#!/bin/sh
#
# Copyright 2003-2020 Intel Corporation.
# 
# This software and the related documents are Intel copyrighted materials, and
# your use of them is governed by the express license under which they were
# provided to you (License). Unless the License provides otherwise, you may
# not use, modify, copy, publish, distribute, disclose or transmit this
# software or the related documents without Intel's prior written permission.
# 
# This software and the related documents are provided as is, with no express
# or implied warranties, other than those that are expressly stated in the
# License.
#
# shellcheck shell=sh

get_script_path() (
    script="$1"
    while [ -L "$script" ] ; do
        script_dir=$(command dirname -- "$script")
        script_dir=$(cd "$script_dir" && command pwd -P)
        script="$(readlink "$script")"
        case $script in
            (/*) ;;
            (*) script="$script_dir/$script" ;;
        esac
    done
    script_dir=$(command dirname -- "$script")
    script_dir=$(cd "$script_dir" && command pwd -P)
    echo "$script_dir"
)

usage() {
    printf "%s\n"   "ERROR: This script must be sourced."
    printf "%s\n"   "Usage: source $1"
    exit 255
}

vars_script_name=""
if [ -n "${ZSH_VERSION:-}" ] ; then
    case $ZSH_EVAL_CONTEXT in (*:file*) vars_script_name="${(%):-%x}" ;; esac ;
    if [ -n "$ZSH_EVAL_CONTEXT" ] && [ "" = "$vars_script_name" ] ; then
        usage "${(%):-%x}" ;
    fi
elif [ -n "${KSH_VERSION:-}" ] ; then
    if [ "$(set | grep KSH_VERSION)" = "KSH_VERSION=.sh.version" ] ; then
        if [ "$(cd "$(dirname -- "$0")" && pwd -P)/$(basename -- "$0")" \
            != "$(cd "$(dirname -- "${.sh.file}")" && pwd -P)/$(basename -- "${.sh.file}")" ] ; then
            vars_script_name="${.sh.file}" || usage "$0" ;
        fi
    else
        _lmksh="$(basename -- "$0")" ;
        if [ "mksh" = "$_lmksh" ] || [ "lksh" = "$_lmksh" ] || [ "ksh" = "$_lmksh" ] ; then
            vars_script_name="$( (echo "${.sh.file}") 2>&1 )" || : ;
            vars_script_name="$(expr "$vars_script_name" : '^.*ksh: \(.*\)\[[0-9]*\]:')" ;
        fi
    fi
elif [ -n "${BASH_VERSION:-}" ] ; then
    (return 0 2>/dev/null) && vars_script_name="${BASH_SOURCE}" || usage "${BASH_SOURCE}"
else
    case ${0##*/} in (sh|dash) vars_script_name="" ;; esac
fi

if [ "" = "$vars_script_name" ] ; then
    >&2 echo ":: ERROR: Unable to proceed: no support for sourcing from '[dash|sh]' shell." ;
    >&2 echo "   This script must be sourced. Did you execute or source this script?" ;
    >&2 echo "   Can be caused by sourcing from inside a \"shebang-less\" script." ;
    >&2 echo "   Can also be caused by sourcing from ZSH version 4.x or older." ;
    return 1 2>/dev/null || exit 1
fi


MPIVARS_DIR=$(get_script_path "$vars_script_name")
I_MPI_ROOT="$( cd "${MPIVARS_DIR}/.."  && pwd )"

export I_MPI_ROOT

print_help()
{
    echo ""
    echo "Usage: vars.sh [-i_mpi_ofi_internal[=0|1]] [-i_mpi_library_kind[=debug|debug_mt|release|release_mt]]"
    echo ""
    echo "-i_mpi_ofi_internal specifies whether to use libfabric from the Intel(R) MPI Library."
    echo ""
    echo "If the arguments to the sourced script are ignored (consult docs"
    echo "for your shell) the alternative way to specify target is environment"
    echo "variable I_MPI_LIBRARY_KIND to pass"
    echo "-i_mpi_library_kind to the script."
    echo "Use variable I_MPI_OFI_LIBRARY_INTERNAL to pass -i_mpi_ofi_internal setting."
    echo ""
}

PLATFORM=""
if [ -z "${PATH}" ]
then
    PATH="${I_MPI_ROOT}/${PLATFORM}/bin"; export PATH
else
    PATH="${I_MPI_ROOT}/${PLATFORM}/bin:${PATH}"; export PATH
fi

if [ -z "${CLASSPATH}" ]
then
    CLASSPATH="${I_MPI_ROOT}/${PLATFORM}/lib/mpi.jar"; export CLASSPATH
else
    CLASSPATH="${I_MPI_ROOT}/${PLATFORM}/lib/mpi.jar:${CLASSPATH}"; export CLASSPATH
fi

if [ -z "${LD_LIBRARY_PATH}" ]
then
    LD_LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/lib/release:${I_MPI_ROOT}/${PLATFORM}/lib"; export LD_LIBRARY_PATH
else
    LD_LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/lib/release:${I_MPI_ROOT}/${PLATFORM}/lib:${LD_LIBRARY_PATH}"; export LD_LIBRARY_PATH
fi

if [ -z "${PLATFORM}" ]
then
    if [ -z "${LIBRARY_PATH}" ]
    then
        LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/lib/release:${I_MPI_ROOT}/${PLATFORM}/lib"; export LIBRARY_PATH
    else
        LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/lib/release:${I_MPI_ROOT}/${PLATFORM}/lib:${LIBRARY_PATH}"; export LIBRARY_PATH
    fi

    if [ -z "${CPATH}" ]
    then
        CPATH="${I_MPI_ROOT}/${PLATFORM}/include"; export CPATH
    else
        CPATH="${I_MPI_ROOT}/${PLATFORM}/include:${CPATH}"; export CPATH
    fi
fi

export MANPATH="${I_MPI_ROOT}/man:${MANPATH}:"

if [ -z "${I_MPI_OFI_LIBRARY_INTERNAL}" ]
then
    i_mpi_ofi_library_internal="1"
else
    i_mpi_ofi_library_internal="${I_MPI_OFI_LIBRARY_INTERNAL}"
fi

i_mpi_library_kind="${I_MPI_LIBRARY_KIND}"

args=$*
for arg in $args
do
    case "$arg" in
        -i_mpi_ofi_internal|--i_mpi_ofi_internal)
            i_mpi_ofi_library_internal=1
            ;;
        -i_mpi_ofi_internal=*|--i_mpi_ofi_internal=*)
            i_mpi_ofi_library_internal="${arg#*=}"
            ;;
        -h|--help)
            print_help
            return 0
            ;;
        -i_mpi_library_kind=*|--i_mpi_library_kind=*)
            i_mpi_library_kind="${arg#*=}"
            ;;
    esac
done

case "$i_mpi_library_kind" in
    debug|debug_mt|release|release_mt)
        LD_LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/lib/${i_mpi_library_kind}:${I_MPI_ROOT}/${PLATFORM}/lib:${LD_LIBRARY_PATH}"; export LD_LIBRARY_PATH
        ;;
esac

case "$i_mpi_ofi_library_internal" in
    0|no|off|disable)
        ;;
    *)
        PATH="${I_MPI_ROOT}/${PLATFORM}/libfabric/bin:${PATH}"; export PATH
        LD_LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/libfabric/lib:${LD_LIBRARY_PATH}"; export LD_LIBRARY_PATH
        if [ -z "${LIBRARY_PATH}" ]
        then
            LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/libfabric/lib"; export LIBRARY_PATH
        else
            LIBRARY_PATH="${I_MPI_ROOT}/${PLATFORM}/libfabric/lib:${LIBRARY_PATH}"; export LIBRARY_PATH
        fi
        FI_PROVIDER_PATH="${I_MPI_ROOT}/${PLATFORM}/libfabric/lib/prov:/usr/lib64/libfabric"; export FI_PROVIDER_PATH
        ;;
esac
