#!/bin/sh
# shellcheck shell=sh

# Copyright Intel Corporation
# SPDX-License-Identifier: MIT
# https://opensource.org/licenses/MIT


# ############################################################################

# Overview

# This script collects all the available oneAPI modulefiles and organizes them
# into a folder that can be added to the $MODULEPATH env variable or by way of
# the "module use" command. For each tool/component that is found in the
# oneAPI installation folder, all available versions of modulefiles associated
# with that tool/component are added to the final output folder.

# NOTE: this script is expected to work on Linux and macOS. This means we
# must be sure to avoid GNU options with external commands since macOS tends
# to use older BSD commands.


# ############################################################################

# To be called if we encounter bad command-line args or user asks for help.

# Inputs:
#   none
#
# Outputs:
#   message to stdout

script_name=$(basename -- "$0")

usage() {
  echo "  "
  echo "usage: ${script_name}" '[--output-dir=dir]' '[--help]'
  echo "  "
  echo "Scans the oneAPI installation folder for available modulefiles and organizes"
  echo "them into a single folder that can be added to the \$MODULEPATH environment"
  echo "variable or by using the 'module use' command. For each tool or library that"
  echo "is found in the oneAPI installation folder, all versions available for those"
  echo "tools and libraries are added to the output folder."
  echo "  "
  echo "  --output-dir=path/to/folder/name"
  echo "  "
  echo "    Specify path/name of folder to contain oneAPI modulefile links."
  echo "    Default location is 'modulefiles' folder inside the oneAPI install dir."
  echo "      e.g., --output-dir=~/intel-oneapi-modulefiles"
# TODO: change output-dir default to ~/modulefiles ; see also TODO below
  echo "  "
  echo "  --help"
  echo "  "
  echo "    Display this help message and exit."
  echo "  "
}

# TODO: add support for input folder option
# usage() {
#   echo "  "
#   echo "usage: ${script_name}" '[--input-dir=dir] [--output-dir=dir] [--help]'
#   echo "  "
#   echo "Scans the oneAPI installation folder for available modulefiles and organizes"
#   echo "them into a single folder that can be added to the \$MODULEPATH environment"
#   echo "variable or by using the 'module use' command. For each tool or library that"
#   echo "is found in the oneAPI installation folder, all versions available for those"
#   echo "tools and libraries are added to the output folder."
#   echo "  "
#   echo "  --input-dir=path/to/oneapi/install/dir"
#   echo "  "
#   echo "    Specify oneAPI installation directory to be scanned."
#   echo "    Multiple instances of --input=dir are allowed."
#   echo "    Defaults to folder containing this script."
#   echo "  "
#   echo "  --output-dir=path/to/folder/name"
#   echo "  "
#   echo "    Specify path/name of folder to contain oneAPI modulefile links."
#   echo "    Default location is 'modulefiles' folder inside the oneAPI install dir."
#   echo "      e.g., --output-dir=~/intel-oneapi-modulefiles"
#   echo "  "
#   echo "  --help"
#   echo "  "
#   echo "    Display this help message and exit."
#   echo "  "
# }


# ############################################################################

# Get absolute path to script. **NOTE:** `readlink` is not a POSIX command!!
# Uses `readlink` to remove links and `pwd -P` to turn into an absolute path.
# see also: https://stackoverflow.com/a/12145443/2914328

# Usage:
#   script_dir=$(get_script_path "$script_rel_path")
#
# Inputs:
#   script/relative/pathname/scriptname
#
# Outputs:
#   /script/absolute/pathname

# executing function in a *subshell* to localize vars and effects on `cd`
get_script_path() (
  script="$1"
  while [ -L "$script" ] ; do
    # combining next two lines fails in zsh shell
    script_dir=$(command dirname -- "$script")
    script_dir=$(cd "$script_dir" && command pwd -P)
    script="$(readlink "$script")"
    case $script in
      (/*) ;;
       (*) script="$script_dir/$script" ;;
    esac
  done
  # combining next two lines fails in zsh shell
  script_dir=$(command dirname -- "$script")
  script_dir=$(cd "$script_dir" && command pwd -P)
  echo "$script_dir"
)


# ###########################################################################

# Make sure we are being executed, not sourced.
# Making this detection for a variety of /bin/sh impersonators is overkill.
# If it becomes necessary to do that, we can add support right here.

# if [ "$0" != "${BASH_SOURCE}" ] ; then
#   echo "  "
#   echo ":: ERROR: Incorrect usage: \"$script_name\" must be executed, not sourced." ;
#   usage
#   return 255 2>/dev/null || exit 255
# fi


# ############################################################################

# Interpret command-line arguments passed to this script.
# Set the default location for the final modulefiles output folder.
# see https://unix.stackexchange.com/a/258514/103967

# TODO: change modulesoutdir default to ~/modulefiles ; see also TODO above

help=0
script_root=$(get_script_path "${0}")
#modulesoutdir=${HOME}/modulefiles
modulesoutdir=${script_root}/modulefiles

for arg do
  shift
  case "$arg" in
    (--help)
      help=1
      ;;
    (--output-dir=*)
      modulesoutdir="$(expr "$arg" : '--output-dir=\(.*\)')"
      ;;
    (*)
      set -- "$@" "$arg"
      ;;
  esac
  # echo "\$@ = " "$@"
done

# Save a copy of the arguments array ($@) in case needed later.
# This copy excludes the arguments consumed by this script.
# TODO: this should be preserving spaces within the arguments
# env_vars_args="$*"

if [ "$help" != "0" ] ; then
  usage
  return 254 2>/dev/null || exit 254
fi


# ############################################################################

# Create the output modulefiles directory.
# Clean it up in case of a pre-existing copy.

echo ":: Initializing oneAPI modulefiles folder ..."

# Create the modulefiles output folder.
command -p mkdir -p "$modulesoutdir"

echo ":: Removing any previous oneAPI modulefiles folder content."

# TODO: ask user if okay to clean a pre-existing modulefiles output folder.
# TODO: option to ask will also require a "--force" command-line option.
# Clean the destination folder of dangling symbolic links and empty folders.
# Can happen after an uninstall of a subset of the currently installed tools.
find -L "$modulesoutdir" -type l -exec rm {} \;
find "$modulesoutdir" -empty -type d -delete >/dev/null 2>&1



# ############################################################################

# Process oneAPI components.
# Scan for modulefiles and create symlinks in the modulefiles output folder.

echo ":: Generating oneAPI modulefiles folder links."

# each subdirectory is a potential oneAPI "component"
# make sure each "component" variable ends with a trailing '/' character
for component in $(command -p ls -d "$script_root"/*/) ; do
  versiondircount=$(find "$component" -mindepth 1 -maxdepth 1 -type d | wc -l)
  if [ "$versiondircount" -gt 0 ] ; then

    # each subdirectory of a component is a version specifier
    # using 'ls -d' rather than find because it sees symlinked dirs
    for versiondir in $(command -p ls -d "$component"*/) ; do
      version=$(basename "$versiondir")
      modulefilesindir=${versiondir}modulefiles

      if [ -d "$modulefilesindir" ] ; then
        files="$modulefilesindir/*"
        for modulefile in $files ; do
          modulename=$(basename "$modulefile")

          # resolve modulefiles that were symlinked into expected location
          if [ -h "$modulefile" ] ; then
            modulefile="$(get_script_path "$modulefile")/${modulename}"
          fi

          echo "-- ${modulename}/${version} -> $modulefile"

          # create module directory
          if [ ! -d "$modulesoutdir/$modulename" ] ; then
            command -p mkdir -p "$modulesoutdir/$modulename"
          fi
          # create a symlink to the modulefile located in the install dir
          # TODO: -f option may be dangerous, seek alternate option
          command -p ln -fs "$modulefile" "$modulesoutdir/$modulename/$version"
        done
      fi

    done

  fi
done

echo ":: oneAPI modulefiles folder initialized."
echo ":: oneAPI modulefiles folder is here: \"$modulesoutdir\"" ;
