#!/bin/sh
# shellcheck shell=sh

#===============================================================================
# Copyright 1985-2019 Intel Corporation All Rights Reserved.
#
# The source code,  information  and material  ("Material") contained  herein is
# owned by Intel Corporation or its  suppliers or licensors,  and  title to such
# Material remains with Intel  Corporation or its  suppliers or  licensors.  The
# Material  contains  proprietary  information  of  Intel or  its suppliers  and
# licensors.  The Material is protected by  worldwide copyright  laws and treaty
# provisions.  No part  of  the  Material   may  be  used,  copied,  reproduced,
# modified, published,  uploaded, posted, transmitted,  distributed or disclosed
# in any way without Intel's prior express written permission.  No license under
# any patent,  copyright or other  intellectual property rights  in the Material
# is granted to  or  conferred  upon  you,  either   expressly,  by implication,
# inducement,  estoppel  or  otherwise.  Any  license   under such  intellectual
# property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing,  you may not remove or alter this
# notice or  any  other  notice   embedded  in  Materials  by  Intel  or Intel's
# suppliers or licensors in any way.
#===============================================================================

# ############################################################################
# Get absolute path to script, when sourced from bash, zsh and ksh shells.
# Uses `readlink` to remove links and `pwd -P` to turn into an absolute path.
# Derived from similar function used by VTune and Advisor.
# Converted into a POSIX-compliant function.

# Usage:
#   script_dir=$(get_root_path "$script_rel_path")
#
# Inputs:
#   script/relative/pathname
#
# Outputs:
#   /script/absolute/pathname

# executing function in a *subshell* to localize vars and effects on `cd`
get_root_path() (
  script="$1"
  while [ -L "$script" ] ; do
    # combining next two lines fails in zsh shell
    script_dir=$(command dirname -- "$script")
    script_dir=$(cd "$script_dir" && command pwd -P)
    script="$(readlink "$script")"
    case $script in
      (/*) ;;
       (*) script="$script_dir/$script" ;;
    esac
  done
  # combining next two lines fails in zsh shell
  script_dir=$(command dirname -- "$script")
  script_dir=$(cd "$script_dir" && command pwd -P)
  echo "$script_dir"
)


# ############################################################################

# Even though this script is designed to be POSIX compatible, there are lines
# in the code block below that are _not_ POSIX compatible. This works within a
# POSIX compatible shell because they are single-pass interpreters. Each "if
# test" that checks for a non-POSIX shell (zsh, bash, etc.) will return a
# "false" condition in a POSIX shell and, thus, will skip the non-POSIX lines.
# This requires that the "if test" constructs _are_ POSIX compatible.

usage() {
  printf "%s\n"   "ERROR: This script must be sourced."
  printf "%s\n"   "Usage: source $1"
  return 2 2>/dev/null || exit 2
}

if [ -n "$ZSH_EVAL_CONTEXT" ] ; then
  # following only executed in zsh
  # shellcheck disable=2039,2015
  [[ $ZSH_EVAL_CONTEXT =~ :file$ ]] && vars_script_name="${(%):-%x}" || usage "${(%):-%x}"
elif [ -n "$KSH_VERSION" ] ; then
  # following only executed in ksh
  # shellcheck disable=2039,2015
  [[ $(cd "$(dirname -- "$0")" && printf '%s' "${PWD%/}/")$(basename -- "$0") != \
  "${.sh.file}" ]] && vars_script_name="${.sh.file}" || usage "$0"
elif [ -n "$BASH_VERSION" ] ; then
  # following only executed in bash
  # shellcheck disable=2039,2015
  (return 0 2>/dev/null) && vars_script_name="${BASH_SOURCE[0]}" || usage "${BASH_SOURCE[0]}"
else
  case ${0##*/} in (sh|dash) vars_script_name="" ;; esac
fi

if [ "" = "$vars_script_name" ] ; then
  >&2 echo ":: ERROR: Unable to proceed: no support for sourcing from '[dash|sh]' shell." ;
  >&2 echo "   Can be caused by sourcing from inside a \"shebang-less\" script." ;
  return 1
fi

# ############################################################################

# Export required env vars for compiler package.
CMPLR_ROOT=$(dirname -- "$(get_root_path "$vars_script_name")")

INTEL_TARGET_ARCH="intel64"
INTEL_TARGET_PLATFORM="linux"

while [ $# -gt 0 ]
do
  case "$1" in
  ia32)
    INTEL_TARGET_ARCH="ia32"
    shift
    ;;
  intel64)
    INTEL_TARGET_ARCH="intel64"
    shift
    ;;
  *)
    shift
    ;;
  esac
done

# This environment variable to switch il0 compiler to support ia32 mode.
if [ ${INTEL_TARGET_ARCH} = "ia32" ]; then
  export INTEL_TARGET_ARCH_IA32=ia32
else
  unset INTEL_TARGET_ARCH_IA32
fi

# Compiler driver and other executables are in intel64 (host), include ia32/intel64 targets.
if [ -z "${PATH}" ]; then
  export PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin/intel64:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/ioc/bin"
else
  export PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin/intel64:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/ioc/bin:${PATH}"
fi

if [ -z "${CPATH}" ]; then
  export CPATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/include"
else
  export CPATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/include:${CPATH}"
fi

if [ -z "${LIBRARY_PATH}" ]; then
  export LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib"
else
  export LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib:${LIBRARY_PATH}"
fi

if [ ${INTEL_TARGET_ARCH} = "ia32" ]; then
  export LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/ia32_lin:${LIBRARY_PATH}"
else
  export LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${LIBRARY_PATH}"
fi

# intel64 libraries also be needed in ia32 target
if [ -z "${LD_LIBRARY_PATH}" ]; then
  export LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib"
else
  export LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib:${LD_LIBRARY_PATH}"
fi

if [ ${INTEL_TARGET_ARCH} = "ia32" ]; then
  export LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/ia32_lin:${LD_LIBRARY_PATH}"
fi

# OpenCL FPGA runtime
if [ -f "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/oclfpga/fpgavars.sh"  ]; then
  # shellcheck source=/dev/null
  source "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/oclfpga/fpgavars.sh" "$@"
fi

# OpenMP offload and CPU LLVM libraries and headers (includes OpenCL CPU run-time)
export LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/emu:${LD_LIBRARY_PATH}"

# adding OpenCL CPU ICD filename, OpenCL FPGA part should be prepend here, added in fpgavars.sh
if [ -z "${OCL_ICD_FILENAMES}" ]; then
  export OCL_ICD_FILENAMES="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64/libintelocl.so"
else
  export OCL_ICD_FILENAMES="${OCL_ICD_FILENAMES}:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64/libintelocl.so"
fi

# TODO: is this MIC_LD_LIBRARY_PATH still requrired? No mic directory under lib.
if [ -d "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/mic" ]; then
    if [ -z "${MIC_LD_LIBRARY_PATH}" ]; then
        export MIC_LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/mic"
    else
        export MIC_LD_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/mic:${MIC_LD_LIBRARY_PATH}"
    fi
    if [ -z "${MIC_LIBRARY_PATH}" ]; then
        export MIC_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/mic"
    else
        export MIC_LIBRARY_PATH="${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/mic:${MIC_LIBRARY_PATH}"
    fi
fi

export MANPATH=${MANPATH}:${CMPLR_ROOT}/documentation/en/man/common

