#include "MAR_pp.def"
subroutine turtke_gen(dt_dif)
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE (TKE)                                XF 09-11-2020  MAR |
    ! |    subroutine turtke_gen includes 1 1/2 Vertical Turbulence Closures   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! | METHOD: 1. `Standard' Closure of Turbulence:                           |
    ! | ^^^^^^^     E - epsilon  , Duynkerke,           JAS 45, 865--880, 1988 |
    ! |  .OR.   2.  E - epsilon  , Huang and Raman,     BLM 55, 381--407, 1991 |
    ! |  .OR.   3.  TKE          , Therry et Lacarrere, BLM 25,  63-- 88, 1983 |
    ! |                                                                        |
    ! | INPUT  : itexpe: Nb of iterations                                      |
    ! | ^^^^^^^^ dt_dif: Local Time Step                                   (s) |
    ! |          explIO: Experiment Label                                  (s) |
    ! |                                                                        |
    ! | INPUT / OUTPUT: The Vertical Turbulent Fluxes are included for:        |
    ! | ^^^^^^^^^^^^^^                                                         |
    ! |   a) Turbulent Kinetic Energy             ect_TE(mx,my,mz)     (m2/s2) |
    ! |   b) Turbulent Kinetic Energy Dissipation eps_TE(mx,my,mz)     (m2/s3) |
    ! |                                                                        |
    ! | OUTPUT :  TUkvm(i,j,k): vertical diffusion coeff. for momentum  (m2/s) |
    ! | ^^^^^^^^  TUkvh(i,j,k): vertical diffusion coeff. for heat...   (m2/s) |
    ! |          zi__TE(i,j)  : inversion height                           (m) |
    ! |                                                                        |
    ! | OPTIONS: #De: Dirichlet Type Top Boundary Condit. for ect_TE & eps_TE  |
    ! | ^^^^^^^^ #WE: Output on MAR.TKE (unit 29)                              |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_dy
    use mar_te
    use mar_tu
    use mar_hy
    use mar_sl
    use mar_wk
    use mar_io
#ifdef De
    ! CAa : MAR_Di.inc does not exist
    ! use mar_di
#endif

    implicit none

    ! +--Global Variables
    ! +  ================

    real dt_dif

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    logical log_KA
    integer locTKE, mz__KA
    common / turtke_gen_loc / locTKE, mz__KA

    integer i5, n5, kect, ke
#ifdef WE
    integer m30, m31
#endif

    real aalme(mx, mz)
    real aalmk(mx, mz)
    real flott(mx, mz)

    real epse, epsd, ectmin, turmx, dkect
    real sqrcmu, sqrcmu0
    real cmu, c1ep, c2ep, rc1c2, betahr, voninv
    real cmu0, c1ep0, c2ep0
    real ectcrt, akz, alme, almk, pousse
    real hrel, base
    real epslme, ectnew, produc, factur, zeta, stab_s, phimm
    real edtnum, edtden

    real phim, dzlev
    real sature, sgnLMO, absLMO, se
    real alamb, Ri_Sat
    real fac_Ri, vuzvun, Kz_vun
    real kz_max, kz_mix, kz2mix, KvhMax(mx, my, mz)

    ! +--DATA
    ! +  ====

    data epse/0.00000100/
    data epsd/0.00000001/

    ! +...     ectmin:Minimum SBL    turbulent kinetic   energy
    data ectmin/0.00010000/

    ! +...Bintanja , 2000, BLM (95),milieu p. 355
    data cmu/0.090/
#ifdef PD
    ! +...Duynkerke, 1988, JAS (45),  haut p. 868
    data cmu0/0.033/
#endif
#ifdef KI
    ! +...Kitada   , 1987, BLM (41),  haut  p.220
    data cmu0/0.090/
#endif
    ! +...Bintanja , 2000, BLM (95),milieu p. 355
    data sqrcmu/3.333/
#ifdef PD
    ! +...Duynkerke, 1988, JAS (45),  (19) p. 869 :(c_mu)^1/2=(0.033)^1/2=5.50
    data sqrcmu0/5.500/
#endif
#ifdef KI
    ! +...Kitada   , 1987, BLM (41),        p.220 :(c_mu)^1/2=(0.090)^1/2=3.333
    data sqrcmu0/3.333/
#endif
#ifdef Kl
    ! +...Schayes and Thunis, 1990, Contrib. 60 Inst.Astr.Geoph. p.8
    data sqrcmu0/4.000/
#endif

    ! +...Duynkerke, 1988, JAS (45),milieu p. 868
    ! +...Bintanja , 2000, BLM (95),milieu p. 355
    data c1ep/1.46/
#ifdef KI
    ! +...Kitada   , 1987, BLM (41),  haut  p.220
    data c1ep0/1.44/
#endif
    ! +...Bintanja , 2000, BLM (95),milieu p. 355
    data c2ep/1.92/
#ifdef PD
    ! +...Duynkerke, 1988, JAS (45),milieu p. 868
    data c2ep0/1.83/
#endif
#ifdef KI
    ! +...Kitada   , 1987, BLM (41),  haut  p.220
    data c2ep0/1.92/
#endif

#ifdef HR
    ! +...Huang and Raman, 1991, BLM (55), p.386 and (A22) p.405
    data betahr/2.00/
#endif

    data alamb/0.1/! 1 / 10 m
    data Ri_Sat/2.0/! Mahalov&al., 2004, GRL
#ifdef PD
    cmu = cmu0
#endif
#ifdef KI
    cmu = cmu0
#endif
#ifdef PD
    sqrcmu = sqrcmu0
#endif
#ifdef KI
    sqrcmu = sqrcmu0
#endif
#ifdef Kl
    sqrcmu = sqrcmu0
#endif
#ifdef KI
    c1ep = c1ep0
#endif
#ifdef PD
    c2ep = c2ep0
#endif
#ifdef KI
    c2ep = c2ep0
#endif

    rc1c2 = c1ep / c2ep

    ! +--Parameters
    ! +  ==========

    voninv = 1./vonkar

    n5 = 5
    n5 = min(mx, n5)

    ! +--Initialisation
    ! +  ==============

    if(.not. ini_KA_TE) then
        ini_KA_TE = .true.
        mz__KA = mz1
11      continue
        if(zsigma(mz__KA) > 5. .OR. mz__KA <= 1) go to 10
        mz__KA = mz__KA - 1
        go to 11
10      continue
        write(6, 1000) mz - mz__KA
1000    format(/, ' TKE: Moving Average until mz -', i2, ' Level',/)
    endif

    if(locTKE == 0 .and. itexpe <= 1) then

        log_KA = .true.
#ifdef Kl
        log_KA = .false.
        if(log_KA) then
            write(6, 6000)
6000        format('!?%@&* Conflicting Use of #KA and  #Kl', &
                   ' ==> Emergency EXIT in turtke_gen', 6x, &
                   ' ### do NOT PRE-PROCESS #KA with #Kl ###')
            STOP
        endif
#endif

        ! +--Minimum Vertical Turbulent Diffusion Coefficient (ARPS.4.0 Users Guide,
        ! +  ------------------------------------------------  fin para 6.3.4 p.143)

        do k = 1, mz1
            dzlev = zsigma(k) - zsigma(k + 1)
            TUmin(k) = akmol
#ifdef ARPS
            TUmin(k) = min(0.15, epsi * dzlev * dzlev)
#endif
        enddo
        k = mz
        dzlev = zsigma(k)
        TUmin(k) = akmol
#ifdef ARPS
        TUmin(k) = min(0.15, epsi * dzlev * dzlev)
#endif

    endif

    ! +--Initial E,e
    ! +  -----------

!$OMP PARALLEL do &
!$OMP private(i,j,k,hrel,sature,base,sgnLMO,absLMO,dkect, &
!$OMP akz,alme,almk,pousse,epslme,ectnew,produc,factur, &
!$OMP se,ke,edtden,edtnum,kz2mix,kz_max,kz_mix, &
!$OMP Kz_vun,fac_Ri,vuzvun)
    do j = 1, my
        if(locTKE == 0 .and. itexpe <= 1) then
            do k = 1, mz
                do i = 1, mx
                    ect_TE(i, j, k) = epse
                    eps_TE(i, j, k) = epsd
                    ! +...     These initial values of TKE and epsilon correspond to K ~ Kmol

                    TUkvm(i, j, k) = akmol
                    TUkvh(i, j, k) = akmol
                enddo
            enddo
            !   end do

            !   do j=1,my
            do i = 1, mx
                zi__TE(i, j) = max(gpmiDY(i, j, mz) * grvinv - sh(i, j), &
                                   zi__TE(i, j))
            enddo
            !   end do

            ! +--Verification: TKE must be Positive Definite
            ! +  ===========================================

        else
            !   do j=1,my
            do k = 1, mz
                do i = 1, mx
                    ect_TE(i, j, k) = max(epse, ect_TE(i, j, k))
                    eps_TE(i, j, k) = max(epsd, eps_TE(i, j, k))
                enddo
            enddo
            !   end do

            ! +--Inversion Height
            ! +  ================

            do i = 1, mx
                WKxy1(i, j) = 0.05 * max(max(ect_TE(i, j, mmz1), &
                                             ect_TE(i, j, mz)), ectmin)
                WKxy2(i, j) = 1.
            enddo

            do k = 1, mz
                do i = 1, mx
                    if(ect_TE(i, j, k) < WKxy1(i, j)) then
                        WKxy2(i, j) = min(mz1, k)
                    endif
                enddo
            enddo

            do i = 1, mx
                k = WKxy2(i, j)
                if(ect_TE(i, j, k + 1) < ectmin) then
                    WKxy1(i, j) = gpmiDY(i, j, mz) * grvinv &
                                  - sh(i, j)
                else
                    dkect = ect_TE(i, j, k) - ect_TE(i, j, k + 1)
                    WKxy1(i, j) = (gpmiDY(i, j, k + 2) &
                                   + (gpmiDY(i, j, k + 1) - gpmiDY(i, j, k + 2)) &
                                   * (WKxy1(i, j) - ect_TE(i, j, k + 1)) &
                                   / (sign(unun, dkect) &
                                      * max(eps9, abs(dkect))) &
                                   - gplvDY(i, j, mzz) &
                                   ) * grvinv
                endif
            enddo

            do i = 1, mx
                zi__TE(i, j) = min(WKxy1(i, j), &
                                   gpmiDY(i, j, 1) * grvinv - sh(i, j))
            enddo

            do i = 1, mx
                zi__TE(i, j) = max(gpmiDY(i, j, mz) * grvinv - sh(i, j), &
                                   zi__TE(i, j))
                WKxy1(i, j) = 0.
                WKxy2(i, j) = 0.
#ifdef WR
                if(zi__TE(i, j) > 10000.) then
                    kect = k
                    kect = min(mz - 3, kect)
                    kect = max(5, kect)
                    write(6, 6001) i, j, k, kect,(ke, ke=kect + 3, kect - 4, -1) &
                        , zi__TE(i, j),(ect_TE(i, j, ke), ke=kect + 3, kect - 4, -1)
6001                format('zi / TKE', 2(i6, i4), 8i10, /, d10.3, 18x, 8d10.3)
                endif
#endif
            enddo
        endif

        ! +--TKE Production/Destruction by the Vertical Wind Shear
        ! +  =====================================================

        ! do j=1,my
        do k = kp1(1), mmz1
            do i = 1, mx
                WKxyz3(i, j, k) = uairDY(i, j, k) - uairDY(i, j, k + 1)
                WKxyz4(i, j, k) = vairDY(i, j, k) - vairDY(i, j, k + 1)
            enddo
        enddo
        ! end do

        do k = 1, mz
            ! do j=1,my
            do i = 1, mx
                WKxyz5(i, j, k) = &
                    gravit / (gplvDY(i, j, k) - gplvDY(i, j, k + 1))
                ! +...   1/dz(k+1/2)
            enddo
            ! end do
        enddo

        do k = kp1(1), mmz1
            ! do j=1,my
            do i = 1, mx
                WKxyz1(i, j, k) = &
                    TUkvm(i, j, k) * (WKxyz3(i, j, k) * WKxyz3(i, j, k) &
                                      + WKxyz4(i, j, k) * WKxyz4(i, j, k)) &
                    * WKxyz5(i, j, k) * WKxyz5(i, j, k)
            enddo
            ! end do
        enddo

        ! do j=1,my
        do i = 1, mx
            WKxyz1(i, j, mz) = 0.0
        enddo
        ! end do

        ! +--Buoyancy
        ! +  ========

        ! +--Reduced (Equivalent) Potential Temperature
        ! +  ------------------------------------------

        ! do j=1,my
        do i = 1, mx
            WKxy5(i, j) = pktaDY(i, j, mzz) &
                          * exp(Lv_H2O * qvapSL(i, j) / (cp * TairSL(i, j)))
        enddo
        ! end do

        do k = 1, mz
            ! do j=1,my
            do i = 1, mx
                WKxyz2(i, j, k) = pktaDY(i, j, k) &
                                  * exp(Lv_H2O * qvDY(i, j, k) / (cp * tairDY(i, j, k)))
            enddo
            ! end do
        enddo

        ! +--Buoyancy Coefficients
        ! +  ---------------------

        do k = 1, mz
            !c#HY    do j=1,my
            do i = 1, mx

                hrel = 0.50 * (qvDY(i, j, k) / qvswDY(i, j, k) + &
                               qvDY(i, j, kp1(k)) / qvswDY(i, j, k))
                WKxy3(i, j) = 0.50 * (qvswDY(i, j, k) + qvswDY(i, j, k + 1))
                WKxy4(i, j) = 0.50 * (tairDY(i, j, k) + tairDY(i, j, kp1(k)))

                sature = max(0., sign(1., hrel + eps12 - 1.))
                base = WKxy3(i, j) * Lv_H2O / (RDryAi * WKxy4(i, j))

                ! +--Vectorization of the unsaturated (H<1) and saturated cases (H=1.)
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                WKxy1(i, j) = &
                    ! H<1.
                    1.-sature &
                    ! H=1.
                    + sature * (1.0 + base) &
                    / (1.0 + base * 0.605 * Lv_H2O / (cp * WKxy4(i, j)))
                ! +...       C_thq (Duynkerke & Driedonks 1987 JAS 44(1), Table 1 p.47)

                WKxy2(i, j) = &
                    ! H<1.
                    (1.-sature) * (Lv_H2O / (cp * WKxy4(i, j)) &
                                   - 0.605) &
                    ! H=1.
                    + sature
                ! +...       C_q_w (Duynkerke and Driedonks 1987)
                ! +                with (1-Ra/Rv)/(Ra/Rv) =  0.605 [Ra=287.J/kg/K;
                ! +                                                 Rv=461.J/kg/K]

                ! +--Unsaturated Case
                ! +  ~~~~~~~~~~~~~~~~
                ! +        if (hrel.lt.1.0) then
                ! +          WKxy1(i,j) = 1.0
                ! +          WKxy2(i,j) =                    Lv_H2O/(cp*WKxy4(i,j))
                ! +  .                                     - 0.605

                ! +--Saturated   Case
                ! +  ~~~~~~~~~~~~~~~~
                ! +        else
                ! +          base       =      WKxy3(i,j)*Lv_H2O/(RDryAi*WKxy4(i,j))
                ! +          WKxy1(i,j) = (1.0+base)
                ! +  .                   /(1.0+base*0.605*Lv_H2O/(cp    *WKxy4(i,j)))
                ! +          WKxy2(i,j) =  1.0
                ! +        end if

            enddo
            !c#HY    end do

            ! +--Buoyancy
            ! +  --------

            if(k <= mmz1) then
                ! do j=1,my
                do i = 1, mx
                    WKxyz3(i, j, k) = gravit &
                                      * WKxyz5(i, j, k) * ((WKxyz2(i, j, k) - WKxyz2(i, j, k + 1)) * 2.0 &
                                                           / (WKxyz2(i, j, k) + WKxyz2(i, j, k + 1)) &
                                                           * WKxy1(i, j) &
                                                           - WKxy2(i, j) &
                                                           * (qvDY(i, j, k) - qvDY(i, j, kp1(k)) &
                                                              + qwHY(i, j, k) - qwHY(i, j, kp1(k)) &
                                                              + qrHY(i, j, k) - qrHY(i, j, kp1(k)) &
                                                              + qiHY(i, j, k) - qiHY(i, j, kp1(k)) &
                                                              + qsHY(i, j, k) - qsHY(i, j, kp1(k))) &
                                                           )
                    ! +...      (g/theta)                  X(dtheta/dz) :
                    ! +          Buoyancy Parameter beta X grad.vert.temp.pot. en k+1/2
                enddo
                ! end do

            else
                ! do j=1,my
                do i = 1, mx
                    WKxyz3(i, j, k) = gravit &
                                      * WKxyz5(i, j, k) * ((WKxyz2(i, j, k) - WKxy5(i, j)) * 2.0 &
                                                           / (WKxyz2(i, j, k) + WKxy5(i, j)) &
                                                           * WKxy1(i, j) &
                                                           )
                    ! +...      (g/theta)                  X(dtheta/dz) :
                    ! +          Buoyancy Parameter beta X grad.vert.temp.pot. en k+1/2
                enddo
                ! end do
            endif
        enddo

        ! +--Length Scales Parameters (Therry and Lacarrere 1983 Model)
        ! +  ========================

        ! do j=1,my
        do i = 1, mx
            WKxy3(i, j) = 1.0 / zi__TE(i, j)

            WKxy1(i, j) = 15.0 * WKxy3(i, j)
            ! +...   Ce1/hi
            WKxy2(i, j) = 5.0 * WKxy3(i, j)
            ! +...   Ce1/hi
            WKxy4(i, j) = 11.0 * WKxy3(i, j)
            ! +...   Ck2/hi
        enddo
        ! end do

        ! do j=1,my
        do i = 1, mx
            sgnLMO = sign(unun, SLlmo(i, j))
            absLMO = abs(SLlmo(i, j))
            SLlmo(i, j) = sgnLMO * max(absLMO, eps12)
            WKxy3(i, j) = -min(0., sgnLMO) &
                          / (1.0 - min(0., SLlmo(i, j)) / zi__TE(i, j))

            ! +--Replacement of:
            ! +      if (SLlmo(i,j).lt.0.0) then
            ! +          SLlmo(i,j) =        min(SLlmo(i,j),-eps12)
            ! +          WKxy3(i,j) =  1.0/(1.d0-SLlmo(i,j)/zi__TE(i,j))
            ! +      else
            ! +          SLlmo(i,j) =        max(SLlmo(i,j), eps12)
            ! +          WKxy3(i,j) =  0.0
            ! +      end if
            ! +...  m2
            ! end do
        enddo

        do k = kp1(1), mz
            ! do j=1,my
            do i = 1, mx
                WKxyz4(i, j, k) = sqrt(max(zero, WKxyz3(i, j, k)) / ect_TE(i, j, k))
                ! +...   1/ls
            enddo
            ! end do
        enddo

        ! +--Dissipation Length
        ! +  ------------------

        do k = kp1(1), mz
            ! do j=1,my
            do i = 1, mx
                akz = voninv * gravit / (gpmiDY(i, j, k + 1) - gplvDY(i, j, mzz))
                ! +...   1/kz(i,j,k+1/2)
                ! +
                alme = akz + WKxy1(i, j) &
                       - (akz + WKxy2(i, j)) * WKxy3(i, j) / (1.+5.0e-3 * zi__TE(i, j) * akz) &
                       + 1.5 * WKxyz4(i, j, k)
                ! +...   alme=1/Dissipation Length (Therry and Lacarrere, 1983 BLM 25 p.75)

                ! +--Mixing Length
                ! +  -------------

                almk = akz + WKxy1(i, j) &
                       - (akz + WKxy4(i, j)) * WKxy3(i, j) / (1.+2.5e-3 * zi__TE(i, j) * akz) &
                       + 3.0 * WKxyz4(i, j, k)
                ! +...   almk=1/Mixing Length      (Therry and Lacarrere, 1983 BLM 25 p.78)

                ! +--Contribution of Vertical Wind Shear + Buoyancy + Dissipation to TKE
                ! +  ===================================================================

                pousse = -TUkvh(i, j, k) * WKxyz3(i, j, k)

                epslme = eps_TE(i, j, k)
#ifdef Kl
                epslme = 0.125 * alme * sqrt(ect_TE(i, j, k)) * ect_TE(i, j, k)
#endif
                ectnew = ect_TE(i, j, k) * &
                         (ect_TE(i, j, k) + dt_dif * (WKxyz1(i, j, k) + max(zero, pousse))) &
                         / (ect_TE(i, j, k) + dt_dif * (-min(zero, pousse) &
                                                        + epslme))
                ! +...   Numerical Scheme : cfr. E. Deleersnijder, 1992 (thesis) pp.59-61

                ! +--Contribution of Vertical Wind Shear + Buoyancy to epsilon
                ! +  =========================================================

#ifdef HR
                tranTE(i, j, k) = zero
#endif
                produc = WKxyz1(i, j, k) + max(pousse, zero) + max(tranTE(i, j, k), zero)

#ifdef KI
                ! based on standard values of Kitada, 1987, BLM 41, p.220
                produc = WKxyz1(i, j, k) + max(zero, pousse)
#endif

#ifdef BH
                ! based on          values of Betts et Haroutunian, 1983
                ! can be used by replacing strings `c #KI' (except the previous one)
                !                     and `c #BH' by blanks
                !              (cfr. Kitada, 1987, BLM 41, p.220):
                ! buoyancy > 0 (unstability) => (1-ce3) X buoyancy = 1.8  X buoyancy
                ! buoyancy < 0 (  stability) => (1-ce3) X buoyancy =-1.15 X buoyancy
                produc = WKxyz1(i, j, k) + max(zero, pousse) * 1.80 - min(zero, pousse) * 1.15
#endif

                factur = eps_TE(i, j, k) / ect_TE(i, j, k)
                ! Numerical Scheme : cfr. E. Deleersnijder, 1992 (thesis) pp.59-61
                eps_TE(i, j, k) = &
                    eps_TE(i, j, k) &
                    * (eps_TE(i, j, k) + dt_dif * factur * c1ep * produc) &
                    / (eps_TE(i, j, k) + dt_dif * factur * c2ep * eps_TE(i, j, k))

#ifdef Kl
                eps_TE(i, j, k) = epslme
#endif

                ! +--New TKE Value
                ! +  =============

                ect_TE(i, j, k) = ectnew

                ! +--Dissipation Lengths Variables are Assigned for Output Purpose
                ! +  =============================================================

                WKxyz1(i, j, k) = alme
                WKxyz2(i, j, k) = almk
            enddo
            ! end do
        enddo

        ! +--OUTPUT Preparation
        ! +  ==================

        !  if((  itexpe.gt.0.and.jmmMAR.eq.0.and.jssMAR.eq.0.and.
        ! .     ((IO_loc.ge.2.and.    jhurGE   .eq.0) .or.
        ! .      (IO_loc.ge.2.and.mod(jhurGE,3).eq.0) .or.
        ! .      (IO_loc.ge.3)                            )       ).or.
        ! .       IO_loc.ge.7                                          ) then
        !
        !    do i5 = 1,n5
        !    aalme(i5,1)  =0.0
        !    aalmk(i5,1)  =0.0
        !    flott(i5,1)  =0.0
        !    do k = kp1(1),mz
        !  if (WKxyz1(igrdIO(i5),jgrdIO(i5),k).gt.0.0) then
        !    aalme(i5,k) = 1.0/WKxyz1(igrdIO(i5),jgrdIO(i5),k)
        !  else
        !    aalme(i5,k) = 0.0
        !  end if
        !C +...       Le
        !  if (WKxyz2(igrdIO(i5),jgrdIO(i5),k).gt.0.d0) then
        !    aalmk(i5,k) = 1.0/WKxyz2(igrdIO(i5),jgrdIO(i5),k)
        !  else
        !    aalmk(i5,k) = 0.0
        !  end if
        !C +...       Lk
        !    flott(i5,k)  = - WKxyz3(igrdIO(i5),jgrdIO(i5),k)
        !C +...       proportional to the buoyant force (e.g. Demoor, 1978, p.47)
        !    end do
        !    end do
        !  end if

        ! +--Lower Boundary Conditions
        ! +  =========================

        do i = 1, mx
            ! do j=1,my
            WKxy1(i, j) = SLuus(i, j) * SLuus(i, j)            ! -> TKE SBC
            WKxy3(i, j) = (gplvDY(i, j, mz) - gplvDY(i, j, mzz)) * grvinv!    z_SBL
            !c#vL  end do
        enddo

        do i = 1, mx
            !c#vL  do j=1,my
            WKxy2(i, j) = WKxy1(i, j) * SLuus(i, j)            ! -> e   SBC
            ect_TE(i, j, mz) = WKxy1(i, j) * sqrcmu                 !    TKE SBC

            WKxy4(i, j) = WKxy3(i, j) / SLlmo(i, j)            !    zeta
            WKxy5(i, j) = max(0.0, sign(unun, SLlmo(i, j)))          !
            !c#vL  end do
        enddo

        do i = 1, mx
            !c#vL  do j=1,my
            eps_TE(i, j, mz) = WKxy2(i, j) &
                               ! phim Stab.
                               * ((WKxy5(i, j) * (1.+A_Turb * WKxy4(i, j)) &
                                   ! phim Inst.
                                   + (1.0 - WKxy5(i, j)) / (1.-20.*min(0., WKxy4(i, j)))) &
                                  * voninv / WKxy3(i, j) &
                                  - voninv / SLlmo(i, j))
            ! +...   Duynkerke, 1988, JAS (45), (19) p. 869

#ifdef KI
            eps_TE(i, j, mz) = WKxy2(i, j) * voninv / WKxy3(i, j)
#endif
            !c#vL  end do
        enddo

        ! +--When TKE Closure is Used, TKE is Modified near the Lower Boundary
        ! +  -----------------------------------------------------------------

        do i = 1, mx
            !c#vL  do j=1,my
#ifdef KC
            se = max(0., sign(unun, ect_TE(i, j, mmz2) - ect_TE(i, j, mmz1)))
            ke = mmz1 - se
            ! Schayes and Thunis, 1990, Contrib. 60 Inst.Astr.Geoph. p.8, 1.4.4.
            ect_TE(i, j, mmz1) = ect_TE(i, j, ke)
            eps_TE(i, j, mmz1) = eps_TE(i, j, ke)
            ! ect_TE(i, j, mmz1) = ect_TE(i, j, mz)
            ! eps_TE(i, j, mmz1) = eps_TE(i, j, mz)
#endif

            ! +--Upper Boundary Conditions
            ! +  =========================

            ect_TE(i, j, 1) = ect_TE(i, j, kp1(1))
#ifdef De
            ect_TE(i, j, 1) = ect_TE(i, j, mz) * rolvDY(i, j, mz) / rolvDY(i, j, 1)
            ect_DI(i, j) = ect_TE(i, j, 1)
#endif

            eps_TE(i, j, 1) = eps_TE(i, j, kp1(1))
#ifdef De
            eps_TE(i, j, 1) = eps_TE(i, j, mz) * rolvDY(i, j, mz) / rolvDY(i, j, 1)
            eps_DI(i, j) = eps_TE(i, j, 1)
#endif
            ! +
            ! end do
        enddo

        ! +--TKE-e Vertical Moving Average
        ! +  =============================

        do k = mz__KA, mz1
            !c#KA  do j=1,my
            do i = 1, mx
                WKxyz7(i, j, k) = (dsigm1(kp1(k)) * ect_TE(i, j, kp1(k)) &
                                   + dsigm1(k) * ect_TE(i, j, k) * 2. &
                                   +dsigm1(km1(k)) * ect_TE(i, j, km1(k))) &
                                  / (dsigm1(kp1(k)) &
                                     + dsigm1(k) * 2. &
                                     +dsigm1(km1(k)))
                WKxyz8(i, j, k) = (dsigm1(kp1(k)) * eps_TE(i, j, kp1(k)) &
                                   + dsigm1(k) * eps_TE(i, j, k) * 2. &
                                   +dsigm1(km1(k)) * eps_TE(i, j, km1(k))) &
                                  / (dsigm1(kp1(k)) &
                                     + dsigm1(k) * 2. &
                                     +dsigm1(km1(k)))
            enddo
            !c#KA  end do
        enddo

        do k = mz__KA, mz1
            !c#KA  do j=1,my
            do i = 1, mx
                ect_TE(i, j, k) = WKxyz7(i, j, k)
                eps_TE(i, j, k) = WKxyz8(i, j, k)
            enddo
            !c#KA  end do
        enddo

        ! +--Verification: TKE must be Positive Definite
        ! +  ===========================================

        do k = 1, mz
            ! do j=1,my
            do i = 1, mx
                ect_TE(i, j, k) = max(epse, ect_TE(i, j, k))
                eps_TE(i, j, k) = max(epsd, eps_TE(i, j, k))
            enddo
            ! end do
        enddo

#ifdef HR
        ! +--Minimum Energy Dissipation Time
        ! +  ===============================
        do i = 1, mx
            edtnum = 0.0
            edtden = 0.0
            do k = 1, mz
                edtnum = edtnum + ect_TE(i, j, k) * dsig_1(k)
                edtden = edtden + eps_TE(i, j, k) * dsig_1(k)
            enddo
            if(edtden > 0.0) then
                edt_TE(i, j) = betahr * edtnum / edtden
            else
                ! edt_TE set to an arbitrary small value
                edt_TE(i, j) = 1.e-8
            endif
        enddo
#endif

        ! +--Turbulent Diffusion Coefficients
        ! +  ================================

        if(locTKE > 0 .or. itexpe > 1) then

            ! +--Richardson Number (contributors)
            ! +  -----------------

            !   do j=1,my
            do i = 1, mx
                TU_Pra(i, j, mz) = 1.
            enddo
            !   end do

            do k = kp1(1), mz1
                !   do j=1,my
                do i = 1, mx
                    WKxyz6(i, j, k) = 0.0
#ifdef RI
                    WKxyz6(i, j, k) = 0.5 * (pktadY(i, j, k) - pktadY(i, j, k + 1)) * pcap &
                                      / (tairDY(i, j, k) + tairDY(i, j, k + 1))
                    WKxyz7(i, j, k) = max((uairDY(i, j, k) - uairDY(i, j, k + 1))**2 &
                                          + (vairDY(i, j, k) - vairDY(i, j, k + 1))**2, epsi)
#endif
                enddo
                !c#vK    end do
            enddo

            ! +--Richardson Number
            ! +  -----------------

            do k = kp1(1), mz1
                !c#vK    do j=1,my
                do i = 1, mx
                    WKxyz8(i, j, k) = 0.0
#ifdef RI
                    ! g * dz     (k+1/2)
                    WKxyz8(i, j, k) = (gravit / WKxyz5(i, j, k)) &
                                      ! d(theta)/T (k+1/2)
                                      * WKxyz6(i, j, k) &
                                      ! d|V|
                                      / WKxyz7(i, j, k)
#endif
                enddo
                !c#vK    end do
            enddo

            ! +--Diffusion Coefficient for Heat
            ! +  ------------------------------

            do k = kp1(1), mz
                !c#vK    do j=1,my
                do i = 1, mx
                    TUkvh(i, j, k) = &
                        cmu * ect_TE(i, j, k) * ect_TE(i, j, k) / (eps_TE(i, j, k) &
                                                                   )
                    ! nu_t =c_mu X ECT          X ECT          / eps
#ifdef Kl
                    TUkvh(i, j, k) = 0.50 * sqrt(ect_TE(i, j, k)) / WKxyz2(i, j, k)
#endif

                    kz_max = vonkar * (gplvDY(i, j, k + 1) - gplvDY(i, j, mzz)) * grvinv
                    kz_mix = kz_max / (1.+kz_max * 0.1)
                    kz2mix = kz_mix * kz_mix
                    KvhMax(i, j, k) = max(5000., 100. &
                                          *kz2mix * abs((ssvSL(i, j, k) - ssvSL(i, j, kp1(k))) &
                                                        * WKxyz5(i, j, k)))
                    TUkvh(i, j, k) = min(KvhMax(i, j, k), TUkvh(i, j, k))
                    TUkvh(i, j, k) = max(akmol, TUkvh(i, j, k))
                enddo
                !c#vK    end do
            enddo

            ! +--Prandtl Number (Sukoriansky et al., 2005,
            ! +  --------------  BLM 117: 231-257, Eq.15, 19, 20 & Fig.2)

            do k = kp1(1), mz1
                !c#vK    do j=1,my
                do i = 1, mx
#ifdef RI
                    fac_Ri = 5.0 * max(WKxyz8(i, j, k), epsi)
                    vuzvun = 0.4 * (1.-(fac_Ri - 1./fac_Ri) / (fac_Ri + 1./fac_Ri)) + 0.2
                    fac_Ri = 4.2 * max(WKxyz8(i, j, k), epsi)
                    Kz_vun = 0.7 * (1.-(fac_Ri - 1./fac_Ri) / (fac_Ri + 1./fac_Ri))
#endif
                    TU_Pra(i, j, k) = 1.
#ifdef RI
                    TU_Pra(i, j, k) = max(0., sign(1., TUkvh(i, j, k) - 0.20)) &
                                      - min(0., sign(1., TUkvh(i, j, k) - 0.20)) &
                                      * min(vuzvun / max(epsi, Kz_vun), 20.00)
#endif
                enddo
                !   end do
            enddo

            ! +--Diffusion Coefficient for Momentum
            ! +  ----------------------------------

            do k = kp1(1), mz
                !   do j=1,my
                do i = 1, mx
                    TUkvm(i, j, k) = TUkvh(i, j, k)
                    ! +...     cfr Machiels, 1992, TFE (FSA/UCL) (3.21) p.21

#ifdef Kl
                    TUkvm(i, j, k) = 0.7 * TUkvh(i, j, k)
#endif

#ifdef RI
                    TUkvm(i, j, k) = TU_Pra(i, j, k) * TUkvh(i, j, k)
#endif
                enddo
                !   end do
            enddo

        endif

#ifdef OB
        ! +--Lateral Boundary Conditions
        ! +  ===========================
        if(openLB) then
            if(mmx > 1) then
                do k = 1, mz
                    do j = jp11, my1
                        ect_TE(1, j, k) = ect_TE(ip11, j, k)
                        ect_TE(mx, j, k) = ect_TE(mx1, j, k)
                        eps_TE(1, j, k) = eps_TE(ip11, j, k)
                        eps_TE(mx, j, k) = eps_TE(mx1, j, k)
                        TUkvm(1, j, k) = TUkvm(ip11, j, k)
                        TUkvm(mx, j, k) = TUkvm(mx1, j, k)
                        TUkvh(1, j, k) = TUkvh(ip11, j, k)
                        TUkvh(mx, j, k) = TUkvh(mx1, j, k)
                    enddo
                enddo
            endif
            if(mmy > 1) then
                do k = 1, mz
                    do i = ip11, mx1
                        ect_TE(i, 1, k) = ect_TE(i, jp11, k)
                        ect_TE(i, my, k) = ect_TE(i, my1, k)
                        eps_TE(i, 1, k) = eps_TE(i, jp11, k)
                        eps_TE(i, my, k) = eps_TE(i, my1, k)
                        TUkvm(i, 1, k) = TUkvm(i, jp11, k)
                        TUkvm(i, my, k) = TUkvm(i, my1, k)
                        TUkvh(i, 1, k) = TUkvh(i, jp11, k)
                        TUkvh(i, my, k) = TUkvh(i, my1, k)
                    enddo
                enddo
            endif
            ! +... Lateral Boundary Values of Kzm and Kzh are used
            ! +    during the Initialisation Procedure in 1-D Zone
        endif
#endif

        ! +--Hourly Output on Listing
        ! +  ========================

        !  if((  itexpe.gt.0.and.jmmMAR.eq.0.and.jssMAR.eq.0.and.
        ! .     ((IO_loc.ge.2.and.    jhurGE   .eq.0) .or.
        ! .      (IO_loc.ge.2.and.mod(jhurGE,3).eq.0) .or.
        ! .      (IO_loc.ge.3)                            )       ).or.
        ! .       IO_loc.ge.7                                          ) then
        !
        !    do i5=1,n5
        !
        !  do k=1,mz
        !    WKxza(i5,k) = 6.6d-1
        ! .                   * (ect_TE(igrdIO(i5),jgrdIO(i5),k)**1.50)
        ! .                    /(eps_TE(igrdIO(i5),jgrdIO(i5),k)
        !C +...            0.066  = cmu(Duynkerke) * 2
        !c #HR.                     +ect_TE(igrdIO(i5),jgrdIO(i5),k)
        !c #HR.                     /edt_TE(igrdIO(i5),jgrdIO(i5))
        ! .                                                )
        !  end do
        !
        !    write(4 ,61)explIO,igrdIO(i5),jgrdIO(i5),
        ! .            jdarGE,jhlrGE(igrdIO(i5),jgrdIO(i5)),
        ! .            jmmMAR,jssMAR,dt_dif,
        ! .                    xxkm(igrdIO(i5)),
        ! .                   SLlmo(igrdIO(i5),jgrdIO(i5)),
        ! .                  zi__TE(igrdIO(i5),jgrdIO(i5))
        !c #WE      if (mod(jhaMAR,12).eq.0)
        !c #WE.       write(25,61)explIO,igrdIO(i5),jgrdIO(i5),
        !c #WE.            jdarGE,jhlrGE(igrdIO(i5),jgrdIO(i5)),
        !c #WE.            jmmMAR,jssMAR,dt_dif,
        !c #WE.                    xxkm(igrdIO(i5)),
        !c #WE.                   SLlmo(igrdIO(i5),jgrdIO(i5)),
        !c #WE.                  zi__TE(igrdIO(i5),jgrdIO(i5))
        ! 61          format(/,' EXP.',a3,' / Pt. (',i3,',',i3,')',
        ! .                      ' / Turbulence Characteristics on',
        ! .           i5,'d',i2,'LT',i2,'m',i2,'s - dt=',f6.0,
        ! .         /,' ------------------------------------------',
        ! .           '-----------------------------------------',
        ! .       /,' x =',f5.0,'km',3x,'Lo =',e9.2,'m' ,3x,'zi =',f6.0,'m',
        ! .       /,' lev  Altit. Temper.  Pot.T. Wind_u Wind_v   TKE   ',
        ! .         ' epsilon Buoyancy     Le    Lk   Le(e) Prandtl     Kvm',
        ! .                                               '     Kvh Kvh MAX',
        ! .       /,'       [m]     [K]      [K]   [m/s]  [m/s] [m2/s2] ',
        ! .         ' [m2/s2]    [s-2]    [m]   [m]  [m]        [-]  [m2/s]',
        ! .                                               '  [m2/s]  [m2/s]')
        !    write(4 ,62)(k,
        ! .              0.1019*gplvDY(igrdIO(i5),jgrdIO(i5),k),
        ! .                     tairDY(igrdIO(i5),jgrdIO(i5),k),
        ! .              3.7300*pktaDY(igrdIO(i5),jgrdIO(i5),k),
        ! .                     uairDY(igrdIO(i5),jgrdIO(i5),k),
        ! .                     vairDY(igrdIO(i5),jgrdIO(i5),k),
        ! .              0.1019*gpmiDY(igrdIO(i5),jgrdIO(i5),k+1),
        ! .                     ect_TE(igrdIO(i5),jgrdIO(i5),k),
        ! .                     eps_TE(igrdIO(i5),jgrdIO(i5),k),
        ! .              flott(i5,k) ,aalme(i5,k) ,aalmk(i5,k),WKxza(i5,k),
        ! .                       TU_Pra(igrdIO(i5),jgrdIO(i5),k),
        ! .                        TUkvm(igrdIO(i5),jgrdIO(i5),k),
        ! .                        TUkvh(igrdIO(i5),jgrdIO(i5),k),
        ! .                       KvhMax(igrdIO(i5),jgrdIO(i5),k),
        ! .               k=1,mz)
        !    write(4 ,620)
        ! 620         format(
        ! .         ' lev  Altit. Temper.  Pot.T. Wind_u Wind_v   TKE   ',
        ! .         ' epsilon Buoyancy     Le    Lk   Le(e) Prandtl     Kvm',
        ! .                                               '     Kvh Kvh MAX',
        ! .       /,'       [m]     [K]      [K]   [m/s]  [m/s] [m2/s2] ',
        ! .         ' [m2/s2]    [s-2]    [m]   [m]  [m]        [-]  [m2/s]',
        ! .                                               '  [m2/s]  [m2/s]',
        ! .       /,1x)
        !c #WE      if (mod(jhaMAR,12).eq.0) then
        !c #WE        m30=min(mz,30)
        !c #WE        write(25,62)(k,
        !c #WE.              0.1019*gplvDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     tairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              3.7300*pktaDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     uairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     vairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              0.1019*gpmiDY(igrdIO(i5),jgrdIO(i5),k+1),
        !c #WE.                     ect_TE(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     eps_TE(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              flott(i5,k) ,aalme(i5,k) ,aalmk(i5,k),WKxza(i5,k),
        !c #WE.                     TU_Pra(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                      TUkvm(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                      TUkvh(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     KvhMax(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.               k=1,m30)
        !c #WE        write(25,64)
        ! 64          format(/////)
        !c #WE        write(25,61)explIO,jhlrGE(igrdIO(i5),jgrdIO(i5)),
        !c #WE.                          minuGE,jsecGE,dt_dif,
        !c #WE.                    xxkm(igrdIO(i5)),
        !c #WE.                   SLlmo(igrdIO(i5),jgrdIO(i5)),
        !c #WE.                  zi__TE(igrdIO(i5),jgrdIO(i5))
        !c #WE        m31=min(mz,31)
        !c #WE        if (mz.ge.m31)
        !c #WE.       write(25,62)(k,
        !c #WE.              0.1019*gplvDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     tairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              3.7300*pktaDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     uairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     vairDY(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              0.1019*gpmiDY(igrdIO(i5),jgrdIO(i5),k+1),
        !c #WE.                     ect_TE(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     eps_TE(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.              flott(i5,k) ,aalme(i5,k) ,aalmk(i5,k),WKxza(i5,k),
        !c #WE.                     TU_Pra(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                      TUkvm(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                      TUkvh(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.                     KvhMax(igrdIO(i5),jgrdIO(i5),k),
        !c #WE.               k=m31,mz)
        !c #WE      end if
        ! 62        format((i4,f8.0,2f8.2,2f7.2,
        ! .           /,4x,f8.0,30x,3e9.2,2f6.1,5f8.2))
        !  write(4,63) SL_z0(igrdIO(i5),jgrdIO(i5),1),
        ! .                TairSL(igrdIO(i5),jgrdIO(i5))  ,
        ! .                 SLuus(igrdIO(i5),jgrdIO(i5))
        ! 63        format( 4x,f8.6, f8.2,8x,f8.3)
        !
        !  do k=1,mz
        !    WKxza(i5,k) = 0.0
        !  end do
        !
        !    end do
        !  end if

        ! +--Work Arrays Reset
        ! +  =================

        ! do j=1,my
        do i = 1, mx
            WKxy1(i, j) = 0.0
            WKxy2(i, j) = 0.0
            WKxy3(i, j) = 0.0
            WKxy4(i, j) = 0.0
            WKxy5(i, j) = 0.0
        enddo
        ! end do

        do k = 1, mz
            ! do j=1,my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
                WKxyz3(i, j, k) = 0.0
                WKxyz4(i, j, k) = 0.0
                WKxyz5(i, j, k) = 0.0
                WKxyz6(i, j, k) = 0.0
                WKxyz7(i, j, k) = 0.0
                WKxyz8(i, j, k) = 0.0
                tranTE(i, j, k) = 0.0
            enddo
        enddo
    enddo
!$OMP END PARALLEL DO

    locTKE = 1
    ! +...locTKE: turn on TKE Evolution

    return
endsubroutine turtke_gen
