#include "MAR_pp.def"
subroutine turtke_difv(dt_dif, alphaD)
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE (TKE)                                   11-14-2022  MAR |
    ! |   subroutine turtke_difv includes TKE Vertical Turbulence Contribution |
    ! |      to Turbulent Kinetic Energy (ect_TE) and Dissipation (eps_TE)     |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |  INPUT: TUkvm(mx,my,mz): Vertical Turbulent Coeffic.(momentum) [m2/s2] |
    ! |  ^^^^^^                                                                |
    ! |                                                                        |
    ! |  INPUT / OUTPUT: The Vertical Turbulent Fluxes are included for:       |
    ! |  ^^^^^^^^^^^^^^                                                        |
    ! |       a) Turbulent Kinetic Energy             ect_TE(mx,my,mz) [m2/s2] |
    ! |       b) Turbulent Kinetic Energy Dissipation eps_TE(mx,my,mz) [m2/s3] |
    ! |                                                                        |
    ! | #OPTIONS: #De: Dirichlet Type Top Boundary Condit. for ect_TE & eps_TE |
    ! | #^^^^^^^^                                                              |
    ! +------------------------------------------------------------------------+

    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_te
    use mar_tu
    use mar_wk
#ifdef De
    ! CAa : MAR_DI.inc doesn't exist
    ! include 'MAR_DI.inc'
#endif

    implicit none

    ! +--Global Variables
    ! +  ================

    real dt_dif

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    integer k1
    real sige, sigk
    real sige0, sigk0, sigek, alpha, beta, ab, alphaD

    ! +--DATA
    ! +  ====

    ! Bintanja , 2000, BLM (95),milieu p. 355 : 1/sig_e = 1./1.16 = 0.862
    data sige/0.862e0/
    ! Duynkerke, 1988, JAS (45), end a. p.868 : 1/sig_e = 1./2.38 = 0.420
#ifdef PD
    data sige0/0.420e0/
#endif
    ! Kitada   , 1987, BLM (41),        p.220 : 1/sig_e = 1./1.30 = 0.769
#ifdef KI
    data sige0/0.769e0/
#endif
    ! For TKE Closure (Therry and Lacarrere, 1983)
#ifdef Kl
    data sige0/0.420e0/
#endif

    ! Duynkerke, 1988, JAS (45), end a. p.868 : 1/sig_k  = 1./1.00=1.000
    data sigk/1.000e0/
    ! Kitada   , 1987, BLM (41),        p.220 : 1/sig_k  = 1./1.00=1.000
#ifdef KI
    data sigk0/1.000e0/
#endif
    ! Schayes and Thunis, Contribution 60 Inst.Astr.Geoph.(1990) p.6
#ifdef Kl
    data sigk0/1.200e0/
#endif

#ifdef PD
    sige = sige0
#endif
#ifdef KI
    sige = sige0
#endif
#ifdef Kl
    sige = sige0
#endif
    sigek = sige / sigk

    ! +--Parameters for the Numerical Scheme of Vertical Turbulent Transport
    ! +  ===================================================================

    alpha = alphaD            ! Expliciteness := 0 (positive definite)
    beta = 1.00 - alpha        ! Impliciteness
    ab = alpha / beta        !

    ! +--Work Arrays Reset
    ! +  =================

    !$OMP PARALLEL do private (i,j,k,k1)
    do j = 1, my
        do k = 1, mz
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
                WKxyz3(i, j, k) = 0.0
            enddo
        enddo
        !      end do

        ! +--Vertical Diffusion of Turbulent Kinetic Energy
        ! +  ==============================================

        ! +--Tridiagonal Matrix Coefficients - ect_TE
        ! +  ----------------------------------------

        do k = mmz2, 1, -1
            ! do j=   1,my
            do i = 1, mx
                WKxyz1(i, j, k) = -gravi2 * (TUkvm(i, j, k) + TUkvm(i, j, k + 1) &
                                             ) * 0.50 * beta * sigk &
                                  * romiDY(i, j, k) * rolvDY(i, j, k) &
                                  / (pstDY2(i, j) * dsigm1(k) * dsig_1(k))
            enddo
            ! end do

            ! do j=   1,my
            do i = 1, mx
                WKxyz3(i, j, kp1(k)) = WKxyz1(i, j, k) * dsigm1(k) / dsigm1(kp1(k)) &
                                       / rolvDY(i, j, k) * rolvDY(i, j, k + 1)
            enddo
            ! end do

        enddo

        do k = 1, mmz1
            ! do j=   1,my
            do i = 1, mx
                WKxyz1(i, j, k) = WKxyz1(i, j, k) * dt_dif
                WKxyz3(i, j, k) = WKxyz3(i, j, k) * dt_dif
                WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
            enddo
            ! end do
        enddo

        ! +--Second Member of the Tridiagonal System - ect_TE
        ! +  ------------------------------------------------

        ! do j=   1,my
        do i = 1, mx
            WKxyz4(i, j, 1) = &
                WKxyz1(i, j, 1) * ab * (ect_TE(i, j, 1) - ect_TE(i, j, kp1(1)))
#ifdef De
            WKxyz1(i, j, 1) = 0.0
            WKxyz2(i, j, 1) = 1.0
            WKxyz4(i, j, 1) = ect_DI(i, j)
#endif
        enddo
        ! end do

        do k = kp1(1), mmz2
            ! do j=   1,my
            do i = 1, mx
                WKxyz4(i, j, k) = &
                    WKxyz1(i, j, k) * ab * (ect_TE(i, j, k) - ect_TE(i, j, kp1(k))) &
                    - WKxyz3(i, j, k) * ab * (ect_TE(i, j, km1(k)) - ect_TE(i, j, k))
            enddo
            ! end do
        enddo

        ! do j=   1,my
        do i = 1, mx
            WKxyz4(i, j, mmz1) = -(alpha * ect_TE(i, j, mmz1) - ect_TE(i, j, mz)) &
                                 * gravi2 * (TUkvm(i, j, mmz1) + TUkvm(i, j, mmz2) &
                                             ) * 0.50 &
                                 * romiDY(i, j, mmz1) * romiDY(i, j, mmz1) &
                                 / (pstDY2(i, j) * dsigm1(mmz1) * dsig_1(mmz1)) &
                                 - WKxyz3(i, j, mmz1) * ab * (ect_TE(i, j, mmz2) - ect_TE(i, j, mmz1))
        enddo
        ! end do

        ! +--Tridiagonal Matrix Inversion - ect_TE
        ! +  -------------------------------------

        k1 = 1
#ifdef De
        k1 = 2
#endif
        do k = k1, mz
            ! do j=   1,my
            do i = 1, mx
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + ect_TE(i, j, k)
            enddo
            ! end do
        enddo

        ! +         ************
        call MARgz_1mx1my(1, mmz1, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +         ************

        do k = 1, mmz1
            ! do j=   1,my
            do i = 1, mx
                tranTE(i, j, k) = tranTE(i, j, k) + (WKxyz7(i, j, k) - ect_TE(i, j, k)) &
                                  / dt_dif
                ect_TE(i, j, k) = WKxyz7(i, j, k)
            enddo
            ! end do
        enddo

        ! +--Vertical Diffusion of Dissipation
        ! +  =================================

        ! +--Update Tridiagonal Matrix Coefficients - eps_TE
        ! +  -----------------------------------------------

        do k = 1, mmz1
            ! do j=   1,my
            do i = 1, mx
                WKxyz1(i, j, k) = WKxyz1(i, j, k) * sigek
                WKxyz3(i, j, k) = WKxyz3(i, j, k) * sigek
                WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
            enddo
            ! end do
        enddo

        ! +--Second Member of the Tridiagonal System - eps_TE
        ! +  ------------------------------------------------

        ! do j=   1,my
        do i = 1, mx
            WKxyz4(i, j, 1) = &
                WKxyz1(i, j, 1) * ab * (eps_TE(i, j, 1) - eps_TE(i, j, kp1(1)))
#ifdef De
            WKxyz1(i, j, 1) = 0.0
            WKxyz2(i, j, 1) = 1.0
            WKxyz4(i, j, 1) = eps_DI(i, j)
#endif
        enddo
        ! end do

        do k = kp1(1), mmz2
            ! do j=   1,my
            do i = 1, mx
                WKxyz4(i, j, k) = &
                    WKxyz1(i, j, k) * ab * (eps_TE(i, j, k) - eps_TE(i, j, kp1(k))) &
                    - WKxyz3(i, j, k) * ab * (eps_TE(i, j, km1(k)) - eps_TE(i, j, k))
            enddo
            ! end do
        enddo

        ! do j=   1,my
        do i = 1, mx
            WKxyz4(i, j, mmz1) = -(alpha * eps_TE(i, j, mmz1) - eps_TE(i, j, mz)) &
                                 * gravi2 * (TUkvm(i, j, mmz1) + TUkvm(i, j, mmz2) &
                                             ) * 0.50 &
                                 * romiDY(i, j, mmz1) * romiDY(i, j, mmz1) &
                                 / (pstDY2(i, j) * dsigm1(mmz1) * dsig_1(mmz1)) &
                                 - WKxyz3(i, j, mmz1) * ab * (eps_TE(i, j, mmz2) - eps_TE(i, j, mmz1))
        enddo
        ! end do

        ! +--Tridiagonal Matrix Inversion - eps_TE
        ! +  -------------------------------------

        k1 = 1
#ifdef De
        k1 = 2
#endif
        do k = k1, mz
            ! do j=   1,my
            do i = 1, mx
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + eps_TE(i, j, k)
            enddo
            ! end do
        enddo

        ! +         ************
        call MARgz_1mx1my(1, mmz1, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +         ************

        do k = 1, mmz1
            ! do j=   1,my
            do i = 1, mx
                eps_TE(i, j, k) = WKxyz7(i, j, k)
            enddo
            ! end do
        enddo

        ! +--Work Arrays Reset
        ! +  =================

        do k = 1, mz
            !     do j=   1,my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.00
                WKxyz2(i, j, k) = 0.00
                WKxyz3(i, j, k) = 0.00
                WKxyz4(i, j, k) = 0.00
                WKxyz7(i, j, k) = 0.00
            enddo
        enddo
    enddo
    !$OMP END PARALLEL DO

    return
endsubroutine turtke_difv
