subroutine turtke_difh(dt_dif)
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE (TKE)                                   08-09-2017  MAR |
    ! |   subroutine turtke_difh includes TKE Horizontal Diffusion Contribution|
    ! |      to Turbulent Kinetic Energy (ect_TE) and Dissipation (eps_TE)     |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |  INPUT:TUkhx(mx,my,mz): Horizontal Diffusion Coefficient (x-Direction) |
    ! |  ^^^^^ TUkhy(mx,my,mz): Horizontal Diffusion Coefficient (y-Direction) |
    ! |                                                                        |
    ! |  INPUT / OUTPUT                                                        |
    ! |  ^^^^^^^^^^^^^^                                                        |
    ! |      The Horizontal Diffusion and Correction Terms are included for :  |
    ! |       a) Turbulent Kinetic Energy             ect_TE(mx,my,mz) [m2/s2] |
    ! |       b) Turbulent Kinetic Energy Dissipation eps_TE(mx,my,mz) [m2/s2] |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_te
    use mar_tu
    use mar_wk

    implicit none

    ! +--Global Variables
    ! +  ================

    real dt_dif

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    real tran, dx2inv

    ! +--Computes Horizontal Diffusion using an Explicit Scheme
    ! +  ======================================================

    do i = 1, mx1
        do j = 1, my
            WKxy1(i, j) = 0.50 * (pstDY(i, j) + pstDY(ip1(i), j))
        enddo
    enddo

    do j = 1, my1
        do i = 1, mx
            WKxy2(i, j) = 0.50 * (pstDY(i, j) + pstDY(i, jp1(j)))
        enddo
    enddo

    dx2inv = 1.d0 / dx / dx
    do j = 1, my
        do i = 1, mx
            WKxy3(i, j) = dx2inv / pstDY(i, j)
        enddo
    enddo

    !$OMP PARALLEL do private (i,j,k,tran)
    ! +   ================
    do k = kp1(1), mmz1
        ! +   ================

        ! +--Horizontal Diffusion, x-Direction
        ! +  =================================

        do i = ip11, mx1
            do j = jp11, my1
                ! +
                tran = WKxy3(i, j) &
                       * (WKxy1(i, j) * TUkhx(i, j, k) &
                          * (ect_TE(ip1(i), j, k) - ect_TE(i, j, k)) &
                          - WKxy1(im1(i), j) * TUkhx(im1(i), j, k) &
                          * (ect_TE(i, j, k) - ect_TE(im1(i), j, k)))
                tranTE(i, j, k) = tranTE(i, j, k) + tran
                WKxyz1(i, j, k) = dt_dif * tran

                WKxyz2(i, j, k) = dt_dif * WKxy3(i, j) &
                                  * (WKxy1(i, j) * TUkhx(i, j, k) &
                                     * (eps_TE(ip1(i), j, k) - eps_TE(i, j, k)) &
                                     - WKxy1(im1(i), j) * TUkhx(im1(i), j, k) &
                                     * (eps_TE(i, j, k) - eps_TE(im1(i), j, k)))
            enddo
        enddo

        do j = jp11, my1
            do i = ip11, mx1
                ect_TE(i, j, k) = ect_TE(i, j, k) + WKxyz1(i, j, k)
                eps_TE(i, j, k) = eps_TE(i, j, k) + WKxyz2(i, j, k)
            enddo
        enddo

        ! +--Horizontal Diffusion, y-Direction
        ! +  =================================

        if(mmy >= 2) then

            do j = jp11, my1
                do i = ip11, mx1

                    tran = WKxy3(i, j) &
                           * (WKxy2(i, j) * TUkhy(i, j, k) &
                              * (ect_TE(i, jp1(j), k) - ect_TE(i, j, k)) &
                              - WKxy2(i, jm1(j)) * TUkhy(i, jm1(j), k) &
                              * (ect_TE(i, j, k) - ect_TE(i, jm1(j), k)))
                    tranTE(i, j, k) = tranTE(i, j, k) + tran
                    WKxyz1(i, j, k) = dt_dif * tran

                    WKxyz2(i, j, k) = dt_dif * WKxy3(i, j) &
                                      * (WKxy2(i, j) * TUkhy(i, j, k) &
                                         * (eps_TE(i, jp1(j), k) - eps_TE(i, j, k)) &
                                         - WKxy2(i, jm1(j)) * TUkhy(i, jm1(j), k) &
                                         * (eps_TE(i, j, k) - eps_TE(i, jm1(j), k)))
                enddo
            enddo

            do j = jp11, my1
                do i = ip11, mx1
                    ect_TE(i, j, k) = ect_TE(i, j, k) + WKxyz1(i, j, k)
                    eps_TE(i, j, k) = eps_TE(i, j, k) + WKxyz2(i, j, k)
                enddo
            enddo

        endif

        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
            enddo
        enddo

        ! +   ======
    enddo
    ! +   ======
    !$OMP END PARALLEL DO

    ! +--Work Arrays Reset
    ! +  =================

    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.0
            WKxy2(i, j) = 0.0
            WKxy3(i, j) = 0.0
            WKxy4(i, j) = 0.0
        enddo
    enddo

    return
endsubroutine turtke_difh
