subroutine turtke_advh(dt_dif)
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE (TKE)                                   15-04-2021  MAR |
    ! |   subroutine turtke_advh includes TKE Horizontal Advection Contribution|
    ! |              solved by a 1st Order Accurate in Space Upstream Scheme   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |  INPUT         :  dt_dif: Local Time Step                          (s) |
    ! |  ^^^^^^^^^^^^^^                                                        |
    ! |                                                                        |
    ! |  INPUT / OUTPUT: The Vertical Turbulent Fluxes are included for:       |
    ! |  ^^^^^^^^^^^^^^                                                        |
    ! |       a) Turbulent Kinetic Energy             ect_TE(mx,my,mz) [m2/s2] |
    ! |       b) Turbulent Kinetic Energy Dissipation eps_TE(mx,my,mz) [m2/s3] |
    ! |                                                                        |
    ! |  REMARK : The Advected Variables Lateral Boundary Conditions           |
    ! |  ^^^^^^^^ are Fixed Implicitely                                        |
    ! |            1)  inflow => no change                                     |
    ! |            2) outflow => advected Value                                |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_te
    use mar_wk

    implicit none

    ! +--Global Variables
    ! +  ================
    real dt_dif

    ! +--Local  Variables
    ! +  ================
    integer i, j, k, m
    real dti, dtxe(mx, my), dtye(mx, my), tran

    ! +--Parameters
    ! +  ==========
    dti = 1.0 / dt_dif
    do j = 1, my
        do i = 1, mx
            dtxe(i, j) = dt_dif / dx3(i, j)
            dtye(i, j) = dt_dif / dy3(i, j)
        enddo
    enddo

    ! +--Courant Number
    ! +  ==============
    !$OMP PARALLEL do private (i,j,k,tran)
    do k = 1, mmz1
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = -max(zero, demi * (uairDY(i, j, k) + uairDY(i, j, kp1(k)))) &
                                  * dtxe(i, j)
                WKxyz2(i, j, k) = -min(zero, demi * (uairDY(i, j, k) + uairDY(i, j, kp1(k)))) &
                                  * dtxe(i, j)
                WKxyz3(i, j, k) = -max(zero, demi * (vairDY(i, j, k) + vairDY(i, j, kp1(k)))) &
                                  * dtye(i, j)
                WKxyz4(i, j, k) = -min(zero, demi * (vairDY(i, j, k) + vairDY(i, j, kp1(k)))) &
                                  * dtye(i, j)
                ! +...Velocities V are computed in the layers (i.e. at k+1/2)
                ! +   Interpolation based on V=0 at the Surface (very approximative)

            enddo
        enddo
        !     end do

        ! +--Advection (x-Direction)
        ! +  =======================
        ! do k=1,mmz1
        do i = ip11, mx1
            do j = jp11, my1
                tran = WKxyz1(i, j, k) * (ect_TE(i, j, k) - ect_TE(im1(i), j, k)) &
                       + WKxyz2(i, j, k) * (ect_TE(ip1(i), j, k) - ect_TE(i, j, k))
                tranTE(i, j, k) = tranTE(i, j, k) + tran * dti
                ect_TE(i, j, k) = ect_TE(i, j, k) + tran
                eps_TE(i, j, k) = eps_TE(i, j, k) &
                                  + WKxyz1(i, j, k) * (eps_TE(i, j, k) - eps_TE(im1(i), j, k)) &
                                  + WKxyz2(i, j, k) * (eps_TE(ip1(i), j, k) - eps_TE(i, j, k))
            enddo
        enddo
        ! end do

        ! +--Advection (y-Direction)
        ! +  =======================
        if(mmy > 1) then
            ! do k=1,mmz1
            do j = jp11, my1
                do i = ip11, mx1
                    tran = WKxyz3(i, j, k) * (ect_TE(i, j, k) - ect_TE(i, jm1(j), k)) &
                           + WKxyz4(i, j, k) * (ect_TE(i, jp1(j), k) - ect_TE(i, j, k))
                    tranTE(i, j, k) = tranTE(i, j, k) + tran * dti
                    ect_TE(i, j, k) = ect_TE(i, j, k) + tran
                    eps_TE(i, j, k) = eps_TE(i, j, k) &
                                      + WKxyz3(i, j, k) * (eps_TE(i, j, k) - eps_TE(i, jm1(j), k)) &
                                      + WKxyz4(i, j, k) * (eps_TE(i, jp1(j), k) - eps_TE(i, j, k))
                enddo
            enddo
            ! end do
        endif

        ! +--Work Arrays Reset
        ! +  =================
        !     do k=1,mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
                WKxyz3(i, j, k) = 0.0
                WKxyz4(i, j, k) = 0.0
            enddo
        enddo
    enddo
    !$OMP END PARALLEL DO

    return
endsubroutine turtke_advh
