#include "MAR_pp.def"
subroutine turhor_dyn(dtHDif)
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE HORIZONTAL                          Sat 08-09-2017  MAR |
    ! |   subroutine turhor_dyn computes Horizontal Diffusion                  |
    ! |                              and Correction Terms                      |
    ! |                         using an Explicit Scheme                       |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! | INPUT: dt_Loc: Time Step between two CaLLs of Horiz. Diffusion Routine |
    ! | ^^^^^  dtHDif: Time Step between two run   of Horiz. Diffusion Scheme  |
    ! |               (dt_Loc = n X dtHDif, n = 1,2,...)                       |
    ! |        micphy: Cloud Microphysical Scheme Switch                       |
    ! |                                                                        |
    ! | INPUT (via common block)                                               |
    ! | ^^^^^  TUkhx(mx,my,mz): Horizontal Diffusion Coefficient (x-Direction) |
    ! |        TUkhy(mx,my,mz): Horizontal Diffusion Coefficient (y-Direction) |
    ! |                                                                        |
    ! | INPUT / OUTPUT                                                         |
    ! | ^^^^^^^^^^^^^^                                                         |
    ! |      The Horizontal Diffusion and Correction Terms are included for :  |
    ! |       1) The Horizontal     x-Wind Component uairDY(mx,my,mz)    [m/s] |
    ! |       2) The Horizontal     y-Wind Component vairDY(mx,my,mz)    [m/s] |
    ! |                                                                        |
    ! |  #NH  3) The Vertical       z-Wind Component wairNH(mx,my,mz)    [m/s] |
    ! |                                                                        |
    ! |       4) The Potential      Temperature      pktaDY(mx,my,mzz)         |
    ! |       5) The Air Specific   Humidity           qvDY(mx,my,mz)  [kg/kg] |
    ! |                                                                        |
    ! |  #HY  6) The Ice Crystals   Concentration      qiHY(mx,my,mz)  [kg/kg] |
    ! |       7) The Ice Crystals   Number           ccniHY(mx,my,mz)  [Nb/m3] |
    ! |       8) The Cloud Droplets Concentration      qwHY(mx,my,mz)  [kg/kg] |
    ! |       9) The Snow Flakes    Concentration      qsHY(mx,my,mz)  [kg/kg] |
    ! |      10) The Rain Drops     Concentration      qrHY(mx,my,mz)  [kg/kg] |
    ! |                                                                        |
    ! |  #TC 11) The Tracer         Concentration      qxTC(mx,my,mz,ntrac)    |
    ! |                                                                        |
    ! |  REMARK:                                                               |
    ! |  ^^^^^^^                                                               |
    ! |  !. `Standard' Horizontal Diffusion is performed on Sigma Surfaces     |
    ! |  !.  Smagorinski Relation (see Tag et al. 1979, JAM 18, 1429--1441)    |
    ! |  !.  CAUTION: Horizontal Diffusion is switched on with turhor = .true. |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_tu
    use mar_fi
    use mar_hy
    use mar_wk
#ifdef TC
    use mar_tc
#endif
#ifdef NH
    use mar_nh
#endif
#ifdef iso
    use mariso, only: niso, wiso, qvDY_iso, Rdefault, negligible
#endif

    implicit none

#ifdef iso
    logical :: is_iso
    real :: WKxyz1_iso(niso, mx, my, mz)
    real :: WKxyz5_iso(niso, mx, my, mz)
#endif

    real dtHDif

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    integer nntrac, ntDifH, iter, n_kq, ivar, itrac
    real akhsta, akhloc, cflakh, dx2inv, gdx2
    real facxx, facxy, facyy, alph2, beta2, akhm2, alph22, beta22

    real vartop(mx, my), varbot(mx, my)

    ! +--DATA
    ! +  ====

    data nntrac/0/
#ifdef TC
    nntrac = ntrac
#endif

    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz3(i, j, k) = 0.0
                WKxyz4(i, j, k) = 0.0
                WKxyz5(i, j, k) = 0.0
#ifdef iso
                ! initialize working variable for isotopes
                do wiso = 1, niso
                    WKxyz1_iso(wiso, i, j, k) = 0.
                    WKxyz5_iso(wiso, i, j, k) = 0.
                enddo
#endif
            enddo
        enddo
    enddo

#ifdef iso
    is_iso = .false.
#endif

    ! +--Update of Local Variables and
    ! +            Mesh Averaged Horizontal Diffusion Coefficient akhm
    ! +  =============================================================

    akhsta = 0.0
    do k = 1, mz

        do i = ip11, mx1
            do j = jp11, my1
                WKxyz1(i, j, k) = FIkhmn + 0.25 * (TUkhx(i, j, k) + TUkhx(im1(i), j, k))
            enddo
        enddo

        do j = jp11, my1
            do i = ip11, mx1
                WKxyz1(i, j, k) = WKxyz1(i, j, k) + 0.25 * (TUkhy(i, j, k) + TUkhy(i, jm1(j), k))
                akhsta = max(WKxyz1(i, j, k), akhsta)
            enddo
        enddo

    enddo

    ! +--Local Time step
    ! +  ===============

    cflakh = dt_Loc * akhsta / dx / dx
    ntDifH = 3 * cflakh
    ntDifH = max(iun, ntDifH)
    dtHDif = dt_Loc / ntDifH

    ! +--Update of Local Coefficients
    ! +  ============================

    dx2inv = 1.0 / dx / dx

    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.0
            WKxy2(i, j) = 0.0
            WKxy3(i, j) = dx2inv / pstDY(i, j)
            WKxy4(i, j) = 1.0 / (pstDY(i, j) * pstDY(i, j))
        enddo
    enddo

    do i = 1, mx1
        do j = 1, my
            WKxy1(i, j) = 0.5 * (pstDY(i, j) + pstDY(ip1(i), j))
        enddo
    enddo

    if(mmy > 1) then
        do j = 1, my1
            do i = 1, mx
                WKxy2(i, j) = 0.5 * (pstDY(i, j) + pstDY(i, jp1(j)))
            enddo
        enddo
    endif

    gdx2 = 0.5 * gravit / dx
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz3(i, j, k) = gdx2 / (pstDY(i, j) * dsig_2(k))
            enddo
        enddo
    enddo

    do j = 1, my
        do i = 1, mx
            WKxy4(i, j) = 0.0
        enddo
    enddo

    ! +--Global Variables Loop
    ! +  =====================

    n_kq = 5    ! Nombre de variables à diffuser: u,v,(w),T,q

    !$OMP PARALLEL DO private (i,j,k) firstprivate(iter,ivar)
    do k = 1, mz
        do iter = 1, ntDifH
            do ivar = 1, n_kq
#ifdef TC
                if(ivar > 10) go to 341
#endif

                go to(331, 332, 333, 334, 335, 336, 337, 338, 339, 340) ivar

                ! +--u Wind Speed Component
                ! +  ----------------------

331             continue

                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = uairDY(i, j, k)
                    enddo
                enddo

                go to 330

                ! +--v Wind Speed Component
                ! +  ----------------------

332             continue

                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = vairDY(i, j, k)
                    enddo
                enddo

                go to 330

333             continue
                ! +--w Wind Speed Component (Non Hydrostatic Option)
                ! +  -----------------------------------------------
#ifdef NH
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = wairNH(i, j, k)
                        enddo
                    enddo
                enddo
                do j = 1, my
                    do i = 1, mx
                        vartop(i, j) = wairNH(i, j, 1)
                        varbot(i, j) = wairNH(i, j, mz)
                    enddo
                enddo
#endif
                go to 330

                ! +--Potential Temperature
                ! +  ---------------------

334             continue

                ! do k=1,mz
                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = pktaDY(i, j, k)
                    enddo
                enddo
                ! end do

                go to 330

                ! +--Specific Humidity
                ! +  -----------------

335             continue
#ifdef iso
                is_iso = .true.
#endif
                ! do k=1,mz
                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = qvDY(i, j, k)
#ifdef iso
                        ! turbulent advection of isotopic ratio
                        if(qvDY(i, j, k) > negligible) then
                            do wiso = 1, niso
                                WKxyz1_iso(wiso, i, j, k) = qvDY_iso(wiso, i, j, k) / qvDY(i, j, k)
                            enddo
                        else
                            WKxyz1_iso(wiso, i, j, k) = Rdefault(wiso)
                        endif
#endif
                    enddo
                enddo
                ! end do

                go to 330

                ! +--Cloud Droplets Concentration
                ! +  ----------------------------

336             continue

                if(micphy) then

                    ! do k=1,mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = qwHY(i, j, k)
                        enddo
                    enddo
                    ! end do

                endif

                go to 330

                ! +--Ice Crystals Concentration
                ! +  --------------------------

337             continue

                if(micphy) then
                    ! do k=1,mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = qiHY(i, j, k)
                        enddo
                    enddo
                    ! end do
                endif

                go to 330

                ! +--Rain Drops Concentration
                ! +  ------------------------

338             continue

                if(micphy) then
                    ! do k=1,mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = qrHY(i, j, k)
                        enddo
                    enddo
                    ! end do
                endif

                go to 330

                ! +--Snow Flakes Concentration
                ! +  -------------------------

339             continue

                if(micphy) then

                    ! do k=1,mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = qsHY(i, j, k)
                        enddo
                    enddo
                    ! end do

                endif

                go to 330

                ! +--Ice Crystals Number
                ! +  -------------------

340             continue

                if(micphy) then

                    ! do k=1,mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = ccniHY(i, j, k)
                        enddo
                    enddo
                    ! end do

                endif

                go to 330

#ifdef TC
                ! +--Tracers
                ! +  -------

341             continue
                itrac = ivar - 10
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = qxTC(i, j, k, itrac)
                        enddo
                    enddo
                enddo
                do j = 1, my
                    do i = 1, mx
                        vartop(i, j) = qxTC(i, j, 1, itrac)
                        varbot(i, j) = qsTC(i, j, itrac)
                    enddo
                enddo
#endif
                go to 330

330             continue

                ! +--Boundary Conditions
                ! +  ===================
                ! do k=1,mz
                do j = 1, my
                    WKxyz1(1, j, k) = WKxyz1(ip11, j, k)
                    WKxyz1(mx, j, k) = WKxyz1(mx1, j, k)
#ifdef iso
                    if(is_iso) then
                        do wiso = 1, niso
                            WKxyz1_iso(wiso, 1, j, k) = WKxyz1_iso(wiso, ip11, j, k)
                            WKxyz1_iso(wiso, mx, j, k) = WKxyz1_iso(wiso, mx1, j, k)
                        enddo
                    endif
#endif
                enddo
                ! end do

                if(mmy > 1) then
                    ! do k=1,mz
                    do i = 1, mx
                        WKxyz1(i, 1, k) = WKxyz1(i, jp11, k)
                        WKxyz1(i, my, k) = WKxyz1(i, my1, k)
#ifdef iso
                        if(is_iso) then
                            do wiso = 1, niso
                                WKxyz1_iso(wiso, i, 1, k) = WKxyz1_iso(wiso, i, jp11, k)
                                WKxyz1_iso(wiso, i, my, k) = WKxyz1_iso(wiso, i, my1, k)
                            enddo
                        endif
#endif
                    enddo
                    ! end do
                endif

                ! +--Lateral Diffusion of non Vectorial Model Variables (c #DF ON)
                ! +  Lateral Diffusion of all           Model Variables (c #DF OFF)
                ! +     (proportional to the gradient, terms in sigma surfaces)
                ! +  ==============================================================

                ! +--Diffusion in the x Direction on Sigma Surfaces
                ! +  ----------------------------------------------

                do i = ip11, mx1
                    do j = 1, my
                        WKxyz5(i, j, k) = WKxy3(i, j) * &
                                          (WKxy1(i, j) * TUkhx(i, j, k) &
                                           * (WKxyz1(ip1(i), j, k) - WKxyz1(i, j, k)) &
                                           - WKxy1(im1(i), j) * TUkhx(im1(i), j, k) &
                                           * (WKxyz1(i, j, k) - WKxyz1(im1(i), j, k)))
#ifdef iso
                        if(is_iso) then
                            do wiso = 1, niso
                                WKxyz5_iso(wiso, i, j, k) = WKxy3(i, j) * &
                                                            (WKxy1(i, j) * TUkhx(i, j, k) &
                                                             * (WKxyz1_iso(wiso, ip1(i), j, k) - WKxyz1_iso(wiso, i, j, k)) &
                                                             - WKxy1(im1(i), j) * TUkhx(im1(i), j, k) &
                                                             * (WKxyz1_iso(wiso, i, j, k) - WKxyz1_iso(wiso, im1(i), j, k)))
                            enddo
                        endif
#endif
                    enddo
                enddo
                !XF         end do

                ! +--Diffusion in the y Direction on Sigma Surfaces
                ! +  ----------------------------------------------

                if(mmy > 2) then
                    !XF         do k=1,mz
                    do j = jp11, my1
                        do i = 1, mx
                            WKxyz5(i, j, k) = WKxyz5(i, j, k) + WKxy3(i, j) * &
                                              (WKxy2(i, j) * TUkhy(i, j, k) &
                                               * (WKxyz1(i, jp1(j), k) - WKxyz1(i, j, k)) &
                                               - WKxy2(i, jm1(j)) * TUkhy(i, jm1(j), k) &
                                               * (WKxyz1(i, j, k) - WKxyz1(i, jm1(j), k)))
#ifdef iso
                            if(is_iso) then
                                do wiso = 1, niso
                                    WKxyz5_iso(wiso, i, j, k) = WKxyz5_iso(wiso, i, j, k) + WKxy3(i, j) * &
                                                                (WKxy2(i, j) * TUkhy(i, j, k) &
                                                                 * (WKxyz1_iso(wiso, i, jp1(j), k) - WKxyz1_iso(wiso, i, j, k)) &
                                                                 - WKxy2(i, jm1(j)) * TUkhy(i, jm1(j), k) &
                                                                 * (WKxyz1_iso(wiso, i, j, k) - WKxyz1_iso(wiso, i, jm1(j), k)))
                                enddo
                            endif
#endif
                        enddo
                    enddo
                    !XF         end do
                endif

                ! +--Update of the Global Variables
                ! +  ==============================
#ifdef TC
                if(ivar > 10) go to 411
#endif

                go to(401, 402, 403, 404, 405, 406, 407, 408, 409, 410) ivar

                ! +--u Wind Speed Component
                ! +  ----------------------

401             continue
                ! do k=1,mz
                do j = jp11, my1
                    do i = ip11, mx1
                        uairDY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                    enddo
                enddo
                ! end do
                go to 400

                ! +--v Wind Speed Component
                ! +  ----------------------

402             continue
                ! do k=1,mz
                do j = jp11, my1
                    do i = ip11, mx1
                        vairDY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                    enddo
                enddo
                ! end do
                go to 400

403             continue
                ! +--w Wind Speed Component (Non Hydrostatic Option)
                ! +  -----------------------------------------------
#ifdef NH
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            wairNH(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                enddo
#endif
                go to 400

                ! +--Potential Temperature
                ! +  ---------------------

404             continue
                ! do k=1,mz
                do j = jp11, my1
                    do i = ip11, mx1
                        pktaDY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                    enddo
                enddo
                ! end do
                go to 400

                ! +-- Specific Humidity
                ! +   -----------------

405             continue
                ! do k=1,mz
                do j = jp11, my1
                    do i = ip11, mx1
                        qvDY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
#ifdef iso
                        ! restoration of mass from turbulent diffusion of isotopic ratio
                        do wiso = 1, niso
                            qvDY_iso(wiso, i, j, k) = (WKxyz1_iso(wiso, i, j, k) + WKxyz5_iso(wiso, i, j, k) * dtHDif) &
                                                      * qvDY(i, j, k)
                        enddo
#endif
                    enddo
                enddo
#ifdef iso
                is_iso = .false.
#endif
                ! end do
                go to 400

                ! +--Cloud Droplets Concentration
                ! +  ----------------------------

406             continue
                if(micphy) then
                    ! do k=1,mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qwHY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                    ! end do
                endif
                go to 400

                ! +--Ice Crystals Concentration
                ! +  --------------------------

407             continue
                if(micphy) then
                    ! do k=1,mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qiHY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                    ! end do
                endif
                go to 400

                ! +--Rain Drops Concentration
                ! +  ------------------------

408             continue
                if(micphy) then
                    ! do k=1,mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qrHY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                    ! end do
                endif
                go to 400

                ! +--Snow Flakes Concentration
                ! +  -------------------------

409             continue
                if(micphy) then
                    ! do k=1,mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qsHY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                    ! end do
                endif
                go to 400

                ! +--Ice Crystals Number
                ! +  -------------------

410             continue
                if(micphy) then
                    ! do k=1,mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            ccniHY(i, j, k) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                    ! end do
                endif
                go to 400

                ! +--Tracers
                ! +  -------
411             continue
#ifdef TC
                itrac = ivar - 10
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qxTC(i, j, k, itrac) = WKxyz1(i, j, k) + WKxyz5(i, j, k) * dtHDif
                        enddo
                    enddo
                enddo
#endif
400             continue
            enddo ! ivar = 1, n_kq
        enddo ! iter = 1, ntDifH
        ! +--Work Arrays Reset
        ! +  =================
        !     do k=1,mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.00
                WKxyz2(i, j, k) = 0.00
                WKxyz3(i, j, k) = 0.00
                WKxyz4(i, j, k) = 0.00
                WKxyz5(i, j, k) = 0.00
            enddo
        enddo
    enddo ! k = 1, mz
    !$OMP END PARALLEL DO

    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.00
            WKxy2(i, j) = 0.00
            WKxy3(i, j) = 0.00
            WKxy4(i, j) = 0.00
        enddo
    enddo

    return
endsubroutine turhor_dyn
