#include "MAR_pp.def"
subroutine TURabl
    ! +------------------------------------------------------------------------+
    ! | MAR TURBULENCE (ABL)                                   10-05-2021  MAR |
    ! |   subroutine TURabl includes the Contribution of Vertical Turbulence   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |  INPUT (via common block)                                              |
    ! |  ^^^^^  micphy         : Cloud Microphysics Switch                     |
    ! |         dt_Loc         : Vertical Diffusion Time Step              [s] |
    ! |                                                                        |
    ! |         TUkvm(mx,my,mz): Vertical Turbulent Coeffic.(momentum) [m2/s]  |
    ! |         TUkvh(mx,my,mz): Vertical Turbulent Coeffic.(heat)     [m2/s]  |
    ! |         SLuus(mx,my)   : Friction Velocity                     [m/s]   |
    ! |         SLuts(mx,my)   : Surface Layer Heat     Turbulent Flux [mK/s]  |
    ! |         SLuqs(mx,my)   : Surface Layer Moisture Turbulent Flux [m/s]   |
    ! |         qvapSL ! CAa#ifdef iso warning, input !
    ! |        uss_HY(mx,my)   : Surface Layer Blowing* Turbulent Flux [m/s]   |
    ! |                                                                        |
    ! |  INPUT / OUTPUT: The Vertical Turbulent Fluxes are included for:       |
    ! |  ^^^^^^^^^^^^^^                                                        |
    ! |       1) The Horizontal     x-Wind Component uairDY(mx,my,mz)    [m/s] |
    ! |       2) The Horizontal     y-Wind Component vairDY(mx,my,mz)    [m/s] |
    ! |                                                                        |
    ! |       3) The Potential      Temperature      pktaDY(mx,my,mzz)         |
    ! |       4) The Air Specific   Humidity           qvDY(mx,my,mz)  [kg/kg] |
    ! |                                                                        |
    ! |       5) The Ice Crystals   Concentration      qiHY(mx,my,mz)  [kg/kg] |
    ! |       6) The Ice Crystals   Number           ccniHY(mx,my,mz)  [Nb/m3] |
    ! |       7) The Cloud Droplets Concentration      qwHY(mx,my,mz)  [kg/kg] |
    ! |       8) The Snow Flakes    Concentration      qsHY(mx,my,mz)  [kg/kg] |
    ! |       9) The Rain Drops     Concentration      qrHY(mx,my,mz)  [kg/kg] |
    ! |                                                                        |
    ! |      10) The Tracer         Concentration      qxTC(mx,my,mz,ntrac)    |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_tu
    use mar_hy
    use mar_sl
    use mar_wk
    use marmagic
    use trackwater, only: track_water_turabl, jtw, &
                          delta_qv, j_turabl_mix, j_turabl_sbl, delta_qv_tmp
#ifdef NH
    use mar_nh
#endif
#ifdef TC
    use mar_tc
#endif
#ifdef EW
    use mar_ew
#endif
#ifdef iso
    !CAa todo : add qiHY, qsHY, qwHY, qrHY
    use mariso, only: wiso, niso, qvDY_iso, &
                      negligible, Rdefault, qvapSL_iso, SLuqs_iso
#endif

    implicit none

#ifdef iso
    ! Riso : isotopic ratio
    real :: Riso(niso, mx, my, mz)
    real :: Riso_uqstar
    real :: Riso_qvap
    real, save :: WKxyz4_iso(niso, mx, my, mz)
    real, save :: WKxyz7_iso(niso, mx, my, mz)
    real, save :: varin_iso(mx, my, mz)
    real, save :: varout_iso(mx, my, mz)
#endif

    logical Q_Impl
    common / TURabl_lo / Q_Impl

    integer lous, lotu
    common / TURabl_IN / lous, lotu

    real alpha, beta, ab
    common / TURabl_re / alpha, beta, ab

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    integer i1_tua, i2_tua, j1_tua, j2_tua, k1_tua, k2_tua, n, km, kp
    real uustar, ssvu(mx, my), ssvv(mx, my)
    real utstar, uqstar, qvap
    real waterb, ussno, dd_sno
    real ratio_rfsf, ratio_temp, ratio_prec

    ! +--Parameters
    ! +  ==========

    if(iterun == 0) then

        ! +--Parameters for the Inclusion of the Friction Velocity u*
        ! +  --------------------------------------------------------

        lous = 1
        lous = 1
        ! +...  lous =1 : SLuus    is          used
        ! +     lous =0 : SLuus**2 is (partly) replaced by  K du / dz
        ! +     CAUTION : do NOT USE lous =0 EXCEPT WHEN VERIFYING THE EKMAN SPIRAL

        ! +--Parameters for the Numerical Scheme of Vertical Turbulent Transport
        ! +  -------------------------------------------------------------------

        Q_Impl = .false.
        Q_Impl = .true.
        Q_Impl = .false.
#ifdef TC
        if(Q_Impl) stop ' #~�@�! BAD Vertical Diffusion of gaseous tracers'
#endif
    endif

    ! +-------------------------------------------------------------------------

    ! +--INITIALIZATION
    ! +  ==============

    if(itexpe == 0) then
        do j = 1, my
            do i = 1, mx
                ssvSL(i, j, mz) = max(sqrt(uairDY(i, j, mz) * uairDY(i, j, mz) &
                                           + vairDY(i, j, mz) * vairDY(i, j, mz)), epsi)
                do n = 1, mw
                    cdmSL(i, j, n) = 0.04
                    cdhSL(i, j, n) = 0.04
                    SLuusl(i, j, n) = cdmSL(i, j, n) * ssvSL(i, j, mz)
                enddo
            enddo
        enddo

        do j = 1, my
            do i = 1, mx
                duusSL(i, j) = 0.
                dutsSL(i, j) = 0.
            enddo
        enddo
    endif

    ! +-------------------------------------------------------------------------

    ! +--Vertical Diffusion of Horizontal Momentum
    ! +  =========================================

    ! +--Implicit Surface Scheme
    ! +  -----------------------

    !$OMP PARALLEL do private(i,j,k,i1_tua,i2_tua,j1_tua,j2_tua, &
    !$OMP k1_tua,k2_tua,n,km,kp,alpha,beta,ab, &
    !$OMP ratio_rfsf,ratio_temp,ratio_prec, &
    !$OMP uustar,utstar,uqstar,qvap,waterb,ussno,dd_sno)
    do j = jp11, my1

        alpha = 0.25               !
        beta = 1.00 - alpha         ! Impliciteness
        ab = alpha / beta         !

        do i = ip11, mx1
            Kv__SL(i, j) = 0.
            aeCdSL(i, j) = 0.
        enddo

        do n = 1, mw
            do i = ip11, mx1
                ! aerodynamic conductance
                aeCdSL(i, j) = aeCdSL(i, j) + cdmSL(i, j, n) * SLuusl(i, j, n) &
                               * Slsrfl(i, j, n)
            enddo
        enddo
        do i = ip11, mx1
            ! Kv Contrib. in SL
            Kv__SL(i, j) = &
                -gravit * aeCdSL(i, j) * beta &
                * rolvDY(i, j, mz) / (pstDY(i, j) * dsigm1(mz))
        enddo

        ! +--Tridiagonal Matrix Coefficients : u and v
        ! +  -----------------------------------------

        do i = ip11, mx1
            ssvu(i, j) = uairDY(i, j, mz) / ssvSL(i, j, mz)
            ssvv(i, j) = vairDY(i, j, mz) / ssvSL(i, j, mz)
        enddo

        ! +--Diagonal A
        ! +  ~~~~~~~~~~
        do i = ip11, mx1
            WKxyz1(i, j, mz) = Kv__SL(i, j)
        enddo

        do k = mmz1, 1, -1
            kp = kp1(k)
            do i = ip11, mx1
                WKxyz1(i, j, k) = -gravi2 * beta * romiDY(i, j, k) * &
                                  TUkvm(i, j, k) * rolvDY(i, j, k) / &
                                  (pstDY2(i, j) * dsigm1(k) * dsig_1(k))
            enddo

            ! +--Diagonal C
            ! +  ~~~~~~~~~~
            do i = ip11, mx1
                WKxyz3(i, j, kp) = WKxyz1(i, j, k) * dsigm1(k) / dsigm1(kp) &
                                   * (rolvDY(i, j, kp) / rolvDY(i, j, k))
            enddo

        enddo

        ! +--A, B, C
        ! +  ~~~~~~~
        do k = 1, mmz1
            do i = ip11, mx1
                WKxyz1(i, j, k) = WKxyz1(i, j, k) * dt_Loc
                WKxyz3(i, j, k) = WKxyz3(i, j, k) * dt_Loc
                WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
            enddo
        enddo

        ! +--Vertical B.C.
        ! +  ~~~~~~~~~~~~~
        do i = ip11, mx1
            WKxyz3(i, j, 1) = 0.0
            WKxyz2(i, j, 1) = 1.0 - WKxyz1(i, j, 1)

            WKxyz1(i, j, mz) = WKxyz1(i, j, mz) * dt_Loc
            WKxyz3(i, j, mz) = WKxyz3(i, j, mz) * dt_Loc
            WKxyz2(i, j, mz) = 1.0 - WKxyz3(i, j, mz) - WKxyz1(i, j, mz)
        enddo

        ! +--Second Member of the Tridiagonal System - u
        ! +  -------------------------------------------

        kp = kp1(1)
        do i = ip11, mx1
            WKxyz4(i, j, 1) = WKxyz1(i, j, 1) &
                              * ab * (uairDY(i, j, 1) - uairDY(i, j, kp))
        enddo

        do k = kp1(1), mmz1
            kp = kp1(k)
            km = km1(k)
            do i = ip11, mx1
                WKxyz4(i, j, k) = &
                    WKxyz1(i, j, k) * ab * (uairDY(i, j, k) - uairDY(i, j, kp)) &
                    - WKxyz3(i, j, k) * ab * (uairDY(i, j, km) - uairDY(i, j, k))
            enddo
        enddo

        do i = ip11, mx1

            uustar = SLuus(i, j) * SLuus(i, j)

            ! +--Implicit Surface Scheme
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~
            ! explicit
            uustar = 0.1 * duusSL(i, j)

            WKxyz4(i, j, mz) = &
                WKxyz1(i, j, mz) * ab * uairDY(i, j, mz) &
                - WKxyz3(i, j, mz) * ab * (uairDY(i, j, mmz1) - uairDY(i, j, mz)) &
                - lous * alpha * gravit * romiDY(i, j, mz) * dt_Loc &
                * uustar * ssvu(i, j) / (pstDY(i, j) * dsigm1(mz))
        enddo

        ! +--Tridiagonal Matrix Inversion - u
        ! +  --------------------------------

        k1_tua = 1
        do k = k1_tua, mz
            do i = ip11, mx1
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + uairDY(i, j, k)
            enddo
        enddo

        k1_tua = 1
        k2_tua = mz

        ! +  ************
        call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +  ************

        do k = 1, mz
            do i = ip11, mx1
                uairDY(i, j, k) = WKxyz7(i, j, k)
            enddo
        enddo

        ! +--Second Member of the Tridiagonal System - v
        ! +  -------------------------------------------

        kp = kp1(1)
        do i = ip11, mx1
            WKxyz4(i, j, 1) = WKxyz1(i, j, 1) &
                              * ab * (vairDY(i, j, 1) - vairDY(i, j, kp))
        enddo

        do k = kp1(1), mmz1
            km = km1(k)
            kp = kp1(k)
            do i = ip11, mx1
                WKxyz4(i, j, k) = &
                    WKxyz1(i, j, k) * ab * (vairDY(i, j, k) - vairDY(i, j, kp)) &
                    - WKxyz3(i, j, k) * ab * (vairDY(i, j, km) - vairDY(i, j, k))
            enddo
        enddo

        do i = ip11, mx1
            uustar = SLuus(i, j) * SLuus(i, j)
            ! +--Implicit Surface Scheme
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~
            uustar = 0.1 * duusSL(i, j)                         ! explicit
            duusSL(i, j) = 0.9 * duusSL(i, j)                         !

            WKxyz4(i, j, mz) = &
                WKxyz1(i, j, mz) * ab * vairDY(i, j, mz) &
                - WKxyz3(i, j, mz) * ab * (vairDY(i, j, mmz1) - vairDY(i, j, mz)) &
                - lous * alpha * gravit * romiDY(i, j, mz) * dt_Loc &
                * uustar * ssvv(i, j) / (pstDY(i, j) * dsigm1(mz))
        enddo

        ! +--Tridiagonal Matrix Inversion - v
        ! +  --------------------------------

        k1_tua = 1
        do k = k1_tua, mz
            ! do j=     jp11,my1
            do i = ip11, mx1
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + vairDY(i, j, k)
            enddo
            ! end do
        enddo

        k1_tua = 1
        k2_tua = mz

        ! +  ************
        call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +  ************

        do i = ip11, mx1
            uustar = aeCdSL(i, j) * ((alpha * WKxy1(i, j) + beta * uairDY(i, j, mz)) * ssvu(i, j) &
                                     + (alpha * vairDY(i, j, mz) + beta * WKxyz7(i, j, mz)) * ssvv(i, j))
            duusSL(i, j) = duusSL(i, j) + SLuus(i, j) * SLuus(i, j) - uustar
        enddo

        do k = 1, mz
            do i = ip11, mx1
                vairDY(i, j, k) = WKxyz7(i, j, k)
            enddo
        enddo

        ! +-------------------------------------------------------------------------

        ! +--Vertical Diffusion of Heat and Water Vapor
        ! +  ==========================================

        ! +--Implicit Surface Scheme
        ! +  -----------------------

        ! do j=jp11,my1
        do i = ip11, mx1
            Kv__SL(i, j) = 0.
            aeCdSL(i, j) = 0.
        enddo
        ! end do

        do n = 1, mw
            do i = ip11, mx1
                ! aerodynamic conductance
                aeCdSL(i, j) = aeCdSL(i, j) + cdhSL(i, j, n) * SLuusl(i, j, n) &
                               * Slsrfl(i, j, n)
            enddo
        enddo

        do i = ip11, mx1
            ! Kv Contrib.  above   SL
            ! +         Kv__SL(i,j) = &
            ! +        -gravi2*romiDY(i,j,mz) * TUkvh(i,j,mz)         *beta &
            ! +               *rolvDY(i,j,mz)/(pstDY2(i,j)* dsigm1(mz)*dsig_1(mz))
            ! Kv Contrib. in      SL
            Kv__SL(i, j) = &
                -gravit * aeCdSL(i, j) * beta &
                * rolvDY(i, j, mz) / (pstDY(i, j) * dsigm1(mz))
        enddo

        ! +--Tridiag. Matrix Coeff. : pktaDY,   qvDY
        ! +  ---------------------------------------

        ! +--Diagonal A
        ! +  ~~~~~~~~~~
        k = mz
        do i = ip11, mx1
            WKxyz1(i, j, k) = Kv__SL(i, j)
        enddo

        do k = mmz1, 1, -1
            do i = ip11, mx1
                ! TUkvh : Vertical Turbulent Coeffic. (heat) [m2/s]
                ! WKxyz1 = - g^2 rho^2 * TUkvh / dp^2  [dp/dz = rho g => rho g / dp = 1/dz]
                ! WKxyz1 in [m2 s-1] * [m-2] -> multiplied by dt hereafter
                WKxyz1(i, j, k) = -gravi2 * beta * &
                                  romiDY(i, j, k) * TUkvh(i, j, k) * rolvDY(i, j, k) / &
                                  (pstDY2(i, j) * dsigm1(k) * dsig_1(k))
            enddo
        enddo

        ! +--Diagonal C
        ! +  ~~~~~~~~~~
        do k = mz, 1, -1
            kp = kp1(k)
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz3(i, j, kp) = WKxyz1(i, j, k) * dsigm1(k) / dsigm1(kp) &
                                   / rolvDY(i, j, k) * rolvDY(i, j, kp)
            enddo
            ! end do
        enddo

        ! +--A, B, C
        ! +  ~~~~~~~
        do k = 1, mz
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz1(i, j, k) = WKxyz1(i, j, k) * dt_Loc
                WKxyz3(i, j, k) = WKxyz3(i, j, k) * dt_Loc
                WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
            enddo
            ! end do
        enddo

        ! +--Vertical B.C.
        ! +  ~~~~~~~~~~~~~
        ! do j=jp11,my1
        do i = ip11, mx1
            WKxyz3(i, j, 1) = 0.0
            WKxyz2(i, j, 1) = 1.0 - WKxyz1(i, j, 1)
        enddo
        ! end do

        ! +--SBC of the Tridiagonal System - pktaDY
        ! +  --------------------------------------

        ! do j=jp11,my1
        do i = ip11, mx1
            utstar = SLuts(i, j)

            ! +--Implicit Surface Scheme
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~
            ! explicit
            utstar = 0.1 * dutsSL(i, j)
            ! set := 0
            dutsSL(i, j) = 0.9 * dutsSL(i, j)
            !
            ! partly explicit
            WKxyz4(i, j, mz) = WKxyz1(i, j, mz) &
                               * (ab * pktaDY(i, j, mz) - pktaSL(i, j) / beta) &
                               - WKxyz3(i, j, mz) * ab * (pktaDY(i, j, mmz1) - pktaDY(i, j, mz)) &
                               ! u*T* all explicit
                               - gravit * dt_Loc * rolvDY(i, j, mz) &
                               * utstar / (pcap * pstDY(i, j) * dsigm1(mz))
        enddo
        ! end do

        ! +--Second Member of the Tridiagonal System - pktaDY
        ! +  ------------------------------------------------

        do k = kp1(1), mmz1
            km = km1(k)
            kp = kp1(k)
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz4(i, j, k) = &
                    WKxyz1(i, j, k) * ab * (pktaDY(i, j, k) - pktaDY(i, j, kp)) &
                    - WKxyz3(i, j, k) * ab * (pktaDY(i, j, km) - pktaDY(i, j, k))
            enddo
            ! end do
        enddo

        ! +--UBC           of the Tridiagonal System - pktaDY
        ! +  ------------------------------------------------

        kp = kp1(1)
        do i = ip11, mx1
            WKxyz4(i, j, 1) = &
                WKxyz1(i, j, 1) * ab * (pktaDY(i, j, 1) - pktaDY(i, j, kp))
        enddo

        ! +--Tridiagonal Matrix Inversion - pktaDY
        ! +  -------------------------------------

        k1_tua = 1
        do k = k1_tua, mz
            do i = ip11, mx1
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + pktaDY(i, j, k)
            enddo
        enddo

        k1_tua = 1
        k2_tua = mz

        ! +  ************
        call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +  ************

        do i = ip11, mx1
            utstar = aeCdSL(i, j) * (alpha * pktaDY(i, j, mz) + beta * WKxyz7(i, j, mz) &
                                     - pktaDY(i, j, mzz)) &
                     * pcap
            dutsSL(i, j) = dutsSL(i, j) + SLuts(i, j) - utstar
        enddo

        do k = 1, mz
            do i = ip11, mx1
                pktaDY(i, j, k) = WKxyz7(i, j, k)
            enddo
        enddo

        ! +-------------------------------------------------------------------------

        ! +--Vertical Diffusion of Moisture
        ! +  ==============================

        ! +--Tridiag. Matrix Coeff. : qvDY
        ! +  -----------------------------

        ! +--Diagonal A
        ! +  ~~~~~~~~~~
        k = mz
        do i = ip11, mx1
            Kv__SL(i, j) = 0.
            WKxyz1(i, j, k) = Kv__SL(i, j)
            ! +--Diagonal B
            ! +  ~~~~~~~~~~
            WKxyz2(i, j, k) = 1.00 - WKxyz3(i, j, k)
        enddo

        ! +--UBC of the Tridiagonal System - qvDY
        ! +  ------------------------------------
#ifdef iso
        ! todo : check if Riso is more pertinent that qiso
        ! initialize isotopic ratios
        do k = 1, mz
            do i = ip11, mx1
                if(qvDY(i, j, k) > negligible) then
                    do wiso = 1, niso
                        Riso(wiso, i, j, k) = qvDY_iso(wiso, i, j, k) / qvDY(i, j, k)
                    enddo
                else
                    do wiso = 1, niso
                        Riso(wiso, i, j, k) = Rdefault(wiso)
                    enddo
                endif
            enddo
        enddo
#endif

        kp = kp1(1)
        do i = ip11, mx1
            WKxyz4(i, j, 1) = WKxyz1(i, j, 1) * ab * (qvDY(i, j, 1) - qvDY(i, j, kp))
            if(track_water_turabl) then
                ! Compute the contribution of qv only, without surface sublimation
                WTxyz4(i, j, 1) = WKxyz4(i, j, 1)
            end if
#ifdef iso
            do wiso = 1, niso
                WKxyz4_iso(wiso, i, j, 1) = WKxyz1(i, j, 1) * ab * (Riso(wiso, i, j, 1) - Riso(wiso, i, j, kp))
            enddo
#endif
        enddo

        ! +--Second Member of the Tridiagonal System - qvDY
        ! +  ----------------------------------------------

        do k = kp1(1), mmz1
            kp = kp1(k)
            km = km1(k)
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz4(i, j, k) = WKxyz1(i, j, k) * ab * (qvDY(i, j, k) - qvDY(i, j, kp)) &
                                  - WKxyz3(i, j, k) * ab * (qvDY(i, j, km) - qvDY(i, j, k))
                if(track_water_turabl) then
                    ! Compute the contribution of qv only, without surface sublimation
                    WTxyz4(i, j, k) = WKxyz4(i, j, k)
                end if
#ifdef iso
                do wiso = 1, niso
                    WKxyz4_iso(wiso, i, j, k) = WKxyz1(i, j, k) * ab &
                                                * (Riso(wiso, i, j, k) - Riso(wiso, i, j, kp)) &
                                                - WKxyz3(i, j, k) * ab * (Riso(wiso, i, j, km) - Riso(wiso, i, j, k))
                enddo
#endif
            enddo
            ! end do
        enddo

        ! +--SBC of the Tridiagonal System - qvDY
        ! +  ------------------------------------

        ! do j=jp11,my1
        do i = ip11, mx1

            ! commented because not used
            !CAaiso ! +--Implicit Surface Scheme
            !CAaiso ! +  ~~~~~~~~~~~~~~~~~~~~~~~
            !CAaiso qvap = qvDY(i, j, mz) - SLuqs(i, j) / aeCdSL(i, j)
            !CAaiso
            !CAaiso ! uqstar is replaced by aeCdSL*(qvDY-qvapSL), set := 0
            !CAaiso uqstar = 0. ! explicit

            ! +--Explicit Surface Scheme
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~
            qvap = qvapSL(i, j)
            ! uqstar = SLuqs is the Moisture Turbulent Flux
            ! uqstar  * rhAir * dt__SV -> evaporation flux, in kg m-2 (= kg kg-1 m s-1 kg m-3 s)
            ! uqstar  * rhAir * dt__SV * g / dp -> g/dp = rho/dz
            uqstar = SLuqs(i, j)

            ! partly explicit
            WKxyz4(i, j, mz) = WKxyz1(i, j, mz) * (ab * qvDY(i, j, mz) - qvap / beta) &
                               - WKxyz3(i, j, mz) * ab * (qvDY(i, j, mmz1) - qvDY(i, j, mz)) &
                               ! u*q* all explicit
                               - gravit * dt_Loc * rolvDY(i, j, mz) * uqstar / (pstDY(i, j) * dsigm1(mz))
            if(track_water_turabl) then
                ! Compute the contribution of qv only, without surface sublimation
                WTxyz4(i, j, mz) = WKxyz1(i, j, mz) * (ab * qvDY(i, j, mz) - qvap / beta) &
                        - WKxyz3(i, j, mz) * ab * (qvDY(i, j, mmz1) - qvDY(i, j, mz))
            end if
#ifdef iso
            do wiso = 1, niso
                if(qvap > negligible) then
                    Riso_qvap = qvapSL_iso(wiso, i, j) / qvap
                else
                    Riso_qvap = Rdefault(wiso)
                endif
                if(uqstar > negligible) then
                    Riso_uqstar = SLuqs_iso(wiso, i, j) / uqstar
                else
                    Riso_uqstar = Rdefault(wiso)
                endif
                WKxyz4_iso(wiso, i, j, mz) = WKxyz1(i, j, mz) * (ab * Riso(wiso, i, j, mz) - Riso_qvap / beta) &
                                             - WKxyz3(i, j, mz) * ab * (Riso(wiso, i, j, mmz1) - Riso(wiso, i, j, mz)) &
                                             ! u*q* all explicit g dt rho uq* / dp | dt rho uq* = [kg m-2] | dt rho uq* g / dp = [kg m-2] / [rho dz]
                                             ! convertion of uqstar into qsurf in kg/kg
                                             ! - gravit * dt_Loc * rolvDY(i, j, mz) * uqstar / (pstDY(i, j) * dsigm1(mz))
                                             - Riso_uqstar
            enddo
#endif
        enddo
        ! end do

        ! +--Tridiagonal Matrix Inversion - qvDY
        ! +  -----------------------------------

        k1_tua = 1
        do k = k1_tua, mz
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz4(i, j, k) = WKxyz4(i, j, k) + qvDY(i, j, k)
                if(track_water_turabl) then
                    ! Compute the contribution of qv only, without surface sublimation
                    WTxyz4(i, j, k) = WTxyz4(i, j, k) + qvDY(i, j, k)
                end if
#ifdef iso
                do wiso = 1, niso
                    WKxyz4_iso(wiso, i, j, k) = WKxyz4_iso(wiso, i, j, k) + Riso(wiso, i, j, k)
                enddo
#endif
            enddo
            ! end do
        enddo

        k1_tua = 1
        k2_tua = mz

        ! +  ************
        call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
        ! +  ************
        if(track_water_turabl) then
            ! Compute the contribution of qv only, without surface sublimation
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WTxyz4, WTxyz7)
        end if
#ifdef iso
        do wiso = 1, niso
            do i = 1, mx
                varin_iso(i, j, :) = WKxyz4_iso(wiso, i, j, :)
            enddo
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, varin_iso, varout_iso)
            do i = 1, mx
                WKxyz7_iso(wiso, i, j, :) = varout_iso(i, j, :)
            enddo
        enddo
#endif
        do k = 1, mz
            ! do j=jp11,my1
            do i = ip11, mx1
                if(track_water_turabl) then
                    ! delta_qv_mix = delta_qv without sublimation
                    delta_qv(i, j, k, j_turabl_mix) = delta_qv(i, j, k, j_turabl_mix) + WTxyz7(i, j, k) - qvDY(i, j, k)
                    ! delta_qv_tot = WKxyz7(i, j, k) - qvDY(i, j, k)
                    ! subl = delta_qv_tot - delta_qv_mix
                    delta_qv(i, j, k, j_turabl_sbl) = delta_qv(i, j, k, j_turabl_sbl) + WKxyz7(i, j, k) - WTxyz7(i, j, k)
                end if
#ifdef iso
                do wiso = 1, niso
                    qvDY_iso(wiso, i, j, k) = WKxyz7_iso(wiso, i, j, k) * qvDY(i, j, k)
                enddo
#endif
                qvDY(i, j, k) = WKxyz7(i, j, k)
            enddo
            ! end do
        enddo

#ifdef TC
        ! +--Vertical Diffusion of gazeous Tracers
        ! +  =====================================
        if(dt_ODE == dtDiff) then
            ! +--Second Member of the Tridiagonal System - qxTC
            ! +  ----------------------------------------------
            ! CAUTION: defines nterr as the Nb of terregenous aerosols (usually 0 .OR. 1)
            do n = nterr + 1, ntrac
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz1(i, j, 1) = 0.0
                        WKxyz2(i, j, 1) = 1.0
                        WKxyz4(i, j, 1) = qxTC(i, j, 1, n)
                    enddo
                enddo
                do k = kp1(1), mmz1
                    kp = kp1(k)
                    km = km1(k)
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = &
                                WKxyz1(i, j, k) * ab * (qxTC(i, j, k, n) - qxTC(i, j, kp, n)) &
                                - WKxyz3(i, j, k) * ab * (qxTC(i, j, km, n) - qxTC(i, j, k, n))
                        enddo
                    enddo
                enddo
                if(.not. BloMod) then
                    do j = jp11, my1
                        do i = ip11, mx1
                            uqTC(i, j, n) = -cdhSL(i, j, 1) * SLuusl(i, j, 1) &
                                            * (qsTC(i, j, n) - qxTC(i, j, mz, n))
                        enddo
                    enddo
                endif
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz4(i, j, mz) = WKxyz1(i, j, mz) &
                                           * (ab * qxTC(i, j, mz, n) - qsTC(i, j, n) / beta) &
                                           - WKxyz3(i, j, mz) * ab * (qxTC(i, j, mmz1, n) - qxTC(i, j, mz, n)) &
                                           + (gravit * dt_Loc &
                                              * rolvDY(i, j, mz) / (pstDY(i, j) * (sigmid(mz) - 1.0d+0))) &
                                           * uqTC(i, j, n)
                    enddo
                enddo
                ! +--Tridiagonal Matrix Inversion - qxTC
                ! +  -----------------------------------
                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = WKxyz4(i, j, k) + qxTC(i, j, k, n)
                        enddo
                    enddo
                enddo
                k1_tua = 1
                k2_tua = mz
                ! +  ************
                call MARgz_2mx1y1(k1_tua, k2_tua)
                ! +  ************
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qxTC(i, j, k, n) = WKxyz7(i, j, k)
                        enddo
                    enddo
                enddo
            enddo
        endif
#endif

#ifdef EW
        ! +--Atmospheric Water Budget
        ! +  ========================
        do j = jp11, my1
            do i = ip11, mx1
                wat0EW(i, j) = 0.0
                do k = 1, mz
                    wat0EW(i, j) = wat0EW(i, j) &
                                   + (qvDY(i, j, k) &
                                      + qwHY(i, j, k) + qrHY(i, j, k) &
                                      + qiHY(i, j, k) + qsHY(i, j, k)) * dsigm1(k)
                enddo
                wat0EW(i, j) = wat0EW(i, j) * pstDY(i, j) * grvinv
            enddo
        enddo
#endif

        ! +-------------------------------------------------------------------------

        ! +--Vertical Diffusion of Hydrometeors and non gazeous Tracers
        ! +  ==========================================================

        ! +--Parameters for the Numerical Scheme of Vertical Turbulent Transport
        ! +  -------------------------------------------------------------------

        ! alpha : Expliciteness = 0 (positive definite)
        alpha = 0.00
        ! beta : Impliciteness
        beta = 1.00 - alpha
        ab = alpha / beta
        !CAa todo : add qiHY, qsHY, qwHY, qrHY
        ! +--Tridiagonal Matrix Coefficients:  qiHY, ccniHY, qwHY, qrHY
        ! +  (Turbulent Diffusion Coefficient X 3:  Bintanja, 2000, BLM) --+
        ! +  -----------------------------------------------------------   V

        ! +--Diagonal A
        ! +  ~~~~~~~~~~
        do k = mz, 1, -1
            kp = kp1(k)

            do i = ip11, mx1
                WKxyz8(i, j, k) = TUkvh(i, j, k) * r_Turb
            enddo

            do i = ip11, mx1
                WKxyz1(i, j, k) = -gravi2 * beta * romiDY(i, j, k) * WKxyz8(i, j, k) &
                                  * rolvDY(i, j, k) / (pstDY2(i, j) * dsigm1(k) * dsig_1(k))
            enddo

            ! +--Diagonal C
            ! +  ~~~~~~~~~~
            do i = ip11, mx1
                WKxyz3(i, j, kp) = WKxyz1(i, j, k) * dsigm1(k) / dsigm1(kp) &
                                   / rolvDY(i, j, k) * rolvDY(i, j, kp)
            enddo

        enddo

        ! +--A, B, C
        ! +  ~~~~~~~
        do k = 1, mz
            do i = ip11, mx1
                WKxyz1(i, j, k) = WKxyz1(i, j, k) * dt_Loc
                WKxyz3(i, j, k) = WKxyz3(i, j, k) * dt_Loc
                WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
            enddo
        enddo

        ! +--Vertical B.C.
        ! +  ~~~~~~~~~~~~~
        ! do j=jp11,my1
        do i = ip11, mx1
            WKxyz3(i, j, 1) = 0.0
            WKxyz2(i, j, 1) = 1.0 - WKxyz1(i, j, 1)
        enddo
        ! end do

        ! +-------------------------------------------------------------------------

        ! +--BEGIN Cloud Microphysics (qiHY, ccniHY, qwHY, qrHY)
        ! +  ===================================================

        if(micphy) then
            ! +--Vertical Diffusion of Ice Crystals
            ! +  ==================================

            ! +--Second Member of the Tridiagonal System - qiHY
            ! +  ----------------------------------------------

            do i = ip11, mx1
                WKxyz1(i, j, 1) = 0.0
                WKxyz2(i, j, 1) = 1.0
                WKxyz4(i, j, 1) = qiHY(i, j, 1)
            enddo

            do k = kp1(1), mmz1
                kp = kp1(k)
                km = km1(k)
                do i = ip11, mx1
                    WKxyz4(i, j, k) = &
                        WKxyz1(i, j, k) * ab * (qiHY(i, j, k) - qiHY(i, j, kp)) &
                        - WKxyz3(i, j, k) * ab * (qiHY(i, j, km) - qiHY(i, j, k))
                enddo
            enddo

            do i = ip11, mx1
                WKxyz4(i, j, mz) = &
                    WKxyz1(i, j, mz) * ab * (qiHY(i, j, mz) - zero) &
                    - WKxyz3(i, j, mz) * ab * (qiHY(i, j, mmz1) - qiHY(i, j, mz))
            enddo

            ! +--Tridiagonal Matrix Inversion - qiHY
            ! +  -----------------------------------

            do k = kp1(1), mz
                do i = ip11, mx1
                    WKxyz4(i, j, k) = WKxyz4(i, j, k) + qiHY(i, j, k)
                enddo
            enddo

            k1_tua = 1
            k2_tua = mz

            ! +  ************
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
            ! +  ************

            do k = 1, mz
                do i = ip11, mx1
                    qiHY(i, j, k) = WKxyz7(i, j, k)
                enddo
            enddo

            ! +--Second Member of the Tridiagonal System - ccniHY
            ! +  ------------------------------------------------

            do i = ip11, mx1
                WKxyz1(i, j, 1) = 0.0
                WKxyz2(i, j, 1) = 1.0
                WKxyz4(i, j, 1) = ccniHY(i, j, 1)
            enddo

            do k = kp1(1), mmz1
                kp = kp1(k)
                km = km1(k)
                do i = ip11, mx1
                    WKxyz4(i, j, k) = &
                        WKxyz1(i, j, k) * ab * (ccniHY(i, j, k) - ccniHY(i, j, kp)) &
                        - WKxyz3(i, j, k) * ab * (ccniHY(i, j, km) - ccniHY(i, j, k))
                enddo
            enddo

            do i = ip11, mx1
                WKxyz4(i, j, mz) = &
                    WKxyz1(i, j, mz) * ab * (ccniHY(i, j, mz) - zero) &
                    - WKxyz3(i, j, mz) * ab * (ccniHY(i, j, mmz1) - ccniHY(i, j, mz))
            enddo

            ! +--Tridiagonal Matrix Inversion - ccniHY
            ! +  -------------------------------------

            do k = kp1(1), mz
                do i = ip11, mx1
                    WKxyz4(i, j, k) = WKxyz4(i, j, k) + ccniHY(i, j, k)
                enddo
            enddo

            k1_tua = 1
            k2_tua = mz

            ! +  ************
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
            ! +  ************

            do k = 1, mz
                do i = ip11, mx1
                    ccniHY(i, j, k) = WKxyz7(i, j, k)
                enddo
            enddo

            ! +--Precipitation
            ! +  -------------

            do i = ip11, mx1
                dd_sno = dt_Loc * rolvDY(i, j, mz) &
                         * WKxyz8(i, j, mz) * qiHY(i, j, mz) &
                         / (gplvDY(i, j, mz) * grvinv - sh(i, j))
                crysHY(i, j) = crysHY(i, j) + dd_sno
                snohSL(i, j) = snohSL(i, j) + dd_sno
            enddo

            ! +-------------------------------------------------------------------------

            ! +--Vertical Diffusion of Cloud Droplets
            ! +  ====================================

            ! +--Second Member of the Tridiagonal System - qwHY
            ! +  ----------------------------------------------

            do i = ip11, mx1
                WKxyz1(i, j, 1) = 0.0
                WKxyz2(i, j, 1) = 1.0
                WKxyz4(i, j, 1) = qwHY(i, j, 1)
            enddo

            do k = kp1(1), mmz1
                km = km1(k)
                do i = ip11, mx1
                    WKxyz4(i, j, k) = &
                        WKxyz1(i, j, k) * ab * (qwHY(i, j, k) - qwHY(i, j, kp)) &
                        - WKxyz3(i, j, k) * ab * (qwHY(i, j, km) - qwHY(i, j, k))
                enddo
            enddo

            do i = ip11, mx1
                WKxyz4(i, j, mz) = &
                    WKxyz1(i, j, mz) * ab * (qwHY(i, j, mz) - zero) &
                    - WKxyz3(i, j, mz) * ab * (qwHY(i, j, mmz1) - qwHY(i, j, mz))
            enddo

            ! +--Tridiagonal Matrix Inversion - qwHY
            ! +  -----------------------------------

            do k = kp1(1), mz
                do i = ip11, mx1
                    WKxyz4(i, j, k) = WKxyz4(i, j, k) + qwHY(i, j, k)
                enddo
            enddo

            k1_tua = 1
            k2_tua = mz

            ! +  ************
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
            ! +  ************

            do k = 1, mz
                do i = ip11, mx1
                    qwHY(i, j, k) = WKxyz7(i, j, k)
                enddo
            enddo

            ! +--Precipitation
            ! +  -------------
            do i = ip11, mx1
                ratio_temp = (tairDY(i, j, mz - 1) + tairDY(i, j, mz - 2) &
                              + tairDY(i, j, mz - 3) + tairDY(i, j, mz - 4)) / 4.

                ratio_prec = dt_Loc * rolvDY(i, j, mz) &
                             * WKxyz8(i, j, mz) * qwHY(i, j, mz) &
                             / (gplvDY(i, j, mz) * grvinv - sh(i, j))

                ratio_rfsf = max(0., min(1.,(ratio_temp - rain_snow_limit) / 2.))

                rainHY(i, j) = rainHY(i, j) + ratio_prec * ratio_rfsf
                snowHY(i, j) = snowHY(i, j) + ratio_prec * (1.-ratio_rfsf)
            enddo

            ! +-------------------------------------------------------------------------

            ! +--Vertical Diffusion of Rain Drops
            ! +  ================================

            ! +--Second Member of the Tridiagonal System - qrHY
            ! +  ----------------------------------------------

            do i = ip11, mx1
                WKxyz1(i, j, 1) = 0.0
                WKxyz2(i, j, 1) = 1.0
                WKxyz4(i, j, 1) = qrHY(i, j, 1)
            enddo

            do k = kp1(1), mmz1
                kp = kp1(k)
                km = km1(k)
                do i = ip11, mx1
                    WKxyz4(i, j, k) = &
                        WKxyz1(i, j, k) * ab * (qrHY(i, j, k) - qrHY(i, j, kp)) &
                        - WKxyz3(i, j, k) * ab * (qrHY(i, j, km) - qrHY(i, j, k))
                enddo
            enddo

            do i = ip11, mx1
                WKxyz4(i, j, mz) = &
                    WKxyz1(i, j, mz) * ab * (qrHY(i, j, mz) - zero) &
                    - WKxyz3(i, j, mz) * ab * (qrHY(i, j, mmz1) - qrHY(i, j, mz))
            enddo

            ! +--Tridiagonal Matrix Inversion - qrHY
            ! +  -----------------------------------

            do k = kp1(1), mz
                do i = ip11, mx1
                    WKxyz4(i, j, k) = WKxyz4(i, j, k) + qrHY(i, j, k)
                enddo
            enddo

            k1_tua = 1
            k2_tua = mz

            ! +  ************
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
            ! +  ************

            do k = 1, mz
                do i = ip11, mx1
                    qrHY(i, j, k) = WKxyz7(i, j, k)
                enddo
            enddo

            ! +--Precipitation
            ! +  -------------

            do i = ip11, mx1
                ratio_temp = (tairDY(i, j, mz - 1) + tairDY(i, j, mz - 2) &
                              + tairDY(i, j, mz - 3) + tairDY(i, j, mz - 4)) / 4.
                ratio_prec = dt_Loc * rolvDY(i, j, mz) &
                             * WKxyz8(i, j, mz) * qrHY(i, j, mz) &
                             / (gplvDY(i, j, mz) * grvinv - sh(i, j))

                ratio_rfsf = max(0., min(1.,(ratio_temp - rain_snow_limit) / 2.))

                rainHY(i, j) = rainHY(i, j) + ratio_prec * ratio_rfsf
                snowHY(i, j) = snowHY(i, j) + ratio_prec * (1.-ratio_rfsf)
            enddo

            ! +--END   Cloud Microphysics (qiHY, ccniHY, qwHY, qrHY)
            ! +  ===================================================

        endif

        ! +-------------------------------------------------------------------------

        ! +--Vertical Diffusion of (Terrigeneous) Hydrometeors and Tracers
        ! +  =============================================================

        ! +--Tridiagonal Matrix Coefficients: Modifications for qsHY, qxTC
        ! +  -------------------------------------------------------------

        ! +--Diagonal A
        ! +  ~~~~~~~~~~
        k = mz
        do i = ip11, mx1
            WKxyz1(i, j, k) = 0.
        enddo

        ! +--A, B, C
        ! +  ~~~~~~~
        k = mz
        do i = ip11, mx1
            WKxyz1(i, j, k) = WKxyz1(i, j, k) * dt_Loc
            WKxyz2(i, j, k) = 1.0 - WKxyz3(i, j, k) - WKxyz1(i, j, k)
        enddo

        ! +-------------------------------------------------------------------------

        ! +--BEGIN Cloud Microphysics (qsHY)
        ! +  ===============================

        if(micphy) then

            ! +--Vertical Diffusion of Snow Flakes
            ! +  =================================

            ! +--Second Member of the Tridiagonal System - qsHY
            ! +  ----------------------------------------------

            do i = ip11, mx1
                WKxyz1(i, j, 1) = 0.0
                WKxyz2(i, j, 1) = 1.0
                WKxyz4(i, j, 1) = qsHY(i, j, 1)
            enddo

            do k = kp1(1), mmz1
                kp = kp1(k)
                km = km1(k)
                do i = ip11, mx1
                    WKxyz4(i, j, k) = &
                        WKxyz1(i, j, k) * ab * (qsHY(i, j, k) - qsHY(i, j, kp)) &
                        - WKxyz3(i, j, k) * ab * (qsHY(i, j, km) - qsHY(i, j, k))
                enddo
            enddo

            do i = ip11, mx1
                ussno = uss_HY(i, j)
                WKxyz4(i, j, mz) = &
                    WKxyz1(i, j, mz) &
                    * (ab * qsHY(i, j, mz) - qsrfHY(i, j) / beta) &
                    - WKxyz3(i, j, mz) * ab * (qsHY(i, j, mmz1) - qsHY(i, j, mz)) &
                    + (gravit * dt_Loc * rolvDY(i, j, mz) &
                       / (pstDY(i, j) * (sigmid(mz) - 1.0))) &
                    * ussno
            enddo

            ! +--Tridiagonal Matrix Inversion - qsHY
            ! +  -----------------------------------

            do k = kp1(1), mz
                do i = ip11, mx1
                    WKxyz4(i, j, k) = WKxyz4(i, j, k) + qsHY(i, j, k)
                enddo
            enddo

            k1_tua = 1
            k2_tua = mz

            ! +  ************
            call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
            ! +  ************

            do k = 1, mz
                do i = ip11, mx1
                    qsHY(i, j, k) = WKxyz7(i, j, k)
                enddo
            enddo

            ! +--END   Cloud Microphysics (qsHY)
            ! +  ===============================

        endif

        ! +-------------------------------------------------------------------------

#ifdef TC
        ! +--Vertical Diffusion of non gazeous Tracers
        ! +  =========================================
        if(dt_ODE == dtDiff .and. nterr > 0) then
            ! +--Second Member of the Tridiagonal System - qxTC
            ! +  ----------------------------------------------
            do n = 1, nterr
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz1(i, j, 1) = 0.0
                        WKxyz2(i, j, 1) = 1.0
                        WKxyz4(i, j, 1) = qxTC(i, j, 1, n)
                    enddo
                enddo
                do k = kp1(1), mmz1
                    kp = kp1(k)
                    km = km1(k)
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = &
                                WKxyz1(i, j, k) * ab * (qxTC(i, j, k, n) - qxTC(i, j, kp, n)) &
                                - WKxyz3(i, j, k) * ab * (qxTC(i, j, km, n) - qxTC(i, j, k, n))
                        enddo
                    enddo
                enddo
                if(.not. BloMod) then
                    do j = jp11, my1
                        do i = ip11, mx1
                            uqTC(i, j, n) = -cdhSL(i, j, 1) * SLuusl(i, j, 1) &
                                            * (qsTC(i, j, n) - qxTC(i, j, mz, n))
                        enddo
                    enddo
                endif
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz4(i, j, mz) = WKxyz1(i, j, mz) &
                                           * (ab * qxTC(i, j, mz, n) - qsTC(i, j, n) / beta) &
                                           - WKxyz3(i, j, mz) * ab * (qxTC(i, j, mmz1, n) - qxTC(i, j, mz, n)) &
                                           + (gravit * dt_Loc &
                                              * rolvDY(i, j, mz) / (pstDY(i, j) * (sigmid(mz) - 1.0d+0))) &
                                           * uqTC(i, j, n)
                    enddo
                enddo
                ! +--Tridiagonal Matrix Inversion - qxTC
                ! +  -----------------------------------
                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = WKxyz4(i, j, k) + qxTC(i, j, k, n)
                        enddo
                    enddo
                enddo
                k1_tua = 1
                k2_tua = mz
                ! +  ************
                call MARgz_2mx1y1_mp(k1_tua, k2_tua, j, WKxyz1, WKxyz2, WKxyz3, WKxyz4, WKxyz7)
                ! +  ************
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qxTC(i, j, k, n) = WKxyz7(i, j, k)
                        enddo
                    enddo
                enddo
            enddo
        endif
#endif

#ifdef EW
        ! +--Atmospheric Water Budget
        ! +  ========================
        do i = ip11, mx1
            do j = jp11, my1
                wat1EW(i, j) = 0.00
                do k = 1, mz
                    wat1EW(i, j) = wat1EW(i, j) &
                                   + (qvDY(i, j, k) &
                                      + qwHY(i, j, k) + qrHY(i, j, k) &
                                      + qiHY(i, j, k) + qsHY(i, j, k)) * dsigm1(k)
                enddo
                wat1EW(i, j) = wat1EW(i, j) * pstDY(i, j) * grvinv
                watfEW(i, j) = -(uss_HY(i, j) + SLuqs(i, j)) &
                               * dt_Loc * rolvDY(i, j, mz)
            enddo
        enddo
        ! +--Atmospheric Water Budget: Output
        ! +  ================================
        ! +
        waterb = wat1EW(imez, jmez) &
                 - wat0EW(imez, jmez) - watfEW(imez, jmez)
        write(6, 606) jdaMAR, jhaMAR, jmmMAR, &
            1.d3 * wat0EW(imez, jmez), 1.d3 * wat1EW(imez, jmez), &
            1.d3 * watfEW(imez, jmez), &
            1.d3 * waterb
606     format(3i3, '  Before vDif:      ', 12x, '  W0 = ', f9.6, &
               /, 9x, '  After  vDif:      ', 12x, '  W1 = ', f9.6, &
               '  W Flux =', f9.6, &
               '  Div(W) =', e9.3)
#endif

        ! +--Work Arrays Reset
        ! +  =================

        do k = 1, mz
            ! do j=jp11,my1
            do i = ip11, mx1
                WKxyz1(i, j, k) = 0.00
                WKxyz2(i, j, k) = 0.00
                WKxyz3(i, j, k) = 0.00
                WKxyz4(i, j, k) = 0.00
                WKxyz5(i, j, k) = 0.00
                WKxyz6(i, j, k) = 0.00
                WKxyz7(i, j, k) = 0.00
            enddo
            ! end do
        enddo

        ! do j=jp11,my1
        do i = ip11, mx1
            WKxy1(i, j) = 0.00
        enddo
    enddo
    !$OMP END PARALLEL DO

    return
endsubroutine TURabl
