subroutine StereoSouth(E, N, GEddxx, lon, lat, GElat0)
    ! +----------------------------------------------------------------------+
    ! |  Compute the lon, lat from Polar   Stereographic Projection          |
    ! |  Written by Cecile Agosta                                 23-12-21   |
    ! |  EPSG Polar Stereographic transformation Variant B                   |
    ! |  (http://www.epsg.org/guides/docs/G7-2.pdf)                          |
    ! |  Equivalent to EPSG 3031 (WGS-84 ellipsoid)  for SH                  |
    ! |  Equivalent to EPSG 3413 (WGS-84 ellipsoid)  for NH                  |
    ! +----------------------------------------------------------------------+
    ! |                                                                      |
    ! | INPUT :  E      : Stereo coordinate on the East  (X, km)             |
    ! | ^^^^^^^  N      : Stereo coordinate on the North (Y, km)             |
    ! |          GEddxx : Longitude of X axis (=GEddxx, 90 = East, clockwise)|
    ! |         [lat true = 71]                                              |
    ! |                                                                      |
    ! | OUTPUT : lon    : longitude (deg)                                    |
    ! | ^^^^^^^  lat    : latitude  (deg)                                    |
    ! |                                                                      |
    ! +----------------------------------------------------------------------+

    use mardim

    implicit none

    ! +-- General Variables
    ! +   -----------------
    real, INTENT(in) :: E, N, GEddxx
    real, INTENT(out) :: lon, lat

    ! +-- Local Variables
    ! +   ---------------
    real ddista

    ! +-- Constants
    ! +   ---------
    real aa, ex, pi, degrad, latF, FE, FN, tF, mF, k0, t, rho, khi, lon0
    real trulat, GElat0

    ! aa : aa (km) = demi grand axe
    aa = 6378.1370
    ! ex : excentricity WGS-84 : 0.081 819 190 842 622 0.081 819 190 842 621
    ex = 0.081819190842621

    if(sign(1., GElat0) <= 0) then
        ! trulat : Latitude of standard parallel, 71S for ESPG 3031
        trulat = -71.
    else
        ! trulat : Latitude of standard parallel, 70N for EPSG 3413
        trulat = 70.
    endif

    pi = 4.*atan(1.)
    degrad = pi / 180.

    latF = trulat * degrad
    lon0 = (GEddxx - 90.) * degrad

    ! FE : False Easting
    FE = 0.
    ! FN : False Northing
    FN = 0.

    ! +-  Polar Stereographic Projection
    ! +   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if(sign(1., GElat0) <= 0) then
        tF = tan(pi / 4 + latF / 2) / &
             ((1 + ex * sin(latF)) / (1 - ex * sin(latF)))**(ex / 2)
    else
        tF = tan(pi / 4 - latF / 2) * &
             ((1 + ex * sin(latF)) / (1 - ex * sin(latF)))**(ex / 2)
    endif

    mF = cos(latF) / (1 - ex**2 * sin(latF)**2)**0.5
    k0 = mF * ((1 + ex)**(1 + ex) * (1 - ex)**(1 - ex))**0.5 / (2 * tF)

    rho = ((E - FE)**2 + (N - FN)**2)**0.5
    t = rho * ((1 + ex)**(1 + ex) * (1 - ex)**(1 - ex))**0.5 / (2 * aa * k0)

    if(sign(1., GElat0) <= 0) then
        khi = 2 * atan(t) - pi / 2
    else
        khi = pi / 2 - 2 * atan(t)
    endif

    lat = khi &
          + (ex**2 / 2 + 5 * ex**4 / 24 + ex**6 / 12 + 13 * ex**8 / 360) &
          * sin(2 * khi) &
          + (7 * ex**4 / 48 + 29 * ex**6 / 240 + 811 * ex**8 / 11520) &
          * sin(4 * khi) &
          + (7 * ex**6 / 120 + 81 * ex**8 / 1120) &
          * sin(6 * khi) &
          + (4279 * ex**8 / 161280) &
          * sin(8 * khi)

    if(E - FE == 0. .and. N - FN == 0) then
        lon = lon0 + pi / 2.
    else if(E - FE == 0. .and. N - FN >= 0) then
        if(sign(1., GElat0) <= 0) then
            lon = lon0
        else
            lon = lon0 - pi
        endif
    else if(E - FE == 0. .and. N - FN <= 0) then
        if(sign(1., GElat0) <= 0) then
            lon = lon0 - pi
        else
            lon = lon0
        endif
    else

        if(sign(1., GElat0) <= 0) then
            lon = lon0 + atan2(E - FE, N - FN)
        else
            lon = lon0 + atan2(E - FE, FN - N)
        endif

    endif

    lat = lat / degrad
    lon = lon / degrad
    if(lon > 180.) then
        lon = lon - 360.
    else if(lon < -180.) then
        lon = lon + 360.
    endif

    return
endsubroutine StereoSouth

subroutine StereoSouth_inverse(lon, lat, lonE, E, N)
    ! +----------------------------------------------------------------------+
    ! |  Compute Polar   Stereographic Projection from lon,lat coordinates   |
    ! |  Written by Cecile Agosta                                 02-02-21   |
    ! |  EPSG Polar Stereographic transformation Variant B                   |
    ! |  (http://www.epsg.org/guides/docs/G7-2.pdf)                          |
    ! |  Equivalent to EPSG 3031 (WGS-84 ellipsoid)  for SH                  |
    ! |  Equivalent to EPSG 3413 (WGS-84 ellipsoid)  for NH                  |
    ! +----------------------------------------------------------------------+
    ! |                                                                      |
    ! | INPUT : lon     : Longitude (deg)                                    |
    ! | ^^^^^^^ lat     : Latitude  (deg)                                    |
    ! |         lon0    : Longitude of X axis (90 = East, clockwise)         |
    ! |        [lat true = 71 S/71N]                                         |
    ! |                                                                      |
    ! | OUTPUT : E      : Stereo coordinate on the East  (X, km)             |
    ! | ^^^^^^^^ N      : Stereo coordinate on the North (Y, km)             |
    ! |                                                                      |
    ! +----------------------------------------------------------------------+

    use mardim

    implicit none

    ! +-- General Variables
    ! +   -----------------
    real, intent(in) :: lon, lat, lonE
    real, intent(out) :: E, N

    ! +-- Local Variables
    ! +   ---------------
    real costru, ddista

    ! +-- Constants
    ! +   ---------
    real aa, ex, pi, degrad, latF, FE, FN, tF, mF, k0, t, rho, lonrad, latrad
    real lon0, trulat, GElat0

    GElat0 = lat
    ! aa : aa (km) = demi grand axe
    aa = 6378.1370
    ! ex : excentricity WGS-84 : 0.081 819 190 842 622 0.081 819 190 842 621
    ex = 0.081819190842621

    if(sign(1., GElat0) <= 0) then
        ! trulat : Latitude of standard parallel, 71S for ESPG 3031
        trulat = -71.
    else
        ! trulat : Latitude of standard parallel, 70N for EPSG 3413
        trulat = 70.
    endif

    pi = 4.*atan(1.)
    degrad = pi / 180.

    latF = trulat * degrad
    lon0 = (lonE - 90.) * degrad
    lonrad = lon * degrad
    latrad = lat * degrad

    ! FE : False Easting
    FE = 0.
    ! FN : False Northing
    FN = 0.

    ! +
    ! +- Polar Stereographic Projection
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    ! +
    if(sign(1., GElat0) <= 0) then
        tF = tan(pi / 4 + latF / 2) / &
             ((1 + ex * sin(latF)) / (1 - ex * sin(latF)))**(ex / 2)
    else
        tF = tan(pi / 4 - latF / 2) * &
             ((1 + ex * sin(latF)) / (1 - ex * sin(latF)))**(ex / 2)
    endif

    mF = cos(latF) / (1 - ex**2 * sin(latF)**2)**0.5
    k0 = mF * ((1 + ex)**(1 + ex) * (1 - ex)**(1 - ex))**0.5 / (2 * tF)

    if(sign(1., GElat0) <= 0) then
        t = tan(pi / 4 + latrad / 2) / &
            ((1 + ex * sin(latrad)) / (1 - ex * sin(latrad)))**(ex / 2)
    else
        t = tan(pi / 4 - latrad / 2) * &
            ((1 + ex * sin(latrad)) / (1 - ex * sin(latrad)))**(ex / 2)
    endif

    rho = 2 * aa * k0 * t / ((1 + ex)**(1 + ex) * (1 - ex)**(1 - ex))**0.5

    E = FE + rho * sin(lonrad - lon0)

    if(sign(1., GElat0) <= 0) then
        N = FN + rho * cos(lonrad - lon0)
    else
        N = FN - rho * cos(lonrad - lon0)
    endif

    return
endsubroutine StereoSouth_inverse
