#include "MAR_pp.def"
subroutine SISVATeSBL
    !--------------------------------------------------------------------------+
    !   MAR          SISVATeSBL                           Tue 20-JAN-2023  MAR |
    !     subroutine SISVATeSBL generates Surface Boundary Layers Properties   |
    !     (and computes usthSV since 24-sep 2018)                              |
    !--------------------------------------------------------------------------+
    !                                                                          |
    !     PARAMETERS:  klonv: Total Number of columns                          |
    !     ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    !                       X       Number of Mosaic Cell per grid box         |
    !                                                                          |
    !     INPUT:   za__SV   : Surface Boundary Layer (SBL) Height          [m] |
    !     ^^^^^    VV__SV   :(SBL Top)   Wind Velocity                   [m/s] |
    !              TaT_SV   : SBL Top    Temperature                       [K] |
    !              ExnrSV   : Exner      Potential                         [-] |
    !              qsnoSV   : SBL Mean   Snow      Content             [kg/kg] |
    !              uqs_SV   : Specific   Humidity  Turbulent Flux        [m/s] |
    !              usthSV   : Threshd. friction velocity for snow erosion[m/s] |
    !              Z0m_SV   : Momentum     Roughness Length                [m] |
    !              Z0h_SV   : Heat         Roughness Length                [m] |
    !              Tsrfsv   : Surface    Temperature                       [K] |
    !              sqrCm0   : Contribution of Z0m to Neutral Drag Coefficient  |
    !              sqrCh0   : Contribution of Z0h to Neutral Drag Coefficient  |
    !                                                                          |
    !     INPUT /  LMO_SV   : Monin-Obukhov       Scale                    [m] |
    !     OUTPUT:  us__SV   : Friction  Velocity                         [m/s] |
    !     ^^^^^^   uts_SV   : Temperature         Turbulent Flux       [K.m/s] |
    !              uss_SV   : Blowing Snow        Turbulent Flux         [m/s] |
    !                                                                          |
    !     OUTPUT:  hSalSV   : Saltating Layer Height                       [m] |
    !     ^^^^^^   qSalSV   : Saltating Snow  Concentration            [kg/kg] |
    !              ram_sv   : Aerodynamic Resistance for Momentum        [s/m] |
    !              rah_sv   : Aerodynamic Resistance for Heat            [s/m] |
    !                                                                          |
    !   # OPTIONS: #BS: Blowing Snow turbulent Fluxes are computed             |
    !   # ^^^^^^^  #ss: Additional Output                                      |
    !                                                                          |
    !--------------------------------------------------------------------------+
    
    use marphy
    use mar_sv
    use mardsv
    use marxsv
    use marysv
    
    implicit none

#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
    common / SISVAT_EV / iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
#endif
    
    !  V,  dT(a-s)    Time Moving Averages
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer ntaver, nt
    parameter(ntaver = 4)
    real V__mem(klonv, ntaver)
    real VVmmem(klonv)
    common / SVeSBLmem / V__mem, VVmmem
    real T__mem(klonv, ntaver)
    real dTmmem(klonv)
    common / STeSBLmem / T__mem, dTmmem
    
    !$OMP threadprivate(/SVeSBLmem/,/STeSBLmem/)

#ifdef AM
    !  u*, u*T*, u*s* Time Moving Averages
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    real u__mem(klonv, ntaver)
    common / S_eSBLmem / u__mem
#endif
#ifdef AT
    real uT_mem(klonv, ntaver)
    common uT_mem
#endif
#ifdef AS
    real us_mem(klonv, ntaver)
    common us_mem
#endif
    
    !  Internal Variables
    !  ==================
    integer ikl, icount, isn
    real VVaSBL(klonv), VVa_OK         ! VVaSBL, VVa_OK : effective SBL wind speed
    real dTa_Ts(klonv)                 ! dTa_Ts : effective SBL Temperature diff.
    real Theta0                        ! Theta0 : Potential Reference Temperature
    real LMOmom(klonv)                 ! LMOmom : Monin-Obukhov Scale Momentum
    real LMOsgn                        ! LMOsgn : Monin-Obukhov Scale Sign
    real LMOabs                        ! LMOabs : Monin-Obukhov Scale Abs.Value
    real uustar, thstar, qqstar, ssstar, thstarv, thstars, thstara
    real zetam, zetah, zeta_S, zeta_A, zeta0m, zeta0h
    real psim_s, xpsimi, psim_i, psim_z
    real psis_s, psis_z, psis_0
    real psih_s, xpsihi, psih_i, psih_z
    real psim_0, psih_0
    real CDm(klonv)                    ! CDm : Drag Coefficient, Momentum
    real CDs(klonv), rCDs(klonv)       ! CDs, rCDs : Drag Coefficient, Blown **
    real CDh(klonv)                    ! CDh : Drag Coefficient, Scalar
    real dustar, u0star, uTstar, usstar
    real sss__F, sss__N, usuth0
    real zetMAX
    real coef_m, coef_h, stab_s
    real Richar(klonv)                 ! Richar : Richardson Number
#ifdef wr
    real W_pLMO                        ! W_pLMO : Pseudo Obukhov Length  (WRITE)
    real W_psim                        ! W_psim : Pseudo psim(z)         (WRITE)
#endif
#ifdef w1
    real W_NUs1                        ! W_NUs1 : Contrib to U* numerat.1(WRITE)
    real W_NUs2                        ! W_NUs2 : Contrib to U* numerat.2(WRITE)
    real W_NUs3                        ! W_NUs3 : Contrib to U* numerat.3(WRITE)
    real W_DUs1                        ! W_DUs1 : Contrib to U* denomin.1(WRITE)
    real W_DUs2                        ! W_DUs2 : Contrib to U* denomin.2(WRITE)
#endif
    real fac_Ri, vuzvun, Kz_vun
    character * 3 qsalt_param              ! qsalt_param : Switch for saltation flux param.
    character * 3 usth_param               ! usth_param : Switch for u*t param

#ifdef AE
    integer nit, iit
    real dusuth, signus
    real sss__K, sss__G
    real us_127, us_227, us_327, us_427, us_527
    real SblPom
    ! rCd10n : Square root of drag coefficient
    real rCd10n
    ! DendOK : Dendricity Switch
    real DendOK
    ! SaltOK : Saltation  Switch
    real SaltOK
    ! MeltOK : Saltation  Switch (Melting Snow)
    real MeltOK
    ! SnowOK : Pack Top   Switch
    real SnowOK
    ! SaltM1, SaltM2, SaltMo, SaltMx : Saltation  Parameters
    real SaltM1, SaltM2, SaltMo, SaltMx
    ! ShearX, ShearS : Arg. Max Shear Stress
    real ShearX, ShearS
    ! Por_BS : Snow Porosity
    real Por_BS
    ! Salt_us : New thresh.friction velocity u*t
    real Salt_us
    ! Fac_Mo, ArguSi, FacRho : Numerical factors for u*t
    real Fac_Mo, ArguSi, FacRho
    ! SaltSI : Snow Drift Index              !
    real SaltSI(klonv, 0:nsno)
    ! MIN_Mo : Minimum Mobility Fresh Fallen *
    real MIN_Mo
#endif
    
    !  Internal DATA
    !  =============
    
    ! Theta0 :  Potential Reference Temperature
    data Theta0/288.0/
#ifdef ZX
    ! zetMAX :  Strong Stability Limit
    data zetMAX/1.e6/
#endif
#ifdef zx
    ! zetMAX :  Strong Stability Limit (Mahalov et al. 2004, GRL  31 2004GL021055)
    data zetMAX/1.e0/
#endif
    ! zetMAX :  Strong Stability Limit
    data zetMAX/4.28/
    ! coef_m :  Stabil.Funct.for Moment.: unstab.coef.
    ! (King    et al. 1996, JGR 101(7) p.19121)
    data coef_m/20./
    ! coef_h :  Stabil.Funct.for Heat:    unstab.coef.
    data coef_h/15./
#ifdef AE
    ! Lower Boundary Height Parameter for Suspension
    ! Pommeroy, Gray and Landine 1993, J. Hydrology, 144(8) p.169
    ! SblPom :  us(is0,uth) recursivity: Nb Iterations
    data SblPom/1.27/
    ! nit :  saltation part. conc. from Pomeroy and Gray
    data nit/5/
    ! qsalt_param :  u*t from Gallee et al. 2001
    ! data qsalt_param/"bin"/ ! saltation part. conc. from Bintanja 2001 (p
    data qsalt_param/"pom"/
    ! data  usth_param/"lis"/  ! u*t from Liston et al. 2007
    data usth_param/"gal"/
    data SaltMx/-5.83e-2/
    ! +--Computation of threshold friction velocity for snow erosion
    ! +  ===========================================================
    rCd10n = 1. / 26.5 ! Vt / u*t = 26.5
    ! Budd et al. 1965, Antarct. Res. Series Fig.13
    ! ratio developped during assumed neutral conditions
    ! +--Snow Properties
    ! +  ~~~~~~~~~~~~~~~
    ! do isn = 1, nsno
    do ikl = 1, klonv
        isn = isnoSV(ikl)
        DendOK = max(zero, sign(unun, epsi - G1snSV(ikl, isn)))
        SaltOK = min(1, max(istdSV(2) - istoSV(ikl, isn), 0))
        MeltOK = (unun &
                - max(zero, sign(unun, TfSnow - epsi &
                        - TsisSV(ikl, isn)))) &
                * min(unun, DendOK &
                        + (1. - DendOK) &
                                ! 1.0 for 1mm
                                * sign(unun, G2snSV(ikl, isn) - 1.0))
        ! Snow Switch
        SnowOK = min(1, max(isnoSV(ikl) + 1 - isn, 0))
        G1snSV(ikl, isn) = SnowOK * G1snSV(ikl, isn) &
                + (1. - SnowOK) * min(G1snSV(ikl, isn), G1_dSV)
        G2snSV(ikl, isn) = SnowOK * G2snSV(ikl, isn) &
                + (1. - SnowOK) * min(G2snSV(ikl, isn), G1_dSV)
        SaltOK = min(unun, SaltOK + MeltOK) * SnowOK
        ! +--Mobility Index (Guyomarc'h & Merindol 1997, Ann.Glaciol.)
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        SaltM1 = -0.750e-2 * G1snSV(ikl, isn) &
                - 0.500e-2 * G2snSV(ikl, isn) + 0.500e00 !dendritic case
        ! +     CAUTION:  Guyomarc'h & Merindol Dendricity Sign is +
        ! +     ^^^^^^^^                    MAR Dendricity Sign is -
        SaltM2 = -0.833d-2 * G1snSV(ikl, isn) &
                - 0.583d-2 * G2snSV(ikl, isn) + 0.833d00 !non-dendritic case
        !       SaltMo   = (DendOK   * SaltM1 + (1.-DendOK) *     SaltM2       )
        SaltMo = 0.625 !SaltMo pour d=s=0.5
        
        ! weighting SaltMo with surface snow density (Vionnet et al. 2012)
        ! FacRho   = 1.25 - 0.0042 * ro__SV(ikl,isn)
        ! SaltMo   = 0.34 * SaltMo + 0.66 * FacRho !needed for polar snow
        MIN_Mo = 0.
        !       SaltMo   =  max(SaltMo,MIN_Mo)
        !       SaltMo   =  SaltOK   * SaltMo + (1.-SaltOK) * min(SaltMo,SaltMx)
#ifdef TUNE
        SaltMo = SaltOK * SaltMo - (1. - SaltOK) * 0.9500
#endif
        SaltMo = max(SaltMo, epsi - unun)
        ! +--Influence of Density on Threshold Shear Stress
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        Por_BS = 1. - 300. / ro_Ice
        ! SheaBS =  Arg(sqrt(shear = max shear stress in snow)):
        ! shear  =  3.420d00 * exp(-(Por_BS +Por_BS) /(unun -Por_BS))
        ! SheaBS :  see de Montmollin         (1978),
        !           These Univ. Sci. Medic. Grenoble, Fig. 1 p. 124
        ShearS = Por_BS / (1. - Por_BS)
        
        ! +--Snow Drift Index (Guyomarc'h & Merindol 1997, Ann.Glaciol.)
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ArguSi = -0.085 * us__SV(ikl) / rCd10n
        !V=u*/sqrt(CD) eqs 2 to 4 Gallee et al. 2001
        SaltSI(ikl, isn) = -2.868 * exp(ArguSi) + 1 + SaltMo
        ! +--Threshold Friction Velocity
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(ro__SV(ikl, isn) > 300.) then
            Por_BS = 1.000 - ro__SV(ikl, isn) / ro_Ice
        else
            Por_BS = 1.000 - 300. / ro_Ice
        endif
        ShearX = Por_BS / max(epsi, 1. - Por_BS)
        ! +     Gallee et al., 2001    eq 5, p5
        Fac_Mo = exp(-ShearX + ShearS)
        if(usth_param == "gal") then
            Salt_us = (log(2.868) - log(1 + SaltMo)) * rCd10n / 0.085
            ! Salt_us :  Extension of  Guyomarc'h & Merindol 1998 with
            !           de Montmollin (1978). Gallee et al. 2001
            Salt_us = Salt_us * Fac_Mo
        endif
        if(usth_param == "lis") then !Liston et al. 2007
            if(ro__SV(ikl, isn) > 300.) then
                Salt_us = 0.005 * exp(0.013 * ro__SV(ikl, isn))
            else
                Salt_us = 0.01 * exp(0.003 * ro__SV(ikl, isn))
            endif
        endif
        SnowOK = 1 - min(1, iabs(isn - isnoSV(ikl))) !Switch new vs old snow
        usthSV(ikl) = SnowOK * (Salt_us) + (1. - SnowOK) * usthSV(ikl)
    enddo
    ! end do
#endif
    
    !  Effective SBL variables
    !  =======================
    
    do ikl = 1, klonv
        VVaSBL(ikl) = VV__SV(ikl)
        VVaSBL(ikl) = VVmmem(ikl)
        dTa_Ts(ikl) = TaT_SV(ikl) - Tsrfsv(ikl)
        dTa_Ts(ikl) = dTmmem(ikl)
    enddo
    
    !  Convergence Criterion
    !  =====================
    
    icount = 0
    
    1   continue
    icount = icount + 1
    dustar = 0.
    
    do ikl = 1, klonv
        
        u0star = us__SV(ikl)

#ifdef AM
        !  u*, u*T*, u*s* Time Moving Averages
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        u0star = 0.0
#ifdef AT
        uTstar = 0.0
#endif
#ifdef AS
        usstar = 0.0
#endif
        do nt = 1, ntaver
            u0star = u0star + u__mem(ikl, nt)
#ifdef AT
            uTstar = uTstar + uT_mem(ikl, nt)
#endif
#ifdef AS
            usstar = usstar + us_mem(ikl, nt)
#endif
        enddo
        u0star = u0star / ntaver
        us__SV(ikl) = u0star
#ifdef AT
        uts_SV(ikl) = uTstar / ntaver
#endif
#ifdef AS
        uss_SV(ikl) = usstar / ntaver
#endif
#endif
        
        !  Turbulent Scales from previous Time Step
        !  ----------------------------------------
        
        u0star = max(epsi, u0star)      ! Friction Velocity     u*
        uustar = u0star * u0star       ! Friction Velocity^2  uu*
        thstar = uts_SV(ikl) / u0star       ! Temperature       theta*
        qqstar = uqs_SV(ikl) / u0star       ! Specific Humidity    qq*
        ssstar = uss_SV(ikl) / u0star       ! Blown    Snow        ss*
        
        !  Monin-Obukhov Stability Parameter for Momentum
        !  ----------------------------------------------
        
        !  Pseudo Virtual Temperature Turbulent Scale thetav*
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        thstarv = thstar + Theta0 * (0.608 * qqstar) &
                / (1. + 0.608 * QaT_SV(ikl) - qsnoSV(ikl))
        thstars = sign(unun, thstarv)
        thstara = abs(thstarv)
        thstarv = max(epsi, thstara) * thstars
        
        !  Pseudo Obukhov Length Scale        (Gall?e et al., 2001 BLM 99, (A2) p.17)
        !  Full   Obukhov Length Scale        (when Blowing * is ##NOT## switched ON)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LMO_SV(ikl) = Theta0 * max(epsi, uustar) &
                / (vonkar * gravit * thstarv)
#ifdef wr
        W_pLMO = LMO_SV(ikl)
#endif
        
        zetah = za__SV(ikl) / LMO_SV(ikl)
        zetam = min(zetMAX, zetah)! Strong Stability Limit
        !                                                  !(Mahalov et al. 2004
        !                                                  ! GRL 31 2004GL021055)
        LMOmom(ikl) = za__SV(ikl) / (max(epsi, abs(zetam)) &
                * sign(unun, zetam))
        zeta0m = Z0m_SV(ikl) / LMOmom(ikl)
        zeta0h = Z0h_SV(ikl) / LMO_SV(ikl)
        
        !  Momentum Pseudo Stability Function (Gall?e et al. 2001, BLM 99, (11) p. 7)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        stab_s = max(zero, sign(unun, zetam))
        
        psim_s = -A_Turb * zetam
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zetam)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psim_z = stab_s * psim_s + (1. - stab_s) * psim_i
#ifdef wr
        W_psim = psim_z
#endif
        
        psim_s = -A_Turb * zeta0m
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zeta0m)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psim_0 = stab_s * psim_s + (1. - stab_s) * psim_i

#ifdef AE
        !  Virtual Temperature Turbulent Scale thetav*    (ss* impact included   )
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ needed for new ss*)
        thstarv = thstar + Theta0 * (0.608 * qqstar &
                - ssstar &
                ) &
                / (1. + 0.608 * QaT_SV(ikl) - qsnoSV(ikl))
        thstars = sign(unun, thstarv)
        thstara = abs(thstarv)
        thstarv = max(epsi, thstara) * thstars
        !  Full   Obukhov Length Scale        (Gallee et al. 2001, BLM 99, (A1) p.16)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LMO_SV(ikl) = Theta0 * us__SV(ikl) * us__SV(ikl) &
                / (vonkar * gravit * thstarv)
        zetah = za__SV(ikl) / LMO_SV(ikl)
        ! Strong Stability Limit (Mahalov et al. 2004 GRL 31 2004GL021055)
        zetam = min(zetMAX, zetah)
        LMOmom(ikl) = za__SV(ikl) / (max(epsi, abs(zetam)) &
                * sign(unun, zetam))
        zeta0m = Z0m_SV(ikl) / LMOmom(ikl)
        !  Snow Erosion    Stability Function (Gall?e et al. 2001, BLM 99, (11) p. 7)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        stab_s = max(zero, sign(unun, zetam))
        psis_s = -AsTurb * zetam
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zetam)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psis_z = stab_s * psis_s + (1. - stab_s) * psim_i
        psis_s = -AsTurb * zeta0m
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zeta0m)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psis_0 = stab_s * psis_s + (1. - stab_s) * psim_i
        !  Square Roots of the Drag Coefficient for Snow Erosion Turbulent Flux
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        rCDmSV(ikl) = vonkar / (sqrCm0(ikl) - psim_z + psim_0)
#endif

#ifdef ss
        if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                nn__SV(ikl) == nwr_SV) &
                write(6, 6600) Z0m_SV(ikl), psim_z &
                        , LMO_SV(ikl), uustar &
                        , sqrCm0(ikl), psim_0 &
                        , LMOmom(ikl), thstarv
        6600    format(/, ' ** SISVATeSBL *0  ' &
                , '  Z0m_SV  = ', e12.4, '  psim_z  = ', e12.4 &
                , '  LMO_SV  = ', e12.4, '  uustar  = ', e12.4 &
                , /, '                   ' &
                , '  sqrCm0  = ', e12.4, '  psim_0  = ', e12.4 &
                , '  LMOmom  = ', e12.4, '  thstarv = ', e12.4)
#endif
        
        !  Momentum            Turbulent Scale  u*
        !  ---------------------------------------
        
        !  Momentum            Turbulent Scale  u*          in case of NO Blow. Snow
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        VVa_OK = max(0.000001, VVaSBL(ikl))
        sss__N = vonkar * VVa_OK
        sss__F = (sqrCm0(ikl) - psim_z + psim_0)
        usuth0 = sss__N / sss__F                ! u* if NO Blow. Snow

#ifdef AE
        !  Momentum  Turbulent Scale  u* in case of Blow. Snow
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        sss__G = 0.27417 * gravit
        !  ______________               _____
        !  Newton-Raphson (! Iteration, BEGIN)
        !  ~~~~~~~~~~~~~~               ~~~~~
        do iit = 1, nit
            sss__K = gravit * r_Turb * A_Turb * za__SV(ikl) &
                    * rCDmSV(ikl) * rCDmSV(ikl) &
                    / (1. + 0.608 * QaT_SV(ikl) - qsnoSV(ikl))
            us_127 = exp(SblPom * log(us__SV(ikl)))
            us_227 = us_127 * us__SV(ikl)
            us_327 = us_227 * us__SV(ikl)
            us_427 = us_327 * us__SV(ikl)
            us_527 = us_427 * us__SV(ikl)
            us__SV(ikl) = us__SV(ikl) &
                    - (us_527 * sss__F / sss__N &
                            - us_427 &
                            - us_227 * qsnoSV(ikl) * sss__K &
                            + (us__SV(ikl) * us__SV(ikl) - usthSV(ikl) * usthSV(ikl)) / sss__G) &
                            / (us_427 * 5.27 * sss__F / sss__N &
                                    - us_327 * 4.27 &
                                    - us_127 * 2.27 * qsnoSV(ikl) * sss__K &
                                    + us__SV(ikl) * 2.0 / sss__G)
            us__SV(ikl) = min(us__SV(ikl), usuth0)
            us__SV(ikl) = max(us__SV(ikl), epsi)
            rCDmSV(ikl) = us__SV(ikl) / VVa_OK
            ! sss__F = vonkar / rCDmSV(ikl)
        enddo
        us__SV(ikl) = usuth0 !desactivate feedback between BS and u*
        !  ______________               ___
        !  Newton-Raphson (! Iteration, END  )
        !  ~~~~~~~~~~~~~~               ~~~
        us_127 = exp(SblPom * log(us__SV(ikl)))
        us_227 = us_127 * us__SV(ikl)
        !  Momentum            Turbulent Scale  u*: 0-Limit in case of no Blow. Snow
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        dusuth = us__SV(ikl) - usthSV(ikl)
        signus = max(sign(unun, dusuth), zero)
#endif
#ifdef AE
        us__SV(ikl) = us__SV(ikl) * signus + & ! u* (_BS)
                usuth0 * (1. - signus) ! u* (nBS)
#else
        us__SV(ikl) = usuth0
#endif
#ifdef AE
        !  Blowing Snow        Turbulent Scale ss*
        !  ---------------------------------------
        hSalSV(ikl) = 8.436e-2 * us__SV(ikl)**SblPom
        if(qsalt_param == "pom") then
            qSalSV(ikl) = (us__SV(ikl)**2 - usthSV(ikl)**2) * signus &
                    / (hSalSV(ikl) * gravit * us__SV(ikl) * 4.2)
        endif
        if(qsalt_param == "bin") then
            qSalSV(ikl) = (us__SV(ikl) * us__SV(ikl) &
                    - usthSV(ikl) * usthSV(ikl)) * signus &
                    * 0.535 / (hSalSV(ikl) * gravit)
        endif
        ssstar = rCDmSV(ikl) * (qsnoSV(ikl) - qSalSV(ikl)) &
                * r_Turb !Bintanja 2000, BLM
        ! r_Turb compensates for an overestim. of the blown snow part. fall velocity
        uss_SV(ikl) = min(zero, us__SV(ikl) * ssstar)
#endif
#ifdef BS
        uss_SV(ikl) = max(-0.0001, uss_SV(ikl))
#endif

#ifdef ss
        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                nn__SV(ikl) == nwr_SV) &
                write(6, 6610) usuth0, us__SV(ikl) &
                        , qsnoSV(ikl), uss_SV(ikl) &
                        , usthSV(ikl), LMO_SV(ikl) &
                        , qSalSV(ikl), VVa_OK
        6610    format(/, ' ** SISVATeSBL *1  ' &
                , '  u*(nBS) = ', e12.4, '  u*(_BS) = ', e12.4 &
                , '  Qs      = ', e12.4, '  u*Qs*   = ', e12.4 &
                , /, '                   ' &
                , '  u*(_th) = ', e12.4, '  LMO     = ', e12.4 &
                , '  QSalt   = ', e12.4, '  VVa     = ', e12.4)
#endif

#ifdef wx
        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(ikl == kSV_v1 .and. lSV_v1 > 0 &
                .and. lSV_v1 <= 2) &
                write(6, 6000) daHost, icount, &
                        us__SV(ikl), 1.e3 * hSalSV(ikl), &
                        1.e3 * Z0m_SV(ikl), &
                        1.e3 * qsnoSV(ikl), 1.e3 * qSalSV(ikl) &
                        , usthSV(ikl), us__SV(ikl) - usthSV(ikl), &
                        1.e3 * ssstar, 1.e3 * us__SV(ikl) * ssstar
        6000    format(a18, i3, 6x, 'u*   [m/s] =', f6.3, '   hSalt[mm]=', e9.3, &
                '   Z0m   [mm] =', f9.3, '   q   [g/kg] =', f9.3, &
                /, 91x, '   qSa [g/kg] =', f9.3, &
                /, 27x, 'ut*[m/s]=', e9.3, '   u*-ut*   =', e9.3, &
                '   s*  [g/kg] =', f9.3, '   us* [mm/s] =', f9.3)
#endif

#ifdef AE
        !  Virtual Temperature Turbulent Scale thetav*    (ss* impact included)
        !  --------------------------------------------------------------------
        thstarv = thstar + Theta0 * (0.608 * qqstar &
                - ssstar &
                ) &
                / (1. + 0.608 * QaT_SV(ikl) - qsnoSV(ikl))
        thstars = sign(unun, thstarv)
        thstara = abs(thstarv)
        thstarv = max(epsi, thstara) * thstars
        !  Full   Obukhov Length Scale (Gall?e et al., 2001, BLM 99, (A1) p.16)
        !  --------------------------------------------------------------------
        LMO_SV(ikl) = Theta0 * us__SV(ikl) * us__SV(ikl) &
                / (vonkar * gravit * thstarv)
        zetah = za__SV(ikl) / LMO_SV(ikl)
        ! Strong Stability Limit (Mahalov et al. 2004 GRL 31 2004GL021055)
        zetam = min(zetMAX, zetah)
        LMOmom(ikl) = za__SV(ikl) / (max(epsi, abs(zetam)) &
                * sign(unun, zetam))
        zeta0m = Z0m_SV(ikl) / LMOmom(ikl)
        zeta0h = Z0h_SV(ikl) / LMO_SV(ikl)
#endif

#ifdef wx
        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(ikl == kSV_v1 .and. lSV_v1 > 0 &
                .and. lSV_v1 <= 2) &
                write(6, 6001) LMO_SV(ikl), zetah
#endif
        6001    format(18x, 9x, 'LMO  [m]=', f9.1, '   zetah[-] =', f9.3)
        
        !  Turbulent Scales
        !  ----------------
        
        !  Momentum Stability Function (Gall?e et al., 2001, BLM 99, (11) p. 7)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        stab_s = max(zero, sign(unun, zetam))
        
        psim_s = -A_Turb * zetam
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zetam)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psim_z = stab_s * psim_s + (1. - stab_s) * psim_i
        
        psim_s = -A_Turb * zeta0m
        xpsimi = sqrt(sqrt(unun - coef_m * min(zero, zeta0m)))
        psim_i = 2. * log(demi * (unun + xpsimi)) &
                + log(demi * (unun + xpsimi * xpsimi)) &
                - 2. * atan(xpsimi) + demi * pi
        psim_0 = stab_s * psim_s + (1. - stab_s) * psim_i
        
        !  Heat     Stability Function (Gall?e et al., 2001, BLM 99, (11) p. 7)
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        stab_s = max(zero, sign(unun, zetah))
        
        psih_s = -AhTurb * zetah
        xpsihi = sqrt(sqrt(unun - coef_h * min(zero, zetah)))
        psih_i = 2. * log(demi * (unun + xpsihi))
        psih_z = stab_s * psih_s + (1. - stab_s) * psih_i
        
        psih_s = -AhTurb * zeta0h
        xpsihi = sqrt(sqrt(unun - coef_h * min(zero, zeta0h)))
        psih_i = 2. * log(demi * (unun + xpsihi))
        psih_0 = stab_s * psih_s + (1. - stab_s) * psih_i
        
        !  Square Roots of the Drag Coefficients
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        rCDhSV(ikl) = vonkar * (ExnrSV(ikl) / pcap) &
                / (sqrCh0(ikl) - psih_z + psih_0)
        rCDmSV(ikl) = vonkar / (sqrCm0(ikl) - psim_z + psim_0)
        
        !  Drag Coefficients
        !  ~~~~~~~~~~~~~~~~~
        CDh(ikl) = rCDmSV(ikl) * rCDhSV(ikl)
        CDm(ikl) = rCDmSV(ikl) * rCDmSV(ikl)
        
        !  real Temperature Turbulent Scale theta*
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        thstar = rCDhSV(ikl) * dTa_Ts(ikl) &
                * (pcap / ExnrSV(ikl))
        uts_SV(ikl) = us__SV(ikl) * thstar
        
        !  Convergence Criterion
        !  =====================
        
        dustar = max(dustar, abs(us__SV(ikl) - u0star))

#ifdef AM
        !  u*, u*T*, u*s* Time Moving Averages
        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        do nt = 1, ntaver - 1
            u__mem(ikl, nt) = u__mem(ikl, nt + 1)
#ifdef AT
            uT_mem(ikl, nt) = uT_mem(ikl, nt + 1)
#endif
#ifdef AS
            us_mem(ikl, nt) = us_mem(ikl, nt + 1)
#endif
        enddo
        u__mem(ikl, ntaver) = us__SV(ikl)
#ifdef AT
        uT_mem(ikl, ntaver) = uts_SV(ikl)
#endif
#ifdef AS
        us_mem(ikl, ntaver) = uss_SV(ikl)
#endif
#endif

#ifdef wr
        !  OUTPUT for Verification (general)
        !  ~~~~~~~~~~~~~~~~~~~~~~~
        if(icount == 1) then
            write(6, 6004)
            6004        format(122('-'))
            if(mod(VVaSBL(ikl), 4.) < 0.1) then
                write(6, 6003)
                6003            format('   V  Ta-Ts  Z0      It' &
                        , ' du*     u*    sss__F   CD       Qss       Qs*     ' &
                        , ' PseudOL Full-OL zetam   zetah   psim_z  psih_z')
                write(6, 6004)
            endif
        endif
        write(6, 6002) VVaSBL(ikl), dTa_Ts(ikl), Z0m_SV(ikl), icount &
                , dustar, us__SV(ikl), sss__F &
                , CDm(ikl), qSalSV(ikl), ssstar &
                , W_pLMO, LMO_SV(ikl) &
                , zetam, zetah, W_psim, psih_z
        6002    format(2f6.1, f8.4, i3, f9.6, f6.3, f9.3, 3f9.6, 2f8.2, 2f8.4, 2f8.2)
#endif

#ifdef w1
        !  OUTPUT for Verification (u*_AE)
        !  ~~~~~~~~~~~~~~~~~~~~~~~
        if(icount == 1) then
            write(6, 6014)
            6014        format(100('-'))
            if(mod(VVaSBL(ikl), 4.) < 0.1) then
                write(6, 6013)
                6013            format('   V  Ta-Ts  Z0      It' &
                        , ' du*     u*    sss__F   W_NUs1   W_NUs2   W_NUs3      ' &
                        , ' W_DUs1     W_DUs2 ')
                write(6, 6014)
            endif
        endif
        write(6, 6012) VVaSBL(ikl), dTa_Ts(ikl), Z0m_SV(ikl), icount &
                , dustar, us__SV(ikl), sss__F &
                , W_NUs1, W_NUs2, W_NUs3 &
                , W_DUs1, W_DUs2
        6012    format(2f6.1, f8.4, i3, f9.6, f6.3, f9.3, 3f9.3, 2f12.3)
#endif
    
    enddo

#ifdef IX
    if(icount < 3) go to 1
#endif
    !     if (dustar.gt.0.0001.and.icount.lt. 6)                     go to 1

#ifdef AM
    do ikl = 1, klonv
        u0star = 0.0
#ifdef AT
        uTstar = 0.0
#endif
#ifdef AS
        usstar = 0.0
#endif
        do nt = 1, ntaver
            u0star = u0star + u__mem(ikl, nt)
#ifdef AT
            uTstar = uTstar + uT_mem(ikl, nt)
#endif
#ifdef AS
            usstar = usstar + us_mem(ikl, nt)
#endif
        enddo
        us__SV(ikl) = u0star / ntaver
#ifdef AT
        uts_SV(ikl) = uTstar / ntaver
#endif
#ifdef AS
        uss_SV(ikl) = usstar / ntaver
#endif
    enddo
#endif
    
    !  Aerodynamic Resistances
    !  -----------------------
    
    do ikl = 1, klonv
        ram_sv(ikl) = 1. / (CDm(ikl) * max(VVaSBL(ikl), epsi))
        rah_sv(ikl) = 1. / (CDh(ikl) * max(VVaSBL(ikl), epsi))
    enddo
    
    return
endsubroutine SISVATeSBL
