#include "MAR_pp.def"
subroutine SISVAT_zSn
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_zSn                                02-10-2021  MAR |
    ! |   subroutine SISVAT_zSn manages the Snow Pack vertical Discretization  |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT /  NLaysv   = New             Snow Layer  Switch               |
    ! |   OUTPUT:  isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |   ^^^^^^   ispiSV   = 0,...,nsno: Uppermost Superimposed Ice Layer     |
    ! |            iiceSV   = total Nb of Ice      Layers                      |
    ! |            istoSV   = 0,...,5 :   Snow     History (see istdSV data)   |
    ! |                                                                        |
    ! |   INPUT /  TsisSV   : Soil/Ice Temperatures (layers -nsol,-nsol+1,..,0)|
    ! |   OUTPUT:           & Snow     Temperatures (layers  1,2,...,nsno) [K] |
    ! |   ^^^^^^   ro__SV   : Soil/Snow Volumic Mass                   [kg/m3] |
    ! |            eta_SV   : Soil/Snow Water   Content                [m3/m3] |
    ! |            dzsnSV   : Snow Layer        Thickness                  [m] |
    ! |            G1snSV   : Dendricity (<0) or Sphericity (>0) of Snow Layer |
    ! |            G2snSV   : Sphericity (>0) or Size            of Snow Layer |
    ! |            agsnSV   : Snow       Age                             [day] |
    ! |                                                                        |
    ! |   METHOD:  1) Agregate the thinest Snow Layer                          |
    ! |   ^^^^^^      if a new Snow Layer has been precipitated   (NLaysv = 1) |
    ! |            2) Divide   a too thick Snow Layer except                   |
    ! |               if the maximum Number of Layer is reached                |
    ! |               in this case forces                          NLay_s = 1  |
    ! |            3) Agregate the thinest Snow Layer                          |
    ! |               in order to divide a too thick Snow Layer                |
    ! |               at next Time Step when                       NLay_s = 1  |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! |   FILE                 |      CONTENT                                  |
    ! |   ~~~~~~~~~~~~~~~~~~~~~+~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ |
    ! | # SISVAT_zSn.vz        | #vz: OUTPUT/Verification: Snow Layers Agrega. |
    ! |                        |      unit 41, subroutine  SISVAT_zSn **ONLY** |
    ! | # SISVAT_GSn.vp        | #vp: OUTPUT/Verification: Snow   Properties   |
    ! |                        |      unit 47, subroutines SISVAT_zSn, _GSn    |
    ! +------------------------------------------------------------------------+

    use marphy
    use mar_sv
    use mardsv
    use mar0sv
    use marxsv
    use marysv

    implicit none

    ! +--Internal Variables
    ! +  ==================

    integer ikl, isn, i
    ! NLay_s : Split Snow Layer Switch
    integer NLay_s(klonv)
    ! isagr1 : 1st Layer History
    integer isagr1(klonv)
    ! isagr2 : 2nd Layer History
    integer isagr2(klonv)
    ! LstLay : 0 -> isnoSV = 1
    integer LstLay
    ! isno_n : Snow Normal.Profile
    integer isno_n
    ! iice_n : Ice  Normal.Profile
    integer iice_n
    ! iiceOK : Ice Switch
    integer iiceOK
    ! icemix : 0 -> Agregated Snow+Ice=Snow  1 -> Ice
    integer icemix
    ! isn1 : 1st layer to stagger
    integer isn1(klonv)
    ! staggr : stagger Switch
    real staggr
    ! WEagre : Snow Water Equivalent Thickness
    real WEagre(klonv)
    ! dzthin : Thickness of the thinest layer
    real dzthin(klonv)
    ! OKthin : Swich ON a new thinest layer
    real OKthin
    ! dz_dif : difference from ideal discret.
    real dz_dif
    ! thickL : Thick Layer Indicator
    real thickL
    ! OK_ICE : Swich ON  uppermost Ice Layer
    real OK_ICE

    ! Agrege : 1. when Agregation constrained
    real Agrege(klonv)
    ! dzepsi : Min Single Snw Layer Thickness
    real dzepsi
    ! dzxmin : Min Acceptable Layer Thickness
    real dzxmin
    ! dz_min : Min Layer Thickness
    real dz_min
    ! dz_max : Max Layer Thickness
    real dz_max
    ! dzagr1 : 1st Layer Thickness
    real dzagr1(klonv)
    ! dzagr2 : 2nd Layer Thickness
    real dzagr2(klonv)
    ! T_agr1 : 1st Layer Temperature
    real T_agr1(klonv)
    ! T_agr2 : 2nd Layer Temperature
    real T_agr2(klonv)
    ! roagr1 : 1st Layer Density
    real roagr1(klonv)
    ! roagr2 : 2nd Layer Density
    real roagr2(klonv)
    ! etagr1 : 1st Layer Water Content
    real etagr1(klonv)
    ! etagr2 : 2nd Layer Water Content
    real etagr2(klonv)
    ! G1agr1 : 1st Layer Dendricity/Spher.
    real G1agr1(klonv)
    ! G1agr2 : 2nd Layer Dendricity/Spher.
    real G1agr2(klonv)
    ! G2agr1 : 1st Layer Sphericity/Size
    real G2agr1(klonv)
    ! G2agr2 : 2nd Layer Sphericity/Size
    real G2agr2(klonv)
    ! agagr1 : 1st Layer Age
    real agagr1(klonv)
    ! agagr2 : 2nd Layer Age
    real agagr2(klonv)

#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
    common / SISVAT_EV / iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
#endif

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    ! IO   Switch
    logical as_opn
    common / SI_zSn_L / as_opn
    ! Snow Reference Discretization
    real dz_ref(nsno)
    real dzwdif(nsno)
#endif

#ifdef vp
    ! +--Snow Properties Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    ! IO   Switch
    logical VP_opn
    common / SI_GSn_L / VP_opn
#endif

    ! +--DATA
    ! +  ====

    data icemix/0/             ! 0 ====> Agregated Snow+Ice=Snow
    data dzepsi/0.0020/            ! Min single Layer Thickness
    data dzxmin/0.0025/            ! Min accept.Layer Thickness
#ifdef EU
    data dz_min/0.0050/            ! Min Local  Layer Thickness < SMn
#endif
    data dz_min/0.0040/            ! Min Local  Layer Thickness < SMn
    data dz_max/0.0300/            ! Min Gener. Layer Thickness
    ! +   CAUTION:  dz_max > dz_min*2 is required ! Otherwise re-agregation is
    ! +                                           ! activated  after splitting

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  =====================
    if(.not. as_opn) then
        as_opn = .true.
        open(unit=41, status='unknown', file='SISVAT_zSn.vz')
        rewind 41
    endif
#endif

#ifdef vp
    ! +--Snow Properties:   IO
    ! +  =====================
    if(.not. VP_opn) then
        VP_opn = .true.
        open(unit=47, status='unknown', file='SISVAT_GSn.vp')
        rewind 47
    endif
#endif

    ! +--Constrains Agregation of too thin Layers
    ! +  ========================================

    ! +--Search the thinest non-zero Layer
    ! +  ---------------------------------

    do ikl = 1, klonv

        if(isnoSV(ikl) <= 2) dz_min = max(0.0050, dz_min)

        dzepsi = 0.0015
        if(ro__SV(ikl, isnoSV(ikl)) > 920) dzepsi = 0.0020

        dzthin(ikl) = 0.                              ! Arbitrary unrealistic
    enddo                             !       Layer Thickness
    !XF
    do ikl = 1, klonv
        ! no agregation of 3 first snowlayers
        ! XF 04/07/2019
        do isn = 1, isnoSV(ikl) - 3
            isno_n = isnoSV(ikl) - isn + 1     ! Snow Normal.Profile
            iice_n = iiceSV(ikl) - isn       ! Ice  Normal.Profile
            iiceOK = min(1, max(0, iice_n + 1))   ! Ice         Switch
#ifdef vz
            ! Theoretical Profile
            dz_ref(isn) = &
                dz_min * ((1 - iiceOK) * isno_n * isno_n &
                          + iiceOK * 2**iice_n) &
                / max(1, isnoSV(ikl))
#endif
            ! Actual Profile
            dz_dif = max(zero, &
                         ! Theoretical Profile
                         dz_min * ((1 - iiceOK) * isno_n * isno_n &
                                   + iiceOK * 2.**iice_n) &
                         ! Actual Profile
                         - dzsnSV(ikl, isn))
#ifdef vz
            dzwdif(isn) = dz_dif
#endif
            OKthin = max(zero, &
                         ! 1.=> New thinest Lay.
                         sign(unun, dz_dif - dzthin(ikl))) &
                     ! 1 => .le. isnoSV => isn is in the Snow Pack
                     * max(0, min(1, isnoSV(ikl) - isn + 1)) &
                     * min(unun, max(zero, &
                                     ! combination G1 with same sign => OK
                                     sign(unun, G1snSV(ikl, isn) &
                                          * G1snSV(ikl, max(1, isn - 1)))) &
                           ! G1>0 => OK
                           + max(zero, sign(unun, G1snSV(ikl, isn))) &
                           ! dz too small => OK
                           + max(zero, sign(unun, dzxmin - dzsnSV(ikl, isn))))
            ! Update   thinest Lay. Index
            i_thin(ikl) = (1.-OKthin) * i_thin(ikl) + OKthin * isn
            dzthin(ikl) = (1.-OKthin) * dzthin(ikl) + OKthin * dz_dif
        enddo
    enddo

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 4150) daHost, n___SV(lwriSV(1)) &
        , i_thin(1), dzsnSV(1, i_thin(1))
4150 format(/, '-', a18, i5, ' ', 70('-'), &
            /, ' Thinest ', i3, ':', f9.3)
#endif

    do ikl = 1, klonv
        do isn = 1, isnoSV(ikl)
            OKthin = max(zero, &
                         ! ON if dz < dz_min and dz > 0
                         sign(unun, dz_min - dzsnSV(ikl, isn))) &
                     * max(zero, sign(unun, dzsnSV(ikl, isn) - epsi)) &
                     ! Multiple Snow Layers
                     * min(1, max(0, &
                                  ! Switch = 1 if isno > iice + 1
                                  min(1, isnoSV(ikl) - iiceSV(ikl) - 1)) &
                           + int(max(zero, &
                                     ! Minimum accepted for 1 Snow Layer over Ice
                                     sign(unun, dzepsi - dzsnSV(ikl, isn)))) &
                           ! ON if dz > 0
                           * int(max(zero, sign(unun, dzsnSV(ikl, isn) - epsi))) &
                           ! Switch = 1 if isno = iice + 1
                           * (1 - min(abs(isnoSV(ikl) - iiceSV(ikl) - 1), 1)) &
                           ! Ice Switch
                           + max(0, min(1, iiceSV(ikl) + 1 - isn))) &
                     * min(unun, &
                           ! combination G1>0 + G1<0
                           max(zero, &
                               sign(unun, G1snSV(ikl, isn) * G1snSV(ikl, max(1, isn - 1)))) &
                           + max(zero, sign(unun, G1snSV(ikl, isn))) &
                           + max(zero, sign(unun, dzxmin - dzsnSV(ikl, isn))))
            ! Update thinest Layer Index
            i_thin(ikl) = (1.-OKthin) * i_thin(ikl) + OKthin * isn
        enddo
    enddo

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 4151) i_thin(1), dzsnSV(1, i_thin(1)) &
        , isnoSV(1), dzsnSV(1, isnoSV(1))
4151 format(' Thinest ', i3, ':', f9.3, '   Max   =', i3, f12.3)
#endif

#ifdef vp
    ! +--Snow Properties Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    write(47, 470)(G1snSV(1, isn), isn=1, isnoSV(1))
470 format('Before _zCr1: G1 = ', 10f8.1,(/, 19x, 10f8.1))
    write(47, 472)(G2snSV(1, isn), isn=1, isnoSV(1))
472 format('              G2 = ', 10f8.1,(/, 19x, 10f8.1))
#endif

    ! +--Index of the contiguous Layer to agregate
    ! +  -----------------------------------------
    ! +   *********
    call SISVAT_zCr
    ! +   *********

    ! +--Assign the 2 Layers to agregate
    ! +  -------------------------------
    do ikl = 1, klonv
        isn = i_thin(ikl)
        if(LIndsv(ikl) > 0) isn = min(nsno - 1, isn) ! cXF
        isagr1(ikl) = istoSV(ikl, isn)
        isagr2(ikl) = istoSV(ikl, isn + LIndsv(ikl))
        dzagr1(ikl) = dzsnSV(ikl, isn)
        dzagr2(ikl) = dzsnSV(ikl, isn + LIndsv(ikl))
        T_agr1(ikl) = TsisSV(ikl, isn)
        T_agr2(ikl) = TsisSV(ikl, isn + LIndsv(ikl))
        roagr1(ikl) = ro__SV(ikl, isn)
        roagr2(ikl) = ro__SV(ikl, isn + LIndsv(ikl))
        etagr1(ikl) = eta_SV(ikl, isn)
        etagr2(ikl) = eta_SV(ikl, isn + LIndsv(ikl))
        G1agr1(ikl) = G1snSV(ikl, isn)
        G1agr2(ikl) = G1snSV(ikl, isn + LIndsv(ikl))
        G2agr1(ikl) = G2snSV(ikl, isn)
        G2agr2(ikl) = G2snSV(ikl, isn + LIndsv(ikl))
        agagr1(ikl) = agsnSV(ikl, isn)
        agagr2(ikl) = agsnSV(ikl, isn + LIndsv(ikl))
        ! 0  if single Layer
        LstLay = min(1, max(0, isnoSV(ikl) - 1))
        ! decrement   isnoSV if downmost  Layer <  1.e-21 m
        isnoSV(ikl) = isnoSV(ikl) &
                      - (1 - LstLay) * max(zero, &
                                           sign(unun, eps_21 - dzsnSV(ikl, 1)))
        isnoSV(ikl) = max(0, isnoSV(ikl))
        Agrege(ikl) = max(zero, &
                          ! No Agregation if too thick Layer
                          sign(unun, dz_min - dzagr1(ikl))) &
                      ! if  a single Layer
                      * LstLay &
                      ! if Agregation with a Layer above the Pack
                      * min(max(0, isnoSV(ikl) + 1 - i_thin(ikl) - LIndsv(ikl)), 1)
        WEagre(ikl) = 0.
    enddo

    do ikl = 1, klonv
        do isn = 1, isnoSV(ikl)
            WEagre(ikl) = WEagre(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * min(1, max(0, i_thin(ikl) + 1 - isn))
        enddo
    enddo

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 410)
410 format(/, ' Agregation of too THIN Layers')
    write(41, 411)(100.*dz_ref(isn), isn=1, nsno)
    write(41, 412)(100.*dzwdif(isn), isn=1, nsno)
    write(41, 413)(100.*dzsnSV(1, isn), isn=1, nsno)
    write(41, 414)(isn, isn=1, nsno)
411 format(' dz_ref [cm]:', 10f8.2, /,('             ', 10f8.2))
412 format(' dz_dif [cm]:', 10f8.2, /,('             ', 10f8.2))
413 format(' dzsnSV [cm]:', 10f8.2, /,('             ', 10f8.2))
414 format('             ', 10(i5, 3x), /,('             ', 10(i5, 3x)))
    write(41, 4111) isnoSV(1)
    write(41, 4112) i_thin(1)
    write(41, 4113) LIndsv(1)
    write(41, 4114) Agrege(1)
    write(41, 4115) 1.e2 * dzagr1(1)
    write(41, 4116) 1.e2 * dzagr2(1)
4111 format(' isnoSV     :', i8)
4112 format(' i_thin     :', i8)
4113 format(' LIndsv     :', i8)
4114 format(' Agrege     :', f8.2)
4115 format(' dzagr1     :', f8.2)
4116 format(' dzagr2     :', f8.2)
#endif

#ifdef vp
    ! +--Snow Properties Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    write(47, 471)(G1snSV(1, isn), isn=1, isnoSV(1))
471 format('Before _zAg1: G1 = ', 10f8.1,(/, 19x, 10f8.1))
    write(47, 472)(G2snSV(1, isn), isn=1, isnoSV(1))
#endif

    ! +--Agregates
    ! +  ---------
    ! +  **********
    call SISVAT_zAg(isagr1, isagr2, WEagre &
                    , dzagr1, dzagr2, T_agr1, T_agr2 &
                    , roagr1, roagr2, etagr1, etagr2 &
                    , G1agr1, G1agr2, G2agr1, G2agr2 &
                    , agagr1, agagr2, Agrege)
    ! +  **********

    ! +--Rearranges the Layers
    ! +  ---------------------

    ! +--New (agregated) Snow layer
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isn = i_thin(ikl)
        isn = min(isn, isn + LIndsv(ikl))
        isnoSV(ikl) = max(0., isnoSV(ikl) - Agrege(ikl))
        iiceSV(ikl) = iiceSV(ikl) &
                      - max(0, sign(1, iiceSV(ikl) - isn + icemix)) &
                      * Agrege(ikl) &
                      * max(0, sign(1, iiceSV(ikl) - 1))
        istoSV(ikl, isn) = (1.-Agrege(ikl)) * istoSV(ikl, isn) &
                           + Agrege(ikl) * isagr1(ikl)
        dzsnSV(ikl, isn) = (1.-Agrege(ikl)) * dzsnSV(ikl, isn) &
                           + Agrege(ikl) * dzagr1(ikl)
        TsisSV(ikl, isn) = (1.-Agrege(ikl)) * TsisSV(ikl, isn) &
                           + Agrege(ikl) * T_agr1(ikl)
        ro__SV(ikl, isn) = (1.-Agrege(ikl)) * ro__SV(ikl, isn) &
                           + Agrege(ikl) * roagr1(ikl)
        eta_SV(ikl, isn) = (1.-Agrege(ikl)) * eta_SV(ikl, isn) &
                           + Agrege(ikl) * etagr1(ikl)
        G1snSV(ikl, isn) = (1.-Agrege(ikl)) * G1snSV(ikl, isn) &
                           + Agrege(ikl) * G1agr1(ikl)
        G2snSV(ikl, isn) = (1.-Agrege(ikl)) * G2snSV(ikl, isn) &
                           + Agrege(ikl) * G2agr1(ikl)
        agsnSV(ikl, isn) = (1.-Agrege(ikl)) * agsnSV(ikl, isn) &
                           + Agrege(ikl) * agagr1(ikl)
    enddo

    ! +--Above
    ! +  ^^^^^
    do ikl = 1, klonv
        isn1(ikl) = max(i_thin(ikl), i_thin(ikl) + LIndsv(ikl))
    enddo
    do i = 1, nsno - 1
        do ikl = 1, klonv
            staggr = min(1, max(0, i + 1 - isn1(ikl)))
            istoSV(ikl, i) = (1.-staggr) * istoSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * istoSV(ikl, i) &
                                         + Agrege(ikl) * istoSV(ikl, i + 1))
            dzsnSV(ikl, i) = (1.-staggr) * dzsnSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * dzsnSV(ikl, i) &
                                         + Agrege(ikl) * dzsnSV(ikl, i + 1))
            TsisSV(ikl, i) = (1.-staggr) * TsisSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * TsisSV(ikl, i) &
                                         + Agrege(ikl) * TsisSV(ikl, i + 1))
            ro__SV(ikl, i) = (1.-staggr) * ro__SV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * ro__SV(ikl, i) &
                                         + Agrege(ikl) * ro__SV(ikl, i + 1))
            eta_SV(ikl, i) = (1.-staggr) * eta_SV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * eta_SV(ikl, i) &
                                         + Agrege(ikl) * eta_SV(ikl, i + 1))
            G1snSV(ikl, i) = (1.-staggr) * G1snSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * G1snSV(ikl, i) &
                                         + Agrege(ikl) * G1snSV(ikl, i + 1))
            G2snSV(ikl, i) = (1.-staggr) * G2snSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * G2snSV(ikl, i) &
                                         + Agrege(ikl) * G2snSV(ikl, i + 1))
            agsnSV(ikl, i) = (1.-staggr) * agsnSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * agsnSV(ikl, i) &
                                         + Agrege(ikl) * agsnSV(ikl, i + 1))
        enddo
    enddo

    do ikl = 1, klonv
        isn = min(isnoSV(ikl) + 1, nsno)
        istoSV(ikl, isn) = (1.-Agrege(ikl)) * istoSV(ikl, isn)
        dzsnSV(ikl, isn) = (1.-Agrege(ikl)) * dzsnSV(ikl, isn)
        TsisSV(ikl, isn) = (1.-Agrege(ikl)) * TsisSV(ikl, isn)
        ro__SV(ikl, isn) = (1.-Agrege(ikl)) * ro__SV(ikl, isn)
        eta_SV(ikl, isn) = (1.-Agrege(ikl)) * eta_SV(ikl, isn)
        G1snSV(ikl, isn) = (1.-Agrege(ikl)) * G1snSV(ikl, isn)
        G2snSV(ikl, isn) = (1.-Agrege(ikl)) * G2snSV(ikl, isn)
        agsnSV(ikl, isn) = (1.-Agrege(ikl)) * agsnSV(ikl, isn)
    enddo

#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if(ikl == kSV_v1 .and. lSV_v1 == 3) then
        write(6, 5991) i_thin(ikl)
5991    format(/, 'First Agregation / Layer', i3, &
                /, '  i', 11x, 'T', 9x, 'rho', 10x, 'dz', 11x, 'H')
        write(6, 5995)(isn, TsisSV(ikl, isn), ro__SV(ikl, isn) &
                       , dzsnSV(ikl, isn), istoSV(ikl, isn), &
                       isn=isnoSV(ikl), 1, -1)
5995    format(i3, 3f12.3, i12)
    endif
#endif

    ! +--Constrains Splitting of too thick Layers
    ! +  ========================================

    ! +--Search the thickest non-zero Layer
    ! +  ----------------------------------

    do ikl = 1, klonv
        ! Arbitrary unrealistic
        dzthin(ikl) = 0.
    enddo
    do ikl = 1, klonv
        do isn = 1, isnoSV(ikl)
            ! Snow Normal.Profile
            isno_n = isnoSV(ikl) - isn + 1
            ! Ice  Normal.Profile
            iice_n = iiceSV(ikl) - isn
            ! Ice         Switch
            iiceOK = min(1, max(0, iice_n + 1))
            ! Actual      Profile
            dz_dif = (dzsnSV(ikl, isn) &
                      ! Theoretical Profile
                      - dz_max * ((1 - iiceOK) * isno_n * isno_n &
                                  + iiceOK * 2.**iice_n)) &
                     / max(dzsnSV(ikl, isn), epsi)
            OKthin = max(zero, &
                         sign(unun, &
                              ! 1.=>New thickest Lay.
                              dz_dif - dzthin(ikl))) &
                     ! 1 =>.le. isnoSV
                     * max(0, &
                           min(1, &
                               isnoSV(ikl) - isn + 1))
            !  Update thickest Lay. Index
            i_thin(ikl) = (1.-OKthin) * i_thin(ikl) &
                          + OKthin * isn
            dzthin(ikl) = (1.-OKthin) * dzthin(ikl) &
                          + OKthin * dz_dif
        enddo

        isn = 1
        if(isnoSV(ikl) > 1 .and. dzsnSV(ikl, isn) > 5) then
            ! layer > 5m
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

        isn = 2
        if(isnoSV(ikl) > 2 .and. dzsnSV(ikl, isn) > 5) then
            ! layer > 5m
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

        isn = max(1, isnoSV(ikl) - 3)
        ! surface layer > 30cm ! XF 04/07/2019
        if(dzsnSV(ikl, isn) > 0.30) then
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

        isn = max(1, isnoSV(ikl) - 2)
        ! surface layer > 10cm ! XF 04/07/2019
        if(dzsnSV(ikl, isn) > 0.10) then
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

        isn = max(1, isnoSV(ikl) - 1)
        ! surface layer > 5cm ! XF 04/07/2019
        if(dzsnSV(ikl, isn) > 0.05) then
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

        isn = max(1, isnoSV(ikl))
        ! surface layer > 2cm ! XF 04/07/2019
        if(dzsnSV(ikl, isn) > 0.02) then
            i_thin(ikl) = isn
            dzthin(ikl) = dzsnSV(ikl, isn)
        endif

    enddo

    do ikl = 1, klonv
        ! 1. => a too   thick Layer exists
        ThickL = max(zero, &
                     sign(unun, dzthin(ikl) &
                          - epsi)) &
                 ! No spliting allowed if isno > nsno - 1
                 * max(0, 1 - max(0, isnoSV(ikl) &
                                  - nsno + 1))
        ! 1. => effective split
        Agrege(ikl) = ThickL &
                      * max(0, 1 - max(0, NLaysv(ikl) &
                                       + isnoSV(ikl) &
                                       - nsno + 1))
        ! Agregation to allow Splitting at next Time Step
        NLay_s(ikl) = ThickL &
                      * max(0, 1 - max(0, NLaysv(ikl) &
                                       + isnoSV(ikl) &
                                       - nsno)) &
                      - Agrege(ikl)
        ! Agregation effective
        NLay_s(ikl) = max(0, NLay_s(ikl))
    enddo

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 4152) i_thin(1), dzthin(1), ThickL
4152 format(/, ' Thickest', i3, ':', f9.3, '   Split =', f4.0)
#endif

    ! +--Rearranges the Layers
    ! +  ---------------------

    do isn = nsno, 2, -1
        do ikl = 1, klonv
            if(Agrege(ikl) > 0. .and. i_thin(ikl) < isnoSV(ikl)) then
                staggr = min(1, max(0, isn - i_thin(ikl) - 1)) &
                         * min(1, max(0, isnoSV(ikl) - isn + 2))
                istoSV(ikl, isn) = staggr * istoSV(ikl, isn - 1) &
                                   + (1.-staggr) * istoSV(ikl, isn)
                dzsnSV(ikl, isn) = staggr * dzsnSV(ikl, isn - 1) &
                                   + (1.-staggr) * dzsnSV(ikl, isn)
                TsisSV(ikl, isn) = staggr * TsisSV(ikl, isn - 1) &
                                   + (1.-staggr) * TsisSV(ikl, isn)
                ro__SV(ikl, isn) = staggr * ro__SV(ikl, isn - 1) &
                                   + (1.-staggr) * ro__SV(ikl, isn)
                eta_SV(ikl, isn) = staggr * eta_SV(ikl, isn - 1) &
                                   + (1.-staggr) * eta_SV(ikl, isn)
                G1snSV(ikl, isn) = staggr * G1snSV(ikl, isn - 1) &
                                   + (1.-staggr) * G1snSV(ikl, isn)
                G2snSV(ikl, isn) = staggr * G2snSV(ikl, isn - 1) &
                                   + (1.-staggr) * G2snSV(ikl, isn)
                agsnSV(ikl, isn) = staggr * agsnSV(ikl, isn - 1) &
                                   + (1.-staggr) * agsnSV(ikl, isn)
            endif
        enddo
    enddo

    do ikl = 1, klonv
        isn = i_thin(ikl)
        dzsnSV(ikl, isn) = 0.5 * Agrege(ikl) * dzsnSV(ikl, isn) &
                           + (1.-Agrege(ikl)) * dzsnSV(ikl, isn)

        isn = min(i_thin(ikl) + 1, nsno)
        istoSV(ikl, isn) = Agrege(ikl) * istoSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * istoSV(ikl, isn)
        dzsnSV(ikl, isn) = Agrege(ikl) * dzsnSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * dzsnSV(ikl, isn)
        TsisSV(ikl, isn) = Agrege(ikl) * TsisSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * TsisSV(ikl, isn)
        ro__SV(ikl, isn) = Agrege(ikl) * ro__SV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * ro__SV(ikl, isn)
        eta_SV(ikl, isn) = Agrege(ikl) * eta_SV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * eta_SV(ikl, isn)
        G1snSV(ikl, isn) = Agrege(ikl) * G1snSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * G1snSV(ikl, isn)
        G2snSV(ikl, isn) = Agrege(ikl) * G2snSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * G2snSV(ikl, isn)
        agsnSV(ikl, isn) = Agrege(ikl) * agsnSV(ikl, isn - 1) &
                           + (1.-Agrege(ikl)) * agsnSV(ikl, isn)
        isnoSV(ikl) = min(Agrege(ikl) + isnoSV(ikl), real(nsno))
        iiceSV(ikl) = iiceSV(ikl) &
                      + Agrege(ikl) * max(0, sign(1, iiceSV(ikl) &
                                                  - isn + icemix)) &
                      * max(0, sign(1, iiceSV(ikl) &
                                    - 1))
    enddo

    ! +--Constrains Agregation in case of too much  Layers
    ! +  =================================================

    ! +--Search the thinest   non-zero Layer
    ! +  -----------------------------------

#ifdef La
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(6, *) ' '
    write(6, *) 'Agregation 2'
    write(6, 6000) NLaysv(1)
6000 format(i3, 6x, &
           'dzsnSV      dz_min      dz_dif ', &
           'OKthin      dzthin   i_thin')
#endif

    do ikl = 1, klonv
        ! Arbitrary unrealistic Layer Thickness
        dzthin(ikl) = 0.
    enddo
    do ikl = 1, klonv
        ! no agregation of 3 first snowlayers ! XF 04/07/2019
        do isn = 1, isnoSV(ikl) - 3
            ! Snow Normal.Profile
            isno_n = isnoSV(ikl) - isn + 1
            ! Ice  Normal.Profile
            iice_n = iiceSV(ikl) - isn
            ! Ice Switch
            iiceOK = min(1, max(0, iice_n + 1))
#ifdef vz
            ! Theoretical Profile
            dz_ref(isn) = &
                dz_min * ((1 - iiceOK) * isno_n * isno_n &
                          + iiceOK * 2**iice_n) &
                / max(1, isnoSV(ikl))
#endif
            ! Actual      Profile
            dz_dif = dz_min &
                     - dzsnSV(ikl, isn) &
                     ! Theoretical Profile
                     / max(epsi,((1 - iiceOK) * isno_n * isno_n &
                                 + iiceOK * 2.**iice_n))
#ifdef vz
            dzwdif(isn) = dz_dif
#endif
            OKthin = max(zero, &
                         sign(unun, &
                              ! 1.=> New thinest Lay.
                              dz_dif - dzthin(ikl))) &
                     ! 1 => .le. isnoSV
                     * max(0, &
                           min(1, &
                               isnoSV(ikl) - isn + 1))
            ! Update   thinest Lay. Index
            i_thin(ikl) = (1.-OKthin) * i_thin(ikl) &
                          + OKthin * isn
            dzthin(ikl) = (1.-OKthin) * dzthin(ikl) &
                          + OKthin * dz_dif

#ifdef La
            ! +--Layers Agregation: IO
            ! +  ~~~~~~~~~~~~~~~~~~~~~
            if(isn <= isnoSV(1) .and. ikl == 1) &
                write(6, 6001) isn, dzsnSV(ikl, isn), dz_min * isno_n * isno_n, dz_dif &
                , OKthin, dzthin(ikl), i_thin(ikl)
6001        format(i3, 5f12.6, i9)
#endif
        enddo
    enddo

#ifdef La
    write(6, *) ' '
#endif

#ifdef vz
    write(41, 4153) i_thin(1), dzsnSV(1, i_thin(1))
4153 format(/, ' Thinest ', i3, ':', f9.3)
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 4151) i_thin(1), dzsnSV(1, i_thin(1)) &
        , isnoSV(1), dzsnSV(1, isnoSV(1))
#endif

#ifdef vp
    ! +--Snow Properties Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    write(47, 473)(G1snSV(1, isn), isn=1, isnoSV(1))
473 format('Before _zCr2: G1 = ', 10f8.1,(/, 19x, 10f8.1))
    write(47, 472)(G2snSV(1, isn), isn=1, isnoSV(1))
#endif

    ! +--Index of the contiguous Layer to agregate
    ! +  -----------------------------------------

    ! +   *********
    call SISVAT_zCr
    ! +   *********

    ! +--Assign the 2 Layers to agregate
    ! +  -------------------------------

    do ikl = 1, klonv
        isn = i_thin(ikl)
        if(LIndsv(ikl) > 0) isn = min(isn, nsno - 1) !cXF
        isagr1(ikl) = istoSV(ikl, isn)
        isagr2(ikl) = istoSV(ikl, isn + LIndsv(ikl))
        dzagr1(ikl) = dzsnSV(ikl, isn)
        dzagr2(ikl) = dzsnSV(ikl, isn + LIndsv(ikl))
        T_agr1(ikl) = TsisSV(ikl, isn)
        T_agr2(ikl) = TsisSV(ikl, isn + LIndsv(ikl))
        roagr1(ikl) = ro__SV(ikl, isn)
        roagr2(ikl) = ro__SV(ikl, isn + LIndsv(ikl))
        etagr1(ikl) = eta_SV(ikl, isn)
        etagr2(ikl) = eta_SV(ikl, isn + LIndsv(ikl))
        G1agr1(ikl) = G1snSV(ikl, isn)
        G1agr2(ikl) = G1snSV(ikl, isn + LIndsv(ikl))
        G2agr1(ikl) = G2snSV(ikl, isn)
        G2agr2(ikl) = G2snSV(ikl, isn + LIndsv(ikl))
        agagr1(ikl) = agsnSV(ikl, isn)
        agagr2(ikl) = agsnSV(ikl, isn + LIndsv(ikl))
        LstLay = min(1, max(0, isnoSV(ikl) - 1))
        Agrege(ikl) = min(1, &
                          max(0, &
                              NLaysv(ikl) + isnoSV(ikl) - nsno &
                              + NLay_s(ikl)) &
                          * LstLay)

        if(isnoSV(ikl) > 3) then
            ! surface layers> 2-5-10 ! XF 04/07/2019
            if(dzsnSV(ikl, max(1, isnoSV(ikl) - 0)) > 0.02 .or. &
               dzsnSV(ikl, max(1, isnoSV(ikl) - 1)) > 0.05 .or. &
               dzsnSV(ikl, max(1, isnoSV(ikl) - 2)) > 0.10 .or. &
               dzsnSV(ikl, max(1, isnoSV(ikl) - 3)) > 0.30 .or. &
               dzsnSV(ikl, 1) > 5. .or. dzsnSV(ikl, 2) > 5.) then
                Agrege(ikl) = min(1, &
                                  ! nsno-1 layers ma
                                  max(0, NLaysv(ikl) + isnoSV(ikl) + 1 - nsno &
                                      + NLay_s(ikl)) * LstLay)
            endif
        endif

        isnoSV(ikl) = isnoSV(ikl) &
                      - (1 - LstLay) * max(zero, &
                                           sign(unun, eps_21 &
                                                - dzsnSV(ikl, 1)))
        isnoSV(ikl) = max(0, isnoSV(ikl))

        WEagre(ikl) = 0.
    enddo

    do isn = 1, nsno
        do ikl = 1, klonv
            WEagre(ikl) = WEagre(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * min(1, max(0, i_thin(ikl) + 1 - isn))
        enddo
    enddo

#ifdef vz
    ! +--Layers Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    write(41, 4120)
4120 format(' Agregation of too MUCH Layers')
    write(41, 411)(100.*dz_ref(isn), isn=1, nsno)
    write(41, 412)(100.*dzwdif(isn), isn=1, nsno)
    write(41, 413)(100.*dzsnSV(1, isn), isn=1, nsno)
    write(41, 414)(isn, isn=1, nsno)
    write(41, 4111) isnoSV(1)
    write(41, 4112) i_thin(1)
    write(41, 4113) LIndsv(1)
    write(41, 4114) Agrege(1)
#endif

#ifdef vp
    ! +--Snow Properties Agregation: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    write(47, 474)(G1snSV(1, isn), isn=1, isnoSV(1))
474 format('Before _zAg2: G1 = ', 10f8.1,(/, 19x, 10f8.1))
    write(47, 472)(G2snSV(1, isn), isn=1, isnoSV(1))
#endif

    ! +--Agregates
    ! +  ---------

    ! +  ***************
    call SISVAT_zAg &
        (isagr1, isagr2, WEagre &
         , dzagr1, dzagr2, T_agr1, T_agr2 &
         , roagr1, roagr2, etagr1, etagr2 &
         , G1agr1, G1agr2, G2agr1, G2agr2 &
         , agagr1, agagr2, Agrege &
         )
    ! +  ***************

    ! +--Rearranges the Layers
    ! +  ---------------------

    ! +--New (agregated) Snow layer
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isn = i_thin(ikl)
        isn = min(isn, isn + LIndsv(ikl))
        isnoSV(ikl) = max(0., isnoSV(ikl) - Agrege(ikl))
        iiceSV(ikl) = iiceSV(ikl) &
                      - max(0, sign(1, iiceSV(ikl) - isn + icemix)) &
                      * Agrege(ikl) &
                      * max(0, sign(1, iiceSV(ikl) - 1))
        istoSV(ikl, isn) = (1.-Agrege(ikl)) * istoSV(ikl, isn) &
                           + Agrege(ikl) * isagr1(ikl)
        dzsnSV(ikl, isn) = (1.-Agrege(ikl)) * dzsnSV(ikl, isn) &
                           + Agrege(ikl) * dzagr1(ikl)
        TsisSV(ikl, isn) = (1.-Agrege(ikl)) * TsisSV(ikl, isn) &
                           + Agrege(ikl) * T_agr1(ikl)
        ro__SV(ikl, isn) = (1.-Agrege(ikl)) * ro__SV(ikl, isn) &
                           + Agrege(ikl) * roagr1(ikl)
        eta_SV(ikl, isn) = (1.-Agrege(ikl)) * eta_SV(ikl, isn) &
                           + Agrege(ikl) * etagr1(ikl)
        G1snSV(ikl, isn) = (1.-Agrege(ikl)) * G1snSV(ikl, isn) &
                           + Agrege(ikl) * G1agr1(ikl)
        G2snSV(ikl, isn) = (1.-Agrege(ikl)) * G2snSV(ikl, isn) &
                           + Agrege(ikl) * G2agr1(ikl)
        agsnSV(ikl, isn) = (1.-Agrege(ikl)) * agsnSV(ikl, isn) &
                           + Agrege(ikl) * agagr1(ikl)
    enddo

    ! +--Above
    ! +  ^^^^^
    do ikl = 1, klonv
        isn1(ikl) = max(i_thin(ikl), i_thin(ikl) + LIndsv(ikl))
    enddo
    do i = 1, nsno - 1
        do ikl = 1, klonv
            staggr = min(1, max(0, i + 1 - isn1(ikl)))
            istoSV(ikl, i) = (1.-staggr) * istoSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * istoSV(ikl, i) &
                                         + Agrege(ikl) * istoSV(ikl, i + 1))
            dzsnSV(ikl, i) = (1.-staggr) * dzsnSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * dzsnSV(ikl, i) &
                                         + Agrege(ikl) * dzsnSV(ikl, i + 1))
            TsisSV(ikl, i) = (1.-staggr) * TsisSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * TsisSV(ikl, i) &
                                         + Agrege(ikl) * TsisSV(ikl, i + 1))
            ro__SV(ikl, i) = (1.-staggr) * ro__SV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * ro__SV(ikl, i) &
                                         + Agrege(ikl) * ro__SV(ikl, i + 1))
            eta_SV(ikl, i) = (1.-staggr) * eta_SV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * eta_SV(ikl, i) &
                                         + Agrege(ikl) * eta_SV(ikl, i + 1))
            G1snSV(ikl, i) = (1.-staggr) * G1snSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * G1snSV(ikl, i) &
                                         + Agrege(ikl) * G1snSV(ikl, i + 1))
            G2snSV(ikl, i) = (1.-staggr) * G2snSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * G2snSV(ikl, i) &
                                         + Agrege(ikl) * G2snSV(ikl, i + 1))
            agsnSV(ikl, i) = (1.-staggr) * agsnSV(ikl, i) &
                             + staggr * ((1.-Agrege(ikl)) * agsnSV(ikl, i) &
                                         + Agrege(ikl) * agsnSV(ikl, i + 1))
        enddo
    enddo

    do ikl = 1, klonv
        isn = min(isnoSV(ikl) + 1, nsno)
        istoSV(ikl, isn) = (1.-Agrege(ikl)) * istoSV(ikl, isn)
        dzsnSV(ikl, isn) = (1.-Agrege(ikl)) * dzsnSV(ikl, isn)
        TsisSV(ikl, isn) = (1.-Agrege(ikl)) * TsisSV(ikl, isn)
        ro__SV(ikl, isn) = (1.-Agrege(ikl)) * ro__SV(ikl, isn)
        eta_SV(ikl, isn) = (1.-Agrege(ikl)) * eta_SV(ikl, isn)
        G1snSV(ikl, isn) = (1.-Agrege(ikl)) * G1snSV(ikl, isn)
        G2snSV(ikl, isn) = (1.-Agrege(ikl)) * G2snSV(ikl, isn)
        agsnSV(ikl, isn) = (1.-Agrege(ikl)) * agsnSV(ikl, isn)
    enddo

#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if(kSV_v1 > 0 .and. lSV_v1 == 3) then
        write(6, 5992) i_thin(kSV_v1)
5992    format(/, 'Secnd Agregation / Layer', i3, &
                /, '  i', 11x, 'T', 9x, 'rho', 10x, 'dz', 11x, 'H')
        write(6, 5995)(isn, TsisSV(kSV_v1, isn), ro__SV(kSV_v1, isn) &
                       , dzsnSV(kSV_v1, isn), istoSV(kSV_v1, isn), &
                       isn=isnoSV(kSV_v1), 1, -1)
    endif
#endif

#ifdef vp
    write(47, 475)(G1snSV(1, isn), isn=1, isnoSV(1))
475 format('At End _zSn : G1 = ', 10f8.1,(/, 19x, 10f8.1))
    write(47, 472)(G2snSV(1, isn), isn=1, isnoSV(1))
#endif

    ! +--Search new Ice/Snow Interface
    ! +  =============================
    do ikl = 1, klonv
        iiceSV(ikl) = 0
    enddo
    do ikl = 1, klonv
        do isn = 1, isnoSV(ikl)
            OK_ICE = max(zero, sign(unun, ro__SV(ikl, isn) - ro_ice + 20.)) &
                     * max(zero, sign(unun, dzsnSV(ikl, isn) - epsi))
            iiceSV(ikl) = (1.-OK_ICE) * iiceSV(ikl) + OK_ICE * isn
        enddo
    enddo

    return
end
