#include "MAR_pp.def"
subroutine SISVAT_TVg(ETVg_d)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_TVg                                13-09-2003  MAR |
    ! |   subroutine SISVAT_TVg computes the Canopy Energy Balance             |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   ivgtSV   = 0,...,12:   Vegetation Type                      |
    ! |   ^^^^^               0:          Water, Solid or Liquid               |
    ! |            isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |                                                                        |
    ! |   INPUT:   sol_SV   : Downward Solar Radiation                  [W/m2] |
    ! |   ^^^^^    IRd_SV   : Surface  Downward Longwave Radiation      [W/m2] |
    ! |            TaT_SV   : SBL Top  Temperature                         [K] |
    ! |            rhT_SV   : SBL Top  Air  Density                    [kg/m3] |
    ! |            QaT_SV   : SBL Top  Specific  Humidity              [kg/kg] |
    ! |            psivSV   : Leaf     Water     Potential                 [m] |
    ! |            IRs_SV   : Soil     IR Flux  (previous time step)    [W/m2] |
    ! |            dt__SV   : Time     Step                                [s] |
    ! |                                                                        |
    ! |            SoCasv   : Absorbed Solar Radiation by Canopy (Normaliz)[-] |
    ! |            tau_sv   : Fraction of Radiation transmitted by Canopy  [-] |
    ! |            Evg_sv   : Soil+Vegetation Emissivity                   [-] |
    ! |            Eso_sv   : Soil+Snow       Emissivity                   [-] |
    ! |            rah_sv   : Aerodynamic Resistance for Heat            [s/m] |
    ! |            Sigmsv   : Canopy Ventilation Factor                    [-] |
    ! |            LAI_sv   : Leaf Area  Index                             [-] |
    ! |            LAIesv   : Leaf Area  Index (effective / transpiration) [-] |
    ! |            glf_sv   : Green Leaf Fraction of NOT fallen Leaves     [-] |
    ! |            rrMxsv   : Canopy Maximum Intercepted Rain          [kg/m2] |
    ! |                                                                        |
    ! |   INPUT /  TvegSV   : Canopy   Temperature                         [K] |
    ! |   OUTPUT:  rrCaSV   : Canopy     Water     Content             [kg/m2] |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   OUTPUT:  IRv_sv   : Vegetation IR Flux                        [W/m2] |
    ! |   ^^^^^^   HSv_sv   : Sensible Heat Flux                        [W/m2] |
    ! |            HLv_sv   : Latent   Heat Flux                        [W/m2] |
    ! |            Evp_sv   : Evaporation                              [kg/m2] |
    ! |            EvT_sv   : Evapotranspiration                       [kg/m2] |
    ! |            ETVg_d   : Vegetation  Energy Power Forcing          [W/m2] |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |                                                                        |
    ! |   METHOD: The Newton-Raphson Scheme is preferable                      |
    ! |   ^^^^^^  when computing over a long time step the heat content        |
    ! |           of a medium having a very small or zero heat capacity.       |
    ! |           This is to handle strong non linearities arising             |
    ! |           in conjunction with rapid temperature variations.            |
    ! |                                                                        |
    ! | # OPTIONS: #NN: Newton-Raphson Increment not added in last Iteration   |
    ! | # ^^^^^^^                                                              |
    ! +------------------------------------------------------------------------+

    use marphy
    use mar_sv
    use mardsv
    use marxsv
    use marysv

    implicit none

    ! +--Global Variables
    ! +  ================

    ! +--OUTPUT
    ! +  ------

    real ETVg_d(klonv)                 ! VegetationPower, Forcing
#ifdef NC
    ! +--OUTPUT for Stand Alone NetCDF File
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    ! SOsoKL : Absorbed Solar Radiation
    real SOsoKL(klonv)
    ! IRsoKL : Absorbed IR Radiation
    real IRsoKL(klonv)
    ! HSsoKL : Absorbed Sensible Heat Flux
    real HSsoKL(klonv)
    ! HLsoKL : Absorbed Latent Heat Flux
    real HLsoKL(klonv)
    ! HLs_KL : Evaporation
    real HLs_KL(klonv)
    ! HLv_KL : Transpiration
    real HLv_KL(klonv)
    common / DumpNC / SOsoKL, IRsoKL, HSsoKL, HLsoKL, HLs_KL, HLv_KL
#endif

    ! +--Internal Variables
    ! +  ==================
    ! ikl : Grid Point Index
    integer ikl
    ! nitmax, nit : Iterations Counter
    integer nitmax, nit
    ! d_Tveg : Canopy Temperat. Increment
    real d_Tveg
    ! dTvMAX : Canopy Temperat. Increment MAX
    real dTvMAX
    ! dHvdTv : Derivativ.of Canopy Energ.Budg.
    real dHvdTv
    ! Hv_Tv0 : Imbalance of Canopy Energ.Budg.
    real Hv_Tv0
    ! Hv_MAX : MAX Imbal.of Canopy Energ.Budg.
    real Hv_MAX
    ! Hv_MIN : MIN Imbal.of Canopy Energ.Budg.
    real Hv_MIN
    ! Hswich : Newton-Raphson         Switch
    real Hswich
    ! Tveg_0 : Canopy Temperature, Previous t
    real Tveg_0(klonv)
    ! tau_Ca : Canopy IR Radiation Absorption
    real tau_Ca
    ! IR_net : InfraRed NET(t)
    real IR_net
    ! dIRdTv : InfraRed NET(t), Derivative(t)
    real dIRdTv(klonv)
    ! dHSdTv : Sensible Heat FL. Derivative(t)
    real dHSdTv(klonv)
    ! dHLdTv : Latent Heat FL. Derivative(t)
    real dHLdTv(klonv)
#ifdef HC
    ! dHCdTv : Heat Storage
    real dHCdTv(klonv)
#endif
    ! EvFrac : Condensat./Transpirat. Switch
    real EvFrac
    ! SnoMsk : Canopy Snow Switch
    real SnoMsk
    ! den_qs, arg_qs, qsatvg : Canopy Saturat. Spec. Humidity
    real den_qs, arg_qs, qsatvg
    ! dqs_dT : d(qsatvg)/dTv
    real dqs_dT
    ! FacEvp, FacEvT, Fac_Ev : Evapo(transpi)ration Factor
    real FacEvp, FacEvT, Fac_Ev
    ! dEvpdT, dEvTdT : Evapo(transpi)ration Derivative
    real dEvpdT(klonv), dEvTdT(klonv)
    ! F_Stom : Funct.  (Leaf Water Potential)
    real F_Stom
    ! R0Stom : Minimum Stomatal Resistance
    real R0Stom
    ! R_Stom : Stomatal Resistance
    real R_Stom
    ! LAI_OK : 1. ==>  Leaves   exist
    real LAI_OK
    ! rrCaOK, snCaOK, dEvpOK : Positive Definiteness Correct.
    real rrCaOK, snCaOK, dEvpOK

    ! +--Internal DATA
    ! +  =============
    ! nitmax : Maximum  Iterations    Number
    data nitmax/5/
    ! dTvMAX : Canopy Temperat. Increment MAX
    data dTvMAX/5./
    ! Hv_MIN : MIN Imbal. of Surf.Energy Budg.
    data Hv_MIN/0.1/
    ! SnoMsk : Canopy Snow Switch    (Default)
    data SnoMsk/0.0/

    ! +--Newton-Raphson Scheme
    ! +  =====================
    nit = 0
101 continue
    nit = nit + 1
    HV_MAX = 0.
    ! +--Temperature of the Previous Time Step
    ! +  -------------------------------------
    do ikl = 1, klonv
        Tveg_0(ikl) = TvegSV(ikl)
        ! +--IR Radiation Absorption
        ! +  -----------------------
        ! Canopy Absorption
        tau_Ca = 1.-tau_sv(ikl)
        ! Downward IR (OUT) + Upward IR (OUT)
        IRv_sv(ikl) = -2.0 * Evg_sv(ikl) * stefan &
                      * TvegSV(ikl) * TvegSV(ikl) &
                      * TvegSV(ikl) * TvegSV(ikl)
        ! Downward IR (OUT) + Upward IR (OUT)
        dIRdTv(ikl) = &
            -Evg_sv(ikl) * &
            8.*stefan * TvegSV(ikl) * TvegSV(ikl) &
            * TvegSV(ikl)
        ! Downward IR (IN) - Upward IR (IN) + IR (OUT)
        IR_net = tau_Ca * (Evg_sv(ikl) * IRd_SV(ikl) &
                           - IRs_SV(ikl) &
                           + IRv_sv(ikl))
        ! +--Sensible Heat Flux
        ! +  ------------------
        ! Derivative, t(n)
        dHSdTv(ikl) = rhT_SV(ikl) * Sigmsv(ikl) * Cp &
                      / rah_sv(ikl)
        ! Value, t(n)
        HSv_sv(ikl) = dHSdTv(ikl) &
                      * (TaT_SV(ikl) - TvegSV(ikl))
        ! +--Latent   Heat Flux
        ! +  ------------------

        ! +--Canopy Saturation Specific Humidity
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        den_qs = TvegSV(ikl) - 35.8
        arg_qs = 17.27 * (TvegSV(ikl) - 273.16) / den_qs
        qsatvg = .0038 * exp(arg_qs)
        dqs_dT = qsatvg * 4099.2 / (den_qs * den_qs)

        ! +--Canopy Stomatal Resistance
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! Min Stomatal R.
        R0Stom = min(StodSV(ivgtSV(ikl)) &
                     / max(epsi, glf_sv(ikl)), StxdSV)
        ! F(Leaf Wat.Pot.) DR97, eqn. 3.22
        F_Stom = pscdSV / max(pscdSV - psivSV(ikl), epsi)
        ! Can.Stomatal R. DR97, eqn. 3.21
        R_Stom = (R0Stom / max(LAIesv(ikl), R0Stom / StxdSV)) &
                 * F_Stom

        ! +--Evaporation / Evapotranspiration
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        SnoMsk = max(zero, sign(unun, snCaSV(ikl) - eps_21))
        ! Condensation/
        EvFrac = max(zero, sign(unun, QaT_SV(ikl) - qsatvg))
        ! Transpiration Switch
        EvFrac = EvFrac &
                 + (1.-EvFrac) * ((1 - SnoMsk) * rrCaSV(ikl) &
                                  / rrMxsv(ikl) &
                                  + SnoMsk * min(unun, snCaSV(ikl) &
                                                 / rrMxsv(ikl)))
        ! Idem,  Factor
        Fac_Ev = rhT_SV(ikl) * Sigmsv(ikl)
        FacEvp = Fac_Ev * EvFrac / rah_sv(ikl)
        ! Evaporation
        Evp_sv(ikl) = FacEvp * (qsatvg - QaT_SV(ikl))
        ! Evp Derivative
        dEvpdT(ikl) = FacEvp * dqs_dT
        FacEvt = Fac_Ev * (1.-EvFrac) / (rah_sv(ikl) &
                                         + R_Stom * Sigmsv(ikl))
        ! EvapoTranspir.
        EvT_sv(ikl) = FacEvt * (qsatvg - QaT_SV(ikl))
        ! EvT Derivative
        dEvTdT(ikl) = FacEvt * dqs_dT
        ! Latent   Heat  (Subli.Contrib.)
        HLv_sv(ikl) = -Lv_H2O * (Evp_sv(ikl) + EvT_sv(ikl)) &
                      - Lf_H2O * Evp_sv(ikl) * SnoMsk
        dHLdTv(ikl) = Lv_H2O * (dEvpdT(ikl) + dEvTdT(ikl)) &
                      + Lf_H2O * dEvpdT(ikl) * SnoMsk
#ifdef HC
        ! Heat  Storage
        dHCdTv(ikl) = Cn_dSV * snCaSV(ikl) / dt__SV
#endif

        ! +--Imbalance  of the Canopy  Energy Budget
        ! +  ---------------------------------------
        ! NO Budget if no Leaves
        LAI_OK = max(zero, &
                     sign(unun, LAI_sv(ikl) - eps_21))
        ! Absorbed Solar
        ! NET      IR
        ! Sensible Heat
        ! Latent   Heat
        Hv_Tv0 = (SoCasv(ikl) * sol_SV(ikl) &
                  + IR_net &
                  + HSv_sv(ikl) &
                  + HLv_sv(ikl) &
                  ) * LAI_OK
        ! Veg.Energ.Bal.
        ETVg_d(ikl) = Hv_Tv0
        ! +
        Hswich = unun
#ifdef NN
        ! Newton-Raphson Switch
        Hswich = max(zero, &
                     sign(unun, abs(Hv_Tv0) &
                          - Hv_MIN))
#endif

        ! +--Derivative of the Canopy  Energy Budget
        ! +  ---------------------------------------

        dHvdTv = dIRdTv(ikl) * max(eps_21, tau_Ca) &
                 - dHSdTv(ikl) &
                 - dHLdTv(ikl)
#ifdef HC
        dHvdTv = dHvdTv - dHCdTv(ikl)
#endif

        ! +--Update Canopy and Surface/Canopy Temperatures
        ! +  ---------------------------------------------

        d_Tveg = Hv_Tv0 / dHvdTv
        ! Increment Limitor
        d_Tveg = sign(unun, d_Tveg) &
                 * min(abs(d_Tveg), dTvMAX)
        ! Newton-Raphson
        TvegSV(ikl) = TvegSV(ikl) - Hswich * d_Tveg
        Hv_MAX = max(Hv_MAX, abs(Hv_Tv0))

        ! +--Update Vegetation Fluxes
        ! +  ------------------------
#ifdef NN
        ! Emitted  IR
        IRv_sv(ikl) = IRv_sv(ikl) - dIRdTv(ikl) * d_Tveg
        ! Sensible Heat
        HSv_sv(ikl) = HSv_sv(ikl) + dHSdTv(ikl) * d_Tveg
        ! Evapotranspir.
        Evp_sv(ikl) = Evp_sv(ikl) - dEvpdT(ikl) * d_Tveg
        ! Evapotranspir.
        EvT_sv(ikl) = EvT_sv(ikl) - dEvTdT(ikl) * d_Tveg
        ! Latent   Heat
        HLv_sv(ikl) = HLv_sv(ikl) + dHLdTv(ikl) * d_Tveg
#endif
        ! +
        IRv_sv(ikl) = IRv_sv(ikl) * LAI_OK
        HSv_sv(ikl) = HSv_sv(ikl) * LAI_OK
        Evp_sv(ikl) = Evp_sv(ikl) * LAI_OK
        EvT_sv(ikl) = EvT_sv(ikl) * LAI_OK
        HLv_sv(ikl) = HLv_sv(ikl) * LAI_OK
    enddo

#ifdef IX
    if(nit < nitmax) go to 101
#endif
    if(Hv_MAX > Hv_MIN .and. nit < nitmax) go to 101

    do ikl = 1, klonv
        ! Emitted  IR
        IRv_sv(ikl) = IRv_sv(ikl) &
                      + dIRdTv(ikl) * (TvegSV(ikl) - Tveg_0(ikl))
        ! Sensible Heat
        HSv_sv(ikl) = HSv_sv(ikl) &
                      - dHSdTv(ikl) * (TvegSV(ikl) - Tveg_0(ikl))
        ! Evaporation
        Evp_sv(ikl) = Evp_sv(ikl) &
                      + dEvpdT(ikl) * (TvegSV(ikl) - Tveg_0(ikl))
        ! Transpiration
        EvT_sv(ikl) = EvT_sv(ikl) &
                      + dEvTdT(ikl) * (TvegSV(ikl) - Tveg_0(ikl))
        ! Latent   Heat
        HLv_sv(ikl) = HLv_sv(ikl) &
                      - dHLdTv(ikl) * (TvegSV(ikl) - Tveg_0(ikl))

        ! +--OUTPUT for Stand Alone NetCDF File
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef NC
        HLv_KL(ikl) = HLv_sv(ikl)
#endif

        ! +--Update Canopy Water Content
        ! +  ---------------------------

        rrCaSV(ikl) = rrCaSV(ikl) - (1.-SnoMsk) * Evp_sv(ikl) * dt__SV
        snCaSV(ikl) = snCaSV(ikl) - SnoMsk * Evp_sv(ikl) * dt__SV

        ! +--Correction for Positive Definiteness (see WKarea/EvpVeg/EvpVeg.f)
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        rrCaOK = max(rrCaSV(ikl), 0.)
        snCaOK = max(snCaSV(ikl), 0.)
        dEvpOK = (rrCaOK - rrCaSV(ikl) &
                  + snCaOK - snCaSV(ikl)) / dt__SV

        ! Evaporation
        Evp_sv(ikl) = Evp_sv(ikl) - dEvpOK
        ! Latent   Heat
        HLv_sv(ikl) = HLv_sv(ikl) &
                      + (1.-SnoMsk) * Lv_H2O * dEvpOK &
                      + SnoMsk * (Lv_H2O + Lf_H2O) * dEvpOK

        rrCaSV(ikl) = rrCaOK
        snCaSV(ikl) = snCaOK

        wee_SV(ikl, 2) = wee_SV(ikl, 2) + dt__SV * EvT_sv(ikl)
        wee_SV(ikl, 1) = wee_SV(ikl, 1) + dt__SV * Evp_sv(ikl)

    enddo

    return
endsubroutine SISVAT_TVg
