#include "MAR_pp.def"
subroutine SISVAT_TSo(ETSo_0, ETSo_1, ETSo_d)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_TSo                                31-12-2024  MAR |
    ! |   subroutine SISVAT_TSo computes the Soil/Snow Energy Balance          |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   isotSV   = 0,...,11:   Soil       Type                      |
    ! |   ^^^^^               0:          Water, Solid or Liquid               |
    ! |            isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |            dQa_SV   = Limitation of  Water Vapor  Turbulent Flux       |
    ! |                                                                        |
    ! |   INPUT:   sol_SV   : Downward Solar Radiation                  [W/m2] |
    ! |   ^^^^^    IRd_SV   : Surface Downward  Longwave   Radiation    [W/m2] |
    ! |            za__SV   : SBL Top    Height                            [m] |
    ! |            VV__SV   : SBL Top    Wind Speed                      [m/s] |
    ! |            TaT_SV   : SBL Top    Temperature                       [K] |
    ! |            rhT_SV   : SBL Top    Air  Density                  [kg/m3] |
    ! |            QaT_SV   : SBL Top    Specific  Humidity            [kg/kg] |
    ! |            LSdzsv   : Vertical   Discretization Factor             [-] |
    ! |                     =    1. Soil                                       |
    ! |                     = 1000. Ocean                                      |
    ! |            dzsnSV   : Snow Layer Thickness                         [m] |
    ! |            ro__SV   : Snow/Soil  Volumic Mass                  [kg/m3] |
    ! |            eta_SV   : Soil Water Content                       [m3/m3] |
    ! |            dt__SV   : Time Step                                    [s] |
    ! |                                                                        |
    ! |            SoSosv   : Absorbed Solar Radiation by Surfac.(Normaliz)[-] |
    ! |            IRv_sv   : Vegetation  IR Radiation                  [W/m2] |
    ! |            tau_sv   : Fraction of Radiation transmitted by Canopy  [-] |
    ! |            Evg_sv   : Soil+Vegetation Emissivity                   [-] |
    ! |            Eso_sv   : Soil+Snow       Emissivity                   [-] |
    ! |            rah_sv   : Aerodynamic Resistance for Heat            [s/m] |
    ! |            Lx_H2O   : Latent Heat of Vaporization/Sublimation   [J/kg] |
    ! |            Sigmsv   : Canopy Ventilation Factor                    [-] |
    ! |            sEX_sv   : Verticaly Integrated Extinction Coefficient  [-] |
    ! |                                                                        |
    ! |   INPUT /  TsisSV   : Soil/Ice Temperatures (layers -nsol,-nsol+1,..,0)|
    ! |   OUTPUT:           & Snow     Temperatures (layers  1,2,...,nsno) [K] |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   OUTPUT:  IRs_SV   : Soil      IR Radiation                    [W/m2] |
    ! |   ^^^^^^   HSs_sv   : Sensible  Heat Flux                       [W/m2] |
    ! |            HLs_sv   : Latent    Heat Flux                       [W/m2] |
    ! |            ETSo_0   : Snow/Soil Energy Power, before Forcing    [W/m2] |
    ! |            ETSo_1   : Snow/Soil Energy Power, after  Forcing    [W/m2] |
    ! |            ETSo_d   : Snow/Soil Energy Power         Forcing    [W/m2] |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |                                                                        |
    ! |   METHOD: NO   Skin Surface Temperature                                |
    ! |   ^^^^^^  Semi-Implicit Crank Nicholson Scheme                         |
    ! |                                                                        |
    ! | # OPTIONS: #E0: Energy Budget Verification                             |
    ! | # ^^^^^^^  #kd: KDsvat Option:NO Flux  Limitor     on HL               |
    ! | #          #KD: KDsvat Option:Explicit Formulation of HL               |
    ! | #          #NC: OUTPUT for Stand Alone NetCDF File                     |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mar_sv
    use mardsv
    use mar0sv
    use marxsv
    use marysv

    implicit none

    ! +--Global Variables
    ! +  ================

    ! +--OUTPUT
    ! +  ------
    ! Soil/Snow Power, before Forcing
    real ETSo_0(klonv)
    ! Soil/Snow Power, after  Forcing
    real ETSo_1(klonv)
    ! Soil/Snow Power, Forcing
    real ETSo_d(klonv)

    ! +--Internal Variables
    ! +  ==================

    real zt(klonv), ti(klonv), ww
    integer ikl, isl, jsl, ist, izt(klonv)
    ! ist__s, ist__w : Soil/Water  Body Identifier
    integer ist__s, ist__w
    ! islsgn : Soil/Snow Surfac.Identifier
    integer islsgn
    ! eps__3 : Arbitrary Low Number
    real eps__3
    ! etaMid, psiMid : Layer Interface's Humidity
    real etaMid, psiMid
    ! mu_eta : Soil thermal Conductivity
    real mu_eta
    ! mu_exp : arg Soil thermal Conductivity
    real mu_exp
    ! mu_min : Min Soil thermal Conductivity
    real mu_min
    ! mu_max : Max Soil thermal Conductivity
    real mu_max
    ! mu_sno, mu_aux : Snow thermal Conductivity
    real mu_sno(klonv), mu_aux
    ! mu__dz : mu_(eta,sno) / dz
    real mu__dz(klonv, -nsol:nsno + 1)
    ! dtC_sv : dt / C
    real dtC_sv(klonv, -nsol:nsno)
    ! IRs__D : UpwardIR Previous Iter.Contr.
    real IRs__D(klonv)
    ! dIRsdT : UpwardIR T Derivat.
    real dIRsdT(klonv)
    ! f_HSHL : Factor common to HS and HL
    real f_HSHL(klonv)
    ! dRidTs : d(Rib)/d(Ts)
    real dRidTs(klonv)
    ! HS___D : Sensible Heat Flux Atm.Contr.
    real HS___D(klonv)
    ! f___HL :
    real f___HL(klonv)
    ! HL___D : Latent Heat Flux Atm.Contr.
    real HL___D(klonv)
    ! TSurf0 : Previous Surface Temperature
    real TSurf0(klonv), dTSurf
    ! qsatsg : Soil Saturat. Spec. Humidity
    real qsatsg(klonv)
    ! dqs_dT : d(qsatsg)/dTv
    real dqs_dT(klonv)
    ! Psi : 1st Soil Layer Water Potential
    real Psi(klonv)
    ! RHuSol : Soil Surface Relative Humidity
    real RHuSol(klonv)
    ! etaSol : Soil Surface Humidity
    real etaSol
    ! Elem_A, Elem_C : Diagonal Coefficients
    real Elem_A, Elem_C
    ! Diag_A : A Diagonal
    real Diag_A(klonv, -nsol:nsno)
    ! Diag_B : B Diagonal
    real Diag_B(klonv, -nsol:nsno)
    ! Diag_C : C Diagonal
    real Diag_C(klonv, -nsol:nsno)
    ! Term_D : Independant Term
    real Term_D(klonv, -nsol:nsno)
    ! Aux__P : P Auxiliary Variable
    real Aux__P(klonv, -nsol:nsno)
    ! Aux__Q : Q Auxiliary Variable
    real Aux__Q(klonv, -nsol:nsno)
    ! Ts_Min, Ts_Max : Temperature Limits
    real Ts_Min, Ts_Max
    ! Exist0 : Existing Layer Switch
    real Exist0
    ! psat_wat, psat_ice, sp, dzVap0 : computation of qsat
    real psat_wat, psat_ice, sp, dzVap0

    ! nt_srf, it_srf, itEuBk : HL Surface Scheme
    integer nt_srf, it_srf, itEuBk
    ! nt_srf = 10 before
    parameter(nt_srf=6)
    real agpsrf, xgpsrf, dt_srf, dt_ver
    real etaBAK(klonv)
    real etaNEW(klonv)
    real etEuBk(klonv)
    real fac_dt(klonv), faceta(klonv)
    real PsiArg(klonv), SHuSol(klonv)
#ifdef NC
    ! +--OUTPUT for Stand Alone NetCDF File
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    ! Absorbed Solar Radiation
    real SOsoKL(klonv)
    ! Absorbed IR    Radiation
    real IRsoKL(klonv)
    ! Absorbed Sensible Heat Flux
    real HSsoKL(klonv)
    ! Absorbed Latent   Heat Flux
    real HLsoKL(klonv)
    ! Evaporation
    real HLs_KL(klonv)
    ! Transpiration
    real HLv_KL(klonv)
    common / DumpNC / SOsoKL, IRsoKL, HSsoKL, HLsoKL, HLs_KL, HLv_KL
#endif

    ! +--Internal DATA
    ! +  =============
    ! eps__3 : Arbitrary    Low Number
    data eps__3/1.e-3/
    ! mu_exp : Soil Thermal Conductivity
    data mu_exp/-0.4343/
    ! mu_min : Min Soil Thermal Conductivity
    data mu_min/0.172/
    ! mu_max : Max Soil Thermal Conductivity
    data mu_max/2.000/
    ! Ts_Min : Temperature Minimum
    data Ts_Min/175./
    ! Ts_Max : Temperature Acceptable Maximum including   Snow Melt  Energy
    data Ts_Max/300./

    ! +--Heat Conduction Coefficient (zero in the Layers over the highest one)
    ! +  ===========================
    ! +                             ---------------- isl    eta_SV, rho C (isl)
    ! +
    ! +--Soil                       ++++++++++++++++        etaMid,    mu (isl)
    ! +  ----
    ! +                             ---------------- isl-1  eta_SV, rho C (isl-1)
    isl = -nsol
    do ikl = 1, klonv
        mu__dz(ikl, isl) = 0.
        ! dt / (dz X rho C)
        ! [s / (m.J/m3/K)]
        dtC_sv(ikl, isl) = dtz_SV2(isl) * dt__SV &
                           / ((rocsSV(isotSV(ikl)) &
                               + rcwdSV * eta_SV(ikl, isl)) &
                              * LSdzsv(ikl))
    enddo
    do isl = -nsol + 1, 0
        do ikl = 1, klonv
            ! Soil Type
            ist = isotSV(ikl)
            ! 1 => Soil
            ist__s = min(ist, 1)
            ! 1 => Water Body
            ist__w = 1 - ist__s
            ! eta at layers
            !     interface
            ! LSdzsv implicit
            etaMid = 0.5 * (dz_dSV(isl - 1) * eta_SV(ikl, isl - 1) &
                            + dz_dSV(isl) * eta_SV(ikl, isl)) &
                     / dzmiSV(isl)
            etaMid = max(etaMid, epsi)
            psiMid = psidSV(ist) &
                     * (etadSV(ist) / etaMid)**bCHdSV(ist)
            ! Soil Thermal Conductivity DR97 eq.3.31
            mu_eta = 3.82 * (psiMid)**mu_exp
            mu_eta = min(max(mu_eta, mu_min), mu_max)
            ! +
            ! Water Bodies Correction
            mu_eta = ist__s * mu_eta + ist__w * vK_dSV
            ! +
            mu__dz(ikl, isl) = mu_eta / (dzmiSV(isl) &
                                         * LSdzsv(ikl))
            ! dt / (dz X rho C)
            dtC_sv(ikl, isl) = dtz_SV2(isl) * dt__SV &
                               / ((rocsSV(isotSV(ikl)) &
                                   + rcwdSV * eta_SV(ikl, isl)) &
                                  * LSdzsv(ikl))
        enddo
    enddo

    ! +--Soil/Snow Interface
    ! +  -------------------

    ! +--Soil Contribution
    ! +  ^^^^^^^^^^^^^^^^^
    isl = 1
    do ikl = 1, klonv
        ist = isotSV(ikl)                       ! Soil Type
        ist__s = min(ist, 1)                           ! 1 => Soil
        ist__w = 1 - ist__s                            ! 1 => Water Body
        psiMid = psidSV(ist)                       ! Snow => Saturation
        mu_eta = 3.82 * (psiMid)**mu_exp       ! Soil Thermal
        mu_eta = min(max(mu_eta, mu_min), mu_max)      ! Conductivity
        ! +                                                       ! DR97 eq.3.31
        mu_eta = ist__s * mu_eta + ist__w * vK_dSV       ! Water Bodies

        ! +--Snow Contribution
        ! +  ^^^^^^^^^^^^^^^^^
        ! mu_sno :  Snow Heat Conductivity Coefficient [Wm/K]
        !           (Yen 1981, CRREL Rep., 81-10)
        mu_sno(ikl) = CdidSV &
                      * (ro__SV(ikl, isl) / ro_Wat)**1.88
        mu_sno(ikl) = max(epsi, mu_sno(ikl))

        ! +--Combined Heat Conductivity
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        mu__dz(ikl, isl) = 2./(dzsnSV(ikl, isl) / mu_sno(ikl) + LSdzsv(ikl) * dz_dSV(isl - 1) / mu_eta)

        ! +--Inverted Heat Capacity
        ! +  ^^^^^^^^^^^^^^^^^^^^^^
        ! dt / (dz X rho C)
        dtC_sv(ikl, isl) = dt__SV / max(epsi, dzsnSV(ikl, isl) * ro__SV(ikl, isl) * Cn_dSV)
    enddo

    ! +--Snow
    ! +  ----

    do ikl = 1, klonv
        do isl = 1, min(nsno, isnoSV(ikl) + 1)
            ro__SV(ikl, isl) = &
                ro__SV(ikl, isl) &
                * max(0, min(isnoSV(ikl) - isl + 1, 1))
        enddo
    enddo

    do ikl = 1, klonv
        do isl = 1, min(nsno, isnoSV(ikl) + 1)

            ! +--Combined Heat Conductivity
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
            mu_aux = CdidSV &
                     * (ro__SV(ikl, isl) / ro_Wat)**1.88
            ! Combined Heat Conductivity For upper Layer
            mu__dz(ikl, isl) = &
                2.*mu_aux * mu_sno(ikl) &
                / max(epsi, dzsnSV(ikl, isl) * mu_sno(ikl) &
                      + dzsnSV(ikl, isl - 1) * mu_aux)
            mu_sno(ikl) = mu_aux

            ! +--Inverted Heat Capacity
            ! +  ^^^^^^^^^^^^^^^^^^^^^^
            ! dt / (dz X rho C)
            dtC_sv(ikl, isl) = dt__SV / max(eps__3, &
                                            dzsnSV(ikl, isl) * ro__SV(ikl, isl) * Cn_dSV)
        enddo
    enddo

    ! +--Uppermost Effective Layer: NO conduction
    ! +  ----------------------------------------

    do ikl = 1, klonv
        mu__dz(ikl, isnoSV(ikl) + 1) = 0.0
    enddo

    ! +--Energy Budget (IN)
    ! +  ==================
    do ikl = 1, klonv
        ETSo_0(ikl) = 0.
    enddo
    do isl = -nsol, nsno
        do ikl = 1, klonv
            Exist0 = isl - isnoSV(ikl)
            Exist0 = 1.-max(zero, min(unun, Exist0))
            ETSo_0(ikl) = ETSo_0(ikl) &
                          + (TsisSV(ikl, isl) - TfSnow) * Exist0 &
                          / dtC_sv(ikl, isl)
        enddo
    enddo

    ! +--Tridiagonal Elimination: Set Up
    ! +  ===============================

    ! +--Soil/Snow Interior
    ! +  ^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        do isl = -nsol + 1, min(nsno - 1, isnoSV(ikl) + 1)
            Elem_A = dtC_sv(ikl, isl) * mu__dz(ikl, isl)
            Elem_C = dtC_sv(ikl, isl) * mu__dz(ikl, isl + 1)
            Diag_A(ikl, isl) = -Elem_A * Implic
            Diag_C(ikl, isl) = -Elem_C * Implic
            Diag_B(ikl, isl) = 1.0d+0 - Diag_A(ikl, isl) - Diag_C(ikl, isl)
            Term_D(ikl, isl) = Explic * (Elem_A * TsisSV(ikl, isl - 1) &
                                         + Elem_C * TsisSV(ikl, isl + 1)) &
                               + (1.0d+0 - Explic * (Elem_A + Elem_C)) * TsisSV(ikl, isl) &
                               + dtC_sv(ikl, isl) * sol_SV(ikl) * SoSosv(ikl) &
                               * (sEX_sv(ikl, isl + 1) &
                                  - sEX_sv(ikl, isl))
        enddo
    enddo

    ! +--Soil  lowest Layer
    ! +  ^^^^^^^^^^^^^^^^^^
    isl = -nsol
    do ikl = 1, klonv
        Elem_A = 0.
        Elem_C = dtC_sv(ikl, isl) * mu__dz(ikl, isl + 1)
        Diag_A(ikl, isl) = 0.
        Diag_C(ikl, isl) = -Elem_C * Implic
        Diag_B(ikl, isl) = 1.0d+0 - Diag_A(ikl, isl) - Diag_C(ikl, isl)
        Term_D(ikl, isl) = Explic * Elem_C * TsisSV(ikl, isl + 1) &
                           + (1.0d+0 - Explic * Elem_C) * TsisSV(ikl, isl) &
                           + dtC_sv(ikl, isl) * sol_SV(ikl) * SoSosv(ikl) &
                           * (sEX_sv(ikl, isl + 1) &
                              - sEX_sv(ikl, isl))
    enddo

    ! +--Snow highest Layer (dummy!)
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isl = min(isnoSV(ikl) + 1, nsno)
        Elem_A = dtC_sv(ikl, isl) * mu__dz(ikl, isl)
        Elem_C = 0.
        Diag_A(ikl, isl) = -Elem_A * Implic
        Diag_C(ikl, isl) = 0.
        Diag_B(ikl, isl) = 1.0d+0 - Diag_A(ikl, isl)
        Term_D(ikl, isl) = Explic * Elem_A * TsisSV(ikl, isl - 1) &
                           + (1.0d+0 - Explic * Elem_A) * TsisSV(ikl, isl) &
                           + dtC_sv(ikl, isl) * (sol_SV(ikl) * SoSosv(ikl) &
                                                 * (sEX_sv(ikl, isl + 1) &
                                                    - sEX_sv(ikl, isl)))
    enddo

    ! +--Surface: UPwardIR Heat Flux
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        ! - d(IR)/d(T)
        dIRsdT(ikl) = Eso_sv(ikl) * stefan * 4. &
                      *TsisSV(ikl, isl) &
                      * TsisSV(ikl, isl) &
                      * TsisSV(ikl, isl)
        IRs__D(ikl) = dIRsdT(ikl) * TsisSV(ikl, isl) * 0.75
#ifdef RC
        ! +--Surface: Richardson Number:   T Derivative
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        dRidTs(ikl) = -gravit * za__SV(ikl) &
                      * (1.-Sigmsv(ikl)) &
                      / (TaT_SV(ikl) * VV__SV(ikl) &
                         * VV__SV(ikl))
#endif
        ! +--Surface: Turbulent Heat Flux: Factors
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! common factor to  HS, HL
        f_HSHL(ikl) = rhT_SV(ikl) * (1.-Sigmsv(ikl)) / rah_sv(ikl)
        f___HL(ikl) = f_HSHL(ikl) * Lx_H2O(ikl)

        ! +--Surface: Sensible  Heat Flux: T Derivative
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        dSdTSV(ikl) = f_HSHL(ikl) * Cp                    !#- d(HS)/d(T)
        ! Richardson Nb. Correct.
#ifdef RC
        dSdTSV(ikl) = dSdTSV(ikl) &
                      * (1.0 - (TsisSV(ikl, isl) - TaT_SV(ikl)) &
                         * dRidTs(ikl) * dFh_sv(ikl) / rah_sv(ikl))
#endif
        HS___D(ikl) = dSdTSV(ikl) * TaT_SV(ikl)

        ! +--Surface: Latent Heat Flux: Saturation Specific Humidity
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        sp = (pst_SV(ikl) + ptopSV) * 10.
        psat_ice = 6.1070 * exp(6150.*(1./273.16 - 1./tsrf_SV(ikl)))
        psat_wat = 6.1078 * exp(5.138 * log(273.16 / tsrf_SV(ikl))) &
                   * exp(6827.*(1./273.16 - 1./tsrf_SV(ikl)))

        if(tsrf_SV(ikl) <= 273.15) then
            qsatsg(ikl) = 0.622 * psat_ice / (sp - 0.378 * psat_ice)
        else
            qsatsg(ikl) = 0.622 * psat_wat / (sp - 0.378 * psat_wat)
        endif
        fac_dt(ikl) = f_HSHL(ikl) / (ro_Wat * dz_dSV(0))
    enddo

    ! +--Surface: Latent Heat Flux: Surface Relative Humidity
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    xgpsrf = 1.05
    agpsrf = dt__SV * (1.0 - xgpsrf) &
             / (1.0 - xgpsrf**nt_srf)
    dt_srf = agpsrf
    dt_ver = 0.
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        ist = max(0, isotSV(ikl) - 100 * isnoSV(ikl)) ! 0 if H2O
        ist__s = min(1, ist)
        etaBAK(ikl) = max(epsi, eta_SV(ikl, isl))
        etaNEW(ikl) = etaBAK(ikl)
        etEuBk(ikl) = etaNEW(ikl)
    enddo
    if(ist__s == 1) then ! to reduce computer time
        do it_srf = 1, nt_srf
            dt_ver = dt_ver + dt_srf
            do ikl = 1, klonv
                faceta(ikl) = fac_dt(ikl) * dt_srf
#ifdef VX
                !    Limitation by Atm.Conten NO Limitation of Downw.Flux
                faceta(ikl) = faceta(ikl) &
                              / (1.+faceta(ikl) * dQa_SV(ikl))
                ! *max(0,sign(1.,qsatsg(ikl)-QaT_SV(ikl))))
#endif
            enddo
            do itEuBk = 1, 2
                do ikl = 1, klonv
                    ! 0 if    H2O
                    ist = max(0, isotSV(ikl) - 100 * isnoSV(ikl))
                    ! DR97, Eqn 3.34
                    Psi(ikl) = &
                        psidSV(ist) &
                        * (etadSV(ist) &
                           / max(etEuBk(ikl), epsi)) &
                        **bCHdSV(ist)
                    PsiArg(ikl) = 7.2E-5 * Psi(ikl)
                    RHuSol(ikl) = exp(-min(argmax, PsiArg(ikl)))
                    ! DR97, Eqn 3.15
                    SHuSol(ikl) = qsatsg(ikl) * RHuSol(ikl)
                    etEuBk(ikl) = &
                        (etaNEW(ikl) + faceta(ikl) * (QaT_SV(ikl) &
                                                      - SHuSol(ikl) &
                                                      * (1.-bCHdSV(ist) &
                                                         * PsiArg(ikl)))) &
                        / (1.+faceta(ikl) * SHuSol(ikl) &
                           * bCHdSV(ist) &
                           * PsiArg(ikl) &
                           / etaNEW(ikl))
                    etEuBk(ikl) = etEuBk(ikl) - Rootsv(ikl, 0) &
                                  * dt_srf / (Ro_Wat * dz_dSV(0))
                enddo
            enddo
            do ikl = 1, klonv
                etaNEW(ikl) = max(etEuBk(ikl), epsi)
            enddo
            dt_srf = dt_srf * xgpsrf
        enddo
    endif

    ! +--Surface: Latent Heat Flux: Soil/Water Surface Contributions
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        ! 0 if H2O
        ist = max(0, isotSV(ikl) - 100 * isnoSV(ikl))
        ! 1 if no H2O
        ist__s = min(1, ist)
        ! 1 if H2O
        ist__w = 1 - ist__s
        ! latent heat flux computation
        if(isotSV(ikl) > 1) then
            ! to avoid too high flux
            etaNEW(ikl) = max(etaNEW(ikl), 0.95 * etaBAK(ikl))
        endif
        HL___D(ikl) = (ist__s * ro_Wat * dz_dSV(0) * (etaNEW(ikl) - etaBAK(ikl)) / dt__SV &
                       + 1.1 * ist__w * f_HSHL(ikl) * (QaT_SV(ikl) - qsatsg(ikl))) * Lx_H2O(ikl)
                       ! 02/05/2025: +10% of the surface evaporation over ocean
        dzVap0 = dt__SV * HL___D(ikl) * min(isl, 1) &
                 / (Lx_H2O(ikl) * max(ro__SV(ikl, isl), epsi))

        if(isnoSV(ikl) > 0 .and. dzVap0 + dzsnSV(ikl, isl) <= 0) then
            ist__s = 0
            do while(dzVap0 + dzsnSV(ikl, isl) <= 0 .and. ist__s <= 10)
                ist__s = ist__s + 1
                HL___D(ikl) = HL___D(ikl) * 0.5
                !HL___D(ikl)            = HL___D(ikl) *0 ! for MAR-offline
                dzVap0 = dt__SV * HL___D(ikl) * min(isl, 1) &
                         / (Lx_H2O(ikl) * max(ro__SV(ikl, isl), epsi))
                print *, "sisvat_tso.f: HL___D too high on", ii__sv(ikl), jj__sv(ikl), nn__sv(ikl)
            enddo
        endif

#ifdef DL
        RHuSol(ikl) = (QaT_SV(ikl) - HL___D(ikl) / f___HL(ikl)) / qsatsg(ikl)
#endif

        ! +--Surface: Latent    Heat Flux: T Derivative
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        dLdTSV(ikl) = 0.
        ! - d(HL)/d(T)
#ifdef DL
        dLdTSV(ikl) = f___HL(ikl) * RHuSol(ikl) * dqs_dT(ikl)
        HL___D(ikl) = HL___D(ikl) + dLdTSV(ikl) * TsisSV(ikl, isl)
#endif
    enddo

    ! +--Surface: Tridiagonal Matrix Set Up
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        TSurf0(ikl) = TsisSV(ikl, isl)
        Elem_A = dtC_sv(ikl, isl) * mu__dz(ikl, isl)
        Elem_C = 0.
        Diag_A(ikl, isl) = -Elem_A * Implic
        Diag_C(ikl, isl) = 0.
        Diag_B(ikl, isl) = 1.0d+0 - Diag_A(ikl, isl)
        Diag_B(ikl, isl) = Diag_B(ikl, isl) &
                           ! Upw. Sol IR
                           + dtC_sv(ikl, isl) * (dIRsdT(ikl) &
                                                 ! HS/Surf.Contr.
                                                 + dSdTSV(ikl) &
                                                 ! HL/Surf.Contr.
                                                 + dLdTSV(ikl))
        Term_D(ikl, isl) = Explic * Elem_A * TsisSV(ikl, isl - 1) &
                           + (1.0d+0 - Explic * Elem_A) * TsisSV(ikl, isl)
        Term_D(ikl, isl) = Term_D(ikl, isl) &
                           ! Absorbed
                           + dtC_sv(ikl, isl) * (sol_SV(ikl) * SoSosv(ikl) &
                                                 ! Solar
                                                 * (sEX_sv(ikl, isl + 1) &
                                                    - sEX_sv(ikl, isl)) &
                                                 ! Down Atm IR
                                                 + tau_sv(ikl) * IRd_SV(ikl) * Eso_sv(ikl) &
                                                 ! Down Veg IR
                                                 - (1.0 - tau_sv(ikl)) * 0.5 * IRv_sv(ikl) &
                                                 ! Upw. Sol IR
                                                 + IRs__D(ikl) &
                                                 ! HS/Atmo.Contr.
                                                 + HS___D(ikl) &
                                                 ! HL/Atmo.Contr.
                                                 + HL___D(ikl))
    enddo

    ! +--Tridiagonal Elimination
    ! +  =======================

    ! +--Forward  Sweep
    ! +  ^^^^^^^^^^^^^^
    do ikl = 1, klonv
        Aux__P(ikl, -nsol) = Diag_B(ikl, -nsol)
        Aux__Q(ikl, -nsol) = -Diag_C(ikl, -nsol) / Aux__P(ikl, -nsol)
    enddo

    do ikl = 1, klonv
        do isl = -nsol + 1, min(nsno, isnoSV(ikl) + 1)
            Aux__P(ikl, isl) = Diag_A(ikl, isl) * Aux__Q(ikl, isl - 1) &
                               + Diag_B(ikl, isl)
            Aux__Q(ikl, isl) = -Diag_C(ikl, isl) / Aux__P(ikl, isl)
        enddo
    enddo

    do ikl = 1, klonv
        TsisSV(ikl, -nsol) = Term_D(ikl, -nsol) / Aux__P(ikl, -nsol)
    enddo

    do ikl = 1, klonv
        do isl = -nsol + 1, min(nsno, isnoSV(ikl) + 1)
            TsisSV(ikl, isl) = (Term_D(ikl, isl) &
                                - Diag_A(ikl, isl) * TsisSV(ikl, isl - 1)) &
                               / Aux__P(ikl, isl)
        enddo
    enddo

    ! +--Backward Sweep
    ! +  ^^^^^^^^^^^^^^
    zt = 0.; ti = 0; izt = 1
    do ikl = 1, klonv
        do isl = isnoSV(ikl), 1, -1
            zt(ikl) = zt(ikl) + dzsnSV(ikl, isl)
            ti(ikl) = ti(ikl) + dzsnSV(ikl, isl) * TsisSV(ikl, isl)
            if(zt(ikl) > 5) izt(ikl) = max(izt(ikl), isl)
        enddo
        ti(ikl) = min(271.15, ti(ikl) / max(0.01, zt(ikl)))
        do isl = min(nsno - 1, isnoSV(ikl) + 1), -nsol, -1
            TsisSV(ikl, isl) = Aux__Q(ikl, isl) * TsisSV(ikl, isl + 1) &
                               + TsisSV(ikl, isl)
            if(isl == 0 .and. isnoSV(ikl) == 0) then

                TsisSV(ikl, isl) = min(TaT_SV(ikl) + 30, TsisSV(ikl, isl))
                TsisSV(ikl, isl) = max(TaT_SV(ikl) - 30, TsisSV(ikl, isl))
#ifdef EU
                TsisSV(ikl, isl) = max(TaT_SV(ikl) - 25., TsisSV(ikl, isl))
                ! 2024/12/09: impacts on Tmax/Tmin
#endif
            endif
            if(zt(ikl) > 15 .and. isl <= 0) then ! ice sheet
                TsisSV(ikl, isl) = max(223.15, min(ti(ikl), TsisSV(ikl, isl)))
                eta_SV(ikl, isl) = epsi
            endif
        enddo
    enddo

    ! +--Temperature Limits (avoids problems in case of no Snow Layers)
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        dTSurf = TsisSV(ikl, isl) - TSurf0(ikl)
        ! 180.0 dgC/hr  = 0.05 dgC/s
        TsisSV(ikl, isl) = TSurf0(ikl) + sign(1., dTSurf) &
                           * min(abs(dTSurf), 5.e-2 * dt__SV)
    enddo
    do ikl = 1, klonv
        do isl = min(nsno, isnoSV(ikl) + 1), 1, -1
            TsisSV(ikl, isl) = max(Ts_Min, TsisSV(ikl, isl))
            TsisSV(ikl, isl) = min(Ts_Max, TsisSV(ikl, isl))
        enddo

        if(zt(ikl) > 15 .and. isnoSV(ikl) > 4) then
            ww = 3600.*24.*30./dt__SV ! 1 month
            do isl = 1, min(izt(ikl), isnoSV(ikl) - 1)
                if(TsisSV(ikl, isl + 1) < TsisSV(ikl, isl)) then
                    TsisSV(ikl, isl) = max(0.999 * TsisSV(ikl, isl), &
                                           min(1.001 * TsisSV(ikl, isl), &
                                               (TsisSV(ikl, isl) * dzsnSV(ikl, isl) * ww &
                                                + TsisSV(ikl, isl + 1) * dzsnSV(ikl, isl + 1)) &
                                               / (dzsnSV(ikl, isl) * ww &
                                                  + dzsnSV(ikl, isl + 1))))
                    TsisSV(ikl, isl + 1) = min(1.001 * TsisSV(ikl, isl + 1) &
                                               , max(0.999 * TsisSV(ikl, isl + 1), &
                                                     (TsisSV(ikl, isl) * dzsnSV(ikl, isl) &
                                                      + TsisSV(ikl, isl + 1) * dzsnSV(ikl, isl + 1) * ww) &
                                                     / (dzsnSV(ikl, isl) &
                                                        + ww * dzsnSV(ikl, isl + 1))))

                endif
            enddo
            if(ro__SV(ikl, 1) > 600 .and. TsisSV(ikl, 1) > 273.1) then
                TsisSV(ikl, 1) = (TsisSV(ikl, 1) * dzsnSV(ikl, 1) &
                                  + TsisSV(ikl, 2) * dzsnSV(ikl, 2)) / &
                                 (dzsnSV(ikl, 1) + dzsnSV(ikl, 2))
            endif
            if(ro__SV(ikl, 2) > 600 .and. TsisSV(ikl, 2) > 273.1) then
                TsisSV(ikl, 2) = (TsisSV(ikl, 1) * dzsnSV(ikl, 1) &
                                  + TsisSV(ikl, 2) * dzsnSV(ikl, 2) &
                                  + TsisSV(ikl, 3) * dzsnSV(ikl, 3)) / &
                                 (dzsnSV(ikl, 1) + dzsnSV(ikl, 2) + dzsnSV(ikl, 3))
            endif
        endif
    enddo

    ! +--Update Surface    Fluxes
    ! +  ========================
    do ikl = 1, klonv
        isl = isnoSV(ikl)
        IRs_SV(ikl) = IRs__D(ikl) - dIRsdT(ikl) * TsisSV(ikl, isl)
        ! Sensible Heat Downward > 0
        HSs_sv(ikl) = HS___D(ikl) - dSdTSV(ikl) * TsisSV(ikl, isl)
        ! Latent   Heat Downward > 0
        HLs_sv(ikl) = HL___D(ikl) - dLdTSV(ikl) * TsisSV(ikl, isl)
#ifdef NC
        ! +--OUTPUT for Stand Alone NetCDF File
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ! Absorbed Sol.
        SOsoKL(ikl) = sol_SV(ikl) * SoSosv(ikl)
        ! Up Surf. IR
        ! Down Atm IR
        ! Down Veg IR
        IRsoKL(ikl) = IRs_SV(ikl) &
                      + tau_sv(ikl) * IRd_SV(ikl) * Eso_sv(ikl) &
                      - (1.0 - tau_sv(ikl)) * 0.5 * IRv_sv(ikl)
        ! HS
        HSsoKL(ikl) = HSs_sv(ikl)
        ! HL
        HLsoKL(ikl) = HLs_sv(ikl)
        ! mm w.e./sec
        HLs_KL(ikl) = HLs_sv(ikl) / Lv_H2O
#endif
    enddo

    ! +--Energy Budget (OUT)
    ! +  ===================
    do ikl = 1, klonv
        ! Net   Solar
        ! Up Surf. IR
        ! Down Atm IR
        ! Down Veg IR
        ! Sensible
        ! Latent
        ETSo_d(ikl) = (SoSosv(ikl) * sol_SV(ikl) &
                       + IRs_SV(ikl) &
                       + tau_sv(ikl) * IRd_SV(ikl) * Eso_sv(ikl) &
                       - (1.0 - tau_sv(ikl)) * 0.5 * IRv_sv(ikl) &
                       + HSs_sv(ikl) &
                       + HLs_sv(ikl))
        ETSo_1(ikl) = 0.
    enddo
    do isl = -nsol, nsno
        do ikl = 1, klonv
            Exist0 = isl - isnoSV(ikl)
            Exist0 = 1.-max(zero, min(unun, Exist0))
            ETSo_1(ikl) = ETSo_1(ikl) + (TsisSV(ikl, isl) - TfSnow) * Exist0 / dtC_sv(ikl, isl)
        enddo
    enddo

    return
endsubroutine SISVAT_TSo
