#include "MAR_pp.def"
subroutine SISVAT_SIc(SIvAcr)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_SIc                                26-09-2006  MAR |
    ! |   subroutine SISVAT_SIc treats Sea-Ice and Ocean Latent Heat Exchanges |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT:   TaT_SV   : SBL Top    Temperature                       [K] |
    ! |   ^^^^^    isnoSV   : total Nb of Ice/Snow Layers                  [-] |
    ! |            LSmask   : Land-Sea   Mask                              [-] |
    ! |            dsn_SV   : Snow  Intensity                      [mm w.e./s] |
    ! |                                                                        |
    ! |   INPUT /  TsisSV   : Snow/Ice/Soil-Water Temperature              [K] |
    ! |   OUTPUT:  eta_SV   : Soil/Snow Water   Content                [m3/m3] |
    ! |   ^^^^^^   dzsnSV   : Snow Layer        Thickness                  [m] |
    ! |                                                                        |
    ! |   OUTPUT:  HFraSV   : Frazil            Thickness                  [m] |
    ! |   ^^^^^^                                                               |
    ! +------------------------------------------------------------------------+

    use marphy
    use mar_sv
    use mardsv
    use marxsv

    implicit none

    ! +--INPUT/OUTPUT
    ! +  ------------

    real SIvAcr(klonv)                 ! Sea-Ice      Vertical Acretion

    ! +--Local Variables
    ! +  ===============

    integer ikl, isn
    real OCN_OK
    real SIceOK
    real SIcFrz
    real Twat_n
    real Crodzw, Lro__I
    common / SISVAT_SIc_R / Crodzw, Lro__I
    logical SIcINI
    common / SISVAT_SIc_L / SIcINI

    real SalIce              ! Sea-Ice   Salinity
    real SalWat              ! Sea-Water Salinity

    ! +--DATA
    ! +  ====

    data SalIce/10./        ! Sea-Ice   Salinity
    data SalWat/35./        ! Sea-Water Salinity
    ! +...                    Typical Salinities in Terra Nova Bay
    ! +                       (Bromwich and Kurtz,   1984, JGR, p.3568;
    ! +                        Cavalieri and Martin, 1985,      p. 248)

    ! +--Initialisation
    ! +  ==============

    if(.not. SIcINI) then
        SIcINI = .true.
        ! [J/m2/K]
        Crodzw = C__Wat * ro_Wat * dz_dSV(0)
        ! [J/m3]
        Lro__I = Lf_H2O * ro_Ice * (1.-1.e-3 * SalIce &
                                    - (SalIce / SalWat) * (1.-1.e-3 * SalWat))
#ifdef e1
        Lro__I = Lf_H2O * ro_Ice
#endif
    endif

    ! +--Snow Fall cools Sea Water
    ! +  =========================

    do ikl = 1, klonv
        ! (1 - LSmask(ikl)) : Free Ocean
        OCN_OK = (1 - LSmask(ikl)) * max(0, 1 - isnoSV(ikl))
#ifdef IA
        ! [K]
        ! [J/kg]
        ! [J/kg]
        ! [kg/m2]
        TsisSV(ikl, 0) = TsisSV(ikl, 0) &
                         - OCN_OK * (Cn_dSV * (TfSnow - TaT_SV(ikl)) &
                                     + Lf_H2O * (1.-eta_SV(ikl, 0))) &
                         * dsn_SV(ikl) * dt__SV / Crodzw
#endif

        ! +--Sea-Ice Formation
        ! +  =================
#ifdef IA
        ! +***  Hibler (1984), Ocean Heat Flux: 25% of cooling (ANTARCTIC Ocean)
        ! +    (Hansen and Takahashi Eds)
        ! +     Geophys. Monogr. 29, M. Ewing Vol. 5, AGU, p. 241
        ! [K]
        Twat_n = max(TsisSV(ikl, 0), tfrwat)
        ! [m]
        SIcFrz = (Twat_n - TsisSV(ikl, 0)) * Crodzw / Lro__I * 0.75
        ! +--Frazil  Formation
        ! +  -----------------
        HFraSV(ikl) = SIcFrz * OCN_OK
        ! +--Growth of the Sea-Ice First Ice Floe
        ! +  ------------------------------------
        ! Ice Cover.Ocean
        SIceOK = (1 - LSmask(ikl)) &
                 * min(1, isnoSV(ikl))
        ! Vertical Acret.
        dzsnSV(ikl, 1) = dzsnSV(ikl, 1) &
                         + SIcFrz * SIceOK
#endif
        ! +--Diagnostic of the Surface Mass Balance
        ! +  --------------------------------------
#ifdef m2
        SIvAcr(ikl) = ro_Ice * SIcFrz * (OCN_OK + SIceOK) &
                      - dt__SV * dsn_SV(ikl) * OCN_OK
#else
        SIvAcr(ikl) = 0.
#endif

        ! +--Water Fluxes Update
        ! +  -------------------

        RnofSV(ikl) = RnofSV(ikl) &
                      + dsn_SV(ikl) * OCN_OK
        RuofSV(ikl, 1) = RuofSV(ikl, 1) &
                         + dsn_SV(ikl) * OCN_OK
        dsn_SV(ikl) = dsn_SV(ikl) * (1.-OCN_OK)

    enddo

    return
endsubroutine SISVAT_SIc
