subroutine SISVAT_qVg
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_qVg                                22-09-2001  MAR |
    ! |   subroutine SISVAT_qVg computes the Canopy Water  Balance             |
    ! |                                  including  Root   Extraction          |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   ivgtSV   = 0,...,12:   Vegetation Type                      |
    ! |   ^^^^^               0:          Water, Solid or Liquid               |
    ! |                                                                        |
    ! |   INPUT:   rhT_SV   : SBL Top    Air  Density                  [kg/m3] |
    ! |   ^^^^^    QaT_SV   : SBL Top    Specific  Humidity            [kg/kg] |
    ! |                                                                        |
    ! |            TvegSV   : Canopy   Temperature                         [K] |
    ! |            rrCaSV   : Canopy     Water     Content             [kg/m2] |
    ! |            rrMxsv   : Canopy Maximum Intercepted Rain          [kg/m2] |
    ! |            rah_sv   : Aerodynamic Resistance for Heat            [s/m] |
    ! |            EvT_sv   : EvapoTranspiration                       [kg/m2] |
    ! |            Sigmsv   : Canopy Ventilation Factor                    [-] |
    ! |            glf_sv   : Green Leaf Fraction of NOT fallen Leaves     [-] |
    ! |            LAIesv   : Leaf Area  Index (effective / transpiration) [-] |
    ! |            psi_sv   : Soil       Water    Potential                [m] |
    ! |            Khydsv   : Soil   Hydraulic    Conductivity           [m/s] |
    ! |                                                                        |
    ! |   INPUT /  psivSV   : Leaf       Water    Potential                [m] |
    ! |   OUTPUT:                                                              |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   OUTPUT:  Rootsv   : Root Water Pump                        [kg/m2/s] |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |                                                                        |
    ! |   REMARK: Water Extraction by roots calibrated by Evapotranspiration   |
    ! |   ^^^^^^  (computed in the Canopy Energy Balance)                      |
    ! |                                                                        |
    ! | # OPTIONS: #KW: Root Water Flow slowed by Soil Hydraulic Conductivity  |
    ! | # ^^^^^^^                                                              |
    ! +------------------------------------------------------------------------+

    use marphy
    use mar_sv
    use mardsv
    use mar0sv
    use marxsv
    use marysv

    implicit none

    ! +--Internal Variables
    ! +  ==================
    ! ikl, isl : Grid Point, Layer Indices
    integer ikl, isl
    ! nitmax, nit : Iterations Counter
    integer nitmax, nit
    ! PlantW : Plant  Water
    real PlantW(klonv)
    ! dPdPsi : Plant  Water psi Derivative
    real dPdPsi(klonv)
    ! psidif : Soil-Canopy  Water Pot. Differ.
    real psidif
    ! Root_W : Root   Water     Flow
    real Root_W
    ! RootOK : Roots take Water in Soil Layer
    real RootOK
    ! d_psiv : Canopy Water     Increment
    real d_psiv
    ! dpvMAX : Canopy Water     Increment MAX
    real dpvMAX
    ! BWater : Imbalance of Canopy Water Budg.
    real BWater
    ! BW_MAX : MAX Imbal.of Canopy Water Budg.
    real BW_MAX
    ! BW_MIN : MIN Imbal.of Canopy Water Budg.
    real BW_MIN
    ! dBwdpv : Derivativ.of Canopy Water Budg.
    real dBwdpv
    ! Bswich : Newton-Raphson         Switch
    real Bswich
    ! psiv_0 : Canopy Temperature,  Previous t
    real psiv_0(klonv)
    ! EvFrac : Condensat./Transpiration Switch
    real EvFrac
    ! den_qs, arg_qs, qsatvg : Canopy Saturat. Spec. Humidity
    real den_qs, arg_qs, qsatvg
    ! EvTran : EvapoTranspiration
    real EvTran
    ! dEdpsi : Evapotranspiration  Derivative
    real dEdpsi
    ! Fac_Ev, FacEvT : Evapotranspiration  Factors
    real Fac_Ev, FacEvT
    ! denomE : Evapotranspiration  Denominator
    real denomE
    ! F_Stom : Funct.  (Leaf Water Potential)
    real F_Stom
    ! dFdpsi : Derivative  of F_Stom
    real dFdpsi
    ! denomF : Denominator of F_Stom
    real denomF
    ! F___OK : (psi>psi_c) => F_Stom swich  ON
    real F___OK
    ! R0Stom : Minimum Stomatal Resistance
    real R0Stom
    ! R_Stom : Stomatal Resistance
    real R_Stom
    ! dRdpsi : Derivat.Stomatal Resistance
    real dRdpsi
    ! numerR : Numerat.Stomatal Resistance
    real numerR

    ! +--Internal DATA
    ! +  =============
    ! nitmax : Maximum  Iterations    Number
    data nitmax/5/
    ! dpvMAX : Canopy   Water   Increment MAX
    data dpvMAX/20./
    ! BW_MIN : MIN Imbal. of Surf.Energy Budg.
    data BW_MIN/4.e-8/

    ! +--Newton-Raphson Scheme
    ! +  =====================
    nit = 0
101 continue
    nit = nit + 1
    BW_MAX = 0.

    ! +--W.Potential of the Previous Time Step
    ! +  -------------------------------------
    do ikl = 1, klonv
        psiv_0(ikl) = psivSV(ikl)
        ! +--Extraction of Soil Water through the Plant Roots
        ! +  ------------------------------------------------
        ! PlantW : Plant Water
        PlantW(ikl) = 0.
        ! dPdPsi : Idem, Derivat.
        dPdPsi(ikl) = 0.
    enddo
    do isl = -nsol, 0
        do ikl = 1, klonv
            ! Soil-Canopy Water
            psidif = psivSV(ikl) - (DH_dSV(ivgtSV(ikl)) &
                                    + psi_sv(ikl, isl))    ! Potential  Diff.
            ! If > 0, Contrib. to Root Water
            Root_W = Ro_Wat * RF__SV(ivgtSV(ikl), isl) &
                     / max(eps_21, PR_dSV(ivgtSV(ikl)))
            ! +!!!    Pas de prise en compte de la resistance sol/racine dans proto-svat
            ! +       (DR97, eqn.3.20)
            RootOK = max(zero, sign(unun, psidif))
            Rootsv(ikl, isl) = Root_W * max(zero, psidif)         ! Root  Water
            PlantW(ikl) = PlantW(ikl) + Rootsv(ikl, isl)  ! Plant Water
            dPdPsi(ikl) = dPdPsi(ikl) + RootOK * Root_W     ! idem, Derivat.
        enddo
    enddo

    ! +--Latent   Heat Flux
    ! +  ------------------

    ! +--Canopy Saturation Specific Humidity
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        den_qs = TvegSV(ikl) - 35.8
        arg_qs = 17.27 * (TvegSV(ikl) - 273.16) / den_qs
        qsatvg = .0038 * exp(arg_qs)
        !XF       qsatvg = .0038 * exp(arg_qs) * 0.875 ! A TESTER 04/2019

        ! +--Canopy Stomatal Resistance
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        R0Stom = min(StodSV(ivgtSV(ikl)) &
                     / max(epsi, glf_sv(ikl)), StxdSV) ! Min Stomatal R.
        denomF = pscdSV - psivSV(ikl)
        F___OK = max(zero, sign(unun, denomF))
        denomF = max(epsi, denomF)           !
        F_Stom = pscdSV / denomF            ! F(Leaf Wat.Pot.)
        dFdpsi = -F_Stom / denomF            !
        ! +                                                         ! DR97, eqn. 3.22
        numerR = R0Stom / max(LAIesv(ikl), R0Stom / StxdSV) !
        R_Stom = numerR * F_Stom         ! Can.Stomatal R.
        ! +                                                         ! DR97, eqn. 3.21
        dRdpsi = R_Stom * dFdpsi         !

        ! +--Evaporation / Evapotranspiration
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        EvFrac = max(zero, sign(unun, QaT_SV(ikl) - qsatvg)) ! Condensation/
        ! Transpiration  Switch
        EvFrac = EvFrac &
                 + (1.-EvFrac) * rrCaSV(ikl) / rrMxsv(ikl)
        ! idem,  Factor
        Fac_Ev = rhT_SV(ikl) * Sigmsv(ikl)
        denomE = rah_sv(ikl) + R_Stom * Sigmsv(ikl)
        FacEvT = Fac_Ev * (1.-EvFrac) / denomE
        ! EvapoTranspir.
        EvTran = FacEvT * (qsatvg - QaT_SV(ikl))
        ! EvT Derivative
        dEdpsi = (EvTran / denomE) * dRdpsi

        ! +--Imbalance  of the Canopy  Water  Budget
        ! +  ---------------------------------------

        ! Available  Water
        BWater = (PlantW(ikl) &
                  ! Transpired Water
                  - EvTran) * F___OK
        ! Newton-Raphson Switch
        Bswich = max(zero, &
                     sign(unun, abs(BWater) &
                          - BW_MIN))

        ! +--Derivative of the Canopy  Water  Budget
        ! +  ---------------------------------------

        dBwdpv = dPdpsi(ikl) &
                 - dEdpsi
        dBwdpv = sign(unun, dBwdpv) &
                 * max(eps_21, abs(dBwdpv))

        ! +--Update Canopy and Surface/Canopy Temperatures
        ! +  ---------------------------------------------

        d_psiv = BWater / dBwdpv
        ! Increment Limitor
        d_psiv = sign(unun, d_psiv) &
                 * min(abs(d_psiv), dpvMAX)
        ! Newton-Raphson
        psivSV(ikl) = psivSV(ikl) - Bswich * d_psiv
        BW_MAX = max(BW_MAX, abs(BWater))
    enddo

    ! +--Update Root Water Fluxes | := Evapotranspiration
    ! +  ------------------------------------------------

    do isl = -nsol, 0
        do ikl = 1, klonv
            ! Root  Water
            Rootsv(ikl, isl) = Rootsv(ikl, isl) * EvT_SV(ikl) &
                               / max(eps_21, PlantW(ikl))
        enddo
    enddo

    if(BW_MAX > BW_MIN .and. nit < nitmax) go to 101

    return
end
