#include "MAR_pp.def"
subroutine SISVAT_qSo(Wats_0, Wats_1, Wats_d)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_qSo                                14-03-2022  MAR |
    ! |   subroutine SISVAT_qSo computes the Soil      Water  Balance          |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |   ^^^^^    isotSV   = 0,...,11:   Soil       Type                      |
    ! |                       0:          Water, Solid or Liquid               |
    ! |                                                                        |
    ! |   INPUT:   rhT_SV   : SBL Top    Air  Density                  [kg/m3] |
    ! |   ^^^^^    drr_SV   : Rain   Intensity                       [kg/m2/s] |
    ! |            LSdzsv   : Vertical   Discretization Factor             [-] |
    ! |                     =    1. Soil                                       |
    ! |                     = 1000. Ocean                                      |
    ! |            dt__SV   : Time   Step                                  [s] |
    ! |                                                                        |
    ! |            Lx_H2O   : Latent Heat of Vaporization/Sublimation   [J/kg] |
    ! |            HLs_sv   : Latent Heat  Flux                         [W/m2] |
    ! |            Rootsv   : Root   Water Pump                      [kg/m2/s] |
    ! |                                                                        |
    ! |   INPUT /  eta_SV   : Water      Content                       [m3/m3] |
    ! |   OUTPUT:  Khydsv   : Soil   Hydraulic    Conductivity           [m/s] |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   OUTPUT:  RnofSV   : RunOFF Intensity                       [kg/m2/s] |
    ! |   ^^^^^^   Wats_0   : Soil Water,  before Forcing                 [mm] |
    ! |            Wats_1   : Soil Water,  after  Forcing                 [mm] |
    ! |            Wats_d   : Soil Water          Forcing                 [mm] |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |            z_Bump   : (Partly)Bumpy Layers Height                  [m] |
    ! |            z0Bump   :         Bumpy Layers Height                  [m] |
    ! |            dzBump   :  Lowest Bumpy Layer:                         [m] |
    ! |            etBump   :         Bumps Layer Averaged Humidity    [m3/m3] |
    ! |            etaMid   : Layer Interface's Humidity               [m3/m3] |
    ! |            eta__f   : Layer             Humidity  (Water Front)[m3/m3] |
    ! |            Dhyd_f   : Soil  Hydraulic Diffusivity (Water Front) [m2/s] |
    ! |            Dhydif   : Soil  Hydraulic Diffusivity               [m2/s] |
    ! |            WgFlow   : Water         gravitational     Flux   [kg/m2/s] |
    ! |            Wg_MAX   : Water MAXIMUM gravitational     Flux   [kg/m2/s] |
    ! |            SatRat   : Water         Saturation        Flux   [kg/m2/s] |
    ! |            WExces   : Water         Saturation Excess Flux   [kg/m2/s] |
    ! |            Dhydtz   : Dhydif * dt / dz                             [m] |
    ! |            FreeDr   : Free Drainage Fraction                       [-] |
    ! |            Elem_A   : A Diagonal Coefficient                           |
    ! |            Elem_C   : C Diagonal Coefficient                           |
    ! |            Diag_A   : A Diagonal                                       |
    ! |            Diag_B   : B Diagonal                                       |
    ! |            Diag_C   : C Diagonal                                       |
    ! |            Term_D   :   Independant Term                               |
    ! |            Aux__P   : P Auxiliary Variable                             |
    ! |            Aux__Q   : Q Auxiliary Variable                             |
    ! |                                                                        |
    ! |   TUNING PARAMETER:                                                    |
    ! |   ^^^^^^^^^^^^^^^^                                                     |
    ! |            z0soil   : Soil Surface averaged Bumps Height           [m] |
    ! |                                                                        |
    ! |   METHOD: NO   Skin Surface Humidity                                   |
    ! |   ^^^^^^  Semi-Implicit Crank Nicholson Scheme                         |
    ! |           (Partial) free Drainage, Water Bodies excepted (Lakes, Sea)  |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! | #          #m0: Water  Budget Verification                             |
    ! | #          #m1: Snow/I Budget Verification                             |
    ! |                                                                        |
    ! | # OPTIONS: #GF: Saturation Front                                       |
    ! | # ^^^^^^^  #GH: Saturation Front allows Horton Runoff                  |
    ! | #          #GA: Soil Humidity Geometric Average                        |
    ! | #          #BP: Parameterization of Terrain Bumps                      |
    ! |                                                                        |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! |   FILE                 |      CONTENT                                  |
    ! |   ~~~~~~~~~~~~~~~~~~~~~+~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ |
    ! | # SISVAT_qSo.vw        | #vw: OUTPUT/Verif+Detail: H2O    Conservation |
    ! |                        |      unit 42, subroutine  SISVAT_qSo **ONLY** |
    ! +------------------------------------------------------------------------+

    use marphy
    use marctr
    use mar_sv
    use mardsv
    use mar0sv
    use marxsv
    use marysv

    implicit none

    ! +--OUTPUT
    ! +  ------

    ! Water (Mass) Budget
    ! ~~~~~~~~~~~~~~~~~~~
    real Wats_0(klonv)                 ! Soil Water,  before forcing
    real Wats_1(klonv)                 ! Soil Water,  after  forcing
    real Wats_d(klonv)                 ! Soil Water          forcing

    ! +--Internal Variables
    ! +  ==================

    integer isl, jsl, ist, ikl
    integer ikm, ikp, ik0, ik1
    ! ist__s, ist__w : Soil/Water Body Identifier
    integer ist__s, ist__w
#ifdef BP
    ! z0soil : Soil Surface Bumps Height  [m]
    real z0soil
    data z0soil/0.020/
    ! z_Bump : (Partly) Bumpy Layers Height [m]
    real z_Bump
    ! z0Bump : Bumpy Layers Height [m]
    real z0Bump
    ! dzBump : Lowest Bumpy Layer:
    real dzBump
    ! etBump : Bumps Layer Averaged Humidity
    real etBump(klonv)
#endif
    ! etaMid : Layer Interface's Humidity
    real etaMid
    ! Dhydif : Hydraulic Diffusivity [m2/s]
    real Dhydif
    ! eta__f : Water Front Soil Water Content
    real eta__f
    ! Khyd_f : Water Front Hydraulic Conduct.
    real Khyd_f
    ! Khydav : Hydraulic Conductivity [m/s]
    real Khydav
    ! WgFlow : Water gravitat. Flux [kg/m2/s]
    real WgFlow
    ! Wg_MAX : Water MAX.grav. Flux [kg/m2/s]
    real Wg_MAX
    ! SatRat : Saturation Flux [kg/m2/s]
    real SatRat
    ! WExces : Saturat. Excess Flux [kg/m2/s]
    real WExces
    ! SoRnOF, SoRnOF2 : Soil Run OFF
    real SoRnOF(klonv), SoRnOF2(klonv)
    ! Dhydtz : Dhydif * dt / dz [m]
    real Dhydtz(klonv, -nsol:0)
    ! Elem_A, Elem_B, Elem_C : Diagonal Coefficients
    real Elem_A, Elem_B, Elem_C
    ! Diag_A : A Diagonal
    real Diag_A(klonv, -nsol:0)
    ! Diag_B : B Diagonal
    real Diag_B(klonv, -nsol:0)
    ! Diag_C : C Diagonal
    real Diag_C(klonv, -nsol:0)
    ! Term_D :  Independant Term
    real Term_D(klonv, -nsol:0)
    ! Aux__P : P Auxiliary Variable
    real Aux__P(klonv, -nsol:0)
    ! Aux__Q : Q Auxiliary Variable
    real Aux__Q(klonv, -nsol:0)
    ! etaaux : Soil Water Content [m3/m3]
    real etaaux(klonv, -nsol:-nsol + 1)
    ! FreeDr : Free Drainage Fraction (actual)
    real FreeDr
    ! FreeD0 : Free Drainage Fraction (1=Full)
    real FreeD0
    ! aKdtSV3 : Khyd=a*eta+b: a * dt
    real aKdtSV3(0:nsot, 0:nkhy)
    ! bKdtSV3 : Khyd=a*eta+b: b * dt
    real bKdtSV3(0:nsot, 0:nkhy)
    real sum1(klonv), sum2(klonv)

#ifdef mw
    ! Water (Mass) Budget
    ! ~~~~~~~~~~~~~~~~~~~
    ! mwopen : IO Switch
    logical mwopen
    common / Sm_qSo_L / mwopen
    real hourwr, timewr
    common / Sm_qSo_R / timewr
    real Evapor(klonv)
#endif

    ! +--Internal DATA
    ! +  =============
    ! FreeD0 : Free Drainage Fraction (1=Full)
    data FreeD0/1.000/

    aKdtSV3 = aKdtSV2 * dt__SV
    bKdtSV3 = bKdtSV2 * dt__SV
#ifdef m0
    ! Water  Budget (IN)
    ! ==================
    do ikl = 1, klonv
        ! OLD RunOFF Contrib.
        Wats_0(ikl) = 0.
        ! Water Surface Forc.
        Wats_d(ikl) = drr_SV(ikl)
    enddo
    isl = -nsol
    do ikl = 1, klonv
        Wats_0(ikl) = Wats_0(ikl) &
                      + ro_Wat * (eta_SV(ikl, isl) * dz78SV(isl) &
                                  + eta_SV(ikl, isl + 1) * dz_8SV(isl)) * LSdzsv(ikl)
    enddo
    do isl = -nsol + 1, -1
        do ikl = 1, klonv
            Wats_0(ikl) = Wats_0(ikl) &
                          + ro_Wat * (eta_SV(ikl, isl) * dz34SV(isl) &
                                      + (eta_SV(ikl, isl - 1) &
                                         + eta_SV(ikl, isl + 1)) * dz_8SV(isl)) * LSdzsv(ikl)
        enddo
    enddo
    isl = 0
    do ikl = 1, klonv
        Wats_0(ikl) = Wats_0(ikl) &
                      + ro_Wat * (eta_SV(ikl, isl) * dz78SV(isl) &
                                  + eta_SV(ikl, isl - 1) * dz_8SV(isl)) * LSdzsv(ikl)
    enddo
#else
    Wats_0 = 0.
    Wats_1 = 0.
    Wats_d = 0.
#endif

    ! +--Gravitational Flow
    ! +  ==================

    ! +...    METHOD: Surface Water Flux saturates successively the soil layers
    ! +       ^^^^^^  from up to below, but is limited by infiltration capacity.
    ! +               Hydraulic Conductivity again contributes after this step,
    ! +               not redundantly because of a constant (saturated) profile.

    ! +--Flux  Limitor
    ! +  ^^^^^^^^^^^^^
    isl = 0
    do ikl = 1, klonv
        ! Soil Type
        ist = isotSV(ikl)
        ! 1 => Soil
        ist__s = min(ist, 1)
        ! 1 => Water Body
        ist__w = 1 - ist__s
        ! Hydraulic Diffusivity DR97, Eqn.(3.36)
        Dhydif = s1__SV(ist) &
                 * max(epsi, eta_SV(ikl, isl)) &
                 **(bCHdSV(ist) + 2.)
        ! Water Bodies
        Dhydif = ist__s * Dhydif &
                 + ist__w * vK_dSV
        ! DR97  Assumption Water Bodies
        Khydav = ist__s * Ks_dSV(ist) &
                 + ist__w * vK_dSV
        ! +
        ! MAXimum  Infiltration Rate
        Wg_MAX = ro_Wat * Dhydif &
                 * (etadSV(ist) - eta_SV(ikl, isl)) &
                 / (dzAvSV(isl) * LSdzsv(ikl)) &
                 + ro_Wat * Khydav

        ! +--Surface Horton RunOFF
        ! +  ^^^^^^^^^^^^^^^^^^^^^
        if(ivgtSV(ikl) >= 0 .and. ivgtSV(ikl) <= 6) then ! crop/grass
            Wg_MAX = max(Wg_MAX, 0.10 * drr_SV(ikl))
        endif

        if(ivgtSV(ikl) >= 7 .and. ivgtSV(ikl) <= 12) then ! forest
            Wg_MAX = max(Wg_MAX, 0.05 * drr_SV(ikl))
        endif

        if(ivgtSV(ikl) == 13) then ! city
            Wg_MAX = max(Wg_MAX, 0.5 * drr_SV(ikl))
        endif
        SoRnOF(ikl) = max(zero, drr_SV(ikl) - Wg_MAX)
        RuofSV(ikl, 4) = RuofSV(ikl, 4) + SoRnOF(ikl)
        drr_SV(ikl) = drr_SV(ikl) - SoRnOF(ikl)
        SoRnOF2(ikl) = 0.
    enddo
#ifdef GF
    do isl = 0, -nsol, -1
        do ikl = 1, klonv
            ist = isotSV(ikl)                     ! Soil Type
            ist__s = min(ist, 1)                     ! 1 => Soil
            ist__w = 1 - ist__s                      ! 1 => Water Body
            ! +--Water Diffusion
            ! +  ^^^^^^^^^^^^^^^
            ! Hydraulic Diffusivity DR97, Eqn.(3.36)
            Dhydif = s1__SV(ist) &
                     * max(epsi, eta_SV(ikl, isl)) &
                     **(bCHdSV(ist) + 2.)
            ! Water Bodies
            Dhydif = ist__s * Dhydif &
                     + ist__w * vK_dSV
            ! +--Water Conduction (without Horton Runoff)
            ! +  ^^^^^^^^^^^^^^^^
            Khyd_f = Ks_dSV(ist)
            ! +...    Uses saturated K ==> Horton Runoff ~0    !
#ifdef GH
            ! +--Water Conduction (with    Horton Runoff)
            ! +  ^^^^^^^^^^^^^^^^
            ik0 = nkhy * eta_SV(ikl, isl) &
                  / etadSV(ist)
            eta__f = 1. &
                     -aKdtSV3(ist, ik0) / (2.*dzAvSV(isl) &
                                           * LSdzsv(ikl))
            eta__f = max(eps_21, eta__f)
            eta__f = min(etadSV(ist), &
                         eta_SV(ikl, isl) + &
                         (aKdtSV3(ist, ik0) * eta_SV(ikl, isl) &
                          + bKdtSV3(ist, ik0)) / (dzAvSV(isl) &
                                                  * LSdzsv(ikl)) &
                         / eta__f)
            eta__f = .5 * (eta_SV(ikl, isl) &
                           + eta__f)
#ifdef gh
            eta__f = eta_SV(ikl, isl)
#endif
            ik0 = nkhy * eta__f &
                  / etadSV(ist)
            Khyd_f = &
                (aKdtSV3(ist, ik0) * eta__f &
                 + bKdtSV3(ist, ik0)) / dt__SV
#endif
            ! DR97  Assumption Water Bodies
            Khydav = ist__s * Khyd_f &
                     + ist__w * vK_dSV
            ! +--Gravitational Flow
            ! +  ^^^^^^^^^^^^^^^^^^
            ! MAXimum  Infiltration Rate
            Wg_MAX = &
                ro_Wat * Dhydif &
                * (etadSV(ist) - eta_SV(ikl, isl)) &
                / (dzAvSV(isl) * LSdzsv(ikl)) &
                + ro_Wat * Khydav
#ifdef WR
            write(6, 6001) isl, drr_SV(ikl) * 3.6e3, Wg_MAX * 3.6e3
6001        format(i3, '  vRain ,Wg_MAX ', 2e12.3, ' mm/hr')
#endif
            ! Infiltration Rate
            WgFlow = min(Wg_MAX, drr_SV(ikl))
            ! Water Excess => RunOff
            WExces = max(zero, drr_SV(ikl) - WgFlow)
#ifdef WR
            write(6, 6002) WgFlow * 3.6e3, WExces * 3.6e3
6002        format(3x, '  WgFlow,WExces ', 2e12.3, ' mm/hr')
#endif
            SoRnOF(ikl) = SoRnOF(ikl) + WExces
            drr_SV(ikl) = WgFlow
#ifdef WR
            write(6, 6003) SoRnOF(ikl) * 3.6e3, drr_SV(ikl) * 3.6e3
6003        format(3x, '  SoRnOF,drr_SV ', 2e12.3, ' mm/hr')
#endif
            ! Saturation   Rate
            SatRat = (etadSV(ist) - eta_SV(ikl, isl)) &
                     * ro_Wat * dzAvSV(isl) &
                     * LSdzsv(ikl) / dt__SV
            SatRat = min(SatRat, drr_SV(ikl))
            ! Water Flux for Below
            drr_SV(ikl) = drr_SV(ikl) - SatRat
#ifdef WR
            write(6, 6004) SatRat * 3.6e3, drr_SV(ikl) * 3.6e3
6004        format(3x, '  SatRat,drr_SV ', 2e12.3, ' mm/hr')
#endif
#ifdef WR
            write(6, 6005) eta_SV(ikl, isl) * 1.e3
#endif
            ! Saturation
            eta_SV(ikl, isl) = eta_SV(ikl, isl) &
                               + SatRat * dt__SV &
                               / (ro_Wat * dzAvSV(isl) &
                                  * LSdzsv(ikl))
#ifdef WR
            write(6, 6005) eta_SV(ikl, isl) * 1.e3
6005        format(3x, '  eta_SV,       ', e12.3, ' g/kg')
#endif
        enddo
    enddo
    do ikl = 1, klonv
        ! RunOFF Intensity [kg/m2/s]
        SoRnOF(ikl) = SoRnOF(ikl) &
                      + drr_SV(ikl)
        ! +!!!    Inclure la possibilite de creer une mare sur un bedrock impermeable
        drr_SV(ikl) = 0.
    enddo
#endif

    ! +--Temperature Correction due to a changed Soil Energy Content
    ! +  ===========================================================

    ! +!!!    Mettre en oeuvre le couplage humidit?-?nergie

    ! +--Full Resolution of the Richard's Equation
    ! +  =========================================

    ! +...    METHOD: Water content evolution results from water fluxes
    ! +       ^^^^^^  at the layer boundaries
    ! +               Conductivity is approximated by a piecewise linear profile.
    ! +               Semi-Implicit Crank-Nicholson scheme is used.
    ! +              (Bruen, 1997, Sensitivity of hydrological processes
    ! +                            at the land-atmosphere interface.
    ! +                            Proc. Royal Irish Academy,  IGBP symposium
    ! +                            on global change and the Irish Environment.
    ! +                            Publ.: Maynooth)
    ! +                      - - - - - - - -   isl+1/2   - -  ^
    ! +                                                       |
    ! +   eta_SV(isl)        ---------------   isl     -----  +--dz_dSV(isl)  ^
    ! +                                                       |               |
    ! +   Dhydtz(isl) etaMid - - - - - - - -   isl-1/2   - -  v  dzmiSV(isl)--+
    ! +                                                                       |
    ! +   eta_SV(isl-1)      ---------------   isl-1   -----                  v

    ! +--Transfert       Coefficients
    ! +  ----------------------------

    do isl = -nsol + 1, 0
        do ikl = 1, klonv
            ! Soil Type
            ist = isotSV(ikl)
            ! 1 => Soil
            ist__s = min(ist, 1)
            ! 1 => Water Body
            ist__w = 1 - ist__s
            ! eta at layers interface LSdzsv implicit
            etaMid = (dz_dSV(isl) * eta_SV(ikl, isl - 1) &
                      + dz_dSV(isl - 1) * eta_SV(ikl, isl)) &
                     / (2.0 * dzmiSV(isl))
#ifdef GA
            ! Idem, geometric average (Vauclin&al.1979)
            etaMid = sqrt(dz_dSV(isl) * eta_SV(ikl, isl - 1) &
                          * dz_dSV(isl - 1) * eta_SV(ikl, isl)) &
                     / (2.0 * dzmiSV(isl))
#endif
            ! Hydraul.Diffusi. DR97, Eqn.(3.36)
            Dhydif = s1__SV(ist) &
                     * (etaMid**(bCHdSV(ist) + 2.))
            Dhydtz(ikl, isl) = Dhydif * dt__SV &
                               / (dzmiSV(isl) &
                                  * LSdzsv(ikl))
            ! Soil Water bodies
            Dhydtz(ikl, isl) = Dhydtz(ikl, isl) * ist__s &
                               + 0.5 * dzmiSV(isl) * LSdzsv(ikl) * ist__w

        enddo
    enddo
    isl = -nsol
    do ikl = 1, klonv
        Dhydtz(ikl, isl) = 0.0                        !
    enddo

    ! +--Tridiagonal Elimination: Set Up
    ! +  -------------------------------

    ! +--Soil/Snow Interior
    ! +  ^^^^^^^^^^^^^^^^^^

    sum1 = 0
    do isl = 0, -nsol, -1
        do ikl = 1, klonv
            eta_SV(ikl, isl) = max(epsi, eta_SV(ikl, isl))
            sum1(ikl) = sum1(ikl) + eta_SV(ikl, isl) * dzAvSV(isl)
            ! -dt__SV *Rootsv(ikl,isl)
            ! /(ro_Wat *LSdzsv(ikl))
        enddo
    enddo

    do ikl = 1, klonv
        ! rainfall
        sum1(ikl) = sum1(ikl) + dt__SV * drr_SV(ikl) / (ro_Wat * LSdzsv(ikl))
        ! subli/evapo from soil
        sum1(ikl) = sum1(ikl) + dt__SV * HLs_sv(ikl) * (1 - min(1, isnoSV(ikl))) &
                    / (Lx_H2O(ikl) * ro_Wat * LSdzsv(ikl))
        ! evapotranspiration
        sum1(ikl) = sum1(ikl) - dt__SV * EvT_SV(ikl) / (ro_Wat * LSdzsv(ikl))
        if(isnoSV(ikl) == 0) then
            wee_SV(ikl, 4) = wee_SV(ikl, 4) - dt__SV * HLs_sv(ikl) / Lx_H2O(ikl)
        endif
    enddo

    do isl = -nsol, -nsol + 1
        do ikl = 1, klonv
            etaaux(ikl, isl) = eta_SV(ikl, isl)
        enddo
    enddo

    do isl = -nsol + 1, -1
        do ikl = 1, klonv
            ist = isotSV(ikl)
            ikm = nkhy * eta_SV(ikl, isl - 1) / etadSV(ist)
            ik0 = nkhy * eta_SV(ikl, isl) / etadSV(ist)
            ikp = nkhy * eta_SV(ikl, isl + 1) / etadSV(ist)
            if(ikm < 0 .or. ik0 < 0 .or. ikp < 0) then
                call time_steps
                print *, "CRASH1 in sisvat_qso.f on pixel (i,j,n)", &
                    ii__SV(ikl), jj__SV(ikl), nn__SV(ikl)
                print *, "decrease your time step or increase ntphys "// &
                    "and ntdiff in time_steps.f90"
                stop
            endif

            Elem_A = Dhydtz(ikl, isl) &
                     - aKdtSV3(ist, ikm) * dziiSV(isl) * LSdzsv(ikl)
            Elem_B = -(Dhydtz(ikl, isl) &
                       + Dhydtz(ikl, isl + 1) &
                       - aKdtSV3(ist, ik0) * (dziiSV(isl + 1) &
                                              - dzi_SV(isl)) * LSdzsv(ikl))
            Elem_C = Dhydtz(ikl, isl + 1) &
                     + aKdtSV3(ist, ikp) * dzi_SV(isl + 1) * LSdzsv(ikl)
            Diag_A(ikl, isl) = dz_8SV(isl) * LSdzsv(ikl) &
                               - Implic * Elem_A
            Diag_B(ikl, isl) = dz34SV(isl) * LSdzsv(ikl) &
                               - Implic * Elem_B
            Diag_C(ikl, isl) = dz_8SV(isl) * LSdzsv(ikl) &
                               - Implic * Elem_C

            Term_D(ikl, isl) = (dz_8SV(isl) * LSdzsv(ikl) &
                                + Explic * Elem_A) * eta_SV(ikl, isl - 1) &
                               + (dz34SV(isl) * LSdzsv(ikl) &
                                  + Explic * Elem_B) * eta_SV(ikl, isl) &
                               + (dz_8SV(isl) * LSdzsv(ikl) &
                                  + Explic * Elem_C) * eta_SV(ikl, isl + 1) &
                               + (bKdtSV3(ist, ikp) * dzi_SV(isl + 1) &
                                  + bKdtSV3(ist, ik0) * (dziiSV(isl + 1) &
                                                         - dzi_SV(isl)) &
                                  - bKdtSV3(ist, ikm) * dziiSV(isl)) &
                               * LSdzsv(ikl) &
                               - dt__SV * Rootsv(ikl, isl) / ro_Wat
        enddo
    enddo

    isl = -nsol
    do ikl = 1, klonv
        ist = isotSV(ikl)
        ! FreeDr   =         FreeD0            *  min(ist,1)
        FreeDr = iWaFSV(ikl) * min(ist, 1)
        ik0 = nkhy * eta_SV(ikl, isl) / etadSV(ist)
        ikp = nkhy * eta_SV(ikl, isl + 1) / etadSV(ist)

        if(ik0 < 0 .or. ikp < 0) then
            print *, "CRASH2 in sisvat_qso.f on pixel (i,j,n)", &
                ii__SV(ikl), jj__SV(ikl), nn__SV(ikl)
            print *, "decrease your time step or increase ntphys "// &
                "and ntdiff in time_steps.f"
            stop
        endif

        Elem_A = 0.
        Elem_B = -(Dhydtz(ikl, isl + 1) &
                   - aKdtSV3(ist, ik0) * (dziiSV(isl + 1) * LSdzsv(ikl) &
                                          - FreeDr))
        Elem_C = Dhydtz(ikl, isl + 1) &
                 + aKdtSV3(ist, ikp) * dzi_SV(isl + 1) * LSdzsv(ikl)
        Diag_A(ikl, isl) = 0.
        Diag_B(ikl, isl) = dz78SV(isl) * LSdzsv(ikl) &
                           - Implic * Elem_B
        Diag_C(ikl, isl) = dz_8SV(isl) * LSdzsv(ikl) &
                           - Implic * Elem_C

        Term_D(ikl, isl) = (dz78SV(isl) * LSdzsv(ikl) &
                            + Explic * Elem_B) * eta_SV(ikl, isl) &
                           + (dz_8SV(isl) * LSdzsv(ikl) &
                              + Explic * Elem_C) * eta_SV(ikl, isl + 1) &
                           + (bKdtSV3(ist, ikp) * dzi_SV(isl + 1) * LSdzsv(ikl) &
                              + bKdtSV3(ist, ik0) * (dziiSV(isl + 1) * LSdzsv(ikl) &
                                                     - FreeDr)) &
                           - dt__SV * Rootsv(ikl, isl) / ro_Wat
    enddo

    isl = 0
    do ikl = 1, klonv
        ist = isotSV(ikl)
        ikm = nkhy * eta_SV(ikl, isl - 1) / etadSV(ist)
        ik0 = nkhy * eta_SV(ikl, isl) / etadSV(ist)
        Elem_A = Dhydtz(ikl, isl) &
                 - aKdtSV3(ist, ikm) * dziiSV(isl) * LSdzsv(ikl)
        Elem_B = -(Dhydtz(ikl, isl) &
                   + aKdtSV3(ist, ik0) * dzi_SV(isl) * LSdzsv(ikl))
        Elem_C = 0.
        Diag_A(ikl, isl) = dz_8SV(isl) * LSdzsv(ikl) &
                           - Implic * Elem_A
        Diag_B(ikl, isl) = dz78SV(isl) * LSdzsv(ikl) &
                           - Implic * Elem_B
        Diag_C(ikl, isl) = 0.
        ! +
        Term_D(ikl, isl) = (dz_8SV(isl) * LSdzsv(ikl) &
                            + Explic * Elem_A) * eta_SV(ikl, isl - 1) &
                           + (dz78SV(isl) * LSdzsv(ikl) &
                              + Explic * Elem_B) * eta_SV(ikl, isl) &
                           - (bKdtSV3(ist, ik0) * dzi_SV(isl) &
                              + bKdtSV3(ist, ikm) * dziiSV(isl)) * LSdzsv(ikl) &
                           + dt__SV * (HLs_sv(ikl) * 1.*(1 - min(1, isnoSV(ikl))) &
                                       / (ro_Wat * dz_dSV(0) * Lx_H2O(ikl)) &
                                       + drr_SV(ikl) &
                                       - Rootsv(ikl, isl)) / ro_Wat
    enddo

    do ikl = 1, klonv
        drr_SV(ikl) = 0. ! drr is included in the 1st soil layer
    enddo

    ! +
    ! +
    ! +--Tridiagonal Elimination
    ! +  =======================
    ! +
    ! +--Forward  Sweep
    ! +  ^^^^^^^^^^^^^^
    do ikl = 1, klonv
        Aux__P(ikl, -nsol) = Diag_B(ikl, -nsol)
        Aux__Q(ikl, -nsol) = -Diag_C(ikl, -nsol) / Aux__P(ikl, -nsol)
    enddo
    ! +
    do isl = -nsol + 1, 0
        do ikl = 1, klonv
            Aux__P(ikl, isl) = Diag_A(ikl, isl) * Aux__Q(ikl, isl - 1) &
                               + Diag_B(ikl, isl)
            Aux__Q(ikl, isl) = -Diag_C(ikl, isl) / Aux__P(ikl, isl)
        enddo
    enddo
    ! +
    do ikl = 1, klonv
        eta_SV(ikl, -nsol) = Term_D(ikl, -nsol) / Aux__P(ikl, -nsol)
    enddo
    ! +
    do isl = -nsol + 1, 0
        do ikl = 1, klonv
            eta_SV(ikl, isl) = (Term_D(ikl, isl) &
                                - Diag_A(ikl, isl) * eta_SV(ikl, isl - 1)) &
                               / Aux__P(ikl, isl)
        enddo
    enddo

    ! +--Backward Sweep
    ! +  ^^^^^^^^^^^^^^
    do isl = -1, -nsol, -1
        do ikl = 1, klonv
            eta_SV(ikl, isl) = Aux__Q(ikl, isl) * eta_SV(ikl, isl + 1) &
                               + eta_SV(ikl, isl)
        enddo
    enddo

    ! +--Horton RunOFF Intensity
    ! +  =======================
    do isl = 0, -nsol, -1
        do ikl = 1, klonv
            ! Soil Type
            ist = isotSV(ikl)
            ! OverSaturation Rate
            SatRat = (eta_SV(ikl, isl) - etadSV(ist)) &
                     * ro_Wat * dzAvSV(isl) &
                     * LSdzsv(ikl) &
                     / dt__SV
            SoRnOF(ikl) = SoRnOF(ikl) + max(zero, SatRat)
            SoRnOF2(ikl) = SoRnOF2(ikl) + max(zero, SatRat)
            RuofSV(ikl, 5) = RuofSV(ikl, 5) + max(zero, SatRat)
            eta_SV(ikl, isl) = max(epsi, eta_SV(ikl, isl))
            eta_SV(ikl, isl) = min(eta_SV(ikl, isl), etadSV(ist))
        enddo
    enddo

    ! +--IO, for Verification
    ! +  ~~~~~~~~~~~~~~~~~~~~
    ! #ifdef WR
    !     write(6, 6010)
    !     6010     format(/, 1x)
    ! #endif
    !     do isl = 0, -nsol, -1
    !         do ikl = 1, klonv
    !             ist = isotSV(ikl)
    !             ikp = nkhy * eta_SV(ikl, isl) / etadSV(ist)
    !             Khydsv(ikl, isl) = (aKdtSV3(ist, ikp) * eta_SV(ikl, isl)&
    !                     + bKdtSV3(ist, ikp)) * 2.0 / dt__SV
    ! #ifdef WR
    !             write(6, 6011) ikl, isl, eta_SV(ikl, isl) * 1.e3, &
    !                     ikp, aKdtSV3(ist, ikp), bKdtSV3(ist, ikp), &
    !                     Khydsv(ikl, isl)
    ! #endif
    !             6011 format(2i3, f8.1, i3, 3e12.3)
    !         end do
    !     end do

    ! +--Additional RunOFF Intensity
    ! +  ===========================
    do ikl = 1, klonv
        ist = isotSV(ikl)
        ik0 = nkhy * etaaux(ikl, -nsol) / etadSV(ist)
        ! FreeDr = FreeD0 *  min(ist,1)
        FreeDr = iWaFSV(ikl) * min(ist, 1)
        WExces = (aKdtSV3(ist, ik0) * (etaaux(ikl, -nsol) * Explic &
                                       + eta_SV(ikl, -nsol) * Implic) &
                  + bKdtSV3(ist, ik0)) &
                 * FreeDr * ro_Wat / dt__SV

        eta_SV(ikl, -nsol) = eta_SV(ikl, -nsol) - WExces * dt__SV &
                             / (ro_Wat * dzAvSV(-nsol) * LSdzsv(ikl))
        SoRnOF(ikl) = SoRnOF(ikl) + WExces
        SoRnOF2(ikl) = SoRnOF2(ikl) + WExces
        RuofSV(ikl, 6) = RuofSV(ikl, 6) + WExces

        ! +--Full Run OFF: Update
        ! +  ~~~~~~~~~~~~~~~~~~~~
        RnofSV(ikl) = RnofSV(ikl) + SoRnOF(ikl)
    enddo

    ! +--Mass conservation
    ! +  ^^^^^^^^^^^^^^^^^
    sum2 = 0
    do ikl = 1, klonv
        do isl = 0, -nsol, -1
            eta_SV(ikl, isl) = max(epsi, eta_SV(ikl, isl))
            sum2(ikl) = sum2(ikl) + eta_SV(ikl, isl) * dzAvSV(isl)
        enddo
        sum2(ikl) = sum2(ikl) + dt__SV * SoRnOF2(ikl) / (ro_Wat * LSdzsv(ikl))
    enddo

    do ikl = 1, klonv
        if(isotSV(ikl) > 0) then
            do isl = -nsol + 2, -nsol, -1
                eta_SV(ikl, isl) = eta_SV(ikl, isl) + (sum1(ikl) - sum2(ikl)) &
                                   / (3.*dzAvSV(isl))
                eta_SV(ikl, isl) = max(epsi, eta_SV(ikl, isl))
            enddo
        endif
    enddo

    ! +--Temperature Correction due to a changed Soil Energy Content
    ! +  ===========================================================

    ! +!!!    Mettre en oeuvre le couplage humidit?-?nergie

    ! +--Bumps/Asperites Treatment
    ! +  =========================

    ! +--Average over Bump Depth (z0soil)
    ! +  --------------------------------
#ifdef BP
    z_Bump = 0.
    do ikl = 1, klonv
        etBump(ikl) = 0.
    enddo
    do isl = 0, -nsol, -1
        z0Bump = z_Bump
        z_Bump = z_Bump + dzAvSV(isl)
        if(z_Bump < z0soil) then
            do ikl = 1, klonv
                etBump(ikl) = etBump(ikl) + dzAvSV(isl) * eta_SV(ikl, isl)
            enddo
        endif
        if(z_Bump > z0soil .and. z0Bump < z0soil) then
            do ikl = 1, klonv
                etBump(ikl) = etBump(ikl) + (z0soil - z0Bump) * eta_SV(ikl, isl)
                etBump(ikl) = etBump(ikl) / z0soil
            enddo
        endif
    enddo
    ! +--Correction
    ! +  ----------
    z_Bump = 0.
    do isl = 0, -nsol, -1
        z0Bump = z_Bump
        z_Bump = z_Bump + dzAvSV(isl)
        if(z_Bump < z0soil) then
            do ikl = 1, klonv
                eta_SV(ikl, isl) = etBump(ikl)
            enddo
        endif
        if(z_Bump > z0soil .and. z0Bump < z0soil) then
            dzBump = z_Bump - z0soil
            do ikl = 1, klonv
                eta_SV(ikl, isl) = (etBump(ikl) * (dzAvSV(isl) - dzBump) &
                                    + eta_SV(ikl, isl) * dzBump) &
                                   / dzAvSV(isl)
            enddo
        endif
    enddo
    ! +--Positive Definite
    ! +  =================
    do isl = 0, -nsol, -1
        do ikl = 1, klonv
            eta_SV(ikl, isl) = max(epsi, eta_SV(ikl, isl))
        enddo
    enddo
#endif
#ifdef m0
    ! +--Water  Budget (OUT)
    ! +  ===================
    do ikl = 1, klonv
        ! Precipitation is already included + Evaporation + Soil RunOFF Contrib.
        Wats_d(ikl) = Wats_d(ikl) &
                      + drr_SV(ikl) * zero &
                      + HLs_sv(ikl) &
                      * (1 - min(isnoSV(ikl), 1)) / Lx_H2O(ikl) &
                      - SoRnOF(ikl)
        Wats_1(ikl) = 0.
#ifdef mw
        Evapor(ikl) = HLs_sv(ikl) * dt__SV &
                      * (1 - min(isnoSV(ikl), 1)) / Lx_H2O(ikl)
#endif
    enddo
    do isl = -nsol, 0
        do ikl = 1, klonv
            ! Root Extract.
            Wats_d(ikl) = Wats_d(ikl) - Rootsv(ikl, isl)
        enddo
    enddo
    do ikl = 1, klonv
        Wats_d(ikl) = Wats_d(ikl) * dt__SV        !
    enddo
    isl = -nsol
    do ikl = 1, klonv
        Wats_1(ikl) = Wats_1(ikl) &
                      + ro_Wat * (eta_SV(ikl, isl) * dz78SV(isl) &
                                  + eta_SV(ikl, isl + 1) * dz_8SV(isl)) * LSdzsv(ikl)
    enddo
    do isl = -nsol + 1, -1
        do ikl = 1, klonv
            Wats_1(ikl) = Wats_1(ikl) &
                          + ro_Wat * (eta_SV(ikl, isl) * dz34SV(isl) &
                                      + (eta_SV(ikl, isl - 1) &
                                         + eta_SV(ikl, isl + 1)) * dz_8SV(isl)) * LSdzsv(ikl)
        enddo
    enddo
    isl = 0
    do ikl = 1, klonv
        Wats_1(ikl) = Wats_1(ikl) &
                      + ro_Wat * (eta_SV(ikl, isl) * dz78SV(isl) &
                                  + eta_SV(ikl, isl - 1) * dz_8SV(isl)) * LSdzsv(ikl)
    enddo
#endif
#ifdef mw
    ! +--Water  Budget (IO)
    ! +  ==================
    if(.not. mwopen) then
        mwopen = .true.
        open(unit=42, status='unknown', file='SISVAT_qSo.vw')
        rewind 42
        write(42, 42)
42      format('subroutine SISVAT_qSo: Local Water Budget', &
               /, '=========================================')
    endif
    timewr = timewr + dt__SV
    hourwr = 3600.0
    if(mod(timewr, hourwr) < epsi) write(42, 420) timewr / hourwr
420 format(11('-'), '----------+--------------+-', &
           3('-'), '----------+--------------+-', &
           '----------------+----------------+', &
           /, f8.2, 3x, 'Wats_0(1) |    Wats_d(1) | ', &
           3x, 'Wats_1(1) | W_0+W_d-W_1  | ', &
           '   Soil Run OFF |   Soil Evapor. |', &
           /, 11('-'), '----------+--------------+-', &
           3('-'), '----------+--------------+-', &
           '----------------+----------------+')
    write(42, 421) Wats_0(1), Wats_d(1) &
        , Wats_1(1) &
        , Wats_0(1) + Wats_d(1) - Wats_1(1) &
        , SoRnOF(1), Evapor(1)
421 format(8x, f12.6, ' + ', f12.6, ' - ', f12.6, ' = ', f12.6, ' | ', f12.6, &
           '      ', f15.6)
#endif
    return
end
