#include "MAR_pp.def"
subroutine SISVAT_qSn()
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_qSn                                02-12-2024  MAR |
    ! |   subroutine SISVAT_qSn updates  the Snow Water Content                |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |   ^^^^^                                                                |
    ! |                                                                        |
    ! |   INPUT:   TaT_SV   : SBL Top    Temperature                       [K] |
    ! |   ^^^^^    dt__SV   : Time Step                                    [s] |
    ! |                                                                        |
    ! |   INPUT /  drr_SV   : Rain Intensity                         [kg/m2/s] |
    ! |   OUTPUT:  dzsnSV   : Snow Layer Thickness                         [m] |
    ! |   ^^^^^^   eta_SV   : Snow Water Content                       [m3/m3] |
    ! |            ro__SV   : Snow/Soil  Volumic Mass                  [kg/m3] |
    ! |            TsisSV   : Soil/Ice Temperatures (layers -nsol,-nsol+1,..,0)|
    ! |                     & Snow     Temperatures (layers  1,2,...,nsno) [K] |
    ! |                                                                        |
    ! |   OUTPUT:  SWS_SV   : Surficial Water Status                           |
    ! |   ^^^^^^                                                               |
    ! |            EExcsv   : Snow Energy in Excess, initial Forcing    [J/m2] |
    ! |            EqSn_d   : Snow Energy in Excess, remaining          [J/m2] |
    ! |            EqSn_0   : Snow Energy, before Phase Change          [J/m2] |
    ! |            EqSn_1   : Snow Energy, after  Phase Change          [J/m2] |
    ! |            SIsubl   : Snow sublimed/deposed Mass             [mm w.e.] |
    ! |            SImelt   : Snow Melted           Mass             [mm w.e.] |
    ! |            SIrnof   : Surficial Water + Run OFF Change       [mm w.e.] |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |                                                                        |
    ! | # OPTIONS: #E0: IO for Verification: Energy       Budget               |
    ! | # ^^^^^^^                                                              |
    ! | #          #su: IO for Verification: Slush        Diagnostic           |
    ! |                                                                        |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! |   FILE                 |      CONTENT                                  |
    ! |   ~~~~~~~~~~~~~~~~~~~~~+~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ |
    ! | # SISVAT_qSn.vm        | #vm: OUTPUT/Verification: Energy/Water Budget |
    ! |                        |      unit 43, subroutine  SISVAT_qSn **ONLY** |
    ! | # SISVAT_qSn.vu        | #vu: OUTPUT/Verification: Slush  Parameteriz. |
    ! |                        |      unit 44, subroutine  SISVAT_qSn **ONLY** |
    ! +------------------------------------------------------------------------+

    use marphy
    use mar_sv
    use mardsv
    use mar0sv
    use marxsv
    use marysv

    implicit none

#ifdef e1
    ! Energy Budget
    ! ~~~~~~~~~~~~~
    ! EqSn_d : Energy in Excess, initial
    real EqSn_d(klonv)
    ! EqSn_0 : Snow Energy, befor Phase Change
    real EqSn_0(klonv)
    ! EqSn_1 : Snow Energy, after Phase Change .and. Mass Redistr.
    real EqSn_1(klonv)
#endif

#ifdef vm
    ! EqSn01 : Snow Energy, after Phase Change
    real EqSn01(klonv)
    ! EqSn02 : Snow Energy, after Phase Change .and. Last Melting
    real EqSn02(klonv)
#endif
    ! Snow/Ice (Mass) Budget
    ! ~~~~~~~~~~~~~~~~~~~~~~
#ifdef m1
    ! SIsubl : Snow Deposed Mass
    real SIsubl(klonv)
    ! SImelt : Snow Melted  Mass
    real SImelt(klonv)
    ! SIrnof : Local Surficial Water + Run OFF
    real SIrnof(klonv)
#endif

    ! +--Internal Variables
    ! +  ==================
    integer ikl, isn, flag
    ! nh : Non erodible Snow: up.lay.Index
    integer nh
    ! LayrOK : 1 (0)  if In(Above) Snow Pack
    integer LayrOK
    ! k_face : 1 (0)  if Crystal(no) faceted
    integer k_face
    ! LastOK : 1 ==>  1! Snow Layer
    integer LastOK
    ! NOLayr : 1     Layer  Update
    integer NOLayr
    ! noSnow : Nb of Layers Updater
    integer noSnow(klonv)
    ! noSnow : Slush Switch
    integer kSlush
    ! dTSnow : Temperature[C]
    real dTSnow
    ! EExdum : Energy in Excess when no Snow
    real EExdum(klonv)
    ! OKmelt : 1 (0)  if(no) Melting
    real OKmelt
    ! EnMelt : Energy in excess, for Melting
    real EnMelt
    ! SnHLat : Energy consumed   in  Melting
    real SnHLat
    ! AdEnrg, B_Enrg : Additional Energy from  Vapor
    real AdEnrg, B_Enrg
    ! Vaporized Thickness [m]
    real dzVap0, dzVap1
    ! Melted    Thickness [m]
    real dzMelt(klonv)
    ! rosDry : Snow volumic Mass if no Water in
    real rosDry
    ! PorVol : Pore volume
    real PorVol
    ! PClose : Pore Hole Close OFF Switch
    real PClose
    ! SGDiam : Snow Grain Diameter
    real SGDiam
    ! SGDmax : Max. Snow Grain Diameter
    real SGDmax
    ! rWater : Retained Water [kg/m2]
    real rWater
    ! drrNEW : New available Water [kg/m2]
    real drrNEW
    ! rdzNEW : Snow Mass [kg/m2]
    real rdzNEW
    ! rdzsno : Snow Mass [kg/m2]
    real rdzsno
    ! EnFrez : Energy Release in  Freezing
    real EnFrez
    ! WaFrez : Water  consumed in  Melting
    real WaFrez
    ! RapdOK : 1. ==> Snow melts rapidly
    real RapdOK
    ! ThinOK : 1. ==> Snow Layer is thin
    real ThinOK
    ! dzepsi : Minim. Snow Layer Thickness (!)
    real dzepsi
    ! dz_Min : Minim. Snow Layer Thickness
    real dz_Min
    ! z_Melt : Last (thin) Layer Melting
    real z_Melt
    ! rusnew : Surficial Water Thickness [mm]
    real rusnew
    ! zWater : Max Slush Water Thickness [mm]
    real zWater
    ! zSlush : Slush Water Thickness [mm]
    real zSlush
    ! ro_new : New Snow/ice Density [kg/m3]
    real ro_new
    ! zc, zt : Non erod.Snow Thickness [mm w.e.]
    real zc, zt
    ! dru : Surficial Water [kg/m2]
    real dru
    real rusnSV0(klonv), sum1(klonv), sum2(klonv)
    real drr1(klonv), drr2(klonv), ws0dSV2

    ! +--OUTPUT of SISVAT Trace Statistics (see assignation in PHY_SISVAT)
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer isnnew, isinew, isnUpD, isnitr
#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
    common / SISVAT_EV / iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
#endif
#ifdef vm
    ! +--Energy and Mass Budget
    ! +  ~~~~~~~~~~~~~~~~~~~~~~
    ! WqSn_0 : Snow Water+Forcing  Initial
    real WqSn_0(klonv)
    ! WqSn_1 : Snow Water+Forcing, Final
    real WqSn_1(klonv)
    ! emopen : IO Switch
    logical emopen
    common / Se_qSn_L / emopen
    integer no_err
    common / Se_qSn_I / no_err
    real hourer, timeer
    common / Se_qSn_R / timeer
#endif
#ifdef vu
    ! +--Slush Diagnostic: IO
    ! +  ~~~~~~~~~~~~~~~~~~~~
    ! su_opn : IO   Switch
    logical su_opn
    common / SI_qSn_L / su_opn
#endif

    ! +--DATA
    ! +  ====
    ! dzepsi : Minim. Snow Layer Thickness (!)
    data dzepsi/0.0001/
    ! dzepsi = 0.005 : Warning: Too high for Col de Porte: precludes 1st snow (layer) apparition
    ! data dz_Min/0.005/
    ! dz_Min : Minim. Snow Layer Thickness
    data dz_Min/2.5e-3/
    ! SGDmax : Maxim. Snow Grain Diameter [m] (Rowe et al. 1995, JGR p.16268)
    data SGDmax/0.003/

#ifdef e1
    ! +--Energy Budget (IN)
    ! +  ==================
    do ikl = 1, klonv
        EqSn_0(ikl) = 0.
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            EqSn_0(ikl) = EqSn_0(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * (Cn_dSV * (TsisSV(ikl, isn) - TfSnow) &
                             - Lf_H2O * (1.-eta_SV(ikl, isn)))
        enddo
    enddo
#endif

#ifdef vm
    ! +--Water  Budget (IN)
    ! +  ==================
    do ikl = 1, klonv
        WqSn_0(ikl) = drr_SV(ikl) * dt__SV &
                      + rusnSV(ikl)
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            WqSn_0(ikl) = WqSn_0(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn)
        enddo
    enddo
#endif

#ifdef m1
    ! +--Snow Melt Budget
    ! +  ================
    do ikl = 1, klonv
        SImelt(ikl) = 0.
        SIrnof(ikl) = rusnSV(ikl) + RnofSV(ikl) * dt__SV
    enddo
#endif

    ! +--Initialization
    ! +  ==============

    do ikl = 1, klonv
        ! noSnow : Nb of Layers Updater
        noSnow(ikl) = 0
        ! ispiSV : Pore Hole Close OFF Index
        ! (assumed to be the Top of the surimposed Ice Layer)
        ispiSV(ikl) = 0
        zn5_SV(ikl) = 0.
        rusnSV0(ikl) = 0.
        sum1(ikl) = 0
        sum2(ikl) = 0
        do isn = 1, isnoSV(ikl)
            sum1(ikl) = sum1(ikl) + dzsnSV(ikl, isn) * ro__SV(ikl, isn)
        enddo
        drr1(ikl) = drr_SV(ikl) * dt__SV
    enddo

    ! +--Melting/Freezing Energy
    ! +  =======================

    ! +...REMARK: Snow liquid Water Temperature assumed = TfSnow
    ! +   ^^^^^^
    do ikl = 1, klonv
        EExdum(ikl) = drr_SV(ikl) * C__Wat * (TaT_SV(ikl) - TfSnow) &
                      * dt__SV
        EExcsv(ikl) = EExdum(ikl) * min(1, isnoSV(ikl)) ! Snow exists
        EExdum(ikl) = EExdum(ikl) - EExcsv(ikl)  !
#ifdef e1
        EqSn_d(ikl) = EExcsv(ikl)
#endif
    enddo

    ! +--Surficial Water Status
    ! +  ----------------------

    do ikl = 1, klonv
        SWS_SV(ikl) = max(zero, sign(unun, TfSnow &
                                     - TsisSV(ikl, isnoSV(ikl))))
    enddo

    do ikl = 1, klonv
        zt = 0. ; zc=0.
        do isn = 1, isnoSV(ikl)
            zt = zt + dzsnSV(ikl, isn)
        enddo
        do isn = isnoSV(ikl), 1, -1
            zc = zc + dzsnSV(ikl, isn)
            if(ws0dSV>=0.02) then
             ws0dSV2=0.02 + (ws0dSV - 0.02) * (1 - zc/1.)  
             ws0dSV2=max(0.02,min(ws0dSV,ws0dSV2))
            else
             ws0dSV2=ws0dSV
            endif

            ! +--Energy, store Previous Content
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            dTSnow = TsisSV(ikl, isn) - TfSnow
            EExcsv(ikl) = EExcsv(ikl) &
                          + ro__SV(ikl, isn) * Cn_dSV * dTSnow &
                          * dzsnSV(ikl, isn)
            TsisSV(ikl, isn) = TfSnow

            ! +--Water,  store Previous Content
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            drr_SV(ikl) = drr_SV(ikl) &
                          + ro__SV(ikl, isn) * eta_SV(ikl, isn) &
                          * dzsnSV(ikl, isn) &
                          / dt__SV
            ro__SV(ikl, isn) = &
                ro__SV(ikl, isn) * (1.-eta_SV(ikl, isn))
            eta_SV(ikl, isn) = 0.

            ! +--Melting  if EExcsv > 0
            ! +  ======================

            EnMelt = max(zero, EExcsv(ikl))

            ! +--Energy Consumption
            ! +  ^^^^^^^^^^^^^^^^^^
            SnHLat = ro__SV(ikl, isn) * Lf_H2O
            dzMelt(ikl) = EnMelt / max(SnHLat, epsi)
            noSnow(ikl) = noSnow(ikl) &
                          + max(zero, sign(unun, dzMelt(ikl) &
                                           ! 1 if full Melt
                                           - dzsnSV(ikl, isn))) &
                          ! 1 in the  Pack
                          * min(1, max(0, 1 + isnoSV(ikl) - isn))
            dzMelt(ikl) = &
                min(dzsnSV(ikl, isn), dzMelt(ikl))
            dzsnSV(ikl, isn) = &
                dzsnSV(ikl, isn) - dzMelt(ikl)
            zn5_SV(ikl) = zn5_SV(ikl) + dzMelt(ikl)
            EExcsv(ikl) = EExcsv(ikl) - dzMelt(ikl) * SnHLat
            wem_SV(ikl) = wem_SV(ikl) - dzMelt(ikl) * ro__SV(ikl, isn)

            ! +--Water  Production
            ! +  ^^^^^^^^^^^^^^^^^
            drr_SV(ikl) = drr_SV(ikl) &
                          + ro__SV(ikl, isn) * dzMelt(ikl) / dt__SV
#ifdef m1
            SImelt(ikl) = SImelt(ikl) &
                          + ro__SV(ikl, isn) * dzMelt(ikl)
#endif
            OKmelt = max(zero, sign(unun, drr_SV(ikl) - epsi))

            ! +--Snow History
            ! +  ^^^^^^^^^^^^
            ! = 1  if faceted
            k_face = min(istoSV(ikl, isn), istdSV(1)) &
                     * max(0, 2 - istoSV(ikl, isn))
            istoSV(ikl, isn) = &
                (1.-OKmelt) * istoSV(ikl, isn) &
                + OKmelt * ((1 - k_face) * istdSV(2) &
                            + k_face * istdSV(3))

            ! +--Freezing if EExcsv < 0
            ! +  ======================
            flag = 0
            if(drr_SV(ikl) == 0       .and. isn == isnoSV(ikl)   .and. &
               ro__SV(ikl, isn) > 900 .and. ro__SV(ikl, 1) > 900 .and. & 
               rusnSV(ikl) > 0        .and. TaT_SV(ikl)<273.and.EExcsv(ikl)<0) then
                flag        = 1
                drr_SV(ikl) = rusnSV(ikl) / dt__SV
                dru         = rusnSV(ikl)
            endif
            rdzsno = ro__SV(ikl, isn) * dzsnSV(ikl, isn)
            LayrOK = min(1, max(0, isnoSV(ikl) - isn + 1))
            EnFrez = min(zero, EExcsv(ikl))
            WaFrez = -(EnFrez * LayrOK / Lf_H2O)
            drrNEW = max(zero, drr_SV(ikl) - WaFrez / dt__SV)
            WaFrez = (drr_SV(ikl) - drrNEW) * dt__SV
            drr_SV(ikl) = drrNEW
            EExcsv(ikl) = EExcsv(ikl) + WaFrez * Lf_H2O
            EnFrez = min(zero, EExcsv(ikl)) * LayrOK
            rdzNEW = WaFrez + rdzsno
            if(ro__SV(ikl, isn)<=900) then
               ro__SV(ikl, isn) = rdzNEW / max(epsi, dzsnSV(ikl, isn))
            else
               dzsnSV(ikl, isn) = rdzNEW / ro__SV(ikl, isn)
            endif
            TsisSV(ikl, isn) = TfSnow &
                               + EnFrez / (Cn_dSV * max(epsi, rdzNEW))
            EExcsv(ikl) = EExcsv(ikl) - EnFrez
            wer_SV(ikl) = WaFrez + wer_SV(ikl)
            if(flag == 1) then
                rusnSV(ikl)    = drr_SV(ikl) * dt__SV
                dru            =             dru - rusnSV(ikl)
                drr1(ikl)      = drr1(ikl) + dru 
                RuofSV(ikl, 2) = max(0., RuofSV(ikl, 2) - dru / dt__SV)
                drr_SV(ikl)    = 0.
                flag           = 0.
            endif

            ! +--Snow Water Content
            ! +  ==================

            ! +--Percolation Velocity
            ! +  ^^^^^^^^^^^^^^^^^^^^
#ifdef PW
            SGDiam = 1.6d-4 &
                     + 1.1d-13 * (ro__SV(ikl, isn) * ro__SV(ikl, isn) &
                                  * ro__SV(ikl, isn) * ro__SV(ikl, isn))
#endif

            ! +--Pore   Volume [-]
            ! +  ^^^^^^^^^^^^^^^^^
            rosDry = (1.-eta_SV(ikl, isn)) * ro__SV(ikl, isn) !
            PorVol = 1.-rosDry / ro_Ice          !
            PorVol = max(PorVol, zero)         !

            ! +--Water  Retention
            ! +  ^^^^^^^^^^^^^^^^
            rWater = ws0dSV2 * PorVol * ro_Wat * dzsnSV(ikl, isn)
            drrNEW = max(zero, drr_SV(ikl) - rWater / dt__SV)
            rWater = (drr_SV(ikl) - drrNEW) * dt__SV
            drr_SV(ikl) = drrNEW
            rdzNEW = rWater &
                     + rosDry * dzsnSV(ikl, isn)
            eta_SV(ikl, isn) = rWater / max(epsi, rdzNEW)
            ro__SV(ikl, isn) = rdzNEW / max(epsi, dzsnSV(ikl, isn))

            ! +--Pore Hole Close OFF
            ! +  ^^^^^^^^^^^^^^^^^^^
            PClose = max(zero, &
                         sign(unun, ro__SV(ikl, isn) &
                              - roCdSV))
            ! Water under SuPer.Ice contributes to Surficial Water
            ispiSV(ikl) = ispiSV(ikl) * (1.-PClose) &
                          + max(ispiSV(ikl), isn) * Pclose
            ! PClose = max(0, min (1, ispiSV(ikl) - isn))
            PClose = 1.-(ro_ice - ro__SV(ikl, isn)) / (ro_ice - roCdSV)
            PClose = max(0., min(1., PClose))

            if(ro__SV(ikl, isn) > 900) then
                dzsnSV(ikl, isn) = dzsnSV(ikl, isn) * ro__SV(ikl, isn) / ro_ice
                ro__SV(ikl, isn) = ro_ice
                PClose = 1
                ! eta_SV(ikl, isn) = 0
            endif

            if(isnoSV(ikl) >= 3 .and. &
               ro__SV(ikl, isn) >= roCdSV .and. &
               (ro__SV(ikl, 1) * dzsnSV(ikl, 1) + &
                ro__SV(ikl, 2) * dzsnSV(ikl, 2)) / &
               (dzsnSV(ikl, 1) + dzsnSV(ikl, 2)) < 900) &
                PClose = PClose / 3. ! ice lense
#ifdef EU
            zt = 1.
#endif
            if(isn == 1 .and. zt > 15) then
                ! > 15 meter of snwow => ice sheet
                PClose = 1
                ispiSV(ikl) = max(ispiSV(ikl), 1)
            endif

            if(isnoSV(ikl) == 0) PClose = 0.
            rusnSV(ikl) = rusnSV(ikl) &
                          + drr_SV(ikl) * dt__SV * PClose
            rusnSV0(ikl) = rusnSV0(ikl) &
                           + drr_SV(ikl) * dt__SV * PClose
            drr_SV(ikl) = drr_SV(ikl) * (1.-PClose)

        enddo
    enddo

    ! +--Remove Zero-Thickness Layers
    ! +  ============================

1000 continue
    isnitr = 0
    do ikl = 1, klonv
        isnUpD = 0
        isinew = 0
        !XF
        do isn = 1, min(nsno - 1, isnoSV(ikl))
            isnnew = (unun - max(zero, sign(unun, dzsnSV(ikl, isn) - dzepsi))) &
                     * max(0, min(1, isnoSV(ikl) + 1 - isn))
            isnUpD = max(isnUpD, isnnew)
            isnitr = max(isnitr, isnnew)
            ! LowerMost  0-Layer
            isinew = isn * isnUpD * max(0, 1 - isinew) &
                     + isinew       ! Index
            dzsnSV(ikl, isn) = dzsnSV(ikl, isn + isnnew)
            ro__SV(ikl, isn) = ro__SV(ikl, isn + isnnew)
            TsisSV(ikl, isn) = TsisSV(ikl, isn + isnnew)
            eta_SV(ikl, isn) = eta_SV(ikl, isn + isnnew)
            G1snSV(ikl, isn) = G1snSV(ikl, isn + isnnew)
            G2snSV(ikl, isn) = G2snSV(ikl, isn + isnnew)
            dzsnSV(ikl, isn + isnnew) = (1 - isnnew) * dzsnSV(ikl, isn + isnnew)
            ro__SV(ikl, isn + isnnew) = (1 - isnnew) * ro__SV(ikl, isn + isnnew)
            eta_SV(ikl, isn + isnnew) = (1 - isnnew) * eta_SV(ikl, isn + isnnew)
            G1snSV(ikl, isn + isnnew) = (1 - isnnew) * G1snSV(ikl, isn + isnnew)
            G2snSV(ikl, isn + isnnew) = (1 - isnnew) * G2snSV(ikl, isn + isnnew)
        enddo
        isnoSV(ikl) = isnoSV(ikl) - isnUpD            ! Nb of Snow   Layer
        ! Nb of SuperI Layer
        ispiSV(ikl) = ispiSV(ikl) &
                      ! Update  if I=0
                      - isnUpD * max(0, min(ispiSV(ikl) - isinew, 1))

        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef wx
        if(isnUpD > 0 .and. ikl == kSV_v1 .and. lSV_v1 == 3) &
            write(6, *) ' MERGE ', isnoSV(ikl), ' Grid ', iSV_v1, jSV_v1, nSV_v1
#endif

    enddo
    if(isnitr > 0) go to 1000

    ! +--New upper Limit of the non erodible Snow (istoSV .GT. 1)
    ! +  ========================================

    do ikl = 1, klonv
        nh = 0
        !XF
        do isn = isnoSV(ikl), 1, -1
            nh = nh + isn * min(istoSV(ikl, isn) - 1, 1) * max(0, 1 - nh)
        enddo
        zc = 0.
        zt = 0.
        !XF
        do isn = 1, isnoSV(ikl)
            zc = zc + dzsnSV(ikl, isn) * ro__SV(ikl, isn) &
                 * max(0, min(1, nh + 1 - isn))
            zt = zt + dzsnSV(ikl, isn) * ro__SV(ikl, isn)
        enddo
        zWE_SV(ikl) = zt
        zWEcSV(ikl) = min(zWEcSV(ikl), zt)
        zWEcSV(ikl) = max(zWEcSV(ikl), zc)
    enddo

    ! +--Energy Budget (OUT)
    ! +  ===================

#ifdef vm
    do ikl = 1, klonv
        EqSn01(ikl) = -EqSn_0(ikl) &
                      - EExcsv(ikl)
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            EqSn01(ikl) = EqSn01(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * (Cn_dSV * (TsisSV(ikl, isn) - TfSnow) &
                             - Lf_H2O * (1.-eta_SV(ikl, isn)))
        enddo
    enddo
#endif

    ! +--"Negative Heat" from supercooled rain
    ! +   ------------------------------------

    do ikl = 1, klonv
        EExcsv(ikl) = EExcsv(ikl) + EExdum(ikl)

        ! +--Surficial Water Run OFF
        ! +  -----------------------

        rusnew = rusnSV(ikl) * SWf_SV(ikl)

        if(isnoSV(ikl) <= 1) rusnew = 0.
        if(ivgtSV(ikl) >= 1) rusnew = 0.

#ifdef EU
        rusnew = 0.
#endif
#ifdef AC
        rusnew = 0.
#endif
        RnofSV(ikl) = RnofSV(ikl) + (rusnSV(ikl) - rusnew) / dt__SV
        RuofSV(ikl, 2) = RuofSV(ikl, 2) + (rusnSV0(ikl)) / dt__SV
        rusnSV(ikl) = rusnew
    enddo

    ! +--Percolation down the Continental Ice Pack
    ! +  -----------------------------------------

    ! do ikl = 1, klonv
    !     drr_SV(ikl) = drr_SV(ikl) + &
    !             rusnSV(ikl) * (1 - min(1, ispiSV(ikl))) / dt__SV
    !     rusnSV(ikl) = rusnSV(ikl) * min(1, ispiSV(ikl))
    ! end do

    !XF removal of too thin snowlayers if TT> 275.15 + bug if TT>> 273.15
    do ikl = 1, klonv
        zt = 0.
        do isn = 1, isnoSV(ikl)
            zt = zt + dzsnSV(ikl, isn)
        enddo
        if(zt < 0.005 + (TaT_SV(ikl) - 275.15) / 1000. .and. &
           isnoSV(ikl) > 0 .and. &
           TaT_SV(ikl) >= 275.15 .and. &
           istoSV(ikl, isnoSV(ikl)) > 1) then
            do isn = 1, isnoSV(ikl)
                drr_SV(ikl) = drr_SV(ikl) &
                              + dzsnSV(ikl, isn) * ro__SV(ikl, isn) / dt__SV
                dzsnSV(ikl, isn) = 0.
            enddo
            isnoSV(ikl) = 0
        endif
    enddo

    ! +--Slush Formation (CAUTION: ADD RunOff Possibility before Activation)
    ! +  ---------------  ^^^^^^^  ^^^

#ifdef vu
    if(.not. su_opn) then
        su_opn = .true.
        open(unit=44, status='unknown', file='SISVAT_qSn.vu')
        rewind 44
    endif
    write(44, 440) daHost
440 format('iSupI    i       dz       ro      eta', &
           '   PorVol   zSlush     ro_n    eta_n', 2x, a18)
#endif

    ! #ifdef SU
    !     do ikl = 1, klonv
    !         do isn = 1, isnoSV(ikl)
    !             kSlush = min(1, max(0, isn + 1 - ispiSV(ikl)))        ! Slush Switch
    !             ! +--Available Additional Pore   Volume [-]
    !             ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    !             ! [--]
    !             PorVol = 1. - ro__SV(ikl, isn) &
    !                     * (1. - eta_SV(ikl, isn)) / ro_Ice &
    !                     - eta_SV(ikl, isn) &
    !                             * ro__SV(ikl, isn) / ro_Wat
    !             PorVol = max(PorVol, zero)
    !             ! [mm] OR [kg/m2]
    !             ! 0 <=> freezing
    !             ! 1 <=> isn=isnoSV
    !             zWater = dzsnSV(ikl, isn) * PorVol * 1000. &
    !                     * (1. - SWS_SV(ikl) &
    !                             * (1 - min(1, iabs(isn - isnoSV(ikl)))))
    !             ! [mm] OR [kg/m2]
    !             zSlush = min(rusnSV(ikl), zWater)
    !             ro_new = (dzsnSV(ikl, isn) * ro__SV(ikl, isn) &
    !                     + zSlush) &
    !                     / max(dzsnSV(ikl, isn), epsi)
    !             if(ro_new<ro_Ice + 20) then ! MAX 940kg/m3
    !                 ! [mm] OR [kg/m2]
    !                 rusnSV(ikl) = rusnSV(ikl) - zSlush
    !                 rusnSV0(ikl) = rusnSV0(ikl) - zSlush
    !                 RuofSV(ikl, 2) = max(0., RuofSV(ikl, 2) - zSlush / dt__SV)
    !                 eta_SV(ikl, isn) = (ro_new - ro__SV(ikl, isn) &
    !                         * (1. - eta_SV(ikl, isn))) &
    !                         / max (ro_new, epsi)
    !                 ro__SV(ikl, isn) = ro_new
    !             end if
    !         end do
    !     end do
    ! #endif

    ! +--Mass conservation
    ! +  =================
    do ikl = 1, klonv
        RuofSV(ikl, 3) = RuofSV(ikl, 3) + drr_SV(ikl)
        drr2(ikl) = drr_SV(ikl) * dt__SV + rusnSV0(ikl)
        do isn = 1, isnoSV(ikl)
            sum2(ikl) = sum2(ikl) + dzsnSV(ikl, isn) * ro__SV(ikl, isn)
        enddo
        sum1(ikl) = sum1(ikl) + (drr1(ikl) - drr2(ikl))
        isn = 1
        if(isnoSV(ikl) > 1) then
            dzsnSV(ikl, isn) = dzsnSV(ikl, isn) + (sum1(ikl) - sum2(ikl)) / ro__SV(ikl, isn)
        endif
    enddo

    ! +--Impact of the Sublimation/Deposition on the Surface Mass Balance
    ! +  ================================================================

    do ikl = 1, klonv
        isn = isnoSV(ikl)
        dzVap0 = dt__SV * HLs_sv(ikl) * min(isn, 1) &
                 / (Lx_H2O(ikl) * max(ro__SV(ikl, isn), epsi))
        NOLayr = min(zero, sign(unun, dzsnSV(ikl, isn) + dzVap0))
        dzVap1 = min(zero, dzsnSV(ikl, isn) + dzVap0)

        ! +--Additional Energy
        ! +  -----------------

#ifdef VH
        ! Water   Vapor Sensible Heat
        AdEnrg = dzVap0 * ro__SV(ikl, isnoSV(ikl)) &
                 * C__Wat * (TsisSV(ikl, isnoSV(ikl)) - TfSnow)
#endif

#ifdef aH
        B_Enrg = (Cn_dSV * (TsisSV(ikl, isn) - TfSnow) &
                  - Lf_H2O * (1.-eta_SV(ikl, isn))) &
                 / (1.+dzVap0 / max(epsi, dzsnSV(ikl, isn)))
        eta_SV(ikl, isn) = &
            max(zero, unun + (B_Enrg &
                              - (TsisSV(ikl, isn) - TfSnow) * Cn_dSV) &
                / Lf_H2O)
        TsisSV(ikl, isn) = (B_Enrg &
                            + (1.-eta_SV(ikl, isn)) &
                            * Lf_H2O) &
                           / Cn_dSV &
                           + TfSnow
#endif

#ifdef e1
        STOP "PLEASE add Energy (#aH) from deposition/sublimation"
#endif
        ! +--Update of the upper Snow layer Thickness
        ! +  ----------------------------------------
        dzsnSV(ikl, isn) = max(zero, dzsnSV(ikl, isnoSV(ikl)) + dzVap0)
        isnoSV(ikl) = isnoSV(ikl) + NOLayr
        isn = isnoSV(ikl)
        dzsnSV(ikl, isn) = dzsnSV(ikl, isn) + dzVap1
        wee_SV(ikl, 3) = wee_SV(ikl, 3) - ro__SV(ikl, isn) * dzVap0
    enddo

#ifdef vm
    ! +--Energy Budget (OUT)
    ! +  ===================
    do ikl = 1, klonv
        EqSn02(ikl) = -EqSn_0(ikl) - EExcsv(ikl)
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            EqSn02(ikl) = EqSn01(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * (Cn_dSV * (TsisSV(ikl, isn) - TfSnow) &
                             - Lf_H2O * (1.-eta_SV(ikl, isn)))
        enddo
    enddo
#endif

#ifdef m1
    ! +--Snow/I Budget
    ! +  -------------
    do ikl = 1, klonv
        SIsubl(ikl) = dt__SV * HLs_sv(ikl) * min(isnoSV(ikl), 1) &
                      / Lx_H2O(ikl)
        SIrnof(ikl) = rusnSV(ikl) + RnofSV(ikl) * dt__SV &
                      - SIrnof(ikl)
    enddo
#endif

    ! +--Anticipated Disappearance of a rapidly Melting too thin Last Snow Layer
    ! +  =======================================================================
    do ikl = 1, klonv
        LastOK = min(1, max(0, iiceSV(ikl) - isnoSV(ikl) + 2) &
                     * min(1, isnoSV(ikl) - iiceSV(ikl)) &
                     + min(1, isnoSV(ikl)))
        RapdOK = max(zero, sign(unun, dzMelt(ikl) - epsi))
        ThinOK = max(zero, sign(unun, dz_Min - dzsnSV(ikl, 1)))
        z_Melt = LastOK * RapdOK * ThinOK
        noSnow(ikl) = noSnow(ikl) + z_Melt
        z_Melt = z_Melt * dzsnSV(ikl, 1)
        dzsnSV(ikl, 1) = dzsnSV(ikl, 1) - z_Melt
        EExcsv(ikl) = EExcsv(ikl) - z_Melt * ro__SV(ikl, 1) &
                      * (1.-eta_SV(ikl, 1)) * Lf_H2O

        ! +--Water  Production
        ! +  ^^^^^^^^^^^^^^^^^
        drr_SV(ikl) = drr_SV(ikl) + ro__SV(ikl, 1) * z_Melt / dt__SV
    enddo

    ! +--Update Nb of Layers
    ! +  ===================
#ifdef EF
    if(isnoSV(1) > 0) &
        write(6, 6005) noSnow(1)
6005 format(i3, ' (noSnow) ')
#endif
    do ikl = 1, klonv
        isnoSV(ikl) = isnoSV(ikl) * min(1, iabs(isnoSV(ikl) - noSnow(ikl)))
    enddo

#ifdef e1
    ! Energy Budget (OUT)
    ! ===================
    do ikl = 1, klonv
        EqSn_1(ikl) = 0.
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            EqSn_1(ikl) = EqSn_1(ikl) + ro__SV(ikl, isn) * dzsnSV(ikl, isn) &
                          * (Cn_dSV * (TsisSV(ikl, isn) - TfSnow) &
                             - Lf_H2O * (1.-eta_SV(ikl, isn)))
        enddo
    enddo
#endif

#ifdef vm
    ! +--Water  Budget (OUT)
    ! +  ===================
    do ikl = 1, klonv
        WqSn_0(ikl) = WqSn_0(ikl) &
                      + HLs_sv(ikl) * dt__SV &
                      * min(isnoSV(ikl), 1) / Lx_H2O(ikl)
        WqSn_1(ikl) = drr_SV(ikl) * dt__SV &
                      + rusnSV(ikl) &
                      + RnofSV(ikl) * dt__SV
    enddo
    do isn = nsno, 1, -1
        do ikl = 1, klonv
            WqSn_1(ikl) = WqSn_1(ikl) &
                          + ro__SV(ikl, isn) * dzsnSV(ikl, isn)
        enddo
    enddo
    ! +--OUTPUT Budget
    ! +  =============
    if(.not. emopen) then
        emopen = .true.
        open(unit=43, status='unknown', file='SISVAT_qSn.vm')
        rewind 43
        write(43, 43)
43      format('subroutine SISVAT_qSn: Local Energy and Water Budgets', &
               /, '=====================================================')
    endif
    do ikl = 1, klonv
        if(EqSn01(ikl) > 1.e-3) write(43, 431) dahost, EqSn01(ikl)
431     format(' WARNING (1) in _qSn,', a18, &
               ': Energy Unbalance in Phase Change = ', e15.6)
    enddo
    do ikl = 1, klonv
        if(EqSn02(ikl) > 1.e-3) write(43, 432) dahost, EqSn01(ikl)
432     format(' WARNING (2) in _qSn,', a18, &
               ': Energy Unbalance in Phase Change = ', e15.6)
    enddo
    timeer = timeer + dt__SV
    hourer = 3600.0
    if(mod(no_err, 11) == 0) then
        no_err = 1
        write(43, 435) timeer / hourer
435     format(11('-'), '----------+-', 3('-'), '----------+-', &
               3('-'), '----------+-', 3('-'), '----------+-', &
               '----------------+----------------+', &
               /, f8.2, 3x, 'EqSn_0(1) | ', 3x, 'EqSn_d(1) | ', &
               3x, 'EqSn_1(1) | ', 3x, 'EExcsv(1) | ', &
               'E_0+E_d-E_1-EE  |   Water Budget |', &
               /, 11('-'), '----------+-', 3('-'), '----------+-', &
               3('-'), '----------+-', 3('-'), '----------+-', &
               '----------------+----------------+')
    endif
    if(abs(EqSn_0(1) + EqSn_d(1) - EqSn_1(1) - EExcsv(1)) > epsi .OR. &
       abs(WqSn_1(1) - WqSn_0(1)) > epsi) then
        no_err = no_err + 1
        write(43, 436) EqSn_0(1), EqSn_d(1) &
            , EqSn_1(1), EExcsv(1) &
            , EqSn_0(1) + EqSn_d(1) - EqSn_1(1) - EExcsv(1) &
            , WqSn_1(1) - WqSn_0(1)
436     format(8x, f12.0, ' + ', f12.0, ' - ', f12.0, ' - ', f12.3, ' = ', f12.3, &
               '    | ', f15.9)
    endif
#endif
#ifdef e1
    do ikl = 1, klonv
        EqSn_d(ikl) = EqSn_d(ikl) - EExcsv(ikl)
    enddo
#endif

    return
endsubroutine SISVAT_qSn
