#include "MAR_pp.def"
subroutine SISVAT_ini
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_ini                            Sun 01-04-2025  MAR |
    ! |   subroutine SISVAT_ini generates non time dependant SISVAT parameters |
    ! +------------------------------------------------------------------------+
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   dt__SV   : Time  Step                                   [s] |
    ! |   ^^^^^    dz_dSV   : Layer Thickness                              [m] |
    ! |                                                                        |
    ! |   OUTPUT:  RF__SV   : Root Fraction in Layer isl                   [-] |
    ! |   ^^^^^^   rocsSV   : Soil Contrib. to (ro c)_s exclud.Water  [J/kg/K] |
    ! |            etamSV   : Soil Minimum Humidity                    [m3/m3] |
    ! |                      (based on a prescribed Soil Relative Humidity)    |
    ! |            s1__SV   : Factor of eta**( b+2) in Hydraul.Diffusiv.       |
    ! |            s2__SV   : Factor of eta**( b+2) in Hydraul.Conduct.        |
    ! |            aKdtSV   : KHyd: Piecewise Linear Profile:  a * dt    [m]   |
    ! |            bKdtSV   : KHyd: Piecewise Linear Profile:  b * dt    [m/s] |
    ! |            dzsnSV(0): Soil first Layer Thickness                   [m] |
    ! |            dzmiSV   : Distance between two contiguous levels       [m] |
    ! |            dz78SV   : 7/8 (Layer Thickness)                        [m] |
    ! |            dz34SV   : 3/4 (Layer Thickness)                        [m] |
    ! |            dz_8SV   : 1/8 (Layer Thickness)                        [m] |
    ! |            dzAvSV   : 1/8  dz_(i-1) + 3/4 dz_(i) + 1/8 dz_(i+1)    [m] |
    ! |            dtz_SV   : dt/dz                                      [s/m] |
    ! |            OcndSV   : Swab Ocean / Soil Ratio                      [-] |
    ! |            Implic   : Implicit Parameter  (0.5:  Crank-Nicholson)      |
    ! |            Explic   : Explicit Parameter = 1.0 - Implic                |
    ! |                                                                        |
    ! | # OPTIONS: #ER: Richards Equation is not smoothed                      |
    ! | # ^^^^^^^  #kd: De Ridder   Discretization                             |
    ! | #          #SH: Hapex-Sahel Values                                     |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use mardim
    use marphy
    use mar_sv
    use mar_tv
    use mardsv
    use mar0sv
    use marxsv
    use margrd
    use marctr

    implicit none

    ! +--Internal Variables
    ! +  ==================

    integer i, j, k, ivt, ist, kk, ikl, isl, isn, ikh
    integer misl_2, nisl_2
    real zDepth
    real d__eta, eta__1, eta__2, Khyd_1, Khyd_2
    real RHsMin                          ! Min.Soil Relative Humidity
    real PsiMax                          ! Max.Soil Water    Potential
    real a_Khyd, b_Khyd                   ! Piecewis.Water Conductivity
#ifdef WR
    real Khyd_x, Khyd_y
#endif

    ! +--DATA
    ! +  ====

    data RHsMin/0.001/                   ! Min.Soil Relative Humidity

    ! +--Non Time Dependant SISVAT parameters
    ! +  ====================================

    ! +--Soil Discretization
    ! +  -------------------

    ! +--Numerical Scheme Parameters
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^
    Implic = 0.75                           ! 0.5 <==> Crank-Nicholson
    Explic = 1.00 - Implic                  !

    ! +--Soil/Snow Layers Indices
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^
    do isl = -nsol, 0
        islpSV(isl) = isl + 1
        islpSV(isl) = min(islpSV(isl), 0)
        islmSV(isl) = isl - 1
        islmSV(isl) = max(-nsol, islmSV(isl))
    enddo

    do isn = 1, nsno
        isnpSV(isn) = isn + 1
        isnpSV(isn) = min(isnpSV(isn), nsno)
    enddo
#ifdef kd
    ! +--Soil      Layers Thicknesses
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    if(nsol > 4) then
        do isl = -5, -nsol, -1
            dz_dSV(isl) = 1.
        enddo
    endif
#endif

    dz_dSV(-4:0) = (/0.72, 0.20, 0.060, 0.019, 0.001/)

    if(nsol /= 4) then
        do isl = 0, -nsol, -1
            misl_2 = -mod(isl, 2)
            nisl_2 = -isl / 2
            dz_dSV(isl) = (((1 - misl_2) * 0.001 &
                            + misl_2 * 0.003) * 10**(nisl_2)) * 3.
                          ! + misl_2 * 0.003) * 10**(nisl_2)) * 4. for version < 3.14.1
            ! +...    dz_dSV(0)  =         Hapex-Sahel Calibration:       4 mm

        enddo
        dz_dSV(0) = 0.001
        dz_dSV(-1) = dz_dSV(-1) - dz_dSV(0) + 0.004
    endif

    zz_dSV = 0.
    do isl = 0, -nsol, -1
     if (isl ==0 ) then
     write(6,'(a17,i2,f7.4)') "soil level:", isl,dz_dSV(isl)/2.
     else
     write(6,'(a17,i2,f7.4)') "soil level:", isl,dz_dSV(isl+1)+(dz_dSV(isl)+dz_dSV(min(0,isl+1)))/2.
     endif
     zz_dSV = zz_dSV+dz_dSV(isl)
    enddo
    write(6,'(a17,2x,f7.4,x,a1)') "total soil depth:", zz_dSV,"m"

    zz_dSV = 0.
    do isl = -nsol, 0
        dzmiSV(isl) = 0.500 * (dz_dSV(isl) + dz_dSV(islmSV(isl)))
        dziiSV(isl) = 0.500 * dz_dSV(isl) / dzmiSV(isl)
        dzi_SV(isl) = 0.500 * dz_dSV(islmSV(isl)) / dzmiSV(isl)
        dtz_SV(isl) = dt__SV / dz_dSV(isl)
        dtz_SV2(isl) = 1./dz_dSV(isl)
        dz78SV(isl) = 0.875 * dz_dSV(isl)
        dz34SV(isl) = 0.750 * dz_dSV(isl)
        dz_8SV(isl) = 0.125 * dz_dSV(isl)
        dzAvSV(isl) = 0.125 * dz_dSV(islmSV(isl)) &
                      + 0.750 * dz_dSV(isl) &
                      + 0.125 * dz_dSV(islpSV(isl))
#ifdef ER
        dz78SV(isl) = dz_dSV(isl)
        dz34SV(isl) = dz_dSV(isl)
        dz_8SV(isl) = 0.
        dzAvSV(isl) = dz_dSV(isl)
#endif
        zz_dSV = zz_dSV + dz_dSV(isl)
    enddo
    do ikl = 1, klonv
        dzsnSV(ikl, 0) = dz_dSV(0)
    enddo

    ! +--Conversion to a 50 m Swab Ocean Discretization
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    OcndSV = 0.
    do isl = -nsol, 0
        OcndSV = OcndSV + dz_dSV(isl)
    enddo
    OcndSV = 50./OcndSV

    ! +--Secondary Vegetation Parameters
    ! +  -------------------------------
#ifdef SH
    ! +--Minimum Stomatal Resistance (Hapex Sahel Data)
    ! +  (Taylor et al. 1997, J.Hydrol 188-189, p.1047)
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ivg = 1, 3
        StodSV(ivg) = 210.             ! Millet
    enddo
    StodSV(4) = 120.             ! Sparse Tiger Bush
    do ivg = 5, 6
        StodSV(ivg) = 80.             ! Dense  Tiger Bush
    enddo
    StodSV(7) = 80.             ! Low    Trees (Fallow)
    StodSV(10) = 80.
    ! +--Minimum Stomatal Resistance (Tropical Forest)
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    StodSV(8) = 60.             ! Medium Trees
    StodSV(11) = 60.
    StodSV(9) = 40.             ! High   Trees
    StodSV(12) = 40.
#endif
    ! +--Root Fraction
    ! +  ^^^^^^^^^^^^^
    ! +  * GENERAL REFERENCE
    ! +    Jackson et al., 1996: A global analysis of root distributions for
    ! +    terrestrial biomes. In Oecologia, 108, 389-411.

    ! +  * ROOT PROFILE
    ! +    The cumulative root fraction Y is given by
    ! +        Y = 1 - beta**d   with d    the depth (in cm),
    ! +                               beta a coefficient (vegetation dependent).

    ! +  * BETA VALUES (for 11 world biomes)
    ! +  1  boreal forest                0.943
    ! +  2  crops                        0.961
    ! +  3  desert                       0.975
    ! +  4  sclerophyllous shrubs        0.964
    ! +  5  temperate coniferous forest  0.976
    ! +  6  temperate deciduous forest   0.966
    ! +  7  temperate grassland          0.943
    ! +  8  tropical deciduous forest    0.961
    ! +  9  tropical evergreen forest    0.962
    ! +  10 tropical grassland savanna   0.972
    ! +  11 tundra                       0.914

    ! +  * ADVISED BETA VALUES FOR MAR
    ! +    (see 'block data SISVAT_dat', variable rbtdSV)
    ! +
    ! +    SVAT veg. type         default      West Africa
    ! +    0  barren soil         0.000        0.000
    ! +    1  crops low           0.961 (2)    0.961 (2)
    ! +    2  crops medium        0.961 (2)    0.961 (2)
    ! +    3  crops high          0.961 (2)    0.961 (2)
    ! +    4  grass low           0.943 (7)    0.943 (7)
    ! +    5  grass medium        0.943 (7)    0.964 (4)
    ! +    6  grass high          0.943 (7)    0.972 (10)
    ! +    7  broadleaf low       0.966 (6)    0.968 (4,10)
    ! +    8  broadleaf medium    0.966 (6)    0.962 (8,9)
    ! +    9  broadleaf high      0.966 (6)    0.962 (8,9)
    ! +    10 needleleaf low      0.976 (5)    0.971 (5,6)
    ! +    11 needleleaf medium   0.976 (5)    0.976 (5)
    ! +    12 needleleaf high     0.976 (5)    0.976 (5)

    ! +    Numbers between brackets refer to Jackson's biomes. For more details
    ! +    about some choices, see the correspondance between the IGBP and SVAT
    ! +    vegetation classes (i.e. in NESTOR).

    ! +  * WARNING
    ! +    Most of the roots are located in the first 2 m of soil. The root
    ! +    fraction per layer depends on the definition of the soil layer
    ! +    thickness. It will get wrong if a thick layer is defined around 2 m
    ! +    deep.

    !     write(*,'(/a)') 'ROOT PROFILES (Jackson, 1996) :'

    do ivt = 0, nvgt
        zDepth = 0.
        do isl = 0, -nsol, -1
            if(ivt /= 0) then
                RF__SV(ivt, isl) = rbtdSV(ivt)**zDepth * &
                                   (1.-rbtdSV(ivt)**(dz_dSV(isl) * 100))
                zDepth = zDepth + dz_dSV(isl) * 100  !in cm
            else
                RF__SV(ivt, isl) = 0.
            endif
        enddo
        !       write(*,'(a,i2,a,i3,a,99f10.5:)') &
        !            '  RF__SV(', ivt, ',', -nsol, ':0) =', RF__SV(ivt,:)
    enddo
    !     write(6,6600)
    ! 6600 format(&
    ! '  NOTE: If root fraction is not close to 0  around 2 m deep,', &
    ! /, '        Then you should redefine the soil layer thicknesses.', &
    ! /, '        See the code for more details.')

    ! +--Secondary Soil       Parameters
    ! +  -------------------------------

    do ist = 0, nsot
        rocsSV(ist) = (1.0 - etadSV(ist)) * 1.2E+6  ! Soil Contrib. to (ro c)_s
        ! Factor of (eta)**(b+2) in DR97, Eqn.(3.36)
        s1__SV(ist) = bCHdSV(ist) &
                      * psidSV(ist) * Ks_dSV(ist) &
                      / (etadSV(ist)**(bCHdSV(ist) + 3.))
        ! Factor of (eta)**(2b+3) in DR97, Eqn.(3.35)
        s2__SV(ist) = Ks_dSV(ist) &
                      / (etadSV(ist)**(2.*bCHdSV(ist) + 3.))

        ! +--Soil Minimum Humidity (from a prescribed minimum relative Humidity)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        Psimax = -(log(RHsMin)) / 7.2E-5        ! DR97, Eqn 3.15 Inversion
        etamSV(ist) = etadSV(ist) &
                      * (PsiMax / psidSV(ist))**(-min(10., 1./bCHdSV(ist)))
    enddo
    etamSV(12) = 0.

    ! +--Piecewise Hydraulic Conductivity Profiles
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ist = 0, nsot

#ifdef WR
        write(6, 6000)
6000    format(' Type |    etaSat | No |    eta__1 |    eta__2 |', &
               '    Khyd_1 |    Khyd_x |    Khyd_2 |    Khyd_y |' &
               /, ' -----+-----------+----+-----------+-----------+', &
               '-----------+-----------+-----------+-----------+')
#endif

        d__eta = etadSV(ist) / nkhy
        eta__1 = 0.
        eta__2 = d__eta
        do ikh = 0, nkhy
            ! DR97, Eqn.(3.35)
            Khyd_1 = s2__SV(ist) &
                     * (eta__1**(2.*bCHdSV(ist) + 3.))
            Khyd_2 = s2__SV(ist) &
                     * (eta__2**(2.*bCHdSV(ist) + 3.))

            a_Khyd = (Khyd_2 - Khyd_1) / d__eta
            b_Khyd = Khyd_1 - a_Khyd * eta__1
#ifdef WR
            Khyd_x = a_Khyd * eta__1 + b_Khyd
            Khyd_y = a_Khyd * eta__2 + b_Khyd
#endif
            aKdtSV(ist, ikh) = a_Khyd * dt__SV
            bKdtSV(ist, ikh) = b_Khyd * dt__SV
            aKdtSV2(ist, ikh) = a_Khyd * 1.
            bKdtSV2(ist, ikh) = b_Khyd * 1.

#ifdef WR
            write(6, 6001) ist, etadSV(ist), ikh, eta__1, &
                eta__2, Khyd_1, Khyd_x, Khyd_2, Khyd_y
#endif
6001        format(i5, ' |', e10.2, ' |', i3, ' |', &
                   6(e10.2, ' |'))

            eta__1 = eta__1 + d__eta
            eta__2 = eta__2 + d__eta
        enddo
    enddo

    if(itexpe <= 2) then
        do j = 1, my
            do i = 1, mx
                do k = 1, nvx
                    do kk = 1, llx
                        if(TsolTV(i, j, k, kk) <= 273.15) then
                            Eta_TV(i, j, k, kk) = min(Eta_TV(i, j, k, kk), 0.2 * etadSV(isolTV(i, j)))
                        else
                            Eta_TV(i, j, k, kk) = max(Eta_TV(i, j, k, kk), &
                                                      min(0.7, 0.4 + (TsolTV(i, j, k, kk) - 273.15) / 100.) * etadSV(isolTV(i, j)))
                        endif
                    enddo
                enddo
            enddo
        enddo
    endif

    return
end
