#include "MAR_pp.def"
subroutine SISVAT(itPhys)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT                                    02-04-2024  MAR |
    ! |   subroutine SISVAT contains the fortran 77 code of the                |
    ! |              Soil/Ice Snow Vegetation Atmosphere Transfer Scheme       |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |   PARAMETERS:  klonv: Total Number of columns =                        |
    ! |   ^^^^^^^^^^        = Total Number of continental     grid boxes       |
    ! |                     X       Number of Mosaic Cell per grid box         |
    ! |                                                                        |
    ! |   INPUT:   daHost   : Date Host Model                                  |
    ! |   ^^^^^                                                                |
    ! |                                                                        |
    ! |   INPUT:   LSmask   : 1:          Land       MASK                      |
    ! |   ^^^^^               0:          Sea        MASK                      |
    ! |            ivgtSV   = 0,...,12:   Vegetation Type                      |
    ! |            isotSV   = 0,...,12:   Soil       Type                      |
    ! |                       0:          Water,          Liquid (Sea, Lake)   |
    ! |                      12:          Water, Solid           (Ice)         |
    ! |                                                                        |
    ! |   INPUT:   coszSV   : Cosine of the Sun Zenithal Distance          [-] |
    ! |   ^^^^^    sol_SV   : Surface Downward  Solar      Radiation    [W/m2] |
    ! |            IRd_SV   : Surface Downward  Longwave   Radiation    [W/m2] |
    ! |            drr_SV   : Rain  Intensity                        [kg/m2/s] |
    ! |            dsn_SV   : Snow  Intensity                      [mm w.e./s] |
    ! |            dsnbSV   : Snow  Intensity,  Drift Fraction             [-] |
    ! |            dbs_SV   : Drift Amount                           [mm w.e.] |
    ! |            za__SV   : Surface Boundary Layer (SBL) Height          [m] |
    ! |            VV__SV   :(SBL Top)   Wind Velocity                   [m/s] |
    ! |            VV10SV   : 10-m       Wind Velocity                   [m/s] |
    ! |            TaT_SV   : SBL Top    Temperature                       [K] |
    ! |            rhT_SV   : SBL Top    Air  Density                  [kg/m3] |
    ! |            QaT_SV   : SBL Top    Specific  Humidity            [kg/kg] |
    ! |            qsnoSV   : SBL Mean   Snow      Content             [kg/kg] |
    ! |            LAI0SV   : Leaf Area  Index                             [-] |
    ! |            glf0SV   : Green Leaf Fraction                          [-] |
    ! |            alb0SV   : Soil Basic Albedo               [-] |
    ! |            slopSV   : Surface    Slope                             [-] |
    ! |            dt__SV   : Time  Step                                   [s] |
    ! |                                                                        |
    ! |   INPUT /  isnoSV   = total Nb of Ice/Snow Layers                      |
    ! |   OUTPUT:  ispiSV   = 0,...,nsno: Uppermost Superimposed Ice Layer     |
    ! |   ^^^^^^   iiceSV   = total Nb of Ice      Layers                      |
    ! |            istoSV   = 0,...,5 :   Snow     History (see istdSV data)   |
    ! |                                                                        |
    ! |   INPUT /  alb_SV   : Surface-Canopy Albedo           [-] |
    ! |   OUTPUT:  emi_SV   : Surface-Canopy Emissivity                    [-] |
    ! |   ^^^^^^   IRs_SV   : Soil           IR Flux  (negative)        [W/m2] |
    ! |            LMO_SV   : Monin-Obukhov               Scale            [m] |
    ! |            us__SV   : Friction          Velocity                 [m/s] |
    ! |            uts_SV   : Temperature       Turbulent Scale          [m/s] |
    ! |            uqs_SV   : Specific Humidity Velocity                 [m/s] |
    ! |            uss_SV   : Blowing Snow      Turbulent Scale          [m/s] |
    ! |            usthSV   : Blowing Snow      Erosion   Threshold      [m/s] |
    ! |            Z0m_SV   : Momentum     Roughness Length                [m] |
    ! |            Z0mmSV   : Momentum     Roughness Length (time mean)    [m] |
    ! |            Z0mnSV   : Momentum     Roughness Length (instantaneous)[m] |
    ! |            Z0SaSV   : Sastrugi     Roughness Length                [m] |
    ! |            Z0e_SV   : Erosion Snow Roughness Length                [m] |
    ! |            Z0emSV   : Erosion Snow Roughness Length (time mean)    [m] |
    ! |            Z0enSV   : Erosion Snow Roughness Length (instantaneous)[m] |
    ! |            Z0roSV   : Subgrid Topo Roughness Length                [m] |
    ! |            Z0h_SV   : Heat         Roughness Length                [m] |
    ! |            snCaSV   : Canopy   Snow     Thickness            [mm w.e.] |
    ! |            rrCaSV   : Canopy   Water    Content                [kg/m2] |
    ! |            psivSV   : Leaf     Water    Potential                  [m] |
    ! |            TvegSV   : Canopy   Temperature                         [K] |
    ! |            TsisSV   : Soil/Ice Temperatures (layers -nsol,-nsol+1,..,0)|
    ! |                     & Snow     Temperatures (layers  1,2,...,nsno) [K] |
    ! |            ro__SV   : Soil/Snow Volumic Mass                   [kg/m3] |
    ! |            eta_SV   : Soil/Snow Water   Content                [m3/m3] |
    ! |            G1snSV   : snow dendricity/sphericity                       |
    ! |            G2snSV   : snow sphericity/grain size                       |
    ! |            dzsnSV   : Snow Layer        Thickness                  [m] |
    ! |            agsnSV   : Snow       Age                             [day] |
    ! |            BufsSV   : Snow Buffer Layer              [kg/m2] .OR. [mm] |
    ! |            BrosSV   : Snow Buffer Layer Density      [kg/m3]           |
    ! |            BG1sSV   : Snow Buffer Layer Dendricity / Sphericity    [-] |
    ! |            BG2sSV   : Snow Buffer Layer Sphericity / Size [-] [0.1 mm] |
    ! |            rusnSV   : Surficial   Water              [kg/m2] .OR. [mm] |
    ! |                                                                        |
    ! |   OUTPUT:  no__SV   : OUTPUT file Unit Number                      [-] |
    ! |   ^^^^^^   i___SV   : OUTPUT point   i Coordinate                  [-] |
    ! |            j___SV   : OUTPUT point   j Coordinate                  [-] |
    ! |            n___SV   : OUTPUT point   n Coordinate                  [-] |
    ! |            lwriSV   : OUTPUT point vec Index                       [-] |
    ! |                                                                        |
    ! |   OUTPUT:  IRu_SV   : Upward     IR Flux (+, upw., effective)      [K] |
    ! |   ^^^^^^   hSalSV   : Saltating Layer Height                       [m] |
    ! |            qSalSV   : Saltating Snow  Concentration            [kg/kg] |
    ! |            RnofSV   : RunOFF Intensity                       [kg/m2/s] |
    ! |                                                                        |
    ! |   Internal Variables:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^                                                   |
    ! |            NLaysv   = New            Snow Layer Switch             [-] |
    ! |            albisv   : Snow/Ice/Water/Soil Integrated Albedo [-] |
    ! |            SoCasv   : Absorbed Solar Radiation by Canopy (Normaliz)[-] |
    ! |            SoSosv   : Absorbed Solar Radiation by Surfac.(Normaliz)[-] |
    ! |            tau_sv   : Fraction of Radiation transmitted by Canopy  [-] |
    ! |            TBr_sv   : Brightness Temperature                       [K] |
    ! |            IRupsv   : Upward     IR Flux (-, upw.)              [W/m2] |
    ! |            IRv_sv   : Vegetation IR Flux                        [W/m2] |
    ! |            rrMxsv   : Canopy Maximum Intercepted Rain          [kg/m2] |
    ! |            Sigmsv   : Canopy Ventilation Factor                    [-] |
    ! |            ram_sv   : Aerodynamic Resistance for Momentum        [s/m] |
    ! |            rah_sv   : Aerodynamic Resistance for Heat            [s/m] |
    ! |            HSv_sv   : Vegetation Sensible Heat Flux             [W/m2] |
    ! |            HLv_sv   : Vegetation Latent   Heat Flux             [W/m2] |
    ! |            Rootsv   : Root Water Pump                        [kg/m2/s] |
    ! |            Evp_sv   : Evaporation                              [kg/m2] |
    ! |            EvT_sv   : Evapotranspiration                       [kg/m2] |
    ! |            HSs_sv   : Surface    Sensible Heat Flux + => absorb.[W/m2] |
    ! |            HLs_sv   : Surface    Latent   Heat Flux + => absorb.[W/m2] |
    ! |            Lx_H2O   : Latent Heat of Vaporization/Sublimation   [J/kg] |
    ! |            Tsrfsv   : Surface    Temperature                       [K] |
    ! |            LAI_sv   : Leaf Area  Index (snow included)             [-] |
    ! |            LAIesv   : Leaf Area  Index (effective / transpiration) [-] |
    ! |            glf_sv   : Green Leaf Fraction of NOT fallen Leaves     [-] |
    ! |            sEX_sv   : Verticaly Integrated Extinction Coefficient  [-] |
    ! |            LSdzsv   : Vertical   Discretization Factor             [-] |
    ! |                     =    1. Soil                                       |
    ! |                     = 1000. Ocean                                      |
    ! |            z_snsv   : Snow Pack  Thickness                         [m] |
    ! |            zzsnsv   : Snow Pack  Thickness                         [m] |
    ! |            albssv   : Soil       Albedo               [-] |
    ! |            Evg_sv   : Soil+Vegetation Emissivity                   [-] |
    ! |            Eso_sv   : Soil+Snow       Emissivity                   [-] |
    ! |            psi_sv   : Soil       Water    Potential                [m] |
    ! |            Khydsv   : Soil   Hydraulic    Conductivity           [m/s] |
    ! |                                                                        |
    ! |            ETVg_d   : VegetationEnergy Power         Forcing    [W/m2] |
    ! |            ETSo_0   : Snow/Soil Energy Power, before Forcing    [W/m2] |
    ! |            ETSo_1   : Snow/Soil Energy Power, after  Forcing    [W/m2] |
    ! |            ETSo_d   : Snow/Soil Energy Power         Forcing    [W/m2] |
    ! |            EqSn_0   : Snow      Energy, before Phase Change     [J/m2] |
    ! |            EqSn_1   : Snow      Energy, after  Phase Change     [J/m2] |
    ! |            EqSn_d   : Snow      Energy,       net    Forcing    [J/m2] |
    ! |            Enrsvd   : SVAT      Energy Power         Forcing    [W/m2] |
    ! |            Enrbal   : SVAT      Energy Balance                  [W/m2] |
    ! |            Wats_0   : Soil Water,  before Forcing                 [mm] |
    ! |            Wats_1   : Soil Water,  after  Forcing                 [mm] |
    ! |            Wats_d   : Soil Water          Forcing                 [mm] |
    ! |            SIWm_0   : Snow        initial Mass               [mm w.e.] |
    ! |            SIWm_1   : Snow        final   Mass               [mm w.e.] |
    ! |            SIWa_i   : Snow Atmos. initial Forcing            [mm w.e.] |
    ! |            SIWa_f   : Snow Atmos. final   Forcing(noConsumed)[mm w.e.] |
    ! |            SIWe_i   : SnowErosion initial Forcing            [mm w.e.] |
    ! |            SIWe_f   : SnowErosion final   Forcing(noConsumed)[mm w.e.] |
    ! |            SIsubl   : Snow sublimed/deposed  Mass            [mm w.e.] |
    ! |            SImelt   : Snow Melted            Mass            [mm w.e.] |
    ! |            SIrnof   : Surficial Water + Run OFF Change       [mm w.e.] |
    ! |            SIvAcr   : Sea-Ice    vertical Acretion           [mm w.e.] |
    ! |            Watsvd   : SVAT Water          Forcing                 [mm] |
    ! |            Watbal   : SVAT Water  Balance                       [W/m2] |
    ! |                                                                        |
    ! |            dsn_Ca,snCa_n :     Snow Contribution to the Canopy[m w.e.] |
    ! |            drr_Ca,rrCa_n,drip: Rain Contribution to the Canopy [kg/m2] |
    ! |            vk2      : Square of Von Karman Constant                [-] |
    ! |            sqrCm0   : Factor of   Neutral Drag Coeffic.Momentum  [s/m] |
    ! |            sqrCh0   : Factor of   Neutral Drag Coeffic.Heat      [s/m] |
    ! |            EmiVeg   : Vegetation    Emissivity                     [-] |
    ! |            EmiSol   : Soil          Emissivity                     [-] |
    ! |            EmiSno   : Snow          Emissivity                     [-] |
    ! |            EmiWat   : Water         Emissivity                     [-] |
    ! |            Z0mSea   :          Sea  Roughness Length               [m] |
    ! |            Z0mLnd   :          Land Roughness Length               [m] |
    ! |            sqrrZ0   : u*t/u*                                           |
    ! |            f_eff    : Marticorena & B. 1995 JGR (20)                   |
    ! |            A_Fact   : Fundamental * Roughness                          |
    ! |            Z0mBSn   :         BSnow Roughness Length               [m] |
    ! |            Z0mBS0   : Mimimum BSnow Roughness Length (blown* )     [m] |
    ! |            Z0m_Sn   :          Snow Roughness Length (surface)     [m] |
    ! |            Z0m_S0   : Mimimum  Snow Roughness Length               [m] |
    ! |            Z0m_S1   : Maximum  Snow Roughness Length               [m] |
    ! |            Z0_GIM   : Minimum GIMEX Roughness Length               [m] |
    ! |            Z0_ICE   : Sea Ice ISW   Roughness Length               [m] |
    ! |                                                                        |
    ! | # TUNING PARAMETERS :                                                  |
    ! | # OPTIONS: #BS: Wind Dependant Roughness Length of Snow                |
    ! | # ^^^^^^^  #ZS: Wind Dependant Roughness Length of Sea                 |
    ! | #          #FL: Dead Leaves are assumed to been fallen                 |
    ! | #          #RS: Z0h = Z0m / 100 over the ocean                         |
    ! | #          #US: u*   computed from aerodynamic resistance              |
    ! | #          #WV: OUTPUT                                                 |
    ! | #          #WR: OUTPUT    for Verification                             |
    ! | #          #SR: Variable      Tracing                                  |
    ! | #          #CP: Col de Porte  Turbulence        Parameterization       |
    ! | #          #GL: Greenland                       Parameterization       |
    ! |                                                                        |
    ! |                                                                        |
    ! |   TUNING PARAMETER:                                                    |
    ! |   ^^^^^^^^^^^^^^^^                                                     |
    ! |            z0soil   : Soil Surface averaged Bumps Height (see _qSo)[m] |
    ! |                                                                        |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! |   FILE                 |      CONTENT                                  |
    ! |   ~~~~~~~~~~~~~~~~~~~~~+~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ |
    ! | # SISVAT_iii_jjj_n     | #E0: OUTPUT on ASCII  File (SISVAT Variables) |
    ! | #                      |      Energy Budg. Verif.: Soil+(Sea-Ice)+Snow |
    ! | #                      |(#E0  MUST BE PREPROCESSED BEFORE #e1 & #e2 !) |
    ! | # SISVAT_iii_jjj_n     | #m0: OUTPUT/Verification: H2O    Conservation |
    ! |                        |                                               |
    ! | # stdout               | #s0: OUTPUT of Snow Buffer Layer              |
    ! |                        |      unit  6, subroutine  SISVAT     **ONLY** |
    ! | # stdout               | #sb: OUTPUT of Snow Erosion                   |
    ! |                        |      unit  6, subroutine  SISVAT_BSn **ONLY** |
    ! | # stdout               | #sz: OUTPUT of Roughness Length & Drag Coeff. |
    ! |                        |      unit  6, subroutine  SISVAT     **ONLY** |
    ! | # stdout               | #wz: OUTPUT of Roughness Length (Blown Snow)  |
    ! |                        |      unit  6, subroutines SISVAT, PHY_SISVAT  |
    ! |                                                                        |
    ! |   SUGGESTIONS of MODIFICATIONS: see lines beginning with "C +!!!"      |
    ! |   ^^^^^^^^^^^^^^^^^^^^^^^^^^^^                                         |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mar_sv
    use marlsv
    use mardsv
    use mar0sv
    use marxsv
    use mardim
    use mar_ge
    use marysv
    use margrd
#ifdef CP
    use marcdp
#endif
    
    implicit none
    
    ! +--Internal Variables
    ! +  ==================
    
    ! TBr_sv : Brightness Temperature
    real TBr_sv(klonv)
    ! IRdwsv : DOWNward   IR Flux
    real IRdwsv(klonv)
    ! IRupsv : UPward     IR Flux
    real IRupsv(klonv)
    ! d_Bufs : Buffer Snow Layer Increment
    real d_Bufs, Bufs_N
    ! Buf_ro : Buffer Snow Layer Density
    real Buf_ro, Bros_N
    ! BufPro : Buffer Snow Layer Density
    real BufPro
    ! dt__SV2 : saved dt
    real dt__SV2
    ! Buf_G1 : Buffer Snow Layer Dendr/Sphe[-]
    real Buf_G1, BG1__N
    ! Buf_G2 : Buffer Snow Layer Spher/Size[-]
    real Buf_G2, BG2__N
    ! Bdzssv : Buffer Snow Layer Thickness
    real Bdzssv(klonv)
    ! z_snsv : Snow-Ice, current Thickness
    real z_snsv(klonv)
    
    ! +--Energy         Budget
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    ! ETVg_d : VegetationPower, Forcing
    real ETVg_d(klonv)
    ! ETSo_0 : Soil/Snow Power, before Forcing
    real ETSo_0(klonv)
    ! ETSo_1 : Soil/Snow Power, after  Forcing
    real ETSo_1(klonv)
    ! ETSo_d : Soil/Snow Power, Forcing
    real ETSo_d(klonv)
#ifdef e1
    ! EqSn_0 : Snow Energy, befor Phase Change
    real EqSn_0(klonv)
    ! EqSn_1 : Snow Energy, after Phase Change
    real EqSn_1(klonv)
    ! EqSn_d : Energy in Excess
    real EqSn_d(klonv)
#endif
    
    ! +--Water   (Mass) Budget
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    ! Wats_0 : Soil Water,  before Forcing
    real Wats_0(klonv)
    ! Wats_1 : Soil Water,  after  Forcing
    real Wats_1(klonv)
    ! Wats_d : Soil Water,         Forcing
    real Wats_d(klonv)

#ifdef m1
    ! +--Snow/Ice(Mass) Budget
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    ! SIsubl : Snow Sublimed/Deposed Mass
    real SIsubl(klonv)
    ! SImelt : Snow Melted           Mass
    real SImelt(klonv)
    ! SIrnof : Local Surficial Water + Run OFF
    real SIrnof(klonv)
#endif
    
    ! +--Sea-Ice (Mass) Budget
    ! +  ~~~~~~~~~~~~~~~~~~~~~
    ! SIvAcr : Sea-Ice Vertical Acretion
    real SIvAcr(klonv)
    
    ! +--Local
    ! +  -----

#ifdef MT
    ! Garrat : SBL     Scheme    Switch
    logical Garrat
#endif
    ! SepLab : OUTPUT ASCII File Labels
    character * 1 SepLab
    character * 6 FilLab
    character * 16 FilNam
    common / SISVAT_loc_abc / SepLab, FilLab
    ! FilLab : OUTPUT File  Unit Number
    integer noUNIT
    ! FilNam : OUTPUT File  Unit Number (New)
    integer nwUNIT
    common / SISVAT_loc_num / nwUNIT
    integer iwr, itPhys
    integer ikl, isn, isl, ist, it
    ! Soil/Water Body Identifier
    integer ist__s, ist__w
    ! Seasonal               Mask
    integer growth
    ! Land+Ice / Open    Sea Mask
    integer LISmsk
    ! Snow-Ice / No Snow-Ice Mask
    integer LSnMsk
    !      Ice / No      Ice Mask
    integer IceMsk, IcIndx(klonv)
    ! Snow     / No Snow     Mask
    integer SnoMsk
    ! Rain Contribution to the Canopy
    real drr_Ca, rrCa_n, drip
    ! Snow Contribution to the Canopy
    real dsn_Ca, snCa_n, FallOK(klonv)
    ! Fallen Snow Density (PAHAUT)
    real roSMin, roSn_1, roSn_2, roSn_3
    ! Max. Fallen Snow Density
    real roSMax
    ! Fallen Snow Dendric.(GIRAUD)
    real Dendr1, Dendr2, Dendr3
    ! Fallen Snow Spheric.(GIRAUD)
    real Spher1, Spher2, Spher3, Spher4
    ! Polar  Snow Switch
    real Polair, factor
    real PorSno, Por_BS, Salt_f, PorRef
#ifdef sw
    real PorVol, rWater
    real rusNEW, rdzNEW, etaNEW
#endif
    real ro_new
    ! Maximum     Polar Temperature
    real TaPole
    ! Minimum realistic Temperature
    real T__Min
    ! Emissivity of Vegetation
    real EmiVeg
    ! Emissivity of       Soil
    real EmiSol
    ! Emissivity of            Snow
    real EmiSno
    ! Emissivity of a Water Area
    real EmiWat
    ! Square of Von Karman Constant
    real vk2
    ! (u*)**2
    real u2star
#ifdef WR
    ! Friction Velocity (aer.resist.)
    real u_star
#endif
    ! Fallen   Leaves         Switch
    real fallen
    ! Sea  Roughness Length
    real Z0mSea, Z0hSea
    ! Land Roughness Length
    real Z0mLnd
#ifdef ZN
    ! u*t/u*
    real sqrrZ0
#endif
    ! Marticorena & B. 1995 JGR (20)
    real f_eff
    ! Fundamental * Roughness
    real A_Fact
    ! Smooth R Snow Roughness Length
    real Z0m_nu
    !         BSnow Roughness Length
    real Z0mBSn
    ! Mimimum BSnow Roughness Length
    real Z0mBS0
    ! Mimimum  Snow Roughness Length
    real Z0m_S0
    ! Maximum  Snow Roughness Length
    real Z0m_S1
#ifdef SZ
    ! Regime   Snow Roughness Length
    real Z0Sa_N
    ! 1.if Rgm Snow Roughness Length
    real Z0SaSi
#endif
#ifdef GL
    ! Mimimum GIMEX Roughness Length
    real Z0_GIM
#endif
    ! Sea-Ice ISW   Roughness Length
    real Z0_ICE
    ! Snow  Surface Roughness Length
    real Z0m_Sn, Z0m_90
    ! Snow Layer    Switch
    real SnoWat
#ifdef RN
    real rstar, alors
    real rstar0, rstar1, rstar2
#endif
    ! 1. => Same Type of Grains
    real SameOK
    ! Averaged G1,  same Grains
    real G1same
    ! Averaged G2,  same Grains
    real G2same
    ! 1. => Lay1 Type: Dendritic
    real typ__1
    ! dz X ro, if fresh Snow
    real zroNEW
    ! G1,      if fresh Snow
    real G1_NEW
    ! G2,      if fresh Snow
    real G2_NEW
    ! dz X ro, if old   Snow
    real zroOLD
    ! G1,      if old   Snow
    real G1_OLD
    ! G2,      if old   Snow
    real G2_OLD
    ! Size,    if fresh Snow
    real SizNEW
    ! Spheric.,if fresh Snow
    real SphNEW
    ! Size,    if old   Snow
    real SizOLD
    ! Spheric.,if old   Snow
    real SphOLD
    ! Averaged    Grain Size
    real Siz_av
    ! Averaged    Grain Spher.
    real Sph_av
    ! Averaged    Grain Dendr.
    real Den_av
    ! 1. => Average is  Dendr.
    real DendOK
    ! Averaged G1, diff. Grains
    real G1diff
    ! Averaged G2, diff. Grains
    real G2diff
    ! Averaged G1
    real G1
    ! Averaged G2
    real G2
    ! Polynomial   fit z0=f(T)
    real param
    ! Fit Z0_obs=f(T) (m)
    real Z0_obs
    ! min T of linear fit (K)
    real tamin
    ! max T of linear fit (K)
    real tamax
    ! Coefs for z0=f(T)
    real coefa, coefb, coefc, coefd
    ! Air temperature thresholds
    real ta1, ta2, ta3
    ! z0 thresholds
    real z01, z02, z03
    ! Critical param.
    real tt_c, vv_c
    ! Temporary variables
    real tt_tmp, vv_tmp, vv_virt
    ! .true. if Kotlyakov 1961 else density from obs.
    logical density_kotlyakov
#ifdef BS
    ! Correc. factor for drift ratio
    real Fac
#endif
    
    ! +--Energy       Budget
    ! +  ~~~~~~~~~~~~~~~~~~~
#ifdef e1
    ! Energy Imbalances Counter
    integer noEBal
    common / SISVAT__EBal / noEBal
    ! Soil+Vegetat  Power  Forcing
    real Enrsvd(klonv)
    ! Soil+Snow   , Power  Balance
    real EnsBal
    !      Vegetat, Power  Balance
    real EnvBal
#endif
    
    ! +--Water (Mass) Budget
    ! +  ~~~~~~~~~~~~~~~~~~~
#ifdef m0
    ! Water  Imbalances Counter
    integer noWBal
    common / SISVAT__WBal / noWBal
    ! Soil+Vegetat, before Forcing
    real Watsv0(klonv)
    ! Soil+Vegetat  Water  Forcing
    real Watsvd(klonv)
    ! Soil+Vegetat, Water  Balance
    real Watbal
#endif
    
    ! +--Snow  (Mass) Budget
    ! +  ~~~~~~~~~~~~~~~~~~~
#ifdef m1
    ! Water  Imbalances Counter
    integer noSBal
    common / SISVAT__SBal / noSBal
    ! Snow Initial/Final        Mass
    real SIWm_0(klonv), SIWm_1(klonv)
    ! Snow Initial/Final ATM Forcing
    real SIWa_i(klonv), SIWa_f(klonv)
    ! Snow Initial/Final BLS Forcing
    real SIWe_i(klonv), SIWe_f(klonv)
    ! Snow Pack     Mass   Balance
    real SnoBal
#endif
    
    ! +--Internal DATA
    ! +  =============

#ifdef MT
    ! SBL Scheme    Switch
    data Garrat/.true./
#endif
    ! Minimum realistic Temperature
    data T__Min/200.00/
#ifdef AC
    ! Maximum Polar Temperature
    data TaPole/283.15/
#endif
    ! Maximum Polar Temperature
    data TaPole/268.15/
#ifdef EU
    ! Minimum Snow Density
    data roSMin/25./
#endif
#ifdef BS
    ! Minimum Snow Density
    data roSMin/300./
#endif
#ifdef AC
    ! Minimum Snow Density
    data roSMin/300./
#endif
    ! Minimum Snow Density
    data roSMin/300./
    !mode Greenland over temperate climate: roSMin=100
#ifdef EU
    ! Max Fresh Snow Density
    data roSMax/150./
#endif
    ! Max Fresh Snow Density
    data roSMax/400./
    ! Critical Temp. (degC)
    ! tt_c=-2.  => rho->quickly to rho(vv_c) when T->-inf
    data tt_c/-2.0/
    ! Critical Wind speed (m/s)
    ! vv_c=14.3 => rho->300 when T->-inf
    data vv_c/14.3/
    ! Fall.Sno.Density, Indep. Param.
    data roSn_1/109./
    ! Fall.Sno.Density, Temper.Param.
    data roSn_2/6./
    ! Fall.Sno.Density, Wind   Param.
    data roSn_3/26./
    ! Fall.Sno.Dendric.,Wind 1/Param.
    data Dendr1/17.12/
    ! Fall.Sno.Dendric.,Wind 2/Param.
    data Dendr2/128./
    ! Fall.Sno.Dendric.,Indep. Param.
    data Dendr3/-20./
    ! Fall.Sno.Spheric.,Wind 1/Param.
    data Spher1/7.87/
    ! Fall.Sno.Spheric.,Wind 2/Param.
    data Spher2/38./
    ! Fall.Sno.Spheric.,Wind 3/Param.
    data Spher3/50./
    ! Fall.Sno.Spheric.,Indep. Param.
    data Spher4/90./
    ! Emissivities : Pielke, 1984, pp. 383,409
    ! EmisnoAO and EmiWatAO are defined and fixed in MAR_phy.inc !AO_CK 20/02/2020
    ! Emissivity of Soil
    data EmiSol/0.94/
    ! Emissivity of Vegetation
    data EmiVeg/0.98/
    ! Emissivity of a Water Area
    data EmiWat/0.99/
    ! Emissivity of Snow
    data EmiSno/0.99/
    
    ! Fallen  Leaves         Switch
    data fallen/0./
    ! MINimum Snow Roughness Length
    ! for Momentum if Blowing Snow
    ! Gallee et al. 2001 BLM 99 (19)
    data Z0mBS0/0.5e-6/
    ! MINimum Snow Roughness Length
    data Z0m_S0/0.00005/
    ! MINimum Snow Roughness Length
#ifdef MG
    ! MegaDunes    included
    data Z0m_S0/0.00200/
#endif
    ! MAXimum Snow Roughness Length
    !        (Sastrugis)
    data Z0m_S1/0.030/
#ifdef GL
    ! Ice Min Z0 = 0.0013 m (Broeke)
    ! Old Ice Z0 = 0.0500 m (Bruce)
    !              0.0500 m (Smeets)
    !              0.1200 m (Broeke)
    data Z0_GIM/0.0017/
#endif
    ! Sea-Ice Z0 = 0.0010 m (Andreas)
    !    (Ice Station Weddel -- ISW)
    data Z0_ICE/0.0010/
    ! Square of Von Karman Constant
    vk2 = vonkar * vonkar
#ifdef FL
    ! Fallen  Leaves Switch
    fallen = 1.
#endif
    
    ! +..BEGIN.main.
    ! +--SISVAT Forcing VERIFICATION
    ! +  ===========================
    
    if(.not. iniOUT) then
        iniOUT = .true.
        if(IRs_SV(1) > -epsi) &
                write(6, 600)
        600     format(/, '### SISVAT ERROR, Soil IR Upward  not defined ###', &
                /, '###               Initialize and Store IRs_SV ###')
        
        ! OUTPUT
        ! ======
        
        FilLab = 'SISVAT'
        SepLab = '_'
        nwUNIT = 51
    endif
#ifdef E0
    do ikl = 1, klonv
        if(lwriSV(ikl) /= 0 .and. no__SV(lwriSV(ikl)) == 0) then
            nwUNIT = nwUNIT + 1
            no__SV(lwriSV(ikl)) = nwUNIT
            write(FilNam, '(a6,a1,2(i3.3,a1),i1)') &
                    FilLab, SepLab, i___SV(lwriSV(ikl)), &
                    SepLab, j___SV(lwriSV(ikl)), &
                    SepLab, n___SV(lwriSV(ikl))
            open(unit = nwUNIT, status = 'unknown', file = FilNam)
            rewind nwUNIT
        endif
    enddo
    do ikl = 1, klonv
        if(lwriSV(ikl) /= 0) then
            noUNIT = no__SV(lwriSV(ikl))
            write(noUNIT, 5000) daHost, i___SV(lwriSV(ikl)), &
                    j___SV(lwriSV(ikl)), &
                    n___SV(lwriSV(ikl)), &
                    Z0m_SV(ikl), &
                    albisv(ikl)
            5000        format(&
                    /, a18, '|           Grid Point ', 2i4, &
                    ' (', i2, ')', &
                    '    | Z0m =', f12.6, ' | Albedo = ', f6.3, ' |', &
                    /, ' -------+', 7('---------+'), 2('--------+'))
        endif
    enddo
#endif
    
    ! +--"Soil" Humidity of Water Bodies
    ! +  ===============================
    
    do ikl = 1, klonv
        ! Soil Type
        ist = isotSV(ikl)
        ! 1 => Soil
        ist__s = min(ist, 1)
        ! 1 => Water Body
        ist__w = 1 - ist__s
        do isl = -nsol, 0
            ! Soil + Water Body
            eta_SV(ikl, isl) = eta_SV(ikl, isl) * ist__s &
                    + etadSV(ist) * ist__w
        enddo
        
        ! +--Vertical Discretization Factor
        ! +  ==============================
        ! Soil + Water Body
        LSdzsv(ikl) = ist__s &
                + OcndSV * ist__w
    enddo
    
    ! +--Vegetation Temperature Limits
    ! +  =============================
    
    do ikl = 1, klonv
        TvegSV(ikl) = max(TvegSV(ikl), T__Min)         ! T__Min = 200.K
        
        ! +--LAI Assignation and Fallen Leaves Correction (#FL)
        ! +  ==================================================
        
        LAI0SV(ikl) = LAI0SV(ikl) * min(1, ivgtSV(ikl)) ! NO LAI if
        ! +                                                          ! no vegetation
        
        if(ivgtSV(ikl) == 13) then ! city
            LAI0SV(ikl) = 0.
            glf0SV(ikl) = 0.
        endif
        
        glf_sv(ikl) = glf0SV(ikl)
#ifdef FL
        glf_sv(ikl) = 1.
#endif
        LAI_sv(ikl) = LAI0SV(ikl)
#ifdef FL
        LAI_sv(ikl) = LAI_sv(ikl) * glf0SV(ikl)
#endif
    enddo
    
    ! +--LAI in Presence of Snow
    ! +  =======================
    do ikl = 1, klonv
        z_snsv(ikl) = 0.0
    enddo
    do isn = 1, nsno
        do ikl = 1, klonv
            z_snsv(ikl) = z_snsv(ikl) + dzsnSV(ikl, isn)
            zzsnsv(ikl, isn) = z_snsv(ikl)
        enddo
    enddo
    ! + ASSUMPTION: LAI decreases   when Snow Thickness increases,
    ! + ^^^^^^^^^^      becoming  0 when Snow Thickn. = Displac.Height
    do ikl = 1, klonv
        LAI_sv(ikl) = LAI_sv(ikl) &
                * (1.0 - zzsnsv(ikl, isnoSV(ikl)) &
                        / (DH_dSV(ivgtSV(ikl)) + epsi))
        LAI_sv(ikl) = max(LAI_sv(ikl), zero)
        LAI_sv(ikl) = min(LAI_sv(ikl), argmax)
    enddo
    
    ! +--Interception of Rain by the Canopy
    ! +  ==================================
    
    ! Vegetation Forcing
    ! ------------------
#ifdef m0
    do ikl = 1, klonv
        Watsv0(ikl) = rrCaSV(ikl)           ! Canopy Water Cont.
        Watsvd(ikl) = drr_SV(ikl)           ! Precipitation
    enddo
#endif
    
    ! +--New Canopy Water Content
    ! +  ------------------------
    
    do ikl = 1, klonv
        ! Precip. Max. Intercept.
        rrMxsv(ikl) = 0.2 * max(epsi, LAI_sv(ikl))
        ! Canopy Ventilation Coe. (DR97, eqn 3.6)
        Sigmsv(ikl) = 1.0 - exp(-demi * LAI_sv(ikl))
        ! Intercepted Rain
        drr_Ca = drr_SV(ikl) * Sigmsv(ikl) &
                * dt__SV
        ! New Canopy Water Contnt (DR97, eqn 3.28)
        rrCa_n = rrCaSV(ikl) + drr_Ca
        ! Water  Drip
        drip = rrCa_n - rrMxsv(ikl)
        drip = max(zero, drip)
        rrCa_n = rrCa_n - drip
        ! Update Rain  Contribut.
        drr_SV(ikl) = drr_SV(ikl) + (rrCaSV(ikl) &
                - rrCa_n) &
                / dt__SV
        RuofSV(ikl, 1) = RuofSV(ikl, 1) + max(0., drr_SV(ikl))
        ! Upd. Canopy Water Content
        rrCaSV(ikl) = rrCa_n
        
        ! +--Interception of Snow by the Canopy
        ! +  ==================================
        
        ! cXF 03/03/2021 not relevant
        ! ! Intercepted Snow
        ! dsn_Ca = dsn_SV(ikl) * Sigmsv(ikl) * dt__SV
        ! ! New Canopy Snow Thickn.
        ! snCa_n = snCaSV(ikl) + dsn_Ca
        ! drip = snCa_n - rrMxsv(ikl)
        ! drip = max(zero, drip)
        ! snCa_n = snCa_n - drip
        ! ! Update Snow  Contribut.
        ! dsn_SV(ikl) = dsn_SV(ikl) + (snCaSV(ikl) - snCa_n) / dt__SV
        ! ! Upd.Canopy Snow Thickn.
        ! snCaSV(ikl) = snCa_n
        snCaSV(ikl) = 0.
    enddo
    
    ! +--Snow Fall from the Canopy
    ! +  =========================
    
    ! +       ASSUMPTION: snow fall from the canopy,
    ! +       ^^^^^^^^^^  when the temperature of the vegetation is positive
    ! +             (.OR. when snow over the canopy is saturated  with water)
    
    ! do ikl = 1, klonv
    !     FallOK(ikl) = max(zero, sign(unun, TvegSV(ikl) - TfSnow + epsi))&
    !             * max(zero, sign(unun, snCaSV(ikl) - epsi))
    !     dsn_SV(ikl) = dsn_SV(ikl) + snCaSV(ikl) * FallOK(ikl)&
    !             / dt__SV
    !     snCaSV(ikl) = snCaSV(ikl) * (1. - FallOK(ikl))
#ifdef AE
    ! +--Blowing Particles Threshold Friction velocity
    ! +  =============================================
    !     usthSV(ikl) = 1.0e+2
#endif
    ! end do
    
    ! +--Contribution of Snow to the Surface Snow Pack
    ! +  =============================================
    
    if(SnoMod) then
#ifdef m1
        ! Snow Initial Mass (below the Canopy) and Forcing
        ! ------------------------------------------------
        do ikl = 1, klonv
            ! [mm w.e.]
            SIWa_i(ikl) = (drr_SV(ikl) + dsn_SV(ikl)) * dt__SV
            SIWe_i(ikl) = dbs_SV(ikl)
            SIWm_0(ikl) = BufsSV(ikl) + HFraSV(ikl) * ro_Ice
            do isn = 1, nsno
                SIWm_0(ikl) = SIWm_0(ikl) + dzsnSV(ikl, isn) * ro__SV(ikl, isn)
            enddo
        enddo
#endif
        ! +--Blowing Snow
        ! +  ------------
        if(BloMod) then
            if(klonv == 1) then
                if(isnoSV(1) >= 2 .and. &
                        TsisSV(1, max(1, isnoSV(1))) < 273. .and. &
                        ro__SV(1, max(1, isnoSV(1))) < 500. .and. &
                        eta_SV(1, max(1, isnoSV(1))) < epsi) then
                    ! +  **********
                    call SISVAT_BSn
                    ! +  **********
                endif
            else
                ! +  **********
                call SISVAT_BSn
                ! +  **********
            endif
        endif
#ifdef ve
        !    **********
        call SISVAT_wEq('_BSn  ', 1)
        !    **********
#endif
        
        ! +--Sea Ice
        ! +  -------
        
        ! +  **********
        call SISVAT_SIc(SIvAcr)
        ! +  **********
#ifdef ve
        ! **********
        call SISVAT_wEq('_SIc  ', 0)
        ! **********
#endif
        
        ! +--Buffer Layer
        ! +  ------------
        do ikl = 1, klonv
            !  BufsSV(ikl) [mm w.e.], i.e [kg/m2]
            d_Bufs = max(dsn_SV(ikl) * dt__SV, 0.)
            dsn_SV(ikl) = 0.
            Bufs_N = BufsSV(ikl) + d_Bufs
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6601) BufsSV(ikl), d_Bufs, Bufs_N
            6601        format(/, 'Buffer *: ', 3e15.6)
#endif
            ! +--Snow Density
            ! +  ^^^^^^^^^^^^
            Polair = zero
#ifdef NP
            Polair = max(zero, &
                    sign(unun, TaPole &
                            - TaT_SV(ikl)))
#endif
            Polair = max(zero, sign(unun, TaPole - TaT_SV(ikl)))
            ! Fallen Snow Density [kg/m3] Pahaut (CEN)
            Buf_ro = max(rosMin, &
                    roSn_1 + roSn_2 * (TaT_SV(ikl) - TfSnow) &
                            + roSn_3 * sqrt(VV10SV(ikl)))
            ! Fallen Snow Density Kotlyakov (1961)
#ifdef NP
            BufPro = max(rosMin, &
                    104. * sqrt(max(VV10SV(ikl) - 6.0, 0.0)))
#endif
            
            ! Fallen Snow Density, Adapted for Antarctica
            density_kotlyakov = .true.
#ifdef AC
            ! C.Agosta snow densisty as if BS is on b
            density_kotlyakov = .false.
#endif
#ifdef BS
            ! C.Amory BS 2018
            density_kotlyakov = .false.
#endif
            if(density_kotlyakov) then
                tt_tmp = TaT_SV(ikl) - TfSnow
                vv_tmp = VV10SV(ikl)
                ! + ...         [ A compromise between
                ! + ...           Kotlyakov (1961) and Lenaerts (2012, JGR, Part1) ]
                if(tt_tmp >= -10) then
                    BufPro = max(rosMin, &
                            104. * sqrt(max(vv_tmp - 6.0, 0.0))) ! Kotlyakov (1961)
                else
                    vv_virt = (tt_c * vv_tmp + vv_c * (tt_tmp + 10)) &
                            / (tt_c + tt_tmp + 10)
                    BufPro = 104. * sqrt(max(vv_virt - 6.0, 0.0))
                endif
            else
                ! + ...         [ density derived from observations of the first 50cm of
                ! + ...           snow - cf. Rajashree Datta - and multiplied by 0.8 ]
                ! + ...           C. Agosta, 2016-09
                !c #SD           BufPro = 149.2 + 6.84*VV10SV(ikl) + 0.48*Tsrfsv(ikl)
                !c #SD           BufPro = 125 + 14*VV10SV(ikl) + 0.6*Tsrfsv(ikl) !MAJ CK and CAm
                BufPro = 200 + 21 * VV10SV(ikl)!CK 29/07/19
            endif
            ! Temperate Snow or Polar     Snow
            Bros_N = (1. - Polair) * Buf_ro &
                    + Polair * BufPro
            
            !XF !!!!
            Bros_N = max(20., max(rosMin, Bros_N))
            Bros_N = min(400., min(rosMax - 1, Bros_N)) ! for dz_min in SISVAT_zSn
            !XF !!!!

#ifdef BS
            !    Density of deposited blown snow
            !    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            Bros_N = frsno
            ro_new = ro__SV(ikl, max(1, isnoSV(ikl)))
            ro_new = max(Bros_N, min(roBdSV, ro_new))
            Fac = 1 - ((ro__SV(ikl, max(1, isnoSV(ikl))) &
                    - roBdSV) / (500. - roBdSV))
            Fac = max(0., min(1., Fac))
            dsnbSV(ikl) = Fac * dsnbSV(ikl)
            Bros_N = Bros_N * (1.0 - dsnbSV(ikl)) &
                    + ro_new * dsnbSV(ikl)
#endif
            !    Instantaneous Density if deposited blown Snow (Melted* from Canopy)
            !    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            Bros_N = Bros_N * (1.0 - FallOK(ikl)) &
                    + 300. * FallOK(ikl)
            
            !    Time averaged Density of deposited blown Snow
            !    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            BrosSV(ikl) = (Bros_N * d_Bufs &
                    + BrosSV(ikl) * BufsSV(ikl)) &
                    / max(epsi, Bufs_N)
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6602) Buf_ro, Bros_N, BrosSV(ikl), dsnbSV(ikl)
            6602        format('rho    *: ', 3e15.6, '    dsnbSV: ', e15.6)
#endif
            ! +-- S.Falling Snow Properties (computed as in SISVAT_zAg)
            ! +     ^^^^^^^^^^^^^^^^^^^^^^^
            ! Temperate Snow  Dendricity
            Buf_G1 = max(-G1_dSV, &
                    min(Dendr1 * VV__SV(ikl) - Dendr2, &
                            Dendr3))
            ! Temperate Snow Sphericity
            Buf_G2 = min(Spher4, &
                    max(Spher1 * VV__SV(ikl) + Spher2, &
                            Spher3))    !
            ! Temperate Snow OR Polar Snow
            Buf_G1 = (1. - Polair) * Buf_G1 &
                    + Polair * G1_dSV
            Buf_G2 = (1. - Polair) * Buf_G2 &
                    + Polair * ADSdSV
            ! NO  Blown Snow
            G1 = Buf_G1
            ! NO  Blown Snow
            G2 = Buf_G2
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6603) BG1sSV(ikl), BG2sSV(ikl)
            6603        format('G1,G2  *: ', 3e15.6)
#endif
#ifdef BS
            !     S.1. Meme  Type  de Neige  / same Grain Type
            !          ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            SameOK = max(zero, &
                    sign(unun, Buf_G1 * G1_dSV &
                            - eps_21))
            G1same = ((1.0 - dsnbSV(ikl)) * Buf_G1 + dsnbSV(ikl) * G1_dSV)
            G2same = ((1.0 - dsnbSV(ikl)) * Buf_G2 + dsnbSV(ikl) * ADSdSV)
            !           Blowing Snow Properties:                         G1_dSV, ADSdSV
            !     S.2. Types differents / differents Types
            !          ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            ! =1.=> Dendritic
            typ__1 = max(zero, sign(unun, epsi - Buf_G1))
            ! fract.Dendr.Lay.
            zroNEW = typ__1 * (1.0 - dsnbSV(ikl)) &
                    + (1. - typ__1) * dsnbSV(ikl)
            ! G1 of Dendr.Lay.
            G1_NEW = typ__1 * Buf_G1 &
                    + (1. - typ__1) * G1_dSV
            ! G2 of Dendr.Lay.
            G2_NEW = typ__1 * Buf_G2 &
                    + (1. - typ__1) * ADSdSV
            ! fract.Spher.Lay.
            zroOLD = (1. - typ__1) * (1.0 - dsnbSV(ikl)) &
                    + typ__1 * dsnbSV(ikl)
            ! G1 of Spher.Lay.
            G1_OLD = (1. - typ__1) * Buf_G1 &
                    + typ__1 * G1_dSV
            ! G2 of Spher.Lay.
            G2_OLD = (1. - typ__1) * Buf_G2 &
                    + typ__1 * ADSdSV
            ! Size  Dendr.Lay.
            SizNEW = -G1_NEW * DDcdSV / G1_dSV &
                    + (1. + G1_NEW / G1_dSV) &
                            * (G2_NEW * DScdSV / G1_dSV &
                                    + (1. - G2_NEW / G1_dSV) * DFcdSV)
            ! Spher.Dendr.Lay.
            SphNEW = G2_NEW / G1_dSV
            ! Size  Spher.Lay.
            SizOLD = G2_OLD
            ! Spher.Spher.Lay.
            SphOLD = G1_OLD / G1_dSV
            ! Averaged Size
            Siz_av = (zroNEW * SizNEW + zroOLD * SizOLD)
            ! Averaged Sphericity
            Sph_av = min(zroNEW * SphNEW + zroOLD * SphOLD &
                    , unun)
            Den_av = min((Siz_av - (Sph_av * DScdSV &
                    + (1. - Sph_av) * DFcdSV)) &
                    / (DDcdSV - (Sph_av * DScdSV &
                            + (1. - Sph_av) * DFcdSV)) &
                    , unun)
            ! Small   Grains
            ! Faceted Grains
            DendOK = max(zero, &
                    sign(unun, Sph_av * DScdSV &
                            + (1. - Sph_av) * DFcdSV &
                            - Siz_av))
            ! +...      REMARQUE: le  type moyen (dendritique ou non) depend
            ! +         ^^^^^^^^  de la  comparaison avec le diametre optique
            ! +                   d'une neige recente de   dendricite nulle
            ! +...      REMARK:   the mean type  (dendritic   or not) depends
            ! +         ^^^^^^    on the comparaison with the optical diameter
            ! +                   of a recent snow    having zero dendricity
            G1diff = (-DendOK * Den_av &
                    + (1. - DendOK) * Sph_av) * G1_dSV
            G2diff = DendOK * Sph_av * G1_dSV &
                    + (1. - DendOK) * Siz_av
            G1 = SameOK * G1same &
                    + (1. - SameOK) * G1diff
            G2 = SameOK * G2same &
                    + (1. - SameOK) * G2diff
#endif
            ! FallOK : Melted *  from Canopy
            BG1__N = ((1. - FallOK(ikl)) * G1 &
                    + FallOK(ikl) * 99.) &
                    * d_Bufs / max(epsi, d_Bufs)
            ! FallOK : Melted *  from Canopy
            BG2__N = ((1. - FallOK(ikl)) * G2 &
                    + FallOK(ikl) * 30.) &
                    * d_Bufs / max(epsi, d_Bufs)
            
            ! +-- S.Buffer  Snow Properties (computed as in SISVAT_zAg)
            ! +     ^^^^^^^^^^^^^^^^^^^^^^^
            ! Falling   Snow
            Buf_G1 = BG1__N
            ! Falling   Snow
            Buf_G2 = BG2__N
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6604) Buf_G1, Buf_G2, FallOK(ikl) &
                            , TvegSV(ikl)
            6604        format('G1,G2 F*: ', 3e15.6, '    T__Veg: ', e15.6)
#endif
            !     S.1. Meme  Type  de Neige  / same Grain Type
            !          ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            SameOK = max(zero, &
                    sign(unun, Buf_G1 * BG1sSV(ikl) &
                            - eps_21))
            G1same = (d_Bufs * Buf_G1 + BufsSV(ikl) * BG1sSV(ikl)) &
                    / max(epsi, Bufs_N)
            G2same = (d_Bufs * Buf_G2 + BufsSV(ikl) * BG2sSV(ikl)) &
                    / max(epsi, Bufs_N)
            
            !     S.2. Types differents / differents Types
            !          ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            typ__1 = max(zero, sign(unun, epsi - Buf_G1))   ! =1.=> Dendritic
            ! fract.Dendr.Lay.
            zroNEW = (typ__1 * d_Bufs &
                    + (1. - typ__1) * BufsSV(ikl)) &
                    / max(epsi, Bufs_N)
            ! G1 of Dendr.Lay.
            G1_NEW = typ__1 * Buf_G1 &
                    + (1. - typ__1) * BG1sSV(ikl)
            ! G2 of Dendr.Lay.
            G2_NEW = typ__1 * Buf_G2 &
                    + (1. - typ__1) * BG2sSV(ikl)
            ! fract.Spher.Lay.
            zroOLD = ((1. - typ__1) * d_Bufs &
                    + typ__1 * BufsSV(ikl)) &
                    / max(epsi, Bufs_N)
            ! G1 of Spher.Lay.
            G1_OLD = (1. - typ__1) * Buf_G1 &
                    + typ__1 * BG1sSV(ikl)
            ! G2 of Spher.Lay.
            G2_OLD = (1. - typ__1) * Buf_G2 &
                    + typ__1 * BG2sSV(ikl)
            ! Size  Dendr.Lay.
            SizNEW = -G1_NEW * DDcdSV / G1_dSV &
                    + (1. + G1_NEW / G1_dSV) &
                            * (G2_NEW * DScdSV / G1_dSV &
                                    + (1. - G2_NEW / G1_dSV) * DFcdSV)
            ! Spher.Dendr.Lay.
            SphNEW = G2_NEW / G1_dSV
            ! Size  Spher.Lay.
            SizOLD = G2_OLD
            ! Spher.Spher.Lay.
            SphOLD = G1_OLD / G1_dSV
            ! Averaged Size
            Siz_av = (zroNEW * SizNEW + zroOLD * SizOLD)
            Sph_av = min(zroNEW * SphNEW + zroOLD * SphOLD &
                    , unun)  ! Averaged Sphericity
            Den_av = min((Siz_av - (Sph_av * DScdSV &
                    + (1. - Sph_av) * DFcdSV)) &
                    / (DDcdSV - (Sph_av * DScdSV &
                            + (1. - Sph_av) * DFcdSV)) &
                    , unun)
            ! Small   Grains
            ! Faceted Grains
            DendOK = max(zero, &
                    sign(unun, Sph_av * DScdSV &
                            + (1. - Sph_av) * DFcdSV &
                            - Siz_av))
            ! +...      REMARQUE: le  type moyen (dendritique ou non) depend
            ! +         ^^^^^^^^  de la  comparaison avec le diametre optique
            ! +                   d'une neige recente de   dendricite nulle
            ! +...      REMARK:   the mean type  (dendritic   or not) depends
            ! +         ^^^^^^    on the comparaison with the optical diameter
            ! +                   of a recent snow    having zero dendricity
            
            G1diff = (-DendOK * Den_av &
                    + (1. - DendOK) * Sph_av) * G1_dSV
            G2diff = DendOK * Sph_av * G1_dSV &
                    + (1. - DendOK) * Siz_av
            G1 = SameOK * G1same &
                    + (1. - SameOK) * G1diff
            G2 = SameOK * G2same &
                    + (1. - SameOK) * G2diff
            
            BG1sSV(ikl) = G1 &
                    * Bufs_N / max(epsi, Bufs_N)
            BG2sSV(ikl) = G2 &
                    * Bufs_N / max(epsi, Bufs_N)
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6605) Buf_G1, typ__1 &
                            , DendOK, Den_av, Sph_av, Siz_av &
                            , G1same, G1diff, G1
            6605        format('B1,Typ  : ', 2e15.6, 11x, 'OK,Den,Sph,Siz: ', 4e15.6 &
                    , /, '          ', 30x, 11x, 'sam,dif,G1    : ', 3e15.6)
#endif
            ! +--Update of Buffer Layer Content & Decision about creating a new snow layer
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            BufsSV(ikl) = Bufs_N                  !     [mm w.e.]
            ! Allows to create
            ! a new snow Layer
            ! if Buffer > SMndSV
            ! Except if * Erosion dominates
            NLaysv(ikl) = min(unun, &
                    max(zero, &
                            sign(unun, BufsSV(ikl) &
                                    - SMndSV)) &
                            * max(zero, &
                            sign(unun, 0.50 &
                                    - dsnbSV(ikl))) &
                            ! Allows to create
                            ! a new snow Layer
                            ! is Buffer > SMndSV*3
                            + max(zero, &
                            sign(unun, BufsSV(ikl) &
                                    - SMndSV * 3.00)))
            ! [mm w.e.] -> [m]
            Bdzssv(ikl) = 1.e-3 * BufsSV(ikl) * ro_Wat &
                    / max(epsi, BrosSV(ikl))!& [m w.e.] -> [m]
#ifdef s0
            ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
            ! OUTPUT           for Buffer G1, G2 variables
            if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                    nn__SV(ikl) == nwr_SV) &
                    write(6, 6606) BG1sSV(ikl), BG2sSV(ikl) &
                            , NLaysv(ikl), BdzsSV(ikl)
            6606        format('G1,G2 N*: ', 2e15.6, i15, e27.6)
#endif
        enddo
        
        ! +--Snow Pack Discretization
        ! +  ========================
        if(klonv == 1) then
            if(isnoSV(1) >= 1 .or. NLaysv(1) >= 1) then
                ! +  **********
                call SISVAT_zSn
                ! +  **********
            endif
        else
            ! +  **********
            call SISVAT_zSn
            ! +  **********
        endif
#ifdef ve
        ! +  **********
        call SISVAT_wEq('_zSn  ', 0)
        ! +  **********
#endif
#ifdef EF
        if(isnoSV(1) > 0) &
                write(6, 6004) isnoSV(1), dsn_SV(1) * dt__SV + BufsSV(1), &
                        ((dzsnSV(1, isn) * ro__SV(1, isn)), isn = 1, isnoSV(1))
        6004    format(i3, '  dsn+Buf=', f6.2, 6x, 'z dz *ro =', 10f6.2, &
                (/, 35x, 10f6.2))
#endif
        
        ! +--Add a new Snow Layer
        ! +  ====================
        
        do ikl = 1, klonv
#ifdef EC
            if(NLaysv(ikl) > 0) &
                    write(6, 6005) isnoSV(ikl), 1.e3 * Bdzssv(ikl), Brossv(ikl), &
                            BG1ssv(ikl), BG2ssv(ikl)
            6005        format(i3, ' dz     = ', f6.3, 3x, ' ro     = ', f6.1, 3x, &
                    ' G1     = ', f6.3, 3x, ' G2     = ', f6.1)
#endif
            ! +
            isnoSV(ikl) = isnoSV(ikl) + NLaysv(ikl)
            isn = isnoSV(ikl)
            dzsnSV(ikl, isn) = dzsnSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + Bdzssv(ikl) * NLaysv(ikl)
            TsisSV(ikl, isn) = TsisSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + min(TaT_SV(ikl), TfSnow) * NLaysv(ikl)
            ro__SV(ikl, isn) = ro__SV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + Brossv(ikl) * NLaysv(ikl)
            eta_SV(ikl, isn) = eta_SV(ikl, isn) * (1 - NLaysv(ikl))   ! + 0.
            agsnSV(ikl, isn) = agsnSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + (real(jdarGE + njyrGE(mmarGE)) / 365. + iyrrGE) &
                            * NLaysv(ikl)
            G1snSV(ikl, isn) = G1snSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + BG1ssv(ikl) * NLaysv(ikl)
            G2snSV(ikl, isn) = G2snSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + BG2ssv(ikl) * NLaysv(ikl)
            istoSV(ikl, isn) = istoSV(ikl, isn) * (1 - NLaysv(ikl)) &
                    + max(zero, sign(unun, TaT_SV(ikl) &
                            - TfSnow - eps_21)) * istdSV(2) &
                            * NLaysv(ikl)
            BufsSV(ikl) = BufsSV(ikl) * (1 - NLaysv(ikl))
            NLaysv(ikl) = 0
        enddo
        
        ! +--Snow Pack Thickness
        ! +  -------------------
        
        do ikl = 1, klonv
            z_snsv(ikl) = 0.0
        enddo
        do isn = 1, nsno
            do ikl = 1, klonv
                z_snsv(ikl) = z_snsv(ikl) + dzsnSV(ikl, isn)
                zzsnsv(ikl, isn) = z_snsv(ikl)
            enddo
        enddo
        
        ! +--Diffusion of Surficial Water in the Snow Pack
        ! +  ---------------------------------------------

#ifdef sw
        do isn = 1, nsno
            do ikl = 1, klonv
                PorVol = 1. - ro__SV(ikl, isn) / ro_Ice
                PorVol = max(PorVol, zero)
                rWater = ws0dSV * PorVol * ro_Wat * dzsnSV(ikl, isn) &
                        * max(zero, &
                                sign(unun, rusnSV(ikl) / ro_Wat - zzsnsv(ikl, isn) &
                                        + dzsnSV(ikl, isn)))
                rusNEW = max(rusnSV(ikl) - rWater, zero)
                rWater = rusnSV(ikl) - rusNEW
                rdzNEW = rWater &
                        + ro__SV(ikl, isn) * dzsnSV(ikl, isn)
                etaNEW = rWater / max(epsi, rdzNEW)
                rusnSV(ikl) = rusNEW
                ro__SV(ikl, isn) = rdzNEW / max(epsi, dzsnSV(ikl, isn))
                eta_SV(ikl, isn) = eta_SV(ikl, isn) + etaNEW
            enddo
        enddo
#endif
    
    endif

#ifdef EF
    if(isnoSV(1) > 0) &
            write(6, 6006) isnoSV(1), dsn_SV(1) * dt__SV + BufsSV(1), &
                    ((dzsnSV(1, isn) * ro__SV(1, isn)), isn = 1, isnoSV(1))
    6006 format(i3, '  dsn+Buf=', f6.2, 6x, '* dz *ro =', 10f6.2, &
            (/, 35x, 10f6.2))
#endif

#ifdef BD
    ! +--Blowing Dust
    ! +  ============
    
    if(BloMod) then
        ! +       ***************
        call SISVAT_BDu
        ! +       ***************
    endif
#endif
    
    ! +--Soil      Albedo: Soil Humidity Correction
    ! +  ==========================================
    
    ! +...    REFERENCE: McCumber and Pielke (1981), Pielke (1984)
    ! +       ^^^^^^^^^
    do ikl = 1, klonv
        albssv(ikl) = &
                alb0SV(ikl) * (1.0 - min(demi, eta_SV(ikl, 0) &
                        / etadSV(isotSV(ikl))))**0.5
        !XF
        ! +...      REMARK: Albedo of Water Surfaces (isotSV=0):
        ! +         ^^^^^^  alb0SV := 2  X  effective value, while
        ! +                 eta_SV :=          etadSV
    enddo
    
    ! +--Snow Pack Optical Properties
    ! +  ============================
    
    if(SnoMod) then
        
        ! +  ******
        call SnOptP
        ! +  ******
    
    else
        do ikl = 1, klonv
            sEX_sv(ikl, 1) = 1.0
            sEX_sv(ikl, 0) = 0.0
            albisv(ikl) = albssv(ikl)
        enddo
    endif
#ifdef ve
    ! +  **********
    call SISVAT_wEq('SnOptP', 0)
    ! +  **********
#endif
    
    ! +--Solar Radiation Absorption and Effective Leaf Area Index
    ! +  ========================================================
    
    ! +  ******
    call VgOptP
    ! +  ******
    
    ! +--Surface-Canopy Emissivity
    ! +  =========================
    
    do ikl = 1, klonv
        
        LSnMsk = min(iun, isnoSV(ikl))
        ! Veg Transmit.Frac.
        tau_sv(ikl) = exp(-LAI_sv(ikl))
        ! Veg+Sno Emissivity
        Evg_sv(ikl) = EmiVeg * (1 - LSnMsk) + EmiSno * LSnMsk
        ! Sol+Sno Emissivity
        Eso_sv(ikl) = EmiSol * (1 - LSnMsk) + EmiSno * LSnMsk
        emi_SV(ikl) = &
                (((EmiSol * tau_sv(ikl) &
                        + EmiVeg * (1.0 - tau_sv(ikl))) * LSmask(ikl)) &
                        + EmiWat * (1 - LSmask(ikl))) * (1 - LSnMsk) &
                        + EmiSno * LSnMsk

#ifdef AO
        ! ocean
        if(LSmask(ikl) == 0) then
            ! covered by snow/ice
            if(LSnMSK == 1) then
                ! Imposed Values from NEMO
                Evg_sv(ikl) = EmiSnoAO
                Eso_sv(ikl) = EmiSnoAO
                Emi_sv(ikl) = EmiSnoAO
                !  open water
            else
                ! Imposed Values from NEMO
                emi_sv(ikl) = EmiwatAO
            endif
        endif
#endif
    enddo
    
    ! +--Soil/Vegetation Forcing/ Upward IR (INPUT, from previous time step)
    ! +  ===================================================================
    
    do ikl = 1, klonv
#ifdef e1
        Enrsvd(ikl) = -IRs_SV(ikl)
#endif
        IRupsv(ikl) = IRs_SV(ikl) * tau_sv(ikl) ! Upward   IR
    enddo
    
    ! +--Turbulence
    ! +  ==========
    
    ! +--Latent Heat of Vaporization/Sublimation
    ! +  ---------------------------------------
    
    do ikl = 1, klonv
        SnoWat = min(isnoSV(ikl), 1)
        Lx_H2O(ikl) = &
                (1. - SnoWat) * Lv_H2O &
                        + SnoWat * (Ls_H2O * (1. - eta_SV(ikl, isnoSV(ikl))) &
                        + Lv_H2O * eta_SV(ikl, isnoSV(ikl)))
    enddo
    
    ! +--Roughness Length for Momentum
    ! +  -----------------------------
    
    ! +--Land+Sea-Ice / Ice-free Sea Mask
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    do ikl = 1, klonv
        IcIndx(ikl) = 0
    enddo
    do isn = 1, nsno
        do ikl = 1, klonv
            IcIndx(ikl) = max(IcIndx(ikl), &
                    isn * max(0, &
                            sign(1, &
                                    int(ro__SV(ikl, isn) - 900.))))
        enddo
    enddo
    
    do ikl = 1, klonv
        LISmsk = min(iiceSV(ikl), 1)
        LISmsk = max(LSmask(ikl), LISmsk)
        IceMsk = max(0, sign(1, IcIndx(ikl) - 1))
        SnoMsk = max(min(isnoSV(ikl) - iiceSV(ikl), 1), 0)
        
        ! +--Sea  Roughness Length
        ! +  ^^^^^^^^^^^^^^^^^^^^^
        Z0mSea = 0.0002
        Z0hSea = 0.000049

#ifdef zs
        ! Doyle MWR 130 p.3088 2e col
        Z0mSea = 0.0185 * us__SV(ikl) * us__SV(ikl) * grvinv
#endif
        ! Wang  MWR 129 p.1377 (21)
        Z0mSea = 0.016 * us__SV(ikl) * us__SV(ikl) &
                * grvinv &
                + 0.11 * akmol &
                        / max(epsi, us__SV(ikl))

#ifdef zs
        ! Wang  MWR 129 p.1377 (21) (adapted)
        Z0mSea = 0.0185 * us__SV(ikl) * us__SV(ikl) &
                * grvinv &
                + 0.135 * akmol &
                        / max(epsi, us__SV(ikl))
#endif
        ! Wang  MWR 129 p.1377 (22)
        Z0hSea = max(0.000049, &
                0.20 * akmol &
                        / max(epsi, us__SV(ikl)))
        
        Z0mSea = max(Z0mSea, epsi)                   !
        
        ! +--Land Roughness Length, Snow Contribution excluded
        ! +  ^^^^^^^^^^^^^^^^^^^^^^ Ice  Contribution included
        ! +                         ^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! +--If vegetation Seasonal Cycle described by  LAI     :
        growth = min(max(0, 7 - ivgtSV(ikl)), 1)
        Z0mLnd = Z0mdSV(ivgtSV(ikl)) * LAI_sv(ikl) * growth &
                / LAIdSV &
                + Z0mdSV(ivgtSV(ikl)) * (1 - growth)
        
        ! +--If vegetation Seasonal Cycle described by  GLF only:
        Z0mLnd = &
                fallen * Z0mLnd &
                        + (1. - fallen) * Z0mdSV(ivgtSV(ikl)) * glf_sv(ikl) * growth &
                        + Z0mdSV(ivgtSV(ikl)) * (1 - growth)
        
        ! +--Land Roughness Length, Influence of the Masking by Snow
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0mLnd = max(Z0mLnd, &
                Z0mdSV(0) * (iun - IceMsk) &
                        + Z0_ICE * IceMsk)
        Z0mLnd = Z0mLnd &
                - (zzsnsv(ikl, isnoSV(ikl)) &
                        - zzsnsv(ikl, max(IcIndx(ikl), 0))) / 7.
        Z0mLnd = max(Z0mLnd, 5.e-5)  ! Min set := Z0 on *
        ! +...    Roughness  disappears under Snow
        ! +       Assumption Height/Roughness Length =  7 is used
        
        ! +--Z0 Smooth Regime over Snow (Andreas 1995, CRREL Report 95-16, p. 8)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0m_nu = 5.e-5 ! z0s~(10-d)*exp(-vonkar/sqrt(1.1e-03))
        
        ! +--Z0 Saltat.Regime over Snow (Gallee  et al., 2001, BLM 99 (19) p.11)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        u2star = us__SV(ikl) * us__SV(ikl)
        Z0mBSn = u2star * 0.536e-3 - 61.8e-6
        Z0mBSn = max(Z0mBS0, Z0mBSn)
        
        ! +--Z0 Smooth + Saltat. Regime
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0enSV(ikl) = Z0m_nu &
                + Z0mBSn
        
        ! +--Rough   Snow Surface Roughness Length (Typical Value)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
#ifdef tz
        Z0m_Sn = 0.250e-3 ! Andreas 1995, CRREL Report 95-16, fig.1&p.2
#endif
        !                              ! z0r~(10-d)*exp(-vonkar/sqrt(1.5e-03))-5.e-5
        Z0m_Sn = 2.000e-3 ! Calibration    of MAR
#ifdef TZ
        Z0m_Sn = 1.000e-3 ! Exemple Tuning in RACMO
        Z0m_Sn = 0.500e-3 ! Exemple Tuning in MAR
#endif
        
        ! +--Rough   Snow Surface Roughness Length (Variable Sastrugi Height)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        A_Fact = 1.0000        ! Andreas et al., 2004, p.4
        !                                      ! ams.confex.com/ams/pdfpapers/68601.pdf
        
        ! Parameterization of z0 dependance on Temperature (C. Amory, 2017)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ! Z0=f(T) deduced from observations, Adelie Land, dec2012-dec2013
        coefa = 0.1658 !0.1862 !Ant
        coefb = -50.3869 !-55.7718 !Ant
        ta1 = 253.15 !255. Ant
        ta2 = 273.15
        ta3 = 273.15 + 3
        z01 = exp(coefa * ta1 + coefb) !~0.2 ! ~0.25 mm
        z02 = exp(coefa * ta2 + coefb) !~6  !~7 mm
        z03 = z01
        coefc = log(z03 / z02) / (ta3 - ta2)
        coefd = log(z03) - coefc * ta3
        if(TaT_SV(ikl) < ta1) then
            Z0_obs = z01
        else if(TaT_SV(ikl) >= ta1 .and. TaT_SV(ikl) < ta2) then
            Z0_obs = exp(coefa * TaT_SV(ikl) + coefb)
        else if(TaT_SV(ikl) >= ta2 .and. TaT_SV(ikl) < ta3) then
            !           ! if st > 0, melting induce smooth surface
            Z0_obs = exp(coefc * TaT_SV(ikl) + coefd)
        else
            Z0_obs = z03
        endif
        
        ! Z0_obs = 1.000e-3
        
        !CAa Snow roughness lenght deduced from observations
        !CAa (parametrization if no Blowing Snow)
        !CAa ----------------------------------- C. Agosta 09-2016 -----
        !CAa Substract Z0enSV(ikl) because re-added later in Z0mnSV(ikl)
        Z0m_Sn = Z0_obs - Z0enSV(ikl)
        !CAa -----------------------------------------------------------
        
        param = Z0_obs / 1. ! param(s) | 1.(m/s)=TUNING
#ifdef SZ
        ! 0.0001 = TUNING
        Z0Sa_N = (us__SV(ikl) - 0.2) * param &
                * max(zero, sign(unun, TfSnow - eps9 &
                        - TsisSV(ikl, isnoSV(ikl))))
        ! 1 if erosion
        ! Z0SaSi = max(zero,sign(unun,Z0Sa_N                  ))
        Z0SaSi = max(zero, sign(unun, zero - eps9 - uss_SV(ikl)))!
        Z0Sa_N = max(zero, Z0Sa_N)
        Z0SaSV(ikl) = &
                max(Z0SaSV(ikl), Z0SaSV(ikl) &
                        + Z0SaSi * (Z0Sa_N - Z0SaSV(ikl)) * exp(-dt__SV / 43200.)) &
                        - min(dz0_SV(ikl), Z0SaSV(ikl))
        ! CAUTION: The influence of the sastrugi direction is not yet included
        ! A=5 if h~10cm
        A_Fact = Z0SaSV(ikl) * 5.0 / 0.15
        Z0m_Sn = Z0SaSV(ikl) &
                - Z0m_nu
#endif
#ifdef ZN
        ! +--Z0 Saltat.Regime over Snow (Shao & Lin, 1999, BLM 91 (46)  p.222)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        sqrrZ0 = usthSV(ikl) / max(us__SV(ikl), 0.001)
        sqrrZ0 = min(sqrrZ0, 0.999)
        Z0mBSn = 0.55 * 0.55 * exp(-sqrrZ0 * sqrrZ0) &
                * us__SV(ikl) * us__SV(ikl) * grvinv * 0.5
        ! +--Z0 Smooth + Saltat. Regime (Shao & Lin, 1999, BLM 91 (46)  p.222)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0enSV(ikl) = (Z0m_nu**sqrrZ0) &
                * (Z0mBSn**(1. - sqrrZ0))
        Z0enSV(ikl) = max(Z0enSV(ikl), Z0m_nu)
#endif
#ifdef ZA
        ! +--Z0 Smooth Regime over Snow (Andreas etAl., 2004
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^  ams.confex.com/ams/pdfpapers/68601.pdf)
        Z0m_nu = 0.135 * akmol / max(us__SV(ikl), epsi)
        ! +--Z0 Saltat.Regime over Snow (Andreas etAl., 2004
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^  ams.confex.com/ams/pdfpapers/68601.pdf)
        Z0mBSn = 0.035 * u2star * grvinv
        ! +--Z0 Smooth + Saltat. Regime (Andreas etAl., 2004
        !    (      used by Erosion)     ams.confex.com/ams/pdfpapers/68601.pdf)
        !    ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0enSV(ikl) = Z0m_nu &
                + Z0mBSn
        ! +--Z0 Rough  Regime over Snow (Andreas etAl., 2004
        ! +  (.not. used by Erosion)     ams.confex.com/ams/pdfpapers/68601.pdf)
        !    ^^^^^^^^^^^^^^^^^^^^^^^^^^
        u2star = (us__SV(ikl) - 0.1800) / 0.1
        Z0m_Sn = A_Fact * Z0mBSn * exp(-u2star * u2star)
        Z0m_90 = (10. - 0.025 * VVs_SV(ikl) / 5.) &
                * exp(-0.4 / sqrt(.00275 + .00001 * max(0., VVs_SV(ikl) - 5.)))
        Z0m_Sn = DDs_SV(ikl) * Z0m_90 / 45. &
                - DDs_SV(ikl) * DDs_SV(ikl) * Z0m_90 / (90. * 90.)
#endif
        ! +--Z0  (Erosion)    over Snow (instantaneous or time average)
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Z0e_SV(ikl) = Z0enSV(ikl)
        Z0e_SV(ikl) = Z0emSV(ikl)
        
        ! +--Momentum  Roughness Length
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! Contribution of
        ! Vegetation Form
        ! Sastrugi   Form
        ! Snow    Erosion
        Z0mnSV(ikl) = Z0mLnd &
                + (Z0m_Sn &
                        + Z0enSV(ikl)) * SnoMsk
        
        ! +--Mom. Roughness Length, Discrimination among Ice/Land  and Ice-Free Ocean
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! Ice and  Land +  Ice-Free Ocean
        Z0mnSV(ikl) = Z0mnSV(ikl) * LISmsk &
                + Z0mSea * (1 - LISmsk)
#ifdef OR
        ! Subgrid  Topogr.
        Z0mnSV(ikl) = Z0mnSV(ikl) + Z0roSV(ikl)
#endif

#ifdef GL
        ! +--GIS  Roughness Length
        ! +  ^^^^^^^^^^^^^^^^^^^^^
                                                  factor=10
        if(aiceSV(ikl)>=0.3.and.aiceSV(ikl)<=0.6) factor=2.5
        Z0mnSV(ikl) = &
                (1 - LSmask(ikl)) * Z0mnSV(ikl) &
                        + LSmask(ikl) * max(Z0mnSV(ikl), max(Z0_GIM, &
                        Z0_GIM + (Z0_GIM * factor - Z0_GIM) * (ro__SV(ikl, isnoSV(ikl)) - 600.) &
                                                 / (ro_ice - 600.)))
#endif
        
        ! +--Mom. Roughness Length, Instantaneous OR Box Moving Average in Time
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        ! Z0mnSV  instant.
        Z0m_SV(ikl) = Z0mnSV(ikl)
        ! Z0mnSV  Average
        Z0m_SV(ikl) = Z0mmSV(ikl)
        
        ! +--Corrected Threshold Friction Velocity before Erosion    ! Marticorena and
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^    ! Bergametti 1995
#ifdef BS
        ! not used anymore since Marticorena and Bergametti disabled !CK 18/07/2018
        ! Z0e_SV(ikl) =   min(Z0m_SV(ikl),Z0e_SV(ikl))
#endif
#ifdef MB
        ! f_eff=    log(0.35*(0.1        /Z0e_SV(ikl))**0.8) ! JGR 100
        ! (20) p. 16420  p.16426 2nd ?
        ! f_eff=1.-(log(      Z0m_SV(ikl)/Z0e_SV(ikl)      )) &
        !         /(max(      f_eff      ,epsi             ))
        ! CONTROL
        ! f_eff=    max(      f_eff      ,epsi              )
        ! TUNING
        ! f_eff=1.0   -(1.0 - f_eff)     /5.00
        ! f_eff=    min(      f_eff      ,1.00              )
        ! usthSV(ikl) =       usthSV(ikl)/f_eff
#endif
        
        ! +--Roughness Length for Scalars
        ! +  ----------------------------
        Z0hnSV(ikl) = Z0mnSV(ikl) / 7.4
        ! (Taylor & Clark, QJRMS 127,p864)
        ! Z0h = Z0m  /100.0 over the Sahel
#ifdef SH
        Z0hnSV(ikl) = Z0mnSV(ikl) / 100.0
#endif
#ifdef RN
        !XF #RN does not work over bare ice
        !XF MAR is then too warm and not enough melt
        rstar = Z0mnSV(ikl) * us__SV(ikl) / akmol
        rstar = max(epsi, min(rstar, thous))
        alors = log(rstar)
        rstar0 = 1.250e0 * max(zero, sign(unun, 0.135e0 - rstar)) &
                + (1. - max(zero, sign(unun, 0.135e0 - rstar))) &
                        * (0.149e0 * max(zero, sign(unun, 2.500e0 - rstar)) &
                                + 0.317e0 &
                                        * (1. - max(zero, sign(unun, 2.500e0 - rstar))))
        rstar1 = 0. * max(zero, sign(unun, 0.135e0 - rstar)) &
                + (1. - max(zero, sign(unun, 0.135e0 - rstar))) &
                        * (-0.55e0 * max(zero, sign(unun, 2.500e0 - rstar)) &
                                - 0.565 &
                                        * (1. - max(zero, sign(unun, 2.500e0 - rstar))))
        rstar2 = 0. * max(zero, sign(unun, 0.135e0 - rstar)) &
                + (1. - max(zero, sign(unun, 0.135e0 - rstar))) &
                        * (0. * max(zero, sign(unun, 2.500e0 - rstar)) &
                                - 0.183 &
                                        * (unun - max(zero, sign(unun, 2.500e0 - rstar))))
        if(ro__SV(ikl, isnoSV(ikl)) > 50 &
                .and. ro__SV(ikl, isnoSV(ikl)) < roSdSV) then
            Z0hnSV(ikl) = max(zero &
                    , sign(unun, zzsnsv(ikl, isnoSV(ikl)) - epsi)) &
                    * exp(rstar0 + rstar1 * alors + rstar2 * alors * alors) &
                    * 0.001e0 + Z0hnSV(ikl) * (1. - max(zero &
                    , sign(unun, zzsnsv(ikl, isnoSV(ikl)) - epsi)))
        endif
#endif
        
        ! Ice-free Ocean + Ice and  Land
        Z0hnSV(ikl) = Z0hSea * (1 - LISmsk) &
                + Z0hnSV(ikl) * LISmsk
        
        Z0h_SV(ikl) = Z0hnSV(ikl)
        Z0h_SV(ikl) = Z0hmSV(ikl)
        
        ! +--Contributions of the Roughness Lenghths to the neutral Drag Coefficient
        ! +  -----------------------------------------------------------------------

#ifdef MT
        ! Min Z0_m (Garrat Scheme)
        Z0m_SV(ikl) = max(2.0e-6, Z0m_SV(ikl))
#endif
        Z0m_SV(ikl) = min(Z0m_SV(ikl), za__SV(ikl) * 0.3333)
        sqrCm0(ikl) = log(za__SV(ikl) / Z0m_SV(ikl))
        sqrCh0(ikl) = log(za__SV(ikl) / Z0h_SV(ikl))

#ifdef wz
        if(ikl == 1) write(6, 6661) dsn_SV(ikl), us__SV(ikl), Z0SaSi &
                , Z0Sa_N, Z0SaSV(ikl), Z0m_Sn, Z0m_SV(ikl)
        6661    format(7f9.6)
#endif

#ifdef sz
        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! OUTPUT           of Roughness Length and Drag Coefficients
        if(ii__SV(ikl) == iwr_SV .and. jj__SV(ikl) == jwr_SV .and. &
                nn__SV(ikl) == nwr_SV) &
                write(6, 6600) za__SV(ikl), Z0m_SV(ikl) &
                        , sqrCm0(ikl), za__SV(ikl) / Z0m_SV(ikl) &
                        , Z0SaSV(ikl), Z0h_SV(ikl) &
                        , sqrCh0(ikl), za__SV(ikl) / Z0h_SV(ikl)
        6600    format(/, ' ** SISVAT     *0  ' &
                , '  za__SV  = ', e12.4, '  Z0m_SV  = ', e12.4 &
                , '  sqrCm0  = ', e12.4, '  Za/Z0m  = ', e12.4 &
                , /, '                   ' &
                , '  Z0SaSV  = ', e12.4, '  Z0h_SV  = ', e12.4 &
                , '  sqrCh0  = ', e12.4, '  Za/Z0h  = ', e12.4)
#endif
        
        ! +--Vertical Stability Correction
        ! +  -----------------------------
        
        ! +--Surface/Canopy Temperature
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Tsrfsv(ikl) = Sigmsv(ikl) * TvegSV(ikl) &
                + (1. - Sigmsv(ikl)) * TsisSV(ikl, isnoSV(ikl))
    enddo
    
    ! +--Aerodynamic Resistance
    ! +  ^^^^^^^^^^^^^^^^^^^^^^
#ifdef CP
    if(SnoMod .and. ColPrt) then
#endif

#ifdef CP
        ! +  **********
        call ColPrt_SBL
        ! +  **********
#endif

#ifdef CP
    else
#endif

#ifdef MT
        if(Garrat) then
            ! +  **********
            call SISVAT_SBL
            ! +  **********
        else
#endif
            
            ! +  **********
            call SISVATeSBL
            ! +  **********

#ifdef MT
        endif
#endif
#ifdef CP
    endif
#endif
    
    ! +--Friction Velocity
    ! +  -----------------

#ifdef US
    do ikl = 1, klonv
#ifdef WR
#endif
        u_star = sqrt(VV__SV(ikl) / ram_sv(ikl))
        write(6, *) u_star, us__SV(ikl)
#endif
#ifdef US
        us__SV(ikl) = sqrt(VV__SV(ikl) / ram_sv(ikl))
    enddo
#endif
    
    ! Canopy Energy Balance
    ! =====================
    
    ! +  **********
    call SISVAT_TVg(ETVg_d)
    ! +  **********
    
    ! +--Surface/Canopy Temperature
    ! +  ==========================
    
    do ikl = 1, klonv
        Tsrfsv(ikl) = Sigmsv(ikl) * TvegSV(ikl) &
                + (1. - Sigmsv(ikl)) * TsisSV(ikl, isnoSV(ikl))
    enddo
    
    ! Soil Energy Balance
    ! ===================
    do ikl = 1, klonv
        ist = ntPhys
        if(isnoSV(ikl) >= 1 .and. TaT_SV(ikl) >= 273.15) ist = ntPhys + 1 ! melting snow
        if(isnoSV(ikl) <= 0 .and. isotSV(ikl) <= 0) ist = 1 ! sea
    enddo
    dt__SV2 = dt__SV
    dt__SV = dt__SV / real(ist)
    do it = 1, max(1, ist)
        ! +  **********
        call SISVAT_TSo(ETSo_0, ETSo_1, ETSo_d)
        ! +  **********
    enddo
    dt__SV = dt__SV2

#ifdef ve
    ! +  **********
    call SISVAT_wEq('_TSo  ', 0)
    ! +  **********
#endif
    
    ! +--Canopy Water  Balance
    ! +  =====================
    
    ! +--Soil Water     Potential
    ! +  ------------------------
    
    do isl = -nsol, 0
        do ikl = 1, klonv
            ! Soil Type
            ist = isotSV(ikl)
            ! DR97, Eqn.(3.34)
            psi_sv(ikl, isl) = psidSV(ist) &
                    * (etadSV(ist) / max(epsi, eta_SV(ikl, isl))) &
                            **bCHdSV(ist)
            
            ! +--Soil Hydraulic Conductivity
            ! +  ---------------------------
            ! DR97, Eqn.(3.35)
            Khydsv(ikl, isl) = s2__SV(ist) &
                    * (eta_SV(ikl, isl)**(2. * bCHdSV(ist) + 3.))
        enddo
    enddo
    
    ! +  **********
    call SISVAT_qVg
    ! +  **********
    
    ! Vegetation Forcing
    ! ------------------
#ifdef m0
    do ikl = 1, klonv
        ! Canopy Precip. IN
        ! Canopy Precip. OUT
        ! Canopy Water Evap.
        Watsvd(ikl) = (Watsvd(ikl) &
                - drr_SV(ikl) &
                - Evp_sv(ikl)) * dt__SV
    enddo
#endif
    
    ! +--Melting / Refreezing in the Snow Pack
    ! +  =====================================
    
    if(SnoMod) then
        
        ! +  **********
        call SISVAT_qSn()
        ! +  **********

#ifdef ve
        ! +  **********
        call SISVAT_wEq('_qSn  ', 0)
        ! +  **********
#endif

#ifdef EF
        if(isnoSV(1) > 0) &
                write(6, 6007) isnoSV(1), dsn_SV(1) * dt__SV + BufsSV(1), &
                        ((dzsnSV(1, isn) * ro__SV(1, isn)), isn = 1, isnoSV(1))
        6007    format(i3, '  dsn+Buf=', f6.2, 6x, 'q dz *ro =', 10f6.2, &
                (/, 35x, 10f6.2))
#endif
        
        ! +--Snow Pack Thickness
        ! +  -------------------
        do ikl = 1, klonv
            z_snsv(ikl) = 0.0
        enddo
        do isn = 1, nsno
            do ikl = 1, klonv
                z_snsv(ikl) = z_snsv(ikl) + dzsnSV(ikl, isn)
                zzsnsv(ikl, isn) = z_snsv(ikl)
            enddo
        enddo
        
        ! +--Energy in Excess is added to the first Soil Layer
        ! +  -------------------------------------------------
        do ikl = 1, klonv
            z_snsv(ikl) = max(zero, &
                    sign(unun, epsi - z_snsv(ikl)))
            TsisSV(ikl, 0) = TsisSV(ikl, 0) + EExcsv(ikl) &
                    / (rocsSV(isotSV(ikl)) &
                            + rcwdSV * eta_SV(ikl, 0))
            EExcsv(ikl) = 0.
        enddo

#ifdef m1
        ! Snow Final   Mass (below the Canopy) and Forcing
        ! ------------------------------------------------
        do ikl = 1, klonv
            ! [mm w.e.]
            SIWa_f(ikl) = (drr_SV(ikl) + dsn_SV(ikl)) * dt__SV
            SIWe_f(ikl) = dbs_SV(ikl)
            SIWm_1(ikl) = BufsSV(ikl) + HFraSV(ikl) * ro_Ice
            do isn = 1, nsno
                SIWm_1(ikl) = SIWm_1(ikl) + dzsnSV(ikl, isn) * ro__SV(ikl, isn)
            enddo
        enddo
#endif
    endif
    
    ! Soil   Water  Balance
    ! =====================
    
    ! +  **********
    call SISVAT_qSo(Wats_0, Wats_1, Wats_d)
    ! +  **********
    
    ! +--Surface/Canopy Fluxes
    ! +  =====================
    
    do ikl = 1, klonv
        ! Downward IR
        IRdwsv(ikl) = tau_sv(ikl) * IRd_SV(ikl) * Eso_sv(ikl) &
                + (1.0 - tau_sv(ikl)) * IRd_SV(ikl) * Evg_sv(ikl)
        ! Upward   IR
        IRupsv(ikl) = IRupsv(ikl) &
                + 0.5 * IRv_sv(ikl) * (1. - tau_sv(ikl))
        ! Upward   IR
        ! (effective)
        ! (positive)
        IRu_SV(ikl) = -IRupsv(ikl) &
                + IRd_SV(ikl) &
                - IRdwsv(ikl)
        ! Brightness Temperature
        TBr_sv(ikl) = sqrt(sqrt(IRu_SV(ikl) / stefan))
        ! u*T*
        uts_SV(ikl) = (HSv_sv(ikl) + HSs_sv(ikl)) / (rhT_SV(ikl) * Cp)
        ! u*q*
        uqs_SV(ikl) = (HLv_sv(ikl) + HLs_sv(ikl)) / (rhT_SV(ikl) * Lv_H2O)
        
        ! +--Surface/Canopy Temperature
        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
        Tsrfsv(ikl) = Sigmsv(ikl) * TvegSV(ikl) + (1. - Sigmsv(ikl)) * TsisSV(ikl, isnoSV(ikl))
    enddo
    
    ! +--Snow Pack Properties (sphericity, dendricity, size)
    ! +  ===================================================
    
    if(SnoMod) then
        if(klonv == 1) then
            if(isnoSV(1) >= 1 .and. itphys == 1) then
                ! +    **********
                call SISVAT_GSn
                ! +    **********
            endif
        else
            ! +  **********
            call SISVAT_GSn
            ! +  **********
        endif

#ifdef ve
        ! +  **********
        call SISVAT_wEq('_GSn  ', 0)
        ! +  **********
#endif
        
        ! +--Surficial Water Freezing, including that of a Water Surface (isotSV=0)
        ! +  ======================================================================
    
    endif
    
    ! +--OUTPUT
    ! +  ======

#ifdef E0
    do ikl = 1, klonv
        if(lwriSV(ikl) /= 0) then
            noUNIT = no__SV(lwriSV(ikl))
            write(noUNIT, 5001) &
                    (SoSosv(ikl) + SoCasv(ikl)) * sol_SV(ikl), &
                    IRdwsv(ikl), IRu_SV(ikl), &
                    HSv_sv(ikl) + HSs_sv(ikl), &
                    HLv_sv(ikl) + HLs_sv(ikl), TaT_SV(ikl), &
                    dsn_SV(ikl) * 3.6e3, drr_SV(ikl) * 3.6e3, &
                    SoSosv(ikl) * sol_SV(ikl), &
                    IRv_sv(ikl) * 0.5, &
                    HSv_sv(ikl), HLv_sv(ikl), TvegSV(ikl), &
                    SoCasv(ikl) * sol_SV(ikl), &
                    HSs_sv(ikl), HLs_sv(ikl), TsisSV(ikl, isnoSV(ikl))
            5001        format(&
                    '        |Net Solar| IR Down | IR Up   | HS/Dwn=+|', &
                    ' HL/Dwn=+| Temper. |         |  Snow  |  Rain  |', &
                    /, '        | [W/m2]  | [W/m2]  | [W/m2]  | [W/m2]  |', &
                    ' [W/m2]  | [K]     |         | [mm/h] | [mm/h] |', &
                    /, ' -------+', 7('---------+'), 2('--------+'), &
                    /, ' SISVAT |', f8.1, ' |', f8.1, ' |', f8.1, ' |', f8.1, ' |', &
                    f8.1, ' |A', f7.2, ' |', 8x, ' |', 2(f7.2, ' |'), &
                    /, ' Canopy |', f8.1, ' |', 8x, ' |', f8.1, ' |', f8.1, ' |', &
                    f8.1, ' |', f8.2, ' |', 8x, ' |', 2(7x, ' |') &
                    /, ' Soil   |', f8.1, ' |', 8x, ' |', 8x, ' |', f8.1, ' |', &
                    f8.1, ' |', f8.2, ' |', 8x, ' |', 2(7x, ' |'))
#endif

#ifdef e1
            ! +--Energy Budget
            ! +  -------------
            ! Up Surf. IR
            Enrsvd(ikl) = Enrsvd(ikl) &
                    ! Offset
                    + IRs_SV(ikl) &
                    ! Net   Solar
                    + ((SoSosv(ikl) &
                            + SoCasv(ikl)) * sol_SV(ikl) &
                            ! Downward IR
                            + IRdwsv(ikl) &
                            ! Upward   IR
                            + IRupsv(ikl) &
                            ! Sensible
                            + HSv_sv(ikl) + HSs_sv(ikl) &
                            ! Latent
                            + HLv_sv(ikl) + HLs_sv(ikl))
            write(noUNIT, 5002) Enrsvd(ikl), &
                    ETSo_0(ikl), ETSo_d(ikl), &
                    ETSo_0(ikl) + ETSo_d(ikl), ETSo_1(ikl), &
                    EqSn_0(ikl) / dt__SV, &
                    EqSn_d(ikl) / dt__SV, &
                    (EqSn_1(ikl) - EqSn_0(ikl) - EqSn_d(ikl)) / dt__SV, &
                    EqSn_1(ikl) / dt__SV
            5002        format(&
                    ' -----------------+-------------------+', &
                    '-----------------+-+-----------------+', &
                    '-------------------+', &
                    /, ' SOIL/SNOW/VEGET. |                   |', &
                    ' Power,  Forcing |                   |', &
                    '                   |', &
                    /, '                  |', 11x, '        |', &
                    f9.2, ' [W/m2] |', 11x, '        |', &
                    11x, '        |', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+', &
                    !        ETSo_0
                    /, ' SOIL/SNOW  (TSo) | Energy/dt, Time 0 |', &
                    ! ETSo_d/ETSo_0+d
                    ' Power,  Forcing |   Sum Tim.0+Forc. |', &
                    ! ETSo_1
                    ' Energy/dt, Time 1 |', &
                    !        ETSo_0
                    /, '                  |', f11.2, ' [W/m2] |', &
                    ! ETSo_d/ETSo_0+d
                    f9.2, ' [W/m2] |', f11.2, ' [W/m2] |', &
                    ! ETSo_1
                    f11.2, ' [W/m2] |', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+', &
                    ! EqSn_0/dt
                    /, '      SNOW  (qSn) | Energy/dt, Time 0 |', &
                    ! EqSn_d/dt, 1-0-d
                    ' Power,  Excess  |   D(Tim.1-0-Forc.)|', &
                    ! EqSn_1/dt
                    ' Energy/dt, Time 1 |', &
                    ! EqSn_0/dt
                    /, '                  |', f12.2, '[W/m2] |', &
                    ! EqSn_d/dt, 1-0-d
                    f9.2, ' [W/m2] |', f11.2, ' [W/m2] |', &
                    ! EqSn_1/dt
                    f12.2, '[W/m2] | ', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+')
            
            EnsBal = ETSo_1(ikl) - (ETSo_0(ikl) + Enrsvd(ikl))
            EnvBal = Enrsvd(ikl) - ETVg_d(ikl)

#ifdef e2
            if((abs(EnsBal) > 5.e-1) .OR. (lwriSV(ikl) == 2)) then
#else
                if(abs(EnsBal) .gt. 5.e-1) then
#endif
                write(6, 6001) daHost, i___SV(lwriSV(ikl)), &
                        j___SV(lwriSV(ikl)), &
                        n___SV(lwriSV(ikl)), &
                        ETSo_1(ikl), ETSo_0(ikl), ETSo_d(ikl), &
                        ETSo_1(ikl) - ETSo_0(ikl) - ETSo_d(ikl), &
                        Enrsvd(ikl), ETVg_d(ikl), ETSo_d(ikl), &
                        Enrsvd(ikl) - ETVg_d(ikl) - ETSo_d(ikl)
                6001                format(a18, 3i4, ' (EB1', f15.6, &
                        ')  - [(EB0           ', f15.6, ')', &
                        /, 55x, '+(ATM->Snow/Soil', f15.6, ')] ', &
                        '= EBAL', f15.6, ' [W/m2]', &
                        /, 55x, ' (ATM->SISVAT', f18.6, &
                        /, 55x, '- Veg. ImBal.', f18.6, ')  ', &
                        /, 55x, '- ATM->SnoSol', f18.6, ')  ', &
                        '= ????', f15.6, ' [W/m2]')
                noEBal = noEBal + 1
#ifdef e2
                noEBal = noEBal - 1
#endif
                if(noEBal >= 10) stop 'TOO MUCH ENERGY IMBALANCES'
            endif
#endif
            
            ! +--Snow   Budget [mm w.e.]
            ! +  -----------------------
#ifdef m1
            write(noUNIT, 5010) &
                    SIWm_0(ikl), SIWa_i(ikl) - SIWa_f(ikl) &
                    , SIWm_0(ikl) + SIWa_i(ikl) - SIWa_f(ikl) &
                    + SIWe_i(ikl) - SIWe_f(ikl) &
                    + SIsubl(ikl) &
                    - SImelt(ikl) &
                    - SIrnof(ikl) &
                    + SIvAcr(ikl) &
                    , SIWm_1(ikl), SIWe_i(ikl) - SIWe_f(ikl) &
                    , SIsubl(ikl) &
                    , -SImelt(ikl) &
                    , -SIrnof(ikl) &
                    , SIvAcr(ikl)
            5010            format(' SNOW             |   Snow,   Time 0  |', &
                    ' Snow,   Forcing |           Sum     |', &
                    '   Snow,   Time 1  |', &
                    /, '                  |', f13.3, ' [mm] |', &
                    ' A', f9.3, ' [mm] |', f13.3, ' [mm] |', &
                    f13.3, ' [mm] |', &
                    /, '                  |', 13x, '      |', &
                    ' E', f9.3, ' [mm] |', 13x, '      |', &
                    13x, '      |', &
                    /, '                  |', 13x, '      |', &
                    ' S', f9.3, ' [mm] |', 13x, '      |', &
                    13x, '      |', &
                    /, '                  |', 13x, '      |', &
                    '(M', f9.3, ' [mm])|  (included in A)  |', &
                    13x, '      |', &
                    /, '                  |', 13x, '      |', &
                    ' R', f9.3, ' [mm] |', 13x, '      |', &
                    13x, '      |', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+')
            SnoBal = SIWm_1(ikl) - (SIWm_0(ikl) &
                    + SIWa_i(ikl) - SIWa_f(ikl) &
                    + SIWe_i(ikl) - SIWe_f(ikl)) &
                    - SIsubl(ikl) &
                    + SIrnof(ikl)
            
            SnoBal = SnoBal - SIvAcr(ikl)
            if(abs(SnoBal) > epsi) then
                write(6, 6010) daHost, i___SV(lwriSV(ikl)), &
                        j___SV(lwriSV(ikl)), &
                        n___SV(lwriSV(ikl)), &
                        SIWm_1(ikl), SIWm_0(ikl), &
                        SIWa_i(ikl), SIWa_f(ikl), &
                        SIWe_i(ikl), SIWe_f(ikl), &
                        SIsubl(ikl), SImelt(ikl), &
                        SIrnof(ikl), SIvAcr(ikl), &
                        SnoBal
                6010                format(a18, 3i4, ' (MB1', f12.6, &
                        ') - [(MB0        ', f12.6, 15x, ')', &
                        /, 51x, '+(ATM Forcing', f12.6, ' - ', f12.6, ')', &
                        /, 51x, '+(BLS Forcing', f12.6, ' - ', f12.6, ')', &
                        /, 51x, '-(Depo/Sublim', f12.6, 15x, ')', &
                        /, 51x, ' !Melting    ', f12.6, '  included in A!', &
                        /, 51x, '+(Run  OFF   ', f12.6, 15x, ')', &
                        /, 51x, '-(Sea-Ice Acr', f12.6, 15x, ')', &
                        /, 29x, '= *BAL', f12.6, ' [mm w.e.]')
                noSBal = noSBal + 1
                if(noSBal >= 10) stop 'TOO MUCH SNOW MASS IMBALANCE'
            endif
#endif
            
            ! +--Water  Budget
            ! +  -------------
#ifdef m0
            ! Canopy Water Cont. + Soil   Water Cont.
            Watsv0(ikl) = Watsv0(ikl) &
                    + Wats_0(ikl)
            ! Canopy Forcing + Soil   Forcing
            Watsvd(ikl) = Watsvd(ikl) &
                    + Wats_d(ikl)
            write(noUNIT, 5003) &
                    Wats_0(ikl), Wats_d(ikl), &
                    Wats_0(ikl) + Wats_d(ikl), Wats_1(ikl), &
                    Watsv0(ikl), Watsvd(ikl), &
                    Watsv0(ikl) + Watsvd(ikl), Wats_1(ikl) &
                    + rrCaSV(ikl)
            5003            format(' SOIL/SNOW  (qSo) |   Water,  Time 0  |', &
                    ' Water,  Forcing |           Sum     |', &
                    '   Water,  Time 1  |', &
                    /, '                  |', f13.3, ' [mm] |', &
                    f11.3, ' [mm] |', f13.3, ' [mm] |', &
                    f13.3, ' [mm] |', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+', &
                    /, ' SOIL/SNOW/VEGET. |   Water,  Time 0  |', &
                    ' Water,  Forcing |           Sum     |', &
                    '   Water,  Time 1  |', &
                    /, '                  |', f13.3, ' [mm] |', &
                    f11.3, ' [mm] |', f13.3, ' [mm] |', &
                    f13.3, ' [mm] |', &
                    /, ' -----------------+-------------------+', &
                    '-----------------+-------------------+', &
                    '-------------------+')
            
            WatBal = Wats_1(ikl) + rrCaSV(ikl) &
                    - (Watsv0(ikl) + Watsvd(ikl))
            if(abs(WatBal) > epsi) then
                write(6, 6002) daHost, i___SV(lwriSV(ikl)), &
                        j___SV(lwriSV(ikl)), &
                        n___SV(lwriSV(ikl)), &
                        Wats_1(ikl), rrCaSV(ikl), &
                        Watsv0(ikl), Watsvd(ikl), WatBal, &
                        Wats_1(ikl), &
                        Wats_0(ikl), Wats_d(ikl), &
                        Wats_1(ikl) - Wats_0(ikl) - Wats_d(ikl)
                6002                format(30x, ' NEW Soil Water', 3x, ' Canopy   Water', 3x, &
                        ' OLD SVAT Water', 4x, ' FRC SVAT Water', &
                        /, a18, 3i4, f15.6, ' + ', f15.6, ' - ', f15.6, &
                        ' -  ', f15.6, '    ', 15x, '    ', &
                        /, 31x, '= ', f12.6, ' [mm] (Water Balance)', &
                        /, 30x, ' NEW Soil Water', 3x, '               ', 3x, &
                        ' OLD Soil Water', 4x, ' FRC Soil Water', &
                        /, 30x, f15.6, '   ', 15x, ' - ', f15.6, &
                        ' -  ', f15.6, '    ', 15x, '    ', &
                        /, 31x, '= ', f12.6, ' [mm] (3 terms SUM)')
                noWBal = noWBal + 1
                if(noWBal >= 10) stop 'TOO MUCH WATER  IMBALANCES'
            endif
#endif
            
            ! +--Water/Temperature Profiles
            ! +  --------------------------
#ifdef E0
            write(noUNIT, 5004)
            5004            format(' -----+--------+--+-----+--------+----+---+', &
                    '--------+----+---+--------+------+-+--------+--------+', &
                    /, '    n |     z  |     dz |     ro |    eta |', &
                    '     T  |     G1 |     G2 | Extinc |        | HISTORY|', &
                    /, '      |    [m] |    [m] | [kg/m3]| [m3/m3]|', &
                    '    [K] |    [-] |    [-] |    [-] |        |   [-]  |', &
                    /, ' -----+--------+--------+--------+--------+', &
                    '--------+--------+--------+--------+--------+--------+')
            write(noUNIT, 5005) rusnSV(ikl), albisv(ikl)
            5005            format('      |        |        |        |W', f6.3, ' |', &
                    '        |        |        |A', f6.3, ' |        |        |')
            write(noUNIT, 5015) &
                    (isn, zzsnsv(ikl, isn), dzsnSV(ikl, isn), &
                            ro__SV(ikl, isn), eta_SV(ikl, isn), &
                            TsisSV(ikl, isn), &
                            G1snSV(ikl, isn), G2snSV(ikl, isn), &
                            sEX_sv(ikl, isn), istoSV(ikl, isn), &
                            isn = isnoSV(ikl), 1, -1)
            5015            format((i5, ' |', 2(f7.3, ' |'), f7.1, ' |', &
                    f7.3, ' |', f7.2, ' |', 2(f7.1, ' |'), f7.3, ' |', &
                    7x, ' |', i5, '   |'))
            write(noUNIT, 5006)
            5006            format(' -----+--------+--------+--------+--------+', &
                    '--------+--------+--------+--------+--------+--------+')
            write(noUNIT, 5007) TBr_sv(ikl), &
                    TvegSV(ikl), rrCaSV(ikl) * 1.e3, &
                    EvT_sv(ikl) * 86.4e3
            5007            format(' Brgh |', 4(8x, '|'), f7.2, ' | [micm] |', 4(8x, '|'), &
                    /, ' VEGE |', 4(8x, '|'), 2(f7.2, ' |'), 2(8x, '|'), &
                    f7.3, ' |', 8x, '|')
            write(noUNIT, 5014)
            5014            format(' -----+--------+--------+--------+--------+', &
                    '--------+--------+--------+--------+--------+--------+', &
                    /, '    n |        |     dz |        |    eta |', &
                    '     T  |        |        |        | Root W.| W.Flow |', &
                    /, '      |        |    [m] |        | [m3/m3]|', &
                    '    [K] |        |        |        | [mm/d] | [mm/h] |', &
                    /, ' -----+--------+--------+--------+--------+', &
                    '--------+--------+--------+--------+--------+--------+')
            do isl = 0, -nsol, -1
                write(noUNIT, 5008) isl, LSdzsv(ikl) * dz_dSV(isl), &
                        eta_SV(ikl, isl), &
                        TsisSV(ikl, isl), &
                        86.4e3 * Rootsv(ikl, isl), &
                        3.6e3 * Khydsv(ikl, isl)
            enddo
            5008            format((i5, ' |', 7x, ' |', f7.3, ' |', 7x, ' |', &
                    f7.3, ' |', f7.2, ' |', 2(7x, ' |'), 7x, ' |', &
                    f7.3, ' |', f7.2, ' |'))
            write(noUNIT, 5006)
            write(noUNIT, 5009) RnofSV(ikl) * 3.6e3
            5009            format('      |', 9(8x, '|'), f7.3, ' |')
            write(noUNIT, 5006)
        endif
    enddo
#endif
    ! +..END  .main.
    return
endsubroutine SISVAT
