subroutine qsat3d
    ! +------------------------------------------------------------------------+
    ! | MAR PHYSICS                                         Mc 30-05-2007  MAR |
    ! |   subroutine qsat3d computes the Saturation Specific Humidity  (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT :   TairSL: Surface Air Temperature                        (K) |
    ! |   ^^^^^^^   TairDY:         Air Temperature                        (K) |
    ! |              pstDY: Model Pressure Thickness                     (kPa) |
    ! |                                                                        |
    ! |   OUTPUT :  qvswDY: Saturation Specific Humidity  over Water   (kg/kg) |
    ! |   ^^^^^^^   qvsiDY: Saturation Specific Humidity  over Ice     (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marphy
    use mardim
    use margrd, only: sigma
    use mar_dy, only: pstDY, ptopDY, tairDY, qvswDY, qvsiDY
    use mar_sl, only: TairSL
    use mar_wk, only: WKxyz5, WKxyz6, WKxyz7, WKxyz8

    implicit none

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    real WatIce, ExpWat, ExpWa2, ExpIce

    ! +--DATA
    ! +  ====
    data WatIce/273.16e0/
    data ExpWat/5.138e0/
    data ExpWa2/6827.e0/
    data ExpIce/6150.e0/

    ! +--Temperature (K) and Pressure (hPa)
    ! +  ==================================

    !$OMP PARALLEL do default(shared) private(i,j,k)
    do j = 1, my
        do k = 1, mz
            ! do j=1,my
            do i = 1, mx
                WKxyz5(i, j, k) = tairDY(i, j, k)
                WKxyz6(i, j, k) = (pstDY(i, j) * sigma(k) + ptopDY) * 10.0d0
            enddo
            ! end do
        enddo

        ! do j=1,my
        do i = 1, mx
            WKxyz5(i, j, mzz) = TairSL(i, j)
            WKxyz6(i, j, mzz) = (pstDY(i, j) + ptopDY) * 10.0d0
        enddo
        ! end do

        ! +--Saturation Vapor Pressure over Ice
        ! +  ==================================
        do k = 1, mzz
            ! do j=1,my
            do i = 1, mx
                WKxyz7(i, j, k) = 6.1070d0 * exp(ExpIce * (unun / WatIce - unun / WKxyz5(i, j, k)))
                ! +...    Dudhia (1989) JAS, (B1) and (B2) p.3103

                WKxyz8(i, j, k) = .622d0 * WKxyz7(i, j, k) / (WKxyz6(i, j, k) - .378d0 * WKxyz7(i, j, k))

                ! +--Saturation Vapor Pressure over Water
                ! +  ====================================
                WKxyz7(i, j, k) = 6.1078d0 * exp(ExpWat * log(WatIce / WKxyz5(i, j, k))) &
                                  * exp(ExpWa2 * (unun / WatIce - unun / WKxyz5(i, j, k)))
                ! +...    Dudhia (1989) JAS, (B1) and (B2) p.3103
                ! +       See also Pielke (1984), p.234 and Stull (1988), p.276

                qvswDY(i, j, k) = max(eps9, .622d0 * WKxyz7(i, j, k) &
                                      / (WKxyz6(i, j, k) - .378d0 * WKxyz7(i, j, k)))
                ! +...    Saturation Vapor Specific Concentration over Water
                ! +       (even for temperatures less than freezing point)

                ! +--Water Phase Discriminator
                ! +  =========================
                WKxyz7(i, j, k) = max(zero, sign(unun, WKxyz5(i, j, k) - WatIce))
                ! +...    WKxyz7(i,j,k) =     1    if   Tair     >    273.16
                ! +                           0    if   Tair     <  273.16

                ! +--Saturation Vapor Specific Concentration over Ice
                ! +  ================================================
                qvsiDY(i, j, k) = max(eps9, qvswDY(i, j, k) * WKxyz7(i, j, k) &
                                      + WKxyz8(i, j, k) * (unun - WKxyz7(i, j, k)))

                ! +--Work Area Reset
                ! +  ===============
                WKxyz5(i, j, k) = 0.0
                WKxyz6(i, j, k) = 0.0
                WKxyz7(i, j, k) = 0.0
                WKxyz8(i, j, k) = 0.0
            enddo
        enddo
    enddo
    !$OMP END PARALLEL DO

    return
end
