subroutine qsat2D(tair2D, pst2D, tsrf2D, qvsi2D, qvsw2D)

    ! +------------------------------------------------------------------------+
    ! | MAR PHYSICS                                         Mc 30-05-2007  MAR |
    ! |   subroutine qsat2D computes the Saturation Specific Humidity  (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT :   tair2D: Air Temperature                                (K) |
    ! |              pst2D: Model Pressure Thickness                     (kPa) |
    ! |             tsrf2D: Surface Air Temperature                        (K) |
    ! |                                                                        |
    ! |   OUTPUT :  qvsi2D: Saturation Specific Humidity over Ice      (kg/kg) |
    ! |   ^^^^^^^   qvsw2D: Saturation Specific Humidity over Water    (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marphy
    use mardim
    use margrd
    use mar_dy

    implicit none

    !  Input - Output
    !  ==============
    ! in
    ! --
    real, intent(in) :: tair2D(klon, klev)
    real, intent(in) :: pst2D(klon)
    real, intent(in) :: tsrf2D(klon)
    ! out
    ! ---
    real, intent(out) :: qvsi2D(klon, klev + 1)
    real, intent(out) :: qvsw2D(klon, klev + 1)

    ! +--Local Variables
    ! +  ===============

    integer i, j, k, m
    integer il, klq

    real :: W2xyz5(klon, klev + 1)
    real :: W2xyz6(klon, klev + 1)
    real :: W2xyz7(klon, klev + 1)
    real :: W2xyz8(klon, klev + 1)

    real WatIce, ExpWat, ExpWa2, ExpIce

    ! +--DATA
    ! +  ====
    data WatIce/273.16e0/
    data ExpWat/5.138e0/
    data ExpWa2/6827.e0/
    data ExpIce/6150.e0/

    ! +--Work Area Init
    ! +  ===============
    do klq = 1, klev + 1
        do il = 1, klon
            W2xyz5(il, klq) = 0.0
            W2xyz6(il, klq) = 0.0
            W2xyz7(il, klq) = 0.0
            W2xyz8(il, klq) = 0.0
        enddo
    enddo

    ! +--Temperature (K) and Pressure (hPa)
    ! +  ==================================

    do klq = 1, klev
        do il = 1, klon
            W2xyz5(il, klq) = tair2D(il, klq)
            W2xyz6(il, klq) = (pst2D(il) * sigma(klq) + ptopDY) * 10.0d0
        enddo
    enddo

    do il = 1, klon
        W2xyz5(il, klev + 1) = tsrf2D(il)
        W2xyz6(il, klev + 1) = (pst2D(il) + ptopDY) * 10.0d0
    enddo

    ! +--Saturation Vapor Pressure over Ice
    ! +  ==================================

    do klq = 1, klev + 1
        do il = 1, klon
            ! +...    Dudhia (1989) JAS, (B1) and (B2) p.3103
            W2xyz7(il, klq) = 6.1070d0 * exp(ExpIce * (unun / WatIce - unun / W2xyz5(il, klq)))

            W2xyz8(il, klq) = .622d0 * W2xyz7(il, klq) &
                              / (W2xyz6(il, klq) - .378d0 * W2xyz7(il, klq))

            ! +--Saturation Vapor Pressure over Water
            ! +  ====================================
            ! +...    Dudhia (1989) JAS, (B1) and (B2) p.3103
            ! +       See also Pielke (1984), p.234 and Stull (1988), p.276
            W2xyz7(il, klq) = 6.1078d0 * exp(ExpWat * log(WatIce / W2xyz5(il, klq))) &
                              * exp(ExpWa2 * (unun / WatIce - unun / W2xyz5(il, klq)))
            ! +...    Saturation Vapor Specific Concentration over Water
            ! +       (even for temperatures less than freezing point)
            qvsw2D(il, klq) = max(eps9, .622d0 * W2xyz7(il, klq) &
                                  / (W2xyz6(il, klq) - .378d0 * W2xyz7(il, klq)))

            ! +--Water Phase Discriminator
            ! +  =========================
            ! +...    W2xyz7(il,klq) =     1    if   Tair     >    273.16
            ! +                            0    if   Tair     <  273.16
            W2xyz7(il, klq) = max(zero, sign(unun, W2xyz5(il, klq) - WatIce))

            ! +--Saturation Vapor Specific Concentration over Ice
            ! +  ================================================
            ! +
            qvsi2D(il, klq) = max(eps9, qvsw2D(il, klq) * W2xyz7(il, klq) &
                                  + W2xyz8(il, klq) * (unun - W2xyz7(il, klq)))

            ! +--Work Area Reset
            ! +  ===============
            W2xyz5(il, klq) = 0.0
            W2xyz6(il, klq) = 0.0
            W2xyz7(il, klq) = 0.0
            W2xyz8(il, klq) = 0.0
        enddo
    enddo

    return
endsubroutine qsat2D
