function qsat0D(ttq, ss, pstar, pt, lsf)
    ! +------------------------------------------------------------------------+
    ! | MAR PHYSICS                                         Mc 30-05-2007  MAR |
    ! |   Function qsat0D computes the Saturation Specific Humidity    (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT :   ttq            : Air Temperature                       (K) |
    ! |   ^^^^^^^   pstar * ss + pt: Pressure of sigma level ss          (kPa) |
    ! |                                                                        |
    ! |   OUTPUT :  esat: Saturation Vapor Pressure                      (hPa) |
    ! |   ^^^^^^^   qsat0D: Saturation Specific Humidity               (kg/kg) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marphy

    implicit none

    real qsat0D, ttq, ss, pstar, pt
    integer lsf

    ! +--Local  Variables
    ! +  ================

    real pr, esat
    real, parameter :: tfreeze = 273.16d0

    pr = 10.d0 * (pstar * ss + pt)
    ! +...pr   : pressure (hPa)

    if(ttq >= tfreeze .or. lsf == 0) then
        esat = 6.1078d0 * exp(5.138d0 * log(tfreeze / ttq)) &
               * exp(6827.d0 * (unun / tfreeze - unun / ttq))
        ! +...  esat : saturated vapor pressure with respect to water
        ! +***  Dudhia (1989) JAS, (B1) and (B2) p.3103
        ! +     See also Pielke (1984), p.234 and Stull (1988), p.276
    else
        esat = 6.107d0 * exp(6150.d0 * (unun / tfreeze - unun / ttq))
        ! +...  esat : saturated vapor pressure with respect to ice
        ! +***  Dudhia (1989) JAS, 1989, (B1) and (B2) p.3103
    endif

    ! +  ******
    qsat0D = max(eps9, .622d0 * esat / (pr - .378d0 * esat))
    ! +  ******

    return
end
