#include "MAR_pp.def"
subroutine PHYrad_ECRAD_in(dST_UA)

    ! +------------------------------------------------------------------------+
    ! | MAR PHYSICS                                     XF,JFG 25-07-2023  MAR |
    ! |                                                                        |
    ! |   The subroutine PHYrad_ECRAD interfaces MAR with the new ECMWF        |
    ! |   solar/infrared radiative transfer scheme ecRad. See also the         |
    ! |   subroutines PHYrad_ecRad_init and PHYrad_ecRad in the ecRad          |
    ! |   subdirectory for more details on how MAR and ecRad are interfaced.   |
    ! |                                                                        |
    ! |   ecRad code by Hogan et al., cf. https://github.com/ecmwf-ifs/ecrad   |
    ! |                                                                        |
    ! |   The code of this subroutine was initially an adaptation of the code  |
    ! |   of the phyrad_cep_mp subroutine from MAR v3.12 to ecRad by J.-F.     |
    ! |   Grailet (late 2022). It was subsequently adapted to Fortran 90       |
    ! |   based on the new phyrad_cep_mp from MAR v3.13 (January 2023). Later  |
    ! |   modifications (MAR v3.13, v3.14 and onwards) added more control of   | 
    ! |   ecRad for MAR users and improved the preparation of ecRad inputs.    |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_dy
    use mar_hy
    use mar_ra
    use mar_sl
    use mar_wk
    use mar_io
#if(AR)
    use mar_tc
#endif
    ! +--Interface  Variables
    ! +  ====================
    use mar_ecrad

    implicit none

    integer i, j, k, m
    real dST_UA                    ! Distance Soleil-Terre [UA]
    ! real RAcldE(mx, my, mz)        ! Cloud Emissivity       [-]
    ! real htngIR(mx, my, mz)        ! IR      Heating      [K/s]
    ! real htngSO(mx, my, mz)        ! Solar   Heating      [K/s]

    ! +--Inputs to ecRad
    ! +  ---------------
    
    integer yyyy, mm, dd ! Year, month, day
    integer i_hhss       ! Number of seconds in the day
    
    ! Variables computed/adjusted for the needs of ecRad
    real, dimension(:, :, :), allocatable :: pa_RAD ! Pressure levels (layer)
    real, dimension(:, :, :), allocatable :: pahRAD ! Pressure levels (interf.)
    real, dimension(:, :),    allocatable :: lsmRAD ! Land/sea mask for ecRad
    real, dimension(:, :),    allocatable :: larRAD ! Latitude (radian)
    real, dimension(:, :),    allocatable :: lorRAD ! Longitude (radian)
    real, dimension(:, :, :), allocatable :: qv_RAD ! Vapor Concentr. [kg/kg]
    real, dimension(:, :, :), allocatable :: sw_RAD ! Satur. % water [kg/kg]

    real, dimension(:, :, :), allocatable :: rqiRAD ! qiHY
    real, dimension(:, :, :), allocatable :: rqsRAD ! qsHY
    
    ! Additional variables
    real, dimension(:,:), allocatable :: cldMAX  ! Cloud max fraction
    integer n, l
    real ww, nn, ss

    ! +--Outputs from ecRad
    ! +  ------------------
    
    ! J.-F. Grailet remark (05/09/2023): these arrays are allocated and set within the 
    ! PHYrad_ecRad subroutine but deallocated here (cf. end).
    
    real, dimension(:, :, :), allocatable :: FIRn_c ! CLEAR-SKY LW NET FLUXES
    real, dimension(:, :, :), allocatable :: FIRn_t ! TOTAL LW NET FLUXES
    real, dimension(:, :, :), allocatable :: FSOn_c ! CLEAR-SKY SW NET FLUXES
    real, dimension(:, :, :), allocatable :: FSOn_t ! TOTAL SW NET FLUXES
    
    real, dimension(:, :, :), allocatable :: rPLWdv ! Partial deriv. of LW
    real, dimension(:, :, :), allocatable :: rPSWdf
    real, dimension(:, :, :), allocatable :: rPSWdr
    real, dimension(:, :, :), allocatable :: rFLopt ! Additional outputs
    ! rFLopt includes TOTAL-SKY SURFACE SW DOWNWARD FLUX (formerly FSOs_t)

    ! +--Additional input variables w.r.t. surface albedo
    ! +  ------------------------------------------------

    real bsegal, albmax, albx, dalb, albu  
    real czeMAX, czrx
    real siceOK, ciceOK, zangOK, sign_T, ColdOK
    real sign_S, snowOK

    ! +--Additional output variables
    ! +  ---------------------------

    integer RADerr(mx, my, 3) ! Advertises up to 3 different errors
    real tmp
    
    ! Variables used to print the time
    character(len=10) dateStr
    character(len=8) timeStr
    character(len=22) dateTimeStr
    
    ! +--Additional variables for "MAR override"
    ! +  ---------------------------------------
    ! +  Added for MAR v3.14 to provide some control of the ecRad configuration through a module
    ! +  similar to marmagic rather than through a namelist configuration file.
    
    logical, dimension(6) :: override_logical
    integer, dimension(2) :: override_int
    real, dimension(8) :: override_real

    ! +--DATA
    ! +  ====

    data bsegal /2.00e0/
    data albmax /0.99e0/
    data czeMAX /0.173648178/ ! czeMAX: 80.deg (Segal et al., 1991 JAS)

    ! J.-F. Grailet remark (03/05/2023, updated 05/09/2023): including the full definition of the 
    ! PHYrad_ecRad_init subroutine is necessary to be able to use optional parameters for MAR 
    ! override. The location of the header file was chosen so it can also be used for the 
    ! stand-alone ecRad test program (i.e., outside of MAR).

#include "./ecRad/include/PHYrad_ecRad_init.intfb.h"
#include "./ecRad/include/PHYrad_ecRad.intfb.h"

    ! +--INITIALIZATION
    ! +  ==============
    
    ! +--Should the radiative scheme be called now ?
    ! +  ===========================================

    if (mod(iterun, jtRadi) /= 0) then
        return
    end if

    ! +--Time & Insolation (top of the atmosphere)
    ! +  =========================================

    yyyy = iyrrGE
    mm = mmarGE
    dd = jdarGE
    
    i_hhss = jhurGE * 3600 + minuGE * 60 + jsecGE
    
    ! Prepares the time in string format
    write(dateStr, '(i0.2,a,i0.2,a,i4)') dd, '/' , mm, '/', yyyy
    write(timeStr, '(i0.2,a,i0.2,a,i0.2)') jhurGE, 'h', minuGE, 'm', jsecGE
    write(dateTimeStr, '(a,a,a)') dateStr, ' at ', timeStr
    
    ! +--Allocation of variables adjusted for ecRad
    ! +  ==========================================

    allocate(pa_RAD(mx, my, mz))
    allocate(pahRAD(mx, my, mz + 1))
    allocate(lsmRAD(mx, my))
    allocate(larRAD(mx, my))
    allocate(lorRAD(mx, my))
    allocate(qv_RAD(mx, my, mz))
    allocate(sw_RAD(mx, my, mz))
    allocate(cldMAX(mx, my))
    allocate(rqiRAD(mx, my, mz))
    allocate(rqsRAD(mx, my, mz))

    ! +--Zenith Angle Correction of Snow Albedo
    ! +  ======================================
    
    if (.not.VSISVAT) then ! CTR

        do j=jp11, my1; do i=ip11, mx1

            siceOK = 1 - min(iabs(isolSL(i, j) - 2), iun)
            ciceOK = 1 - min(iabs(isolSL(i, j) - 3), iun)

            zangOK = max(siceOK, ciceOK)

            sign_T = sign(unun, TfSnow - tairSL(i, j))
            ColdOK = max(zero, sign_T)
            zangOK = max(zangOK, ColdOK)

            sign_S = zero
            snowOK = max(zero, sign_S)
            zangOK = max(zangOK, snowOK)

#if(CP)
            zangOK = 0.0e+0
#endif
            ! Snow and/or ice covered surface 
            albx = alb0SL(i, j)
            czrx = max(czeMAX, czenGE(i, j))
            dalb = 0.32e0 * ((bsegal+unun) / (unun + 2.e0 * bsegal * czrx) &
             &               - unun) / bsegal
            dalb = max(dalb, zero)
            albx = dalb + alb0SL(i, j)
            albx = min(albx, albmax)
            ! Influence of Sun Zenith Angle 
            ! (Segal et al., 1991 JAS 48, p.1025)

            ! Underlying Surface Albedo
            albu = alb0SL(i, j)

            ! Actual albedo
            albeSL(i, j) = zangOK * albx + (1 - zangOK) * albu

        end do; end do

    end if ! CTR

    ! +--Effective Radiating Surface Temperature
    ! +  =======================================

    write(6,'(a,a)') ' Call of PHYrad_ecRad: ', dateTimeStr
    
    ! Adjusts some MAR variables for the needs of ecRad
    do j=1, my; do i=1, mx
      
        cld_SL(i, j) = 0.
        cldMAX(i, j) = 0.

        ! Pressure levels (kPa -> Pa)
        do k=1, mz
            pa_RAD(i, j, k) = (pstDY(i, j) * sigma(k) + ptopDY) * 1.e3
            pahRAD(i, j, k) = (pstDY(i, j) * sigmid(k) + ptopDY) * 1.e3
        end do
        pahRAD(i, j, mz + 1) = (pstDY(i, j) * sigmid(mz + 1) + ptopDY) * 1.e3

        ! Land/sea mask
        lsmRAD(i,j) = 1 - maskSL(i,j)
          
        ! Geographical coordinates
        larRad(i, j) = (GElatr(i, j) * 180.) / pi
        lorRAD(i, j) = (GElonh(i, j) * hourad * 180.) / pi
        ! Old code (no longer what ecRad expects)
        ! larRAD(i, j) = sign(1., GElatr(i, j)) * min(89.9 * degrad, abs(GElatr(i, j)))
        ! lorRAD(i, j) = GElonh(i, j) * hourad
        ! lorRAD(i, j) = lorRAD(i, j) - pi * 2. * min(sign(1., lorRAD(i, j)), 0.)

        ! Water species
        do k=1, mz
            qv_RAD(i, j, k) = max(1.e-6, qvDY(i, j, k)) ! Vapor
            sw_RAD(i, j, k) = min(qvswDY(i, j, k), 0.03) ! Saturation % W
            rqiRAD(i, j, k) = qiHY(i, j, k)
            rqsRAD(i, j, k) = qsHY(i, j, k)
#if(GR)
            rqiRAD(i, j, k) = qiHY(i, j, k)+0.90*qsHY(i, j, k)
            rqsRAD(i, j, k) =               0.10*qsHY(i, j, k)
#endif
#if(AC)
            rqiRAD(i, j, k) = qiHY(i, j, k)+0.75*qsHY(i, j, k)
            rqsRAD(i, j, k) =               0.25*qsHY(i, j, k)
#endif
        end do

    end do; end do
    
    if (.not. ecRad_initialized) then

      ! Remarks by J.-F. Grailet for MAR v3.14 (added 02/05/2023, updated 05/09/2023):
      ! 1) ecRad_initialized is a variable from the mar_ecrad module used to set ecRad and the 
      !    MAR code/data interfacing MAR with ecRad at the first call of this subroutine.
      ! 2) CMIP_scenario is from the mar_ecrad module too and handled in an updated phymar.f90 
      !    (i.e. same as v3.13 with scenario names adjusted for the new ECMWF time series).
      ! 3) Parameters of MAR for ecRad, provided via the "MAR override" feature, are all declared 
      !    and set in the mar_ecrad module, following the example of the marmagic module. The 
      !    instructions just before calling PHYrad_ecRad_init put together these parameters as 
      !    arrays to more easily send them.
      ! 4) Dedicated namelist file for ecRad is necessarily expected in the directory where MAR 
      !    has been placed, with the name "ecRad_config.nam" (default configuration with the 
      !    "MAR override" parameters are used otherwise).
      
      override_logical(1) = use_ecckd_high_res
      override_logical(2) = do_longwave_scattering
      override_logical(3) = output_extra_fluxes
      override_logical(4) = output_clim_interpolated
      override_logical(5) = output_clim_scaled
      override_logical(6) = output_raw_spectral_fluxes
      
      override_int(1) = cloud_parametrization
      override_int(2) = cloud_decorrelation_length
      
      override_real(1) = solar_constant_scaling
      override_real(2) = cloud_water_fractional_std
      override_real(3) = cloud_fraction_ecmwf_gamma
      override_real(4) = water_species_scaling_vapor
      override_real(5) = water_species_scaling_ice
      override_real(6) = water_species_scaling_liquid
      override_real(7) = water_species_scaling_rain
      override_real(8) = water_species_scaling_snow
      
      ! Initializes MAR data/parameters added for ecRad and ecRad itself
      call PHYrad_ecRad_init(mx, my, mz, dx, yyyy, mm,               &
       &                     './ecRad_config.nam',                   &
       &                     CMIP_scenario,                          &
       &                     pa_RAD, larRAD, lorRAD,                 &
       &                     MAR_override_logical=override_logical,  &
       &                     MAR_override_int=override_int,          &
       &                     MAR_override_real=override_real,        &
       &                     MAR_override_strato=blanket_layers,     &
       &                     MAR_override_spectral=sw_spectral_def)
      
      ecRad_initialized = .true.
    
    end if
    
    ! Runs the ecRad radiation scheme
    call PHYrad_ecRad(mx, my, mz, yyyy, mm, dd, i_hhss,       &
     &                dst_UA, albeSL, pa_RAD, pahRAD, cfraHY, &
     &                eps0SL, lsmRAD, czenGE, larRAD, lorRAD, &
     &                qv_RAD, rqiRAD, qwHY  , sw_RAD, qrHY  , &
     &                rqsRAD, tairDY, tairSL,                 &
     &                FIRn_c, FIRn_t, FSOn_c, FSOn_t, rPLWdv, &
     &                rPSWdf, rPSWdr, rFLopt)
    
    ! Now formats the outputs of ecRad for the MAR
    RADerr = 0
    do i=1, mx; do j=1, my

        ! Surface downward radiative fluxes
        if (.not. isnan(FIRn_t(i, j, 1))) then
            RAdOLR(i, j) = -FIRn_t(i, j, 1)
        end if
        
        ! J.-F. Grailet important remark: in PHYrad_ecRad, the equivalent of total-sky surface 
        ! shortwave downward flux is stored in the first slice of rFLopt (optional outputs of 
        ! ecRad). The ", 1" may eventually be replaced by a constant.
        
        if (.not. isnan(rFLopt(i, j, 1)) .and. rFLopt(i, j, 1) < 1350) then
            if (rFLopt(i, j, 1) < 0.1) rFLopt(i, j, 1) = 0. ! Kept just in case
            RAdsol(i, j)  = rFLopt(i, j, 1) ! total
            RAdsolc(i, j) = rFLopt(i, j, 3) ! clear-sky
            RAdsold(i, j) = rFLopt(i, j, 5) ! direct
            sol_SL(i, j)  = rFLopt(i, j, 1) * (1. - albeSL(i, j))
        else
            RADerr(i, j, 1) = RADerr(i, j, 1) + 1
        end if

        if (.not.isnan(FSOn_t(i,j,1))) then
            RAdOSR(i, j) = -FSOn_t(i, j, 1)
        end if

        ! JFG: previous code for "correcting" IR heating has been removed
        if (.not. isnan(FIRn_t(i, j, mz + 1))) then
            RAd_ir(i, j) = FIRn_t(i, j, mz + 1) + eps0SL(i, j) * tairSL(i, j) * tairSL(i, j) &
             &             * tairSL(i, j) * tairSL(i, j) * 5.670373e-8
        else
            RADerr(i, j, 2) = RADerr(i, j, 2) + 1
        endif

        ! Surface IR net radiative fluxes
        if (.not. isnan(FIRn_t(i, j, mz + 1))) then
            RAfnIR(i, j, mzz) = FIRn_t(i, j, mz + 1)
        end if

        if (.not. isnan(FIRn_c(i, j, mz + 1))) then
            RAfncIR(i, j, mzz) = FIRn_c(i, j, mz + 1)
        end if

        ! 3D variables
        do k=1, mz

            ! Atmospheric net radiative fluxes
            if (.not. isnan(FIRn_t(i, j, k))) then 
                RAfnIR(i, j, k) = FIRn_t(i, j, k)
            end if

            if (.not. isnan(FSOn_t(i, j, k))) then
                RAfnSO(i, j, k) = FSOn_t(i, j, k)
            end if

            if (.not. isnan(FIRn_c(i, j, k))) then
                RAfncIR(i, j, k) = FIRn_c(i, j, k)
            end if

            if (.not. isnan(FSOn_c(i, j, k))) then
                RAfncSO(i, j, k) = FSOn_c(i, j, k)
            end if

            ! Cloud fraction
            if (.not. isnan(cfraHY(i, j, k))) then
                cldMAX(i, j) = max(cfraHY(i, j, k), cldMAX(i, j))
                CldFRA(i, j, k) = cfraHY(i, j, k)
            end if

            ! Radiative heating
            WKxyz1(i, j, k) = -(FIRn_t(i, j, k + 1) - FIRn_t(i, j, k)) &
             &                * gravit / (cp * 1.e3 * pstDY(i, j) * dsigm1(k))
            WKxyz2(i, j, k) = -(FSOn_t(i, j, k + 1) - FSOn_t(i, j, k)) &
             &                * gravit / (cp * 1.e3 * pstDY(i, j) * dsigm1(k))

        end do
        
        ! Cloud fraction (2D)
        cld_SL(i, j) = cldMAX(i, j)
        clduSL(i, j) = 0.
        cldmSL(i, j) = 0.
        clddSL(i, j) = 0.
        do k=1, mz
            if (pahRAD(i, j, k) < 44000) then
                clduSL(i, j) = max(clduSL(i, j), CldFRA(i, j, k))
            end if
            if (pahRAD(i, j, k) >= 44000 .and. pahRAD(i, j, k) <= 68000) then
                cldmSL(i, j) = max(cldmSL(i, j), CldFRA(i, j, k))
            end if
            if (pahRAD(i, j, k) > 68000) then
                clddSL(i, j) = max(clddSL(i, j), CldFRA(i, j, k))
            end if
        end do
        
        ! Radiative heating adjustment (dt = time step from margrd)
        do k=1, mz
            tmp = (WKxyz1(i, j, k) + WKxyz2(i, j, k)) * dt / pkDY(i, j, k)

            if(.not. isnan(tmp) .and. abs(tmp) < 10) then 
                pktRAd(i, j, k) = tmp 
                ! htngIR(i, j, k) = WKxyz1(i, j, k) * 86400.
                ! htngSO(i, j, k) = WKxyz2(i, j, k) * 86400.
            else
                RADerr(i, j, 3) = RADerr(i, j, 3) + 1
            endif
        end do
    
    end do; end do ! End of formatting output variables

    ! +--Lateral Boundary Conditions for Radiative Variables
    ! +  ===================================================
    
    do k = 1, mz
        do j = 1, my
            pktRAd(1, j, k) = pktRAd(ip11, j, k)
            pktRAd(mx, j, k) = pktRAd(mx1, j, k)
        end do
        do i = 1, mx
            pktRAd(i, 1, k) = pktRAd(i, jp11, k)
            pktRAd(i, my, k) = pktRAd(i, my1, k)
        end do
    end do

    do i=1, mx; do j=1, my; do k=1, mz
        WKxyz1(i, j, k) = 0.
        WKxyz2(i, j, k) = 0.
    end do; end do; end do
    
    ! +--Some error control (check for NaN values)
    ! +  =========================================

    do j = 4, my - 3

        do i = 4, mx - 3
            ! Advertises NaN errors (3 potential sources, see code above)
            if (RADerr(i, j, 1) > 0 .or. RADerr(i, j, 2) > 0 .or. RADerr(i, j, 3) > 0) then
                write(6, '(a,a)') 'WARNING: PHYrad_ECRAD_in NaN on ', dateStr
                write(6, '(a,i4,i4,a,i2,a,i2,a,i2)') '(i, j)=', i, j, &
                 &                                   ', (1)=', RADerr(i, j, 1), & 
                 &                                   ', (2)=', RADerr(i, j, 2), & 
                 &                                   ', (3)=', RADerr(i, j, 3)
            end if

            ww = 0; nn = 0; ss = 0
            do k = -1, 1; do l = -1, 1
                ww = 1
                if (k == 0 .or. l == 0) ww = 2
                if (k == 0 .and. l == 0) ww = 0
                if (RAdsol(i + k, j + l) < 100) ww = 0
                nn = nn + ww
                ss = ss + RAdsol(i + k, j + l) * ww
            end do; end do

            if (nn == 12 .and. RAdsol(i, j) < (ss / nn) * 0.10) then
                write(6, '(a,a,a,i4,i4)') 'ERROR: likely error of // in PHYrad_ECRAD_in on', &
                 &                        dateTimeStr, ', (i, j)=', i, j
                RAdsol(i, j) = ss / nn
                write(6, '(a)') 'CHECK your SWD output !!!'
            end if
        end do

        ss = 0; ww = 0
        do i = 4 ,mx - 3
            ss = ss + RAdsol(i, j + 0)
            ww = ww + RAdsol(i, j + 1)
        end do
        ss = ss / real(mx - 3 - 4 + 1)
        ww = ww / real(mx - 3 - 4 + 1)

        if ((ss < 1 .and. ww > 100) .or. (ss > 100 .and. ww < 1)) then
            write(6, '(a,i4,i4)') dateTimeStr, 0, j
            write(6, '(a)') 'CHECK your SWD output !!!'
            stop
        endif

    end do

    ! +--Some error control (boundaries)
    ! +  ===============================

    do j = 2, my - 1; do i = 2, mx - 1; do n = 1, mz

        ww = 0; nn = 0; ss = 0
        do k = -1, 1; do l = -1, 1
            ww = 1
            if (k == 0 .or. l == 0) ww = 2
            if (k == 0 .and. l == 0) ww = 0
            nn = nn + ww
            ss = ss + abs(pktRAd(i + k, j + l, n)) * ww
        end do; end do

        if (abs(pktRAd(i, j, n)) > (ss / nn) + 1.5 .or. & 
         &  abs(pktRAd(i, j, n)) > 3. + 2.* n / mz) then
            write(6, '(a,a,a,i4,i4,i4)') 'ERROR: likely error of pktRAd in PHYrad_ECRAD_in on', &
             &  dateTimeStr, ', (i, j)=', i, j, ', n=', n
            pktRAd(i, j, n) = sign(1., pktRAd(i, j, n)) * (ss / nn)
        end if

    end do; end do; end do
    
    ! +--Freeing all previously allocated variables
    ! +  ==========================================

    ! Intermediate variables (MAR variables adjusted to ecRad)
    deallocate(pa_RAD)
    deallocate(pahRAD)
    deallocate(lsmRAD)
    deallocate(larRAD)
    deallocate(lorRAD)
    deallocate(qv_RAD)
    deallocate(sw_RAD)
    deallocate(cldMAX)
    deallocate(rqiRAD)
    deallocate(rqsRAD)

    ! Output variables from ecRad (allocated within PHYrad_ecRad)
    deallocate(FIRn_c)
    deallocate(FIRn_t)
    deallocate(FSOn_c)
    deallocate(FSOn_t)
    deallocate(rPLWdv)
    
    deallocate(rPSWdf)
    deallocate(rPSWdr)
    
    deallocate(rFLopt)

    return
endsubroutine PHYrad_ECRAD_in
