#include "MAR_pp.def"
subroutine PHYrad_CEP_in(dST_UA)

    ! +------------------------------------------------------------------------+
    ! | MAR PHYSICS                                         XF 07-12-2020  MAR |
    ! |                                                                        |
    ! |   subroutine PHYrad_CEP  interfaces MAR        with the    new         |
    ! |              ECMWF Solar/Infrared   Radiative  Transfer Scheme         |
    ! |                                                                        |
    ! |   f77 / f90  MAR /ECMWF  Interface                                     |
    ! |                                                                        |
    ! |   ECMWF Code Source:  J.-J. Morcrette, 28 nov 2002                     |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_dy
    use mar_hy
    use mar_ra
    use mar_sl
    use mar_wk
    use mar_io
#ifdef AR
    use mar_tc
#endif
    ! +--Interface  Variables
    ! +  ====================
    use radcep

    implicit none

    integer i, j, k, m
    real dST_UA                  ! Distance Soleil-Terre [UA]
    ! real RAcldE(mx, my, mz)    ! Cloud Emissivity       [-]
    ! real htngIR(mx, my, mz)    ! IR      Heating      [K/s]
    ! real htngSO(mx, my, mz)    ! Solar   Heating      [K/s]

    ! +--INPUT
    ! +  -----

    logical PHYrad_CEP_ERR

    integer yymmdd                 ! Date   in the form  yyyyMMdd
    integer i_hhss                 ! Number of seconds in the day

    real AlbCEP(klonr)          ! Surface Albedo
    real pa_CEP(klonr, klevr)    ! Air     Pressure (layer)
    real pahCEP(klonr, klevr + 1)  ! Air     Pressure (layer interface)
    real fcdCEP(klonr, klevr)    ! Cloud   Fraction (dropplets)
    real emsCEP(klonr)          ! Surface IR Emissivity
    real lsmCEP(klonr)          ! Land/Sea Mask: (1.=land  0.=ocean)
    real cszCEP(klonr)          ! cosine   (solar zenithal Distance)
    real czeMIN                 ! Minimum accepted for cszCEP
    real larCEP(klonr)          ! Latitude                  [radian]
    real lorCEP(klonr)          ! Longitude                 [radian]
    real ta_CEP(klonr, klevr)    ! Air     Temperature
    real tasCEP(klonr)          ! Surface Temperature

    real AerCEP(klonr, nn_aer, klevr)     ! Aerosol Concentration     !
    real O3rCEP(klonr, klevr)            ! O3 Concentration

    real cldMAX(mx, my)                  ! Cloud Max  Fraction    [-]
    real CD_OD1(klonr, klevr)            ! Cloud Optical Depth    [-]
    real CDtOD1(klonr)                  ! Cloud Optical Depth    [-]
    !                                              ! (vertically integrated)
    real Ae_ODa(klonr, klevr)            ! Aeros.Optical Depth    [-]
    real AetODa(klonr)                  ! Aeros.Optical Depth    [-]
    !                                              ! (vertically integrated)
    real qv_CEP(klonr, klevr)            ! Vapor   Concentr.  [kg/kg]
    real qi_CEP(klonr, klevr)            ! Cryst.  Concentr.  [kg/kg]
    real qw_CEP(klonr, klevr)            ! Droppl. Concentr.  [kg/kg]
    real sw_CEP(klonr, klevr)            ! Saturation % water [kg/kg]
    real qr_CEP(klonr, klevr)            ! Drops   Concentr.  [kg/kg]
    integer n, l, nae
    real ww, nn, ss

    ! +--OUTPUT
    ! +  ------

    real FIRn_c(klonr, klevr + 1)  ! CLEAR-SKY         LW NET      FLUXES
    real FIRn_t(klonr, klevr + 1)  ! TOTAL             LW NET      FLUXES
    real FSOn_c(klonr, klevr + 1)  ! CLEAR-SKY         SW NET      FLUXES
    real FSOn_t(klonr, klevr + 1)  ! TOTAL             SW NET      FLUXES
    real FSOs_t(klonr)          ! TOTAL-SKY SURFACE SW DOWNWARD FLUX

    integer ij0MAX, ij_MAX, OMP_GET_THREAD_NUM
    parameter(ij0MAX=mx2 * my2)
    integer nbvMAX, nb_MAX
    parameter(nbvMAX=ij0max / klonr)
    integer klonrb
    parameter(klonrb=ij0max - klonr * nbvMAX)

    integer ikl, lkl, nkl
    integer ij, nnn, nvc
    integer k2i(klonr)         ! i index corresp. to kl
    integer k2j(klonr)         ! j index corresp. to kl
#ifdef VR
    integer ij0ver(mx, my)         ! For Verif. of Vectoriz.
    integer ij_ver(mx, my)         ! For Verif. of Vectoriz.
    integer ij2, ijdver(mx, my)         ! For Verif. of Vectoriz.
#endif

    ! +--Surface Albedo
    ! +  --------------

    real bsegal, albmax, albx, dalb, albu
    real czeMAX, czrx
    real siceOK, ciceOK, zangOK, sign_T, ColdOK
    real sign_S, snowOK

    real qsfac

    ! +--OUTPUT
    ! +  ------

    integer io, CEPerr(mx, my)
    real zlevel, pr_atm, qcloud, fcloud, tmp
    real heatng                                 ! Total Heating [K/s]

    ! +--DATA
    ! +  ====

#ifdef GR
    ! qsfac : qs_HY contribution to qi_cep
    data qsfac/1.0/
#elif(AC)
    ! qsfac for Antarctica
    data qsfac/0.3/
#else
    data qsfac/0.5/
#endif
    data bsegal/2.00e0/
    data albmax/0.99e0/
    ! czeMAX: 80.deg (Segal et al., 1991 JAS)
    data czeMAX/0.173648178/
    !    MIN (Solar Zenithal Distance)
    data czeMIN/5.00e-6/

    ! +--INITIALIZATION
    ! +  ==============

#ifdef DB
    open(unit=30, status='unknown', file='PHYrad_CEP.txt')
    rewind 30
#endif

    if(iterun == 0) then
        ij_MAX = mx2 * my2
        if(mod(ij_MAX, klonr) == 0) then
            nb_MAX = ij_MAX / klonr
        else
            nb_MAX = ij_MAX / klonr + 1
        endif
    endif

    qcloud = 0.
    fcloud = 0.

    ! +--Time & Insolation (top of the atmosphere)
    ! +  =========================================

    yymmdd = min(iyrrGE, 2004) * 10000 + mmarGE * 100 + jdarGE
    i_hhss = jhurGE * 3600 + minuGE * 60 + jsecGE

    ! +--Zenith Angle Correction of Snow Albedo
    ! +  ======================================

    if(mod(iterun, jtRadi) == 0) then ! CTR
        if(.not. VSISVAT) then ! CTR

            do j = jp11, my1
                do i = ip11, mx1

                    siceOK = 1 - min(iabs(isolSL(i, j) - 2), iun)
                    ciceOK = 1 - min(iabs(isolSL(i, j) - 3), iun)

                    zangOK = max(siceOK, ciceOK)

                    sign_T = sign(unun, TfSnow - TairSL(i, j))
                    ColdOK = max(zero, sign_T)
                    zangOK = max(zangOK, ColdOK)

                    sign_S = zero
                    snowOK = max(zero, sign_S)
                    zangOK = max(zangOK, snowOK)

#ifdef CP
                    zangOK = 0.0e+0
#endif

                    ! +--Snow and/or ice covered surface
                    ! +  -------------------------------

                    albx = alb0SL(i, j)
                    czrx = max(czeMAX, czenGE(i, j))
                    dalb = 0.32e0 * ((bsegal + unun) / (unun + 2.e0 * bsegal * czrx) &
                                     - unun) / bsegal
                    dalb = max(dalb, zero)
                    albx = dalb + alb0SL(i, j)
                    albx = min(albx, albmax)
                    ! +***      Influence of Sun Zenith Angle
                    ! +         (Segal et al., 1991 JAS 48, p.1025)

                    ! +--Underlying Surface Albedo
                    ! +  -------------------------

                    albu = alb0SL(i, j)

                    ! +--Actual Albedo
                    ! +  -------------

                    albeSL(i, j) = zangOK * albx + (1 - zangOK) * albu

                enddo
            enddo

        ENDif                                                    ! CTR

        ! +--Effective Radiating Surface Temperature
        ! +  =======================================

        write(6, 397) jdarGE, mmarGE, iyrrGE, jhurGE, minuGE, jsecGE

397     format(' Call of PHYrad_CEP_mp IN : ' &
               , i2, '/', i2, '/', i4, ' ', i2, ':', i2, ':', i2)

        !$OMP PARALLEL
        RADin2 = .false.
        !$OMP END PARALLEL

        !$OMP PARALLEL do &
        !$OMP firstprivate(AlbCEP,pa_CEP,pahCEP,fcdCEP, &
        !$OMP emsCEP,lsmCEP,cszCEP,larCEP,lorCEP, &
        !$OMP AerCEP,O3rCEP,qv_CEP,qi_CEP,qw_CEP, &
        !$OMP sw_CEP,qr_CEP,ta_CEP,tasCEP,k2i,k2j, &
        !$OMP FIRn_c,FIRn_t,FSOn_c,FSOn_t,FSOs_t,tmp,n, &
        !$OMP CD_OD1,CDtOD1,Ae_ODa,AetODa,i,lkl,ikl,nae)
        do j = jp11, my1
            do i = ip11, mx1

                WKxy1(i, j) = 0.0

                do n = 1, mw
                    WKxy1(i, j) = WKxy1(i, j) + &
                                  eps0SL(i, j) * SLsrfl(i, j, n) * &
                                  tsrfSL(i, j, n)**4.
                enddo

                tviRA(i, j) = sqrt(sqrt(WKxy1(i, j) &
                                        + (1.-eps0SL(i, j)) * RAd_ir(i, j) / stefan))
                cld_SL(i, j) = 0.
                cldMAX(i, j) = 0.
                CEPerr(i, j) = 0
#ifdef VR
                ij0ver(i, j) = 0
                ij_ver(i, j) = 0
                ijdver(i, j) = 0
#endif

                ! +--Solar and IR Transfer through the Atmosphere
                ! +  ============================================

                ! +--Grid  Point   Dependant Variables --> PHYrad_CEP "Vector"Variables
                ! +  ------------------------------------------------------------------

                do ikl = 1, klonr
                    k2i(ikl) = i
                    k2j(ikl) = j
#ifdef VR
                    ij0ver(i, j) = ij0ver(i, j) + 1
                    ijdver(i, j) = ijdver(i, j) + ij
#endif

                    ! +--Geographic Coordinates
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^
                    larCEP(ikl) = sign(1., GElatr(i, j)) * &
                                  min(89.9 * degrad, abs(GElatr(i, j)))
                    lorCEP(ikl) = GElonh(i, j) * hourad
                    lorCEP(ikl) = lorCEP(ikl) &
                                  - pi * 2.*min(sign(1., lorCEP(ikl)), 0.)

                    ! +--Albedo
                    ! +  ^^^^^^
                    AlbCEP(ikl) = albeSL(i, j)

                    ! +--Surface
                    ! +  ^^^^^^^
                    pahCEP(ikl, mzz) = (pstDY(i, j) * sigmid(mzz) + ptopDY) * 1.e3

                    emsCEP(ikl) = eps0SL(i, j)         ! Emissivity
                    lsmCEP(ikl) = 1 - maskSL(i, j)         ! Land/sea Mask
                    cszCEP(ikl) = max(czenGE(i, j), czeMIN) ! cos(zenith.Dist.)

                    tasCEP(ikl) = tairSL(i, j)

                enddo

                do lkl = 1, mz
                    do ikl = 1, klonr
                        ! +--Pressure
                        ! +  ^^^^^^^^
                        pahCEP(ikl, lkl) = (pstDY(i, j) * sigmid(lkl) + ptopDY) * 1.e3
                        pa_CEP(ikl, lkl) = (pstDY(i, j) * sigma(lkl) + ptopDY) * 1.e3

                        ! +--Temperature
                        ! +  ^^^^^^^^^^^
                        ta_CEP(ikl, lkl) = tairDY(i, j, lkl)

                        ! +--Water   Concentration       (qsHY: Von Walden et al. 2003, JAM 42, p.1400)
                        ! +  ^^^^^^^^^^^^^^^^^^^^^       (      Fall  snow 24 mim                     )
                        !                                (      Blown snow 11 mim, only over ice sheet)
                        ! Vapor
                        qv_CEP(ikl, lkl) = max(1.e-6, qvDY(i, j, lkl))
                        ! Crystals
                        qi_CEP(ikl, lkl) = 0.
                        ! XF qi_CEP : exp((tairDY(i, j, lkl)-273.15)
                        qi_CEP(ikl, lkl) = qiHY(i, j, lkl) &
                                           + (1.-min(1., exp((tairDY(i, j, lkl) - 273.15) * 0.1))) &
                                           * (qsHY(i, j, lkl) * qsfac)
                        ! Dropplets
                        qw_CEP(ikl, lkl) = 0.
                        qw_CEP(ikl, lkl) = qwHY(i, j, lkl)
                        ! Saturation % W
                        sw_CEP(ikl, lkl) = min(qvswDY(i, j, lkl), 0.03)
                        ! Rain Drops
                        qr_CEP(ikl, lkl) = 0.
                        qr_CEP(ikl, lkl) = qrHY(i, j, lkl)

                        if(gplvDY(i, j, lkl) * grvinv - sh(i, j) > 50.) then
#ifdef GR
                            qi_CEP(ikl, lkl) = qiHY(i, j, lkl) + 0.75 * qsHY(i, j, lkl)
#endif

#ifdef EU
                            qi_CEP(ikl, lkl) = qiHY(i, j, lkl) + qsHY(i, j, lkl)
                            ! qw_CEP(ikl,lkl) =   qw_CEP(ikl,lkl) * 1.15
                            ! qr_CEP(ikl,lkl) =   qr_CEP(ikl,lkl) * 1.15
                            ! qi_CEP(ikl,lkl) =   qi_CEP(ikl,lkl) * 1.1
                            ! qv_CEP(ikl,lkl) =   qv_CEP(ikl,lkl) * 1.05
#endif
                        endif

                        !C #EU ... * 1.15 instead

                        ! +--Cloud Fraction (liquid water)
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        fcdCEP(ikl, lkl) = cfraHY(i, j, lkl)

                        ! +--O3      Concentration
                        ! +  ^^^^^^^^^^^^^^^^^^^^^
                        O3rCEP(ikl, lkl) = O3_MAR(i, j, lkl)

                    enddo
                enddo

                ! +--Aerosol Concentration
                ! +  ^^^^^^^^^^^^^^^^^^^^^
                do nae = 1, nn_aer
                    do lkl = 1, mz
                        do ikl = 1, klonr
                            AerCEP(ikl, nae, lkl) &
                                = Ae_MAR(k2i(ikl), k2j(ikl), nae, lkl)
                        enddo
                    enddo
                enddo

                ! +--Radiative Transfert Computation
                ! +  -------------------------------

                ! +  **********
                call PHYrad2CEP(klonr, klevr, nn_aer, yymmdd, i_hhss &
                                , dST_UA, AlbCEP, pa_CEP, pahCEP, fcdCEP &
                                , emsCEP, lsmCEP, cszCEP, larCEP, lorCEP &
                                , AerCEP, O3rCEP, qv_CEP, qi_CEP, qw_CEP &
                                , sw_CEP, qr_CEP, ta_CEP, tasCEP &
                                , FIRn_c, FIRn_t, FSOn_c, FSOn_t, FSOs_t &
                                , CD_OD1, CDtOD1, Ae_ODa, AetODa, iyrrGE &
                                , radINI, radIN2, CMIP_scenario)
                ! +  **********

                ! +  -------------------------------

                RADini = .true.; RADin2 = .true.

                ! +--Grid  Point   Dependant Variables <-- PHYrad_CEP "Vector"Variables
                ! +  ------------------------------------------------------------------

                do ikl = 1, klonr

#ifdef VR
                    ij2 = ij2 + 1
                    ijdver(i, j) = ijdver(i, j) - ij2
                    ij_ver(i, j) = ij_ver(i, j) + 1
#endif

                    ! +--Surface Cloud/Aerosol Optical Depth
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                    if(.not. isnan(CDtOD1(ikl))) &
                        RAcdtO(i, j) = CDtOD1(ikl)
                    if(.not. isnan(AetODa(ikl))) &
                        RAertO(i, j) = AetODa(ikl)

                    ! +--Surface Downward Radiative Fluxes
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                    if(.not. isnan(FIRn_t(ikl, 1))) &
                        RAdOLR(i, j) = -FIRn_t(ikl, 1)

                    if(.not. isnan(FSOs_t(ikl)) .and. FSOs_t(ikl) < 1350) then
                        if(FSOs_t(ikl) < 0.1) FSOs_t(ikl) = 0.
                        RAdsol(i, j) = FSOs_t(ikl)
                        sol_SL(i, j) = FSOs_t(ikl) * (1.-albeSL(i, j))
                    else
                        CEPerr(i, j) = CEPerr(i, j) + 1
                    endif

                    if(.not. isnan(FSOn_t(ikl, 1))) &
                        RAdOSR(i, j) = -FSOn_t(ikl, 1)

                    if(.not. isnan(FIRn_t(ikl, 1 + klevr))) then
                        RAd_ir(i, j) = FIRn_t(ikl, 1 + klevr) &
                                       + eps0SL(i, j) * TairSL(i, j) * TairSL(i, j) &
                                       * TairSL(i, j) * TairSL(i, j) &
                                       * 5.670373e-8
#ifdef EU
                        !XF 04/12/2020
                        RAd_ir(i, j) = RAd_ir(i, j) + 1.
#endif
#ifdef GR
                        !XF 04/12/2020
                        RAd_ir(i, j) = RAd_ir(i, j) + 1.
#endif

#ifdef EU
                        if(sol_SL(i, j) > 0) then
                            tmp = RAd_ir(i, j) * 0.03
                            RAd_ir(i, j) = RAd_ir(i, j) * 1.03
                            RAdsol(i, j) = max(RAdsol(i, j) * 0.90, RAdsol(i, j) - tmp * 3.)
                            sol_SL(i, j) = RAdsol(i, j) * (1.-albeSL(i, j))
                        endif
#endif
                    else
                        CEPerr(i, j) = CEPerr(i, j) + 1
                    endif

                    ! +--Surface IR Net   Radiative Fluxes
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                    if(.not. isnan(FIRn_t(ikl, mzz))) &
                        RAfnIR(i, j, mzz) = FIRn_t(ikl, mzz)
                    if(.not. isnan(FIRn_c(ikl, mzz))) &
                        RAfncIR(i, j, mzz) = FIRn_c(ikl, mzz)
                enddo

                do lkl = 1, mz
                    do ikl = 1, klonr

                        ! +--Atmosph. Net   Radiative Fluxes
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        if(.not. isnan(FIRn_t(ikl, lkl))) &
                            RAfnIR(i, j, lkl) = FIRn_t(ikl, lkl)
                        if(.not. isnan(FSOn_t(ikl, lkl))) &
                            RAfnSO(i, j, lkl) = FSOn_t(ikl, lkl)
                        if(.not. isnan(FIRn_c(ikl, lkl))) &
                            RAfncIR(i, j, lkl) = FIRn_c(ikl, lkl)
                        if(.not. isnan(FSOn_c(ikl, lkl))) &
                            RAfncSO(i, j, lkl) = FSOn_c(ikl, lkl)

                        ! +--Cloud   Fraction
                        ! +  ^^^^^^^^^^^^^^^^
                        if(.not. isnan(fcdCEP(ikl, lkl))) then
                            cldMAX(i, j) = max(fcdCEP(ikl, lkl), cldMAX(i, j))
                            CldFRA(i, j, lkl) = fcdCEP(ikl, lkl)
                        endif

                        ! +--Atmosph.Cloud/Aerosol Optical Depth
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        if(.not. isnan(CD_OD1(ikl, lkl))) &
                            RAcd_O(i, j, lkl) = CD_OD1(ikl, lkl)
                        if(.not. isnan(Ae_ODa(ikl, lkl))) &
                            RAer_O(i, j, lkl) = Ae_ODa(ikl, lkl)

                        ! +--Radiative Heating
                        ! +  ^^^^^^^^^^^^^^^^^
                        WKxyz1(i, j, lkl) = -(FIRn_t(ikl, lkl + 1) - FIRn_t(ikl, lkl)) &
                                            * gravit / (cp * 1.e3 * pstDY(i, j) * dsigm1(lkl))
                        WKxyz2(i, j, lkl) = -(FSOn_t(ikl, lkl + 1) - FSOn_t(ikl, lkl)) &
                                            * gravit / (cp * 1.e3 * pstDY(i, j) * dsigm1(lkl))

                        ! +--O3      Concentration
                        ! +  ^^^^^^^^^^^^^^^^^^^^^
                        if(.not. isnan(O3rCEP(ikl, lkl))) &
                            O3_MAR(i, j, lkl) = O3rCEP(ikl, lkl)

                    enddo
                enddo

                ! +--Cloud   Fraction
                ! +  ^^^^^^^^^^^^^^^^
                do ikl = 1, klonr

                    cld_SL(i, j) = cldMAX(i, j)
                    clduSL(i, j) = 0.
                    cldmSL(i, j) = 0.
                    clddSL(i, j) = 0.
                    do lkl = 1, mz
                        if(pahCEP(ikl, lkl) < 44000) &
                            clduSL(i, j) = max(clduSL(i, j), CldFRA(i, j, lkl))
                        if(pahCEP(ikl, lkl) >= 44000 .and. pahCEP(ikl, lkl) <= 68000) &
                            cldmSL(i, j) = max(cldmSL(i, j), CldFRA(i, j, lkl))
                        if(pahCEP(ikl, lkl) > 68000) &
                            clddSL(i, j) = max(clddSL(i, j), CldFRA(i, j, lkl))
                    enddo
                enddo

                ! +--Radiative Heating
                ! +  ^^^^^^^^^^^^^^^^^
                do lkl = 1, mz
                    do ikl = 1, klonr
                        tmp = (WKxyz1(i, j, lkl) + WKxyz2(i, j, lkl)) &
                              * dt / pkDY(i, j, lkl)

                        if(.not. isnan(tmp) .and. abs(tmp) < 10) then
                            pktRAd(i, j, lkl) = tmp
                            ! htngIR(i, j, lkl) = WKxyz1(i, j, lkl) * 86400.
                            ! htngSO(i, j, lkl) = WKxyz2(i, j, lkl) * 86400.
                        else
                            CEPerr(i, j) = CEPerr(i, j) + 1
                        endif
                    enddo
                enddo

                ! +--Aerosol Concentration
                ! +  ^^^^^^^^^^^^^^^^^^^^^
                do nae = 1, nn_aer
                    do lkl = 1, mz
                        do ikl = 1, klonr
                            if(.not. isnan(AerCEP(ikl, nae, lkl))) &
                                Ae_MAR(k2i(ikl), k2j(ikl), nae, lkl) &
                                = AerCEP(ikl, nae, lkl)
                        enddo
                    enddo
                enddo

            enddo
        enddo
        !$OMP END PARALLEL DO

        write(6, 398) jdarGE, mmarGE, iyrrGE, jhurGE, minuGE, jsecGE

398     format(' Call of PHYrad_CEP_mp OUT : ' &
               , i2, '/', i2, '/', i4, ' ', i2, ':', i2, ':', i2)

        ! +--Lateral Boundary Conditions for Radiative Variables
        ! +  ===================================================

        do k = 1, mz
            do j = 1, my
                pktRAd(1, j, k) = pktRAd(ip11, j, k)
                pktRAd(mx, j, k) = pktRAd(mx1, j, k)
            enddo
            do i = 1, mx
                pktRAd(i, 1, k) = pktRAd(i, jp11, k)
                pktRAd(i, my, k) = pktRAd(i, my1, k)
            enddo
        enddo

        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    WKxyz1(i, j, k) = 0.
                    WKxyz2(i, j, k) = 0.
                enddo
            enddo
        enddo

        !      ------------------------------------------------------------

        PHYrad_CEP_ERR = .false.

        do j = 4, my - 3
            do i = 4, mx - 3
                if(CEPerr(i, j) > 0) then
                    write(6, 399) iyrrGE, mmarGE, jdarGE, &
                        jhurGE, i, j, CEPerr(i, j)
399                 format('XF WARNING: PHYrad_CEP_mp NaN on ', i4, '/', i2, '/', &
                           i2, i3, 'h, (i,j)=', i4, i4, ', #err=', i4)
                    PHYrad_CEP_ERR = .true.
                endif

                ww = 0; nn = 0; ss = 0

                do k = -1, 1
                    do l = -1, 1
                        ww = 1
                        if(k == 0 .or. l == 0) ww = 2
                        if(k == 0 .and. l == 0) ww = 0
                        if(RAdsol(i + k, j + l) < 100) ww = 0
                        nn = nn + ww
                        ss = ss + RAdsol(i + k, j + l) * ww
                    enddo
                enddo

                if(nn == 12 .and. RAdsol(i, j) < (ss / nn) * 0.10) then
                    write(6, 400) iyrrGE, mmarGE, jdarGE, &
                        jhurGE, i, j
400                 format('ERROR: likely error of // in PHYrad_CEP_mp on', &
                           i4, '/', i2, '/', i2, i3, 'h, (i,j)=', i4, i4)
                    RAdsol(i, j) = ss / nn
                    print *, "CHECK your SWD output!!!!"
                endif
            enddo

            ss = 0; ww = 0

            do i = 4, mx - 3
                ss = ss + RAdsol(i, j + 0)
                ww = ww + RAdsol(i, j + 1)
            enddo
            ss = ss / real(mx - 3 - 4 + 1)
            ww = ww / real(mx - 3 - 4 + 1)

            if((ss < 1 .and. ww > 100) .or. (ss > 100 .and. ww < 1)) then
                write(6, 400) iyrrGE, mmarGE, jdarGE, jhurGE, 0, j
                print *, "CHECK your SWD output!!!!"
                stop
            endif

        enddo

        !      ------------------------------------------------------------

        do j = 2, my - 1
            do i = 2, mx - 1
                do n = 1, mz

                    ww = 0; nn = 0; ss = 0

                    do k = -1, 1
                        do l = -1, 1
                            ww = 1
                            if(k == 0 .or. l == 0) ww = 2
                            if(k == 0 .and. l == 0) ww = 0
                            nn = nn + ww
                            ss = ss + abs(pktRAd(i + k, j + l, n)) * ww
                        enddo
                    enddo

                    if(abs(pktRAd(i, j, n)) > (ss / nn) + 1.5 &
                       .or. abs(pktRAd(i, j, n)) > 3.+2.*n / mz) then
                        write(6, 390) iyrrGE, mmarGE, jdarGE, jhurGE, i, j, n
390                     format('ERROR: likely error of pktRAd in PHYrad_CEP_mp on', &
                               i4, '/', i2, '/', i2, i3, 'h, (i,j)=', i4, i4, i4)
                        pktRAd(i, j, n) = sign(1., pktRAd(i, j, n)) * (ss / nn)
                    endif

                enddo
            enddo
        enddo

        !       ------------------------------------------------------------
        
#ifdef VR
        ! C +--OUTPUT for Verification
        ! C +  -----------------------
        write(6, 6000)
6000    format(/, 'Verification of Vectorization: Before CALL')
        do j = my, 1, -1
            write(6, 6010)(ij0ver(i, j), i=1, mx)
6010        format(132i1)
        enddo

        write(6, 6001)
6001    format(/, 'Verification of Vectorization: After  CALL')
        do j = my, 1, -1
            write(6, 6010)(ij_ver(i, j), i=1, mx)
        enddo

        do j = 1, my
            do i = 1, mx
                if(ijdver(i, j) /= 0 .and. ij_ver(i, j) /= 1) write(6, 6002) i, j, ijdver(i, j)
6002            format(' Vectorization ERROR on', 2i4, '   (', i6, ')')
            enddo
        enddo
#endif

    endif

#ifdef DB
    close(unit=30)
#endif

    return
endsubroutine PHYrad_CEP_in
