#include "MAR_pp.def"
subroutine PHY_SISVAT_MP(ihamr_SIS, nhamr_SIS)
    ! +------------------------------------------------------------------------+
    ! | MAR          SISVAT_Driver                             02-04-2024  MAR |
    ! |   subroutine PHY_SISVAT    interfaces MAR        with        the       |
    ! |              Soil/Ice Snow Vegetation Atmosphere Transfer Scheme       |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT: ihamr_SIS: Time Digital Filter Status                         |
    ! |   ^^^^^  nhamr_SIS: Time Digital Filter Set Up                         |
    ! |                                                                        |
    ! |   INPUT    (via common block)                                          |
    ! |   ^^^^^     VegMod: SISVAT    is set up when .T.                       |
    ! |             SnoMod: Snow Pack is set up when .T.                       |
    ! |             reaLBC: Update Bound.Condit.when .T.                       |
    ! |             iterun: Run Iterations Counter                             |
    ! |                                                                        |
    ! |   INPUT    (via common block)                                          |
    ! |   ^^^^^     xxxxTV: SISVAT/MAR interfacing variables                   |
    ! |                                                                        |
    ! | # CAUTION: #sa: Stand Alone Preprocessing Label must be removed        |
    ! | # ^^^^^^^       when SISVAT is coupled with MAR                        |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT PHYSICS                                |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^^^^^^                                |
    ! | #                        #HY                                           |
    ! | #                        #SN: Snow         Model                       |
    ! | #                        #BS: Blowing Snow Parameterization            |
    ! | #                        #SI  Sea-Ice      Parameterization            |
    ! | #                        #GP  LAI and GLF  Variations not specified    |
    ! | #                        #OP  SST       is interactive                 |
    ! |                                                                        |
    ! | #                        #DS: diffuse radiation differing from direct  |
    ! |                              (variable RADsod must still be included)  |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT PHYSICS: Col de Porte                  |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^                  |
    ! | #                        #CP: SBL,                       Col de Porte  |
    ! | #                        #cp  Solar Radiation,           Col de Porte  |
    ! | #                        #AG: Snow Ageing,               Col de Porte  |
    ! |                                                                        |
    ! |                                                                        |
    ! |   Preprocessing  Option: SISVAT IO (not always a standard preprocess.) |
    ! |   ^^^^^^^^^^^^^^^^^^^^^  ^^^^^^^^^                                     |
    ! |   FILE                 |      CONTENT                                  |
    ! |   ~~~~~~~~~~~~~~~~~~~~~+~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ |
    ! | # ANI.yyyymmdd.LAB.nc  | #NC: OUTPUT on NetCDF File (Stand Alone EXP.) |
    ! |                        |                                               |
    ! | # SISVAT_iii_jjj_n     | #E0: OUTPUT on ASCII  File (SISVAT Variables) |
    ! | #                      |(#E0  MUST BE PREPROCESSED BEFORE #e1 & #e2 !) |
    ! | # SISVAT_iii_jjj_n     | #e1: OUTPUT/Verification: Energy Conservation |
    ! | # SISVAT_iii_jjj_n     | #e2: OUTPUT/Verification: Energy Consrv.2e pt.|
    ! |                        |                           (no premature stop) |
    ! |                        |                                               |
    ! | # SISVAT_iii_jjj_n     | #m0: OUTPUT/Verification: H2O    Conservation |
    ! | # SISVAT_iii_jjj_n     | #m1: OUTPUT/Verification: * Mass Conservation |
    ! | # SISVAT_iii_jjj_n     | #m2: OUTPUT/Verification: SeaIce Conservation |
    ! |                        |                                               |
    ! | # SISVAT_zSn.vz        | #vz: OUTPUT/Verification: Snow Layers Agrega. |
    ! |                        |      unit 41, subroutine  SISVAT_zSn **ONLY** |
    ! | # SISVAT_qSo.vw        | #vw: OUTPUT/Verif+Detail: H2O    Conservation |
    ! |                        |      unit 42, subroutine  SISVAT_qSo **ONLY** |
    ! | # SISVAT_qSn.vm        | #vm: OUTPUT/Verification: Energy/Water Budget |
    ! |                        |      unit 43, subroutine  SISVAT_qSn **ONLY** |
    ! | # SISVAT_qSn.vu        | #vu: OUTPUT/Verification: Slush  Parameteriz. |
    ! |                        |      unit 44, subroutine  SISVAT_qSn **ONLY** |
    ! | # SISVAT_wEq.ve        | #ve: OUTPUT/Verification: Snow/Ice Water Eqv. |
    ! |                        |      unit 45, subroutine  SISVAT_wEq **ONLY** |
    ! | # SnOptP____.va        | #va: OUTPUT/Verification: Albedo Parameteriz. |
    ! |                        |      unit 46, subroutine  SnOptP     **ONLY** |
    ! | # SISVAT_GSn.vp        | #vp: OUTPUT/Verification: Snow   Properties   |
    ! |                        |      unit 47, subroutines SISVAT_zSn, _GSn    |
    ! | # PHY_SISVAT.v0        | #v0: OUTPUT/Verification: DUMP                |
    ! |                        |      unit 50, subroutine  PHY_SISVAT **ONLY** |
    ! |                        |                                               |
    ! | # stdout               | #s0: OUTPUT of Snow Buffer Layer              |
    ! |                        |      unit  6, subroutine  SISVAT     **ONLY** |
    ! | # stdout               | #wx: OUTPUT/Verification: specified i,j,k,n   |
    ! | # stdout               | #wz: OUTPUT of Roughness Length (Blown Snow)  |
    ! |                        |      unit  6, subroutines SISVAT, PHY_SISVAT  |
    ! |                        |                                               |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_ra
    use mar_lb
    use mar_dy
    use mar_hy
    use mar_tu
    use mar_sv
    use mardSV
    use mar0SV
    use mar_sl
    use mar_TV
    use mar_bs
    use marssn
    use mar_ib
    use marsib
    use mar_wk
    use marmagic
#ifdef BW
    use mar_te
#endif
#ifdef TC
    use mar_tc
#endif
#ifdef AO
    use mar_AO
#endif
#ifdef PO
    use mar_po
#endif
    ! +--INTERFACE Variables
    ! +  ===================
    use marxsv
    use marysv
#ifdef iso
    use mariso, only: wiso, niso, qsrfHY_iso, SLuqsl_iso, uqs_SV_iso, &
                      SLuqs_iso, qvapSL_iso, evapTV_iso, Rdefault
#endif
    implicit none

    ! +--Global Variables
    ! +  ================

    integer ihamr_SIS, nhamr_SIS          ! Hamming Filter Counters
    integer newglfSIS                     !
    integer newsicSI                      !

    real rtime
    integer ntime
    common / c_time / ntime

    integer mw0
    parameter(mw0=3)

    ! +--Level of negligible blown Snow Particles Concentration
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer kB
    common / SISVAT_MAR__BS / kB

    ! +--10-m Level
    ! +  ~~~~~~~~~~
    integer kSBL
    common / PHY_SISVAT_SBLi / kSBL
    real rSBL10, VV__10(mx, my), ERprev(mx, my, mw)
    common / PHY_SISVAT_SBLr / rSBL10, ERprev

    ! +--V,  dT(a-s)    Time Moving Averages
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    real V__mem(klonv, ntaver)   ! ntaver defined in MAR_SL.inc
    real VVmmem(klonv)          !
    common / SVeSBLmem / V__mem, VVmmem          !
    real T__mem(klonv, ntaver)   !
    real dTmmem(klonv)          !
    common / STeSBLmem / T__mem, dTmmem          !

    !$OMP threadprivate(/SVeSBLmem/,/STeSBLmem/)

    ! +--u*, u*T*, u*s* Time Moving Averages
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef AM
    ! ntaver defined in mar_sl
    real u__mem(klonv, ntaver)
    common / S_eSBLmem / u__mem
#endif
#ifdef AT
    real uT_mem(klonv, ntaver), uT_mem
#endif
#ifdef AS
    real us_mem(klonv, ntaver), us_mem
#endif

    ! +--OUTPUT for Stand Alone NetCDF File
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef NC
    ! SOsoKL : Absorbed Solar Radiation
    real SOsoKL(klonv)
    ! IRsoKL : Absorbed IR    Radiation
    real IRsoKL(klonv)
    ! HSsoKL : Absorbed Sensible Heat Flux
    real HSsoKL(klonv)
    ! HLsoKL : Absorbed Latent   Heat Flux
    real HLsoKL(klonv)
    ! HLs_KL : Evaporation
    real HLs_KL(klonv)
    ! HLv_KL : Transpiration
    real HLv_KL(klonv)
    common / DumpNC / SOsoKL, IRsoKL, HSsoKL, HLsoKL, HLs_KL, HLv_KL
    ! SOsoNC : Absorbed Solar Radiation
    real SOsoNC(mx, my, nvx)
    ! IRsoNC : Absorbed IR    Radiation
    real IRsoNC(mx, my, nvx)
    ! HSsoNC : Absorbed Sensible Heat Flux
    real HSsoNC(mx, my, nvx)
    ! HLsoNC : Absorbed Latent   Heat Flux
    real HLsoNC(mx, my, nvx)
    ! HLs_NC : Evaporation
    real HLs_NC(mx, my, nvx)
    ! HLv_NC : Transpiration
    real HLv_NC(mx, my, nvx)
    ! eta_NC : Soil Humidity
    real eta_NC(mx, my, nvx)
    common / writNC / SOsoNC, IRsoNC, HSsoNC, HLsoNC, HLs_NC, HLv_NC, eta_NC
#endif

#ifdef wx
    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    integer iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
    common / SISVAT_EV / iSV_v1, jSV_v1, nSV_v1, kSV_v1, lSV_v1
#endif

    ! +--Internal  Variables
    ! +  ===================
    integer i, j, k, m
    logical StandA, snow_filter
    logical glfFIX
    ! ijnmax, nvcmax : Control Indices Distribution
    integer ijnmax, nvcmax
    common / SISVAT_MAR_Loc / ijnmax, nvcmax
    ! k2i : Distributed i      Index
    integer k2i(klonv)
    ! k2j : Distributed j      Index
    integer k2j(klonv)
    ! k2n : Distributed mosaic Index
    integer k2n(klonv)
#ifdef VR
    ! ij0ver : Verification of Vectorization
    integer ij0ver(mx, my, mw)
    ! ij_ver : Verification of Vectorization
    integer ij_ver(mx, my, mw)
    ! ij2, ijdver : Verification of Vectorization
    integer ij2, ijdver(mx, my, mw)
#endif
    character * 1 cha, chb
    integer iwr, ipt, l, nvcmax2, ijnmax2, itPhys
    integer ikl, isl, isn
    integer ijn, ij, nnn
    integer nvc, nkl, n, nt

    ! slopx, slopy : Surf.Slope, x, y
    real slopx, slopy
    ! czemin : Minimum accepted cos(Solar zenith.Dist.)
    real czemin
    ! Upw_IR : Upward IR Flux
    real Upw_IR(mx, my)
    ! IR_aux : Upward IR Flux  (dummy)
    real IR_aux
    ! uqstar : u*q*
    real uqstar
    ! rhAir : Air    Densitity
    real rhAir
    ! Ua_min : Minimum Air Velocity
    real Ua_min
    ! rr__DR : Desagregated  Rain
    real rr__DR(mx, my, mw)
    ! hfra : Frazil Thickness
    real hfra(mx, my, mw)
    ! Rnof : RunOFF Intensity
    real Rnof(mx, my, mw)
    ! Ruof : RunOFF Intensity
    real Ruof(mx, my, mw, 6)
    ! EvSu : RunOFF Intensity
    real EvSu(mx, my, mw, 4)
    ! d_snow, SnowOK : Snow Precip.: Total
    real d_snow, SnowOK
#ifdef SZ
    ! dsastr : z0(Sastrugi): Variation
    real dsastr(mx, my)
#endif
    real WVaLim(mx, my)
    ! FixSST, VarSST : SST forcing switch
    real FixSST, VarSST
    ! SSTnud : SST Nudging Rate
    real SSTnud
    common / SISVAT_MAR_ocn / FixSST, VarSST, SSTnud

    real ifra_t
    !XF
    ! SrfSIC, SIc0OK : Oceanic Fraction: previous
    real SrfSIC, SIc0OK
    ! FraOcn, SIceOK : Oceanic Fraction
    real FraOcn, SIceOK
    ! TocnSI : Ocn Temp.=> S-Ice Covered
    real TocnSI
    ! OcnMin : Oceanic Fraction: Minimum
    real OcnMin
    ! dzSIce : Sea-Ice Layers    Thickness
    real dzSIce(4)
    ! SIcMIN : Sea-Ice Layer Min Thickness
    real SIcMIN
    ! SIc_OK : Sea-Ice Switch
    real SIc_OK(2)
    ! c1_zuo, c2_zuo, c3_zuo : Run Off Parameters
    real c1_zuo, c2_zuo, c3_zuo
    ! SnowWE : Snow Water Equivalent[m w.e.]
    real SnowWE
    ! rosNEW : Added Snow Density    [kg/m3]
    real rosNEW
    ! S_Eros, SnEros : Snow Erosion (status) = (1,0)
    real S_Eros, SnEros
#ifdef BW
    integer noUNIT
    real BlowST, SnowSB
#endif
#ifdef WR
    ! ifrVER : Verification Variable: Total Fraction must be 100%
    integer ifrVER(mx, my)
#endif
    real tairDY_2D(mx, my), qvDY_2D(mx, my)
    real uu, vv, ww
#ifdef AO
    !coupling ck AO
    real zntot
#endif

    ! +--DATA
    ! +  ====
    data StandA/.true./
    data glfFIX/.false./
    data cha/'-'/
    data chb/':'/

    data czemin/1.e-3/

    data TocnSI/270.70/       ! Ocn Temp.=> S-Ice Covered
#ifdef AO
    !AO_CK 20/02/2020 same as in NEMO
    data OcnMin/0.01/
#endif
    data OcnMin/0.05/       ! Oceanic Fraction: Minimum
    data dzSIce/0.5, 0.05, 0.001, 0.0/    ! Sea-Ice Layers    Thickness
    data SIcMIN/0.1/                   ! Sea-Ice Layer Min Thickness
    data SIc_OK/1.0, 0.00/              ! Sea-Ice Switch
    ! +
    !     data    c1_zuo/12.960e+4/,c2_zuo/2.160e+6/,c3_zuo/1.400e+2/ ! Zuoriginal
    !     data    c1_zuo/ 2.796e+4/,c2_zuo/2.160e+6/,c3_zuo/1.400e+2/ ! ETH Tuning
    !     data    c1_zuo/    86400/,c2_zuo/  777600/,c3_zuo/1.400e+2/ ! 1-10 days
    ! from 3h (c1_zuo) to 15h (c1_zuo+c2_zuo).
    data c1_zuo/10800/, c2_zuo/54000/, c3_zuo/1.400e+2/ ! 18h day max

    ! +...        Run Off Parameters
    ! +           86400*1.5 day     ...*25 days (Modif. ETH Camp: 86400*0.3day)
    ! +           (Zuo and Oerlemans 1996, J.Glacio. 42, 305--317)

    ! +--SISVAT Time             Variable
    ! +  ================================
    dt__SV = dt

    ! +   ++++++++++++++++  INITIALISATION: BEGIN +++
    if(.not. INI_SV) then
        ! +   ++++++++++++++++

        ! +--OUTPUT point (i,j,n) coordinates
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        iwr_SV = 1
        jwr_SV = 1
        nwr_SV = 1

        ! +--Level of negligible blown Snow Particles Concentration ( ~ 100magl)
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        kB = mz
11      continue
#ifdef AE
        if(zsigma(kB) > 100. .OR. kB <= 1) go to 10
        kB = kB - 1
        go to 11
#endif
10      continue
#ifdef AE
        write(6, 1000) kB
#endif
1000    format(/, ' BS : Level of negligible ' &
                , 'blown Snow Particles Concentration is', i4 &
                , ' (i.e., ~ 100. magl)',/)

#ifdef wx
        ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
        ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ! Snow Erosion Statistics: Grid Point Coordinate
        iSV_v1 = imez
        ! Id.
        jSV_v1 = jmez
        ! Id.
        nSV_v1 = 1
        ! Snow Erosion Statistics: OUTPUT SWITCH (if   >   0)
        !                          .LE. 1 Blowing Snow
        !                          .LE. 2 Blowing Snow (FULL)
        !                          .EQ. 3 Snow    Agregation
        lSV_v1 = 1
#endif

        ! +--SISVAT Time Independant Variables
        ! +  =================================

        StandA = .false.
        if(VSISVAT .and. StandA) then
            write(6, 600)
600         format(/, '### MAR_SISVAT CRASH,', &
                    ' STAND ALONE LABEL #sa is ON ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 15x, 'EMERGENCY STOP')
            stop
        endif

        if(mw /= nvx) then
            write(6, 601) mw, nvx
601         format(/, '### MAR_SISVAT CRASH, mw =', i6, &
                    ' .NE. nvx =', i6, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 15x, 'EMERGENCY STOP')
            stop
        endif

        if(mw /= nsx) then
            write(6, 602) mw, nsx
602         format(/, '### MAR_SISVAT CRASH, mw =', i6, &
                    ' .NE. nvx =', i6, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 15x, 'EMERGENCY STOP')
            stop
        endif

        if(nsol + 1 /= llx) then
            write(6, 603) nsol + 1, llx
603         format(/, '### MAR_SISVAT CRASH, ns =', i6, &
                    ' .NE. nvx =', i6, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 15x, 'EMERGENCY STOP')
            stop
        endif

        if(nb_wri > mz) then
            write(6, 604) nb_wri, mz
604         format(/, '### MAR_SISVAT CRASH, nb_wri =', i6, &
                    ' .GT. mz      =', i3, ' ', 2x, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 23x, 'EMERGENCY STOP')
            !         stop
        endif

        if(nb_wri > mw * iptx) then
            write(6, 605) nb_wri, mw, iptx
605         format(/, '### MAR_SISVAT CRASH, nb_wri =', i6, &
                    ' .GT. mw *iptx=', i3, '*', i2, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 23x, 'EMERGENCY STOP')
            !          stop
        endif

        if(nb_wri > nsx * iptx) then
            write(6, 606) nb_wri, nsx, iptx
606         format(/, '### MAR_SISVAT CRASH, nb_wri =', i6, &
                    ' .GT. nsx*iptx=', i3, '*', i2, ' ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 23x, 'EMERGENCY STOP')
            stop
        endif


        if(klonv /= 1) then
            write(6, 608) klonv
608         format(/, '#BS MAR_SISVAT CRASH, klonv =', i6, '.ne.256 ###', &
                    /, '    ?!&~@|@[#@#]=!!!', 15x, 'EMERGENCY STOP')
            stop
        endif

        ! +  ****************
        call SISVAT_ini
        ! +  ****************

        ! +--Grids Correspondance
        ! +  --------------------

        do isl = -nsol, 0
            deptTV(1 - isl) = dz_dSV(isl)
        enddo

        ntime = 0

        ijnmax = mx2 * my2 * nvx
        if(mod(ijnmax, klonv) == 0) then
            nvcmax = ijnmax / klonv
        else
            nvcmax = ijnmax / klonv + 1
        endif
        ! +
        ! +
        ! +--Surface Fall Line Slope
        ! +  -----------------------
        ! +
        if(SnoMod) then
            if(mx == 1 .and. my == 1) then
                ! Normalized Decay of the
                ! Surficial Water Content
                !(Zuo and Oerlemans 1996, J.Glacio. 42, 305--317)
                SWfSNo(1, 1) = &
                    exp(-dt__SV &
                        / (c1_zuo &
                           + c2_zuo * exp(-c3_zuo * slopTV(1, 1))))
            else
                do j = jp11, my1
                    do i = ip11, mx1
                        slopx = (sh(ip1(i), j) - sh(im1(i), j)) * dxinv3(i, j)
                        slopy = (sh(i, jp1(j)) - sh(i, jm1(j))) * dyinv3(i, j)
                        slopTV(i, j) = sqrt(slopx * slopx + slopy * slopy)
                        ! SWfSNo
                        ! Normalized Decay of the
                        ! Surficial Water Content
                        ! (Zuo and Oerlemans 1996, J.Glacio. 42, 305--317)
                        SWfSNo(i, j) = &
                            exp(-dt__SV &
                                / (c1_zuo &
                                   + c2_zuo * exp(-c3_zuo * slopTV(i, j))))
                        slopGE(i, j) = cos(atan(slopTV(i, j)))
                    enddo
                enddo
            endif
        endif

        ! +--Initialization of Surface Types
        ! +  ===============================
        if(itexpe == 0) then
            do j = jp11, my1
                do i = ip11, mx1
                    if(maskSL(i, j) == 1) then
                        nSLsrf(i, j) = 1                           ! Ocean Grid Pt
                        SLsrfl(i, j, 1) = 1.
                        if(mw > 1) then
                            do n = min(2, mw), nvx
                                SLsrfl(i, j, n) = 0.
                            enddo
                        endif
                    else
                        nSLsrf(i, j) = nvx                         ! Land  Grid Pt
                        do n = 1, nvx
                            SLsrfl(i, j, n) = ifraTV(i, j, n)
                            SLsrfl(i, j, n) = SLsrfl(i, j, n) * 0.01
                        enddo
                    endif

                    ! +--Initialization of z0(Sastrugi)
                    ! +  ==============================

                    ! Influence of the Angle(Wind,Sastrugi) (Andreas, 1995, CCREL report 95-16)
                    ! -------------------------------------------------------------------------

#ifdef ZA
                    ua_0BS(i, j) = uairDY(i, j, mz)
                    va_0BS(i, j) = vairDY(i, j, mz)
#endif

                    !  Sastrugi Height
                    !  ---------------

                    do n = 1, mw
#ifdef SZ
                        Z0SaBS(i, j, n) = 0.
#endif
                        do nt = 1, ntavSL
                            SLn_z0(i, j, n, nt) = 0.5e-6
                            SLn_b0(i, j, n, nt) = 0.5e-6
                            SLn_r0(i, j, n, nt) = 0.5e-6
                        enddo
                    enddo
                enddo
            enddo

        endif

#ifdef ZA
        ! Influence of the Angle(Wind,Sastrugi) (Andreas, 1995, CCREL report 95-16)
        ! -------------------------------------------------------------------------
        FracBS = exp(-dt__SV / 43200.)
#endif

#ifdef OR
        ! +--Initialization of z0(Orography Roughness)
        ! +  =========================================
        do k = 1, mw
            do j = 1, my
                do i = 1, mx
                    SL_z0(i, j, k) = min(SL_z0(i, j, k), zsigma(mz) / 3.)
                    SLzoro(i, j, k) = min(SLzoro(i, j, k), zsigma(mz) / 3.)
                enddo
            enddo
        enddo
#endif

        ! +--Ocean Status
        ! +  ============
        VarSST = 0.
#ifdef OP
        VarSST = 1.
#endif
        FixSST = 1.-VarSST
        SSTnud = exp(-dt__SV / 2.592e6)  ! SST    Nudging:
        ! +...                                              ! e-folding time: 30 Days
        ! +
        if(itexpe == 0) then
            do j = jp11, my1
                do i = ip11, mx1
                    if(maskSL(i, j) > 0 .and. ifraTV(i, j, 1) < 100) then
                        write(6, 6000) i, j,(ifraTV(i, j, n), n=1, nvx)
6000                    format(' WARNING on Grid Point', 2i4, ' Mosaic = (', 3i4, &
                               '): ISLANDS must(will) be excluded')
                        do n = 1, nvx
                            ifraTV(i, j, n) = 0
                            ivegTV(i, j, n) = 0
                        enddo
                        ifraTV(i, j, 1) = 100
                    endif
                enddo
            enddo

            ! +--Prescription from SST
            ! +  ---------------------
            Tfr_LB = TocnSI
#ifdef RE
            Tfr_LB = 271.35 + epsi
#endif
            do j = jp11, my1
                do i = ip11, mx1
                    FraOcn = (TsolTV(i, j, 1, 1) - Tfr_LB) / TSIdSV! Open Ocean
                    FraOcn = 1.-sicsIB(i, j)                   ! Prescribed
                    FraOcn = min(unun, FraOcn)                !      Fract.
                    FraOcn = max(OcnMin, FraOcn)                !
                    ! New  Ocean
                    SLsrfl(i, j, 1) = (1 - maskSL(i, j)) * SLsrfl(i, j, 1) &
                                      + maskSL(i, j) * FraOcn        !
                    SrfSIC = SLsrfl(i, j, 2) ! Old  Sea Ice
                    SIc0OK = max(zero, sign(unun, SrfSIC - epsi)) !
                    ! New  Sea Ice
                    SLsrfl(i, j, 2) = (1 - maskSL(i, j)) * SLsrfl(i, j, 2) &
                                      + maskSL(i, j) * (1.-FraOcn)       !
                    SIceOK = max(zero, sign(unun, SLsrfl(i, j, 2) &
                                            - epsi)) !
                    ifra_t = ifraTV(i, j, 1) + ifraTV(i, j, 2) ! OCN  Fract.
                    ifraTV(i, j, 1) = SLsrfl(i, j, 1) * 100.          !
                    ifraTV(i, j, 1) = min(ifraTV(i, j, 1), ifra_t)       !
                    ifraTV(i, j, 2) = ifra_t - ifraTV(i, j, 1) !

                    ! +--Sea-Ice Vertical Discretization
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    nssSNo(i, j, 2) = &
                        nssSNo(i, j, 2) * (1 - maskSL(i, j)) &
                        + (nssSNo(i, j, 2) * SIc0OK &
                           + 3 * (1.-SIc0OK) * SIceOK) * maskSL(i, j)

                    nisSNo(i, j, 2) = &
                        nisSNo(i, j, 2) * (1 - maskSL(i, j)) &
                        + (nisSNo(i, j, 2) * SIc0OK &
                           + 3 * (1.-SIc0OK) * SIceOK) * maskSL(i, j)
                    issSNo(i, j, 2) = nisSNo(i, j, 2)

                    do l = 1, nsno
                        dzsSNo(i, j, 2, l) = &
                            dzsSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (dzsSNo(i, j, 2, l) * SIc0OK &
                               + dzSIce(min(4, l)) * (1.-SIc0OK) * SIceOK) * maskSL(i, j)

                        tisSNo(i, j, 2, l) = &
                            tisSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (tisSNo(i, j, 2, l) * SIc0OK &
                               + TsolTV(i, j, 1, 1) * (1.-SIc0OK)) * maskSL(i, j)

                        rosSNo(i, j, 2, l) = &
                            rosSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (rosSNo(i, j, 2, l) * SIc0OK &
                               + ro_Ice * (1.-SIc0OK) * SIceOK) * maskSL(i, j)

                        g1sSNo(i, j, 2, l) = &
                            g1sSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (g1sSNo(i, j, 2, l) * SIc0OK &
                               + G1_dSV * (1.-SIc0OK) * SIceOK) * maskSL(i, j)

                        g2sSNo(i, j, 2, l) = &
                            g2sSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (g2sSNo(i, j, 2, l) * SIc0OK &
                               + 30.*(1.-SIc0OK) * SIceOK) * maskSL(i, j)

                        nhsSNo(i, j, 2, l) = &
                            nhsSNo(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + istdSV(2) * maskSL(i, j)
                    enddo
                    do l = 1, llx
                        TsolTV(i, j, 2, l) = &
                            TsolTV(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + (TsolTV(i, j, 2, l) * SIc0OK &
                               + TsolTV(i, j, 1, l) * (1.-SIc0OK)) * maskSL(i, j)

                        eta_TV(i, j, 2, l) = &
                            eta_TV(i, j, 2, l) * (1 - maskSL(i, j)) &
                            + eta_TV(i, j, 2, l) * SIc0OK * maskSL(i, j)
                        ! +...                            No Pore in Ice => No Water
                    enddo

#ifdef WI
                    write(6, 6001) jdarGE, labmGE(mmarGE), iyrrGE &
                        , jhurGE, minuGE, jsecGE, TsolTV(i, j, 1, 1) &
                        , FraOcn, ifraTV(i, j, 1), TsolTV(i, j, 2, 1) &
                        , nisSNo(i, j, 2), nssSNo(i, j, 2)
6001                format(/, 98('_'), &
                            /, i3, '-', a3, '-', i4, 3(':', i2), &
                            2x, 'T OCN = ', f7.3, 4x, '% OCN = ', f7.3, '(', i3, ')', &
                            2x, 'T ICE = ', f7.3, &
                            /, 42x, 'NbIce = ', i3, 11x, 'NbSno = ', i3)
#endif

                enddo
            enddo
        endif

        ! +--Soil Upward IR Flux
        ! +  ===================

        if(itexpe == 0) then
            do j = jp11, my1
                do i = ip11, mx1
                    ! Upward IR Flux
                    IR_aux = -eps0SL(i, j) * stefan * TairSL(i, j)**4
                    do n = 1, nvx
                        IRsoil(i, j, n) = IR_aux
                    enddo
                    ! +--Water Vapor Flux Limitor
                    ! +  ========================
#ifdef VX
                    do n = 1, nLimit
                        WV__SL(i, j, n) = 1.
                    enddo
#endif
                enddo
            enddo

            ! +--SBL  Characteristics
            ! +  ====================
            do nt = 1, ntaver
                do j = 1, my
                    do i = 1, mx
                        V_0aSL(i, j, nt) = ssvSL(i, j, mz)
                        do n = 1, nvx
                            dT0aSL(i, j, n, nt) = tairDY(i, j, mz) - tsrfSL(i, j, n)
                        enddo
                    enddo
                enddo
            enddo
        endif

        ! +--OUTPUT Files Definition
        ! +  =======================
#ifdef v0
        open(unit=50, status='unknown', file='PHY_SISVAT.v0')
        rewind 50
#endif
        if(mmy <= 1 .and. mw > mw0) then
            open(unit=51, status='unknown', file='Dsagrr.OUT')
            rewind 51
            write(51, 5100)
5100        format(/, ' Simple Disagregation Model', &
                    /, ' ==========================')
        endif

        iwr = 0
        do ipt = 1, iptx
            if(IOi_TV(ipt) == 0) IOi_TV(ipt) = imez
            if(IOj_TV(ipt) == 0) IOi_TV(ipt) = jmez
            do n = 1, nvx
                iwr = 1 + iwr
                if(iwr <= nb_wri) then
                    no__SV(iwr) = 0
                    i___SV(iwr) = IOi_TV(ipt)
                    j___SV(iwr) = IOj_TV(ipt)
                    n___SV(iwr) = n
                endif
            enddo
        enddo

        ! +--Initialization of V10 Interpolation
        ! +  ===================================
        if(zsigma(1) > 10.) then
            k = 0
301         continue
            k = k + 1
            if(zsigma(k) < 10 .OR. k > mz) go to 300
            go to 301
300         continue
            kSBL = k

            if(kSBL == mz) then
                ! 0.002: typical Z0
                rSBL10 = log(10./0.002) &
                         / log(zsigma(kSBL) / 0.002)    !
            else
                rSBL10 = (10.-zsigma(kSBL)) &
                         / (zsigma(kSBL - 1) - zsigma(kSBL))
            endif
        else
            kSBL = mz
            rSBL10 = 1.
        endif

        erprev = 0.
        qbs_HY = 0.
        ! +   ++++++
    endif
    ! +   ++++++       +++  INITIALISATION:  END  +++

    ! +--Preparation    of V10 Interpolation
    ! +  ===================================

    if(kSBL == mz) then
        do j = 1, my
            do i = 1, mx
                VV__10(i, j) = rSBL10 * ssvSL(i, j, kSBL)
            enddo
        enddo
    else
        do j = 1, my
            do i = 1, mx
                VV__10(i, j) = ssvSL(i, j, kSBL) &
                               + rSBL10 * (ssvSL(i, j, kSBL - 1) - ssvSL(i, j, kSBL))
            enddo
        enddo
    endif

    ! +--Preparation of OUTPUT
    ! +  =====================

    ! do n = 1, nvx
    !     do j = jp11, my1
    !         do i = ip11, mx1
    !             WKxyz1(i, j, n) = 0.
    !         end do
    !     end do
    ! end do
    ! do iwr = 1, nb_wri
    !     WKxyz1(i___SV(iwr), j___SV(iwr), n___SV(iwr)) = iwr
    ! end do

    ! +--Update Sea-Ice    Fraction
    ! +  ==========================

    if(reaLBC) then
        ! +  ******
        call INIsic(ihamr_SIS, nhamr_SIS, newsicSI)
        ! +  ******
    endif

    ! +--Update Green Leaf Fraction
    ! +  ==========================
#ifdef GP
    glfFIX = .true.
#endif
    if(vegmod .and. reaLBC .and. .not. glfFIX) then
        ! +
        ! +  ******
        call INIglf(ihamr_SIS, nhamr_SIS, newglfSIS)
        ! +  ******
        ! +
    endif

    ! +--SISVAT Time   Dependant Variables
    ! +  =================================

#ifdef VX
    ! +--Water Vapor Flux Limitor
    ! +  ------------------------
    do n = 1, nLimit - 1
        do j = jp11, my1
            do i = ip11, mx1
                WV__SL(i, j, n) = WV__SL(i, j, n + 1)
            enddo
        enddo
    enddo
    do j = jp11, my1
        do i = ip11, mx1
            uqstar = max(abs(SLuqs(i, j)), epsi) * sign(1., SLuqs(i, j))
            WV__SL(i, j, n) &
                = TUkvh(i, j, mmz1) * (qvDY(i, j, km2(mz)) - qvDY(i, j, mmz1)) &
                  / (uqstar * (zsigma(km2(mz)) - zsigma(mmz1)))
        enddo
    enddo
    do j = jp11, my1
        do i = ip11, mx1
            WVaLim(i, j) = 0.
            do n = 1, nLimit
                WVaLim(i, j) = WVaLim(i, j) + WV__SL(i, j, n)
            enddo
            WVaLim(i, j) = WVaLim(i, j) / nLimit
        enddo
    enddo
#endif

    ! +--Simple "Mosaic" Rain Disagregation Model
    ! +  ----------------------------------------

    if(mmy <= 1 .and. mw > mw0) then !

        ! +--White Noise Generator
        ! +  ~~~~~~~~~~~~~~~~~~~~~
        if(jhurGE == 6 .and. minuGE == 0 .and. jsecGE == 0) then
            rtime = tairDY(imez, jmez, mz) * 1.e3
            ntime = int(rtime)
            rtime = (rtime - ntime) * 1.e3
            ntime = rtime
            ntime = mod(ntime, mw) + 1
        endif

        ! +--Averaged Soil Humidity
        ! +  ~~~~~~~~~~~~~~~~~~~~~~
        do j = 1, my
            do i = 1, mx
                ! Averaged Soil Humidity
                WKxy1(i, j) = 0.
                do n = 1, mw
                    do k = -nsol, 0
                        WKxy1(i, j) = WKxy1(i, j) &
                                      + eta_TV(i, j, n, 1 - k) * dz_dSV(k)
                    enddo
                enddo
                WKxy1(i, j) = WKxy1(i, j) / (mw * zz_dSV)

                ! +--Rain Distribution
                ! +  ~~~~~~~~~~~~~~~~~
                !  Rain Persistance over "wetter" surfaces
                ! (Taylor et al., 1997, MWR 125, pp.2211-2227)
                ! Normalization Factor
                WKxy2(i, j) = 0.
                do n = 1, mw
                    ! Rain Distribution Arg.
                    ! dry ===> sparse Rain
                    ! Persistance   Impact
                    rr__DR(i, j, n) = (mod(mw - n + ntime, mw) + 1) &
                                      / (mw * WKxy1(i, j) * WKxy1(i, j)) &
                                      * eta_TV(i, j, n, 1) / WKxy1(i, j)
                    rr__DR(i, j, n) = min(rr__DR(i, j, n) &
                                          , argmax * 0.1)
                    ! Rain Distribution
                    rr__DR(i, j, n) = exp(-rr__DR(i, j, n))
                    ! mw0 basic Mosaics
                    k = (n - 1) / mw0
                    k = k * mw0 + 1
                    ! Rain Distribution Arg.
                    rr__DR(i, j, n) = rr__DR(i, j, k)
                    ! Normalization Factor
                    WKxy2(i, j) = WKxy2(i, j) &
                                  + rr__DR(i, j, n)        !
                enddo                           !

            enddo
        enddo

        do j = 1, my
            do i = 1, mx
                do n = 1, mw
                    rr__DR(i, j, n) = rr__DR(i, j, n) * mw &
                                      / WKxy2(i, j)
                enddo
            enddo
        enddo

        if(mod(jdarGE, 3) == 0 .and. jhurGE == 6 .and. &
           minuGE == 0 .and. &
           jsecGE == 0 .and. &
           mmy <= 1) then
            do i = 1, mx
                if(isolSL(i, 1) > 2) then
                    write(51, 5101) jdarGE, labmGE(mmarGE), iyrrGE, jhurGE, &
                        i,(eta_TV(i, 1, n, 1), n=1, mw)
5101                format(i3, '-', a3, '-', i4, ':', i2, i6, 15f6.3, /,(21x, 15f6.3))
                    write(51, 5102) &
                        ntime, isolSL(i, 1),(rr__DR(i, 1, n), n=1, mw)
5102                format(i12, 3x, i6, 15f6.2, /,(21x, 15f6.2))
                endif
            enddo
            write(51, 5103)
5103        format(111('-'))
        endif
    else
        do j = 1, my
            do i = 1, mx
                do n = 1, mw
                    rr__DR(i, j, n) = 1.
                enddo
            enddo
        enddo
    endif

    !     - Interpolation of temp. and spec. hum. on sub_grid - *CL*
    !     --------------------------------------------------------------

    do i = 1, mx; do j = 1, my
            do n = 1, nsx
                tairDY_int(i, j, n) = tairDY(i, j, mz)
                qvDY_int(i, j, n) = qvDY(i, j, mz)
                tairDY_2D(i, j) = tairDY(i, j, mz)
                qvDY_2D(i, j) = qvDY(i, j, mz)
            enddo
        enddo; 
    enddo

    if(mw == 5) then
        ! +       ************************************************************
        call interp_subpix(tairDY_2D, tairDY_int, 1, -0.01, 0.05 &
                           , gradTM)
        ! +       ************************************************************

        ! +       ************************************************************
        call interp_subpix(qvDY_2D, qvDY_int, 2, -1.0, 1.0 &
                           , gradQM)
        ! +       ************************************************************
    endif

    ! +--Grid Averages
    ! +  -------------

    !$OMP PARALLEL DO default(shared) &
    !$OMP private(i,j,k,n,l,ikl,isl,isn,nt,Ua_min,d_snow,SnowOK, &
    !$OMP         rhAir,FraOcn,SrfSIC,SIc0OK,SIceOK,ifra_t, &
    !$OMP         S_Eros,SnEros,k2i,k2j,k2n,itphys) &
    !$OMP schedule(dynamic)

    do j = jp11, my1
        do i = ip11, mx1
            dt__SV = dt
            ! if(tairdy(i, j, mz)>273.15.and.&
            !         max(nssSNo(i, j, 1), nssSNo(i, j, mw))>1.)&
            !         dt__SV = dt / real(ntphys)

            ! if(isolSL(i, j)<=2)         dt__SV = dt ! sea or ice
            ! if(i<=n7.or.j<=n7.or.i>=mx - n6.or.j>=my - n6)&
            !         dt__SV = dt

            ! +--Surface Fall Line Slope
            ! +  -----------------------
            ! +
            if(SnoMod) then
                if(mx == 1 .and. my == 1) then
                    ! Normalized Decay of the
                    ! Surficial Water Content
                    !(Zuo and Oerlemans 1996, J.Glacio. 42, 305--317)
                    SWfSNo(1, 1) = exp(-dt__SV / (c1_zuo + c2_zuo * exp(-c3_zuo * slopTV(1, 1))))
                else
                    ! Normalized Decay of the
                    ! Surficial Water Content
                    !(Zuo and Oerlemans 1996, J.Glacio. 42, 305--317)
                    SWfSNo(i, j) = exp(-dt__SV / (c1_zuo + c2_zuo * exp(-c3_zuo * slopTV(i, j))))
                endif
            endif

#ifdef WR
            ifrVER(i, j) = 0
#endif
            albeSL(i, j) = 0.
            eps0SL(i, j) = 0.
            Upw_IR(i, j) = 0.
            SLlmo(i, j) = 0.
            SLuus(i, j) = 0.
            SLuts(i, j) = 0.
            SLuqs(i, j) = 0.
            uss_HY(i, j) = 0.
            qsrfHY(i, j) = 0.
#ifdef iso
            do wiso = 1, niso
                qsrfHY_iso(wiso, i, j) = 0.
                SLuqs_iso(wiso, i, j) = 0.
            enddo
#endif
            TairSL(i, j) = 0.
            draiTV(i, j) = 0.
#ifdef TC
            uqTC(i, j, 1) = 0.
            qsTC(i, j, 1) = 0.
#endif

            dzsnSV = 0.
            ro__SV = 0.

#ifdef ZA
            FracBS = exp(-dt__SV / 43200.)
#endif

            ! +--Sastrugi Height decreased by Precipitation if V < 6 m/s (Kotlyakov, 1961)
            ! + --------------------------------------------------------------------------

#ifdef SZ
            dsastr(i, j) = max(0.00,(snowHY(i, j) - sno0HY(i, j)) &
                               / max(0.05, 0.104 * sqrt(max(0.00, VV__10(i, j) - 6.00))))
#endif

            ! Influence of the Angle(Wind,Sastrugi) (Andreas, 1995, CCREL report 95-16)
            ! -------------------------------------------------------------------------

#ifdef ZA
            S_Eros = max(zero, sign(unun, -uss_HY(i, j) - eps9))
            SnEros = max(zero, sign(unun, uss_HY(i, j) + eps9))
            VVs_BS(i, j) = &
                SnEros * VVs_BS(i, j) &
                + S_Eros * (VVs_BS(i, j) * FracBS + &
                            VV__10(i, j))
            RRs_BS(i, j) = &
                SnEros * RRs_BS(i, j) &
                + S_Eros * (RRs_BS(i, j) * FracBS + 1.0)
            DDs_BS(i, j) = &
                SnEros * DDs_BS(i, j) &
                + S_Eros * DDs_BS(i, j) * FracBS &
                + ((vairDY(i, j, mz) * (uairDY(i, j, mz) - ua_0BS(i, j)) &
                    - uairDY(i, j, mz) * (vairDY(i, j, mz) - va_0BS(i, j)))) &
                / (degrad * max(0.3, ssvSL(i, j, mz) * ssvSL(i, j, mz)))
            if(DDs_BS(i, j) > 360.) DDs_BS(i, j) = DDs_BS(i, j) - 360.
            if(DDs_BS(i, j) < 0.) DDs_BS(i, j) = DDs_BS(i, j) + 360.
#endif

            ! +--Grid  Point   Dependant Variables
            ! +  ---------------------------------

            ! +--Verification of Vectorization
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
#ifdef VR
            ij2 = 0
            do n = 1, mw
                ij0ver(i, j, n) = 0
                ij_ver(i, j, n) = 0
                ijdver(i, j, n) = 0
            enddo
#endif

            ! +--SISVAT Variables Update
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^
            ptopSV = ptopDY
            do n = 1, mw
                if(SLsrfl(i, j, n) /= 0) then
                    do ikl = 1, klonv
                        k2i(ikl) = i
                        k2j(ikl) = j
                        k2n(ikl) = n
                        ! Work pt. i Coord.
                        ii__SV(ikl) = i
                        ! Work pt. j Coord.
                        jj__SV(ikl) = j
                        ! Work pt. n Coord.
                        nn__SV(ikl) = n

#ifdef wz
                        if(ikl == 1 .and. jsecGE == 0) write(6, 6659)
6659                    format(20x, '   dsn_SV   us__SV   Z0SaSi   Z0Sa_N' &
                               , '   Z0SaSV   Z0m_Sn   Z0m_SV')
#endif

                        ! +--Atmospheric Forcing (INPUT)
                        ! +  ^^^^^^^^^^^^^^^^^^^ ^^^^^
                        ! zSBLSV [m]
                        zSBLSV = z__SBL
                        za__SV(ikl) = (gplvDY(i, j, mz) &
                                       - gplvDY(i, j, mzz)) * grvinv
                        VV__SV(ikl) = ssvSL(i, j, mz)
                        VV10SV(ikl) = VV__10(i, j)
#ifdef ZA
                        VVs_SV(ikl) = VVs_BS(i, j) / RRs_BS(i, j)
                        DDs_SV(ikl) = max(zero, DDs_BS(i, j) - 180.) &
                                      + 180.*min(unun, zero - min(zero, DDs_BS(i, j) - 180.)) &
                                      + min(zero, DDs_BS(i, j) - 180.)
#endif
                        Ua_min = epsi
#ifdef VM
                        Ua_min = 0.2 * sqrt(za__SV(ikl))
#endif
                        VV__SV(ikl) = max(Ua_min, ssvSL(i, j, mz))
                        TaT_SV(ikl) = tairDY_int(i, j, n)
                        ExnrSV(ikl) = pkDY(i, j, mz)
                        ! [kg/m3] *CL*
                        rhT_SV(ikl) = (pstDYn(i, j) + ptopDY) * 1.e3 &
                                      / (tairDY_int(i, j, n) * RDryAi)
                        QaT_SV(ikl) = qvDY_int(i, j, n)
                        tsrf_SV(ikl) = tsrfSL(i, j, n)
                        pst_SV(ikl) = pstDY(i, j)
#ifdef VX
                        ! Water  Vapor Flux Limitor
                        dQa_SV(ikl) = max(0., 1.-WVaLim(i, j)) * dtDiff / zsigma(mz)
#endif
                        qsnoSV(ikl) = 0.+min(demi, qsHY(i, j, mz))

                        ! +--Energy Fluxes (INPUT)
                        ! +  ^^^^^^^^^^^^^ ^^^^^
                        ! cos(zenith.Dist.)
                        coszSV(ikl) = max(czemin, czenGE(i, j))
                        !    downward Solar
                        sol_SV(ikl) = RAdsol(i, j)
                        !    downward IR
                        IRd_SV(ikl) = RAd_ir(i, j)

                        ! +--Water  Fluxes (INPUT)
                        ! +  ^^^^^^^^^^^^^ ^^^^^
                        ! [m/s] -> [mm/s] = [kg/m2/s]
                        drr_SV(ikl) = (rainHY(i, j) - rai0HY(i, j)) * 1.e3 &
                                      * rr__DR(i, j, n) / dt__SV
                        ! Only SnowFall
                        d_snow = snowHY(i, j) - sfa0HY(i, j)
                        ! Erosion NOT incl.
                        dsn_SV(ikl) = d_snow * 1.e3 / dt__SV
                        ! Correction
                        SnowOK = &
                            max(zero, sign(unun, qsHY(i, j, mz) - epsi)) &
                            * max(zero, min(unun,(rain_snow_limit - 1 - tairDY_int(i, j, n))))
                        dsn_SV(ikl) = dsn_SV(ikl) + drr_SV(ikl) * SnowOK
                        drr_SV(ikl) = drr_SV(ikl) * (1.-SnowOK)
#ifdef BS
                        ! Erosion
                        ! dsnbSV is used and modified in SISVAT_BSn,
                        ! then used for Buffer Layer Update
                        dbs_Ac(ikl) = 0.
                        dbs_Er(ikl) = 0.
#endif
                        ! +--Soil/Canopy (INPUT)
                        ! +  ^^^^^^^^^^^ ^^^^^
                        ! Land/Sea   Mask
                        LSmask(ikl) = 1 - maskSL(i, j)
                        ! Soil       Type
                        isotSV(ikl) = isolTV(i, j)
                        ! Soil Drainage
                        iWaFSV(ikl) = iWaFTV(i, j)
                        ! Fall Line Slope
                        slopSV(ikl) = atan(slopTV(i, j))
                        ! Soil Albedo
                        alb0SV(ikl) = AlbSTV(i, j)
                        ! Bare ice Albedo
                        aiceSV(ikl) = aiceSL(i,j)
                        ! Vegetation Type
                        ivgtSV(ikl) = ivegTV(i, j, n)
                        ! LAI
                        LAI0SV(ikl) = alaiTV(i, j, n)
                        ! Green Leaf Frac.
                        glf0SV(ikl) = glf_TV(i, j, n)
                        wem_SV(ikl) = 0.
                        wer_SV(ikl) = 0.
                        wee_SV(ikl, :) = 0.

                        ! +--Energy Fluxes (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^ ^^^^^^^^^^^^
                        ! Cloudiness
                        cld_SV(ikl) = cld_SL(i, j)
                        ! Soil upward IR
                        IRs_SV(ikl) = IRsoil(i, j, n)
                        ! Monin-Obukhov L.
                        LMO_SV(ikl) = SLlmol(i, j, n)
                        ! Frict. Velocity
                        us__SV(ikl) = SLuusl(i, j, n)
                        ! u*T*
                        uts_SV(ikl) = SLutsl(i, j, n)

                        ! +--Water  Fluxes (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^ ^^^^^^^^^^^^
                        ! u*q*
                        uqs_SV(ikl) = SLuqsl(i, j, n)
#ifdef iso
                        do wiso = 1, niso
                            uqs_SV_iso(wiso, ikl) = SLuqsl_iso(wiso, i, j, n)
                        enddo
#endif
#ifdef AE
                        ! u*_th
                        usthSV(ikl) = SaltSN(i, j, n)
#endif

                        ! +--Soil/Canopy (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^ ^^^^^^^^^^^^
                        ! Moment.Roughn.L.
                        Z0m_SV(ikl) = SL_z0(i, j, n)
                        ! Heat   Roughn.L.
                        Z0h_SV(ikl) = SL_r0(i, j, n)
#ifdef OR
                        ! Orogr. Roughn.L.
                        Z0roSV(ikl) = SLzoro(i, j, n)
#endif
                        ! Vegetation Temp.
                        TvegSV(ikl) = TvegTV(i, j, n)
                        ! Canopy SnowCover
                        snCaSV(ikl) = CaSnTV(i, j, n)
                        ! Canopy RainWater
                        rrCaSV(ikl) = CaWaTV(i, j, n)
                        ! Vegetation Pot.
                        psivSV(ikl) = psivTV(i, j, n)
                        do isl = -nsol, 0
                            ! Soil Temperature
                            TsisSV(ikl, isl) = TsolTV(i, j, n, 1 - isl)
                            ! Soil Humidity
                            eta_SV(ikl, isl) = eta_TV(i, j, n, 1 - isl)
                        enddo
                    enddo

                    ! +--Snow Roughness (INPUT/OUTPUT)
                    ! +  ^^^^^^^^^^^^^^ ^^^^^^^^^^^^
                    do ikl = 1, klonv
                        ! +--Verification of Vectorization
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
#ifdef VR
                        if(ijn <= ijnmax) then
                            ij0ver(i, j, n) = ij0ver(i, j, n) + 1
                            ijdver(i, j, n) = ijdver(i, j, n) + ij
                        endif
#endif

                        Z0mmSV(ikl) = 0.                         !
                        Z0emSV(ikl) = 0.                         !
                        Z0hmSV(ikl) = 0.                         !
                        do nt = 1, ntavSL
                            Z0mmSV(ikl) = Z0mmSV(ikl) &
                                          + SLn_z0(i, j, n, nt)
                            Z0emSV(ikl) = Z0emSV(ikl) &
                                          + SLn_b0(i, j, n, nt)
                            Z0hmSV(ikl) = Z0hmSV(ikl) &
                                          + SLn_r0(i, j, n, nt)
                        enddo
                        !  z0(Mom., Box Av.)
                        Z0mmSV(ikl) = min(Z0mmSV(ikl) / ntavSL, zsigma(mz) / 3.)
                        !  z0(Eros, Box Av.)
                        Z0emSV(ikl) = Z0emSV(ikl) / ntavSL
                        !  z0(Heat, Box Av.)
                        Z0hmSV(ikl) = Z0hmSV(ikl) / ntavSL

#ifdef SZ
                        !  z0(Sastrugi h)
                        Z0SaSV(ikl) = Z0SaBS(i, j, n)
#endif
#ifdef SZ
                        ! dz0(Sastrugi dh)
                        dz0_SV(ikl) = .01 * dsastr(i, j) * max(2 - n, 0)
#endif

                        ! +--V,  dT(a-s)    Time Moving Averages
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        do nt = 1, ntaver
                            V__mem(ikl, nt) = V_0aSL(i, j, nt)
                            T__mem(ikl, nt) = dT0aSL(i, j, n, nt)
                        enddo

                        do nt = 1, ntaver - 1
                            V__mem(ikl, nt) = V__mem(ikl, nt + 1)
                            T__mem(ikl, nt) = T__mem(ikl, nt + 1)
                        enddo
                        V__mem(ikl, ntaver) = VV__SV(ikl)
                        T__mem(ikl, ntaver) = TaT_SV(ikl) - tsrfSL(i, j, n)

                        VVmmem(ikl) = 0.0
                        dTmmem(ikl) = 0.0
                        do nt = 1, ntaver
                            VVmmem(ikl) = VVmmem(ikl) + V__mem(ikl, nt)
                            dTmmem(ikl) = dTmmem(ikl) + T__mem(ikl, nt)
                        enddo
                        VVmmem(ikl) = VVmmem(ikl) / ntaver
                        dTmmem(ikl) = dTmmem(ikl) / ntaver

                        ! +--u*, u*T*, u*s* Time Moving Averages
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef AM
                        do nt = 1, ntaver
                            u__mem(ikl, nt) = u_0aSL(i, j, n, nt)
#ifdef AT
                            uT_mem(ikl, nt) = uT0aSL(i, j, n, nt)
#endif
#ifdef AS
                            us_mem(ikl, nt) = us0aSL(i, j, n, nt)
#endif
                        enddo
#endif
                    enddo

#ifdef BS
                    do ikl = 1, klonv
                        Z0emBS(i, j, n) = Z0emSV(ikl)
                    enddo
#endif

                    ! +--Snow Pack (INPUT/OUTPUT)
                    ! +  ^^^^^^^^^ ^^^^^^^^^^^^
                    do ikl = 1, klonv
#ifdef AO
                        !weightA0= 1 if full MAR, 1> weightao>0 if transition, 0 for NEMO coupling area
                        AOmask = weightao_al(i, j)
                        albAOsisv(ikl) = albAO(i, j, n)
#endif
                        ! Snow Buffer Lay.
                        BufsSV(ikl) = snohSN(i, j, n)
                        dsn_SV(ikl) = dsn_SV(ikl) &
                                      + max(BufsSV(ikl) - SMndSV, 0.) &
                                      / dt__SV
                        BufsSV(ikl) = min(BufsSV(ikl), SMndSV)
                        ! Snow Buffer dens.
                        BrosSV(ikl) = BrosSN(i, j, n)
                        ! Snow Buffer D./S.
                        BG1sSV(ikl) = BG1sSN(i, j, n)
                        ! Snow Buffer S./S.
                        BG2sSV(ikl) = BG2sSN(i, j, n)
                        ! Nb Snow/Ice L
                        isnoSV(ikl) = min(nsno, max(0, nssSNo(i, j, n)))
                        ! Nb Supr.Ice L
                        ispiSV(ikl) = min(isnoSV(ikl), max(0, issSNo(i, j, n)))
                        ! Nb      Ice L
                        iiceSV(ikl) = min(isnoSV(ikl), max(0, nisSNo(i, j, n)))
                        ! Non-Erod.*Thick.
                        zWEcSV(ikl) = zWEcSN(i, j, n)
                        ! Surficial Water
                        rusnSV(ikl) = SWaSNo(i, j, n)
                        ! Surficial Wat.St.
                        SWS_SV(ikl) = SWSSNo(i, j, n)
                        ! Normalized Decay
                        SWf_SV(ikl) = SWfSNo(i, j)
                    enddo
                    do ikl = 1, klonv
                        do isn = 1, nsno
                            ! istoSV [-]
                            istoSV(ikl, isn) = nhsSNo(i, j, n, isn)
                            ! dzsnSV [m]
                            dzsnSV(ikl, isn) = dzsSNo(i, j, n, isn)
                            ! ro__SV [kg/m3]
                            ro__SV(ikl, isn) = rosSNo(i, j, n, isn)
                            ! eta_SV [m3/m3]
                            eta_SV(ikl, isn) = wasSNo(i, j, n, isn)
                            ! TsisSV [K]
                            TsisSV(ikl, isn) = tisSNo(i, j, n, isn)
                            ! G1snSV [-]        [-]
                            G1snSV(ikl, isn) = max(-G1_dSV, min(G1_dSV, g1sSNo(i, j, n, isn)))
                            ! G2snSV [-] [0.0001 m]
                            G2snSV(ikl, isn) = max(-G1_dSV, min(G1_dSV, g2sSNo(i, j, n, isn)))
                            ! agsnSV [day]
                            agsnSV(ikl, isn) = agsSNo(i, j, n, isn)
                        enddo
                    enddo

                    ! Grid Point                                                (OUTPUT)
                    ! ^^^^^^^^^^                                                 ^^^^^^
#ifdef wx
                    kSV_v1 = 0
#endif

                    do ikl = 1, klonv
                        HFraSV(ikl) = 0.                         ! Frazil Thickness

                        ! +--RunOFF Intensity                                       (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^^^^                                        ^^^^^^^^^^^^
                        RnofSV(ikl) = 0.                         ! RunOFF Intensity
                        RuofSV(ikl, :) = 0.                         ! RunOFF Intensity
                        zn4_SV(ikl) = 0.
                        zn5_SV(ikl) = 0.

                        ! Grid Point                                                (OUTPUT)
                        ! ^^^^^^^^^^                                                 ^^^^^^
#ifdef wx
                        if(i == iSV_v1 .and. j == jSV_v1 .and. n == nSV_v1) kSV_v1 = ikl
#endif
                        ! lwriSV(ikl) = WKxyz1(i, j, n)
#ifdef BW
                        if(lwriSV(ikl) /= 0 .and. iterun > 0) then
                            noUNIT = no__SV(lwriSV(ikl))
                            write(noUNIT, 5012)
5012                        format(/, 1x)
                            write(noUNIT, 5013)
5013                        format(' -----+--------+--------+--------+--------+', &
                                   '--------+--------+--------+--------+--------+', &
                                   '--------+')
                            write(noUNIT, 5014)
5014                        format('    n |     z  |     qs |      V |        |', &
                                   '     T  | TKE^0.5|        |        |        |', &
                                   '        |', &
                                   /, '      |    [m] | [g/kg] |  [m/s] |        |', &
                                   '    [K] |  [m/s] |        |        |        |', &
                                   '        |')
                            BlowST = 0.
                            k = 0
5011                        continue
                            k = k + 1
                            if(k > mz) go to 5010
                            if(grvinv * gplvDY(i, j, k) - sh(i, j) < 100.) then
                                BlowST = BlowST + ssvSL(i, j, k) * qsHY(i, j, k) &
                                         * pstDY(i, j) * dsigm1(k) * 1.e3 * grvinv
                                write(noUNIT, 5015) mzz - k, grvinv * gplvDY(i, j, k) - sh(i, j), &
                                    1.e3 * qsHY(i, j, k), ssvSL(i, j, k), tairDY(i, j, k), &
                                    sqrt(ect_TE(i, j, k))
5015                            format(i5, ' |', f7.2, ' |', f7.3, ' |', f7.2, ' |', &
                                       8x, '|', f7.2, ' |', f7.3, ' |', 4(8x, '|'))
                            endif
                            go to 5011
5010                        continue
                            SnowSB = snohSN(i, j, n)
                            if(nssSNo(i, j, n) > 0) then
                                do isn = max(0, nssSNo(i, j, n)), nssSNo(i, j, n)
                                    SnowSB = SnowSB &
                                             + dzsSNo(i, j, n, isn) * rosSNo(i, j, n, isn)
                                enddo
                            endif
                            write(noUNIT, 5016) BlowST, SnowSB
5016                        format(' * TRANSPORT = ', e12.3, ' kg/m/s', 8x, '|', &
                                   ' * BUDGET    = ', f12.6, ' mm w.e.|', 2(8x, '|'))
                            write(noUNIT, 5013)
                        endif
#endif
#ifdef v0
                        ! OUTPUT, for Stand-Alone VERIFICATION
                        ! ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        if(i >= 1 .and. i <= mx) then
                            write(50, 5001) iterun, i, j, n, nvc, ikl, &
                                za__SV(ikl), VV__SV(ikl), TaT_SV(ikl), &
                                rhT_SV(ikl), QaT_SV(ikl), qsnoSV(ikl), &
                                coszSV(ikl), sol_SV(ikl), IRd_SV(ikl), &
                                drr_SV(ikl), dsn_SV(ikl), dbs_SV(ikl), &
                                LSmask(ikl), isotSV(ikl), alb0SV(ikl), &
                                IRs_SV(ikl), &
                                ivgtSV(ikl), LAI0SV(ikl), glf0SV(ikl), &
                                TvegSV(ikl), LMO_SV(ikl), us__SV(ikl), &
                                uqs_SV(ikl), uts_SV(ikl), uss_SV(ikl), &
                                snCaSV(ikl), rrCaSV(ikl), psivSV(ikl)
5001                        format(/, 'c #INFO   iterun          = ', i15, &
                                    /, 'c #INFO   i,j,n           = ', 3i5, &
                                    /, 'c #INFO   nvc             = ', i15, &
                                    /, 'c #INFO   ikl             = ', i15, &
                                    /, '          za__SV(ikl)     = ', e15.6, &
                                    /, '          VV__SV(ikl)     = ', e15.6, &
                                    /, '          TaT_SV(ikl)     = ', e15.6, &
                                    /, '          rhT_SV(ikl)     = ', e15.6, &
                                    /, '          QaT_SV(ikl)     = ', e15.6, &
                                    /, '          qsnoSV(ikl)     = ', e15.6, &
                                    /, '          coszSV(ikl)     = ', e15.6, &
                                    /, '          sol_SV(ikl)     = ', e15.6, &
                                    /, '          IRd_SV(ikl)     = ', e15.6, &
                                    /, '          drr_SV(ikl)     = ', e15.6, &
                                    /, '          dsn_SV(ikl)     = ', e15.6, &
                                    /, '          dbs_SV(ikl)     = ', e15.6, &
                                    /, '          LSmask(ikl)     = ', i15, &
                                    /, '          isotSV(ikl)     = ', i15, &
                                    /, '          alb0SV(ikl)     = ', e15.6, &
                                    /, '          IRs_SV(ikl)     = ', e15.6, &
                                    /, '          ivgtSV(ikl)     = ', i15, &
                                    /, '          LAI0SV(ikl)     = ', e15.6, &
                                    /, '          glf0SV(ikl)     = ', e15.6, &
                                    /, '          TvegSV(ikl)     = ', e15.6, &
                                    /, '          LMO_SV(ikl)     = ', e15.6, &
                                    /, '          us__SV(ikl)     = ', e15.6, &
                                    /, '          uqs_SV(ikl)     = ', e15.6, &
                                    /, '          uts_SV(ikl)     = ', e15.6, &
                                    /, '          uss_SV(ikl)     = ', e15.6, &
                                    /, '          snCaSV(ikl)     = ', e15.6, &
                                    /, '          rrCaSV(ikl)     = ', e15.6, &
                                    /, '          psivSV(ikl)     = ', e15.6)
                            do isl = -nsol, 0
                                write(50, 5002) isl, TsisSV(ikl, isl), isl, eta_SV(ikl, isl)
5002                            format('          TsisSV(ikl,', i2, ')  = ', e15.6, &
                                       '          eta_SV(ikl,', i2, ')  = ', e15.6)
                            enddo
                            do isl = 1, nsno
                                write(50, 5003) isl, TsisSV(ikl, isl), isl, dzsnSV(ikl, isl)
5003                            format('          TsisSV(ikl,', i2, ')  = ', e15.6, &
                                       '          dzsnSV(ikl,', i2, ')  = ', e15.6)
                            enddo
                        endif
#endif
                    enddo

                    ! +--SISVAT Execution
                    ! +  ^^^^^^^^^^^^^^^^
                    !          write(daHost,'(i2,a3,i4,i3,2(a1,i2))')
                    !     .          jdarGE,labmGE(mmarGE),iyrrGE,
                    !     .          jhurGE,chb,minuGE,chb,jsecGE
#ifdef wz
                    write(6, 6660) jdarGE, mmarGE, iyrrGE, jhurGE, minuGE, jsecGE
6660                format(2(i2, '-'), 2i4, 2(':', i2), 3x, $)
#endif

                    do ikl = 1, klonv
#ifdef BS
                        ! dbs_SV = Maximum potential erosion amount [kg/m2]
                        ! => Upper bound for eroded snow mass
                        dbs_SV(ikl) = blowSN(i, j, n)
#endif
                        uss_SV(ikl) = SLussl(i, j, n) ! u*qs* (only for Tv in sisvatesbl.f)
#ifdef BS
                        ! dsnbSV is the drift fraction of deposited snow updated in sisvat.f
                        ! will be used for characterizing the Buffer Layer
                        ! (see update of  Bros_N, G1same, G2same, zroOLD, zroNEW)
                        if(dsn_SV(ikl) > eps12 .and. erprev(i, j, n) > eps9) then
                            ! BS neglib. at kb ~100 magl)
                            dsnbSV(ikl) = 1.0 - min(qsHY(i, j, kB) &
                                                    / max(qshy(i, j, mz), eps9), unun)
                            dsnbSV(ikl) = max(dsnbSV(ikl), erprev(i, j, n) / dsn_SV(ikl))
                            dsnbSV(ikl) = max(0., min(1., dsnbSV(ikl)))
                        else
                            dsnbSV(ikl) = 0.
                        endif
                        if(n == 1) qbs_HY(i, j) = dsnbSV(ikl)
#endif

                    enddo
                    ! +       ************
                    ! do itPhys = 1, max(1, nint(dt / dt__SV))
                    call SISVAT(1)
                    !     do ikl = 1, klonv
                    !         dsn_SV(ikl) = 0.
                    !         drr_SV(ikl) = 0.
                    !     end do
                    ! end do
                    ! +       ************

                    do ikl = 1, klonv
                        SLussl(i, j, n) = 0.
#ifdef BS
                        ! Effective erosion ~u*qs* from previous time step
                        SLussl(i, j, n) = (-dbs_ER(ikl)) / (dt * rhT_SV(ikl))
                        ! New max. pot. Erosion [kg/m2] (further bounded in sisvat_bsn.f)
                        blowSN(i, j, n) = dt * uss_SV(ikl) * rhT_SV(ikl)
                        erprev(i, j, n) = dbs_Er(ikl) / dt__SV
#endif
                    enddo

                    ! +--MAR    Variables Update
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^
                    do ikl = 1, klonv
#ifdef VR
                        ! +--Verification of Vectorization
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        ij2 = ij2 + 1
                        ijdver(i, j, n) = ijdver(i, j, n) - ij2
                        ij_ver(i, j, n) = ij_ver(i, j, n) + 1
#endif
                        ! +--Energy Fluxes (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^ ^^^^^^^^^^^^
                        ! Soil upward IR
                        IRsoil(i, j, n) = IRs_SV(ikl)
                        ! Monin-Obukhov L.
                        SLlmol(i, j, n) = LMO_SV(ikl)
                        ! Frict. Velocity
                        SLuusl(i, j, n) = us__SV(ikl)
                        ! u*T*
                        SLutsl(i, j, n) = uts_SV(ikl)
                        ! Sens.H.Flux T-Der.
                        SLdSdT(i, j, n) = dSdTSV(ikl)
#ifdef NC
                        ! +--Energy Fluxes (OUTPUT/NetCDF)
                        ! +  ^^^^^^^^^^^^^ ^^^^^^^^^^^^^
                        ! Absorb.Sol.Rad.
                        SOsoNC(i, j, n) = SOsoKL(ikl)
                        ! Absorb.IR  Rad.
                        IRsoNC(i, j, n) = IRsoKL(ikl)
                        ! HS
                        HSsoNC(i, j, n) = HSsoKL(ikl)
                        ! HL
                        HLsoNC(i, j, n) = HLsoKL(ikl)
                        ! Evaporation
                        HLs_NC(i, j, n) = HLs_KL(ikl)
                        ! Transpiration
                        HLv_NC(i, j, n) = HLv_KL(ikl)
#endif
                        ! +--Water Fluxes (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        ! u*q*
                        SLuqsl(i, j, n) = uqs_SV(ikl)
#ifdef iso
                        do wiso = 1, niso
                            ! todo : compute uqs_SV_iso in sisvat
                            ! todo : SLuqsl_iso -> uqs_SV -> HLv_sv + HLs_sv (sisvat.f90)
                            ! todo : HLv_sv (sisvat_tgv) and HLs_sv (sisvat_tso) -> HL___D : store Rsnow ?
                            ! SLuqsl_iso(wiso, i, j, n) = uqs_SV_iso(wiso, ikl)
                            SLuqsl_iso(wiso, i, j, n) = Rdefault(wiso) * uqs_SV(ikl)
                        enddo
#endif
                        ! Latn.H.Flux T-Der.
                        SLdLdT(i, j, n) = dLdTSV(ikl)
                    enddo
                    do ikl = 1, klonv
#ifdef AE
                        ! u*_th
                        SaltSN(i, j, n) = usthSV(ikl)
#endif
                        ! +--Soil/Canopy (INPUT/OUTPUT)
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^
                        ! Moment.Roughn.L.
                        SL_z0(i, j, n) = Z0m_SV(ikl)
                        ! Heat   Roughn.L.
                        SL_r0(i, j, n) = Z0h_SV(ikl)
                        ! sq.root Contr.Drag
                        cdmSL(i, j, n) = rCDmSV(ikl)
                        ! sq.root Contr.Drag
                        cdhSL(i, j, n) = rCDhSV(ikl)
                        ! Vegetation Temp.
                        TvegTV(i, j, n) = TvegSV(ikl)
                        ! Canopy SnowCover
                        CaSnTV(i, j, n) = snCaSV(ikl)
                        ! Canopy RainWater
                        CaWaTV(i, j, n) = rrCaSV(ikl)
                        ! Vegetation Pot.
                        psivTV(i, j, n) = psivSV(ikl)
                        do isl = -nsol, 0
                            ! Soil Humidity
                            eta_TV(i, j, n, 1 - isl) = eta_SV(ikl, isl)
                            ! Soil Temperature
                            TsolTV(i, j, n, 1 - isl) = TsisSV(ikl, isl)
                        enddo
                    enddo
                    ! +--Snow Roughness (INPUT/OUTPUT)
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                    do ikl = 1, klonv
                        do nt = 1, ntavSL - 1
                            SLn_z0(i, j, n, nt) = SLn_z0(i, j, n, nt + 1)
                            SLn_b0(i, j, n, nt) = SLn_b0(i, j, n, nt + 1)
                            SLn_r0(i, j, n, nt) = SLn_r0(i, j, n, nt + 1)
                        enddo
                    enddo
                    do ikl = 1, klonv
                        ! z0(Momentum)
                        SLn_z0(i, j, n, ntavSL) = Z0mnSV(ikl)
                        ! z0(Mom., Erosion)
                        SLn_b0(i, j, n, ntavSL) = Z0enSV(ikl)
                        ! z0(Heat)
                        SLn_r0(i, j, n, ntavSL) = Z0hnSV(ikl)
#ifdef SZ
                        ! z0(Sastrugi h)
                        Z0SaBS(i, j, n) = Z0SaSV(ikl)
#endif
                        ! +--V,  dT(a-s)    Time Moving Averages
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        do nt = 1, ntaver
                            V_0aSL(i, j, nt) = V__mem(ikl, nt)
                            dT0aSL(i, j, n, nt) = T__mem(ikl, nt)
                        enddo
#ifdef AM
                        ! +--u*, u*T*, u*s* Time Moving Averages
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        do nt = 1, ntaver
                            u_0aSL(i, j, n, nt) = u__mem(ikl, nt)
#ifdef AT
                            uT0aSL(i, j, n, nt) = uT_mem(ikl, nt)
#endif
#ifdef AS
                            us0aSL(i, j, n, nt) = us_mem(ikl, nt)
#endif
                        enddo
#endif
                    enddo

#ifdef BD
                    ! +--Dust   Fluxes (INPUT/OUTPUT)
                    ! +  ^^^^^^^^^^^^^
                    do ikl = 1, klonv
                        ! Snow Free  Surface
                        ! DUST       Erosion
                        ! Tuning Factor (2D)
                        SLubsl(i, j, n) = (1 - min(1, isnoSV(ikl))) &
                                          * uss_SV(ikl) &
                                          * max(1,(2 - mmy) * 3)
                    enddo
#endif
                    ! +--Snow Pack (INPUT/OUTPUT)
                    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^
                    do ikl = 1, klonv
                        snohSN(i, j, n) = BufsSV(ikl)            ! Snow Buffer Lay.
                        BrosSN(i, j, n) = BrosSV(ikl)            ! Snow Buffer dens.
                        BG1sSN(i, j, n) = BG1sSV(ikl)            ! Snow Buffer D./S.
                        BG2sSN(i, j, n) = BG2sSV(ikl)            ! Snow Buffer S./S.
                        nssSNo(i, j, n) = isnoSV(ikl)            ! Nb Snow/Ice Lay.
                        issSNo(i, j, n) = ispiSV(ikl)            ! Nb Supr.Ice Lay.
                        nisSNo(i, j, n) = iiceSV(ikl)            ! Nb      Ice Lay.
                        zWE_SN(i, j, n) = zWE_SV(ikl)            ! Current  *Thick.
                        zWEcSN(i, j, n) = zWEcSV(ikl)            ! Non-Erod.*Thick.
                        hSalSN(i, j, n) = hSalSV(ikl)            ! Salt.Layer Height
                        SWaSNo(i, j, n) = rusnSV(ikl)            ! Surficial Water
                        SWSSNo(i, j, n) = SWS_SV(ikl)            ! Surficial Wat.St.
                    enddo
                    do ikl = 1, klonv
                        do isn = 1, nsno
                            nhsSNo(i, j, n, isn) = istoSV(ikl, isn)        !            [-]
                            dzsSNo(i, j, n, isn) = dzsnSV(ikl, isn)        !            [m]
                            rosSNo(i, j, n, isn) = ro__SV(ikl, isn)        !        [kg/m3]
                            wasSNo(i, j, n, isn) = eta_SV(ikl, isn)        !        [m3/m3]
                            tisSNo(i, j, n, isn) = TsisSV(ikl, isn)        !            [K]
                            g1sSNo(i, j, n, isn) = G1snSV(ikl, isn)        ! [-]        [-]
                            g2sSNo(i, j, n, isn) = G2snSV(ikl, isn)        ! [-] [0.0001 m]
                            agsSNo(i, j, n, isn) = agsnSV(ikl, isn)        !          [day]
                        enddo
                    enddo

                    do ikl = 1, klonv
                        EVSU(i, j, n, :) = wee_SV(ikl, :)  ! Evapo/Sublimation
                        WKxyz4(i, j, n) = wem_SV(ikl)      ! Melting
                        WKxyz5(i, j, n) = wer_SV(ikl)      ! Refreezing
#ifdef BS
                        weerIB(i, j, n) = weerIB(i, j, n) + dbs_Er(ikl) ! BS erosion
#endif
                        zn4IB(i, j, n) = zn4IB(i, j, n) + zn4_SV(ikl)
                        zn5IB(i, j, n) = zn5IB(i, j, n) + zn5_SV(ikl)

                        ! +--Radiative Properties (OUTPUT)
                        ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                        albxSL(i, j, n) = alb_SV(ikl)            ! Mosaic Albedo
                        !Commented !AO_CK 20/02/2020 (bad variable)
                        !c #AO.                          *(1-maskSL(i,j))           !
                        !c #AO.                          +    albAO(i,j,n)          ! Mosaic AlbedoNEMO
                        !c #AO.                          *   maskSL(i,j)            !
                        WKxyz6(i, j, n) = emi_SV(ikl)            ! Mosaic Emissivity
                        WKxyz7(i, j, n) = IRu_SV(ikl)            ! Mosaic Upw.IR
                        WKxyz8(i, j, n) = qSalSV(ikl)            ! Saltating Partic.
                        hfra(i, j, n) = HFraSV(ikl)            ! Frazil  Thickness
                        Rnof(i, j, n) = RnofSV(ikl)            ! Run OFF Intensity
                        Ruof(i, j, n, :) = RuofSV(ikl, :)          ! Run OFF Intensity

                        if(n == 1) then
                            alb1IB(i, j) = alb1SV(ikl)
                            alb2IB(i, j) = alb2SV(ikl)
                            alb3IB(i, j) = alb3SV(ikl)
                        endif

                    enddo
                else
                    Rnof(i, j, n) = 0.
                    Ruof(i, j, n, :) = 0.
                    EVSU(i, j, n, :) = 0.
                endif
            enddo

            ! +--Surface Temperature: Prescription of relevant Medium (Snow, precribed SST)
            ! +  ==========================================================================

            do isl = -nsol, 0                             !
                ! +--Open Ocean
                ! +  ----------
                eta_TV(i, j, 1, 1 - isl) = &
                    eta_TV(i, j, 1, 1 - isl) * (1 - maskSL(i, j)) &
                    + maskSL(i, j)          ! Sea: Humidity:=1
                TsolTV(i, j, 1, 1 - isl) = &
                    ! Soil Temperature
                    ! Prescribed   SST
                    (TsolTV(i, j, 1, 1 - isl) * (1 - maskSL(i, j)) &
                     + sst_LB(i, j) * maskSL(i, j))
#ifdef OP
                ! CAa : OP is weird and not activated
                TsolTV(i, j, 1, 1 - isl) = TsolTV(i, j, 1, 1 - isl) * FixSST + &
                                           (TsolTV(i, j, 1, 1 - isl) &
                                            !~Prescribed   SST
                                            + (sst_LB(i, j) - &
                                               ! (Nudging)
                                               TsolTV(i, j, 1, 1 - isl)) * maskSL(i, j) * SSTnud &
                                            ! Interactive  SST
                                            ) * VarSST
#endif
#ifdef AO
                ! +--Sea Ice
                ! +  -------
                ! AO_CK 20/02/2020 tissno is now modified before the call of sisvat (in oasis_2_mar.f)
                ! Sea: Humidity:=0
                eta_TV(i, j, 2, 1 - isl) = eta_TV(i, j, 2, 1 - isl) * (1 - maskSL(i, j))
                ! Soil Temperature + Prescribed ST
                TsolTV(i, j, 2, 1 - isl) = &
                    (TsolTV(i, j, 2, 1 - isl) * (1 - maskSL(i, j)) &
                     + 271.2 * maskSL(i, j))
#endif
            enddo

            do n = 1, mw
                ! Surf.Temperature
                tsrfSL(i, j, n) = TsolTV(i, j, n, 1) &
                                  * (1 - min(1, nssSNo(i, j, n))) &
                                  + tisSNo(i, j, n, max(1, nssSNo(i, j, n))) &
                                  * min(1, nssSNo(i, j, n))
#ifdef NC
                eta_NC(i, j, n) = 0.
                do isl = -nsol, 0
                    ! Soil Moisture
                    eta_NC(i, j, n) = eta_NC(i, j, n) &
                                      + eta_TV(i, j, n, 1 - isl) * dz_dSV(isl)
                enddo
#endif
            enddo

            ! +--Mosaic Cleaning
            ! +  ===============

            if(maskSL(i, j) == 1) then
                if(nssSNo(i, j, 1) > 0) then
                    nssSNo(i, j, 1) = 0
                    issSNo(i, j, 1) = 0
                    nisSNo(i, j, 1) = 0
                    do isl = 1, nsno
                        tisSNo(i, j, 1, isl) = 0.
                        dzsSNo(i, j, 1, isl) = 0.
                        rosSNo(i, j, 1, isl) = 0.
                        wasSNo(i, j, 1, isl) = 0.
                        g1sSNo(i, j, 1, isl) = 0.
                        g2sSNo(i, j, 1, isl) = 0.
                        agsSNo(i, j, 1, isl) = 0.
                        nhsSNo(i, j, 1, isl) = 0.
                    enddo
                endif
                if(SLsrfl(i, j, 2) < eps9 .and. tsrfSL(i, j, 2) /= tsrfSL(i, j, 1)) then
                    tsrfSL(i, j, 2) = tsrfSL(i, j, 1)
                    do isl = 1, nsol + 1
                        TsolTV(i, j, 2, isl) = TsolTV(i, j, 1, isl)
                    ENDdo !#n2
                    nssSNo(i, j, 2) = nssSNo(i, j, 1) * (1 - maskSL(i, j))
                    issSNo(i, j, 2) = issSNo(i, j, 1) * (1 - maskSL(i, j))
                    nisSNo(i, j, 2) = nisSNo(i, j, 1) * (1 - maskSL(i, j))
                    do isl = 1, nsno       !
                        tisSNo(i, j, 2, isl) = tisSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        dzsSNo(i, j, 2, isl) = dzsSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        rosSNo(i, j, 2, isl) = rosSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        wasSNo(i, j, 2, isl) = wasSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        g1sSNo(i, j, 2, isl) = g1sSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        g2sSNo(i, j, 2, isl) = g2sSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        agsSNo(i, j, 2, isl) = agsSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                        nhsSNo(i, j, 2, isl) = nhsSNo(i, j, 1, isl) * (1 - maskSL(i, j))
                    ENDdo !#n2
                endif    !#n2
            endif

            ! +--Grid Averages / Diagnostics
            ! +  ===========================

            ! +--Grid Averages                                                (OUTPUT)
            ! +  ^^^^^^^^^^^^^                                                 ^^^^^^
            do n = 1, mw
                wee_IB(i, j, n, :) = EvSU(i, j, n, :) + wee_IB(i, j, n, :)  ! evapotrans
                wem_IB(i, j, n) = WKxyz4(i, j, n) + wem_IB(i, j, n)  ! Melting
                wer_IB(i, j, n) = WKxyz5(i, j, n) + wer_IB(i, j, n)  ! Refreezing
                ! Runoff
                weu_IB(i, j, n) = Rnof(i, j, n) * dt__SV &
                                  + weu_IB(i, j, n)
                weo_IB(i, j, n, :) = Ruof(i, j, n, :) * dt__SV &
                                     + weo_IB(i, j, n, :)

#ifdef WR
                ifrVER(i, j) = ifrVER(i, j) + ifraTV(i, j, n)
#endif
                ! albeSL : Grid   Albedo + Mosaic Albedo
                albeSL(i, j) = albeSL(i, j) + SLsrfl(i, j, n) * albxSL(i, j, n)
                ! eps0SL : Grid   Emissivity + Mosaic Emissivity
                eps0SL(i, j) = eps0SL(i, j) + SLsrfl(i, j, n) * WKxyz6(i, j, n)
                ! Upw_IR : + Mosaic Upw.IR
                Upw_IR(i, j) = Upw_IR(i, j) + SLsrfl(i, j, n) * WKxyz7(i, j, n)
                ! SLlmo : +  Mosaic Mon.Ob.
                SLlmo(i, j) = SLlmo(i, j) + SLsrfl(i, j, n) * SLlmol(i, j, n)
                ! SLuus : Grid   u* +  Mosaic u*
                SLuus(i, j) = SLuus(i, j) + SLsrfl(i, j, n) * SLuusl(i, j, n)
                ! SLuts : Grid   u*T* +  Mosaic u*T*
                SLuts(i, j) = SLuts(i, j) + SLsrfl(i, j, n) * SLutsl(i, j, n)
                ! SLuqs : Grid   u*q* +  Mosaic u*q*
                SLuqs(i, j) = SLuqs(i, j) + SLsrfl(i, j, n) * SLuqsl(i, j, n)
#ifdef iso
                do wiso = 1, niso
                    SLuqs_iso(wiso, i, j) = SLuqs_iso(wiso, i, j) + SLsrfl(i, j, n) * SLuqsl_iso(wiso, i, j, n)
                enddo
#endif
#ifdef BS
                ! Grid   u*s* + Mosaic u*s*
                uss_HY(i, j) = uss_HY(i, j) + SLsrfl(i, j, n) * SLussl(i, j, n)
#endif
                !        u*s*
                ! +...NO !    SLussl(i,j,n)       = uss_SV(ikl)
                ! +           Upper Update = wrong Source of Atmospher.Snow!
#ifdef BS
                ! Salt.Part.Concent., only if there is a snow layer
                qsrfHY(i, j) = qsrfHY(i, j) + SLsrfl(i, j, n) * WKxyz8(i, j, n) * min(1, nssSNo(i, j, n))
#endif
#ifdef PO
                ! Frazil  Thickness
                HFraPO(i, j) = HFraPO(i, j) + SLsrfl(i, j, n) * HFra(i, j, n)
#endif
                ! Surface Air Temperature
                TairSL(i, j) = TairSL(i, j) + SLsrfl(i, j, n) * tsrfSL(i, j, n)
                ! Run OFF Intensity
                draiTV(i, j) = draiTV(i, j) + SLsrfl(i, j, n) * Rnof(i, j, n)
#ifdef TC
                ! Grid   u*b* + Mosaic u*b*
                uqTC(i, j, 1) = uqTC(i, j, 1) + SLsrfl(i, j, n) * SLubsl(i, j, n)
                ! Salt.Part.Concent.
                qsTC(i, j, 1) = qsTC(i, j, 1) + SLsrfl(i, j, n) * WKxyz8(i, j, n) * (1 - min(1, nssSNo(i, j, n)))
#endif
            enddo
            sno0HY(i, j) = snowHY(i, j)
            pktaSL(i, j) = TairSL(i, j) / exp(cap * log(pstDY(i, j) + ptopDY))
            ! Brightness Temp.
            tviRA(i, j) = sqrt(sqrt(Upw_IR(i, j) / stefan))
            ! Air    Densitity
            rhAir = rolvDY(i, j, mz) * 1.e3
            ! Sensible Heat Flux
            hsenSL(i, j) = -SLuts(i, j) * rhAir * cp
            ! Surf.Specif.Humid. [to adapt over soil]
            qvapSL(i, j) = qvsiDY(i, j, mzz)
#ifdef iso
            ! todo : compute Riso of the vapor in equilibrium with the surface
            ! todo : get Riso of snow / water + fractcalk
            ! todo : Riso of snow from dzsSNo = dzsnSV
            ! todo : Riso of water from wasSNo_iso (negligible?) and SWaSNo_iso (?) + mixing of the two ?
            do wiso = 1, niso
                qvapSL_iso(wiso, i, j) = Rdefault(wiso) * qvapSL(i, j)
            enddo
#endif
            ! Latent   Heat Flux
            hlatSL(i, j) = -SLuqs(i, j) * rhAir * Lv_H2O
            ! Total    Evaporat. [mm w.e.]
            evapTV(i, j) = evapTV(i, j) - SLuqs(i, j) * rhAir * dt__SV
#ifdef iso
            ! todo : evapTV_iso -> compute Riso
            ! todo : need to track SLuqs -> SLuqsl_iso -> see above
            do wiso = 1, niso
                evapTV_iso(wiso, i, j) = Rdefault(wiso) * evapTV(i, j)
            enddo
#endif
            ! Integrated Run OFF
            runoTV(i, j) = runoTV(i, j) + draiTV(i, j) * dt__SV
            firmSL(i, j) = Upw_IR(i, j)

            ! +--Sea-Ice Ice Floe Size
            ! +  =====================

            ! +--Prescription from SST
            ! +  ---------------------

            if(VarSST <= epsi .and. maskSL(i, j) == 1) then
                ! Prescribed from SST (not use anymore by anyone CK 20/02/20)
                ! FraOcn          =    (TsolTV(i,j,1,1)-Tfr_LB)/TSIdSV
                ! Prescribed from rea/ocean model
                FraOcn = 1.-sicsIB(i, j)

                ! UpperLimit
                FraOcn = min(unun, FraOcn)
                ! LowerLimit
                FraOcn = max(OcnMin, FraOcn)
                ! New Ocean   (ocean % at current dt)
                SLsrfl(i, j, 1) = FraOcn
                ! Old Sea Ice (SIC    at previous dt)
                SrfSIC = SLsrfl(i, j, 2)
                ! 1 if SICdt-1 > epsi ; 0 if SICdt-1<epsi
                SIc0OK = max(zero, sign(unun, SrfSIC - epsi))
                ! New Sea ice (SIC at current dt)
                SLsrfl(i, j, 2) = 1.-FraOcn
                ! 1 if SICdt > epsi ; 0 if Sidct < epsi
                SIceOK = max(zero, sign(unun, SLsrfl(i, j, 2) - epsi))
                ! OCN Fraction
                ifra_t = ifraTV(i, j, 1) + ifraTV(i, j, 2)
                ifraTV(i, j, 1) = SLsrfl(i, j, 1) * 100.
                ifraTV(i, j, 1) = min(ifraTV(i, j, 1), ifra_t)
                ifraTV(i, j, 2) = ifra_t - ifraTV(i, j, 1)

                ! +--Sea-Ice Vertical Discretization
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                ! if no sic dt 0, if SIC dt then
                ! if SICdt-1 then at least 1 layer or number of the dt-1 layers >1
                ! if not SICdt-1 then 3 layers
                nssSNo(i, j, 2) = &
                    (max(1 &
                         , nssSNo(i, j, 2)) * SIc0OK &
                     + 3 * (1.-SIc0OK) * SIceOK)
                ! +
                nisSNo(i, j, 2) = &
                    (max(1 &
                         , nisSNo(i, j, 2)) * SIc0OK &
                     + 3 * (1.-SIc0OK) * SIceOK)
                issSNo(i, j, 2) = nisSNo(i, j, 2)
                ! +
                do l = 1, nsno
                    ! If SIC dt then
                    ! if SICdt-1 then at least the bottom layer
                    ! with 10cm or any value > 10cm (dt-1 thickness for all the other old layers)
                    ! if not SICdt-1 then 4 layers of 0.5,0.05,0.001,0.0/ m (=> 3 real layers)
                    dzsSNo(i, j, 2, l) = &
                        (max &
                         (SIc_OK(min(2, l)) * SIcMIN &
                          , dzsSNo(i, j, 2, l)) * SIc0OK &
                         + dzSIce(min(4, l)) * (1.-SIc0OK) * SIceOK)
                    ! +
                    tisSNo(i, j, 2, l) = &
                        (tisSNo(i, j, 2, l) * SIc0OK &
                         + TsolTV(i, j, 1, 1) * (1.-SIc0OK))
                    ! +
                    ! If sicdt then
                    ! If SICdt-1 then at least the bottom layer
                    ! with 920 or any value of the sea ice pack >920 (dt-1 ro for all the other old layers)
                    ! if not SICdt-1 then 920 for all new layers
                    rosSNo(i, j, 2, l) = &
                        (max &
                         (SIc_OK(min(2, l)) * ro_Ice &
                          , rosSNo(i, j, 2, l)) * SIc0OK &
                         + ro_Ice * (1.-SIc0OK) * SIceOK)
                    ! +
                    g1sSNo(i, j, 2, l) = &
                        (g1sSNo(i, j, 2, l) * SIc0OK &
                         + G1_dSV * (1.-SIc0OK) * SIceOK)
                    ! +
                    g2sSNo(i, j, 2, l) = &
                        (g2sSNo(i, j, 2, l) * SIc0OK &
                         + 30.*(1.-SIc0OK) * SIceOK)
                    ! +
                    nhsSNo(i, j, 2, l) = istdSV(2)
#ifdef SInew
                    nhsSNo(i, j, 2, l) = nhsSNo(i, j, 2, l) + &
                                         (nhsSNo(i, j, 2, l) * SIc0OK + &
                                          istdSV(2) * (1.-SIc0OK) * SIceOK) * maskSL(i, j)
#endif

                enddo

#ifdef AO
                ! COUPLING AO_CK! 20/02/2020
                if(weightAO_sit(i, j) == 0) then !full NEMO
                    ! coupling of sea ice thickness from NEMO
                    if(aohic > 0) then !coupling time step
                        zntot = 0.
                        do l = 1, nsno
                            ! zntot of ice layers
                            if(rosSNo(i, j, 2, l) > 900 .and. dzsSNo(i, j, 2, l) > 0) then
                                zntot = zntot + dzsSNo(i, j, 2, l)
                            endif
                        enddo
                        do l = 1, nsno
                            ! NEMO minimal sea ice thickness is 10cm as in MAR
                            if(rosSNo(i, j, 2, l) > 900 .and. dzsSNo(i, j, 2, l) > 0 &
                               .and. hicAO(i, j) >= 0.1 .and. zntot > 0) then
                                dzsSNo(i, j, 2, l) = dzsSNo(i, j, 2, l) * hicAO(i, j) / zntot
                            endif
                        enddo
                    endif
                endif
                ! end coupling AO_CK SEA ICE THICKNESS 20/02/2020
                ! coupling of snow thickness on the sea ice from NEM
                ! full NEMO
                if(weightAO_snt(i, j) == 0) then
                    if(aohsn > 0) then !coupling time ste
                        if(SIc0OK == 1 .and. SIceOK == 1) then
                            ! Sea ice at previous time step and at the current time step
                            ! => changes the thickness of the snow by applying a ratio
                            zntot = 0.
                            do l = 1, nsno
                                ! zntot of snow layers
                                if(rosSNo(i, j, 2, l) < 900 .and. dzsSNo(i, j, 2, l) > 0) then
                                    zntot = zntot + dzsSNo(i, j, 2, l)
                                endif
                            enddo
                            do l = 1, nsno
                                if(rosSNo(i, j, 2, l) < 900 .and. dzsSNo(i, j, 2, l) > 0 &
                                   .and. zntot > 0 .and. nssSNo(i, j, 2) > 1) then
                                    if(hsnoAO(i, j) >= 0.005) then
                                        ! ratio new/old
                                        dzsSNo(i, j, 2, l) = dzsSNo(i, j, 2, l) * hsnoAO(i, j) / zntot
                                    else
                                        ! if NEMO snow thickness lower than minimal snow thickness in MAR => 0 ?
                                        dzsSNo(i, j, 2, l) = 0.
                                        nssSNo(i, j, 2) = max(3., nssSNo(i, j, 2) - 1.)
                                    endif
                                endif
                            enddo
                        endif
                        if(SIc0OK == 0 .and. SIceOK == 1 &
                           .and. hsnoAO(i, j) >= 0.005) then
                            !No sea ice at the previous time step but new sea ice at the current time step
                            !+1 one snow layer
                            nssSNo(i, j, 2) = nssSNo(i, j, 2) + 1.
                            ! snow thickness from NEMO
                            dzsSNo(i, j, 2, int(nssSNo(i, j, 2))) = hsnoAO(i, j)
                            ! temp neige/surface from NEMO
                            tisSNo(i, j, 2, int(nssSNo(i, j, 2))) = 270.
                            ! density from NEMO
                            rosSNo(i, j, 2, int(nssSNo(i, j, 2))) = 300.
                            ! G1 fresh snow
                            g1sSNo(i, j, 2, int(nssSNo(i, j, 2))) = G1_dSV
                            ! G2 fresh snow
                            g2sSNo(i, j, 2, int(nssSNo(i, j, 2))) = 30.
                            ! faceted cristal
                            nhsSNo(i, j, 2, int(nssSNo(i, j, 2))) = istdSV(1)
                        endif
                    endif
                endif
                ! end coupling AO_CK SNOW THICKNESS 20/02/2020
#endif

                do l = 1, llx
                    TsolTV(i, j, 2, l) = &
                        TsolTV(i, j, 2, l) * (1 - maskSL(i, j)) &
                        + (TsolTV(i, j, 2, l) * SIc0OK &
                           + TsolTV(i, j, 1, l) * (1.-SIc0OK)) * maskSL(i, j)
                    ! +
                    eta_TV(i, j, 2, l) = &
                        eta_TV(i, j, 2, l) * (1 - maskSL(i, j)) &
                        + eta_TV(i, j, 2, l) * SIc0OK * maskSL(i, j)
                    ! +...                            No Pore in Ice => No Water
                enddo
                ! +
#ifdef WI
                write(6, 6001) jdarGE, labmGE(mmarGE), iyrrGE &
                    , jhurGE, minuGE, jsecGE, TsolTV(i, j, 1, 1) &
                    , FraOcn, ifraTV(i, j, 1), TsolTV(i, j, 2, 1) &
                    , nisSNo(i, j, 2), nssSNo(i, j, 2)
#endif
                ! +
            endif

            ! +--Otherwise SST and FrLead have been computed in the Sea-Ice Polynya Model
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            ! +--Rainfall, Snowfall Time Integral at previous Time Step
            ! +  ------------------------------------------------------

            rai0HY(i, j) = rainHY(i, j)    ! Rainfall Time Integral
            sfa0HY(i, j) = snowHY(i, j)    ! Snowfall Time Integral

            !  Wind Horizontal Components         at previous Time Step
            !  --------------------------------------------------------

#ifdef ZA
            ua_0BS(i, j) = uairDY(i, j, mz)
            va_0BS(i, j) = vairDY(i, j, mz)
#endif

            ! +--Work Array Reset
            ! +  ================

            WKxy1(i, j) = 0.
            WKxy2(i, j) = 0.
            WKxy3(i, j) = 0.
            WKxy5(i, j) = 0.
            WKxy6(i, j) = 0.
            WKxy7(i, j) = 0.

            do k = 1, mw
                WKxyz1(i, j, k) = 0.
                WKxyz2(i, j, k) = 0.
                WKxyz3(i, j, k) = 0.
                WKxyz4(i, j, k) = 0.
                WKxyz5(i, j, k) = 0.
                WKxyz6(i, j, k) = 0.
                WKxyz7(i, j, k) = 0.
                WKxyz8(i, j, k) = 0.
            enddo

        ENDdo !i
    ENDdo !j
    !$OMP END PARALLEL DO

    ! +--Blown Snow/Dust Accumulation
    ! +  ============================

    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef wx
    if(lSV_v1 == 2) write(6, 6011) uss_HY(iSV_v1, jSV_v1) * 1.e3
6011 format(10x, 'After SISVAT(1): us* [mm/s] =', f9.3)
#endif

    !c #BS do i=ip11,mx1
    !c #BS do j=   1,my
    !c #BS        WKxy6(i     ,j     ) =
    !c #BS&      uss_HY(im1(i),j     )+2.0*uss_HY(i     ,j     )
    !c #BS& +    uss_HY(ip1(i),j     )
    !c #BS end do
    !c #BS end do

    !c #BS do j=jp11,my1
    !c #BS do i=ip11,mx1
    !c #BS        WKxy5(i     ,j     ) =    WKxy6(i     ,jm1(j))
    !c #BS        WKxy7(i     ,j     ) =    WKxy6(i     ,jp1(j))
    !c #BS end do
    !c #BS end do

    !c#BS do j=jp11,my1
    !c#BS do i=ip11,mx1
    !c#BS       uss_HY(i     ,j     ) =
    !c#BS.       WKxy7(i     ,j     )
    !c#BS.     + WKxy6(i     ,j     ) +    WKxy6(i     ,j     )
    !c#BS.     + WKxy5(i     ,j     )

    !  Previous three Loops Stand for the following unvectorized Loop:
    !       WKxy2(i,j) = uss_HY(im1(i),jp1(j))
    !    .         +2.d0*uss_HY(i     ,jp1(j)) +     uss_HY(ip1(i),jp1(j))
    !    .         +2.d0*uss_HY(im1(i),j)
    !    .         +4.d0*uss_HY(i     ,j)      +2.d0*uss_HY(ip1(i),j)
    !    .         +     uss_HY(im1(i),jm1(j))
    !    .         +2.d0*uss_HY(i     ,jm1(j)) +     uss_HY(ip1(i),jm1(j))
    !c#BS end do
    !c#BS end do

    !c #BD do i=1,mx
    !c #BD do j=1,my
    !c #BD   WKxy3(i,j) = uqTC(im1(i),jp1(j),1)
    !c #BD.         +2.d0*uqTC(i     ,jp1(j),1) +     uqTC(ip1(i),jp1(j),1)
    !c #BD.         +2.d0*uqTC(im1(i),j     ,1)
    !c #BD.         +4.d0*uqTC(i     ,j     ,1) +2.d0*uqTC(ip1(i),j     ,1)
    !c #BD.         +     uqTC(im1(i),jm1(j),1)
    !c #BD.         +2.d0*uqTC(i     ,jm1(j),1) +     uqTC(ip1(i),jm1(j),1)
    !c #BD end do
    !c #BD end do

    !c #BS do j=1,my
    !c #BS do i=1,mx
    !c #BS  WKxy5(i,j) = 0.
    !c #BS  WKxy6(i,j) = 0.
    !c #BS end do
    !c #BS end do

    !spatial smoothing commented - C.Amory BS 2018
    !c #BS do j=jp11,my1
    !c #BS do i=ip11,mx1
    !c #BS  do k=-1,1 ; do n=-1,1
    !c #BS     uu= 1 ; vv=1
    !c #BS     if(sign(1., uairdy(i,j,mz))/=sign(1.,real(k)))
    !c #BS.    uu=sqrt(abs(uairdy(i,j,mz)))
    !c #BS     if(sign(1., vairdy(i,j,mz))/=sign(1.,real(n)))
    !c #BS.    vv=sqrt(abs(vairdy(i,j,mz)))
    !c #BS     uu=max(1.,min(4.,uu))
    !c #BS     vv=max(1.,min(4.,vv))
    !c #BS                                 ww=1.
    !c #BS     if(n==0)                    ww=uu
    !c #BS     if(k==0)                    ww=vv
    !c #BS     if(n==0.and.k==0)           ww=(uu+vv)*2.
    !c #BS     if(abs(k)==1.and.abs(n)==1) ww=(uu+vv)/2.
    !c #BS     WKxy5(i,j)=WKxy5(i,j)+ww*uss_HY(i+k,j+n)
    !c #BS     WKxy6(i,j)=WKxy6(i,j)+ww
    !c #BS  end do ; end do
    !c #BS end do
    !c #BS end do
    !
    !
    !c #BS do j=jp11,my1
    !c #BS do i=ip11,mx1
    !c #BS  uss_HY(i,j)=WKxy5(i,j)/WKxy6(i,j)
    !c #BS end do
    !c #BS end do
    !end spatial smoothing

    snow_filter = .true.
    if(snow_filter .and. mod(iterun, 2) == 0) call sno_filtering

#ifdef BS
    do i = 1, mx
        do j = 1, my
            ! uss_HY(i,j)  = uss_HY(i,j)          * 62.5e-3
            ! snowHY(i,j)  = snowHY(i,j) + dt__SV * rolvDY(i,j,mz)*uss_HY(i,j) !BUGBUG
            weacIB(i, j, 1) = weacIB(i, j, 1) - dt__SV * rolvDY(i, j, mz) * uss_HY(i, j) &
                              * 1000.
#ifdef BD
            uqTC(i, j, 1) = WKxy3(i, j) * 62.5e-3
#endif
        enddo
    enddo
#endif

    ! OUTPUT in SISVAT at specified i,j,k,n (see assignation in PHY_SISVAT)
    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef wx
    if(lSV_v1 == 2) write(6, 6012) uss_HY(iSV_v1, jSV_v1) * 1.e3
6012 format(10x, 'After SISVAT(2): us* [mm/s] =', f9.3)
#endif

    ! +--Additional OUTPUT for VERIFICATION
    ! +  ----------------------------------
#ifdef WR
    do j = jp11, my1
        do i = ip11, mx1
            if(ifrVER(i, j) /= 100) write(6, 660) isolSL(i, j), i, j, ifrVER(i, j) &
                ,(ifraTV(i, j, n), n=1, nvx)
660         format(' WARNING: Mosaic', i2, ' (', 2i4, ') = ', i4, i6, 2i4)
        enddo
    enddo
    i = imez + 10.*111.111e3 / dx
    j = jmez
    write(6, 6060) itexpe, jdarGE, labmGE(mmarGE), iyrrGE &
        , jhurGE, minuGE, GElatr(i, j) / degrad &
        , tairDY(i, j, mz), virDY(i, j, mz), 1.e3 * rolvDY(i, j, mz) &
        , hsenSL(i, j), hlatSL(i, j), -86400.0 * SLuqs(i, j) &
        , 1.e3 * rainHY(i, j), evapTV(i, j), runoTV(i, j)
6060 format(i6, i3, '-', a3, '-', i4, ':', i2, ':', i2, f6.2, '?N', &
           f9.3, ' K', f6.3, f6.3, ' kg/m3', 2(f6.1, ' W/m2'), &
           f6.3, ' mm/day', 3(f9.3, ' mm'))
#endif
    ! +--Verification of Vectorization
    ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
#ifdef VR
    write(6, 6100)
6100 format(/, 'Verification of Vectorization: Before CALL')
    do n = mw, 1, -1
        do j = my, 1, -1
            write(6, 6110)(ij0ver(i, j, n), i=1, mx)
6110        format(132i1)
        enddo
        write(6, 6103)
6103    format(1x)
    enddo
    write(6, 6101)
6101 format(/, 'Verification of Vectorization: After  CALL')
    do n = mw, 1, -1
        do j = my, 1, -1
            write(6, 6110)(ij_ver(i, j, n), i=1, mx)
        enddo
        write(6, 6103)
    enddo
    do n = 1, mw
        do j = 1, my
            do i = 1, mx
                if(ijdver(i, j, n) /= 0 .and. ij_ver(i, j, n) /= 1) write(6, 6102) i, j, n, ijdver(i, j, n)
6102            format(' Vectorization ERROR on', 3i4, '   (', i6, ')')
            enddo
        enddo
    enddo
#endif

    if(.not. INI_SV) &
        INI_SV = .true.

    return
endsubroutine PHY_SISVAT_MP
