subroutine PBLtop(TKE_1D, HHH_1D, h__0SL, h__PSL, h__SSL)
    ! +---------------------------------------------------------------------------+
    ! |                                                               07-APR-2025 |
    ! |              PBLtop computes the height of the Primary   Seeing Layer PSL |
    ! |                                                Secondary Seeing Layer SSL |
    ! |                                                                           |
    ! |   INPUT:   TKE_1D: Turbulent Kinetic Energy                       [m2/s2] |
    ! |            HHH_1D: Height above the Surface                           [m] |
    ! |                                                                           |
    ! |   OUTPUT:  h__PSL: Height of the Primary   Seeing Layer               [m] |
    ! |            h__SSL: Height of the Secondary Seeing Layer               [m] |
    ! |                                                                           |
    ! +---------------------------------------------------------------------------+

    use mardim

    implicit none

    real TKE_1D(mz)
    real HHH_1D(mz)
    real h__PSL
    real h__SSL
    real h__0SL

    real TKEmin
    real TKEtop

    integer k, kmx, kzi

    logical RESET
    logical INTERP

    DATA TKEmin/1.e-6/

    INTERP = .false.

!     +--Height of the zero   Seeing Layer (0SL) (more conservative)
    ! +  ==========================================

    ! +--Search the lowest TKE maximum
    ! +  -----------------------------

    k = mz
    TKEtop = 0.01 * TKE_1D(k)
901 continue
    k = k - 1
    if(k <= mzabso) go to 900
    if(TKE_1D(k) < TKE_1D(k + 1) .and. &
       TKE_1D(k + 1) > TKEmin * 3.00) go to 900
    ! +                                    3.00     = 1/2 order of magnitude
    ! +        (in order to only detect a significant maximum)
    go to 901
900 continue
    kmx = k + 1
    TKEtop = 0.01 * TKE_1D(kmx)
    TKEtop = max(TKEmin * 1.50, TKEtop)
    ! +                                    1.50     = 1/4 order of magnitude

    ! +--Search (from above) the lowest TKE minimum above the lowest TKE maximum
    ! +  ------ (This mimimum may be                                ) ----------
    ! +         (either a  TRUE         minimum  => INTERP = .false.)
    ! +         (    or an arbitrary small value => INTERP = .true. )
    ! +         -----------------------------------------------------

    ! +--Index  of the layer containing the minimum
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    kzi = mzabso
    do k = mzabso, kmx
        if(TKE_1D(k) < TKEtop .OR. &
           TKE_1D(k) < TKE_1D(k - 1) * 0.75) then ! higher 0.75 is, lower PBL is
            kzi = k
            if(TKE_1D(k) < TKEtop) then
                INTERP = .true.
            else
                INTERP = .false.
            endif
        endif
    enddo

    ! +--Height of the                      minimum
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    k = kzi
    if(kzi <= mzabso + 1) then
        h__PSL = HHH_1D(mz)
    else
        if(INTERP) then
            h__0SL = HHH_1D(k + 1) &
                     + (HHH_1D(k) - HHH_1D(k + 1)) &
                     * (TKEtop - TKE_1D(k + 1)) &
                     / (TKE_1D(k) - TKE_1D(k + 1))
        else
            h__0SL = HHH_1D(k)
        endif
    endif

    h__0SL = min(h__0SL,    HHH_1D(3))
    h__0SL = max(HHH_1D(mz-1), h__0SL)

    INTERP = .false.
    ! +--Height of the Primary   Seeing Layer (PSL)
    ! +  ==========================================

    ! +--Search the lowest TKE maximum
    ! +  -----------------------------

    k = mz
    TKEtop = 0.01 * TKE_1D(k)
1001 continue
    k = k - 1
    if(k <= mzabso) go to 1000
    if(TKE_1D(k) < TKE_1D(k + 1) .and. &
       TKE_1D(k + 1) > TKEmin * 3.00) go to 1000
    ! +                                    3.00     = 1/2 order of magnitude
    ! +        (in order to only detect a significant maximum)
    go to 1001
1000 continue
    kmx = k + 1
    TKEtop = 0.01 * TKE_1D(kmx)
    TKEtop = max(TKEmin * 1.50, TKEtop)
    ! +                                    1.50     = 1/4 order of magnitude

    ! +--Search (from above) the lowest TKE minimum above the lowest TKE maximum
    ! +  ------ (This mimimum may be                                ) ----------
    ! +         (either a  TRUE         minimum  => INTERP = .false.)
    ! +         (    or an arbitrary small value => INTERP = .true. )
    ! +         -----------------------------------------------------

    ! +--Index  of the layer containing the minimum
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    kzi = mzabso
    do k = mzabso, kmx
        if(TKE_1D(k) < TKEtop .OR. &
           TKE_1D(k) < TKE_1D(k - 1) * 0.3) then
            kzi = k
            if(TKE_1D(k) < TKEtop) then
                INTERP = .true.
            else
                INTERP = .false.
            endif
        endif
    enddo

    ! +--Height of the                      minimum
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    k = kzi
    if(kzi <= mzabso + 1) then
        h__PSL = HHH_1D(mz)
    else
        if(INTERP) then
            h__PSL = HHH_1D(k + 1) &
                     + (HHH_1D(k) - HHH_1D(k + 1)) &
                     * (TKEtop - TKE_1D(k + 1)) &
                     / (TKE_1D(k) - TKE_1D(k + 1))
        else
            h__PSL = HHH_1D(k)
        endif
    endif

    h__PSL = min(h__PSL, HHH_1D(1))
    h__PSL = max(HHH_1D(mz), h__PSL)

    ! +--Height of the Secondary Seeing Layer (SSL)
    ! +  ==========================================

    RESET = .true.

    ! +--Search the        TKE minimum above the Primary Seeing Layer (PSL)
    ! +  (necessary if the TKE has decreased below the minimum value)
    ! +  ------------------------------------------------------------------

    if(INTERP) then
        k = kzi + 1
1011    continue
        k = k - 1
        if(k <= mzabso) go to 1010
        if(TKE_1D(k) < TKE_1D(k + 1)) go to 1011
1010    continue
    else
        k = kzi
    endif

    ! +--Search the first  TKE maximum above the Primary Seeing Layer (PSL)
    ! +  ------------------------------------------------------------------

    kmx = kzi
    k = k + 1
1021 continue
    k = k - 1
    if(k <= mzabso) go to 1020
    if(TKE_1D(k) > TKE_1D(k - 1) .and. &
       TKE_1D(k) > TKE_1D(k + 1) .and. &
       TKE_1D(k) > TKEmin * 3.0) then
        ! +                                    3.0      = 1/2 order of magnitude
        ! +        (in order to only detect a significant maximum)

        ! +--Define the TKE at the SSL top from the largest maximum in the SSL
        ! +  (thus examine the remaining upper part of the atmospheric column)
        ! +  -----------------------------------------------------------------

        if(RESET) then
            RESET = .false. ! indicates TKEtop is initialized
            TKEtop = 0.00   !
        endif
        if(TKEtop < TKE_1D(k) * 0.01) then
            TKEtop = TKE_1D(k) * 0.01
            kmx = k
        endif
    endif
    go to 1021
1020 continue
    TKEtop = max(TKEmin * 3.0, TKEtop)
    ! +                                    3.0      = 1/2 order of magnitude

    ! +--Search (from above) the SSL top            above the SSL    TKE maximum
    ! +  ------ (This         may be                                ) ----------
    ! +         (either a  TRUE         minimum  => INTERP = .false.)
    ! +         (    or an arbitrary small value => INTERP = .true. )
    ! +         -----------------------------------------------------

    kzi = mzabso
    do k = kmx, mzabso, -1
        if(TKE_1D(k) > TKEtop) &
            kzi = k
    enddo

    k = kzi - 1
    if(kzi <= mzabso + 1) then
        h__SSL = HHH_1D(mz)
    else
        if(INTERP) then
            h__SSL = HHH_1D(k + 1) &
                     + (HHH_1D(k) - HHH_1D(k + 1)) &
                     * (TKEtop - TKE_1D(k + 1)) &
                     / (TKE_1D(k) - TKE_1D(k + 1))
        else
            h__SSL = HHH_1D(k)
        endif
    endif

    h__SSL = min(h__SSL, HHH_1D(1))
    h__SSL = max(h__PSL, h__SSL)

    return
end
