#include "MAR_pp.def"
subroutine out_nc(ipr_nc)
    ! +------------------------------------------------------------------------+
    ! | MAR OUTPUT                                            20-Feb-2025  MAR |
    ! |   subroutine out_nc is used to write the main Model Variables          |
    ! |                                      on  a NetCDF file                 |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT: ipr_nc: Current time step    number                           |
    ! |   ^^^^^^         (starting from ipr_nc=1, which => new file creation)  |
    ! |                                                                        |
    ! |   OUTPUT: NetCDF File adapted to IDL Graphic Software                  |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   CAUTION:    This Routine requires the usual NetCDF library,          |
    ! |   ^^^^^^^^    and the complementary access library  'libUN.a'          |
    ! |                                                                        |
    ! |   MODIF.  4 Nov 2009   : OUTPUT of Map Scaling Factor SFm_DY           |
    ! |   ^^^^^                                                                |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_dy
    use mar_te
    use mar_tu
    use mar_ca
    use mar_hy
    use mar_ra
    use mar_sl
    use mar_sv
    use mardSV
    use mar_TV
    use marsSN
    use mar_wk
    use mar_io
    use mar_bs
    use trackwind, only: track_wind, ntwind, delta_u, delta_v, itw, &
                         name_wind, delta_u_NCsave, delta_v_NCsave, ddelta_wind
    use trackwater, only : jtw, track_water, delta_qv, ntwater, &
            name_water, delta_qv_NCsave, ddelta_water
    
    implicit none
    
    integer, intent(in) :: ipr_nc
    integer i, j, k, m
    real Soilz(mz)
    common / out_nc_rea / Soilz
    
    integer nsnomz, kk
    parameter(nsnomz = min(nsno, mz))
    
    integer kOUTnc, nOUTnc
    integer ijSNOW(mx, my)
    common / out_nc_int / kOUTnc, nOUTnc, ijSNOW
    
    ! +--Local  Variables
    ! +  ================

#ifdef IZ
    logical noZOOM
#endif
    logical LastIO
    real end_YR
    
    integer Lfnam, Ltit, Luni, Lnam, Llnam
    PARAMETER(Lfnam = 40, Ltit = 90, Luni = 90, Lnam = 13, Llnam = 50)
    ! +...Length of char strings
    
    character * (Lfnam) fnamNC
    common / out_nc_loc / fnamNC
    ! +...                   fnamNC: To retain file name.
    
    integer NdimNC
    PARAMETER(NdimNC = 6)
    ! +...Number of defined spatial dimensions (exact)
    
    integer MXdim
    PARAMETER(MXdim = 86401)
    ! +...Maximum Number of all dims: recorded Time Steps
    ! +   and also maximum of spatial grid points for each direction.
    
    integer MX_var
    PARAMETER(MX_var = 120)
    ! +...Maximum Number of Variables
    
    integer NattNC
    PARAMETER(NattNC = 2)
    ! +...Number of real attributes given to all variables
    
    integer RCODE
    
    integer njmo
    integer jourNC(MXdim)
    integer moisNC(MXdim)
    real yearNC(MXdim)
    real dateNC(MXdim)
    common / out_nc_r / yearNC, dateNC
    real timeNC(MXdim)
    real VALdim(MXdim, 0:NdimNC)
    integer nDFdim(0:NdimNC)
    common / out_nc_i / nDFdim
    integer NvatNC(NattNC)
    character * (Lnam) NAMdim(0:NdimNC)
    character * (Luni) UNIdim(0:NdimNC)
    character * (Lnam) SdimNC(4, MX_var)
    character * (Luni) unitNC(MX_var)
    character * (Lnam) nameNC(MX_var)
    character * (Llnam) lnamNC(MX_var)
    character * (Ltit) tit_NC
    character * (Lnam) NAMrat(NattNC)
#ifdef TC
    character * 9 labelc
#endif
    character * 120 tmpINP
    
    integer n1000, n100a, n100, n10_a, n10, n1, m10
    integer n, l, jd10, jd1, MMXstp
    integer it, mois, mill, iu, itotNC, NtotNC, ID__nc
#ifdef TC
    integer itot
#endif
    real starti, DayLen, rhodzk
    real(kind=8) starta(1) 
    character * 1 chn
    character * 7 lab__z, lab_dz, lab_ro, lab_wa, lab_g1, lab_g2, lab_Ti
    
    data lab__z/'z_mmWE '/
    data lab_dz/'dzSNOW '/
    data lab_ro/'roSNOW '/
    data lab_wa/'waSNOW '/
    data lab_g1/'g1SNOW '/
    data lab_g2/'g2SNOW '/
    data lab_Ti/'TiSNOW '/

#ifdef IZ
    data noZOOM/.false./
#endif
    
    ! +--NetCDF File Initialization
    ! +  ==========================
    
    if(ipr_nc == 1) then
        
        ! +--Nb of OUTPUT per File (Option #SO for splitting the MAR*.nc file)
        ! +  ---------------------
        
        nOUTnc = npr_nc
#ifdef SO
        nOUTnc = (2.0 / 0.09) * (161. * 161. * 34.) / (mx * my * mzz)
        kOUTnc = 86400.1 / (dt * nboucl)
        nOUTnc = nOUTnc / kOUTnc
        nOUTnc = nOUTnc * kOUTnc
        nOUTnc = min(nOUTnc, &
                kOUTnc * (365 + max(0, 1 - mod(iyrrGE, 4))))
        if(npr_nc == nprint + 1) then
            nOUTnc = nOUTnc + 1
        endif
        nOUTnc = min(npr_nc, nOUTnc)
#endif
        npr_nc = npr_nc - nOUTnc
        
        n1000 = 1 + iyrrGE / 1000
        n100a = mod(iyrrGE, 1000)
        n100 = 1 + n100a / 100
        n10_a = mod(n100a, 100)
        n10 = 1 + n10_a / 10
        n1 = 1 + mod(n10_a, 10)
        m10 = 1 + mmarGE / 10
        m1 = 1 + mod(mmarGE, 10)
        jd10 = 1 + jdarGE / 10
        jd1 = 1 + mod(jdarGE, 10)
        
        ! +--Output File Label
        ! +  -----------------
        
        fnamNC = 'MAR.' &
                // labnum(n1000) // labnum(n100) &
                // labnum(n10) // labnum(n1) &
                // labnum(m10) // labnum(m1) &
                // labnum(jd10) // labnum(jd1) &
                // '.' // explIO &
                // '.nc    '

#ifdef SO
        write(6, 6000) fnamNC, nOUTnc, npr_nc, kOUTnc
        6000    format(/, '++++++++++++++++++++++++++++++++++++++++++++++', &
                /, 'out_nc: Nb of OUTPUT: ', a19, ':', i4, &
                /, '                      After  present File:', i4, &
                /, '                      Per            Day :', i4, &
                /, '++++++++++++++++++++++++++++++++++++++++++++++')
#endif
        
        ! +--Output Title
        ! +  ------------
        
        tit_NC = 'MAR' &
                // ' - Exp: ' // explIO &
                // ' - ' &
                // labnum(n1000) // labnum(n100) &
                // labnum(n10) // labnum(n1) &
                // labnum(m10) // labnum(m1) &
                // labnum(jd10) // labnum(jd1)
        
        ! +--Create File / Write Constants
        ! +  -----------------------------
        
        MMXstp = MXdim
        ! +...To check array bounds... silently
        
        ! +--Time Variable (hour)
        ! +  ~~~~~~~~~~~~~~~~~~~~
        
        ! +...  To define a NetCDF dimension (size, name, unit):
        nDFdim(0) = nOUTnc
        nDFdim(0) = 0
        NAMdim(0) = 'time'
        UNIdim(0) = 'HOURS since 1900-01-01 00:00:00'
        
        ! +...  Check temporary arrays: large enough ?
        if(nOUTnc > MMXstp) &
                STOP '*** out_nc - ERROR : MXdim to low ***'
        
        starti = jhurGE + minuGE / 60.d0 + jsecGE / 3600.d0
        ! +...        starti : Starting Time (= current time in the day)
        ! Nb Days before iyrrGE
        ! Nb Leap Years
        ! Nb Days before mmarGE
        ! (including Leap Day)
        starta(1) = (365*2 + (iyrrGE - 1902) * 365 &
                + (iyrrGE - 1901) / 4 &
                + njyrGE(mmarGE) &
                + njybGE(mmarGE) &
                        * max(0, 1 - mod(iyrrGE, 4)) &
                + jdarGE - 1) * 24 &
                + jhurGE &
                + (minuGE * 60 + jsecGE) / 3600.
        
        do it = 1, nOUTnc
            timeNC(it) = starti + (it - 1) * nboucl * dt / 3600.
            ! +...                                         nboucl: #iter between output
            
            VALdim(it, 0) = starta(1) + (it - 1) * nboucl * dt / 3600.
            ! +...        VALdim(  ,0) : values of the dimension # 0 (time)
            
            ! +--Time Variable (date)
            ! +  ~~~~~~~~~~~~~~~~~~~~
            dateNC(it) = timeNC(it)
            jourNC(it) = jdarGE + timeNC(it) / 24.
        enddo
        mois = mmarGE
        mill = iyrrGE
        do it = 1, nOUTnc
            if(mois == 2 .and. &
                    mod(mill, 4) == 0) then
                njmo = njmoGE(mois) + 1
            else
                njmo = njmoGE(mois)
            endif
            if(jourNC(it) > njmo) then
                do iu = it, nOUTnc
                    jourNC(iu) = jourNC(iu) - njmo
                enddo
                mois = mois + 1
                if(mois > 12) then
                    mois = 1
                    mill = mill + 1
                endif
            endif
            moisNC(it) = mois
            yearNC(it) = mill
            
            if(dateNC(it) > 24. - epsi) then
                DayLen = 24.
                do iu = it, nOUTnc
                    dateNC(iu) = mod(dateNC(iu), DayLen)
                enddo
            endif
        enddo
        
        do it = 1, nOUTnc
            dateNC(it) = dateNC(it) &
                    + 1.d+2 * jourNC(it) &
                    + 1.d+4 * moisNC(it)
        enddo
        
        ! +--Define horizontal spatial dimensions :
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        
        ! +...Check temporary arrays: large enough ?
        if(mx > MMXstp .or. my > MMXstp &
                .or. mzz > MMXstp .or. mw > MMXstp) &
                STOP '*** out_nc - ERROR : MXdim to low ***'
        
        ! +...To define NetCDF dimensions (size, name, unit):
        
        do i = 1, mx
            VALdim(i, 1) = xxkm(i)
        enddo
        nDFdim(1) = mx
        NAMdim(1) = 'x'
        UNIdim(1) = 'km'
        
        do j = 1, my
            VALdim(j, 2) = yykm(j)
        enddo
        nDFdim(2) = my
        NAMdim(2) = 'y'
        UNIdim(2) = 'km'
        
        do k = 1, mz
            VALdim(k, 3) = sigma(k)
        enddo
        nDFdim(3) = mz
        NAMdim(3) = 'level'
        UNIdim(3) = 'sigma_level'
        ! +... For levels k
        
        do k = 1, mz
            VALdim(k, 4) = sigmid(k + 1)
        enddo
        nDFdim(4) = mz
        NAMdim(4) = 'level2'
        UNIdim(4) = 'sigma_level'
        ! +... For levels k+1/2
        
        do k = 1, mz
            VALdim(k, 5) = sigma(k)
        enddo
        VALdim(mzz, 5) = 1.
        nDFdim(5) = mzz
        NAMdim(5) = 'levelp1'
        UNIdim(5) = 'sigma_level'
        ! +... For levels mzz
        
        do k = 1, mw
            VALdim(k, 6) = k
        enddo
        nDFdim(6) = mw
        NAMdim(6) = 'sector'
        UNIdim(6) = 'level'
        ! +... For Surface Sectors
        
        ! +--Variable's Choice (Table MARvou.dat)
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        
        OPEN(unit = 10, status = 'unknown', file = 'MARvou.dat')
        
        itotNC = 0
        980     continue
        READ(10, '(A120)', end = 990) tmpINP
        if(tmpINP(1:4) == '    ') then
            itotNC = itotNC + 1
            READ(tmpINP, '(4x,5A9,A12,A50)') &
                    nameNC(itotNC), SdimNC(1, itotNC), SdimNC(2, itotNC), &
                    SdimNC(3, itotNC), SdimNC(4, itotNC), &
                    unitNC(itotNC), lnamNC(itotNC)
#ifdef TC
            ! +...          nameNC: Name
            ! +             SdimNC: Names of Selected Dimensions (max.4/variable)
            ! +             unitNC: Units
            ! +             lnamNC: Long_name, a description of the variable
            if(nameNC(itotNC) == 'qxTC     ' .and. nkWri >= 1) then
                nameNC(itotNC) = namTC(1)
                if(nkWri > 1) then
                    itot = itotNC
                    do n = 2, nkWri
                        itot = itot + 1
                        nameNC(itot) = namTC(n)
                        do m = 1, 4
                            SdimNC(m, itot) = SdimNC(m, itotNC)
                        enddo
                        unitNC(itot) = unitNC(itotNC)
                        lnamNC(itot) = lnamNC(itotNC)
                    enddo
                    itotNC = itot
                endif
            endif
#endif
        
        endif
        GOTO 980
        990     continue
        
        CLOSE(unit = 10)
        
        NtotNC = itotNC
        ! +...  NtotNC : Total number of variables writen in NetCDF file.
        
        ! +--List of NetCDF attributes given to all variables:
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ! +... The "actual_range" is the (min,max)
        ! +    of all data for each variable:
        NAMrat(1) = 'actual_range'
        NvatNC(1) = 2
        
        ! +... The "[var]_range" is NOT of attribute type,
        ! +    it is a true variable containing the (min,max) for
        ! +    each level, for 4D (space+time) variables only
        ! +    (automatic handling by UN library;
        ! +     must be the LAST attribute)
        NAMrat(NattNC) = '[var]_range'
        NvatNC(NattNC) = 2
        
        ! +--Automatic Generation of the NetCDF File Structure
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        
        ! +  **************
        call UNscreate(fnamNC, tit_NC, &
                NdimNC, nDFdim, MXdim, NAMdim, UNIdim, VALdim, &
                MX_var, NtotNC, nameNC, SdimNC, unitNC, lnamNC, &
                NattNC, NAMrat, NvatNC, &
                ID__nc)
        ! +  **************
        
        ! +--Write Time - Constants
        ! +  ~~~~~~~~~~~~~~~~~~~~~~
        do j = 1, my
            do i = 1, mx
                WKxy1(i, j) = GElonh(i, j) * 15.
                ! +...    Conversion: Hour->degrees
                
                WKxy2(i, j) = GElatr(i, j) / degrad
                ! +...    Conversion: rad ->degrees
                
                WKxy3(i, j) = isolSL(i, j)
                ! +...    Conversion to REAL type (integer not allowed)
                
                WKxy4(i, j) = isolTV(i, j)
                ! +...    Conversion to REAL type (integer not allowed)
            
            enddo
        enddo
        
        do l = 1, llx
            Soilz(l) = -deptTV(l)
        enddo
        do l = llx + 1, mz
            Soilz(l) = -deptTV(llx) - 1.e-6 * (l - llx)
        enddo
        
        ! +       ************
        call UNwrite(ID__nc, 'lon   ', 1, mx, my, 1, WKxy1)
        call UNwrite(ID__nc, 'lat   ', 1, mx, my, 1, WKxy2)
        call UNwrite(ID__nc, 'MapSC ', 1, mx, my, 1, SFm_DY)
        call UNwrite(ID__nc, 'sh    ', 1, mx, my, 1, sh)
        call UNwrite(ID__nc, 'isol  ', 1, mx, my, 1, WKxy3)
        call UNwrite(ID__nc, 'TEX   ', 1, mx, my, 1, WKxy4)
        call UNwrite(ID__nc, 'DepthS ', ipr_nc, mz, 1, 1, Soilz)
        call UNwrite(ID__nc, 'dsigm1', 1, mz, 1, 1, dsigm1)
        
        ! +       ************
        
        ! +--Snow Mosa?c for OUTPUT
        ! +  ======================
        
        do j = 1, my
            do i = 1, mx
                ijSNOW(i, j) = 1 + maskSL(i, j)
            enddo
        enddo
        
    else
        ! +--Re-Open file if already created.
        ! +  ================================
        
        ! +  ************
        call UNwopen(fnamNC, ID__nc)
        ! +  ************
    
    endif
    
    ! +--Write Time-dependent variables:
    ! +  ===============================
    
    ! +--UNLIMITED Time Dimension
    ! +  ------------------------
    
    if(nDFdim(0) == 0) then
        ! Nb Days before iyrrGE
        ! Nb Leap Years
        ! Nb Days before mmarGE
        ! (including Leap Day)
        starta(1) = (365*2 + (iyrrGE - 1902) * 365 &
                + (iyrrGE - 1901) / 4 &
                + njyrGE(mmarGE) &
                + njybGE(mmarGE) &
                        * max(0, 1 - mod(iyrrGE, 4)) &
                + jdarGE - 1) * 24 &
                + jhurGE &
                + (minuGE * 60 + jsecGE) / 3600.
        
        ! +  ************
        !call UNwrite(ID__nc, 'time   ', ipr_nc, 1, 1, 1, starta(1))
        call UNdwrite(ID__nc, 'time   ', ipr_nc, 1, 1, 1, starta(1))
        ! +  ************
    
    endif
    
    ! +  ************
    call UNwrite(ID__nc, 'date   ', ipr_nc, 1, 1, 1, dateNC(ipr_nc))
    call UNwrite(ID__nc, 'year   ', ipr_nc, 1, 1, 1, yearNC(ipr_nc))
    ! +  ************
    
    ! +--Geopotential Height, Saturation Specific Humidity
    ! +  -------------------------------------------------
    
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = gplvDY(i, j, k) * grvinv
                WKxyz2(i, j, k) = qvswDY(i, j, k)
#ifdef NH
                WKxyz3(i, j, k) = pairNH(i, j, k) * pstDYn(i, j) * sigma(k)
#endif
                WKxyz4(i, j, k) = 0.0
            enddo
        enddo
    enddo
    
    ! +--Surface Humidity and Green Leaf Fraction [%]
    ! +  --------------------------------------------
    
    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.0
            WKxy2(i, j) = 0.0
            WKxy3(i, j) = 0.0
        enddo
    enddo
    
    do n = 1, nvx
        do j = 1, my
            do i = 1, mx
                WKxy1(i, j) = WKxy1(i, j) &
                        + eta_TV(i, j, n, 1) * SLsrfl(i, j, n)
                WKxy2(i, j) = WKxy2(i, j) &
                        + glf_TV(i, j, n) * alaiTV(i, j, n) * SLsrfl(i, j, n)
            enddo
        enddo
    enddo
    
    do l = 1, llx
        do n = 1, nvx
            do j = 1, my
                do i = 1, mx
                    WKxyz4(i, j, l) = WKxyz4(i, j, l) &
                            + eta_TV(i, j, n, l) * SLsrfl(i, j, n)
                enddo
            enddo
        enddo
    enddo
    
    do l = 1, llx
        do j = 1, my
            do i = 1, mx
                WKxy3(i, j) = WKxy3(i, j) &
                        + WKxyz4(i, j, l) * dz_dSV(1 - l)
            enddo
        enddo
    enddo
    
    if(track_wind) then
        ! track momentum budget
        ! ---------------------
        do itw = 1, ntwind
            ! delta u
            ddelta_wind = delta_u(:, :, :, itw) - delta_u_NCsave(:, :, :, itw)
            call UNwrite(ID__nc, 'du_'//name_wind(itw), ipr_nc, mx, my, mz, ddelta_wind)
            ! delta v
            ddelta_wind = delta_v(:, :, :, itw) - delta_v_NCsave(:, :, :, itw)
            call UNwrite(ID__nc, 'dv_'//name_wind(itw), ipr_nc, mx, my, mz, ddelta_wind)
        enddo
        delta_u_NCsave = delta_u
        delta_v_NCsave = delta_v
    endif
    
    if(track_water) then
        ! track water budget
        ! ------------------
        do jtw = 1, ntwater
            ! delta qv
            ddelta_water = delta_qv(:, :, :, jtw) - delta_qv_NCsave(:, :, :, jtw)
            call UNwrite(ID__nc, 'dq_' // name_water(jtw), ipr_nc, mx, my, mz, ddelta_water)
        enddo
        delta_qv_NCsave = delta_qv
    endif
    
    ! +   ************
    call UNwrite(ID__nc, 'uairDY ', ipr_nc, mx, my, mz, uairDY)
    call UNwrite(ID__nc, 'vairDY ', ipr_nc, mx, my, mz, vairDY)
    call UNwrite(ID__nc, 'wairDY ', ipr_nc, mx, my, mz, wairDY)
    call UNwrite (ID__nc, 'psigDY ', ipr_nc, mx, my, mz, psigDY)
    call UNwrite (ID__nc, 'wsigDY ', ipr_nc, mx, my, mz, wsigDY)
    call UNwrite(ID__nc, 'tairDY ', ipr_nc, mx, my, mz, tairDY)
    call UNwrite(ID__nc, 'pktaDY ', ipr_nc, mx, my, mzz, pktaDY)
#ifdef NH
    call UNwrite(ID__nc, 'wairNH ', ipr_nc, mx, my, mz, wairNH)
    call UNwrite(ID__nc, 'pairNH ', ipr_nc, mx, my, mz, WKxyz3)
#endif
    call UNwrite(ID__nc, 'qvDY   ', ipr_nc, mx, my, mz, qvDY)
    call UNwrite(ID__nc, 'zzDY   ', ipr_nc, mx, my, mz, WKxyz1)
    call UNwrite(ID__nc, 'qsatDY ', ipr_nc, mx, my, mz, WKxyz2)
    call UNwrite(ID__nc, 'pstDY  ', ipr_nc, mx, my, 1, pstDY)
    call UNwrite(ID__nc, 'RadOLR ', ipr_nc, mx, my, 1, RAdOLR)
    call UNwrite(ID__nc, 'RadSol ', ipr_nc, mx, my, 1, RAdsol)
    call UNwrite(ID__nc, 'Rad_IR ', ipr_nc, mx, my, 1, RAD_ir)
    call UNwrite(ID__nc, 'hmelSL ', ipr_nc, mx, my, 1, hmelSL)
    call UNwrite(ID__nc, 'tairSL ', ipr_nc, mx, my, 1, TairSL)
    call UNwrite(ID__nc, 'tsrfSL ', ipr_nc, mx, my, mw, tsrfSL)
#ifdef T2
    call UNwrite(ID__nc, 'Ta2mSL ', ipr_nc, mx, my, mw, Ta2mSL)
    call UNwrite(ID__nc, 'TminSL ', ipr_nc, mx, my, mw, TminSL)
    call UNwrite(ID__nc, 'TmaxSL ', ipr_nc, mx, my, mw, TmaxSL)
    call UNwrite(ID__nc, 'Ta3mSL ', ipr_nc, mx, my, mw, Ta3mSL)
    call UNwrite(ID__nc, 'V03mSL ', ipr_nc, mx, my, mw, V03mSL)
    call UNwrite(ID__nc, 'V10mSL ', ipr_nc, mx, my, mw, V10mSL)
#endif
    call UNwrite(ID__nc, 'albxSL ', ipr_nc, mx, my, mw, albxSL)
    call UNwrite(ID__nc, 'hsenSL ', ipr_nc, mx, my, 1, hsenSL)
    call UNwrite(ID__nc, 'hlatSL ', ipr_nc, mx, my, 1, hlatSL)
    call UNwrite(ID__nc, 'ect_TE ', ipr_nc, mx, my, mz, ect_TE)
    call UNwrite(ID__nc, 'eps_TE ', ipr_nc, mx, my, mz, eps_TE)
    call UNwrite(ID__nc, 'TUkvm  ', ipr_nc, mx, my, mz, TUkvm)
    call UNwrite(ID__nc, 'TUkvh  ', ipr_nc, mx, my, mz, TUkvh)
    call UNwrite(ID__nc, 'SL_z0  ', ipr_nc, mx, my, mw, SL_z0)
    call UNwrite(ID__nc, 'SL_r0  ', ipr_nc, mx, my, mw, SL_r0)
#ifdef BS
    call UNwrite(ID__nc, 'ustart ', ipr_nc, mx, my, mw, SaltSN)
    call UNwrite(ID__nc, 'z0emBS ', ipr_nc, mx, my, mw, Z0emBS)
    call UNwrite(ID__nc, 'z0SaBS ', ipr_nc, mx, my, mw, Z0SaBS)
#endif
    call UNwrite(ID__nc, 'SLsrfl ', ipr_nc, mx, my, mw, SLsrfl)
    call UNwrite(ID__nc, 'SLuusl ', ipr_nc, mx, my, mw, SLuusl)
    call UNwrite(ID__nc, 'SLutsl ', ipr_nc, mx, my, mw, SLutsl)
    call UNwrite(ID__nc, 'SLuqsl ', ipr_nc, mx, my, mw, SLuqsl)
    call UNwrite(ID__nc, 'SLussl ', ipr_nc, mx, my, mw, SLussl)
    call UNwrite(ID__nc, 'albeSL ', ipr_nc, mx, my, 1, albeSL)
    call UNwrite(ID__nc, 'Clouds ', ipr_nc, mx, my, 1, cld_SL)
    ! +   ************
    
    ! +   ************
    call UNwrite(ID__nc, 'HumSol ', ipr_nc, mx, my, 1, WKxy1)
    call UNwrite(ID__nc, 'GreenL ', ipr_nc, mx, my, 1, WKxy2)
    call UNwrite(ID__nc, 'WatSol ', ipr_nc, mx, my, 1, WKxy3)
    call UNwrite(ID__nc, 'EvapoT ', ipr_nc, mx, my, 1, evapTV)
    call UNwrite(ID__nc, 'Draing ', ipr_nc, mx, my, 1, draiTV)
    call UNwrite(ID__nc, 'RunOFF ', ipr_nc, mx, my, 1, runoTV)
    call UNwrite(ID__nc, 'H2OSol ', ipr_nc, mx, my, mz, WKxyz4)
    ! +   ************

#ifdef IZ
    if(noZOOM) then
#endif
#ifdef DY
        ! +--Dynamical Budget
        ! +  ----------------
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    dumy3D(i, j, k) = 0.
                enddo
            enddo
        enddo
        ! +  ******
        call dynbil(7, 1, ipr_nc, ID__nc, 0., 0., 0., 0., dumy3D)
        ! +  ******
#endif
#ifdef IZ
    endif
#endif
    
    ! +--Cloud Microphysics, Mass Flux convective Scheme
    ! +  -----------------------------------------------
    
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = qwHY(i, j, k) + qiHY(i, j, k)
                WKxyz2(i, j, k) = qrHY(i, j, k) + qsHY(i, j, k)
                WKxyz3(i, j, k) = dqv_CA(i, j, k) * Lv_H2O / cp &
                        * min(adj_CA(i, j), iun) * 86400. / dt_Loc
                WKxyz4(i, j, k) = (dpktCA(i, j, k) &
                        * (ptopDY + sigma(k) * pstDY(i, j))**cap) &
                        * min(adj_CA(i, j), iun) * 86400. / dt_Loc
            enddo
        enddo
    enddo
    
    do j = 1, my
        do i = 1, mx
            zcd_HY(i, j) = zcd_HY(i, j) / max(eps9, Hcd_HY(i, j)) / gravit
            Tcd_HY(i, j) = Tcd_HY(i, j) / max(eps9, Hcd_HY(i, j))
            zsb_HY(i, j) = zsb_HY(i, j) / max(eps9, Hsb_HY(i, j)) / gravit
            Tsb_HY(i, j) = Tsb_HY(i, j) / max(eps9, Hsb_HY(i, j))
            Hcd_HY(i, j) = Hcd_HY(i, j) / max(1, icntHY)
            Hsb_HY(i, j) = Hsb_HY(i, j) / max(1, icntHY)
        enddo
    enddo
    
    ! +   ************
    call UNwrite(ID__nc, 'QwQi   ', ipr_nc, mx, my, mz, WKxyz1)
    call UNwrite(ID__nc, 'QrQs   ', ipr_nc, mx, my, mz, WKxyz2)
    call UNwrite(ID__nc, 'ccniHY ', ipr_nc, mx, my, mz, ccniHY)
    call UNwrite(ID__nc, 'qiHY   ', ipr_nc, mx, my, mz, qiHY)
    call UNwrite(ID__nc, 'qwHY   ', ipr_nc, mx, my, mz, qwHY)
    call UNwrite(ID__nc, 'qsHY   ', ipr_nc, mx, my, mz, qsHY)
    call UNwrite(ID__nc, 'qrHY   ', ipr_nc, mx, my, mz, qrHY)
    call UNwrite(ID__nc, 'hlatHY ', ipr_nc, mx, my, mz, hlatHY)
    call UNwrite(ID__nc, 'HLCond ', ipr_nc, mx, my, 1, Hcd_HY)
    call UNwrite(ID__nc, 'TaCond ', ipr_nc, mx, my, 1, Tcd_HY)
    call UNwrite(ID__nc, 'z_Cond ', ipr_nc, mx, my, 1, zcd_HY)
    call UNwrite(ID__nc, 'HLSubl ', ipr_nc, mx, my, 1, Hsb_HY)
    call UNwrite(ID__nc, 'TaSubl ', ipr_nc, mx, my, 1, Tsb_HY)
    call UNwrite(ID__nc, 'z_Subl ', ipr_nc, mx, my, 1, zsb_HY)
    call UNwrite(ID__nc, 'rainHY ', ipr_nc, mx, my, 1, rainHY)
    call UNwrite(ID__nc, 'snowHY ', ipr_nc, mx, my, 1, snowHY)
    call UNwrite(ID__nc, 'crysHY ', ipr_nc, mx, my, 1, crysHY)
    ! +   ************
    
    icntHY = 0
    do j = 1, my
        do i = 1, mx
            Hcd_HY(i, j) = 0.
            Tcd_HY(i, j) = 0.
            zcd_HY(i, j) = 0.
            Hsb_HY(i, j) = 0.
            Tsb_HY(i, j) = 0.
            zsb_HY(i, j) = 0.
        enddo
    enddo
    
    ! +   ************
    call UNwrite(ID__nc, 'CAPE   ', ipr_nc, mx, my, 1, capeCA)
    call UNwrite(ID__nc, 'rainCA ', ipr_nc, mx, my, 1, rainCA)
    call UNwrite(ID__nc, 'snowCA ', ipr_nc, mx, my, 1, snowCA)
    call UNwrite(ID__nc, 'dqv_CA ', ipr_nc, mx, my, mz, WKxyz3)
    call UNwrite(ID__nc, 'dpktCA ', ipr_nc, mx, my, mz, WKxyz4)
    ! +   ************

#ifdef WB
    ! +--Water Budget
    ! +  ------------
    ! +  ******
    call H2O_WB(-1, 0., 0., .false., .true.)
    ! +  ******
    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = dq__WB(i, j, 1)
            WKxy2(i, j) = dq__WB(i, j, 2)
            WKxy3(i, j) = dq__WB(i, j, 3)
            WKxy4(i, j) = dq__WB(i, j, 4)
            WKxy5(i, j) = dq__WB(i, j, 5)
            WKxy6(i, j) = dq__WB(i, j, 6)
        enddo
    enddo
    call UNwrite(ID__nc, 'H2O_ADV', ipr_nc, mx, my, 1, WKxy1)
    call UNwrite(ID__nc, 'H2OdifH', ipr_nc, mx, my, 1, WKxy2)
    call UNwrite(ID__nc, 'H2O_CVA', ipr_nc, mx, my, 1, WKxy3)
    call UNwrite(ID__nc, 'H2OdifV', ipr_nc, mx, my, 1, WKxy4)
    call UNwrite(ID__nc, 'H2O_mic', ipr_nc, mx, my, 1, WKxy5)
    call UNwrite(ID__nc, 'H2Ofltr', ipr_nc, mx, my, 1, WKxy6)
    call UNwrite(ID__nc, 'H2OsrfT', ipr_nc, mx, my, 1, uq__WB)
    call UNwrite(ID__nc, 'H2OsrfA', ipr_nc, mx, my, 1, wq__WB)
    call UNwrite(ID__nc, 'H2Oflux', ipr_nc, mx, my, 1, upq_WB)
    call UNwrite(ID__nc, 'H2Ofluy', ipr_nc, mx, my, 1, vpq_WB)
    call UNwrite(ID__nc, 'H2Omean', ipr_nc, mx, my, 1, cpq_WB)
    call UNwrite(ID__nc, 'Snoflux', ipr_nc, mx, my, 1, ups_WB)
    call UNwrite(ID__nc, 'Snofluy', ipr_nc, mx, my, 1, vps_WB)
    call UNwrite(ID__nc, 'Snomean', ipr_nc, mx, my, 1, cps_WB)
    call UNwrite(ID__nc, 'Vap_Liq', ipr_nc, mx, my, mz, dqwHY)
    call UNwrite(ID__nc, 'Vap_Ice', ipr_nc, mx, my, mz, dqiHY)
    ! +  ******
    call H2O_WB(-1, 0., 0., .true., .false.)
    ! +  ******
#endif

#ifdef OD
    ! +--Cloud Optical Depth
    ! +  -------------------
    do j = 1, my
        do i = 1, mx
            ! +...     WKxy1(i,j)  : liquid water path (kg/m2) (droplets)
            WKxy1(i, j) = 0.
            ! +...     WKxy2(i,j)  : liquid water path (kg/m2) (crystals)
            WKxy2(i, j) = 0.
        enddo
    enddo
    do k = mzabso + 1, mz
        do j = 1, my
            do i = 1, mx
                ! rhodzk : (rho / 1000) * (dz * gravit)
                rhodzk = (pstDY(i, j) * sigma(k) + ptopDY) &
                        / (ra * tairDY(i, j, k) * (1. + .608 * qvDY(i, j, k))) &
                        * (gpmiDY(i, j, k) - gpmiDY(i, j, k + 1))
                WKxy1(i, j) = WKxy1(i, j) + rhodzk * qwHY(i, j, k)
                WKxy2(i, j) = WKxy2(i, j) + rhodzk * qiHY(i, j, k)
            enddo
        enddo
    enddo
    do j = 1, my
        do i = 1, mx
            WKxy3(i, j) = 1.5 * (WKxy1(i, j) / 20.d-6 &
                    + WKxy2(i, j) / 40.d-6) * grvinv
        enddo
    enddo
    call UNwrite(ID__nc, 'OptDep ', ipr_nc, mx, my, 1, WKxy3)
#endif
    
    ! +   ************
    !call UNwrite(ID__nc, 'OptDep ', ipr_nc, mx, my, 1, RAcdtO)
    ! +   ************
    
    ! +--Snow Pack
    ! +  ---------
    
    if(SnoMod .and. VSISVAT) then
        
        do k = 1, mw
            do j = 1, my
                do i = 1, mx
                    WRKxys(i, j, k) = zWE_SN(i, j, k) - zWE0SN(i, j, k)
                enddo
            enddo
        enddo
        
        do n = 1, mw
            
            if(n > 1) then
                lab__z(5:5) = '_'
                lab_dz(5:5) = '_'
                lab_ro(5:5) = '_'
                lab_wa(5:5) = '_'
                lab_g1(5:5) = '_'
                lab_g2(5:5) = '_'
                lab_Ti(5:5) = '_'
                write(chn, '(i1)') n
                lab__z(6:6) = chn
                lab_dz(6:6) = chn
                lab_ro(6:6) = chn
                lab_wa(6:6) = chn
                lab_g1(6:6) = chn
                lab_g2(6:6) = chn
                lab_Ti(6:6) = chn
            else
                lab__z(5:6) = 'WE'
                lab_dz(5:6) = 'OW'
                lab_ro(5:6) = 'OW'
                lab_wa(5:6) = 'OW'
                lab_g1(5:6) = 'OW'
                lab_g2(5:6) = 'OW'
                lab_Ti(5:6) = 'OW'
            endif
            
            do k = 1, mz
                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = 0.
                        WKxyz2(i, j, k) = 0.
                        WKxyz3(i, j, k) = 0.
                        WKxyz4(i, j, k) = 0.
                    enddo
                enddo
            enddo
            
            do k = 1, nsnomz
                do j = 1, my
                    do i = 1, mx
                        WKxyz4(i, j, k) = min(max(nssSNo(i, j, n) - k + 1, 0), 1) ! 0: outside SnowPack
                    enddo
                enddo
            enddo
            
            do k = 1, nsnomz
                do j = 1, my
                    do i = 1, mx
#ifdef vS
                        n = ijSNOW(i, j)             ! 1: Land  /  2:Ocean
#endif
                        kk = max(nssSNo(i, j, n) - k + 1, 1)    ! 1: 1st lev SnowPack
                        
                        WKxyz1(i, j, k) = dzsSNo(i, j, n, kk) * WKxyz4(i, j, k) &
                                + epsi * (1. - WKxyz4(i, j, k))
                        WKxyz2(i, j, k) = rosSNo(i, j, n, kk) * WKxyz4(i, j, k)
                        WKxyz3(i, j, k) = wasSNo(i, j, n, kk) * WKxyz4(i, j, k)
                    enddo
                enddo
            enddo
            
            ! +  ************
            call UNwrite(ID__nc, lab__z, ipr_nc, mx, my, mw, WRKxys)
            call UNwrite(ID__nc, lab_dz, ipr_nc, mx, my, mz, WKxyz1)
            call UNwrite(ID__nc, lab_ro, ipr_nc, mx, my, mz, WKxyz2)
            call UNwrite(ID__nc, lab_wa, ipr_nc, mx, my, mz, WKxyz3)
            ! +  ************
            
            do k = 1, mz
                do j = 1, my
                    do i = 1, mx
                        WKxyz1(i, j, k) = 0.
                        WKxyz2(i, j, k) = 0.
                        WKxyz3(i, j, k) = 0.
                        WKxyz4(i, j, k) = 0.
                    enddo
                enddo
            enddo
            
            do k = 1, nsnomz
                do j = 1, my
                    do i = 1, mx
                        kk = max(nssSNo(i, j, n) - k + 1, 1)    ! 1: 1st lev SnowPack
                        WKxyz1(i, j, k) = g1sSNo(i, j, n, kk) * WKxyz4(i, j, k)
                        WKxyz2(i, j, k) = g2sSNo(i, j, n, kk) * WKxyz4(i, j, k)
                        WKxyz3(i, j, k) = tisSNo(i, j, n, kk) * WKxyz4(i, j, k)
                    enddo
                enddo
            enddo
            
            ! +  ************
            call UNwrite(ID__nc, lab_g1, ipr_nc, mx, my, mz, WKxyz1)
            call UNwrite(ID__nc, lab_g2, ipr_nc, mx, my, mz, WKxyz2)
            call UNwrite(ID__nc, lab_Ti, ipr_nc, mx, my, mz, WKxyz3)
            ! +  ************
        
        enddo
    
    endif

#ifdef PO
    ! +--Polynya
    ! +  -------
    ! +   ************
    call UNwrite(ID__nc, 'hatmPO ', ipr_nc, mx, my, 1, hatmPO)
    call UNwrite(ID__nc, 'hfraPO ', ipr_nc, mx, my, 1, hfraPO)
    call UNwrite(ID__nc, 'aicePO ', ipr_nc, mx, my, 1, aicePO)
    call UNwrite(ID__nc, 'hicePO ', ipr_nc, mx, my, 1, hicePO)
    call UNwrite(ID__nc, 'hiavPO ', ipr_nc, mx, my, 1, hiavPO)
    ! +   ************
#endif

#ifdef TC
    ! +--Chemical Species
    ! +  ----------------
    if(nkWri > 0) then
        do n = 1, nkWri
            labelc = namTC(ikTC(n))
            do k = 1, mz
                do j = 1, my
                    do i = 1, mx
                        ! +...      Conversion [kg/kg] ------------------> [micro-g/kg]
                        ! Conversion [mcm]   ------------------> [ppb] if requested
                        WKxyz1(i, j, k) = qxTC(i, j, k, ikTC(n)) * 1.000d+09
#endif
#ifdef CH
                        ! =>Conversion [kg/kg] ------------------> [micro-g/kg]   eliminated
                        !                                      ==> 0.392D-19 = 392D-10/1.d+9
                        WKxyz1(i, j, k) = WKxyz1(i, j, k) * rolvDY(i, j, k) * 0.392D-19 / Unity
#endif
#ifdef TC
                    enddo
                enddo
            enddo
            ! +       ************
            call UNwrite(ID__nc, labelc(1:7), ipr_nc, mx, my, mz, WKxyz1)
            ! +       ************
        enddo
    endif
#endif
    
    ! +--That 's all, folks: NetCDF File Closure
    ! +  =======================================
    
    ! +   ************
    call UNclose(ID__nc)
    ! +   ************
    
    ! +--Work Arrays Reset
    ! +  =================
    
    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.0
            WKxy2(i, j) = 0.0
            WKxy3(i, j) = 0.0
            WKxy4(i, j) = 0.0
        enddo
    enddo
    
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
                WKxyz3(i, j, k) = 0.0
                WKxyz4(i, j, k) = 0.0
            enddo
        enddo
    enddo
    
    if(mmarGE == 12 .and. jdarGE == 31) then
        end_YR = real(24 - jhurGE) * 3599.9 - dt * nboucl
        if(end_YR < 0.) then
            LastIO = .true.
        else
            LastIO = .false.
        endif
    else
        if(mmarGE == 1 .and. jdarGE == 1 .and. jhurGE == 0 .and. &
                iyrrGE > iyr0GE .and. &
                iterun >= nboucl * nprint) then
            LastIO = .true.
        else
            LastIO = .false.
        endif
    endif
    
    ! +   +++++++++++
    ! +   if (LastIO) ipr_nc = 0        ! ipr_nc:=0 => NEW MAR*.nc     created
    ! +   +++++++++++                   !              at the next out_nc call
    
    return
endsubroutine out_nc
