! Created by Cécile Agosta on 21/09/2021
! isotopic routines, taken from LMDZ6iso (Camille Risi, LMD)

subroutine stewart_sublim_nofrac(qa, qa_iso, qr, qr_iso, qevap, qr_new, qr_iso_new, qa_iso_new, qevap_iso)
    !                             q,    xt, rfl,  xtrfl, qevfl,    rfln,       xtrfln,      xtnew,       Exi
    ! ============================================================================================ !
    ! From C. Risi phylmdiso/isotopes_routines_mod.F90/stewart_sublim_nofrac_vectall
    ! sublimation (re-evaporation) of ice: we suppose no fractionation during ice sublimation
    ! ============================================================================================ !
    use mariso, only: niso, iso_wat, iso_nudging, negligible
    implicit none

    ! inputs
    ! ======
    ! qa : specific humidity (kg kg-1)
    ! LMDZ : qa = zq
    real, intent(in) :: qa
    ! qa_iso : isotopic specific humidity (kg kg-1)
    ! LMDZ : qa_iso = zxt
    real, intent(in) :: qa_iso(niso)
    ! qr : precipitation flux in the atmosphere (kg kg-1) *before evaporation*
    ! LMDZ : qr = Pqisup or zrfl
    real, intent(in) :: qr
    ! qr_iso : isotopic precipitation flux in the atmosphere (kg kg-1)
    ! LMDZ : qr_iso = Pxtisup or zxtrfl
    real, intent(in) :: qr_iso(niso)
    ! qevap : evaporation flux (kg kg-1)
    ! LMDZ : qevap = (Eqi or zqevfl) * (fac_ftmr = fac_fluxtomixratio) for conversion from flux to mixing ratio
    ! fac_ftmr = fac_fluxtomixratio = factor for conversion from flux to mixing ratio = g.dt/dp (1 / (kg m-2 s-1))
    ! todo : qevap, verify the unit. Coded here in kg/kg, in LMDZ it was in flux so to be multiplied by fac_fluxtomixratio = g.dt/dp
    real, intent(in) :: qevap
    ! qr_new : precipitation flux in the atmosphere (kg kg-1) *after evaporation*
    ! LMDZ : qr_new = Pqiinf or zrfln (zrfln = zrfl new = rainfall flux new)
    real, intent(in) :: qr_new

    ! outputs
    ! =======
    ! qr_iso_new : isotopic precipitation flux in the atmosphere (kg kg-1) *after evaporation*
    ! LMDZ : qr_iso_new = Pxtiinf or zxtrfln (zxtrfln = zxtrfl new)
    real, intent(out) :: qr_iso_new(niso)
    ! qa_iso_new : isotopic specific humidity after sublimation (kg kg-1)
    ! LMDZ : qa_iso_new = xtnew
    real, intent(out) :: qa_iso_new(niso)
    ! qevap_iso : isotopic evaporation (kg kg-1)
    ! LMDZ : qevap_iso = Exi
    real, intent(out) :: qevap_iso(niso)

    ! local variables
    real Rb0(niso)
    integer wiso
    !#ifdef ISOVERIF
    !#ifdef ISOVERIF

    ! traitement rapide de quelques cas particuliers
    ! todo : check if a threshold is needed here
    if(qr <= 0) then
        ! no precipitation, no qr_new, no change in water vapor
        !#ifdef ISOVERIF
        do wiso = 1, niso
            qr_iso_new(wiso) = 0.
        enddo
        if(iso_nudging) then
            qr_iso_new(iso_wat) = qr_new
        endif
        if(abs(qevap) > negligible) then
            ! attention: pour des raisons obscures, il y a parfois
            ! de le réévaporation significative alors qu'il n'y a
            ! aucun cristal à réévaporer.
            ! Dans ce cas, on admet cette réévaporation obscure et
            ! on suppose qu'elle ne change pas la composition
            ! isotopique de la vapeur.
            if(qa > negligible) then
                do wiso = 1, niso
                    Rb0(wiso) = qa_iso(wiso) / qa
                enddo
            else
                ! there is no water vapor.
                ! It's anoying, but we hope water vapor will be loaded soon.
                do wiso = 1, niso
                    Rb0(wiso) = 0.
                enddo
                Rb0(iso_wat) = 1.
            endif
            do wiso = 1, niso
                qevap_iso(wiso) = Rb0(wiso) * qevap
                qa_iso_new(wiso) = qa_iso(wiso) + qevap_iso(wiso)
            enddo
        else
            ! all is coherent, all fluxes are null
            do wiso = 1, niso
                qa_iso_new(wiso) = qa_iso(wiso)
                qevap_iso(wiso) = 0.
            enddo
        endif
        !#ifdef ISOVERIF
    else
        ! qr is greater than 0.
        ! qr_iso_new and qevap_iso computed without fractionation
        do wiso = 1, niso
            qr_iso_new(wiso) = qr_iso(wiso) / qr * qr_new
            qevap_iso(wiso) = qr_iso(wiso) / qr * qevap
        enddo
        !#ifdef ISOVERIF
        if(iso_nudging) then
            qevap_iso(iso_wat) = qevap
            qr_iso_new(iso_wat) = qr_new
        endif
        ! qa_iso_new
        do wiso = 1, niso
            qa_iso_new(wiso) = qa_iso(wiso) + qevap_iso(wiso)
            qa_iso_new(wiso) = max(0., qa_iso_new(wiso))
        enddo
        !#ifdef ISOVERIF
    endif
endsubroutine stewart_sublim_nofrac

subroutine iso_surf_ocean(psurf, tsurf, qas, wsas, qas_iso, fevap, fevap_iso)
    ! LMDZ : iso_surf_ocean = calcul_iso_surf_oce_vectall
    use mariso, only: wiso, niso, Rocean, Rdefault, negligible
    ! ptopDY : Pressure at Model Top (kPa)
    use mar_dy, only: ptopDY

    !#ifdef ISOVERIF
    !#ifdef ISOTRAC
    implicit none

    ! input
    ! =====
    ! psurf : surface pressure (kPa)
    real, intent(in) :: psurf
    ! tsurf : surface temperature (K)
    real, intent(in) :: tsurf
    ! qas : surface air specific humidity (kg/kg)
    real, intent(in) :: qas
    ! wsas : surface air wind speed (m s-1)
    real, intent(in) :: wsas
    ! qas_iso : isotopes in near surface water vapor
    real, intent(in) :: qas_iso(niso)
    ! fevap : evaporation flux (kg s-1)
    real, intent(in) :: fevap
    ! output
    ! ======
    ! fevap_iso : isotopic evaporation flux (kg s-1)
    real, intent(out) :: fevap_iso(niso)
    ! function
    ! ========
    real qsat0D
    ! local
    ! =====
    ! rh : relative humidity
    real :: rh
    ! qsat : saturation specific humidity (kg/kg)
    real qsat
    ! alpha : fractionation factor
    real alpha(niso)
    ! Riso : isotopic ratio
    real Riso(niso)
    ! kcin : cinetic fractionation coefficient
    real kcin(niso)

    !#ifdef ISOVERIF
    !#ifdef ISOTRAC
    !#ifdef ISOVERIF

    if(fevap > 0.) then
        ! evaporation case
        qsat = qsat0D(tsurf, 1., psurf, ptopDY, 1)
        rh = qas / qsat
        rh = min(1., max(0., rh))
        ! fractionation vapor/liquid
        call fractcalk(tsurf, alpha)
        ! R of surface air
        if(qas > negligible) then
            do wiso = 1, niso
                Riso(wiso) = qas_iso(wiso) / qas
            enddo
        else
            !#ifdef ISOVERIF
            do wiso = 1, niso
                Riso(wiso) = Rdefault(wiso)
            enddo
        endif
        call calcul_kcin(wsas, kcin)
        if(rh < 0.98) then
            do wiso = 1, niso
                fevap_iso(wiso) = fevap * (Rocean(wiso) / alpha(wiso) - rh * Riso(wiso)) / (1.-rh) * (1.-kcin(wiso))
            enddo
        else
            do wiso = 1, niso
                fevap_iso(wiso) = fevap * Rocean(wiso) / alpha(wiso)
            enddo
        endif
        !#ifdef ISOTRAC
        !#ifdef ISOVERIF
        !#ifdef ISOVERIF
        !#ifdef ISOTRAC
    else if(fevap == 0.) then
        ! no evaporation case
        do wiso = 1, niso
            fevap_iso(wiso) = 0.
        enddo
    else
        ! condensation case
        ! call iso_rosee_givre(qas_iso, qas, tsurf, fevap, fevap_iso)
        if(qas > negligible) then
            call fractcalk(tsurf, alpha)
            do wiso = 1, niso
                ! methode 1: condensation à l'équilibre, approx 1er ordre
                Riso(wiso) = qas_iso(wiso) / qas
                fevap_iso(wiso) = fevap * alpha(wiso) * Riso(wiso)
                ! methode 2: condensation, approche sans approximation
                ! call condiso_liq_ice(wiso, qas_iso(wiso), qas, qevap, tsurf, 0.0, zxtice, zxtliq)
                ! fevap_iso(wiso) = -zxtliq / dtime * Mair
            enddo
        else
            do wiso = 1, niso
                Riso(wiso) = Rdefault(wiso)
                fevap_iso(wiso) = fevap * alpha(wiso) * Riso(wiso)
            enddo
        endif
    endif

    return
endsubroutine iso_surf_ocean

subroutine calcul_kcin(ws, kcin)
    use mariso, only: wiso, niso, ws0cin, kcin_0, kcin_1, kcin_2
    implicit none
    ! ==========================================
    ! computes kcin as a function of wind speed
    ! ==========================================
    ! input
    ! =====
    ! ws : surface wind speed (m s-1)
    real, intent(in) :: ws
    ! output
    ! ======
    ! kcin : cinetic fractionation coefficient
    real, intent(out) :: kcin(niso)

    if(ws < ws0cin) then
        do wiso = 1, niso
            kcin(wiso) = kcin_0(wiso)
        enddo
    else
        do wiso = 1, niso
            kcin(wiso) = kcin_1(wiso) * ws + kcin_2(wiso)
        enddo
    endif
endsubroutine calcul_kcin

subroutine iso_rosee_givre(qas_iso, qas, tsurf, fevap, fevap_iso)
    use mariso, only: wiso, niso, negligible
    !#ifdef ISOVERIF
    !#ifdef ISOTRAC
    implicit none
    ! input
    ! =====
    ! fevap : evaporation flux (kg s-1)
    real, intent(in) :: fevap
    ! qas : surface air specific humidity (kg kg-1)
    real, intent(in) :: qas
    ! qas_iso : isotopic surface air specific humidity (kg kg-1)
    real, intent(in) :: qas_iso(niso)
    ! tsurf : surface temperature (K)
    real, intent(in) :: tsurf
    ! output
    ! ======
    ! fevap_iso : isotopic evaporation flux (kg s-1)
    real, intent(out) :: fevap_iso(niso)
    ! local
    ! =====
    ! alpha : fractionation factor
    real alpha(niso)
    real Riso
    ! real zxtliq, zxtice ! (kg kg-1)

    if(fevap == 0.) then
        !#ifdef ISOVERIF
        do wiso = 1, niso
            fevap_iso(wiso) = 0.
        enddo
        return
    endif

    if(qas > negligible) then
        call fractcalk(tsurf, alpha)
        do wiso = 1, niso
            ! methode 1: condensation à l'équilibre, approx 1er ordre
            Riso = qas_iso(wiso) / qas
            fevap_iso(wiso) = fevap * alpha(wiso) * Riso
            ! methode 2: condensation, approche sans approximation
            ! call condiso_liq_ice(wiso, qas_iso(wiso), qas, qevap, tsurf, 0.0, zxtice, zxtliq)
            ! fevap_iso(wiso) = -zxtliq / dtime * Mair
        enddo
    else
        write(*, *) 'iso_surf>iso_rosee_givre 3189: fevap=', fevap
        write(*, *) 'qas=', qas
        stop
    endif
endsubroutine iso_rosee_givre

subroutine fractcalk(ta, alpha)
    use mariso, only: wiso, niso, niso_all, iso_O17, iso_HTO, iso_HDO, iso_O18, iso_O17, iso_wat, frac_tmin, tmelt, &
                      alpha_liq_0, alpha_liq_1, alpha_liq_2, fac_coeff_eq17_liq, &
                      alpha_ice_0, alpha_ice_1, alpha_ice_2, fac_coeff_eq17_ice, diffus_rel, &
                      musi, lambda_sursat
    implicit none
    ! -------------------------------------------------------------------------
    !  Calculation of the fractionation coefficient of water isotopes.
    !  March 2003
    !  Sandrine Bony (LMD/CNRS), after MPI code from Georg Hoffmann (LSCE)
    ! -------------------------------------------------------------------------
    ! input
    ! =====
    ! ta : temperature (K)
    real, intent(in) :: ta
    ! output
    ! ======
    ! alpha : fractionation factor for liquid/vapor if (ta > tfreeze) and for ice/vapor if (ta < tfreeze)
    ! alpha liquid/vapor > 1. and alpha ice/vapor > 1.
    real alpha(niso)
    ! local
    ! =====
    real tt ! temperature (K) = max(ta, frac_tmin)
    real, parameter :: alpha_max = 10.
    ! tfreeze : freezing temperature (K)
    real, parameter :: tfreeze = 273.15
    ! supersat : supersaturation (> 1.)
    real :: supersat
    real :: alphatot(niso_all)

    if(ta >= tfreeze) then
        ! fractionation over liquid water (Majoube, 1971b)
        ! alpha liquid/vapor = Rliquid/Rvapor > 1.
        ! ------------------------------------------------
        tt = max(ta, frac_tmin)
        do wiso = 1, niso
            alpha(wiso) = exp(alpha_liq_0(wiso) + alpha_liq_1(wiso) / tt + alpha_liq_2(wiso) / (tt**2))
            if(wiso == iso_O17) then
                alpha(wiso) = (alpha(wiso))**fac_coeff_eq17_liq
            endif
            alpha(wiso) = max(min(alpha(wiso), alpha_max), 0.)
        enddo
    else
        ! fractionation over ice (Majoube, 1971b)
        ! alpha ice/vapor = Rice/Rvapor > 1.
        ! ---------------------------------------
        tt = max(ta, frac_tmin)
        ! todo : verify fractionation over ice formulas
        do wiso = 1, niso
            alpha(wiso) = exp(alpha_ice_0(wiso) + alpha_ice_1(wiso) / tt + alpha_ice_2(wiso) / (tt**2))
            if(wiso == iso_O17) then
                alpha(wiso) = (alpha(wiso))**fac_coeff_eq17_ice
            endif
            alpha(wiso) = max(min(alpha(wiso), alpha_max), 0.0)
            ! effective fractionation over ice if necessary
            ! ---------------------------------------------
            if(wiso == iso_wat) then
                alpha(wiso) = 1.
            else
                if(tt < tmelt) then
                    supersat = musi - lambda_sursat * (tt - tmelt)
                    alpha(wiso) = alpha(wiso) * (supersat / (1.+alpha(wiso) * (supersat - 1.) * diffus_rel(wiso)))
                endif
            endif
            alpha(wiso) = max(min(alpha(wiso), alpha_max), 0.)
        enddo
    endif
endsubroutine fractcalk

subroutine Riso_from_qiso(wiso, qa_iso, qa, Riso)
    use mariso, only: negligible, Rdefault
    implicit none
    integer, intent(in) :: wiso
    real, intent(in) :: qa_iso, qa
    real, intent(out) :: Riso

    if(qa > negligible) then
        Riso = qa_iso / qa
    else
        Riso = Rdefault(wiso)
    endif
endsubroutine Riso_from_qiso
