module mariso
    use mardim, only: mx, my, mz, mw
    use marssn, only: nsx
    ! klonv = 1, because not
    use mar_sv, only: nsno, klonv
    use mar_tv, only: imx, jmx, nvx, llx
    implicit none
    ! niso_all : number of defined isotopes
    integer, parameter :: niso_all = 5
    ! niso : effective number of isotopes [2 to 4]
    integer, parameter :: niso = 3
    ! iso_wat : index of water
    integer, parameter :: iso_wat = 1
    ! iso_HDO : index of HDO
    integer, parameter :: iso_HDO = 2
    ! iso_O18 : index of H218O
    integer, parameter :: iso_O18 = 3
    ! iso_O17 : index of H217O
    integer, parameter :: iso_O17 = 4
    ! iso_HTO : index of tritium
    integer, parameter :: iso_HTO = 5

    ! local variables
    ! wiso : local loop
    integer :: wiso

    ! options
    ! =======
    ! options for initialization
    ! iso_init_type = 0 : R = Rdefault
    ! iso_init_type = 1 : R from observed delta
    integer, parameter :: iso_init_type = 0

    ! parameters
    ! ==========

    ! constants
    ! =========
    ! initialized in mariso_constants()
    ! negligible : minimal value for a mixing ratio to be considered as negligible
    ! LMDZ : negligible = ridicule
    ! if negligible, we don't test if isotopic composition is aberrant
    real, parameter :: negligible = 1e-12
    ! negligible_rain : negligible value for rain fluxes
    ! negligible_rain in kg m-2 s-1 <-> 1e-3 mm day-1
    real, parameter :: negligible_rain = 1e-8
    ! negligible_evap : negligible value for evap fluxes
    real, parameter :: negligible_evap = 1e-8 * 1e-2
    ! q_minsol : negligible value for qsol
    real, parameter :: q_minsol = 1e-8
    ! negligible_snow : negligible value for snowpack
    real, parameter :: negligible_snow = 1e-8
    ! q_min : negligible value for specific humidity
    ! LMDZ : q_min = qperemin in dyn3dmem/infotrac.f90
    ! real, parameter :: q_min = 1.e-16
    ! ratio_min : negligible value for isotopic ratio (q_iso / q)
    ! LMDZ : ratio_min = ratiomin in dyn3dmem/infotrac.f90
    ! real, parameter :: ratio_min = 1.e-16
    ! TODO : add details on expb_max
    real, parameter :: expb_max = 30.0
    ! deltaO18 at ocean surface
    real, parameter :: deltaO18_ocean = 0.
    ! lab_iso : isotope label
    character(len=3), save :: lab_iso(niso_all)
    ! RNsmow : Vienna Standard Mean Ocean Water (SMOW) [mol mol-1]
    ! LMDZ : RNsmow = tnat
    real, save :: RNsmow(niso_all)
    ! Rsmow : Vienna Standard Mean Ocean Water (SMOW) in mass [kg kg-1]
    ! in the code, R is the mass ratio, so Rsmow must be converted in mass
    ! LMDZ : Rsmow = tnat * Miso / M16
    real, save :: Rsmow(niso_all)
    ! Rdefault : initial ratio for all water components [mol mol-1]
    ! TODO : check Rdefault, why these values?
    ! todo : check if valid with the mass definition of Rsmow
    ! LMDZ : Rdefault = Rdefault for Rdefault_smow = .true.
    real, save :: Rdefault(niso_all)
    ! mwl_slope : slope of meteoric water line of delta_iso vs. delta_18O [-]
    ! LMDZ : mwl_slope = pente_MWL
    real, save :: mwl_slope(niso_all)
    ! fac_Rocean : correction factor for R at ocean surface [-]
    ! Rocean = Rsmow * (1 + delta_ocean / 1000.) * (1 + fac_Rocean)
    ! LMDZ : fac_Rocean = fac_enrichoce18 * mwl_slope
    real, save :: fac_Rocean(niso_all)
    ! Rocean : isotopic ratio at ocean surface [mol mol-1]
    ! todo : check if valid with the mass definition of Rsmow
    ! LMDZ : Rocean = toce * tcorr
    real, save :: Rocean(niso_all)
    ! alpha_liq_0, alpha_liq_1, alpha_liq_2 : coeff of equilibrium fractionation for liquid/vapor phases (Majoube, 1971b)
    ! alpha_liq_0 [-], alpha_liq_1 [K], alpha_liq_2 [K2]
    ! LMDZ : alpha_liq_0, alpha_liq_1, alpha_liq_2 = talph3, talph2, talph1
    real, save :: alpha_liq_0(niso_all), alpha_liq_1(niso_all), alpha_liq_2(niso_all)
    ! alpha_ice_0, alpha_ice_1, alpha_ice_2 : coeff of equilibrium fractionation for ice/vapor phases (Majoube, 1971b)
    ! alpha_ice_0 [-], alpha_ice_1 [K], alpha_ice_2 [K2]
    ! LMDZ : alpha_ice_0, alpha_ice_1, alpha_ice_2 = talps2, talps1
    ! TODO : update with Ellehøj et al. (2013) ? (see with M. Casado)
    real, save :: alpha_ice_0(niso_all), alpha_ice_1(niso_all), alpha_ice_2(niso_all)
    ! kcin [-] : kinetic factor for surface evaporation, with alphak = 1 / (1 - kcin)
    ! ws0cin : wind speed limit for computing kcin [m s-1]
    ! todo : update the equation of evaporation over ocean kcin with Françoise
    ! kcin_0, kcin_1, kcin_2 : kcin = kcin0 for ws < ws0cin and kcin = kcin1 * ws + kcin2 for ws > ws0cin
    ! LMDZ : kcin_0, kcin_1, kcin_2 = tkcin0, tkcin1, tkcin2 ; ws0cin = tv0cin
    real, parameter :: ws0cin = 7.
    real, save :: kcin_0(niso_all), kcin_1(niso_all), kcin_2(niso_all)
    ! diffus_rel : ratio of molecular diffusivities D/Di, with alphak = (D/Di)**diffus_exp
    ! in LMDZ : from Merlivat (1978), here updated with Barkan and Luz (2007)
    ! LMDZ : diffus_rel = tdifrel
    real, save :: diffus_rel(niso_all)
    ! diffus_exp : exponent of relative diffusivity in Stewart (1975), alpha_eff for equilibrium below cloud base
    ! LMDZ : diffus_exp = tdifexp
    real, parameter :: diffus_exp = 0.58
    ! diffus_exp_sol : exponent of relative diffusivity, parametrization of turbulence.
    ! Usually equal to 0.58, but Mathieu and Bariac find the exponent to be between 0.67 and 1
    ! diffus_exp = 0.67 for dry soils and diffus_exp = 1. for saturated soils.
    ! LMDZ : diffus_exp_sol = tdifexp_sol
    ! todo : check tdifexp_sol in LMDZ wateriso, might be equal to 0.8
    real, parameter :: diffus_exp_sol = 0.67
    ! rapport des ln(alphaeq) entre O18 et O17, donné par Amaelle
    real, parameter :: fac_coeff_eq17_liq = 0.529
    real, parameter :: fac_coeff_eq17_ice = 0.529

    !--------------------------------------------------------------
    ! Parameters that do not depend on the nature of water isotopes
    !--------------------------------------------------------------
    ! tmelt : temperature at which ice condensate starts to form (valeur ECHAM?) (K)
    ! LMDZ : tmelt = pxtmelt
    real, parameter :: tmelt = 273.15
    ! tice : temperature at which all condensate is ice (K)
    ! LMDZ : tice = pxtice
    real, parameter :: tice = 273.15 - 10.0
    ! frac_tmin, frac_tmax : minimum and maximun temperature to calculate fractionation coeff (K)
    ! note : coeffs were measured only above -40!
    ! LMDZ : frac_tmin = pxtmin, frac_tmax = pxtmax
    ! frac_tmin : valeur minimum de la température en K. Si elle est de l'ordre de quelques K seulement,
    ! les coeffs de fractionnement deviennent démesurément grands, et en plus ça fait planter l'execution à l'idris.
    real, parameter :: frac_tmin = 273.15 - 120.0
    real, parameter :: frac_tmax = 273.15 + 60.0
    ! factors lambda_sursat and mu in Si = musi - lambda_sursat * T
    real, parameter :: musi = 1.0
    ! Kd : diffusion in soil (m2/s)
    real, parameter :: Kd = 2.5e-9 !
    ! main isotopic parameters, from LMDZ DefLists/iso.def
    ! ====================================================
    ! lambda_sursat : supersaturation. Typical value: 0.002. Range: 0 (no supersaturation) to 0.004 (high supersaturation)
    real, parameter :: lambda_sursat = 0.002
    ! thumxt1: modulate cinetic fractionation during droplets re-evaporations. Typical value: 0.9. Range: 0 (rh near droplets = rh around droplets, strong cinetic fractionation) - 1 (rh = 1 near droplets, no cinetic fractionation)
    real, parameter :: thumxt1 = 0.9
    ! h_land_ice // todo : h_land_ice not relevant for MAR (?)
    ! P_veg: fraction de l'evaporation continentale provenant de la transpiration sans fractionnement de la végétation, au lieu de l évaporation avec fractionnement de sol nu ou d"eau libre. Valeur standard: 1 (comme autres GCMs). Range: 0 - 1.
    ! P_veg = 1.0 // todo : P_veg not relevant for MAR (?)
    ! iso_nudging: if .true. isotope H2O16 is nudged toward normal water. If .false., no nudging. In general, set to .true., but we check if H2O16 = iso_wat is not too far from normal water.
    ! LMDZ : iso_nudging = bidouille_anti_divergence
    logical, parameter :: iso_nudging = .true.
    ! essai_convergence: si T, on fait tout exactement comme pour l eau normale, pour que les 2 versions convergent. Si F, on modifie un peu: ex: initialisations suppl�mentaires de variables, on zappe l homog�n�isation sous le nuage dans cv3_yield...
    ! essai_convergence = .false. // todo : essai_convergence not relevant for MAR (?)
    ! initialisation_iso: la façon dont les isos sont initialisés.
    !   - Lors du lancement de gcm:
    !        Pour la physique:
    !    0: lecture d un fichier, souvent de start (phyiso_etat0_fichier.F)
    !    > 0: initialisation des isotopes académique, donnée dans phyiso_etat0_dur.F
    !        Pour la dynamique:
    !    0: lecture d un fichier (dyniso_etat0_fichier.F)
    !    > 0: initialisation académique:
    !        1: isos à 0 sauf H2016 = eau normale
    !        2: distill de Rayleigh
    !        3: tout est uniforme au SMOW
    !        En général, on met initialisation_iso = 0
    !   - Lors du lancement de create_etat0_limit:
    !        Pour la physique: initialisation en dure, toujours
    !        Pour la dynamique: initialisation académique toujours:
    !        1: isos à 0 sauf H2016 = eau normale
    !        2 ou 0: distill de Rayleigh
    !        3: tout est uniforme au SMOW
    !        En général, on laisse à initialisation_iso=0
    ! initialisation_iso = 2 // todo : iso_init_type = initialisation_iso -> test different initializations?
    ! initialisation_isotrac: comment initialiser les traceurs d'eau?
    ! 1: idéalisé: on met toute l'eau dans le tag izone_init, 0 ailleurs
    ! 0: en lisant dans un fichier. Possible seulement si initialisation_iso = 0
    ! initialisation_isotrac = 1 // todo : add water tracers?
    ! deltaO18_oce: ocean deltaO18: +1.1 for LGM, 0. for current days. Ocean dexcess and O17excess are supposed null.
    real, parameter :: deltaO18_oce = 0.
    ! deltaP_BL: boundary layer height for which soil water tends to equilibrate. Default: 10 mb (optim in ORCHIDEE_LMDZ)
    real, parameter :: deltaP_BL = 10.0
    ! ruissellement_pluie: qu'est-ce qui ruisselle? si 1: c'est la pluie qui ruisselle. elle ne s'infiltre. elle ne s'infiltre donc jamais dans un sol satur�. si 0: c'est le sol qui ruisselle. La pluie s'inglitre donc dans le sol satur�.
    ! ruissellement_pluie = 1 // todo : ruissellement_pluie not relevant for MAR (?)
    ! alphak_stewart: if 1, alphak=(D/Diso)^nsol ; if 0: alphak=1/(1-kcin(vsurf))
    integer, parameter :: alphak_stewart = 1
    ! tdifexp_sol: tdifexp_sol is the exponent of D/Diso, it parameterize turbulence. Usually equal to 0.58, but Mathieu and Bariac estimate it to be between 0.67 and 1: 0.67 for dry soils and 1 for saturated soils.
    real, parameter :: tdifexp_sol = 0.67
    ! deltaD_max : maximum value of deltaD, if deltaD greater than deltaD_max, model crash
    ! LMDZ : deltaD_max = deltalim = deltalimtrac
    real, parameter :: deltaD_max = 1000000.
    ! dexcess_max : maximum value of dexcess_max, if dexcess greater than dexcess_max, model crash
    real, parameter :: dexcess_max = 2000
    ! O17 limits : todo O17 limits
    ! O17_verif = y
    ! o17excess_bas = -1000.0
    ! o17excess_haut = 900.0
    ! nlevmaxO17 = 18
    ! water tagging : todo water tagging
    ! quand on utilise tag 17, on taggue la vapeur résiduelle quand la fraction condensée est supérieure à seuil_tag_tmin, c'est à dire quand l'évènement de condensation affecte suffisemment la compo de la vapeur résiduelle.
    ! Si seuil_tag_tmin=0, le moindre évènement de condensation même très faible agit sur les tags, et les proportions de sfc, condensat et rev sont presques nulles.
    ! par defaut: seuil_tag_tmin=0.01: 1% de condensation <-> 0.1 permil d'effet sur la vapeur résiduelle.
    ! seuil_tag_tmin = 0.01
    ! idem pour condensation dans LS: par defaut égal à seuil_tag_tmin
    ! seuil_tag_tmin_ls = 0.01
    ! option_seuil_tag_tmin: si 1, on recolorise la vapeur résiduelle dès que cond/qt>seuil_tag_tmin.
    ! Ca taggue bien la vap résiduelle mais le problème est que plus on humidifie par la microphysique, plus le seuil est facilement atteint -> plus on retaggue -> au final, on n'a pas plus de tag mcrophysique alors que c'est la source d'humdification.
    ! dans l'option 2: on taggue si (cond-qmicro)/((qt-qmicro)>seuil
    ! option_seuil_tag_tmin = 2
    ! lim_tag20 : latitude de taggage des extra-tropiques quand option de taggage no 20
    ! 35.0 b defaut
    ! lim_tag20 = 35.0
    ! quand tag9, recolorise t'on que le condensat convectif (option 2) ou aussi le condensat starti (otion 1)
    ! option_cond = 2
    ! quand tag 22, on recolorise la vapeur dans les zones ou precip>lim_precip_tag22
    ! 10.0 par défaut
    ! lim_precip_tag22 = 10.0
    ! no_pce : if =1, no post-condensational exchanges as in Field, Jones and Brown. no_pce = 0 : fractionation as usual.
    ! integer, parameter :: no_pce = 0
    ! quand adv 14, saturation limiter tel que on advec min (q,A_satlim*qs)
    ! par defaut, A_satlim=1: c'est le code original de Francis Codron
    ! ce param n'a d'effet que quand adv 14.
    ! A_satlim = 0.8
    ! si ok_restrict_A_satlim=1, alors on n'active adv 14 que dans les extra-tropiques
    ! si ok_restrict_A_satlim=2, alors on n'active adv 14 que dans les tropiques
    ! si ok_restrict_A_satlim=3, alors on n'active adv 14 nulle part
    ! ok_restrict_A_satlim = 0
    ! defaut: 2; pour adv 10: 0
    ! slope_limiter = 2.
    ! on multiplie par 3 ratqs aux extraropiques si modif_ratqs=1.
    ! modif_ratqs = 0

    ! variables
    ! =========

    ! svasav : sisvat snow water
    ! --------------------------
    ! rosSNo_iso : Snow Volumic Mass [kg/m3]
    real, save :: rosSNo_iso(niso, mx, my, nsx, nsno)
    ! wasSNo_iso: Soil humidity content (=> in the snow cover ) [kg/kg]
    real, save :: wasSNo_iso(niso, mx, my, nsx, nsno)
    ! SWaSNo_iso: Surficial Water Mass [kg/m2]
    real, save :: SWaSNo_iso(niso, mx, my, nsx)
    ! snohSN_iso : Snow Buffer Layer Thickness [mmWE]
    real, save :: snohSN_iso(niso, mx, my, nsx)

    ! soil water
    ! ----------
    ! eta_TV : Soil Moisture Content [m3/m3]
    real, save :: eta_TV_iso(niso, imx, jmx, nvx, llx)
    ! evapTV: Time Integrated Evapotranspiration [mm w.e.]
    real, save :: evapTV_iso(niso, imx, jmx)

    ! surface air water (for soil)
    ! ----------------------------
    ! qvapSL : specific humidity close to the surface (kg/kg)
    real, save :: qvapSL_iso(niso, mx, my)
    ! SLuqs_iso :: Surface Specific Humidity Turbulent Flux (Av.) (kg/kg m/s)
    ! evaporation flux from the surface
    real, save :: SLuqs_iso(niso, mx, my)
    ! SLuqsl_iso :: Surface Specific Humidity Turbulent Flux by mosaic (kg/kg m/s)
    ! evaporation flux from the surface by mosaic
    real, save :: SLuqsl_iso(niso, mx, my, mw)
    ! uqs_SV <-> SLuqsl_iso in sisvat
    real, save :: uqs_SV_iso(niso, klonv)

    ! dynamical variables
    ! -------------------
    ! qvDY_iso : Specific Humidity (kg/kg)
    real, save :: qvDY_iso(niso, mx, my, mz)

    ! microphysics water
    ! ------------------
    !CAa : quid de : snfHY = 0.0, sblHY, rnfHY, evpHY? (see iniphy)
    ! qiHY : cloud ice crystals concentration (kg/kg)
    real, save :: qiHY_iso(niso, mx, my, mz)
    ! qsHY : snow flakes concentration (kg/kg)
    real, save :: qsHY_iso(niso, mx, my, mz)
    ! qwHY : cloud dropplets concentration (kg/kg)
    real, save :: qwHY_iso(niso, mx, my, mz)
    ! qrHY : rain concentration (kg/kg)
    real, save :: qrHY_iso(niso, mx, my, mz)
    ! rainHY : integrated precipited rain
    real, save :: rainHY_iso(niso, mx, my)
    ! rai0HY : integrated precipited rain (previous time step)
    real, save :: rai0HY_iso(niso, mx, my)
    ! snowHY : integrated precipited/eroded snow
    real, save :: snowHY_iso(niso, mx, my)
    ! sno0HY : integrated precipited/eroded snow (previous time step)
    real, save :: sno0HY_iso(niso, mx, my)
    ! sfa0HY : integrated precipited snow (previous time step)
    real, save :: sfa0HY_iso(niso, mx, my)
    real, save :: crysHY_iso(niso, mx, my)
    ! qsrfHY : Blowing Snow Concentration (0.325 m above the surface)
    real, save :: qsrfHY_iso(niso, mx, my)
    ! qSalSV <-> qsrfHY in sisvat
    real, save :: qSalSV_iso(niso, klonv)

    ! convective water
    ! ----------------
    real, save :: dqv_CA_iso(niso, mx, my, mz)
    real, save :: dqw_CA_iso(niso, mx, my, mz)
    real, save :: dqi_CA_iso(niso, mx, my, mz)
    real, save :: drr_CA_iso(niso, mx, my)
    real, save :: dss_CA_iso(niso, mx, my)
    real, save :: dsn_CA_iso(niso, mx, my)
    real, save :: rainCA_iso(niso, mx, my)
    real, save :: snowCA_iso(niso, mx, my)

contains

    subroutine mariso_constants()
        ! initialization of constant values for water isotopes
        implicit none
        ! lab_iso : isotope label
        lab_iso(iso_wat) = 'wat'
        lab_iso(iso_O18) = 'O18'
        lab_iso(iso_HDO) = 'HDO'
        lab_iso(iso_O17) = 'O17'
        lab_iso(iso_HTO) = 'HTO'
        ! RNsmow : Vienna Standard Mean Ocean Water (SMOW) [mol mol-1]
        RNsmow(iso_wat) = 1.
        RNsmow(iso_O18) = 2005.2e-6
        RNsmow(iso_HDO) = 155.76e-6
        RNsmow(iso_O17) = 379.9e-6
        RNsmow(iso_HTO) = 0.
        ! Rsmow : SMOW in mass [kg kg-1]
        Rsmow(iso_wat) = 1.
        Rsmow(iso_O18) = RNsmow(iso_O18) * (18.+2.) / (16.+2.)
        Rsmow(iso_HDO) = RNsmow(iso_HDO) * (16.+3.) / (16.+2.)
        Rsmow(iso_O17) = RNsmow(iso_O17) * (17.+2.) / (16.+2.)
        Rsmow(iso_HTO) = 0.
        ! mwl_slope : slope of meteoric water line of delta_iso vs. delta_18O [-]
        mwl_slope(iso_wat) = 0.
        mwl_slope(iso_O18) = 1.
        mwl_slope(iso_HDO) = 8.
        mwl_slope(iso_O17) = 0.528
        mwl_slope(iso_HTO) = 0.
        ! Rdefault : initial ratio for all water components [mol mol-1]
        Rdefault(iso_wat) = Rsmow(iso_wat) * 1.
        Rdefault(iso_O18) = Rsmow(iso_O18) * (1.-6./1000.)
        Rdefault(iso_HDO) = Rsmow(iso_HDO) * (1.-(6.*mwl_slope(iso_HDO) + 10.) / 1000.)
        Rdefault(iso_O17) = Rsmow(iso_O17) * (1.-3.15 / 1000.)
        Rdefault(iso_HTO) = 0.
        ! fac_Rocean : correction factor for R at ocean surface [-]
        fac_Rocean(iso_wat) = 0.
        fac_Rocean(iso_O18) = 0.0005
        fac_Rocean(iso_HDO) = fac_Rocean(iso_O18) * mwl_slope(iso_HDO)
        fac_Rocean(iso_O17) = fac_Rocean(iso_O18) * mwl_slope(iso_O17)
        fac_Rocean(iso_HTO) = 0.
        ! Rocean : isotopic ratio for ocean [mol mol-1]
        ! Rocean : todo : check why these formulas
        Rocean(iso_wat) = Rsmow(iso_wat)
        Rocean(iso_O18) = Rsmow(iso_O18) * (1.+deltaO18_ocean / 1000.) * (1.+fac_Rocean(iso_O18))
        Rocean(iso_HDO) = Rsmow(iso_HDO) * (1.+deltaO18_ocean * mwl_slope(iso_HDO) / 1000.) * (1.+fac_Rocean(iso_HDO))
        Rocean(iso_O17) = Rsmow(iso_O17) * (1.+deltaO18_ocean / 1000.)**mwl_slope(iso_O17) * (1.+fac_Rocean(iso_O17))
        ! Rocean(iso_HTO) : rapport T/H = 0.2 TU Dreisigacker and Roether 1978 (corrigé par Alex Cauquoin)
        ! todo : Rocean(iso_HTO), verify the ratio (16. + 4.) / (16. + 2.) here
        Rocean(iso_HTO) = 4.e-19 * (16.+4.) / (16.+2.)
        ! alpha_liq_0, alpha_liq_1, alpha_liq_2 : coeff of equilibrium fractionation for liquid/vapor phases (Majoube, 1971b)
        alpha_liq_0(iso_wat) = 0.
        alpha_liq_1(iso_wat) = 0.
        alpha_liq_2(iso_wat) = 0.
        alpha_liq_0(iso_O18) = -2.0667e-3
        alpha_liq_1(iso_O18) = -0.4156
        alpha_liq_2(iso_O18) = 1137.
        alpha_liq_0(iso_HDO) = 52.612e-3
        alpha_liq_1(iso_HDO) = -76.248
        alpha_liq_2(iso_HDO) = 24844.
        alpha_liq_0(iso_O17) = alpha_liq_0(iso_O18)
        alpha_liq_1(iso_O17) = alpha_liq_1(iso_O18)
        alpha_liq_2(iso_O17) = alpha_liq_2(iso_O18)
        alpha_liq_0(iso_HTO) = 0.
        alpha_liq_1(iso_HTO) = -103.87
        alpha_liq_2(iso_HTO) = 46480.
        ! alpha_ice_0, alpha_ice_1, alpha_ice_2 : coeff of equilibrium fractionation for ice/vapor phases (Majoube, 1971b)
        ! TODO : update with Ellehøj et al. (2013) ? (see with M. Casado)
        alpha_ice_0(iso_wat) = 0.
        alpha_ice_1(iso_wat) = 0.
        alpha_ice_2(iso_wat) = 0.
        alpha_ice_0(iso_O18) = -0.028244
        alpha_ice_1(iso_O18) = 11.839
        alpha_ice_2(iso_O18) = 0.
        alpha_ice_0(iso_HDO) = -0.0934
        alpha_ice_1(iso_HDO) = 0.
        alpha_ice_2(iso_HDO) = 16288.
        alpha_ice_0(iso_O17) = alpha_ice_0(iso_O18)
        alpha_ice_1(iso_O17) = alpha_ice_1(iso_O18)
        alpha_ice_2(iso_O17) = alpha_ice_2(iso_O18)
        alpha_ice_0(iso_HTO) = 0.
        alpha_ice_1(iso_HTO) = -103.87
        alpha_ice_2(iso_HTO) = 46480.
        ! diffus_rel : ratio of molecular diffusivities D/Di, with alphak = (D/Di)**diffus_exp
        diffus_rel(iso_wat) = 1.
        ! diffus_rel(iso_O18) = 1. / 0.9723 ! Merlivat (1978)
        diffus_rel(iso_O18) = 1./0.9691
        ! diffus_rel(iso_HDO) = 1. / 0.9755 ! Merlivat (1978)
        diffus_rel(iso_HDO) = 1./0.9839
        ! diffus_rel(iso_O17) = 1. / 0.985452 ! donné par Amaelle
        diffus_rel(iso_O17) = 1./0.98555 ! valeur utilisée en 1D et dans modèle de LdG
        diffus_rel(iso_HTO) = 1./0.968
        ! kcin_0, kcin_1, kcin_2 : kinetic factors for surface evaporation, with alphak = 1 / (1 - kcin)
        kcin_0(iso_wat) = 0.
        kcin_1(iso_wat) = 0.
        kcin_2(iso_wat) = 0.
        kcin_0(iso_O18) = 0.006
        kcin_1(iso_O18) = 0.000285
        kcin_2(iso_O18) = 0.00082
        kcin_0(iso_HDO) = kcin_0(iso_O18) * (diffus_rel(iso_HDO) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_1(iso_HDO) = kcin_1(iso_O18) * (diffus_rel(iso_HDO) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_2(iso_HDO) = kcin_2(iso_O18) * (diffus_rel(iso_HDO) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_0(iso_O17) = kcin_0(iso_O18) * (diffus_rel(iso_O17) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_1(iso_O17) = kcin_1(iso_O18) * (diffus_rel(iso_O17) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_2(iso_O17) = kcin_2(iso_O18) * (diffus_rel(iso_O17) - 1.) / (diffus_rel(iso_O18) - 1.)
        kcin_0(iso_HTO) = 0.01056
        kcin_1(iso_HTO) = 0.0005016
        kcin_2(iso_HTO) = 0.0014432
    endsubroutine mariso_constants

    real function R_to_delta(iso, Riso) result(delta)
        implicit none
        integer, intent(in) :: iso
        real, intent(in) :: Riso
        delta = (Riso / Rsmow(iso) - 1.) / 1000.
    endfunction R_to_delta

    real function delta_to_R(iso, delta) result(Riso)
        implicit none
        integer, intent(in) :: iso
        real, intent(in) :: delta
        Riso = (delta / 1000.+1.) * Rsmow(iso)
    endfunction delta_to_R

endmodule mariso
