#include "MAR_pp.def"
! +------------------------------------------------------------------------+
! | MAR          SISVAT_dat                                31-12-2024  MAR |
! |   SubRoutine SISVAT_dat contains the constants of the                  |
! |              Soil/Ice Snow Vegetation Atmosphere Transfer Scheme       |
! +------------------------------------------------------------------------+
module mardsv
    use mar_sv
    implicit none
    ! +--SISVAT Global Variables
    ! +  =======================
    ! INI_SV :Initialisation Switch
    logical, save :: INI_SV = .false.
    ! eps_21 : Arbitrary  very small value
    real, parameter :: eps_21 = 1.e-21
    ! +--Snow
    ! +  ----
    ! istdSV : Snow History
    !   1: faceted cristal
    !   2: liq.watr/no faceted cristal before
    !   3: liq.watr/   faceted cristal before
    integer, parameter :: istdSV(5) = (/1, 2, 3, 4, 5/)
    ! Cn_dSV : Snow Heat Capacity [J/kg/K]
    ! Loth et al. 1993, JGR 98 D6 2.2.2 2e para p.10453
    real, parameter :: Cn_dSV = 2105.
    ! SMndSV : Minimum Thickness of new Layers [mmWE]
    real, parameter :: SMndSV = 1.0
    ! G1_dSV : Conversion 0/99-->0/1 Sphericity/Dendricity
    real, parameter :: G1_dSV = 99.
    ! DDcdSV, DFcdSV, DScdSV : Snow Grains Optical Diameter [1e-4m]
    ! DDcdSV : Dendritic Crystals [0.0001 m]
    real, parameter :: DDcdSV = 1.
    ! DFcdSV : Young Faceted Crystals    [0.0001 m]
    real, parameter :: DFcdSV = 4.
    ! Small Crystals    [0.0001 m]
    real, parameter :: DScdSV = 3.
    ! ADSdSV : Snow Grains Actual Diameter [1e-4m]
    real, parameter :: ADSdSV = 4.
    ! So1dSV, So2dSV, So3dSV : Total Solar Irradiance Fractions [-]
    ! Fractions of total Solar Irradiance in 3 spectral Intervals
    ! (see Feagle and Businger 1981, Int.Geoph.Ser. 25, p.215-222)
    ! So1dSV : 0.3--0.8mim Interval
    real, parameter :: So1dSV = 0.6
    ! So2dSV : 0.8--1.5mim Interval
    real, parameter :: So2dSV = 0.3
    ! So3dSV : 1.5--2.8mim Interval
    real, parameter :: So3dSV = 0.1
    ! initial values
    ! real, parameter :: So1dSV = 0.606
    ! real, parameter :: So2dSV = 0.301
    ! real, parameter :: So3dSV = 0.093
    ! Tuning ETH camp
    !XF real, parameter :: So1dSV = 0.580
    !XF real, parameter :: So2dSV = 0.320
    !XF real, parameter :: So3dSV = 0.100
    ! aI1dSV, aI2dSV, aI3dSV : Bare Ice Albedo [-]
    ! aI1dSV : Minimum bare ICE albedo [-]
    real, parameter :: aI1dSV = 0.50
    ! aI2dSV : Maximum bare ICE albedo [-]
    real, parameter :: aI2dSV = 0.55
    ! aI3dSV : ICE lense albedo at roCdSV kg/m3 and and minimum pure snow albedo
    real, parameter :: aI3dSV = 0.70
    !mode Greenland over temperate climate: aI1dSV=aI2dSV=0.55 or 0.6
    ! ws0dSV : Irreducible Water Saturation in Snow
    !   ws0dSV = 0.07: Coleou et al., 1998, A.Gl
    !   ws0dSV = 0.08-0.15 : Greuell & Konzelmann (199
#ifdef AC
    real, parameter :: ws0dSV = 0.05
#else
    real, parameter :: ws0dSV = 0.07 ! Greenland
#endif
    ! roCdSV : Pore hole close-off density [kg/m3]
    ! roCdSV = 800: Greuell & Konzelmann (1994), Glo
    ! roCdSV = 830: Harper et al. (2012), Nature
    real, parameter :: roCdSV = 830.
    ! roSdSV : Max pure snow density [kg/m3]
    real, parameter :: roSdSV = 450.
    ! roBdSV : Max blowing snow density [kg/m3]
    real, parameter :: roBdSV = 450.
    ! ru_dSV : Surficial Water Scale Factor [kg/m2]
    real, parameter :: ru_dSV = 50.
    ! +--Ice
    ! +  ---
    ! CdidSV : Conductivity of pure Ice [W/m/K]
    real, parameter :: CdidSV = 2.1
    ! +--Vegetation
    ! +  ----------
    ! nvgt : number of vegetation classes
    integer, parameter :: nvgt = 13
    ! DH_dSV : Displacement Height [m]
    real, parameter :: DH_dSV(0:nvgt) = (/ &
                       ! 0 NO VEGETATION
                       0.00, &
                       ! 1 CROPS LOW       | 2 CROPS MEDIUM       | 3 CROPS HIGH
                       0.07, 0.21, 0.70, &
                       ! 4 GRASS LOW       | 5 GRASS MEDIUM       |  6 GRASS HIGH
                       0.07, 0.21, 0.70, &
                       ! 7 BROADLEAF LOW   | 8 BROADLEAF MEDIUM   | 9 BROADLEAF HIGH
                       1.40, 5.60, 14.00, &
                       ! 10 NEEDLELEAF LOW | 11 NEEDLELEAF MEDIUM | 12 NEEDLELEAF HIGH
                       1.40, 5.60, 14.00, &
                       ! 13 City
                       21.00/)
    ! Z0mdSV : Roughness  Length for Momentum [m]
    real, parameter :: Z0mdSV(0:nvgt) = (/ &
                       ! 0 NO VEGETATION
                       0.01, &
                       ! 1 CROPS LOW       | 2 CROPS MEDIUM       | 3 CROPS HIGH
                       0.01, 0.03, 0.10, &
                       ! 4 GRASS LOW       | 5 GRASS MEDIUM       |  6 GRASS HIGH
                       0.01, 0.03, 0.10, &
                       ! 7 BROADLEAF LOW   | 8 BROADLEAF MEDIUM   | 9 BROADLEAF HIGH
                       0.20, 0.80, 2.00, &
                       ! 10 NEEDLELEAF LOW | 11 NEEDLELEAF MEDIUM | 12 NEEDLELEAF HIGH
                       0.20, 0.80, 2.00, &
                       ! 13 City
                       3.00/)
    ! StodSV : Minimum    Stomatal Resistance [s/m]
    real, parameter :: StodSV(0:nvgt) = (/ &
                       ! 0 NO VEGETATION
                       5000., &
                       ! 1 CROPS LOW       | 2 CROPS MEDIUM       | 3 CROPS HIGH
                       50., 50., 50., &
                       ! 4 GRASS LOW       | 5 GRASS MEDIUM       |  6 GRASS HIGH
                       50., 50., 50., &
                       ! 7 BROADLEAF LOW   | 8 BROADLEAF MEDIUM   | 9 BROADLEAF HIGH
                       10., 10., 10., &
                       ! 10 NEEDLELEAF LOW | 11 NEEDLELEAF MEDIUM | 12 NEEDLELEAF HIGH
                       10., 10., 10., &
                       ! 13 City
                       5000./)
    ! PR_dSV : Internal Plant Resistance [s]
    real, parameter :: PR_dSV(0:nvgt) = (/ &
                       ! 0 NO VEGETATION
                       0.0, &
                       ! 1 CROPS LOW       | 2 CROPS MEDIUM       | 3 CROPS HIGH
                       0.5e9, 0.5e9, 0.5e9, &
                       ! 4 GRASS LOW       | 5 GRASS MEDIUM       |  6 GRASS HIGH
                       0.5e9, 0.5e9, 0.5e9, &
                       ! 7 BROADLEAF LOW   | 8 BROADLEAF MEDIUM   | 9 BROADLEAF HIGH
                       1.0e9, 1.0e9, 1.0e9, &
                       ! 10 NEEDLELEAF LOW | 11 NEEDLELEAF MEDIUM | 12 NEEDLELEAF HIGH
                       1.0e9, 1.0e9, 1.0e9, &
                       ! 13 City
                       0.0/)
    ! rbtdSV : Roots Fraction Beta Coefficient [-]
    real, parameter :: rbtdSV(0:nvgt) = (/ &
                       ! 0 NO VEGETATION
                       0.000, &
                       ! 1 CROPS LOW       | 2 CROPS MEDIUM       | 3 CROPS HIGH
                       0.961, 0.961, 0.961, &
                       ! 4 GRASS LOW       | 5 GRASS MEDIUM       |  6 GRASS HIGH
                       0.943, 0.964, 0.972, &
                       ! 7 BROADLEAF LOW   | 8 BROADLEAF MEDIUM   | 9 BROADLEAF HIGH
                       0.968, 0.962, 0.962, &
                       ! 10 NEEDLELEAF LOW | 11 NEEDLELEAF MEDIUM | 12 NEEDLELEAF HIGH
                       0.971, 0.976, 0.976, &
                       ! 13 City
                       0.000/)
    ! pscdSV : Critical Leaf Water Potential [m]
    real, parameter :: pscdSV = 250.
    ! StxdSV : maximum Stomatal Resistance [s/m]
    real, parameter :: StxdSV = 5000.
    ! LAIdSV : maximum LAI
    real, parameter :: LAIdSV = 4.
    ! +--Soil
    ! +  ----
    ! rcwdSV : Density * Water Specific Heat
    real, parameter :: rcwdSV = 4.180e+6
    ! dz_dSV : Soil Vertical Discretization (Layer's Thickness)
    real, save :: dz_dSV(-nsol:0)
    ! zz_dSV : Soil Thickness
    real, save :: zz_dSV
    ! number of soil layers
    integer, parameter :: nsot = 12
    ! etadSV : Water Content at Saturation [m3/m3]
    real, parameter :: etadSV(0:nsot) = (/ &
                       ! 0 WATER           ! 1 SAND            ! 2 LOAMY SAND
                       1.000, 0.395, 0.410, &
                       ! 3 SANDY LOAM      ! 4 SILT LOAM       ! 5 LOAM
                       0.435, 0.485, 0.451, &
                       ! 6 SANDY CLAY LOAM ! 7 SILTY CLAY LOAM ! 8 CLAY LOAM
                       0.420, 0.477, 0.476, &
                       ! 9 SANDY CLAY      ! 10 SILTY CLAY     ! 11 CLAY very wet
                       0.426, 0.492, 0.482, &
                       ! 12 ICE
                       0.001/)
    ! psidSV : Water Succion at Saturation [m]
    real, parameter :: psidSV(0:nsot) = (/ &
                       ! 0 WATER           ! 1 SAND            ! 2 LOAMY SAND
                       1.000, 0.121, 0.090, &
                       ! 3 SANDY LOAM      ! 4 SILT LOAM       ! 5 LOAM
                       0.218, 0.786, 0.478, &
                       ! 6 SANDY CLAY LOAM ! 7 SILTY CLAY LOAM ! 8 CLAY LOAM
                       0.299, 0.356, 0.630, &
                       ! 9 SANDY CLAY      ! 10 SILTY CLAY     ! 11 CLAY very wet
                       0.153, 0.490, 0.405, &
                       ! 12 ICE
                       0.001/)
    ! Ks_dSV : Hydraulic Conductivity at Saturation [m/s]
    real, parameter :: Ks_dSV2(0:nsot) = (/ &
                       ! 0 WATER           ! 1 SAND            ! 2 LOAMY SAND
                       0.0e00, 176.0e-8, 156.3e-8, &
                       ! 3 SANDY LOAM      ! 4 SILT LOAM       ! 5 LOAM
                       34.1e-8, 7.2e-8, 7.0e-8, &
                       ! 6 SANDY CLAY LOAM ! 7 SILTY CLAY LOAM ! 8 CLAY LOAM
                       6.3e-8, 1.7e-8, 2.5e-8, &
                       ! 9 SANDY CLAY      ! 10 SILTY CLAY     ! 11 CLAY very wet
                       2.2e-8, 1.0e-8, 1.3e-8, &
                       ! 12 ICE
                       0.0e00/)
    real, parameter :: Ks_dSV(0:nsot)=Ks_dSV2(0:nsot)*4. ! XF, 31/12/2024
    ! higher Ks_dSV is, colder MAR is! 
    ! bCHdSV : Clapp-Hornberger Coefficient b [-]
    real, parameter :: bCHdSV(0:nsot) = (/ &
                       ! 0 WATER           ! 1 SAND            ! 2 LOAMY SAND
                       1.00, 2.05, 2.38, &
                       ! 3 SANDY LOAM      ! 4 SILT LOAM       ! 5 LOAM
                       2.90, 3.30, 3.39, &
                       ! 6 SANDY CLAY LOAM ! 7 SILTY CLAY LOAM ! 8 CLAY LOAM
                       5.12, 5.75, 6.52, &
                       ! 9 SANDY CLAY      ! 10 SILTY CLAY     ! 11 CLAY very wet
                       8.40, 8.40, 9.40, &
                       ! 12 ICE
                       0.02/)
    ! +--Water Bodies
    ! +  ------------
    ! vK_dSV : Diffusivity in Water [m2/s]
    real, parameter :: vK_dSV = 1000.
    ! TSIdSV : Sea-Ice Fraction: SST Scale [K]
    real, parameter :: TSIdSV = 0.50
endmodule mardsv
