#include "MAR_pp.def"
! mar_ecrad_mod.f90
! 
!   Created on: 02/05/2023
!       Author: J.-F. Grailet (jefgrailet)
! 
! This simple module was added to manage additional variables that are used to interface MAR with 
! the ecRad radiation scheme from the ECMWF. It currently consists in an updated CMIP_scenario 
! variable (longer than before), a logical to remember whether or not ecRad has been initialized, 
! and a set of parameters that can be modified by MAR users to control the ecRad configuration in 
! the same manner as some MAR parameters are controlled via the marmagic module.

module mar_ecrad

    implicit none
    
    ! Initialization of ecRad
    ! =======================
    ! Simple logical used by PHYrad_ECRAD_in to remember whether or not the ecRad radiation scheme 
    ! and some MAR data it relies on (e.g. greenhouse/aerosol climatologies) have been initialized.
    
    logical, save :: ecRad_initialized = .false.
    
    ! CMIP scenario (30 char.)
    ! ========================
    ! Actually a suffix (minus the .nc extension) for a time series NetCDF file from the ECMWF. 
    ! The length allows the longest suffixes found in the ECMWF file names (up to 29 characters).
    
    character(len=30), save :: CMIP_scenario
    
    ! Solar constant scaling
    ! ======================
    ! Knowing that the solar constant in ecRad is hard-coded as RI0 = 1366 W/m², tune this real 
    ! parameter to adjust this constant for MAR needs. It is advised to keep the value 0.9962 in 
    ! most situations, as this scales the solar constant to the value used by the Morcrette 
    ! radiation scheme while interfaced to MAR (1360.8 W/m²). From a physical point of view, 
    ! scaling down amounts to taking account of the small energy loss occurring in the upper 
    ! layers of the atmosphere (1366 W/m² being for the top of atmosphere) that are not modeled 
    ! in the MAR grid.
    
    real, parameter :: solar_constant_scaling = 0.9962
    
    ! Extra stratospheric layers
    ! ==========================
    ! Tune the next parameter to set up extra pressure layers that will be added on top of the MAR 
    ! grid just before calling the ecRad radiation scheme. The idea is to ensure ecRad can 
    ! properly simulate the spectral and broadband effects of the stratosphere, and in particular 
    ! the impact of the ozone peak concentration (which is found in the stratosphere, outside the 
    ! scope of the usual MAR grid). This parameter is especially important if MAR is configured to 
    ! use or produce spectral fluxes in bands that overlap the spectral range where ozone is 
    ! optically active (UV and visible). The extra layers are exclusively used upon calling ecRad 
    ! and are removed when redirecting ecRad outputs to MAR.
    ! 
    ! The blanket_layers array must be configured as follows. It must provide X ascending real 
    ! values corresponding to X extra pressure levels spanning from the stratopause (1 hPa; 
    ! mandatory) to before the top of the MAR grid (maximum 50 hPa; the MAR top sits usually 
    ! around 70 hPa). The given values correspond to the pressure levels from the greenhouse gas 
    ! and aerosol forcings whose mixing ratios will be mapped to each layer, the exact boundaries 
    ! of each layer being halfway between the pressure values within the array, except for the 
    ! first layer whose top is always 0 hPa.
    ! 
    ! E.g., (/ 1., 10., 30. /) configures three layers. The first layer will extend from 0 hPa to 
    ! 5.5 hPa (halfway between 1 and 10 hPa), and will feature the greenhouse gas and aerosol 
    ! mixing ratios found at 1 hPa in the forcings. Likewise, the second layer will extend from 
    ! 5.5 hPa to 20 hPa and will have the mixing ratios found in the forcings at 10 hPa. Adding 
    ! the {1, 10, 30} configuration above the MAR grid yields a vertical integration of ozone (in 
    ! g/m^2) close to the vertical integration of the forcings above Europe, said forcings having 
    ! a higher vertical resolution than MAR. To not use extra layers, set all values to 0.
    
    real, parameter, dimension(3) :: blanket_layers = (/ 1., 10., 30. /) ! 0 to undo
    
    ! Use ecCKD high resolution gas models [ T / F ]
    ! ==============================================
    ! Keep this parameter true to use the ecCKD high resolution gas models (96 intervals for 
    ! shortwave, 64 for longwave) within ecRad. Using such models is especially important if you 
    ! intend to output fine spectral data. However, if such models become problematic, you can 
    ! set this parameter to false to go back to the old RRTM-G model (with 14 bands for shortwave 
    ! and 16 for longwave).
    
    logical, parameter :: use_ecckd_high_res = .true.

    ! Cloud parametrization [ 0 - 4 ]
    ! ===============================
    ! Set to a value between 0 and 4 (included) to pick a cloud parametrization to compute the 
    ! cloud fraction values sent to ecRad. It is advised to use Sundqvist's parametrization (3) or 
    ! Xu & Randall's semiempirical formula (4) over Europe, while the old ECMWF large scale 
    ! cloudiness formula with snowflakes (2) should be preferred for polar areas. Note that the 
    ! ECMWF large scale cloudiness formula has been used up to MAR v3.13 but where it comes from 
    ! seems to have been lost to time (today's ECMWF scientists no longer know about it).
    ! 
    ! 0 = MAR cloud fraction (as computed in hydmic.f90)
    ! 1 = ECMWF large scale cloudiness, crude version (without snowflakes)
    ! 2 = ECMWF large scale cloudiness, complete version (same formula as in hydmic.f90)
    ! 3 = Humidity-based parametrization by Sundqvist (1989 in Monthly Weather Review 117)
    ! 4 = Semiempirical parametrization by Xu and Randall (1996 in JAS)

#if(GR)
    integer, parameter :: cloud_parametrization = 2
#endif
#if(AC)
    integer, parameter :: cloud_parametrization = 2
#endif
#if(EU)
    integer, parameter :: cloud_parametrization = 3
#endif

    ! Cloud fraction gamma
    ! ====================
    ! Tune this parameter to increase/decrease cloud fraction values computed with the ECMWF large 
    ! scale cloudiness formula (1 or 2 in the above parameter). For Europe, this value should be 
    ! set to around 0.02 (value used in MAR v3.13 and prior in all cases), but for polar aeras 
    ! such as Greenland, prefer using 0.005.
    
#if(GR)
    real, parameter :: cloud_fraction_ecmwf_gamma = 0.005
#endif
#if(AC)
    real, parameter :: cloud_fraction_ecmwf_gamma = 0.005
#endif
#if(EU)
    real, parameter :: cloud_fraction_ecmwf_gamma = 0.01 !XF 31/12/2024
   !real, parameter :: cloud_fraction_ecmwf_gamma = 0.02 
#endif
    
    ! Fractional standard deviation of in-cloud water content
    ! =======================================================
    ! Tune this parameter to control the heterogeneity of the water content inside the clouds 
    ! modeled by ecRad solvers for each grid cell. The smaller, the more homogeneous (close to the 
    ! mean) clouds will be. Using more homogeneous clouds tends to reduce shortwave fluxes while 
    ! sligthly increasing longwave fluxes. Shonk et al. (2010) recommends using 0.75±0.18, but in 
    ! the context of a regional climate model (i.e. with high resolution), using 0.5 is acceptable 
    ! (checked with R. Hogan from ECMWF).
    ! 
    ! See also Table 1 in http://www.met.rdg.ac.uk/~swrhgnrj/publications/shonk_part1.pdf

#if(GR)
    real, parameter :: cloud_water_fractional_std = 0.6 
#endif
#if(AC)
    real, parameter :: cloud_water_fractional_std = 0.6
#endif
#if(EU)
    real, parameter :: cloud_water_fractional_std = 0.5 !XF 31/12/2024
   !real, parameter :: cloud_water_fractional_std = 0.75
#endif

    ! Cloud decorrelation length formula [ 0 - 2 ]
    ! ============================================
    ! Tune this parameter to control how the cloud decorrelation length is computed (for more 
    ! info, see Shonk et al., 2010). It is advised to stick to the second formula (decorrelation 
    ! length smoothed towards the equator), but the other options may be explored if necessary 
    ! (this will have a slight impact on shortwave fluxes).
    ! 
    ! 0 = Independant of latitude
    ! 1 = Shonk/Hogan formula (eq. 13 from Shonk et al., 2010)
    ! 2 = Improved Shonk/Hogan formula (smoothed at the equator)
    
    integer, parameter :: cloud_decorrelation_length = 2
    
    ! Longwave scattering [ T / F ]
    ! =============================
    ! Keep this parameter true to configure ecRad to simulate the longwave scattering effect of 
    ! both clouds and aerosols. It is possible to only enable the former in ecRad rather than 
    ! both, but for physical realism, both should always be enabled together.
    
    logical, parameter :: do_longwave_scattering = .true.
    
    ! Scaling parameters for water species
    ! ====================================
    ! Tune these parameters to increase or decrease the water species mixing ratios of MAR prior 
    ! to sending them to ecRad. These parameters are enforced to be in the [0.8, 1.2] range, 
    ! meaning you can remove up to -20% of a water species and conversely increase up to 20%. 
    ! Doing so has various impacts on the radiative transfer, in addition to increasing the input 
    ! cloud fraction. In a normal use of MAR, these scaling parameters should always be neutral 
    ! (i.e., set to 1), but you can change them to test unusual/exotic scenarios.
    
    real, parameter :: water_species_scaling_vapor = 1.
    real, parameter :: water_species_scaling_ice = 1.
    real, parameter :: water_species_scaling_liquid = 1.
    real, parameter :: water_species_scaling_rain = 1.
    real, parameter :: water_species_scaling_snow = 1.
    
    ! Extra output files [ T / F ]
    ! ============================
    ! Set the next parameters to true if you want to output the associated additional output files 
    ! (produced with ecRad or the code interfacing it with MAR).
    ! 
    ! These additional output files are:
    ! 1) extra_fluxes.[...].nc: additional surface fluxes computed by ecRad and not used by MAR, 
    ! 2) clim_interpolated.[...].nc: the climatologies used by each MAR run after scaling them to 
    !    a CMIP scenario and interpolating them to the MAR grid; in case of doubt, it is advised 
    !    to output these files (which are lightweight) to control whether or not the greenhouse 
    !    gases and aerosols climatologies correspond to what you expect,
    ! 3) clim_scaled.[...].nc: same as above but prior to interpolation, the idea being that you 
    !    should be able to see whether a climatology issue comes from the interpolation or the 
    !    scaling to a CMIP scenario,
    ! 4) SW_spectral_raw.[...].nc: surface shortwave spectral fluxes as they are represented 
    !    internally by ecRad during a MAR run. Note that these fluxes are VERY heavy files if you 
    !    use the ecCKD high resolution gas models. Outputting such files should be considered if 
    !    and only if you need to thoroughly check the spectral data as used by ecRad (so, mostly 
    !    for debugging issues if you know what you're doing).
    ! 
    ! All output files are prefixed as announced (the [...] corresponds to the run name and period 
    ! details as traditionally appended by MAR).
    
    logical, parameter :: output_extra_fluxes = .false.
    logical, parameter :: output_clim_interpolated = .false.
    logical, parameter :: output_clim_scaled = .false.
    logical, parameter :: output_raw_spectral_fluxes = .false.
    
    ! Spectral decomposition of surface shortwave fluxes
    ! ==================================================
    ! Tune the next parameter to be an array with ascending real values corresponding to the 
    ! boundaries of custom-defined spectral bands to output a spectral decomposition of the 
    ! surface shortwave downward fluxes. For N spectral bands, there should be N+1 values strictly 
    ! provided in ascending order.
    ! 
    ! If you don't need such spectral data, simply declare a dummy array filled with null values. 
    ! Otherwise, follow the example of the commented lines. The spectral data will be written in a 
    ! side NetCDF file "SW_spectral.[...].nc" where "[...]" provides the simulation name and 
    ! period covered by the file.
    
    real, parameter, dimension(10) :: sw_spectral_def = 0
    !real, parameter, dimension(30) :: sw_spectral_def = (/ 280., 285., 290., 295., 300., 305., &
    !  & 310., 315., 320., 325., 330., 335., 340., 345., 350., 360., 370., 380., 390., 400., &
    !  & 410., 420., 430., 440., 450., 460., 470., 480., 490., 500. /) ! Uccle spectral validation
    !real, parameter, dimension(34) :: sw_spectral_def = (/ 187.5, 312.5, 337.5, 362.5, & 
    ! & 387.5, 412.5, 437.5, 462.5, 487.5, 512.5, 537.5, 562.5, 587.5, 612.5, 637.5, 662.5, & 
    ! & 687.5, 712.5, 750., 800., 900., 1000., 1100., 1200., 1300., 1400., 1500., 1600., 1700., & 
    ! & 1800., 2000., 2400., 3400., 4000. /) ! Black Sea

end module mar_ecrad
