!--VERSION:2005.04.08

!  -----------------------------------------------------------------------
!             libUN : User level NetCDF READ / WRITE routines
!
!                     by Philippe Marbaix and Xavier Fettweis
!
!              Compatible with NetCDF version 3.x (or above).
!  -----------------------------------------------------------------------

!   User-frendly interface :
!   ------------------------

!   CF_INI_FILE   : Initialization of the netcf file
!   CF_CREATE_DIM : Create axis/dimensions
!   CF_CREATE_VAR : Create variables
!   CF_CREATE_FILE: Write the netcdf file
!   CF_WRITE      : Write variables
!   CF_READ3D/2D  : Read variables
!   CF_OPEN       : Open  netcdf file
!   CF_CLOSE      : Close netcdf file

!   Main routines :
!   ---------------

!     UNscreate   : General file creation routine,
!                    defining multiple dimensions + attributes

!     UNwrite     : General variables writting routine
!                    (also updates 'range' attribute and variable if present)
!                   Note: Use UNlwrite to write 2D planes in 3D variables

!     UN(s)read   : Reading routine (grid coordinates + variable)

!   Complementary routines :
!   ------------------------

!     UNparam     : set optional parameters of libUN functions
!     UNwopen     : re-open file for writting
!     UNropen     : open file for reading
!     UNgtime     : Find time index for a given time value
!     UNgindx     : Generalization of UNgtime: find value in any 1D data.
!     UNfindx     : modified version of UNgindx safe for non-monotonic data
!     UNclose     : close the NetCDF file
!     UNwratt     : Real attributes writting
!     UNwcatt     : Characters attributes creation & writing

!   Double Precision :
!   ------------------

!     To be in double precision, type this
!     > sed "s/REAL\*4/REAL\*8/g"      libUN.f  > libUN1.f
!     > sed "s/\_REAL/\_DOUBLE/g"      libUN1.f > libUN2.f
!     > sed "s/NF\_FLOAT/NF\_DOUBLE/g" libUN2.f > libUNd.f
!     > rm -f libUN1.f libUN2.f

!  -----------------------------------------------------------------------

!    +---------------------------+---------------------------------------+
!    +  subroutine CD_INI_FILE : + Initialize the netcdf file            +
!    +---------------------------+---------------------------------------+

subroutine CF_INI_FILE(filename, filetitle)

    use libUN_mod
    !     Input :
    !     =======

    !     filename  = name  of the netcdf file
    !     filetitle = title in the netcdf file

    implicit none

    character * (*) filename, filetitle

    CF_attnam(1) = 'actual_range'
    CF_attnum(1) = 2

    CF_varnbrtot = 0 ! Initialization
    CF_dimnbrtot = -1 ! Initialization

    CF_filenam = filename
    CF_filetit = filetitle

ENDsubroutine CF_INI_FILE

!    +-----------------------------+-------------------------------------+
!    +  subroutine CF_CREATE_DIM : + Create dimensions/axis              +
!    +-----------------------------+-------------------------------------+

subroutine CF_CREATE_DIM(dimname, dimunits, dimdim, vallues)
    use libUN_mod
    !     Input :
    !     =======

    !     dimname  = name of the axis/dimension
    !     dimunits = units of the axis/dimension
    !     dimdim   = dimensions of the axis/dimension
    !     vallues  = vallues of the axis/dimension

    implicit none

    character * (*) dimname, dimunits

    integer dimdim, i
    REAL(kind=4) vallues(dimdim)

    CF_dimnbrtot = CF_dimnbrtot + 1

    CF_dimnbrtot = max(0, CF_dimnbrtot)

    CF_dimnam(CF_dimnbrtot) = dimname
    CF_dimnamuni(CF_dimnbrtot) = dimunits
    CF_dim(CF_dimnbrtot) = dimdim

    do i = 1, dimdim
        CF_dimval(i, CF_dimnbrtot) = vallues(i)
    enddo

ENDsubroutine CF_CREATE_DIM

!    +-----------------------------+-------------------------------------+
!    +  subroutine CF_CREATE_VAR : + Create variables                         +
!    +-----------------------------+-------------------------------------+

subroutine CF_CREATE_VAR(varname, vartitle, varunits, varaxe4, &
                         varaxe1, varaxe2, varaxe3)

    !     Input :
    !     =======

    !     varname  = name of the variable
    !     vartitle = title of the variable
    !     varunits = units of the variable
    !     varaxeX  = axes used by the variable (T,X,Y,Z)
    use libUN_mod
    implicit none

    character * (*) varname, vartitle, varunits
    character * (*) varaxe1, varaxe2, varaxe3, varaxe4

    CF_varnbrtot = max(0, CF_varnbrtot + 1)

    CF_varnam(CF_varnbrtot) = varname
    CF_varnamdim(1, CF_varnbrtot) = varaxe1
    CF_varnamdim(2, CF_varnbrtot) = varaxe2
    CF_varnamdim(3, CF_varnbrtot) = varaxe3
    CF_varnamdim(4, CF_varnbrtot) = varaxe4
    CF_varnamuni(CF_varnbrtot) = varunits
    CF_vardes(CF_varnbrtot) = vartitle

ENDsubroutine CF_CREATE_VAR

!    +--------------------------------------+----------------------------+
!    +  subroutine CF_CREATE_VAR_VIA_FILE : + Create variables           +
!    +--------------------------------------+----------------------------+

subroutine CF_CREATE_VAR_VIA_FILE(filename)

    !     Input :
    !     =======

    !     filename  = name of the file containing informations
    !                 about the variables
    use libUN_mod
    implicit none

    character * 200 filename

    character * 120 tmpvar

    OPEN(unit=999, status='old', file=filename)

980 continue
    READ(999, '(A120)', end=990) tmpvar

    if(tmpvar(1:4) == '    ') then
        CF_varnbrtot = max(0, CF_varnbrtot + 1)
        READ(tmpvar, '(4x,5A9,A12,A50)') &
            CF_varnam(CF_varnbrtot), &
            CF_varnamdim(1, CF_varnbrtot), &
            CF_varnamdim(2, CF_varnbrtot), &
            CF_varnamdim(3, CF_varnbrtot), &
            CF_varnamdim(4, CF_varnbrtot), &
            CF_varnamuni(CF_varnbrtot), &
            CF_vardes(CF_varnbrtot)
    endif

    GOTO 980
990 continue

ENDsubroutine CF_CREATE_VAR_VIA_FILE

!    +------------------------------+------------------------------------+
!    +  subroutine CF_CREATE_FILE : + Create the netcdf file             +
!    +------------------------------+------------------------------------+

subroutine CF_CREATE_FILE(filename)

    !     Input :
    !     =======

    !     filename  = name  of the netcdf file
    use libUN_mod
    implicit none

    character * (*) filename

    integer i, j, id

    integer UN1_dim(0:CF_dimnbrtot)

    real UN1_dimval(CF_dimmaxlen, 0:CF_dimnbrtot)

    character * 31 UN1_dimnam(0:CF_dimnbrtot), &
        UN1_dimnamuni(0:CF_dimnbrtot)

    if(filename /= CF_filenam) then
        write(6, *) "ERROR: not "//CF_filenam
        stop
    endif

    do i = 0, CF_dimnbrtot
        UN1_dim(i) = CF_dim(i)
        UN1_dimnam(i) = CF_dimnam(i)
        UN1_dimnamuni(i) = CF_dimnamuni(i)
        do j = 1, CF_dim(i)
            UN1_dimval(j, i) = CF_dimval(j, i)
        enddo
    enddo

    call UNscreate(CF_filenam, CF_filetit, CF_dimnbrtot, UN1_dim, &
                   CF_dimmaxlen, UN1_dimnam, UN1_dimnamuni, &
                   UN1_dimval, &
                   CF_varmaxnbr, CF_varnbrtot, CF_varnam, &
                   CF_varnamdim, CF_varnamuni, CF_vardes, &
                   CF_attnbr, CF_attnam, CF_attnum, id)

    call UNclose(id)

ENDsubroutine CF_CREATE_FILE

!    +------------------------+------------------------------------------+
!    +  subroutine CF_WRITE : + Writes variables                         +
!    +------------------------+------------------------------------------+

subroutine CF_WRITE(FILEname, VARname, itime, &
                    Ni, Nj, Nlev, var)

    !     Input :
    !     =======

    !     FILEname    = name of the netcdf file
    !     VARname     = name of variables
    !     itime       = index on time axis
    !     Ni,Nj,Nlev  = X,Y,Z dimension
    !     var         = array of vallues of the variable
    use libUN_mod
    implicit none

    character * (*) FILEname, VARname
    integer itime
    integer Ni, Nj, Nlev, fileid
    REAL(kind=4) var(Ni, Nj, Nlev)

    if(CF_filenamopened /= FILEname) then
        call UNwopen(FILEname, fileid)
    else
        fileid = CF_fileidopened
    endif

    call UNwrite(fileid, VARname, itime, Ni, Nj, Nlev, var)

    if(CF_filenamopened /= FILEname) then
        call UNclose(fileid)
    endif

ENDsubroutine CF_WRITE

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine CF_READ2D : + Read variables                          +
!**  +-------------------------+-----------------------------------------+

subroutine CF_READ2D(FILEname, VARname, itime, &
                     Ni, Nj, Nlev, var)

    !     Input :
    !     =======

    !     FILEname    = name of the netcdf file
    !     VARname     = name of variables
    !     itime       = index on time axis
    !     Ni,Nj,Nlev  = X,Y,Z dimension

    !     Output :
    !     ========

    !     var         = array of vallues of the variable
    use libUN_mod
    implicit none

    character * (*) FILEname, VARname
    character * 31 var_units, filetitle
    integer Ni, Nj, Nlev, itime, level
    REAL(kind=4) var(Ni, Nj)

    integer i, j, fileid

    if(CF_filenamopened /= FILEname) then
        call UNropen(FILEname, fileid, filetitle)
    else
        fileid = CF_fileidopened
    endif

    call UNsread(fileid, VARname, itime, Nlev, 1, 1, &
                 Ni, Nj, 1, var_units, var)

    if(CF_filenamopened /= FILEname) then
        call UNclose(fileid)
    endif

ENDsubroutine CF_READ2D

!    +-------------------------+-----------------------------------------+
!    +  subroutine CF_READ3D : + Read variables                          +
!    +-------------------------+-----------------------------------------+

subroutine CF_READ3D(FILEname, VARname, itime, &
                     Ni, Nj, Nlev, var)

    !     Input :
    !     =======

    !     FILEname    = name of the netcdf file
    !     VARname     = name of variables
    !     itime       = index on time axis
    !     Ni,Nj,Nlev  = X,Y,Z dimension

    !     Output :
    !     ========

    !     var         = array of vallues of the variable
    use libUN_mod
    implicit none

    character * (*) FILEname, VARname
    character * 32 var_units, filetitle
    integer Ni, Nj, Nlev, itime, level
    REAL(kind=4) var(Ni, Nj, Nlev)

    integer i, j, fileid

    if(CF_filenamopened /= FILEname) then
        call UNropen(FILEname, fileid, filetitle)
    else
        fileid = CF_fileidopened
    endif

    call UNsread(fileid, VARname, itime, 0, 1, 1, &
                 Ni, Nj, Nlev, var_units, var)

    if(CF_filenamopened /= FILEname) then
        call UNclose(fileid)
    endif

ENDsubroutine CF_READ3D

!**  +------------------------+------------------------------------------+
!**  +  subroutine CF_CLOSE : + Close the file                           +
!**  +------------------------+------------------------------------------+

subroutine CF_CLOSE(FILEname)
    use libUN_mod
    implicit none

    character * (*) FILEname

    if(FILEname == CF_filenamopened) then
        call UNclose(CF_fileidopened)
    else
        print *, FILEname//" not opened"
    endif

    CF_filenamopened = ""
    CF_fileidopened = 0

ENDsubroutine CF_CLOSE

!**  +-----------------------+-------------------------------------------+
!**  +  subroutine CF_OPEN : + open the file                             +
!**  +-----------------------+-------------------------------------------+

subroutine CF_OPEN(FILEname, FILEid)
    use libUN_mod
    implicit none

    integer FILEid

    character * (*) FILEname

    call UNwopen(FILEname, FILEid)

    CF_filenamopened = FILEname

    CF_fileidopened = FILEid

ENDsubroutine CF_OPEN

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNscreate : +                                         +
!**  +-------------------------+                                         +
!**  +  * Purpose :                                                      +
!**  +     Create a NetCDF file, general version.                        +
!**  +     (Staggered grids + other extensions to UNcreate)              +
!**  +                                                                   +
!**  +  * How it works : calling routine must provide                    +
!**  +    -a list of dimensions                                          +
!**  +     (size of each dimens., names, units and values of coordinates)+
!**  +    -a list of variables                                           +
!**  +     (units, number of dimensions, names of selected dimensions)   +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +  -------                                                          +
!**  +                                                                   +
!**  +  General :                                                        +
!**  +   FILEnam          [char]: Name of the file to be created.        +
!**  +   title            [char]: Title attribute                        +
!**  +                                                                   +
!**  +  Dimensions:                                                      +
!**  +   TND                    : Total Number of SPATIAL dimensions     +
!**  +                            Notice : Set "time" to dimension No 0  +
!**  +   DFdim(0:TND)           : # discrete values for each dimension   +
!**  +                            Notice : DFdim(0).eq.0                 +
!**  +                            -> UNLIMITED TIME (coord. not defined) +
!**  +                               WARNING: In this case, the NetCDF   +
!**  +                               use a temporary space to duplicate  +
!**  +                               the file -> NOT RECOMMENDED         +
!**  +   MXdim                  : Maximum value of DFdim, = arrays size  +
!**  +   NAMdim(0:TND)    [char]: Name of dimensions, except time        +
!**  +   UNIdim(0:TND)    [char]: Units of dimensions (attribute)        +
!**  +   VALdim(MXdim,0:TND)[R4]: Values of coordinate for each dimension+
!**  +                                                                   +
!**  +  Variables:                                                       +
!**  +   Dvs                    : Variable's definitions array sizes,    +
!**  +   Nvs                    : Number of defined variables(Nvs.le.Dvs)+
!**  +   name_vs (Dvs)    [char]: name of variable.                      +
!**  +   unit_vs (Dvs)    [char]: physical units of variable (attribute) +
!**  +   Sdim_vs (4,Dvs)  [char]: name of Selected dims (in above list)  +
!**  +                            Blanked or '-' elements = not used     +
!**  +   lnam_vs (Dvs)    [char]: Long_name attribute (descript. of var.)+
!**  +                                                                   +
!**  +  List of real attributes to all variables:                        +
!**  +   Nra                    : Number of Real Attributes (.ge.1 !)    +
!**  +   NAMrat(Nra)      [char]: NAMes of Real ATtributes  (''=none)    +
!**  +                            (initial value= 0; set it with UNwratt)+
!**  +   Nvals(Nra)             : Number of values of these attributes.  +
!**  +   ! Currently limited to 1 value (scalar) or 2 (2 elements vector)+
!**  +   ! EXCEPTION: Setting the last attribute name to '[var]_range'   +
!**  +                does create a variable (!) for level-by-level range+
!**  +                (very usefull for 3D + time fields)                +
!**  +                                                                   +
!**  +  NB : [char] variables may have any length.                       +
!**  +       blanks characters are NOT ALLOWED in any variable,          +
!**  +          except the "title".                                      +
!**  +          and the NetCDF variables defined here are always  REAL(kind=4)  +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +  --------                                                         +
!**  +   FILEid                 : Index of the NetCDF file (remains open)+
!**  +-------------------------------------------------------------------+

subroutine UNscreate(FILEnam, title, &
                     TND, DFdim, MXdim, NAMdim, UNIdim, VALdim, &
                     Dvs, Nvs, name_vs, Sdim_vs, unit_vs, lnam_vs, &
                     Nra, NAMrat, Nvals, &
                     FILEid)
    use libUN_mod
    use mar_ge
    ! +
    implicit none

    ! +
    INTEGER icheck, MXND
    !     ** Maximum number of dimensions
    parameter(MXND=100)

    ! +   INPUT:
    ! +   - - -
    character * (*) FILEnam
    character * (*) title

    integer TND, DFdim(0:TND), MXdim
    character * (*) NAMdim(0:TND)
    character * (*) UNIdim(0:TND)
    REAL(kind=4) VALdim(MXdim, 0:TND)

    integer Nvs, Dvs
    character * (*) name_vs(Dvs)
    character * (*) Sdim_vs(4, Dvs)
    character * (*) unit_vs(Dvs)
    character * (*) lnam_vs(Dvs)

    integer Nra
    character * (*) NAMrat(Nra)
    character * 24 Host, Fdate
    character * 200 tmpchar
    integer Nvals(Nra)

    ! +   OUTPUT:
    ! +   - - - -
    INTEGER FILEid

    ! +   LOCAL:
    ! +   - - -
    integer VARSIZE
    EXTERNAL VARSIZE
    character * (30) tmpchr
    integer dimDID(0:MXND)
    integer dimVID(0:MXND), vsVID, vrVID
    integer dID(4), start(4), count(4), rdID(2)
    integer mimaID
    integer stride(4), imap(4)
    integer Ndim_vs
    integer ivs, igd, idi, ira, itmp
    integer Nlen
    integer dNlen(0:MXND)
    integer Ierro, TTerr, ii, jj
    REAL(kind=4) zero1(1), zero2(2)

    icheck = 0 !Debugging level

    !*    0. Initialisations
    !     ------------------
    if(icheck >= 1) write(*, *) 'UNscreate : Begin'

    ! +   Routines which opens a file must reset libUN internals:
    call UNparam('RESET_PARAMS_', 0.0)

    do ii = 1, 4
        stride(ii) = 1
    enddo
    zero1(1) = 0.
    zero2(1) = 0.
    zero2(2) = 0.
    TTerr = 0 !Total of error flags

    if(TND > MXND) then
        write(*, *) 'UNscreate - Error: so much dimensions ?', TND
    endif

    !     Create a NetCDF file and enter define mode :
    !     --------------------------------------------
    if(icheck >= 2) write(*, *) 'FILEnam :', FILEnam

    !     ** getting FILEnam [char] size :
    Nlen = VARSIZE(FILEnam)

    !Ierro = NF_CREATE(FILEnam(1:Nlen), NF_CLOBBER, FILEid)
    Ierro = NF_CREATE(FILEnam(1:Nlen), NF_NETCDF4, FILEid) ! NETCDF4
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
    !     ** identif.                       =>overwrite =error

    !*    Time coordinate definition.
    !     ---------------------------

    !     ** Define dimension :
    if(icheck >= 3) write(*, *) '# time iters.:', DFdim(0)
    if(DFdim(0) == 0.) then
        Ierro = NF_DEF_DIM(FILEid, 'time', NF_UNLIMITED, dimDID(0))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)
    else
        Ierro = NF_DEF_DIM(FILEid, 'time', DFdim(0), dimDID(0))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)
    endif
    dNlen(0) = 4  ! 4 characters in the name 'time'...
    if(NAMdim(0)(1:4) /= 'time') then
        write(*, *) 'Sorry, NAMdim(0) must be ''time'' .'
        STOP
    endif

    !     ** Define variable for the time coordinate values :
    dID(1) = dimDID(0)
    Ierro = NF_DEF_VAR(FILEid, 'time', NF_DOUBLE, 1, dID, dimVID(0))
    !     **      ^^^^^^^^^^ FILEid  var name  type  dims  DIMid VARid
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
    TTerr = TTerr + ABS(Ierro)


    !     Spatial coordinates definitions : DIMS and VARs (locations).
    !     ------------------------------------------------------------
    !
    do igd = 1, TND            !** BEGIN LOOP over all spatial dims
        if(icheck >= 3) write(*, *) '  spatial dim:', NAMdim(igd)

        !       ** getting NAMdim [char] size :
        Nlen = VARSIZE(NAMdim(igd))
        dNlen(igd) = Nlen  !For further use of NAMdim

        Ierro = NF_DEF_DIM(FILEid, NAMdim(igd)(1:Nlen), &
                           DFdim(igd), dimDID(igd))
        !       **line1 ^^^^^^^^^^ FILEid    | dim name
        !       **line2            # values  | VARid
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)

        dID(1) = dimDID(igd)
        Ierro = NF_DEF_VAR(FILEid, NAMdim(igd)(1:Nlen), &
                           NF_FLOAT, 1, dID, dimVID(igd))
        !       **line1 ^^^^^^^^^^ FILEid    | dim name
        !       **line2            type      | #dims | dimsIDs | VARid
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)

    enddo       !** END   LOOP over all spatial dims

    !     Special coordinate definition: MinMax (for [var]_range)
    !     -------------------------------------------------------
    if(NAMrat(Nra)(1:11) == '[var]_range') then

        Ierro = NF_DEF_DIM(FILEid, 'MinMax', 2, mimaID)
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
    endif

    !     Define the fields.
    !     ------------------

    do ivs = 1, Nvs             !**BEGIN LOOP on var. num.
        if(icheck >= 3) &
            WRITE(*, *) 'Defining variable ', name_vs(ivs)

        !       Set space and time dimensions
        !       - - - - - - - - - - - - - - -
        !        ** Initialise number of dimensions :
        Ndim_vs = 0

        do idi = 1, 4           !** BEGIN LOOP on var dims.
            if(Sdim_vs(idi, ivs)(1:1) /= ' ' &
               .and. Sdim_vs(idi, ivs)(1:1) /= '-') then !**skip undefined.

                !         ** getting Sdim_vs [char] size :
                Nlen = VARSIZE(Sdim_vs(idi, ivs))

                !         ** Searching for the dimension index from its name (Sdim_vs)
                igd = 0
                do WHILE(Sdim_vs(idi, ivs)(1:Nlen) &
                         /= NAMdim(igd)(1:dNlen(igd)))
                    if(igd == TND) then
                        write(*, *) 'UNscreate-ERROR: Dimension not found:', &
                            Sdim_vs(idi, ivs)(1:Nlen)
                        STOP
                    endif
                    igd = igd + 1
                enddo
                !         ** Construct the dimensions id's for that variable (ivs):
                if(icheck >= 3) &
                    WRITE(*, *) 'using dimension ', NAMdim(igd), dimDID(igd)
                Ndim_vs = Ndim_vs + 1
                dID(Ndim_vs) = dimDID(igd)

            endif
        enddo      !** END   LOOP on var dims.

        !       Define our special [var]_range field for 4D variables
        !       - - - - - - - - - - - - - - - - - - - - - - - - - - -
        if(Ndim_vs == 4 &
           .and. NAMrat(Nra)(1:11) == '[var]_range') then

            Nlen = VARSIZE(name_vs(ivs))
            rdID(1) = dID(3)  !(4D variable, 3th dim = level)
            rdID(2) = mimaID   !(for min, max)
            tmpchr = name_vs(ivs)(1:Nlen)//'_range'
            itmp = Nlen + 6
            Ierro = NF_DEF_VAR(FILEid, tmpchr(1:itmp), &
                               NF_FLOAT, 2, rdID, vrVID)
            if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
            TTerr = TTerr + ABS(Ierro)

        endif

        !       Define fields :
        !       - - - - - - - -
        Nlen = VARSIZE(name_vs(ivs))
        Ierro = NF_DEF_VAR(FILEid, name_vs(ivs)(1:Nlen), &
                           NF_FLOAT, Ndim_vs, dID, vsVID)
        !       **line1 ^^^^^^^^^^ FILEid | variable name
        !       **line2            type   | #dims   | dimsIDs | VARid
        if(Ierro /= NF_NOERR) &
            call HANDLE_ERR('UNscreate (field)', Ierro)
        TTerr = TTerr + ABS(Ierro)

        !Ierro=NF_DEF_VAR_DEFLATE(FILEid, vsVID, 1, 1, 1) !NETCDF4, compression

        !     Set the variable's attributes :
        !     -------------------------------

        !       ** Units:
        !       - - - - -
        !       ** getting unit_vs [char] size :
        Nlen = VARSIZE(unit_vs(ivs))

        Ierro = NF_PUT_ATT_TEXT(FILEid, vsVID, 'units', &
                                Nlen, unit_vs(ivs)(1:Nlen))
        !       **line1 ^^^^^^^^^^^^^^^ FILEid |var.id | attr.name
        !       **line2                 length | attr.value
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)

        !       ** Special case : units = sigma
        !       - - - - - - - - - - - - - - - -
        !       In this case, CV convention advises to write the following
        !        attribute :  positive = down
        !
        !BUG    Nlen = VARSIZE(lnam_vs(ivs))

        if(unit_vs(ivs)(1:Nlen) == '[sigma]' &
           .OR. unit_vs(ivs)(1:Nlen) == 'sigma_level') then
            if(icheck >= 3) then
                write(*, *) 'Unit = sigma -> setting positive attr'
            endif

            Ierro = NF_PUT_ATT_TEXT(FILEid, vsVID, 'positive', &
                                    4, 'down')
            !         **line1 ^^^^^^^^^^^^^^^ FILEid |var.id | attr.name
            !         **line2                 length | attr.value
            if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        endif

        !       ** "long_name":
        !       - - - - - - - -
        Nlen = VARSIZE(lnam_vs(ivs))

        if(icheck >= 3) &
            WRITE(*, *) 'Write long_name ', lnam_vs(ivs)(1:Nlen)

        Ierro = NF_PUT_ATT_TEXT(FILEid, vsVID, 'long_name', &
                                Nlen, lnam_vs(ivs)(1:Nlen))

        do jj = 1, Nlen
            if(lnam_vs(ivs)(jj:jj) == " ") lnam_vs(ivs)(jj:jj) = "_"
            if(lnam_vs(ivs)(jj:jj) == ".") lnam_vs(ivs)(jj:jj) = "_"
            if(lnam_vs(ivs)(jj:jj) == "(") lnam_vs(ivs)(jj:jj) = "_"
            if(lnam_vs(ivs)(jj:jj) == ")") lnam_vs(ivs)(jj:jj) = "_"
            if(lnam_vs(ivs)(jj:jj) == "/") lnam_vs(ivs)(jj:jj) = "_"
        enddo

        Ierro = NF_PUT_ATT_TEXT(FILEid, vsVID, 'standard_name', &
                                Nlen, lnam_vs(ivs)(1:Nlen))

        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)

        !       ** From the list of real attributes (input argument) :
        !       - - - - - - - - - - - - - - - - - - - - - - - - - - -
        !
        do ira = 1, Nra
            if(NAMrat(ira)(1:1) /= ' ') then
                if(NAMrat(ira)(1:11) == 'valid_range') then

                    !         ** The "valid_range" special attribute  :
                    Ierro = NF_PUT_ATT_REAL(FILEid, vsVID, 'valid_range', &
                                            NF_FLOAT, 2, ValRange)
                    TTerr = TTerr + ABS(Ierro)

                else if(NAMrat(ira)(1:11) /= '[var]_range') then

                    !         ** All "regular" attributes :
                    Nlen = VARSIZE(NAMrat(ira))
                    if(Nvals(ira) == 1) then
                        Ierro = NF_PUT_ATT_REAL(FILEid, vsVID, NAMrat(ira)(1:Nlen), &
                                                NF_FLOAT, Nvals, zero1)
                        TTerr = TTerr + ABS(Ierro)
                    else if(Nvals(ira) == 2) then
                        Ierro = NF_PUT_ATT_REAL(FILEid, vsVID, NAMrat(ira)(1:Nlen), &
                                                NF_FLOAT, Nvals, zero2)
                        TTerr = TTerr + ABS(Ierro)
                        !
                    endif
                endif
            endif
        enddo

    enddo        ! **END   LOOP on var. num.

    !     Set 'unit' attribute for the dimensions:
    !     ----------------------------------------

    do igd = 0, TND         !** BEGIN LOOP over all spatial dims

        !       ** getting NAMdim [char] size :
        Nlen = VARSIZE(UNIdim(igd))

        Ierro = NF_PUT_ATT_TEXT(FILEid, dimVID(igd), 'units', &
                                Nlen, UNIdim(igd))

        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

        if(igd==0) then !time
        Ierro = NF_PUT_ATT_TEXT(FILEid, dimVID(igd), 'calendar', &
                                VARSIZE(calGE), trim(calGE))

        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        endif 

        Nlen = VARSIZE(NAMdim(igd))

        Ierro = NF_PUT_ATT_TEXT(FILEid, dimVID(igd), 'long_name', &
                                Nlen, NAMdim(igd))

        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

        Ierro = NF_PUT_ATT_TEXT(FILEid, dimVID(igd), 'standard_name', &
                                Nlen, NAMdim(igd))

        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
        TTerr = TTerr + ABS(Ierro)

    enddo

    !     Global attribute(s).
    !     --------------------

    !     ** Title (some general file descriptor) :
    !     ** getting unit_vs [char] size :

    Nlen = VARSIZE(title)

    Ierro = NF_PUT_ATT_TEXT(FILEid, NF_GLOBAL, 'title', &
                            Nlen, title(1:Nlen))

    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

    Nlen = VARSIZE(CF_institution)

    Ierro = NF_PUT_ATT_TEXT(FILEid, NF_GLOBAL, 'institution', &
                            Nlen, CF_institution)

    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

    Nlen = VARSIZE("MARv"//trim(verGE))

    Ierro = NF_PUT_ATT_TEXT(FILEid, NF_GLOBAL, 'model', &
                            Nlen, "MARv"//trim(verGE))

    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

    !     call HostNm(Host, Ierro)

    tmpchar = "libUN ("//CF_libUN_version//") - "//FDate()
    !    &        " - "//Host

    Nlen = VARSIZE(tmpchar)

    Ierro = NF_PUT_ATT_TEXT(FILEid, NF_GLOBAL, 'history', &
                            Nlen, tmpchar)

    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

    Nlen = VARSIZE(NF_INQ_LIBVERS())

    Ierro = NF_PUT_ATT_TEXT(FILEid, NF_GLOBAL, 'netcdf', &
                            Nlen, NF_INQ_LIBVERS())

    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
    TTerr = TTerr + ABS(Ierro)

    !     Leave define mode (!file remains open )
    !     ---------------------------------------
    Ierro = NF_ENDDEF(FILEid)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)
    TTerr = TTerr + ABS(Ierro)

    !     Writing of dimensions coordinates.
    !     ----------------------------------

    !     ** Time :
    !     - - - - -

    start(1) = 1          !Vector of starting indexes values
    count(1) = DFdim(0)   !Vector of total # indexes values
    if(icheck >= 3) &
        WRITE(*, *) 'Write coords for ', NAMdim(0), count(1)

    !     ** Set 'imap' to write with NCVPTG; NCVPT could be enough ?
    !     ** (imap tells NetCDF about the memory locations of var,
    !     **  we choose NCVPTG because
    !     **  only a portion of VALdim is written.)
    imap(1) = 1
    imap(2) = 0                 ! Not used : write only 1 coord.

    Ierro = NF_PUT_VARM_REAL(FILEid, dimVID(0), start, count, &
                             stride, imap, VALdim(1, 0))
    !     **line 1 ^^^^^^^^^^^^^^^ ID file| id var. |read from...  |#data
    !     **line 2                 step   |re-arrang|variable(beg.)
    !     **                      (^^^^stride is not used)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

    !     ** Space coordinates :
    !     - - - - - - - - - - - -

    do igd = 1, TND          !** BEGIN LOOP over all spatial dims

        start(1) = 1
        count(1) = DFdim(igd)
        if(icheck >= 3) &
            WRITE(*, *) 'Write coords for ', NAMdim(igd), count(1)

        Ierro = NF_PUT_VARM_REAL(FILEid, dimVID(igd), start, count, &
                                 stride, imap, VALdim(1, igd))
        !       **      ^^^^^^^^^^^^^^^^ see above
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNscreate', Ierro)

        TTerr = TTerr + ABS(Ierro)

    enddo     !** END   LOOP over all spatial dims

    !     Stop if an error occured.
    !     -------------------------

    if(TTerr /= 0) then
        STOP 'UNscreate : Sorry, an error occured.'
    endif

    ! +
    RETURN
ENDsubroutine UNscreate

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNwrite :   +                                         +
!**  +-------------------------+                                         +
!**  +  * Writes a variable into a NetCDF file,                          +
!**  +    (the NetCDF file must have been created (or re-opened) and     +
!**  +     closed after all writing operations).                         +
!**  +  * Automatically updates attribute 'actual_range' if available    +
!**  +          "          "    special var. '[var]_range'    "          +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from UNcreate OR NetCDF open) +
!**  +    VARname : name given to the variable to write (must be in file)+
!**  +    itime   : No of time step to write to                          +
!**  +    Ni,Nj,Nlev: dimensions of 'var'                                +
!**  +              ! Nlev= 1 for 2D and 1D input variables.             +
!**  +                Nj  = 1 for 1D input variables.                    +
!**  +              NB: can not write 1 level of 3D var only (->UNlwrite)+
!**  +                                                                   +
!**  +    var     : The variable to be writen                            +
!**  +                                                                   +
!**  +  REMARK :                                                         +
!**  +    Truncation of input data is permited:                          +
!**  +    If the dim of "var" > dim in the NetCDF file,                  +
!**  +    "var" is automatically truncted. However, this => WARNING      +
!**  +    message, UNLESS a specific truncation was "announced"          +
!**  +    in var:                                                        +
!**  +       To truncate the first dim to Li, let var(Ni,1,1) = Li       +
!**  +       To truncate the 2nd   dim to Lj, let var(1,Nj,1) = Lj       +
!**  +       ... (this has no effect exept cancel the "WARNING" message) +
!**  +-------------------------------------------------------------------+

subroutine UNwrite(FILEid, VARname, itime, &
                   Ni, Nj, Nlev, var)
    use libUN_mod
    implicit none

    INTEGER icheck

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    integer FILEid
    integer itime
    integer Ni, Nj, Nlev
    character * (*) VARname
    REAL(kind=4) var(Ni, Nj, Nlev)

    !     ** local :
    integer MXlv
    PARAMETER(MXlv=500)
    !                ^^^^Maximal # levels for a special output
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    INTEGER NVRi, NVRj, NVRlev
    INTEGER Ierro, TTerr, Nvatts, vtype
    INTEGER dimID(4), dimSIZ(4), count(4)
    INTEGER start(4), stride(4), imap(4)
    character * (Lvnam) dimNAM(4)
    character * (Lvnam) recname
    character * (30) tmpchr
    INTEGER varVID
    INTEGER VNlen, NDIMvar, NSDIvar, tiDI, itmp
    INTEGER iz, ii, jj, ll
    INTEGER iUNLIMDIM
    REAL(kind=4) chkdim
    REAL(kind=4) Arange(2), sValRange(2)
    REAL(kind=4) Srange(MXlv, 2)
    LOGICAL OkRange

    icheck = 0     !** 'debugging' level
    TTerr = 0     !** 'total number of errors

    if(icheck >= 1) write(*, *) 'UNwrite : Begin'

    !*    1. Get the variable field  and dims IDs
    !     ----------------------------------------

    if(icheck >= 2) write(*, *) 'FILEid  :', FILEid

    !     ** getting VARname  size :
    VNlen = VARSIZE(VARname)
    if(icheck >= 3) write(*, *) 'VNlen  :', VNlen
    if(icheck >= 2) write(*, *) 'VARname   :', VARname(1:VNlen)

    !     ** variable field ID :
    Ierro = NF_INQ_VARID(FILEid, VARname(1:VNlen), varVID)

    !     ** Cancel writing if an error occured : variable undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNwrite  Info  : Variable ', VARname(1:VNlen) &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9999 !** UNwrite_end

    !     ** Inquire about the number of dimensions in var :
    !     **
    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       NDIMvar, dimID, Nvatts)
    !     **  line1          id/file  id/var  var name var type
    !     **  line2          # dims   id/dims #attributes
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro1. ', Ierro

    !*    2. Dimensions : inquire about file + compare with input data.
    !     -------------------------------------------------------------

    !     2.1 Inquire dimensions names and sizes :
    ! +   - - - - - - - - - - - - - - - - - - - - -
    do iz = 1, 4
        dimSIZ(iz) = 0
        dimNAM(iz) = '       '
        !       ** Set any unused dimension to "0" size / no name
    enddo
    do iz = 1, NDIMvar
        Ierro = NF_INQ_DIM(FILEid, dimID(iz), dimNAM(iz), dimSIZ(iz))
        !       **                 id/file  id/dim     dimname      dimsize
        !       **                                     !output      output
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)
    enddo
    if(icheck >= 3) write(*, *) 'NDIMvar  ', NDIMvar
    if(icheck >= 3) write(*, *) 'Ierro 2.0', Ierro

    !     2.2 Set writing region according to field dimension : 2D or 3D
    ! +   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    !     ** Set horizontal dimensions (default, for most data) :
    count(1) = Ni
    count(2) = Nj
    ! +   ** Other default values:
    count(3) = 0
    count(4) = 0
    start(1) = 1
    start(2) = 1
    start(3) = 1
    start(4) = 1

    ! +- ------3D+time variable in file-----------
    if(NDIMvar == 4) then
        !       ** 3D space + time:
        NSDIvar = 3     ! Nb. space dims
        tiDI = 4     ! No. of the time dim
        !       ** write 3D space:
        start(3) = 1    ! Start of index 3 in var (here = vert. levs)
        count(3) = Nlev ! Nb. values of index 3 in var
        !       ** write one time step:
        start(4) = itime
        count(4) = 1
        ! +- ------3D *OR* 2D+time var in file--------
    else if(NDIMvar == 3) then
        if(Nlev == 1) then
            !         ** 2D space + time (standard use of UNlib):
            NSDIvar = 2
            tiDI = 3
            !         ** ...write one time step:
            start(3) = itime
            count(3) = 1
        else
            !         ** 3D (no time slice):
            NSDIvar = 3
            tiDI = 0
            !         ** ...write 3rd dimension:
            start(3) = 1
            count(3) = Nlev
        endif
        ! +- ------2D *OR* 1D+time var in file--------
    else if(NDIMvar == 2) then
        if(Nj == 1 .and. dimNAM(2)(1:4) == 'time') then
            !         ** Write a 1D vector at time= itime:
            NSDIvar = 1
            tiDI = 2
            start(2) = itime
            count(2) = 1
        else
            !         ** Usual MAR 2D space (no time):
            NSDIvar = 2
            tiDI = 0
        endif
        ! +- ------1D *OR* 0D+time var in file--------
    else if(NDIMvar == 1) then
        !       ** 1D space or time
        if(Ni == 1) then
            !         ** Write a single element (at itime)
            start(1) = itime
            count(1) = 1
            count(2) = 0
            NSDIvar = 0
            tiDI = 1
        else
            !         ** Write a vector (use only "space" dim 1)
            NSDIvar = 1
            tiDI = 0
            count(2) = 0
        endif
    else
        write(*, *) 'UNwrite ERROR : data field dimension ?'
        STOP
    endif

    !     2.3 Compare file dimensions to input data.
    ! +   - - - - - - - - - - - - - - - - - - - - - -
    !     ** Save variable size for use as "valid" size (-> range):
    NVRi = Ni
    NVRj = Nj
    NVRlev = Nlev
    !     ** Space dimensions :
    if(NSDIvar > 0) then
        do iz = 1, NSDIvar
            if(dimSIZ(iz) > count(iz)) then
                write(*, *) 'UNwrite - WARNING: '
                write(*, *) ' Your field ', VARname, ' has an empty part.'
                write(*, *) ' (for the dimension:', dimNAM(iz), ')'
            else if(dimSIZ(iz) < count(iz)) then
                !         ** Do display "warning" only if truncation
                !            was not "correctly announced" (see header)
                !            (NVR... => stop here when updating the range attribute)
                if(iz == 1) then
                    chkdim = var(Ni, 1, 1)
                    NVRi = dimSIZ(1)
                else if(iz == 2) then
                    chkdim = var(1, Nj, 1)
                    NVRj = dimSIZ(2)
                else if(iz == 3) then
                    chkdim = var(1, 1, Nlev)
                    NVRlev = dimSIZ(3)
                else
                    chkdim = 0.0
                endif
                Ierro = NF_INQ_UNLIMDIM(FILEid, iUNLIMDIM)
                if(dimID(iz) /= iUNLIMDIM) then
                    if(ABS(chkdim - dimSIZ(iz)) > 0.1) then
                        write(*, *) 'UNwrite - WARNING: '
                        write(*, *) ' Your field ', VARname, ' will be truncated.'
                        write(*, *) ' (for the dimension:', dimNAM(iz), ')'
                    endif
                    count(iz) = dimSIZ(iz)
                endif
            endif
        enddo
    endif

    !     ** Time dimension (when defined):
    if(tiDI /= 0) then
        if(itime > dimSIZ(tiDI)) then
            if(icheck >= 1) write(*, *) 'Time limit, ID', dimID(tiDI)
            Ierro = NF_INQ_UNLIMDIM(FILEid, iUNLIMDIM)
            if(dimID(tiDI) /= iUNLIMDIM) then
                write(*, *) 'UNwrite - ERROR:   '
                write(*, *) ' Time index out of range '
                STOP
            endif
        endif
    endif

    if(icheck >= 2) write(*, *) 'Ierro2. ', Ierro
    if(icheck >= 2) write(*, *) 'Dimension names :', dimNAM
    if(icheck >= 2) write(*, *) 'dimSIZ :', dimSIZ
    if(icheck >= 2) write(*, *) 'count  :', count
    if(icheck >= 2) write(*, *) 'start  :', start
    if(icheck >= 2) write(*, *) 'dimID  :', dimID

    !*    3. Write variable.
    !     ------------------

    !     ** Set 'imap' and WRITE with NCVPTG:
    !     ** NOTE : since the arrays (grid_*) may be over-dimensionned,
    !     **        we use the 'generalised' writing routine NCVPTG
    !     ** (imap tells NetCDF about the memory locations of var)
    imap(1) = 1
    imap(2) = imap(1) * Ni      ! 1st dim of var = Ni
    imap(3) = imap(2) * Nj      ! 2nd dim of var = Nj
    imap(4) = 0                 ! (not used: 0 or 1 time step)
    do iz = 1, 4
        stride(iz) = 1
    enddo
    !     ** NOTE: stride is not used.

    Ierro = NF_PUT_VARM_REAL(FILEid, varVID, start, count, &
                             stride, imap, var(1, 1, 1))
    !     **  line1:              id/file | id/var  |read from...|#data
    !     **  line2:              step    |re-arrang|variable(beg.)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro3.2', Ierro

    !*    4a. Update 'actual_range' attribute.
    !     ------------------------------------

    !     If 'actual_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the old min and max values:
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'actual_range', &
                            Arange)
    !     **line1 ^^^^^^^^^^^^^^  FILEid |var.id | attr.name
    !     **line2                 value

    !     ** Cancel if an error occured : attribute undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNwrite  Info : attribute actual_range ' &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9990 !** Next section

    !     If 'valid_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the min/max valid range (outside = missing val):
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'valid_range', &
                            sValRange)
    if(Ierro /= 0) then
        sValRange(1) = ValRange(1)
        sValRange(2) = ValRange(2)
    endif

    !     Update the min an max
    !     - - - - - - - - - - -

    !     **If this is the first pass, initialise min and max:
    if(Arange(1) == NF_FILL_REAL &
       .OR. (Arange(1) == 0.0 .and. Arange(2) == 0.0)) then
        OkRange = .false.
    else
        OkRange = .true.
    endif

    do ll = 1, NVRlev
        do jj = 1, NVRj
            do ii = 1, NVRi
                if(var(ii, jj, ll) >= sValRange(1) &
                   .and. var(ii, jj, ll) <= sValRange(2)) then
                    if(OkRange) then
                        Arange(1) = MIN(Arange(1), var(ii, jj, ll))
                        Arange(2) = MAX(Arange(2), var(ii, jj, ll))
                    else
                        Arange(1) = var(ii, jj, ll)
                        Arange(2) = var(ii, jj, ll)
                        OkRange = .true.
                    endif
                endif
            enddo
        enddo
    enddo
    if(icheck >= 2) write(*, *) 'Arange', Arange

    !     Set attribute.
    !     - - - - - - - -

    Ierro = NF_PUT_ATT_REAL(FILEid, varVID, 'actual_range', &
                            NF_FLOAT, 2, Arange)
    !     **line1 ^^^^^^^^^^^^^^^ FILEid  |var.id | attr.name
    !     **line2                 type    |len    | attr.value
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)
    TTerr = TTerr + ABS(Ierro)

    !     ** Next section:
9990 continue

    !*    5. Update the optional '[var]_range' special variable.
    !     ------------------------------------------------------
    if(NDIMvar == 4 .and. Nlev < MXlv) then

        !     If '[var]_range' available, get its current value:
        !     - - - - - - - - - - - - - - - - - - - - - - - - - -

        !     ** Get ID of variable [var]_range :
        tmpchr = VARname(1:VNlen)//'_range'
        itmp = VNlen + 6
        Ierro = NF_INQ_VARID(FILEid, tmpchr(1:itmp), varVID)

        !     ** Cancel if an error occured : undefined ?
        if(Ierro /= 0 .and. icheck >= 1) then
            write(*, *) 'UNwrite  Info : [var]_range ' &
                , ' not found -> not written.'
        endif
        if(Ierro /= 0) GOTO 9999 !** UNwrite_end

        !     ** Get the old min and max values:
        !     ** NOTE :
        !     **        we use the 'generalised' reading routine NCVGTG
        !     ** (imap tells NetCDF about the memory locations of var)
        imap(1) = 1
        imap(2) = imap(1) * MXlv
        start(1) = 1
        start(2) = 1
        count(1) = Nlev
        count(2) = 2

        !     ** (See UNread for explanations about NCVGTG)
        Ierro = NF_GET_VARM_REAL(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1, 1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

        !     Update the min an max
        !     - - - - - - - - - - -
        !     **If this is the first pass, initialise min and max:
        !     **(Constant fields shall not be accounted for)
        do ll = 1, Nlev
            if(Srange(ll, 1) == Srange(ll, 2)) then
                Srange(ll, 1) = var(1, 1, ll)
                Srange(ll, 2) = var(1, 1, ll)
            endif
        enddo

        do jj = 1, NVRj
            do ii = 1, NVRi
                do ll = 1, NVRlev
                    Srange(ll, 1) = MIN(Srange(ll, 1), var(ii, jj, ll))
                    Srange(ll, 2) = MAX(Srange(ll, 2), var(ii, jj, ll))
                enddo
            enddo
        enddo
        if(icheck >= 4) write(*, *) 'Srange', Srange

        !     Set special variable [var]_range
        !     - - - - - - - - - - - - - - - - -
        !     **(See UNread for explanations abtout NCVPTG)

        Ierro = NF_PUT_VARM_REAL(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1, 1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    endif  ! End Section 5.

    !     UNwrite_end
    !     -----------
    if(icheck >= 2) write(*, *) 'Errors count:', TTerr
    if(icheck >= 2) write(*, *) 'UNwrite : End'
9999 continue
    RETURN
END

! UNdwrite in double precision
subroutine UNdwrite(FILEid, VARname, itime, &
                    Ni, Nj, Nlev, var)
    use libUN_mod
    implicit none

    INTEGER icheck

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    integer FILEid
    integer itime
    integer Ni, Nj, Nlev
    character * (*) VARname
    REAL(kind=8) var(Ni, Nj, Nlev)

    !     ** local :
    integer MXlv
    PARAMETER(MXlv=500)
    !                ^^^^Maximal # levels for a special output
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    INTEGER NVRi, NVRj, NVRlev
    INTEGER Ierro, TTerr, Nvatts, vtype
    INTEGER dimID(4), dimSIZ(4), count(4)
    INTEGER start(4), stride(4), imap(4)
    character * (Lvnam) dimNAM(4)
    character * (Lvnam) recname
    character * (30) tmpchr
    INTEGER varVID
    INTEGER VNlen, NDIMvar, NSDIvar, tiDI, itmp
    INTEGER iz, ii, jj, ll
    INTEGER iUNLIMDIM
    REAL(kind=4) chkdim
    REAL(kind=4) Arange(2), sValRange(2)
    REAL(kind=4) Srange(MXlv, 2)
    LOGICAL OkRange

    icheck = 0     !** 'debugging' level
    TTerr = 0     !** 'total number of errors

    if(icheck >= 1) write(*, *) 'UNwrite : Begin'

    !*    1. Get the variable field  and dims IDs
    !     ----------------------------------------

    if(icheck >= 2) write(*, *) 'FILEid  :', FILEid

    !     ** getting VARname  size :
    VNlen = VARSIZE(VARname)
    if(icheck >= 3) write(*, *) 'VNlen  :', VNlen
    if(icheck >= 2) write(*, *) 'VARname   :', VARname(1:VNlen)

    !     ** variable field ID :
    Ierro = NF_INQ_VARID(FILEid, VARname(1:VNlen), varVID)

    !     ** Cancel writing if an error occured : variable undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNwrite  Info  : Variable ', VARname(1:VNlen) &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9999 !** UNwrite_end

    !     ** Inquire about the number of dimensions in var :
    !     **
    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       NDIMvar, dimID, Nvatts)
    !     **  line1          id/file  id/var  var name var type
    !     **  line2          # dims   id/dims #attributes
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro1. ', Ierro

    !*    2. Dimensions : inquire about file + compare with input data.
    !     -------------------------------------------------------------

    !     2.1 Inquire dimensions names and sizes :
    ! +   - - - - - - - - - - - - - - - - - - - - -
    do iz = 1, 4
        dimSIZ(iz) = 0
        dimNAM(iz) = '       '
        !       ** Set any unused dimension to "0" size / no name
    enddo
    do iz = 1, NDIMvar
        Ierro = NF_INQ_DIM(FILEid, dimID(iz), dimNAM(iz), dimSIZ(iz))
        !       **                 id/file  id/dim     dimname      dimsize
        !       **                                     !output      output
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)
    enddo
    if(icheck >= 3) write(*, *) 'NDIMvar  ', NDIMvar
    if(icheck >= 3) write(*, *) 'Ierro 2.0', Ierro

    !     2.2 Set writing region according to field dimension : 2D or 3D
    ! +   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    !     ** Set horizontal dimensions (default, for most data) :
    count(1) = Ni
    count(2) = Nj
    ! +   ** Other default values:
    count(3) = 0
    count(4) = 0
    start(1) = 1
    start(2) = 1
    start(3) = 1
    start(4) = 1

    ! +- ------3D+time variable in file-----------
    if(NDIMvar == 4) then
        !       ** 3D space + time:
        NSDIvar = 3     ! Nb. space dims
        tiDI = 4     ! No. of the time dim
        !       ** write 3D space:
        start(3) = 1    ! Start of index 3 in var (here = vert. levs)
        count(3) = Nlev ! Nb. values of index 3 in var
        !       ** write one time step:
        start(4) = itime
        count(4) = 1
        ! +- ------3D *OR* 2D+time var in file--------
    else if(NDIMvar == 3) then
        if(Nlev == 1) then
            !         ** 2D space + time (standard use of UNlib):
            NSDIvar = 2
            tiDI = 3
            !         ** ...write one time step:
            start(3) = itime
            count(3) = 1
        else
            !         ** 3D (no time slice):
            NSDIvar = 3
            tiDI = 0
            !         ** ...write 3rd dimension:
            start(3) = 1
            count(3) = Nlev
        endif
        ! +- ------2D *OR* 1D+time var in file--------
    else if(NDIMvar == 2) then
        if(Nj == 1 .and. dimNAM(2)(1:4) == 'time') then
            !         ** Write a 1D vector at time= itime:
            NSDIvar = 1
            tiDI = 2
            start(2) = itime
            count(2) = 1
        else
            !         ** Usual MAR 2D space (no time):
            NSDIvar = 2
            tiDI = 0
        endif
        ! +- ------1D *OR* 0D+time var in file--------
    else if(NDIMvar == 1) then
        !       ** 1D space or time
        if(Ni == 1) then
            !         ** Write a single element (at itime)
            start(1) = itime
            count(1) = 1
            count(2) = 0
            NSDIvar = 0
            tiDI = 1
        else
            !         ** Write a vector (use only "space" dim 1)
            NSDIvar = 1
            tiDI = 0
            count(2) = 0
        endif
    else
        write(*, *) 'UNwrite ERROR : data field dimension ?'
        STOP
    endif

    !     2.3 Compare file dimensions to input data.
    ! +   - - - - - - - - - - - - - - - - - - - - - -
    !     ** Save variable size for use as "valid" size (-> range):
    NVRi = Ni
    NVRj = Nj
    NVRlev = Nlev
    !     ** Space dimensions :
    if(NSDIvar > 0) then
        do iz = 1, NSDIvar
            if(dimSIZ(iz) > count(iz)) then
                write(*, *) 'UNwrite - WARNING: '
                write(*, *) ' Your field ', VARname, ' has an empty part.'
                write(*, *) ' (for the dimension:', dimNAM(iz), ')'
            else if(dimSIZ(iz) < count(iz)) then
                !         ** Do display "warning" only if truncation
                !            was not "correctly announced" (see header)
                !            (NVR... => stop here when updating the range attribute)
                if(iz == 1) then
                    chkdim = var(Ni, 1, 1)
                    NVRi = dimSIZ(1)
                else if(iz == 2) then
                    chkdim = var(1, Nj, 1)
                    NVRj = dimSIZ(2)
                else if(iz == 3) then
                    chkdim = var(1, 1, Nlev)
                    NVRlev = dimSIZ(3)
                else
                    chkdim = 0.0
                endif
                Ierro = NF_INQ_UNLIMDIM(FILEid, iUNLIMDIM)
                if(dimID(iz) /= iUNLIMDIM) then
                    if(ABS(chkdim - dimSIZ(iz)) > 0.1) then
                        write(*, *) 'UNwrite - WARNING: '
                        write(*, *) ' Your field ', VARname, ' will be truncated.'
                        write(*, *) ' (for the dimension:', dimNAM(iz), ')'
                    endif
                    count(iz) = dimSIZ(iz)
                endif
            endif
        enddo
    endif

    !     ** Time dimension (when defined):
    if(tiDI /= 0) then
        if(itime > dimSIZ(tiDI)) then
            if(icheck >= 1) write(*, *) 'Time limit, ID', dimID(tiDI)
            Ierro = NF_INQ_UNLIMDIM(FILEid, iUNLIMDIM)
            if(dimID(tiDI) /= iUNLIMDIM) then
                write(*, *) 'UNwrite - ERROR:   '
                write(*, *) ' Time index out of range '
                STOP
            endif
        endif
    endif

    if(icheck >= 2) write(*, *) 'Ierro2. ', Ierro
    if(icheck >= 2) write(*, *) 'Dimension names :', dimNAM
    if(icheck >= 2) write(*, *) 'dimSIZ :', dimSIZ
    if(icheck >= 2) write(*, *) 'count  :', count
    if(icheck >= 2) write(*, *) 'start  :', start
    if(icheck >= 2) write(*, *) 'dimID  :', dimID

    !*    3. Write variable.
    !     ------------------

    !     ** Set 'imap' and WRITE with NCVPTG:
    !     ** NOTE : since the arrays (grid_*) may be over-dimensionned,
    !     **        we use the 'generalised' writing routine NCVPTG
    !     ** (imap tells NetCDF about the memory locations of var)
    imap(1) = 1
    imap(2) = imap(1) * Ni      ! 1st dim of var = Ni
    imap(3) = imap(2) * Nj      ! 2nd dim of var = Nj
    imap(4) = 0                 ! (not used: 0 or 1 time step)
    do iz = 1, 4
        stride(iz) = 1
    enddo
    !     ** NOTE: stride is not used.

    Ierro = NF_PUT_VARM_DOUBLE(FILEid, varVID, start, count, &
                             stride, imap, var(1, 1, 1))
    !     **  line1:              id/file | id/var  |read from...|#data
    !     **  line2:              step    |re-arrang|variable(beg.)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro3.2', Ierro

    !*    4a. Update 'actual_range' attribute.
    !     ------------------------------------

    !     If 'actual_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the old min and max values:
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'actual_range', &
                            Arange)
    !     **line1 ^^^^^^^^^^^^^^  FILEid |var.id | attr.name
    !     **line2                 value

    !     ** Cancel if an error occured : attribute undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNwrite  Info : attribute actual_range ' &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9990 !** Next section

    !     If 'valid_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the min/max valid range (outside = missing val):
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'valid_range', &
                            sValRange)
    if(Ierro /= 0) then
        sValRange(1) = ValRange(1)
        sValRange(2) = ValRange(2)
    endif

    !     Update the min an max
    !     - - - - - - - - - - -

    !     **If this is the first pass, initialise min and max:
    if(Arange(1) == NF_FILL_REAL &
       .OR. (Arange(1) == 0.0 .and. Arange(2) == 0.0)) then
        OkRange = .false.
    else
        OkRange = .true.
    endif

    do ll = 1, NVRlev
        do jj = 1, NVRj
            do ii = 1, NVRi
                if(var(ii, jj, ll) >= sValRange(1) &
                   .and. var(ii, jj, ll) <= sValRange(2)) then
                    if(OkRange) then
                        Arange(1) = MIN(Arange(1), var(ii, jj, ll))
                        Arange(2) = MAX(Arange(2), var(ii, jj, ll))
                    else
                        Arange(1) = var(ii, jj, ll)
                        Arange(2) = var(ii, jj, ll)
                        OkRange = .true.
                    endif
                endif
            enddo
        enddo
    enddo
    if(icheck >= 2) write(*, *) 'Arange', Arange

    !     Set attribute.
    !     - - - - - - - -

    Ierro = NF_PUT_ATT_REAL(FILEid, varVID, 'actual_range', &
                            NF_FLOAT, 2, Arange)
    !     **line1 ^^^^^^^^^^^^^^^ FILEid  |var.id | attr.name
    !     **line2                 type    |len    | attr.value
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)
    TTerr = TTerr + ABS(Ierro)

    !     ** Next section:
9990 continue

    !*    5. Update the optional '[var]_range' special variable.
    !     ------------------------------------------------------
    if(NDIMvar == 4 .and. Nlev < MXlv) then

        !     If '[var]_range' available, get its current value:
        !     - - - - - - - - - - - - - - - - - - - - - - - - - -

        !     ** Get ID of variable [var]_range :
        tmpchr = VARname(1:VNlen)//'_range'
        itmp = VNlen + 6
        Ierro = NF_INQ_VARID(FILEid, tmpchr(1:itmp), varVID)

        !     ** Cancel if an error occured : undefined ?
        if(Ierro /= 0 .and. icheck >= 1) then
            write(*, *) 'UNwrite  Info : [var]_range ' &
                , ' not found -> not written.'
        endif
        if(Ierro /= 0) GOTO 9999 !** UNwrite_end

        !     ** Get the old min and max values:
        !     ** NOTE :
        !     **        we use the 'generalised' reading routine NCVGTG
        !     ** (imap tells NetCDF about the memory locations of var)
        imap(1) = 1
        imap(2) = imap(1) * MXlv
        start(1) = 1
        start(2) = 1
        count(1) = Nlev
        count(2) = 2

        !     ** (See UNread for explanations about NCVGTG)
        Ierro = NF_GET_VARM_REAL(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1, 1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

        !     Update the min an max
        !     - - - - - - - - - - -
        !     **If this is the first pass, initialise min and max:
        !     **(Constant fields shall not be accounted for)
        do ll = 1, Nlev
            if(Srange(ll, 1) == Srange(ll, 2)) then
                Srange(ll, 1) = var(1, 1, ll)
                Srange(ll, 2) = var(1, 1, ll)
            endif
        enddo

        do jj = 1, NVRj
            do ii = 1, NVRi
                do ll = 1, NVRlev
                    Srange(ll, 1) = MIN(Srange(ll, 1), var(ii, jj, ll))
                    Srange(ll, 2) = MAX(Srange(ll, 2), var(ii, jj, ll))
                enddo
            enddo
        enddo
        if(icheck >= 4) write(*, *) 'Srange', Srange

        !     Set special variable [var]_range
        !     - - - - - - - - - - - - - - - - -
        !     **(See UNread for explanations abtout NCVPTG)

        Ierro = NF_PUT_VARM_DOUBLE(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1, 1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwrite', Ierro)

    endif  ! End Section 5.

    !     UNwrite_end
    !     -----------
    if(icheck >= 2) write(*, *) 'Errors count:', TTerr
    if(icheck >= 2) write(*, *) 'UNwrite : End'
9999 continue
    RETURN
END

!**
!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNlwrite :  +                                         +
!**  +-------------------------+                                         +
!**  +  * Writes a 2D horizontal LEVEL into a 3D+time NetCDF variable    +
!**  +       OR  a 1D vector           into a 2D+time                    +
!**  +             --            ----         --                         +
!**  +    (SEE ALSO : UNwrite, for all dimensions - this a pecular case  +
!**  +     Note: 1D vectors are writen in the 1st dim of 2D+time)        +
!**  +                                                                   +
!**  +  * Automatically updates attribute 'actual_range' if available    +
!**  +          "          "    special var. '[var]_range'    "          +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from UNcreate OR NetCDF open) +
!**  +    VARname : name given to the variable to write (must be in file)+
!**  +    itime   : No of time step to write to                          +
!**  +    level   : No of level     to write to                          +
!**  +    Ni,  Nj : dimensions of 'var'...                               +
!**  +    var     : A 2D variable to be writen                           +
!**  +-------------------------------------------------------------------+

subroutine UNlwrite(FILEid, VARname, itime, &
                    ilev, Ni, Nj, var)
    use libUN_mod
    implicit none

    INTEGER icheck

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    INTEGER FILEid
    INTEGER itime, ilev
    INTEGER Ni, Nj
    character * (*) VARname
    REAL(kind=4) var(Ni, Nj)

    !     ** local :
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    INTEGER Ierro, TTerr, Nvatts, vtype
    INTEGER dimID(4), dimSIZ(4), count(4)
    INTEGER start(4), stride(4), imap(4)
    INTEGER iUNLIMDIM
    character * (Lvnam) dimNAM(4)
    character * (Lvnam) recname
    character * (30) tmpchr
    INTEGER varVID
    INTEGER VNlen, NDIMvar, NSDIvar, tiDI, ilDI, itmp
    INTEGER iz, ii, jj
    LOGICAL OkRange
    REAL(kind=4) Arange(2), sValRange(2)
    REAL(kind=4) Srange(2)

    icheck = 0     !** 'debugging' level
    TTerr = 0     !** 'total numbe of errors

    if(icheck >= 1) write(*, *) 'UNlwrite : Begin'

    !*    1. Get the variable field  and dims IDs
    !     ----------------------------------------

    if(icheck >= 2) write(*, *) 'FILEid  :', FILEid

    !     ** getting VARname  size :
    VNlen = VARSIZE(VARname)
    if(icheck >= 3) write(*, *) 'VNlen  :', VNlen
    if(icheck >= 2) write(*, *) 'VARname   :', VARname(1:VNlen)

    !     ** variable field ID :
    Ierro = NF_INQ_VARID(FILEid, VARname(1:VNlen), varVID)

    !     ** Cancel writing if an error occured : variable undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNlwrite  Info  : Variable ', VARname(1:VNlen) &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9999 !** UNlwrite_end

    !     ** Inquire about the number of dimensions in var :
    !     **
    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       NDIMvar, dimID, Nvatts)
    !     **  line1          id/file  id/var  var name var type
    !     **  line2          # dims   id/dims #attributes
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro1. ', Ierro

    !*    2. Dimensions : inquire about file + compare with input data.
    !     -------------------------------------------------------------

    !     2.1 Inquire dimensions names and sizes :
    ! +   - - - - - - - - - - - - - - - - - - - - -
    do iz = 1, 4
        dimSIZ(iz) = 0
        dimNAM(iz) = '       '
        !       ** Set any unused dimension to "0" size / no name
    enddo

    do iz = 1, NDIMvar
        Ierro = NF_INQ_DIM(FILEid, dimID(iz), dimNAM(iz), dimSIZ(iz))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)
        !       **           id/file   id/dim    dimname     dimsize    error
        !       **                               !output     output
    enddo
    if(icheck >= 3) write(*, *) 'NDIMvar  ', NDIMvar
    if(icheck >= 3) write(*, *) 'Ierro 2.0', Ierro

    !     2.2 Set writing region according to field dimension :  3D
    ! +   - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    !     ** Set horizontal dimensions (all field dims):
    count(1) = Ni
    count(2) = Nj
    start(1) = 1
    start(2) = 1
    ! +- ------ 3D+time var in file--------
    if(NDIMvar == 4) then
        NSDIvar = 2     ! Nb. input space dims (for a 2D level)
        tiDI = 4     ! No. of the time dim
        !       ** write one level (set the level No) :
        start(3) = ilev ! Start of index 3 in var
        count(3) = 1    ! Nb. values of index 3 in var
        ilDI = 3
        !       ** write one time step:
        start(4) = itime
        count(4) = 1
        ! +- ------ 2D+time var in file--------
    else if(NDIMvar == 3) then
        NSDIvar = 1     ! Nb. input space dims (for a 1D vector)
        tiDI = 3     ! No. of the time dim
        !       ** write one "level" - here a 1D vector in the 1st dim.
        start(2) = ilev ! Start of index 2 in var
        count(2) = 1    ! Nb. values of index 3 in var
        ilDI = 2
        !       ** write one time step:
        start(3) = itime
        count(3) = 1
    else
        write(*, *) 'UNlwrite ERROR : data field dimension ?'
        write(*, *) '  NB: UNlwrite = only for (2 or) 3D +time.'
        STOP
    endif

    !     2.3 Compare file dimensions to input data.
    ! +   - - - - - - - - - - - - - - - - - - - - - -
    !     ** Space dimensions :
    do iz = 1, NSDIvar
        if(dimSIZ(iz) > count(iz)) then
            write(*, *) 'UNlwrite - WARNING: '
            write(*, *) ' Your field ', VARname, ' has an empty part.'
            write(*, *) ' (for the dimension:', dimNAM(iz), ')'
        else if(dimSIZ(iz) < count(iz)) then
            write(*, *) 'UNlwrite - WARNING: '
            write(*, *) ' Your field ', VARname, ' will be truncated.'
            write(*, *) ' (for the dimension:', dimNAM(iz), ')'
            count(iz) = dimSIZ(iz)
        endif
    enddo

    !     ** Space dimensions - check if requested level exists:
    if(dimSIZ(ilDI) < ilev) then
        write(*, *) 'UNlwrite - ERROR: '
        write(*, *) ' The requested level =', ilev
        write(*, *) ' does not exist in the field ', VARname
        write(*, *) ' (for the dimension:', dimNAM(ilDI), ')'
        STOP
    endif

    !     ** Time dimension (when defined):
    if(tiDI /= 0) then
        if(itime > dimSIZ(tiDI)) then
            if(icheck >= 1) write(*, *) 'Time limit, ID', dimID(tiDI)
            Ierro = NF_INQ_UNLIMDIM(FILEid, iUNLIMDIM)
            if(dimID(tiDI) /= iUNLIMDIM) then
                write(*, *) 'UNlwrite - ERROR:  '
                write(*, *) ' Time index out of range '
                STOP
            endif
        endif
    endif

    if(icheck >= 2) write(*, *) 'Ierro2. ', Ierro
    if(icheck >= 2) write(*, *) 'Dimension names :', dimNAM
    if(icheck >= 3) write(*, *) 'dimSIZ :', dimSIZ
    if(icheck >= 3) write(*, *) 'count  :', count
    if(icheck >= 3) write(*, *) 'start  :', start
    if(icheck >= 3) write(*, *) 'dimID  :', dimID

    !*    3. Write variable.
    !     ------------------

    !     ** Set 'imap' and WRITE with NCVPTG:
    !     ** NOTE : since the arrays (grid_*) may be over-dimensionned,
    !     **        we use the 'generalised' writing routine NCVPTG
    !     ** (imap tells NetCDF about the memory locations of var)
    imap(1) = 1
    imap(2) = imap(1) * Ni      ! 1st dim of var = Ni
    imap(3) = imap(2) * Nj      ! (not used: 1 level...)
    imap(4) = 0                 ! (not used: 0 or 1 time step)
    do iz = 1, 4
        stride(iz) = 1
    enddo
    !     ** NOTE: stride is not used.

    Ierro = NF_PUT_VARM_REAL(FILEid, varVID, start, count, &
                             stride, imap, var(1, 1))
    !     **  line1:                id/file | id/var  |read from...|#data
    !     **  line2:                step    |re-arrang|variable(beg.)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)

    if(icheck >= 2) write(*, *) 'Ierro3.2', Ierro

    !*    4a. Update 'actual_range' attribute.
    !     ------------------------------------

    !     If 'actual_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the old min and max values:
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'actual_range', &
                            Arange)
    !     **line1 ^^^^^^^^^^^^^^^ FILEid |var.id | attr.name
    !     **line2                 value

    !     ** Cancel if an error occured : attribute undefined ?
    if(Ierro /= 0 .and. icheck >= 1) then
        write(*, *) 'UNlwrite  Info : attribute actual_range ' &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9990 !** Next section

    !     If 'valid_range' available, get its current value:
    !     - - - - - - - - - - - - - - - - - - - - - - - - - -

    !     ** Get the min/max valid range (outside = missing val):
    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'valid_range', &
                            sValRange)
    if(Ierro /= 0) then
        sValRange(1) = ValRange(1)
        sValRange(1) = ValRange(2)
    endif

    !     Update the min an max
    !     - - - - - - - - - - -

    !     **If this is the first pass, initialise min and max:
    if(Arange(1) == NF_FILL_REAL &
       .OR. (Arange(1) == 0.0 .and. Arange(2) == 0.0)) then
        OkRange = .false.
    else
        OkRange = .true.
    endif

    do jj = 1, Nj
        do ii = 1, Ni
            if(var(ii, jj) >= sValRange(1) &
               .and. var(ii, jj) <= sValRange(2)) then
                if(OkRange) then
                    Arange(1) = MIN(Arange(1), var(ii, jj))
                    Arange(2) = MAX(Arange(2), var(ii, jj))
                else
                    Arange(1) = var(ii, jj)
                    Arange(2) = var(ii, jj)
                    OkRange = .true.
                endif
            endif
        enddo
    enddo
    if(icheck >= 2) write(*, *) 'Arange', Arange

    !     Set attribute.
    !     - - - - - - - -

    Ierro = NF_PUT_ATT_REAL(FILEid, varVID, 'actual_range', &
                            NF_FLOAT, 2, Arange)
    !     **line1 ^^^^^^^^^^^^^^^ FILEid  |var.id | attr.name
    !     **line2                 type    |len    | attr.value
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)
    TTerr = TTerr + ABS(Ierro)

    !     ** Next section:
9990 continue

    !*    5. Update the optional '[var]_range' special variable.
    !     ------------------------------------------------------
    if(NDIMvar == 4) then

        !     If '[var]_range' available, get its current value:
        !     - - - - - - - - - - - - - - - - - - - - - - - - - -

        !     ** Get ID of variable [var]_range :
        tmpchr = VARname(1:VNlen)//'_range'
        itmp = VNlen + 6
        Ierro = NF_INQ_VARID(FILEid, tmpchr(1:itmp), varVID)

        !     ** Cancel if an error occured : undefined ?
        if(Ierro /= 0 .and. icheck >= 1) then
            write(*, *) 'UNlwrite  Info : [var]_range ' &
                , ' not found -> not written.'
        endif
        if(Ierro /= 0) GOTO 9999 !** UNlwrite_end

        !     ** Get the old min and max values:
        !     ** NOTE :
        !     **        we use the 'generalised' reading routine NCVGTG
        !     ** (imap tells NetCDF about the memory locations of var)
        imap(1) = 1
        imap(2) = 0                ! Not used (write only 1 lev)
        start(1) = ilev
        count(1) = 1
        start(2) = 1
        count(2) = 2

        !     ** (See UNread for explanations abtout NCVGTG)
        Ierro = NF_GET_VARM_REAL(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)

        !     Update the min an max
        !     - - - - - - - - - - -
        !     **If this is the first pass, initialise min and max:
        !     **(Constant fields shall not be accounted for)
        if(Srange(1) == Srange(2)) then
            Srange(1) = var(1, 1)
            Srange(2) = var(1, 1)
        endif

        do jj = 1, Nj
            do ii = 1, Ni
                Srange(1) = MIN(Srange(1), var(ii, jj))
                Srange(2) = MAX(Srange(2), var(ii, jj))
            enddo
        enddo
        if(icheck >= 4) write(*, *) 'Srange', Srange

        !     Set special variable [var]_range
        !     - - - - - - - - - - - - - - - - -
        !     **(See UNread for explanations abtout NCVPTG)

        Ierro = NF_PUT_VARM_REAL(FILEid, varVID, start, count, &
                                 stride, imap, Srange(1))
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNlwrite', Ierro)

    endif  ! End Section 5.

    !     UNlwrite_end
    !     -----------
    if(icheck >= 2) write(*, *) 'Errors count:', TTerr
    if(icheck >= 2) write(*, *) 'UNlwrite : End'
9999 continue
    RETURN
END
!**
!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNread :    +                                         +
!**  +-------------------------+                                         +
!**  +  * Reads a model variable from a NetCDF file,                     +
!**  +    and reads the coordinates of the grid upon wich it is defined. +
!**  +    (the NetCDF file must have been opened and must be closed      +
!**  +     after all reading operations). May read an x-y subregion.     +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from NetCDF open)             +
!**  +    VARname  : name of the requested variable.                     +
!**  +    time : [integer*4] is the time index of the data field to read +
!**  +    level: [integer*4] (usefull for 3D-space fields only) :        +
!**  +                       if not=0 --> = no of the level              +
!**  +                                      -> output is 2D (l_dim = 1)  +
!**  +                       if  =0   --> read ALL levels                +
!**  +                                      -> output is 3D              +
!**  +    i_dbeg, j_dbeg      : horizontal indexes of requested region   +
!**  +                          in input data file                       +
!**  +    i_dim, j_dim, l_dim : ...the dimensions of 'var',              +
!**  +                       = the dimensions of the sub-region to read  +
!**  +                       ! l_dim = 1 if level not=0                  +
!**  +                       ! j_dim = 1 if var is 1D                    +
!**  +  OUTPUT :                                                         +
!**  +    varax1[i_dim] (real )                                         +
!**  +    varax2[j_dim]: Horizontal coordinates in the file (lat/lon,...)+
!**  +    varlev[l_dim]: vertical coordinate of the levels               +
!**  +                   (! when level not=0, only varlev(1) is defined) +
!**  +    var_units                 : physical units of var.             +
!**  +    var[i_dim,j_dim,l_dim]    :                                    +
!**  +                            data field values                      +
!**  +                            (var must be defined, and is REAL  )   +
!**  +                                                                   +
!**  +-------------------------------------------------------------------+

subroutine UNread &
    (FILEid, VARname, time, level, i_dbeg, j_dbeg, &
     i_dim, j_dim, l_dim, &
     varax1, varax2, varlev, &
     var_units, var)
    use libUN_mod
    implicit none

    INTEGER icheck

    INTEGER Lvnam
    PARAMETER(Lvnam=21)

    !     ** input
    INTEGER FILEid
    INTEGER time, level, i_dbeg, j_dbeg
    INTEGER i_dim, j_dim, l_dim
    character * (*) VARname

    !     ** output
    REAL(kind=4) varax1(i_dim), varax2(j_dim), varlev(l_dim)
    character * (*) var_units
    REAL(kind=4) var(i_dim, j_dim, l_dim)

    !     ** local :
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    REAL(kind=4) varmin, varmax, add_offset, scale_factor
    INTEGER Ierro, Nvatts, vtype
    INTEGER dimID(4), dimSIZ(4), dimREG(4)
    INTEGER start(4), begREG(4), count(4), stride(4), imap(4)
    character * (Lvnam) dimNAM(4)
    character * (Lvnam) dNAMver, dNAMtim
    character * (Lvnam) recname
    character * (10) Routine
    INTEGER ax1VID, ax2VID, verVID, timVID, varVID
    INTEGER VNlen, varNUMDIM
    INTEGER ii, jj, ll, z

    icheck = 0
    !*    0. Initialisations
    !     ------------------
    Routine = 'UNread'
    if(icheck >= 1) write(*, *) 'UNread : Begin'

    do ii = 1, 4
        stride(ii) = 1
        begREG(ii) = 1
        start(ii) = 1
    enddo

    !*    1. Get the variable field  and dims IDs
    !     ----------------------------------------

    if(icheck >= 3) write(*, *) 'FILEid  :', FILEid

    !     ** getting VARname  size :
    VNlen = VARSIZE(VARname)
    if(icheck >= 3) write(*, *) 'VNlen  :', VNlen
    if(icheck >= 2) write(*, *) 'VARname   :', VARname(1:VNlen)

    !     ** variable field ID :
    Ierro = NF_INQ_VARID(FILEid, VARname(1:VNlen), varVID)

    !*    1b. Handle non-existing variables
    !     ---------------------------------
    if(Ierro /= NF_NOERR) then
        if(Ierro == NF_ENOTVAR .and. iVarWarn <= 1) then
            if(iVarWarn == 1) then
                write(*, *) 'WARNING (UNsread): variable not found:'
                write(*, *) '     ', varName
            endif
            do ll = 1, l_dim
                do jj = 1, j_dim
                    do ii = 1, i_dim
                        var(ii, jj, ll) = VarRepl
                    enddo
                enddo
            enddo
            RETURN  ! EXIT subroutine, read nothing
        endif
        write(*, *) 'Error reading variable: ', VARname(1:VNlen)
        call HANDLE_ERR('UNsread', Ierro)
    endif

    !     1c. Inquire about the number of dimensions in var
    !     -------------------------------------------------

    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       varNUMDIM, dimID, Nvatts)
    !     **  line1          id/file    id/var  var name  var type
    !     **  line2          # dims    id/dims #attributes
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNsread', Ierro)

    if(icheck >= 3) write(*, *) 'Ierro1. ', Ierro

    !*    2. Dimensions : in the reading region and in the file.
    !     ------------------------------------------------------

    !     ** inquire dimensions names and sizes :
    do z = 1, varNUMDIM
        Ierro = NF_INQ_DIM(FILEid, dimID(z), dimNAM(z), dimSIZ(z))
        !       **                 id/file  id/dim    dimname    dimsize
        !       **                                    !output    output
        if(Ierro /= NF_NOERR) call HANDLE_ERR(Routine, Ierro)
    enddo

    !     ** In this version, we read only a xy subregion of the file :
    dimREG(1) = i_dim
    dimREG(2) = j_dim
    begREG(1) = i_dbeg
    begREG(2) = j_dbeg
    if(begREG(1) < 1) begREG(1) = 1
    if(begREG(2) < 1) begREG(2) = 1

    !     ** Set reading region according to field dimension : 2D or 3D
    if(varNUMDIM == 4) then
        !       ** for 3D fields :
        if(level > 0) then
            !       ** one level is read :
            dimREG(3) = 1
            begREG(3) = level
            dNAMver = dimNAM(3)
        else
            !       ** all levels are read :
            dimREG(3) = l_dim
            begREG(3) = 1
            dNAMver = dimNAM(3)
        endif
        !       ** one time step is read:
        dimREG(4) = 1
        begREG(4) = time
        dNAMtim = dimNAM(4)
    else if(varNUMDIM == 3) then
        !       ** for 2D space fields + time:
        !       ** one time step is read:
        dimREG(3) = 1
        begREG(3) = time
        dNAMtim = dimNAM(3)
        dimREG(4) = 0
        begREG(4) = 0
        dimNAM(4) = 'none'
    else if(varNUMDIM == 2) then
        !       ** for 2D fields :
        !       ** no time step is read:
        dimREG(3) = 0
        begREG(3) = 0
        dNAMtim = 'none'
        dimNAM(3) = 'none'
        dimREG(4) = 0
        begREG(4) = 0
        dimNAM(4) = 'none'
    else if(varNUMDIM == 1) then
        !       ** for 1D variable :
        !       ** not assumed to be on a XYZ grid,
        !       ** just read a vector
        dimREG(2) = 0
        begREG(2) = 0
        dimNAM(2) = 'none'
        dimREG(3) = 0
        begREG(3) = 0
        dimNAM(3) = 'none'
        dNAMtim = 'none'
        dimREG(4) = 0
        begREG(4) = 0
        dimNAM(4) = 'none'
    else
        write(*, *) 'UNread ERROR : data field dimension ?'
        STOP
    endif

    do z = 1, varNUMDIM
        if(begREG(z) > dimSIZ(z)) then
            write(*, *) 'UNread - ERROR   : requested area out      '
            write(*, *) '                   of file area.          '
            write(*, *) '  (for the dimension:', dimNAM(z), ')'
            STOP
        endif
        if(dimSIZ(z) < (dimREG(z) + begREG(z) - 1)) then
            write(*, *) 'UNread - WARNING : empty portion in field, '
            write(*, *) '  requested region > file contents       '
            write(*, *) '  (for the dimension:', dimNAM(z), ')'
            dimREG(z) = dimSIZ(z) - begREG(z) + 1
        endif
    enddo

    if(icheck >= 3) write(*, *) 'Ierro2. ', Ierro
    if(icheck >= 2) write(*, *) 'Dimension names :', dimNAM
    if(icheck >= 2) write(*, *) 'dimSIZ :', dimSIZ
    if(icheck >= 2) write(*, *) 'dimREG :', dimREG
    if(icheck >= 2) write(*, *) 'begREG :', begREG
    if(icheck >= 3) write(*, *) 'dimID  :', dimID

    !*    3. Get the variables IDs for the grid points locations.
    !     -------------------------------------------------------

    if(varNUMDIM >= 2) then
        Ierro = NF_INQ_VARID(FILEid, dimNAM(1), ax1VID)
        if(Ierro /= NF_NOERR) then
            if(Ierro == NF_ENOTVAR) then
                write(*, *) 'Coordinate values not found:', dimNAM(1)
            endif
            call HANDLE_ERR(Routine, Ierro)
        endif
        Ierro = NF_INQ_VARID(FILEid, dimNAM(2), ax2VID)
        if(Ierro /= NF_NOERR) then
            if(Ierro == NF_ENOTVAR) then
                write(*, *) 'Coordinate values not found:', dimNAM(2)
            endif
            call HANDLE_ERR(Routine, Ierro)
        endif
    endif
    if(varNUMDIM >= 3) then
        Ierro = NF_INQ_VARID(FILEid, dNAMtim, timVID)
        if(Ierro /= NF_NOERR) then
            if(Ierro == NF_ENOTVAR) then
                write(*, *) 'Coordinate values not found:', dNAMtim
            endif
            call HANDLE_ERR(Routine, Ierro)
        endif
    endif
    if(varNUMDIM == 4) then
        Ierro = NF_INQ_VARID(FILEid, dNAMver, verVID)
        if(Ierro /= NF_NOERR) then
            if(Ierro == NF_ENOTVAR) then
                write(*, *) 'Coordinate values not found:', dNAMver
            endif
            call HANDLE_ERR(Routine, Ierro)
        endif
    endif
    !     **                      id/file  name    id/var

    if(icheck >= 3) write(*, *) 'Ierro3. ', Ierro

    !*    4. Get attributes.
    !     ------------------

    if(varNUMDIM >= 2) then   !Not for 1D vectors (special case)
        !       ** units attribute
        Ierro = NF_GET_ATT_TEXT(FILEid, varVID, 'units', &
                                var_units)
        if(Ierro /= NF_NOERR) then
            if(Ierro == NF_ENOTATT) then
                write(*, *) 'Note (UNread): units not found for'
                write(*, *) '     ', varName
                var_units = ' '
            else
                call HANDLE_ERR('UNread', Ierro)
            endif
        endif

        if(icheck >= 2) write(*, *) 'var_units :', var_units
    endif

    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'scale_factor', &
                            scale_factor)

    Ierro = NF_GET_ATT_REAL(FILEid, varVID, 'add_offset', &
                            add_offset)

    if(Ierro /= NF_NOERR .and. Ierro == NF_ENOTATT) then
        scale_factor = 1.
        add_offset = 0.
    else
        if(icheck >= 2) &
            print *, VARname(1:VNlen)//" scale_factor", scale_factor
        if(icheck >= 2) &
            print *, VARname(1:VNlen)//"   add_offset", add_offset
    endif

    !*    5. Get values.
    !     --------------
    !*    5.1 ...for the grid points locations.
    !     -------------------------------------

    !     ** Horizontal : always read, except for 1D vectors
    if(varNUMDIM >= 2) then
        count(1) = dimREG(1)
        start(1) = begREG(1)
        Ierro = NF_GET_VARA_REAL(FILEid, ax1VID, start, count, varax1)
        !       **                       id/file id/var from  #data data
        if(Ierro /= NF_NOERR) call HANDLE_ERR(Routine, Ierro)
        count(1) = dimREG(2)
        start(1) = begREG(2)
        Ierro = NF_GET_VARA_REAL(FILEid, ax2VID, start, count, varax2)
        if(Ierro /= NF_NOERR) call HANDLE_ERR(Routine, Ierro)
    endif

    !     ** vertical :  only for 3D fields.
    if(varNUMDIM == 4) then
        start(1) = begREG(3)
        count(1) = dimREG(3)
        Ierro = NF_GET_VARA_REAL(FILEid, verVID, start, count, varlev)
        if(Ierro /= NF_NOERR) call HANDLE_ERR(Routine, Ierro)
    endif

    if(icheck >= 3) write(*, *) 'Ierro5.1', Ierro

    !*    5.2 ...for the the variable.
    !     ----------------------------

    !     ** Set 'imap' and READ with NCVGTG:
    !     ** NOTE :
    !     **        we use the 'generalised' reading routine NCVGTG
    !     ** (imap tells NetCDF about the memory locations of var)
    imap(1) = 1
    imap(2) = imap(1) * i_dim  ! 1st dim of var = i_dim
    imap(3) = imap(2) * j_dim  ! 2nd dim of var = j_dim
    imap(4) = 0                !  Should NEVER be used

    Ierro = NF_GET_VARM_REAL(FILEid, varVID, begREG, dimREG, &
                             stride, imap, var(1, 1, 1))
    !     **  line1:               id/file  | id/var  |read from...|#data
    !     **  line2:               step     |re-arrang|variable(beg.)
    !     ** NOTE: stride is not used here.
    if(Ierro /= NF_NOERR) call HANDLE_ERR(Routine, Ierro)

    if(icheck >= 3) write(*, *) 'Ierro5.2', Ierro

    do ll = 1, l_dim
        do jj = 1, j_dim
            do ii = 1, i_dim
                var(ii, jj, ll) = var(ii, jj, ll) * scale_factor &
                                  + add_offset
            enddo
        enddo
    enddo

    !*    6. Check data
    !     -------------
    if(ireadchk >= 1) then
        varmax = var(1, 1, 1)
        varmin = var(1, 1, 1)
        do ll = 1, l_dim
            do jj = 1, j_dim
                do ii = 1, i_dim
                    var(ii, jj, ll) = var(ii, jj, ll) + 0.
                    !           This fixes underflow values but must compile with -fpe1
                    varmax = MAX(var(ii, jj, ll), varmax)
                    varmin = MIN(var(ii, jj, ll), varmin)
                enddo
            enddo
        enddo
        if(varmin < vReadMin .OR. varmax > vReadMax) then
            write(*, *) 'WARNING (UNread): variable ', VARname
            write(*, *) '  is out of specified bounds;'
            write(*, *) '  min is:', varmin
            write(*, *) '  max is:', varmax
        endif
    endif

    if(icheck >= 2) write(*, *) 'UNread : End'

ENDsubroutine UNread

!**
!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNsread :   +                                         +
!**  +-------------------------+                                         +
!**  +  * Reads a model variable from a NetCDF file,                     +
!**  +    SIMPLIFIED VERSION of  UNread  : does NOT read coordinates.    +
!**  +                                                                   +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from NetCDF open)             +
!**  +    VARname  : name of the requested variable.                     +
!**  +    time : [integer*4] is the time index of the data field to read +
!**  +    level: [integer*4] (usefull for 3D-space fields only) :        +
!**  +                       if not=0 --> = no of the level              +
!**  +                                      -> output is 2D (l_dim = 1)  +
!**  +                       if  =0   --> read ALL levels                +
!**  +                                      -> output is 3D              +
!**  +    i_dbeg, j_dbeg      : horizontal indexes of requested region   +
!**  +                          in input data file                       +
!**  +    i_dim, j_dim, l_dim : ...the dimensions of 'var',              +
!**  +                       = the dimensions of the sub-region to read  +
!**  +                       ! l_dim = 1 if level not=0                  +
!**  +                       ! j_dim = 1 if var is 1D                    +
!**  +  OUTPUT :                                                         +
!**  +    var_units                 : physical units of var.             +
!**  +    var[i_dim,j_dim,l_dim]    :                                    +
!**  +                            data field values                      +
!**  +                            (var must be defined, and is REAL  )   +
!**  +                                                                   +
!**  +-------------------------------------------------------------------+

subroutine UNsread &
    (FILEid, VARname, time, level, i_dbeg, j_dbeg, &
     i_dim, j_dim, l_dim, &
     var_units, var)

    implicit none

    !     ** input
    integer FILEid
    integer time, level, i_dbeg, j_dbeg
    integer i_dim, j_dim, l_dim
    character * (*) VARname

    !     ** output
    character * (*) var_units
    REAL(kind=4) var(i_dim, j_dim, l_dim)
    REAL(kind=4) varax1(i_dim), varax2(j_dim), varlev(l_dim)

    call UNread(FILEid, VARname, time, level, i_dbeg, j_dbeg, &
                i_dim, j_dim, l_dim, &
                varax1, varax2, varlev, &
                var_units, var)

ENDsubroutine UNsread

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNwcatt :   +                                         +
!**  +-------------------------+                                         +
!**  +  *Character Attributes creation and (over)writing                 +
!**  +    (the NetCDF file must be open, in data mode)                   +
!**  +  *WARNING: this routine (may?) use a temporary disk space         +
!**  +            equal to the file length (duplicate the file)          +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from UNcreate OR NetCDF open) +
!**  +    varnam  : name of variable to which attribute shall be attached+
!**  +              or 'GLOBAL_ATT'                                      +
!**  +    attnam  : name of writen attribute.                            +
!**  +    attval  : string to be assigned to attribute.                  +
!**  +              (never inclulde more than 3 consecutive blanks !)    +
!**  +                                                                   +
!**  +  Note : all arguments except FILEid  are strings of any length    +
!**  +-------------------------------------------------------------------+

subroutine UNwcatt(FILEid, varnam, attnam, attval)
    use libUN_mod
    implicit none
    !     **Input:

    INTEGER FILEid
    character * (*) varnam
    character * (*) attnam
    character * (*) attval

    !     **Local:
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    INTEGER Nlen, Ierro, varVID, Vlen, TTerr
    INTEGER icheck
    icheck = 0     !** 'debugging' level

    if(icheck >= 1) write(*, *) 'UNwcatt : Begin'

    !*    Get the variable ID
    !     -------------------

    if(icheck >= 2) write(*, *) 'FILEid  :', FILEid

    !     ** getting varnam size :
    Nlen = VARSIZE(varnam)

    !     ** Case of global attributes:
    if(varnam(1:Nlen) == 'GLOBAL_ATT') then
        varVID = NF_GLOBAL

    else

        !     ** Get variable ID to which att is attached to:
        Ierro = NF_INQ_VARID(FILEid, varnam(1:Nlen), varVID)
        TTerr = ABS(Ierro)

        !       ** Cancel writing if an error occured : variable undefined ?
        if(Ierro /= 0) then
            write(*, *) 'UNwcatt -ERROR : Variable ', varnam(1:Nlen) &
                , ' not found -> not written.'
        endif
        if(Ierro /= 0) RETURN !** UNwcatt_end

    endif

    !     Switch to Define Mode,
    !       because attribute may be created or change size.
    !     --------------------------------------------------
    Ierro = NF_REDEF(FILEid)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwcatt', Ierro)

    !     Set attribute.
    !     --------------

    !     ** getting attnam [char] size :
    Nlen = VARSIZE(attnam)
    !     ** getting attval [char] size :
    Vlen = VARSIZE(attval)

    Ierro = NF_PUT_ATT_TEXT(FILEid, varVID, attnam(1:Nlen), &
                            Vlen, attval(1:Vlen))
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwcatt', Ierro)
    !     **line1^^^^ FILEid |var.id | attr.name
    !     **line2     type   | len   | attr.value | flag
    TTerr = TTerr + ABS(Ierro)

    !     Leave define mode (!file remains open )
    !     ---------------------------------------
    Ierro = NF_ENDDEF(FILEid)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwcatt', Ierro)

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNwratt :   +                                         +
!**  +-------------------------+                                         +
!**  +  *real attributes writing  - ! Can not create new attrib !      +
!**  +    (the NetCDF file must be open)                                 +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : input file identifier (from UNcreate OR NetCDF open) +
!**  +    varnam  : name given to the variable to write (must be in file)+
!**  +    attnam  : name of treated attribute.                           +
!**  +    Nvals   : Number of values of that attribute                   +
!**  +    atvalsi(Nvals) : real vector of values for attribute.        +
!**  +                                                                   +
!**  +-------------------------------------------------------------------+

!                    WARNING: this routine uses a temporary disk space
!                             equal to the file length (duplicate the file)
!                             (its use is NOT recommended)

subroutine UNwratt(FILEid, varnam, attnam, Nvals, atvals)
    use libUN_mod
    implicit none

    !     **Input:

    INTEGER FILEid, Nvals
    character * (*) varnam
    character * (*) attnam
    REAL(kind=4) atvals(Nvals)

    !     **Local:
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    INTEGER Nlen, Ierro, varVID
    INTEGER icheck, TTerr
    icheck = 0     !** 'debugging' level
    TTerr = 0

    if(icheck >= 1) write(*, *) 'UNwratt : Begin'

    !*    Get the variable ID
    !     -------------------
    if(icheck >= 2) write(*, *) 'FILEid  :', FILEid

    !     ** getting varnam size :
    Nlen = VARSIZE(varnam)

    !     ** variable ID :
    Ierro = NF_INQ_VARID(FILEid, varnam(1:Nlen), varVID)
    TTerr = TTerr + ABS(Ierro)

    !     ** Cancel writing if an error occured : variable undefined ?
    if(Ierro /= 0) then
        write(*, *) 'UNwratt -ERROR : Variable ', varnam(1:Nlen) &
            , ' not found -> not written.'
    endif
    if(Ierro /= 0) GOTO 9999 !** UNwratt_end

    !     Set attribute.
    !     --------------

    !     ** getting attnam [char] size :
    Nlen = VARSIZE(attnam)

    Ierro = NF_PUT_ATT_REAL(FILEid, varVID, attnam(1:Nlen), &
                            NF_FLOAT, nvals, atvals)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNwratt', Ierro)
    !     **line1^^^^FILEid |var.id | attr.name
    !     **line2    type   | attr.value | flag
    TTerr = TTerr + ABS(Ierro)

9999 continue
    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNwopen :   +                            libUN (0896) +
!**  +-------------------------+-----------------------------------------+
!**  +  * Open a NetCDF file for writing.                                +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEnam : file name                                            +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +    FILEid  : NetCDF file identifier ('logical unit')              +
!**  +---------------------------------------------------------------7++++

subroutine UNwopen(FILEnam, FILEid)
    use libUN_mod
    implicit none

    !     ** input
    character * (*) FILEnam

    !     ** output
    INTEGER FILEid

    !     ** local :
    INTEGER Ierro
    INTEGER icheck

    icheck = 0

    ! +   Routines which opens a file must reset libUN internals:
    call UNparam('RESET_PARAMS_', 0.0)

    !     ** Open NetCDF file, for read-only:
    !     -----------------------------------
    Ierro = NF_OPEN(FILEnam, NF_WRITE, FILEid)
    if(Ierro /= NF_NOERR) then
        write(*, *) 'Error opening file: ', FILEnam
        call HANDLE_ERR('UNwopen', Ierro)
    endif

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNropen :   +                            libUN (0896) +
!**  +-------------------------+-----------------------------------------+
!**  +  * Open a NetCDF file for reading,                                +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEnam : file name                                            +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +    FILEid  : NetCDF file identifier ('logical unit')              +
!**  +    FILEtit : title of the NetCDF file                             +
!**  +              ! [CHAR], must be defined (length > length(title) !) +
!**  +---------------------------------------------------------------7++++

subroutine UNropen(FILEnam, FILEid, FILEtit)
    use libUN_mod
    implicit none

    !     ** input
    character * (*) FILEnam

    !     ** output
    INTEGER FILEid
    character * (*) FILEtit

    !     ** local :
    INTEGER Ierro
    INTEGER icheck

    icheck = 0

    if(icheck >= 2) write(*, *) 'UNropen: Begin'
    if(icheck >= 2) write(*, *) 'FILEnam: ', FILEnam

    ! +   Routines which opens a file must reset libUN internals:
    call UNparam('RESET_PARAMS_', 0.0)

    !     ** Open NetCDF file, for read-only:
    !     -----------------------------------
    Ierro = NF_OPEN(FILEnam, NF_NOWRITE, FILEid)
    if(Ierro /= NF_NOERR) then
        write(*, *) 'Error opening file: ', FILEnam
        call HANDLE_ERR('UNropen', Ierro)
    endif

    !     ** Read title attribute,
    !     ------------------------

    !     ** Read attribute:
    Ierro = NF_GET_ATT_TEXT(FILEid, NF_GLOBAL, 'title', &
                            FILEtit)

    !     ** Display message if an error occured :
    !     **  no title or title too long ?
    !     !if (Ierro.ne.0) then
    !     !   write(*,*) 'UNropen WARNING: no title or title too long'
    !     !end if
    if(icheck >= 2) write(*, *) 'UNropen: End'

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNgtime :   +                            libUN (0896) +
!**  +-------------------------+-----------------------------------------+
!**  +  * From a given value of desired 'time' coordinate,               +
!**  +    gets the coordinate index ('iteration no') + found time value  +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : NetCDF file identifier (from UNropen)                +
!**  +    RQtime  : ReQuested time                                       +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +    RDtime  : The last time for wich RDtime .le. RQtime            +
!**  +    Ftime   : The next time value Following RDtime                 +
!**  +              (-1 if it would be after end-of-file)                +
!**  +    it      : The time index : RDtime = time(it)                   +
!**  +---------------------------------------------------------------7++++

subroutine UNgtime(FILEid, RQtime, RDtime, Ftime, it)
    use libUN_mod
    implicit none

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    INTEGER FILEid
    REAL(kind=4) RQtime

    !     ** output
    REAL(kind=4) RDtime, Ftime
    INTEGER it

    !     ** local :
    INTEGER Ierro, timVID
    INTEGER timDID
    REAL(kind=4) gtim
    INTEGER K, KHI, KLO, Kmax
    INTEGER Mindex(1)
    INTEGER icheck
    character * (Lvnam) dimNAM(1)

    icheck = 0

    !     ** Kmax= nb pas de temps dans le fichier, = dim(time):
    !     ** - - - - - - - - - - - - - - - - - - - - - - - - - -
    !
    Ierro = NF_INQ_DIMID(FILEid, 'time', timDID)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)
    !     **^^ Dimension'time' NetCDF index

    Ierro = NF_INQ_DIM(FILEid, timDID, dimNAM, Kmax)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)
    !     **         id/file  id/dim   dimname dimsize  error
    !     **                           !output output

    !     ** Read/Search the requested time step.
    !     ** - - - - - - - - - - - - - - - - - - -

    Ierro = NF_INQ_VARID(FILEid, 'time', timVID)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)
    !                                         **^^ Variable 'time' NetCDF index

    KLO = 1
    KHI = Kmax

1   if(KHI - KLO > 1) then
        K = (KHI + KLO) / 2

        !       ** Set the position of the needed time step:
        Mindex(1) = K
        !       ** Get 1 time value (gtim = time(K)):
        Ierro = NF_GET_VAR1_REAL(FILEid, timVID, Mindex, gtim)
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)

        if(gtim > RQtime) then
            KHI = K
        else
            KLO = K
        endif
        GOTO 1
    endif
    it = KLO
    !     ** read RDtime= time(KLO)
    Mindex(1) = KLO
    Ierro = NF_GET_VAR1_REAL(FILEid, timVID, Mindex, RDtime)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)
    !     ** read Ftime= time(KHI)
    Mindex(1) = KHI
    Ierro = NF_GET_VAR1_REAL(FILEid, timVID, Mindex, Ftime)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgtime', Ierro)

    !     ** if the last available time step is before
    !     **     the requested time, then KHI and KLO are the
    !     **     two last available time step. Correct this :
    if(RQtime >= Ftime) then
        RDtime = Ftime
        it = KHI
        Ftime = -1.0
    endif

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNgindx :   +                            libUN (0199) +
!**  +-------------------------+-----------------------------------------+
!**  +  * From a given value of a desired coordinate,                    +
!**  +    gets the coordinate index + found the coresp. coordinate value +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : NetCDF file identifier (from UNropen)                +
!**  +    Cname   : The name of the coordinate                           +
!**  +    RQval   : The requested value for that coordinate              +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +    RDval   : The last value for wich RDval .le. RQval             +
!**  +    Fval    : The next val value Following RDval                   +
!**  +              (-1 if it would be after end-of-file)                +
!**  +    indx    : The val index : RDval = value_of_Cname(it)           +
!**  +---------------------------------------------------------------7++++

subroutine UNgindx(FILEid, Cname, RQval, RDval, Fval, indx)
    use libUN_mod
    implicit none

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    INTEGER FILEid
    character * (*) Cname
    REAL(kind=4) RQval

    !     ** output
    REAL(kind=4) RDval, Fval
    INTEGER indx

    !     ** local :
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    REAL(kind=4) gval
    INTEGER Ierro
    INTEGER varDID, VNlen, varVID, varNUMDIM
    INTEGER Nvatts, vtype
    INTEGER K, KHI, KLO, Kmax
    INTEGER Mindex(1), dimID(4)
    INTEGER icheck
    character * (Lvnam) dimNAM(4)
    character * 13 recname

    icheck = 0

    !     ** Kmax= nb pas de temps dans le fichier, = dim(val):
    !     ** - - - - - - - - - - - - - - - - - - - - - - - - - -
    !     ** get Cname string size :
    VNlen = VARSIZE(Cname)
    !
    !     ** get variable ID :
    Ierro = NF_INQ_VARID(FILEid, Cname(1:VNlen), varVID)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)
    !
    !     ** Inquire about the id of the dimension:
    !     **
    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       varNUMDIM, dimID, Nvatts)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)
    !     **  line1  id/file   id/var  var name  var type
    !     **  line2   # dims   id/dims #attributes
    varDID = dimID(1)
    !     ^^^At last, the id of the relevant dimension.

    Ierro = NF_INQ_DIM(FILEid, varDID, dimNAM, Kmax)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)
    !     **         id/file  id/dim   dimname dimsize  error
    !     **                           !output output
    !     ** (Kmax is what we needed: size of the dimension)

    !     ** Read/Search the requested val step.
    !     ** - - - - - - - - - - - - - - - - - - -

    KLO = 1
    KHI = Kmax

1   if(KHI - KLO > 1) then
        K = (KHI + KLO) / 2

        !       ** Set the position of the needed val step:
        Mindex(1) = K
        !       ** Get 1 val value (gval = val(K)):
        Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, gval)
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)

        if(gval > RQval) then
            KHI = K
        else
            KLO = K
        endif
        GOTO 1
    endif
    indx = KLO
    !     ** read RDval= val(KLO)
    Mindex(1) = KLO
    Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, RDval)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)
    !     ** read Fval= val(KHI)
    Mindex(1) = KHI
    Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, Fval)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNgindex', Ierro)

    !     ** if the last available val step is before
    !     **     the requested val, then KHI and KLO are the
    !     **     two last available val step. Correct this :
    if(RQval >= Fval) then
        RDval = Fval
        indx = KHI
        Fval = -1.0
    endif

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNfindx :   +                            (libUN  2003)+
!**  +-------------------------+-----------------------------------------+
!**  +  * Intended to replace UNgindx or UNgtime                         +
!**  +    From a given value of a desired coordinate,                    +
!**  +    gets the coordinate index + the coresp. coordinate value       +
!**  +    This version solves the issue of Dates at year change          +
!**  +    occuring because 1 jan is < 31 dec.  Not optimised.            +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : NetCDF file identifier (from UNropen)                +
!**  +    Cname   : The name of the coordinate                           +
!**  +    RQval   : The requested value for that coordinate              +
!**  +                                                                   +
!**  +  OUTPUT :                                                         +
!**  +    RDval   : The file value closest to RQval                      +
!**  +    Fval    : The next value in the file                           +
!**  +              (-1 if after file end)                               +
!**  +              (This is mainly for compatibility with older version)+
!**  +    indx    : The val index : RDval = value_of_Cname(it)           +
!**  +              (-1 may be returned if the value can't be found)     +
!**  +---------------------------------------------------------------7++++

subroutine UNfindx(FILEid, Cname, RQval, RDval, Fval, indx)
    use libUN_mod
    implicit none

    INTEGER Lvnam
    PARAMETER(Lvnam=20)

    !     ** input
    INTEGER FILEid
    character * (*) Cname
    REAL(kind=4) RQval

    !     ** output
    REAL(kind=4) RDval, Fval
    INTEGER indx

    !     ** local :
    INTEGER VARSIZE
    EXTERNAL VARSIZE
    REAL(kind=4) gval, bmatch, gdist
    INTEGER Ierro
    INTEGER varDID, VNlen, varVID, varNUMDIM
    INTEGER Nvatts, vtype
    INTEGER K, KHI, KLO, Kmax
    INTEGER Mindex(1), dimID(4)
    INTEGER icheck
    character * (Lvnam) dimNAM(4)
    character * 13 recname

    icheck = 0

    !     ** Kmax= nb pas de temps dans le fichier, = dim(val):
    !     ** - - - - - - - - - - - - - - - - - - - - - - - - - -
    !     ** get Cname string size :
    VNlen = VARSIZE(Cname)
    !
    !     ** get variable ID :
    Ierro = NF_INQ_VARID(FILEid, Cname(1:VNlen), varVID)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)
    !
    !     ** Inquire about the id of the dimension:
    !     **
    Ierro = NF_INQ_VAR(FILEid, varVID, recname, vtype, &
                       varNUMDIM, dimID, Nvatts)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)
    !     **  line1  id/file   id/var  var name  var type
    !     **  line2   # dims   id/dims #attributes
    varDID = dimID(1)
    !     ^^^At last, the id of the relevant dimension.

    Ierro = NF_INQ_DIM(FILEid, varDID, dimNAM, Kmax)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)
    !     **         id/file  id/dim   dimname dimsize  error
    !     **                           !output output
    !     ** (Kmax is what we needed: size of the dimension)

    !     ** Read/Search the requested val step.
    !     ** - - - - - - - - - - - - - - - - - - -

    !     This is a workaround, not optimised as stated above.
    !     We simply look at all values sequencially.
    !
    bmatch = 1.E10
    KLO = -1

    do K = 1, KMAX

        !       ** Get 1 val value (gval = val(K)):
        Mindex(1) = K
        Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, gval)
        if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)

        gdist = ABS(gval - RQval)
        if(gdist < bmatch) then

            bmatch = gdist
            KLO = K

        endif

    enddo

    indx = KLO

    KHI = min((KLO + 1), KMAX)

    !     ** read values...

    Mindex(1) = KLO
    Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, RDval)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)
    !     ** read Fval= val(KHI)
    Mindex(1) = KHI
    Ierro = NF_GET_VAR1_REAL(FILEid, varVID, Mindex, Fval)
    if(Ierro /= NF_NOERR) call HANDLE_ERR('UNfindex', Ierro)

    if(KHI == KLO) then
        Fval = -1.0
    endif

    if(bmatch > 1.E9) then
        Fval = -1.0
        indx = -1
    endif

    RETURN
END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNclose :   +                            libUN (0300) +
!**  +-------------------------+-----------------------------------------+
!**  +  * Close the desired file                                         +
!**  +    Created to suppress the need the directly call a netcdf        +
!**  +    routine from a program                                         +
!**  +                                                                   +
!**  +  INPUT :                                                          +
!**  +    FILEid  : NetCDF file identifier (from UNropen)                +
!**  +---------------------------------------------------------------7++++

subroutine UNCLOSE(FILEid)
    use libUN_mod
    implicit none

    integer Ierro, FILEid

    Ierro = NF_CLOSE(FILEid)
    if(Ierro /= NF_NOERR) then
        call HANDLE_ERR('UNclose', Ierro)
    endif

END

!**  +-------------------------+-----------------------------------------+
!**  +  subroutine UNparam :   +                            libUN (0202) +
!**  +-------------------------+-----------------------------------------+
!**  +  Changes some global libUN parameters                             +
!**  +  NB: default values are set at first libUN call                   +
!**  +                                                                   +
!**  +                                                                   +
!**  +  INPUT : pname   name of the parameters to set                    +
!**  +          pvalue  the requested new value                          +
!**  +                                                                   +
!**  +---------------------------------------------------------------7++++

subroutine UNparam(pname, pvalue)
    use libUN_mod
    implicit none

    character * (*) pname
    REAL(kind=4) pvalue

    LOGICAL Lstart
    SAVE Lstart
    DATA Lstart/.true./

    if(pname == 'RESET_PARAMS_') then
        if(Lstart .OR. pvalue > 0.5) then
            vMissVal = 1.0E21   ! for missing values
            VarRepl = vMissVal ! for missing VARIABLES
            ValRange(1) = -vMissVal / 10.
            ValRange(2) = vMissVal / 10.
            iVarWarn = 2
            vReadMin = 0.0
            vReadMax = 0.0
            ireadchk = 0
            Lstart = .false.
        endif

    else if(pname == 'NOVAR_REPLACE') then
        VarRepl = pvalue

    else if(pname == 'NOVAR_WARNING') then
        iVarWarn = NINT(pvalue)

    else if(pname == 'VALID_RANGE_MIN') then
        ValRange(1) = pvalue

    else if(pname == 'VALID_RANGE_MAX') then
        ValRange(2) = pvalue

    else if(pname == 'READOVER_WARN') then
        vReadMin = -pvalue
        vReadMax = pvalue
        ireadchk = 1

    else if(pname == 'READ_MIN_WARN') then
        vReadMin = pvalue
        ireadchk = 1

    else if(pname == 'READ_MAX_WARN') then
        vReadMax = pvalue
        ireadchk = 1

    else
        write(*, *) 'UNparam (libUN) Error: '
        write(*, *) '  parameter undefined:', pname

    endif

END

!**  +-------------------------+-----------------------------------------+
subroutine UNversion(UNver, NCDFver)
    !**  +-------------------------+-----------------------------------------+
    use libUN_mod
    implicit none

    character * 80 UNver, NCDFver

    UNver = '2005.03.31'
    NCDFver = NF_INQ_LIBVERS()

END

!**  +-------------------------------------------------------------------+
FUNCTION VARSIZE(CHAvar)
    !**  +-------------------------------------------------------------------+
    implicit none
    integer maxcha, iz, VARSIZE
    parameter(maxcha=512)
    character * (*) CHAvar
    character * (maxcha) CHAtmp

    write(CHAtmp, '(A)') CHAvar
    iz = 0
    do while((CHAtmp(iz + 1:iz + 3) /= '   ') .and. (iz + 3 <= maxcha))
        iz = iz + 1
    enddo
    VARSIZE = iz

    RETURN
END

!**  +-------------------------------------------------------------------+
subroutine HANDLE_ERR(LOCATION, STATUS)
    !**  +-------------------------------------------------------------------+
    use libUN_mod
    implicit none

    character * (*) LOCATION
    integer STATUS
    if(STATUS /= NF_NOERR) then
        write(*, *) 'IN ROUTINE ', LOCATION
        write(*, *) NF_STRERROR(STATUS)
        STOP 'Stopped'
    endif
END

! UN library: history of fixed bugs and updates.
! ----------------------------------------------
!
!                        961206 - UNgtime, trouble at end-of-file
!                        961218 - - all -, display 'artificial' errors
!                        970318 -   again, display 'artificial' errors
!                        971028 - (3 sub),'syntax'error on Cray computer
!                        971105 - Allowed variable "imap(1)", =8 for Cray
!                        980705 - "single element" extension to UNwrite.
!                        980709 - bug fixes (start) in UNwrite & UNlwrite
!                                 ("DATA" statement incorrectly used).
!                        980825 - Changed default "stride" to 1 for v3.x
!                        981222 - bug fix: allow UNwrite for unlim dims.
!                                 note that this should be tested.
!                        990110 - Added "UNgindx" = general. of UNgtime
!                               - Removed all "DATA" and all "//" in write
!                                 (the later should improve compatibility)
!                        990128 - UNwrite: added a "no warning" option.
!                        990323 - UNwrite: added 1D+time capability.
!                        990807 - UNwrite: added 3D-notime capability.
!  -----------------------------------------------------------------------
!                        000404 - Major upgrade: compatibility with
!                                 NetCDF v3.4
!                               - NOTE: Types other than REAL may be
!                                 accepted in UNread, but not tested
!  -----------------------------------------------------------------------
!                        000614 - Bug fixes: uninitialised error count
!                                 in UNwcatt, bug in UNclose.
!                        000620 - Bug fix: UNropen(args. of get title fn)
!                        000713 - Bug fix: UNgtime (missing arg in a call)
!                                 (last tree caused by 000404 upgrade)
!  -----------------------------------------------------------------------
!                        000928 - UNlwrite: added 2D+time capability.
!                        001008 - All: character*(*) declaration for units
!                                 and longer strings for intern. variables
!                        010417 - UNread: added var not found info
!                                 UNropen: added file not found info
!                        010715 - UNwrite + UNlwrite:
!                                   fixed bug / unlimited time dim
!                        0107xx - UNwrite:
!                                   missing values -> not in "range"
!                        020130 - All:
!                                  .removed obsolete warnings about
!                                   double precision in files.
!                                  .added a version (libUN_dbl) with
!                                   REAL*8 as arguments - but still
!                                   creates  REAL(kind=4) in files.
!                        020526 - Added UNparam function,
!                                 which provide optional features such
!                                 as missing variable behavior control
!                        020808 - Very simple fix for underflows while
!                                 reading some files; must use -fpe1
!                                 Fixed a bug -> out of range msg
!                        030121 - Enabled some non-standard NetCDF files
!                                 (missing units...) -> new warnings
!                                 rather then program stop.
!                        030215 - Added UNfindx for non-monotonic data
!                        030215 - Removed warning related to UNLIM dims
!                        030311 - Added VALID_RANGE attribute (option)
!                                 (if set, the range is accounted for
!                                 in the min/max set while writing vars)
!                        040902 - Improvements to "valid_range" attribute
!                               - Added attribute "positive=down"
!                                 if units are sigma or sigma_level
!                        050331 - Added "user friendly" interfaces
