#include "MAR_pp.def"
subroutine LBCnud_atm(f__LBC, iv_nua, kd_nua)
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS LBC                                   Fri  4-12-2009  MAR |
    ! |   subroutine LBCnud_atm computes the Lateral Boundary Conditions       |
    ! |              following the Davies (1976) scheme                        |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT / OUTPUT : f__LBC, i.e. uairDY, vairDY, qvDY, pktaDY, pstDYn   |
    ! |   ^^^^^^^^     for iv_nua =          1,      2,    3,      4,      5   |
    ! |                    f_LBC0 reevalued on a 5-points width boundary zone  |
    ! |                                                                        |
    ! |   INPUT:   iv_nua: Index of the Variable to relax to Outer Conditions  |
    ! |   ^^^^^^   kd_nua: Maximum Value of the k (vertical) Index             |
    ! |                                                                        |
    ! |   INPUT (via common block)                                             |
    ! |   ^^^^^    reaLBC: Input INI: Previous Dyn.Simulation (MAR .or. GCM)   |
    ! |            rxfact: Lateral Sponge Coefficient         (A89)            |
    ! |            rxLB,ryLB: Nudging Coefficient                              |
    ! |            Independant Term  used in the Implicit Scheme               |
    ! |                                                                        |
    ! |   REFER. : Davies, QJRMS 102, pp.405--418, 1976  (relation 11 p.409)   |
    ! |   ^^^^^^^^                                                             |
    ! |                                                                        |
    ! |   INPUT  : Nudging Coefficient rxLB and ryLB                           |
    ! |   ^^^^^^^^ Inverted Matrices used in the Implicit Scheme               |
    ! |                        wixgLB (zone x <<), wixdLB (zone x >>)          |
    ! |                        wiyiLB (zone y <<), wiysLB (zone y >>)          |
    ! |            Independant Term  used in the Implicit Scheme               |
    ! |            Variable v: tixgLB (zone x <<), tixdLB (zone x >>)          |
    ! |            Variable u: tiyiLB (zone y <<), tiysLB (zone y >>)          |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_lb
    use mar_wk

    implicit none

    integer i, j, k, m
    real f__LBC(mx, my, mz)
    integer kd_nua, iv_nua

    ! +--Local  Variables
    ! +  ================

#ifdef OG
    logical relaxg
#endif

    integer il, ic, jl, jc
    real sx(mx, mz)
    real sy(my, mz)
    real txg(2:n7, mz), txd(mx - n6:mx1, mz)
    real tyi(2:n7, mz), tys(my - n6:my1, mz)

    real fmagng(6), fmagnd(6), fmagnb(6), fmagnh(6)

    data fmagng/1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0/
    data fmagnd/1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0/
    data fmagnb/1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0/
    data fmagnh/1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0, 1.0e0/
    ! +...     fmagnX:magnification factor (=>nudging selectively modified)

#ifdef OG
    ! relaxg=.false.==> NO nudging  at the left boundary.
    data relaxg/.false./
#endif

    ! x Boundaries
    ! ============
!$OMP PARALLEL do default(shared) private(i,j,k,il,ic)
    do k = 1, kd_nua
        if(mmx > 1) then
            if(iv_nua == 2) then
                ! +
                do j = jp11, my1
                    ! +--`Left' Boundary (x <<)
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~
                    do il = ip11, n7
                        ! do k = 1,kd_nua
                        txg(il, k) = 0.d0
                        ! end do
                        do ic = ip11, n7
                            ! do k = 1,kd_nua
                            txg(il, k) = txg(il, k) &
                                         + (tixgLB(ic, j, k) &
                                            + f__LBC(ic, j, k)) * wixgLB(il, ic)
                            ! end do
                        enddo
                    enddo
                    do i = ip11, n7
                        ! do k = 1,kd_nua
                        f__LBC(i, j, k) = txg(i, k)
                        ! end do
                    enddo

                    ! +--`Right' Boundary (x >>)
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                    do il = mx - n6, mx1
                        ! do k = 1,kd_nua
                        txd(il, k) = 0.d0
                        ! end do
                        do ic = mx - n6, mx1
                            ! do k = 1,kd_nua
                            txd(il, k) = txd(il, k) + &
                                         (tixdLB(ic, j, k) &
                                          + f__LBC(ic, j, k)) * wixdLB(il, ic)
                            ! end do
                        enddo
                    enddo
                    do i = mx - n6mxLB, mx1
                        ! do k = 1,kd_nua
                        f__LBC(i, j, k) = txd(i, k)
                        ! end do
                    enddo
                enddo

            else
                do j = jp11, my1
#ifdef OG
                    if(relaxg) then
#endif
                        do i = ip11, n6 - 1
                            ! do k = 1,kd_nua
                            f__LBC(i, j, k) = (f__LBC(i, j, k) &
                                               + fmagng(iv_nua) * rxLB(i) * vaxgLB(i, j, k, iv_nua)) &
                                              / (1.d0 + fmagng(iv_nua) * rxLB(i))
                            ! end do
                        enddo
#ifdef OG
                    endif
#endif
                    do i = mx - n6 + 2, mx1
                        ! do k = 1,kd_nua
                        f__LBC(i, j, k) = (f__LBC(i, j, k) &
                                           + fmagnd(iv_nua) * rxLB(i) * vaxdLB(i, j, k, iv_nua)) &
                                          / (1.d0 + fmagnd(iv_nua) * rxLB(i))
                        ! end do
                    enddo
                enddo
            endif

            ! Zero Gradient at y LBC if fmagng,d = 0 / otherwise prescribed LBC
            ! -----------------------------------------------------------------
            ! do k=1,kd_nua
            do j = 1, my
                f__LBC(1, j, k) = &
                    (1.-fmagng(iv_nua)) * f__LBC(ip11, j, k) &
                    + fmagng(iv_nua) * vaxgLB(1, j, k, iv_nua)
                f__LBC(mx, j, k) = &
                    (1.-fmagnd(iv_nua)) * f__LBC(mx1, j, k) &
                    + fmagnd(iv_nua) * vaxdLB(mx, j, k, iv_nua)
            enddo
            ! end do

            ! Nudging
            ! -------
            do i = ip11, n6 - 1
                ! do k=   1,kd_nua
                do j = jp11, my1
                    WKxyz1(i, j, k) = f__LBC(i, j, k) + rxfact * rxLB(i) &
                                      * (f__LBC(i + 1, j, k) + f__LBC(i - 1, j, k) &
                                         - f__LBC(i, j, k) - f__LBC(i, j, k) &
                                         - vaxgLB(i + 1, j, k, iv_nua) - vaxgLB(i - 1, j, k, iv_nua) &
                                         + vaxgLB(i, j, k, iv_nua) + vaxgLB(i, j, k, iv_nua))
                enddo
                ! end do
            enddo
            do i = ip11, n6 - 1
                ! do k=   1,kd_nua
                do j = jp11, my1
                    f__LBC(i, j, k) = WKxyz1(i, j, k)
                enddo
                ! end do
            enddo

            do i = mx - n6 + 2, mx1
                ! do k=   1   ,kd_nua
                do j = jp11, my1
                    WKxyz1(i, j, k) = f__LBC(i, j, k) + rxfact * rxLB(i) &
                                      * (f__LBC(i + 1, j, k) + f__LBC(i - 1, j, k) &
                                         - f__LBC(i, j, k) - f__LBC(i, j, k) &
                                         - vaxdLB(i + 1, j, k, iv_nua) - vaxdLB(i - 1, j, k, iv_nua) &
                                         + vaxdLB(i, j, k, iv_nua) + vaxdLB(i, j, k, iv_nua))
                enddo
                ! end do
            enddo
            do i = mx - n6 + 2, mx1
                ! do k=  1    ,kd_nua
                do j = jp11, my1
                    f__LBC(i, j, k) = WKxyz1(i, j, k)
                enddo
                ! end do
            enddo

            ! Zero Gradient at y LBC if fmagng,d = 0 / otherwise prescribed LBC
            ! -----------------------------------------------------------------
            ! do k=1,kd_nua
            do j = jp11, my1
                f__LBC(1, j, k) = &
                    (1.-fmagng(iv_nua)) * f__LBC(ip11, j, k) &
                    + fmagng(iv_nua) * vaxgLB(1, j, k, iv_nua)
                f__LBC(mx, j, k) = &
                    (1.-fmagnd(iv_nua)) * f__LBC(mx1, j, k) &
                    + fmagnd(iv_nua) * vaxdLB(mx, j, k, iv_nua)
            enddo
            ! end do
        endif

        !    y Boundaries
        !    ============
        if(mmy > 1) then
            if(iv_nua == 1) then
                do i = 1, mx
                    ! +--`Bottom' Boundary (y <<)
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~
                    do jl = jp11, n7
                        ! do k =    1,kd_nua
                        tyi(jl, k) = 0.d0
                        ! end do
                        do jc = jp11, n7
                            ! do k =    1,kd_nua
                            tyi(jl, k) = tyi(jl, k) &
                                         + (tiyiLB(i, jc, k) &
                                            + f__LBC(i, jc, k)) * wiyiLB(jl, jc)
                            ! end do
                        enddo
                    enddo
                    do j = jp11, n7
                        ! do k = 1,kd_nua
                        f__LBC(i, j, k) = tyi(j, k)
                        ! end do
                    enddo
                    ! +--`Top' Boundary (y >>)
                    ! +  ~~~~~~~~~~~~~~~~~~~~~
                    do jl = my - n6, my - 1
                        ! do k = 1,kd_nua
                        tys(jl, k) = 0.d0
                        ! end do
                        do jc = my - n6, my - 1
                            ! do k = 1,kd_nua
                            tys(jl, k) = tys(jl, k) &
                                         + (tiysLB(i, jc, k) &
                                            + f__LBC(i, jc, k)) * wiysLB(jl, jc)
                            ! end do
                        enddo
                    enddo
                    do j = my - n6, my - 1
                        ! do k = 1,kd_nua
                        f__LBC(i, j, k) = tys(j, k)
                        ! end do
                    enddo
                enddo
            else
                do j = jp11, n6 - 1
                    !   do k=  1 ,kd_nua
                    do i = 1, mx
                        f__LBC(i, j, k) = (f__LBC(i, j, k) &
                                           + fmagnb(iv_nua) * ryLB(j) * vayiLB(i, j, k, iv_nua)) &
                                          / (1.d0 + fmagnb(iv_nua) * ryLB(j))
                    enddo
                    ! end do
                enddo
                do j = my - n6 + 2, my1
                    !   do k=  1    ,kd_nua
                    do i = 1, mx
                        f__LBC(i, j, k) = (f__LBC(i, j, k) &
                                           + fmagnh(iv_nua) * ryLB(j) * vaysLB(i, j, k, iv_nua)) &
                                          / (1.d0 + fmagnh(iv_nua) * ryLB(j))
                    enddo
                    ! end do
                enddo
            endif

            ! Zero Gradient at y LBC if fmagnb,h = 0 / otherwise prescribed LBC
            ! -----------------------------------------------------------------
            ! do k=1,kd_nua
            do i = 1, mx
                f__LBC(i, 1, k) = &
                    (1.-fmagnb(iv_nua)) * f__LBC(i, jp11, k) &
                    + fmagnb(iv_nua) * vayiLB(i, 1, k, iv_nua)
                f__LBC(i, my, k) = &
                    (1.-fmagnh(iv_nua)) * f__LBC(i, my1, k) &
                    + fmagnh(iv_nua) * vaysLB(i, my, k, iv_nua)
            enddo
            ! end do

            !    Nudging
            !    -------
            do j = 2, n6 - 1
                !   do k=1   ,kd_nua
                do i = ip11, mx1
                    WKxyz2(i, j, k) = f__LBC(i, j, k) + rxfact * ryLB(j) &
                                      * (f__LBC(i, j + 1, k) + f__LBC(i, j - 1, k) &
                                         - f__LBC(i, j, k) - f__LBC(i, j, k) &
                                         - vayiLB(i, j + 1, k, iv_nua) - vayiLB(i, j - 1, k, iv_nua) &
                                         + vayiLB(i, j, k, iv_nua) + vayiLB(i, j, k, iv_nua))
                enddo
                ! end do
            enddo
            do j = 2, n6 - 1
                !   do k=1   ,kd_nua
                do i = ip11, mx1
                    f__LBC(i, j, k) = WKxyz2(i, j, k)
                enddo
                ! end do
            enddo

            do j = my - n6 + 2, my1
                !   do k=1      ,kd_nua
                do i = ip11, mx1
                    WKxyz2(i, j, k) = f__LBC(i, j, k) + rxfact * ryLB(j) &
                                      * (f__LBC(i, j + 1, k) + f__LBC(i, j - 1, k) &
                                         - f__LBC(i, j, k) - f__LBC(i, j, k) &
                                         - vaysLB(i, j + 1, k, iv_nua) - vaysLB(i, j - 1, k, iv_nua) &
                                         + vaysLB(i, j, k, iv_nua) + vaysLB(i, j, k, iv_nua))
                enddo
                ! end do
            enddo
            do j = my - n6 + 2, my1
                !   do k=1      ,kd_nua
                do i = ip11, mx1
                    f__LBC(i, j, k) = WKxyz2(i, j, k)
                enddo
                ! end do
            enddo

            ! Zero Gradient at y LBC if fmagnb,h = 0 / otherwise prescribed LBC
            ! -----------------------------------------------------------------
            ! do k=1,kd_nua
            do i = ip11, mx1
                f__LBC(i, 1, k) = &
                    (1.-fmagnb(iv_nua)) * f__LBC(i, jp11, k) &
                    + fmagnb(iv_nua) * vayiLB(i, 1, k, iv_nua)
                f__LBC(i, my, k) = &
                    (1.-fmagnh(iv_nua)) * f__LBC(i, my1, k) &
                    + fmagnh(iv_nua) * vaysLB(i, my, k, iv_nua)
            enddo
            ! end do

#ifdef OB
            do k = 1, kd_nua
                f__LBC(1, 1, k) = (f__LBC(1, jp11, k) &
                                   + f__LBC(ip11, 1, k)) * 0.5d0
                f__LBC(mx, 1, k) = (f__LBC(mx, jp11, k) &
                                    + f__LBC(mx1, 1, k)) * 0.5d0
                f__LBC(1, my, k) = (f__LBC(1, my1, k) &
                                    + f__LBC(ip11, my, k)) * 0.5d0
                f__LBC(mx, my, k) = (f__LBC(mx, my1, k) &
                                     + f__LBC(mx1, my, k)) * 0.5d0
            enddo
#endif
        endif

        ! Work Arrays Reset
        ! =================
        ! do k=1,mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
            enddo
        enddo
    enddo
!$OMP END PARALLEL DO

    return
endsubroutine LBCnud_atm
