#include "MAR_pp.def"
subroutine interp_subpix(var, var_int, opt, inf, sup, grad_MS)
    !     + ---------------------------------------------------------- +
    !     / subroutine interp_subpix interpolates MAR variables        /
    !     / (inputs of SISVAT) on the subgrid.                         /
    !     / Calculates a local gradient for the variable and adjusts   /
    !     / the subgrid values using the elevation difference between  /
    !     / the pixel and its subpixels.                               /
    !     + ---------------------------------------------------------- +
    !     /                                                            /
    !     / INPUT :  var : MAR variable to be interpolated                   /
    !     / ^^^^^^^                                                           /
    !     /                                                            /
    !     / OUTPUT : var_int : variable interpolated on the subgrid    /
    !     / ^^^^^^^  grad_MS : mean local gradient of var              /
    !     /                           (with spat. and temp. smoothing)        /
    !     + ---------------------------------------------------------- +
    !     / /!\ opt = options                                           /
    !     / 1 = only interpolation (eg: for temperature)                   /
    !     / 2 = interpolated value can not be negative                 /
    !     /     (eg: humidity, precipitation ...)                      /
    !     /                                                            /
    !     / inf, sup = lower and upper limits of the gradient          /
    !     /                                                            /
    !     / dSH_min = min value of elevation difference between        /
    !     /           2 pixels to compute the local gradient = 100 m   /
    !     /                                                            /
    !     /                                                            /
    !     /                                 Charlotte Lang        13/03/2015 /
    !     + ---------------------------------------------------------- +

    use mardim
    use marctr
    use mar_ge
    use mar_sl
    use margrd
    use marssn

    implicit none

    integer i, j, k, m, n
    real, parameter :: dSH_min = 100

    real var(mx, my), var_int(mx, my, mw)
    real delta_sh, delta_var, grad_old(mx, my)
    real grad_var, grad_M(mx, my), grad_MS(mx, my)

    ! + grad_var = gradient between the pixel and one of its 8 surrounding pixels
    ! + grad_M   = mean value of the local gradient. Average of the 8 surrounding gr
    ! + grad_MS  = mean gradient after spatial and temporal smoothing

    real w, q, qi(-1:1, -1:1)
    real inf, sup

    integer opt
    real sum_var_int(mx, my), fact(mx, my)

    do i = 1, mx; do j = 1, my
            do k = 1, nsx
                var_int(i, j, k) = var(i, j)
                if(isnan(var(i, j))) then
                    write(6, 399) iyrrGE, mmarGE, jdarGE, &
                        jhurGE, i, j
399                 format('/!\ CL WARNING: VAR is NaN ', i4, '/', i2, '/', &
                           i2, i3, 'h, (i,j)=', i4, i4)
                    stop
                endif
            enddo
        enddo; 
    enddo

    do i = 2, mx - 1; do j = 2, my - 1

            if(itexpe == 0) grad_MS(i, j) = 0.0
            grad_old(i, j) = grad_MS(i, j)

            ! + grad_old = value of the mean local gradient at the previous time step. Used

            ! + *******************************************
            ! + ***   Computing of the local gradient   ***
            ! + *******************************************

            w = 0.0
            grad_M(i, j) = 0.0

            if(isolSL(i, j) >= 3) then ! Gradient is computed only for land pixels. gr

                ! + Computation of the gradient between each pixel and its 8 surroung pixels (gr
                do m = -1, 1; do n = -1, 1

                        grad_var = 0.0

                        if(isolSL(i + m, j + n) >= 3) then
                            delta_sh = sh(i + m, j + n) - sh(i, j)
                            if(abs(delta_sh) > dSH_min) then
                                ! +  If no minimum elevation difference for the computation of the gradient, del
                                ! +  to go through the if loop --> If dSH_min = 0.0, decomment next line.
                                !         if (delta_sh .neq. 0.0) then
                                delta_var = var(i + m, j + n) - var(i, j)
                                grad_var = delta_var / delta_sh
                                w = w + 1.0
                            endif
                        endif
                        ! + Computation of the mean local gradient
                        grad_M(i, j) = grad_M(i, j) + grad_var

                    enddo; 
                enddo

                if(w /= 0.0) then                 ! w = # of pixels among the 8 surro
                    grad_M(i, j) = grad_M(i, j) / w         ! If w = 0, no pixel among the 8 surrou
                endif

                if(isnan(grad_M(i, j))) then
                    write(6, 400) iyrrGE, mmarGE, jdarGE, &
                        jhurGE, i, j
400                 format('/!\ CL WARNING: grad is NaN ', i4, '/', i2, '/', &
                           i2, i3, 'h, (i,j)=', i4, i4)
                    stop
                endif

            endif
        enddo; 
    enddo

    !+ ***   Smoothing of the mean local gradient   ***
    do i = 2, mx - 1; do j = 2, my - 1

            q = 0.0
            grad_MS(i, j) = 0.0

            if(isolSL(i, j) >= 3) then

                ! + ***   Spatial smoothing of the gradient   ***
                do m = -1, 1; do n = -1, 1
                        if(isolSL(i + m, j + n) >= 3) then
                            qi(m, n) = 1.0
                            if(m == 0 .or. n == 0) qi(m, n) = 2.0
                            if(m == 0 .and. n == 0) qi(m, n) = 4.0
                            q = q + qi(m, n)
                            grad_MS(i, j) = grad_MS(i, j) + qi(m, n) * grad_M(i + m, j + n)
                        endif

                    enddo; 
                enddo

                grad_MS(i, j) = grad_MS(i, j) / q

                ! + ***   Temporal smoothing of the gradient   ***

                if(itexpe /= 0) then
                    grad_MS(i, j) = 0.75 * grad_MS(i, j) + 0.25 * grad_old(i, j)
                endif

                ! + ***   Lower and upper limits of the mean gradient value   ***
                grad_MS(i, j) = max(inf, min(sup, grad_MS(i, j)))

                ! + ********************************************************
                ! + ***   Interpolation of the variable on the subgrid   ***
                ! + ********************************************************
                do k = 1, nsx - 1
                    var_int(i, j, k) = var(i, j) &
                                       + grad_MS(i, j) * (sh_int(i, j, k) - sh(i, j))
                enddo

                var_int(i, j, nsx) = var(i, j)

            endif
        enddo; 
    enddo

    ! + *******************
    ! + ***   Options   ***
    ! + *******************

    ! + Opt = 1 --> Nothing more than the interpolation is done
    ! + Opt = 2 --> Interpolated value has to be positive

    if(opt == 2) then
        do i = 1, mx; do j = 1, my
                do k = 1, nsx
                    var_int(i, j, k) = max(0.0, var_int(i, j, k))
                enddo
            enddo; 
        enddo
    endif

    return
endsubroutine interp_subpix
