#include "MAR_pp.def"
subroutine inisnd
    ! +-----------------------------------------------------------------------+
    ! | MAR INPUT      ATMOS                                   19-02-2004  MAR |
    ! |   subroutine inisnd includes Large Scale Conditions from a Sounding    |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |    INPUT (via common block)                                            |
    ! |    ^^^^^^ itexpe: Experiment Iteration Index                           |
    ! |           iterun: Run        Iteration Index (n Run for 1 Experiment)  |
    ! |           log_1D=(0,1) (before,after)  Boundary Layer Initialization   |
    ! |           potvor:"Potential Vorticity conserved" Initialization Switch |
    ! |           potvor=.T. => Potential Vorticity (PV) Conservation          |
    ! |                         Constraint is used (in 2-D Mode only)          |
    ! |           conmas:"Mass                conserved" Initialization Switch |
    ! |           conmas=.T. => Mass Conservation is used                      |
    ! |                                                                        |
    ! |    INPUT: Sounding(s) / File MARSND.dat                                |
    ! |    ^^^^^^                                                              |
    ! |                                                                        |
    ! |   OUTPUT (via common block)                                            |
    ! |   ^^^^^^  MESOSCALE VARIABLES                                          |
    ! |           ^^^^^^^^^^^^^^^^^^^                                          |
    ! |           nSND             : Sounding Number                           |
    ! |           pstDY (mx,my)    : Initial Model Thickness             [kPa] |
    ! |           pstDY1(mx,my)    : Initial Model Thickness -FIXED-     [kPa] |
    ! |           ugeoDY(mx,my,mz) : Initial Geo. Wind (x-Direction)     [m/s] |
    ! |           vgeoDY(mx,my,mz) : Initial Geo. Wind (y-Direction)     [m/s] |
    ! |           uairDY(mx,my,mz) : Initial      Wind (x-Direction)     [m/s] |
    ! |           vairDY(mx,my,mz) : Initial      Wind (y-Direction)     [m/s] |
    ! |           tairDY(mx,my,mz) : Initial                   Temperature [K] |
    ! |           TairSL(mx,my)    : Initial      Surface Air  Temperature [K] |
    ! |                             (Sounding Extrapolated to the Surface;     |
    ! |                              - dtagSL                            )     |
    ! |           pktaDY(mx,my,mzz): Initial Reduced Potential Temperature     |
    ! |             qvDY(mx,my,mz) : Initial Specific Humididity       [kg/kg] |
    ! |           qvapSL(mx,my)    : Initial Specific Humididity       [kg/kg] |
    ! |                             (Sounding Extrapolated to the Surface)     |
    ! |                                                                        |
    ! |   OUTPUT (via common block)                                            |
    ! |   ^^^^^^  LARGE SCALE VARIABLES (BOUNDARIES and DOMAIN AVERAGE)        |
    ! |           ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^        |
    ! |        1. LBC                                                          |
    ! |           vaxgLB  (mx,my,mz,n):Large Scale Values of the Variables     |
    ! |                             (n=1,2,3,4,5) <=> (u,v,qv,pktaDY,pstDY),   |
    ! |                                                               for x << |
    ! |           vaxdLB : idem,                                          x >> |
    ! |           vayiLB : idem,                                          y << |
    ! |           vaysLB : idem,                                          y >> |
    ! |           zetaD  : Potential Vorticity  (CAUTION: Time Independant)    |
    ! |        2. UBC                                                          |
    ! |           uairUB, vairUB, pktaUB                                       |
    ! |                                                                        |
    ! |   METHOD: Vertical  Interpolation on Sigma Levels  is first performed  |
    ! |   ^^^^^^^ For two Soundings nearest in Time (before and after runtime) |
    ! |           Then Time Interpolation                                      |
    ! |           This order was preferred to the reverse because it allows    |
    ! |           to use Soundings having a different vertical Discretization  |
    ! |                                                                        |
    ! | # OPTIONS: _SC  Supersaturation not allowed (do NOT USE for SCu cloud) |
    ! | # ^^^^^^^^ #WR  Additional Output                                      |
    ! |                                                                        |
    ! |   CAUTION: inisnd may be used only for Large Scale Thermodynamics      |
    ! |   ^^^^^^^^ independant of x and y (i.e.for not too large model domain) |
    ! |            zeSND fixed at its initial value whenever Sounding varies   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use marsnd
    use mar_dy
    use mar_lb
    use mar_ub
    use mar_sl
    use mar_wk
    use mar_io
#ifdef NH
    use mar_nh
#endif
#ifdef PV
    use mar_PV
#endif

    implicit none

    ! +--Local Variables
    ! +  ===============
#ifdef CS
    logical consnd
#endif
    integer i, j, k, m
    integer logcon
    integer lbcfix, lsf, lvg
    integer intpol, jhsnd1, jhsnd2, i0snd, j0snd, ksnd, msnd
    integer ishmin, jshmin, ii, jj, inew, jnew
    integer iyrnew, mmanew, jdanew, jhunew
    real tsurf0, qsurf0, gradti, graddt, gra, timar, timarn, timmar
    real shmin, distmn, distij, ta_inv, pksh, gsnd, ddnew
    real zetaD, dul, ttij_1, ttij_2
    real qsat0D, acap, dem1

    real pksnd(0:40, 2)
    ! +...          pksnd: pSND      ** (R/Cp),
    ! +             where (pSND/100) ** (R/Cp) : Exner Function

    real tmsnd(0:40, 2)
    ! +...          tmsnd: Potential Temperature
    ! +                    averaged over the layer between ksnd-1/2 and ksnd+1/2

    real pint(2)
    real tint(2), qint(2)

    real ttij(mz, 2), qvij(mz, 2)
    real ulij(mz, 2), vlij(mz, 2)

    real dpt(mz), dqa(mz)
    real dug(mz), dvg(mz)

    real fftt(3), ddtt(3)

    ! +--DATA
    ! +  ====

    data lbcfix/1/
#ifdef CS
    data consnd/.true./
#endif

    data dem1/1.0e-1/

    data lsf/0/
#ifdef OB
    openLB = .true.
    lbcfix = 0
#endif

    lvg = 0
    ! +...lvg:     is set to 1 if |Vg(sounding)| .ne. 0 anywhere

    acap = 1.0 / cap
    ! +...acap:    Inverse of R / Cp

    ttij_1 = 0.0
    ttij_2 = 0.0

    ! +--Time Parameters
    ! +  ===============

    if(iterun == 0) then
        tiSND1 = -99999999.99
        tiSND2 = -99999999.99
        iSND = imez
        jSND = jmez
        loSND = 1
    endif

    timar = ((iyrrGE * 365 + iyrrGE * 0.25 &
              + njyrGE(mmarGE) + jdarGE) * 24 &
             + jhurGE + itizGE(iSND, jSND)) * 36.d2 &
            + minuGE * 6.d1 + jsecGE
    timarn = timar

    ! +--Interpolation Parameter
    ! +  =======================

    if(tiSND1 >= tiSND2) &
        intpol = 0
    ! +...  intpol = 0 ==> No Interpolation between two soundings
    ! +              1 ==>    Interpolation between two soundings

    ! +  +++++++++++++++++++++++++++++++
    ! +--Search of the relevant Sounding
    ! +  +++++++++++++++++++++++++++++++

    ! +--Main Dependant Variables
    ! +  ========================

    tSND(:, 1) = (/277.0, 272.2, 268.7, 265.2, 261.7, &
                   255.7, 249.7, 243.7, 237.7, 231.7, &
                   225.7, 219.7, 219.2, 218.7, 218.2, &
                   217.7, 217.2, 216.7, 216.2, 215.7, &
                   215.2, 215.2, 215.2, 215.2, 215.2, &
                   215.2, 215.2, 217.4, 227.8, 243.2, &
                   258.5, 265.7, 265.7, 265.7, 265.7, &
                   265.7, 265.7, 265.7, 265.7, 265.7, 265.7/)
    tSND(:, 2) = (/277.0, 272.2, 268.7, 265.2, 261.7, &
                   255.7, 249.7, 243.7, 237.7, 231.7, &
                   225.7, 219.7, 219.2, 218.7, 218.2, &
                   217.7, 217.2, 216.7, 216.2, 215.7, &
                   215.2, 215.2, 215.2, 215.2, 215.2, &
                   215.2, 215.2, 217.4, 227.8, 243.2, &
                   258.5, 265.7, 265.7, 265.7, 265.7, &
                   265.7, 265.7, 265.7, 265.7, 265.7, 265.7/)

    qSND(:, 1) = (/27e-2, .27e-2, .21e-2, .17e-2, .13e-2, &
                   .80e-3, .51e-3, 32e-3, .14e-3, .67e-4, &
                   .35e-4, .18e-4, .20e-4, .20e-4, 70e-5, &
                   .45e-5, .40e-5, .39e-5, .40e-5, .42e-5, &
                   .48e-5, 51e-5, .68e-5, .81e-5, .10e-4, &
                   .13e-4, .17e-4, .20e-4, 14e-4, .10e-4, &
                   .14e-4, .69e-5, .70e-5, .72e-5, .74e-5, &
                   75e-5, .77e-5, .79e-5, .81e-5, .83e-5, .86e-5/)
    qSND(:, 2) = (/27e-2, .27e-2, .21e-2, .17e-2, .13e-2, &
                   .80e-3, .51e-3, 32e-3, .14e-3, .67e-4, &
                   .35e-4, .18e-4, .20e-4, .20e-4, 70e-5, &
                   .45e-5, .40e-5, .39e-5, .40e-5, .42e-5, &
                   .48e-5, 51e-5, .68e-5, .81e-5, .10e-4, &
                   .13e-4, .17e-4, .20e-4, 14e-4, .10e-4, &
                   .14e-4, .69e-5, .70e-5, .72e-5, .74e-5, &
                   75e-5, .77e-5, .79e-5, .81e-5, .83e-5, .86e-5/)

    zSND(:, 1) = (/-143., 0., 1001., 1993., 2992., &
                   3993., 4994., 5983., 6978., 7988., &
                   8984., 9970., 10968., 11975., 12966., &
                   13949., 14945., 15932., 16950., 17900., &
                   18914., 19884., 20894., 21933., 22985., &
                   23799., 24990., 29928., 35068., 38589., &
                   46673., 49408., 49583., 49761., 49948., &
                   50132., 50324., 50521., 50723., 50930., 51143./)
    zSND(:, 2) = (/-143., 0., 1001., 1993., 2992., &
                   3993., 4994., 5983., 6978., 7988., &
                   8984., 9970., 10968., 11975., 12966., &
                   13949., 14945., 15932., 16950., 17900., &
                   18914., 19884., 20894., 21933., 22985., &
                   23799., 24990., 29928., 35068., 38589., &
                   46673., 49408., 49583., 49761., 49948., &
                   50132., 50324., 50521., 50723., 50930., 51143./)

    pSND(:, 1) = (/1036., 1018., 897., 790., 694., &
                   608., 531., 463., 402., 347., &
                   299., 257., 220., 188., 161., &
                   138., 118., 101., 86., 74., &
                   63., 54., 46., 39., 33., &
                   29., 24., 11., 5., 3., &
                   1., 0.7, 0.685, 0.669, 0.654, &
                   0.637, 0.622, 0.606, 0.591, 0.576, 0.560/)
    pSND(:, 2) = (/1036., 1018., 897., 790., 694., &
                   608., 531., 463., 402., 347., &
                   299., 257., 220., 188., 161., &
                   138., 118., 101., 86., 74., &
                   63., 54., 46., 39., 33., &
                   29., 24., 11., 5., 3., &
                   1., 0.7, 0.685, 0.669, 0.654, &
                   0.637, 0.622, 0.606, 0.591, 0.576, 0.560/)

    ! +   ------------------------------------
    if(timar > tiSND2 .and. loSND == 1) then
        ! +   ------------------------------------

        open(unit=2, status='old', file='MARsnd.dat')
        rewind 2
        read(2, 202) iyrSND, mmaSND, jdaSND, jhuSND
202     format(4i4, f4.0)
        read(2, 203)
203     format(1x)
        read(2, 202) iSND, jSND
        read(2, 203)
        read(2, 203)
        read(2, 204)(tSND(ksnd, 1), qSND(ksnd, 1), &
                     zSND(ksnd, 1), pSND(ksnd, 1), &
                     fSND(ksnd, 1), dSND(ksnd, 1), ksnd=40, 0, -1)
204     format((6d13.6))
        read(2, 204) zeSND(1)
        read(2, 202) loSND

        ! +- Time Parameters
        ! +  ~~~~~~~~~~~~~~~
        jhsnd1 = (iyrSND * 365 + iyrSND / 4 &
                  + njyrGE(mmaSND) + jdaSND) * 24 + jhuSND + itizGE(iSND, jSND)
        tiSND1 = ((iyrSND * 365 + iyrSND / 4 &
                   + njyrGE(mmaSND) + jdaSND) * 24 + jhuSND + itizGE(iSND, jSND)) &
                 * 3.6d3

        ! +     ~~~~~~~~~
        ! +- - -do until
        msnd = 1
200     continue
        ! +     ~~~~~~~~~

        if(timar > tiSND1 .and. loSND == 1) then
            msnd = msnd + 1
            intpol = 1

            read(2, 202) iyrnew, mmanew, jdanew, jhunew
            read(2, 203)
            read(2, 202) inew, jnew
            read(2, 203)
            read(2, 203)
            read(2, 204)(tSND(ksnd, 2), qSND(ksnd, 2), &
                         zSND(ksnd, 2), pSND(ksnd, 2), &
                         fSND(ksnd, 2), dSND(ksnd, 2), ksnd=40, 0, -1)
            read(2, 204) zeSND(2)
            read(2, 202) loSND

            if(abs(zeSND(2) - zeSND(1)) > epsi) &
                write(6, 1)
1           format(' **********************************', &
                   '**********************************', &
                   /, ' * CAUTION: zeSND is Time Dependant', &
                   ' (NOT taken into account in MAR) *', &
                   /, ' **********************************', &
                   '**********************************', /, 1x)

            ! +- Time Parameters
            ! +  ~~~~~~~~~~~~~~~
            jhsnd2 = (iyrnew * 365 + iyrnew / 4 &
                      + njyrGE(mmanew) + jdanew) * 24 + jhunew + itizGE(iSND, jSND)
            tiSND2 = ((iyrnew * 365 + iyrnew / 4 &
                       + njyrGE(mmanew) + jdanew) * 24 + jhunew + itizGE(iSND, jSND)) &
                     * 3.6d3

            ! +- Change  of Year
            ! +  ~~~~~~~~~~~~~~~
            timarn = timar
            if(mmaSND == 12 .and. mmanew == 1) then
#ifdef YR
                ! Change in case of iyrSND and iyrnew not defined (#YR)
                jhsnd2 = jhsnd2 + nhyrGE
                tiSND2 = tiSND2 + nhyrGE * 3600.0
#endif

                if(mmarGE == 1) &
                    timarn = timar + nhyrGE * 3600.0
            endif

            ! +- Constant Sounding  if either imposed (logcon = 1) or MARsnd.dat at EOF
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            logcon = 0
#ifdef CS
            if(itexpe == 0 .and. consnd) logcon = 1
#endif
            if(timarn > tiSND2 .or. logcon == 1) then

                intpol = 0

                jhsnd1 = jhsnd2
                tiSND1 = tiSND2
                iyrSND = iyrnew
                mmaSND = mmanew
                jdaSND = jdanew
                jhuSND = jhunew
                do ksnd = 0, 40
                    tSND(ksnd, 1) = tSND(ksnd, 2)
                    qSND(ksnd, 1) = qSND(ksnd, 2)
                    zSND(ksnd, 1) = zSND(ksnd, 2)
                    pSND(ksnd, 1) = pSND(ksnd, 2)
                    fSND(ksnd, 1) = fSND(ksnd, 2)
                    dSND(ksnd, 1) = dSND(ksnd, 2)
                enddo
            endif

        else

            ! +- Constant Sounding  if Simulation starts before 1st Sounding Time
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            intpol = 0

            jhsnd2 = jhsnd1
            tiSND2 = tiSND1
            iyrnew = iyrSND
            mmanew = mmaSND
            jdanew = jdaSND
            jhunew = jhuSND
            do ksnd = 0, 40
                tSND(ksnd, 2) = tSND(ksnd, 1)
                qSND(ksnd, 2) = qSND(ksnd, 1)
                zSND(ksnd, 2) = zSND(ksnd, 1)
                pSND(ksnd, 2) = pSND(ksnd, 1)
                fSND(ksnd, 2) = fSND(ksnd, 1)
                dSND(ksnd, 2) = dSND(ksnd, 1)
            enddo

        endif

        ! +- Continue Read
        ! +  ~~~~~~~~~~~~~

        ! +     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(timarn > tiSND2 .and. loSND == 1) go to 200
        ! +- - -end  do until
        ! +     ~~~~~~~~~~~~~

        ! +- STOP     Read
        ! +  ~~~~~~~~~~~~~

        close(unit=2)

        ! +- Output  Listing
        ! +  ~~~~~~~~~~~~~~~
        write(4, 205) msnd - intpol, jdaSND, labmGE(mmaSND), iyrSND, &
            jhuSND, jhuSND + itizGE(iSND, jSND)
205     format(/, ' SOUNDING No', i2, i6, 1x, a3, i5, i4, ' TU  (', i2, ' LT)', &
                ' --- MARsnd ---', &
                /, ' =============', &
                /, ' z   (m) | p  (Pa) | T   (K) | qv (kg/kg) |', &
                ' ff(m/s) | dd(deg) |' &
                /, '---------+---------+---------+------------+', &
                '---------+---------+')

        write(4, 206)(zSND(ksnd, 1), pSND(ksnd, 1), &
                      tSND(ksnd, 1), qSND(ksnd, 1), &
                      fSND(ksnd, 1), dSND(ksnd, 1), ksnd=40, 0, -1)
206     format((2(f8.0, ' |'), f8.2, ' |', d11.3, ' |', f8.2, ' |', f8.1, ' |'))
        ! +
        if(intpol == 1) then
            write(4, 205) msnd, jdanew, labmGE(mmanew), iyrnew, &
                jhunew, jhunew + itizGE(iSND, jSND)
            write(4, 206)(zSND(ksnd, 2), pSND(ksnd, 2), &
                          tSND(ksnd, 2), qSND(ksnd, 2), &
                          fSND(ksnd, 2), dSND(ksnd, 2), ksnd=40, 0, -1)
        endif

        ! +   ------
    endif
    ! +   ------

    ! +--Additional Variables
    ! +  ====================

    ! +--Time Parameters
    ! +  ---------------

    if(tiSND2 > tiSND1) then
        gradti = (timarn - tiSND1) / (tiSND2 - tiSND1)
        graddt = dt / (tiSND2 - tiSND1)
    else
        gradti = 0.0
        graddt = 0.0
    endif

    ! +--Exner Function and Potential Temperature
    ! +  ----------------------------------------

    do nSND = 1, intpol + 1
        do ksnd = 0, 40
            pksnd(ksnd, nSND) = exp(cap * log(dem1 * pSND(ksnd, nSND)))
            tpSND(ksnd, nSND) = tSND(ksnd, nSND) * pcap / pksnd(ksnd, nSND)
        enddo

        ! +--Potential Temperature Averaging
        ! +  -------------------------------

        tmsnd(0, nSND) = tpSND(1, nSND)
        do ksnd = 1, 39
            tmsnd(ksnd, nSND) = 0.5 * (tpSND(ksnd, nSND) &
                                       + tpSND(ksnd + 1, nSND))
        enddo
        tmsnd(40, nSND) = tpSND(40, nSND)
    enddo

    ! +  ++++++++++++++
    ! +--INITIALISATION
    ! +  ++++++++++++++

    ! +   ++++++++++++++++
    if(itexpe == 0) then
        ! +   ++++++++++++++++

        ! +--Reference Sea Level Air Temperature (K)
        ! +  =======================================

#ifdef NH
        taNH = tSND(1, 1)
#endif

        ! +--Initialisation   of  the  main    Thermodynamical   Variables
        ! +  Pressure Thickness, Surface Temperature and Specific Humidity
        ! +  =============================================================

        gra = -gravit / RDryAi
        pstSND = 0.1 * (pSND(1, 1) + gradti * (pSND(1, intpol + 1) - pSND(1, 1))) &
                 - ptopDY
        tsurf0 = tSND(1, 1) + gradti * (tSND(1, intpol + 1) - tSND(1, 1))
        qsurf0 = qSND(1, 1) + gradti * (qSND(1, intpol + 1) - qSND(1, 1))

        ! +- Reference Grid Point for Temperature Vertical Profile Initialisation
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ishmin = iSND
        jshmin = jSND
        ! +...            indices for the lowest grid point

        shmin = sh(ishmin, jshmin)
        distmn = mx + my

        do j = 1, my
            do i = 1, mx

                if(sh(i, j) < shmin) then
                    ! +...    Constraint 1: Reference Grid Point must be the lowest  One

                    ishmin = i
                    jshmin = j
                    shmin = sh(ishmin, jshmin)
                    distmn = (i - iSND) * (i - iSND) + (j - jSND) * (j - jSND)
                else
                    if(sh(i, j) == shmin) then
                        ! +...      Constraint 2: Reference Grid Point must be the closest One
                        ! +                       from the Sounding Grid Point

                        distij = (i - iSND) * (i - iSND) + (j - jSND) * (j - jSND)
                        if(distij < distmn) then
                            ishmin = i
                            jshmin = j
                            shmin = sh(ishmin, jshmin)
                            distmn = distij
                        endif
                    endif
                endif

                ! +- Surface Elevation is      MSL
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                pstDY(i, j) = pstSND
                TairSL(i, j) = tsurf0 - dtagSL
                qvapSL(i, j) = qsurf0

                ! +- Surface Elevation is NOT  MSL ==> Integration
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                if(sh(i, j) /= zero) then
                    ! +
                    do nSND = 1, intpol + 1
                        ! +
                        ksnd = 1
                        ! +...        ksnd    =  1 (when pSND(mz) -> pSND(0:mz), etc...)
                        ! +                                          zSND(0) = -500m
                        ! +
                        ! + - - - - - do until
110                     continue
                        if(zSND(ksnd, nSND) > sh(i, j)) go to 111
                        ksnd = ksnd + 1
                        go to 110
111                     continue
                        ! + - - - - - end do until
                        ! +
                        pksh = pksnd(ksnd - 1, nSND) &
                               + gravit * (zSND(ksnd - 1, nSND) - sh(i, j)) &
                               * pcap / (cp * tmsnd(ksnd - 1, nSND))
                        pint(nSND) = exp(acap * log(pksh))
                        ! +
                        gsnd = (tSND(ksnd, nSND) - tSND(ksnd - 1, nSND)) &
                               / (zSND(ksnd, nSND) - zSND(ksnd - 1, nSND))
                        tint(nSND) = tSND(ksnd - 1, nSND) &
                                     + gsnd * (sh(i, j) - zSND(ksnd - 1, nSND))
                        gsnd = (qSND(ksnd, nSND) - qSND(ksnd - 1, nSND)) &
                               / (zSND(ksnd, nSND) - zSND(ksnd - 1, nSND))
                        qint(nSND) = qSND(ksnd - 1, nSND) &
                                     + gsnd * (sh(i, j) - zSND(ksnd - 1, nSND))
                        ! +
                    enddo
                    ! +
                    pstDY(i, j) = pint(1) + gradti * (pint(intpol + 1) - pint(1)) &
                                  - ptopDY
                    TairSL(i, j) = tint(1) + gradti * (tint(intpol + 1) - tint(1)) &
                                   - dtagSL
                    qvapSL(i, j) = qint(1) + gradti * (qint(intpol + 1) - qint(1))
                    ! +
                endif
                ! +
                ! _SC     qmax        =  qsat0D(TairSL(i,j),unun,pstDY(i,j),ptopDY,lsf)
                ! _SC     qvapSL(i,j) =    min (qvapSL(i,j),qmax)
                ! +...    avoids supersaturation (_SC)
                ! +
                pstDY1(i, j) = pstDY(i, j)
            enddo
        enddo
        ! +
        ! +
        ! +--Temperature and Specific Humidity Vertical Profiles Initialisation
        ! +  ==================================================================
        ! +
        do j = 1, my
            do i = 1, mx
                ! +
                do nSND = 1, intpol + 1
                    ! +
                    ! +         **************
                    call inisnd_th(pstDY(i, j), ptopDY, sigmid, sigma, ttij, qvij)
                    ! +         **************
                    ! +
                enddo
                ! +
                do k = mz, 1, -1
                    tairDY(i, j, k) = ttij(k, 1) + gradti * (ttij(k, intpol + 1) - ttij(k, 1))
                    ta_inv = min(tairDY(i, j, k), tairDY(i, j, mz) - dtagSL)
                    qvDY(i, j, k) = (qvij(k, 1) + gradti * (qvij(k, intpol + 1) - qvij(k, 1))) &
                                    * qsat0D(ta_inv, sigma(k), &
                                             pstDY(i, j), ptopDY, lsf) &
                                    / qsat0D(tairDY(i, j, k), sigma(k), &
                                             pstDY(i, j), ptopDY, lsf)
                    ! +...     Last two Lines: Correction for possible Surface Inversion
                    ! +
                    ! _SC      qmax        =  qsat0D(tairDY(i,j,k),sigma(k),
                    ! _SC.                            pstDY(i,j),ptopDY,lsf)
                    ! _SC      qvDY(i,j,k) =  min (qvDY(i,j,k),qmax)
#ifdef OM
                    tairDY(i, j, k) = tSND(1, 1) + gradti * (tSND(intpol + 1, 1) - tSND(1, 1))
                    qvDY(i, j, k) = zero
#endif
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Reduced Potential Temperature     Vertical Profiles Initialisation
        ! +  ==================================================================
        ! +
        do j = 1, my
            do i = 1, mx
                pktaDY(i, j, mzz) = TairSL(i, j) &
                                    / ((pstDY1(i, j) + ptopDY)**cap)
            enddo
        enddo
        ! +
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    pktaDY(i, j, k) = tairDY(i, j, k) &
                                      / ((pstDY1(i, j) * sigma(k) + ptopDY)**cap)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Geostrophic Wind Vertical Profile 1st Initialisation
        ! +  ====================================================
        ! +
        do j = 1, my
            do i = 1, mx
                ! +
                do nSND = 1, intpol + 1
                    ! +
                    ! +
                    ! +--Rotation from x in the West-East Direction to x in Direction GEddxx
                    ! +  -------------------------------------------------------------------
                    ! +
                    do ksnd = 0, 40
                        ddnew = (GEddxx - dSND(ksnd, nSND)) * degrad
                        uuSND(ksnd) = fSND(ksnd, nSND) * cos(ddnew)
                        vvSND(ksnd) = fSND(ksnd, nSND) * sin(ddnew)
                    enddo
                    ! +
                    ! +
                    ! +--Vertical Interpolation
                    ! +  ----------------------
                    ! +
                    ! +         **************
                    call inisnd_vl(pstDY(i, j), ptopDY, sigmid, sigma, ulij, vlij)
                    ! +         **************
                    ! +
                enddo
                ! +
                ! +
                ! +--Time Interpolation
                ! +  ------------------
                ! +
                do k = 1, mz
                    ugeoDY(i, j, k) = ulij(k, 1) + gradti * (ulij(k, intpol + 1) - ulij(k, 1))
                    vgeoDY(i, j, k) = vlij(k, 1) + gradti * (vlij(k, intpol + 1) - vlij(k, 1))
                enddo
            enddo
        enddo
        ! +
        zetaD = zeSND(1) + gradti * (zeSND(intpol + 1) - zeSND(1))
        ! +...      zetaD: Large Scale Local Vorticity (CAUTION: Time Independant)
        ! +
        ! +
        ! +--Large Scale Wind Vertical Profile 1st Initialisation
        ! +  ====================================================
        ! +
        ! +
        ! +--Auxiliary Variable for Mass Flux Computation
        ! +  --------------------------------------------
        ! +
        shmin = 100000.0
        ! +...  shmin : minimum surface elevation (for mass flux computation)
        ! +
        do j = 1, my
            do i = 1, mx
                if(sh(i, j) < shmin) then
                    i0snd = i
                    j0snd = j
                    shmin = sh(i, j)
                endif
            enddo
        enddo
#ifdef PV
        ! +--Wind Initialisation, under Constant Potential Vorticity Constraint
        ! +  ------------------------------------------------------------------
        if(potvor .and. mmx > 1 .and. mmy == 1) then
            do k = 1, mz
                ulscPV(k) = ugeoDY(iSND, jSND, k)
                vlscPV(k) = vgeoDY(iSND, jSND, k)
                if(abs(ulscPV(k)) > zero .or. abs(vlscPV(k)) > zero) lvg = 1
            enddo
            ! Initialisation based on Potential Vorticity Conservation
            ! Based on the Direct Integration of the Relative Vorticity
            ! CAa : inisnd_PV does not exist
            !                      ! +                                  **************
            !                      if (mmx.gt.1.and.lvg.eq.1) call inisnd_PV(zetaD)
            !                      ! +                                  **************
        else
            ! +--Wind Initialisation under Constant Mass Flux Constraint
            ! +  -------------------------------------------------------
#endif
            if(conmas) then
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz1(i, j, k) = pstDY(i0snd, j0snd) * ugeoDY(i, j, k)
                            WKxyz2(i, j, k) = pstDY(i0snd, j0snd) * vgeoDY(i, j, k)
                            uairDY(i, j, k) = WKxyz1(i, j, k) / pstDY(i, j)
                            vairDY(i, j, k) = WKxyz2(i, j, k) / pstDY(i, j)
                            ! +...          Geostrophic Wind only used in Large Scale Press.Grad.Force
                            ! +             real Wind takes into Account Mass Conservation
                            ! +
                        enddo
                    enddo
                enddo
                ! +
                if(mmy == 1) then
                    do k = 1, mz
                        do j = 1, my
                            do i = 1, mx
                                vairDY(i, j, k) = vgeoDY(i, j, k)
                            enddo
                        enddo
                    enddo
                endif
                ! +
            else
                ! +
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            uairDY(i, j, k) = ugeoDY(i, j, k)
                            vairDY(i, j, k) = vgeoDY(i, j, k)
                        enddo
                    enddo
                enddo
            endif
#ifdef PV
        endif
#endif
        ! +
        ! +
        ! +--Output
        ! +  ------
        ! +
#ifdef PV
        write(21, 182)(vgeoDY(i, 1, mz), i=1, mx)
182     format(/, '  Vg(i,1,mz)      :', /,(15f8.2))
        write(21, 183)(vairDY(i, 1, mz), i=1, mx)
183     format(/, '  V (i,1,mz)      :', /,(15f8.2))
        write(21, 184)(ugeoDY(i, 1, mz), i=1, mx)
184     format(/, '  Ug(i,1,mz)      :', /,(15f8.2))
        write(21, 185)(uairDY(i, 1, mz), i=1, mx)
185     format(/, '  U (i,1,mz)      :', /,(15f8.2))
#endif
        ! +
        if(IO_loc >= 2) then
            write(21, 1860)
1860        format(1x)
            write(21, 1861)(ugeoDY(iSND, jSND, k), k=1, mz)
1861        format('  ug     =', /,(15f7.1, ' m/sec'))
            ! +
#ifdef PV
            if(potvor .and. mmx > 1 .and. mmy == 1) then
                write(21, 1860)
                write(21, 1862) adugPV
1862            format('  adugPV =', f7.3, ' m/sec')
            endif
#endif
            ! +
            write(21, 1860)
            write(21, 1863)(vgeoDY(iSND, jSND, k), k=1, mz)
1863        format('  vg     =', /,(15f7.1, ' m/sec'))
            ! +
#ifdef PV
            if(potvor .and. mmx > 1 .and. mmy == 1) then
                write(21, 1860)
                write(21, 1864)(1.d3 * advgPV(iSND, k), k=1, mz)
1864            format('  advgPV =', /,(15f7.3, ' mm/sec'))
            endif
#endif
            ! +
            write(21, 1860)
        endif
        ! +
        ! +
        ! +   ++++++
    endif
    ! +   ++++++

    ! +  +++++++++++++++++++++++++++++++++++++
    ! +--INITIALIZATION of BOUNDARY CONDITIONS
    ! +  +++++++++++++++++++++++++++++++++++++

    ! +   ++++++++++++++++++
    if(itexpe == 0) then
        ! +   ++++++++++++++++++

        ! +     ================
        if(log_1D == 0) then
            ! +     ================

            ! +--Upper
            ! +  -----
            do k = 1, mzabso
                do j = 1, my
                    do i = 1, mx
                        uairUB(i, j, k) = uairDY(i, j, k)
                        vairUB(i, j, k) = vairDY(i, j, k)
                        pktaUB(i, j, k) = pktaDY(i, j, k)
                    enddo
                enddo
            enddo

            ! +--x Axis
            ! +  ------
            if(mmx > 1) then

                do k = 1, mz

                    do j = 1, my
                        do i = 1, n7mxLB
                            vaxgLB(i, j, k, 1) = uairDY(i, j, k)
                            vaxgLB(i, j, k, 2) = vairDY(i, j, k)
                            vaxgLB(i, j, k, 3) = qvDY(i, j, k)
                            vaxgLB(i, j, k, 4) = pktaDY(i, j, k)
                            vaxgLB(i, j, 1, 5) = pstDY(i, j)
                        enddo
                        do i = mx - n6mxLB, mx
                            vaxdLB(i, j, k, 1) = uairDY(i, j, k)
                            vaxdLB(i, j, k, 2) = vairDY(i, j, k)
                            vaxdLB(i, j, k, 3) = qvDY(i, j, k)
                            vaxdLB(i, j, k, 4) = pktaDY(i, j, k)
                            vaxdLB(i, j, 1, 5) = pstDY(i, j)
                        enddo
                    enddo

                enddo

            endif

            ! +- y Axis
            ! +  ------
            if(mmy > 1) then

                do k = 1, mz

                    do i = 1, mx
                        do j = 1, n7myLB
                            vayiLB(i, j, k, 1) = uairDY(i, j, k)
                            vayiLB(i, j, k, 2) = vairDY(i, j, k)
                            vayiLB(i, j, k, 3) = qvDY(i, j, k)
                            vayiLB(i, j, k, 4) = pktaDY(i, j, k)
                            vayiLB(i, j, 1, 5) = pstDY(i, j)
                        enddo
                        do j = my - n6myLB, my
                            vaysLB(i, j, k, 1) = uairDY(i, j, k)
                            vaysLB(i, j, k, 2) = vairDY(i, j, k)
                            vaysLB(i, j, k, 3) = qvDY(i, j, k)
                            vaysLB(i, j, k, 4) = pktaDY(i, j, k)
                            vaysLB(i, j, 1, 5) = pstDY(i, j)
                        enddo
                    enddo

                enddo

            endif

            ! +     ====
        else
            ! +     ====

            if(tequil > 0.0) then

                ! +--Upper
                ! +  -----
                do k = 1, mzabso
                    do j = 1, my
                        do i = 1, mx
                            uairUB(i, j, k) = uairDY(i, j, k)
                            vairUB(i, j, k) = vairDY(i, j, k)
                            pktaUB(i, j, k) = pktaDY(i, j, k)
                        enddo
                    enddo
                enddo

                ! +--x Axis
                ! +  ------
                if(mmx > 1) then

                    do k = 1, mz

                        do j = 1, my
                            do i = 2, n7mxLB
                                vaxgLB(i, j, k, 1) = uairDY(i, j, k)
                                vaxgLB(i, j, k, 2) = vairDY(i, j, k)
                                vaxgLB(i, j, 1, 5) = pstDYn(i, j)
                            enddo
                            vaxgLB(1, j, k, 1) = uairDY(ip11 - lbcfix, j, k)
                            vaxgLB(1, j, k, 2) = vairDY(ip11 - lbcfix, j, k)
                            vaxgLB(1, j, 1, 5) = pstDYn(ip11 - lbcfix, j)

                            do i = mx - n6mxLB, mx1
                                vaxdLB(i, j, k, 1) = uairDY(i, j, k)
                                vaxdLB(i, j, k, 2) = vairDY(i, j, k)
                                vaxdLB(i, j, 1, 5) = pstDYn(i, j)
                            enddo
                            vaxdLB(mx, j, k, 1) = uairDY(mx1 + lbcfix, j, k)
                            vaxdLB(mx, j, k, 2) = vairDY(mx1 + lbcfix, j, k)
                            vaxdLB(mx, j, 1, 5) = pstDYn(mx1 + lbcfix, j)
                        enddo

                    enddo

                endif

                ! +- y Axis
                ! +  ------
                if(mmy > 1) then

                    do k = 1, mz

                        do i = 1, mx
                            do j = 1, n7myLB
                                vayiLB(i, j, k, 1) = uairDY(i, j, k)
                                vayiLB(i, j, k, 2) = vairDY(i, j, k)
                                vayiLB(i, j, k, 5) = pstDYn(i, j)
                            enddo
                            vayiLB(i, 1, k, 1) = uairDY(i, jp11 - lbcfix, k)
                            vayiLB(i, 1, k, 2) = vairDY(i, jp11 - lbcfix, k)
                            vayiLB(i, 1, k, 5) = pstDYn(i, jp11 - lbcfix)
                            do j = my - n6myLB, my1
                                vaysLB(i, j, k, 1) = uairDY(i, j, k)
                                vaysLB(i, j, k, 2) = vairDY(i, j, k)
                                vaysLB(i, j, k, 5) = pstDYn(i, j)
                            enddo
                            vaysLB(i, my, k, 1) = uairDY(i, my1 + lbcfix, k)
                            vaysLB(i, my, k, 2) = vairDY(i, my1 + lbcfix, k)
                            vaysLB(i, my, k, 5) = pstDYn(i, my1 + lbcfix)
                        enddo

                    enddo

                endif

            endif

            ! +     ======
        endif
        ! +     ======

        ! +  +++++++++++++++++++++++++++++++++++++
        ! +--UPDATE of LATERAL BOUNDARY CONDITIONS
        ! +  +++++++++++++++++++++++++++++++++++++
        ! +
        ! +   ++++
    else
        ! +   ++++
        ! +
        ! +
        ! +     ================
        if(intpol > 0) then
            ! +     ================
            ! +
            ! +
            ! +--Temperature and Specific Humidity Vertical Profiles Interpolation
            ! +  =================================================================
            ! +
            ! +
            ! +--x Axis / x <<
            ! +  -------------
            ! +
            if(mmx > 1) then
                ! +
                i = 1
                do j = 1, my
#ifdef pv
                    i = iSND
#endif
                    ! +
                    ! +- Vertical  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                    do nSND = 1, 2
                        ! +
                        ! +             **************
                        call inisnd_th(pstDY(i, j), ptopDY, sigmid, sigma, ttij, qvij)
                        ! +             **************
                        ! +
                    enddo
#ifdef pv
                    ! CAUTION: vaxgLB assumed at i=1 assumed to be that of the Sounding Point
                    !          when Potential Temperature is conserved  at the Synoptic Scale
                    i = 1
#endif
                    ! +
                    ! +- Time  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~
                    do k = 1, mz
                        dpt(k) = (ttij(k, 1) + gradti * (ttij(k, 2) - ttij(k, 1))) &
                                 / exp(cap * log(pstDY(i, j) * sigma(k) + ptopDY)) &
                                 - vaxgLB(i, j, k, 4)
                        dqa(k) = qvij(k, 1) + gradti * (qvij(k, 2) - qvij(k, 1)) &
                                 - vaxgLB(i, j, k, 3)
                    enddo
                    ! +
                    if(openLB) then
                        do ii = 1, n7mxLB
                            do k = 1, mz
                                vaxgLB(ii, j, k, 4) = vaxgLB(ii, j, k, 4) + dpt(k)
                                vaxgLB(ii, j, k, 3) = vaxgLB(ii, j, k, 3) + dqa(k)
                            enddo
                        enddo
                    else
                        do ii = 1, n7mxLB
                            do k = 1, mz
                                vaxgLB(ii, j, k, 4) = pktaDY(1, j, k) + dpt(k)
                                vaxgLB(ii, j, k, 3) = qvDY(1, j, k) + dqa(k)
                            enddo
                        enddo
                    endif
                    ! +
                enddo
                ! +
                ! +
                ! +- x Axis / x >>
                ! +  -------------
                ! +
                i = mx
                do j = 1, my
#ifdef pv
                    i = iSND
#endif
                    ! +
                    ! +- Vertical  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                    do nSND = 1, 2
                        ! +
                        ! +             **************
                        call inisnd_th(pstDY(i, j), ptopDY, sigmid, sigma, ttij, qvij)
                        ! +             **************
                        ! +
                    enddo
                    ! +
#ifdef pv
                    i = mx
#endif
                    ! +
                    ! +- Time  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~
                    do k = 1, mz
                        dpt(k) = (ttij(k, 1) + gradti * (ttij(k, 2) - ttij(k, 1))) &
                                 / exp(cap * log(pstDY(i, j) * sigma(k) + ptopDY)) &
                                 - vaxdLB(i, j, k, 4)
                        dqa(k) = qvij(k, 1) + gradti * (qvij(k, 2) - qvij(k, 1)) &
                                 - vaxdLB(i, j, k, 3)
                    enddo
                    ttij_1 = ttij(mz, 1)
                    ttij_2 = ttij(mz, 2)
                    ! +
                    if(openLB) then
                        do ii = mx - n6mxLB, mx
                            do k = 1, mz
                                vaxdLB(ii, j, k, 4) = vaxdLB(ii, j, k, 4) + dpt(k)
                                vaxdLB(ii, j, k, 3) = vaxdLB(ii, j, k, 3) + dqa(k)
                            enddo
                        enddo
                    else
                        do ii = mx - n6mxLB, mx
                            do k = 1, mz
                                vaxdLB(ii, j, k, 4) = pktaDY(mx, j, k) + dpt(k)
                                vaxdLB(ii, j, k, 3) = qvDY(mx, j, k) + dqa(k)
                            enddo
                        enddo
                    endif
                    ! +
                enddo
                ! +
#ifdef WR
                if(mod(minuGE, 30) == 0 .and. jsecGE == 0) then
                    write(6, 608) mmaSND, jdaSND, jhuSND + itizGE(iSND, jSND), &
                        mmarGE, jdarGE, jhurGE, &
                        mmanew, jdanew, jhunew + itizGE(iSND, jSND), &
                        tiSND1, timmar, tiSND2, &
                        gradti, vaxdLB(mx, my, mz, 4) * pcap
608                 format(3(i6, '/', i2, '/', i2, 'LT'), 3f13.0, &
                           ' |Time| =', f5.2, 5x, ' Theta_CLS =', f7.2)
                endif
#endif
                ! +
            ENDif                                    ! {end mmx > 1} CTR
            ! +
            ! +
            ! +- y Axis / y <<
            ! +  -------------
            ! +
            if(mmy > 1) then
                ! +
                j = 1
                do i = 1, mx
                    ! +
                    ! +- Vertical  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                    do nSND = 1, 2
                        ! +
                        ! +           **************
                        call inisnd_th(pstDY(i, j), ptopDY, sigmid, sigma, ttij, qvij)
                        ! +           **************
                        ! +
                    enddo
                    ! +
                    ! +- Time  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~
                    do k = 1, mz
                        dpt(k) = (ttij(k, 1) + gradti * (ttij(k, 2) - ttij(k, 1))) &
                                 / exp(cap * log(pstDY(i, j) * sigma(k) + ptopDY)) &
                                 - vayiLB(i, j, k, 4)
                        dqa(k) = qvij(k, 1) + gradti * (qvij(k, 2) - qvij(k, 1)) &
                                 - vayiLB(i, j, k, 3)
                    enddo
                    ! +
                    if(openLB) then
                        do jj = 1, n7myLB
                            do k = 1, mz
                                vayiLB(i, jj, k, 4) = vayiLB(i, jj, k, 4) + dpt(k)
                                vayiLB(i, jj, k, 3) = vayiLB(i, jj, k, 3) + dqa(k)
                            enddo
                        enddo
                    else
                        do jj = 1, n7myLB
                            do k = 1, mz
                                vayiLB(i, jj, k, 4) = pktaDY(i, 1, k) + dpt(k)
                                vayiLB(i, jj, k, 3) = qvDY(i, 1, k) + dqa(k)
                            enddo
                        enddo
                    endif
                    ! +
                enddo
                ! +
                ! +
                ! +- y Axis / y >>
                ! +  -------------
                ! +
                j = my
                do i = 1, mx
                    ! +
                    ! +- Vertical  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                    do nSND = 1, 2
                        ! +
                        ! +             **************
                        call inisnd_th(pstDY(i, j), ptopDY, sigmid, sigma, ttij, qvij)
                        ! +             **************
                        ! +
                    enddo
                    ! +
                    ! +- Time  Interpolation
                    ! +  ~~~~~~~~~~~~~~~~~~~
                    do k = 1, mz
                        dpt(k) = (ttij(k, 1) + gradti * (ttij(k, 2) - ttij(k, 1))) &
                                 / exp(cap * log(pstDY(i, j) * sigma(k) + ptopDY)) &
                                 - vaysLB(i, j, k, 4)
                        dqa(k) = qvij(k, 1) + gradti * (qvij(k, 2) - qvij(k, 1)) &
                                 - vaysLB(i, j, k, 3)
                    enddo
                    ! +
                    if(openLB) then
                        do jj = my - n6myLB, my
                            do k = 1, mz
                                vaysLB(i, jj, k, 4) = vaysLB(i, jj, k, 4) + dpt(k)
                                vaysLB(i, jj, k, 3) = vaysLB(i, jj, k, 3) + dqa(k)
                            enddo
                        enddo
                    else
                        do jj = my - n6myLB, my
                            do k = 1, mz
                                vaysLB(i, jj, k, 4) = pktaDY(i, my, k) + dpt(k)
                                vaysLB(i, jj, k, 3) = qvDY(i, my, k) + dqa(k)
                            enddo
                        enddo
                    endif
                    ! +
                enddo
                ! +
            ENDif                                    ! {end mmy > 1} CTR
            ! +
            ! +
            ! +--Large Scale Wind Vertical Profiles Interpolation / Dynamical Adjustment
            ! +  =======================================================================
            ! +
            ! +
            ! +--Mass Flux Auxiliary Variable
            ! +  ----------------------------
            ! +
            if(conmas) then
                ! +
                do j = 1, my
                    do i = 1, mx
                        WKxy1(i, j) = pstDY1(iSND, jSND) / pstDY1(i, j)
                    enddo
                enddo
                ! +
            else
                ! +
                do j = 1, my
                    do i = 1, mx
                        WKxy1(i, j) = 1.0
                    enddo
                enddo
                ! +
            endif
            ! +
            ! +
            ! +--Large Scale Wind Sounding: Default
            ! +  ----------------------------------
            ! +
            do k = 1, mz
                dug(k) = 0.0
                dvg(k) = 0.0
            enddo
            ! +
            ! +
            ! +--Rotation from x in the West-East Direction to x in Direction GEddxx
            ! +  -------------------------------------------------------------------
            ! +
            do nSND = 1, intpol + 1
                do ksnd = 0, 40
                    ddnew = (GEddxx - dSND(ksnd, nSND)) * degrad
                    uuSND(ksnd) = fSND(ksnd, nSND) * cos(ddnew)
                    vvSND(ksnd) = fSND(ksnd, nSND) * sin(ddnew)
                enddo
                ! +
                ! +
                ! +--Vertical Interpolation
                ! +  ----------------------
                ! +
                ! +           **************
                call inisnd_vl(pstDY(iSND, jSND), ptopDY, sigmid, sigma, &
                               ulij, vlij)
                ! +           **************
                ! +
            enddo
            ! +
            ! +
            ! +--Time Interpolation
            ! +  ------------------
            ! +
            do k = 1, mz
                dug(k) = graddt * (ulij(k, intpol + 1) - ulij(k, 1))
                dvg(k) = graddt * (vlij(k, intpol + 1) - vlij(k, 1))
            enddo
#ifdef PV
            ! +--Update of Direct Integration of Wind constrained by PV Conservation
            ! +  -------------------------------------------------------------------
            ! +
            if(potvor .and. mmx > 1 .and. mmy == 1) then
                do k = 1, mz
                    ulscPV(k) = ulscPV(k) + dug(k)
                    vlscPV(k) = vlscPV(k) + dvg(k)
                enddo
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            advbPV(i, k) = advbPV(i, k) + dvg(k)
                        enddo
                    enddo
                enddo
            endif
            ! +- PV   Conservation
            ! +  ~~~~~~~~~~~~~~~~~
            if(potvor .and. mmx > 1 .and. mmy == 1) then
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            !  PV Conservation Constraint is included
                            ugeoDY(i, j, k) = adugPV * ulscPV(k)
                            vgeoDY(i, j, k) = advbPV(i, k) &
                                              + advgPV(i, k) * ugeoDY(i, j, k) * ugeoDY(i, j, k)
                            dul = dug(k) * adugPV * adubPV(i, k)
                            ! uairDY_Synop := uairDY        +dul
                            ! vairDY_Synop := vairDY        +dvg(k)
                            ! isallobaric wind contained in (dul,dvl=dvg)
                            ugeoDY(i, j, k) = ugeoDY(i, j, k) + dvg(k) &
                                              / (fcorDY(imez, jmez) * dt)
                            vgeoDY(i, j, k) = vgeoDY(i, j, k) - dul &
                                              / (fcorDY(imez, jmez) * dt)
                        enddo
                    enddo
                enddo
            else
#endif
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            ugeoDY(i, j, k) = ugeoDY(i, j, k) + dug(k)
                            vgeoDY(i, j, k) = vgeoDY(i, j, k) + dvg(k)
                            uairDY(i, j, k) = uairDY(i, j, k) + dug(k) * WKxy1(i, j)
                            vairDY(i, j, k) = vairDY(i, j, k) + dvg(k) * WKxy1(i, j)
                        enddo
                    enddo
                enddo
#ifdef PV
            endif
#endif
            ! +
            ! +
            ! +--Lateral Boundaries
            ! +  ------------------
            ! +
            if(openLB) then
                ! +
                ! +- x Axis / x <<
                ! +  ~~~~~~~~~~~~~
                do i = 1, n7mxLB
                    do j = 1, my
#ifdef PV
                        if(potvor .and. mmx > 1 .and. mmy == 1) then
                            do k = 1, mz
                                vaxgLB(i, j, k, 1) = &
                                    vaxgLB(i, j, k, 1) + dug(k) * adubPV(i, k) * adugPV
                                vaxgLB(i, j, k, 2) = &
                                    vaxgLB(i, j, k, 2) + dvg(k)
                            enddo
                        else
#endif
                            do k = 1, mz
                                vaxgLB(i, j, k, 1) = &
                                    vaxgLB(i, j, k, 1) + dug(k) * WKxy1(i, j)
                                vaxgLB(i, j, k, 2) = &
                                    vaxgLB(i, j, k, 2) + dvg(k) * WKxy1(i, j)
                            enddo
#ifdef PV
                        endif
#endif
                    enddo
                enddo
                ! +
                ! +- x Axis / x >>
                ! +  ~~~~~~~~~~~~~
                do i = mx - n6mxLB, mx
                    do j = 1, my
#ifdef PV
                        if(potvor .and. mmx > 1 .and. mmy == 1) then
                            do k = 1, mz
                                vaxdLB(i, j, k, 1) = &
                                    vaxdLB(i, j, k, 1) + dug(k) * adubPV(i, k) * adugPV
                                vaxdLB(i, j, k, 2) = &
                                    vaxdLB(i, j, k, 2) + dvg(k)
                            enddo
                        else
#endif
                            do k = 1, mz
                                vaxdLB(i, j, k, 1) = &
                                    vaxdLB(i, j, k, 1) + dug(k) * WKxy1(i, j)
                                vaxdLB(i, j, k, 2) = &
                                    vaxdLB(i, j, k, 2) + dvg(k) * WKxy1(i, j)
                            enddo
#ifdef PV
                        endif
#endif
                    enddo
                enddo
                ! +
                ! +- y Axis / y <<
                ! +  ~~~~~~~~~~~~~
                if(mmy > 1) then
                    ! +
                    do i = 1, mx
                        do j = 1, n7myLB
                            do k = 1, mz
                                vayiLB(i, j, k, 1) = &
                                    vayiLB(i, j, k, 1) + dug(k) * WKxy1(i, j)
                                vayiLB(i, j, k, 2) = &
                                    vayiLB(i, j, k, 2) + dvg(k) * WKxy1(i, j)
                            enddo
                        enddo
                    enddo
                    ! +
                    ! +- y Axis / y >>
                    ! +  ~~~~~~~~~~~~~
                    do i = 1, mx
                        do j = my - n6myLB, my
                            do k = 1, mz
                                vaysLB(i, j, k, 1) = &
                                    vaysLB(i, j, k, 1) + dug(k) * WKxy1(i, j)
                                vaysLB(i, j, k, 2) = &
                                    vaysLB(i, j, k, 2) + dvg(k) * WKxy1(i, j)
                            enddo
                        enddo
                    enddo
                    ! +
                endif
                ! +
            else                       ! {end openLB / begin .not. openLB} CTR
                ! +
                ! +- x Axis / x <<
                ! +  ~~~~~~~~~~~~~
                if(mmx > 1) then
                    do i = 1, n7mxLB
                        do j = 1, my
                            do k = 1, mz
                                vaxgLB(i, j, k, 1) = uairDY(1, j, k)
                                vaxgLB(i, j, k, 2) = vairDY(1, j, k)
                            enddo
                        enddo
                    enddo
                    ! +
                    ! +- x Axis / x >>
                    ! +  ~~~~~~~~~~~~~
                    do i = mx - n6mxLB, mx
                        do j = 1, my
                            do k = 1, mz
                                vaxdLB(i, j, k, 1) = uairDY(mx, j, k)
                                vaxdLB(i, j, k, 2) = vairDY(mx, j, k)
                            enddo
                        enddo
                    enddo
                endif
                ! +
                ! +- y Axis / y <<
                ! +  ~~~~~~~~~~~~~
                if(mmy > 1) then
                    do k = 1, mz
                        do j = 1, n7myLB
                            do i = 1, mx
                                vayiLB(i, j, k, 1) = uairDY(i, 1, k)
                                vayiLB(i, j, k, 2) = vairDY(i, 1, k)
                            enddo
                        enddo
                    enddo
                    ! +
                    ! +- y Axis / y >>
                    ! +  ~~~~~~~~~~~~~
                    do k = 1, mz
                        do j = 1, n7myLB
                            do i = 1, mx
                                vaysLB(i, j, k, 1) = uairDY(i, my, k)
                                vaysLB(i, j, k, 2) = vairDY(i, my, k)
                            enddo
                        enddo
                    enddo
                endif
            ENDif                               ! {end .not. openLB} CTR
            ! +
            ! +     ======
        endif
        ! +     ======
        ! +
        ! +
        ! +   ++++++
    endif
    ! +   ++++++
    ! +
    ! +
    ! +--OUTPUT (Each Hour)
    ! +  ==================
    ! +
    ! +   --------------------------------
    if(minuGE == 0 .and. jsecGE == 0) then
        ! +   --------------------------------
        ! +
        do nSND = 1, 2
            fftt(nSND) = sqrt(ulij(mz, nSND) * ulij(mz, nSND) &
                              + vlij(mz, nSND) * vlij(mz, nSND))
            if(ulij(mz, nSND) /= 0.0) then
                ddtt(nSND) = atan(vlij(mz, nSND) / ulij(mz, nSND))
                if(ulij(mz, nSND) < zero) &
                    ddtt(nSND) = ddtt(nSND) + pi
            else
                if(vlij(mz, nSND) > zero) then
                    ddtt(nSND) = 0.5 * pi
                else
                    ddtt(nSND) = -0.5 * pi
                endif
            endif
        enddo
        ! +
        fftt(3) = sqrt(ugeoDY(iSND, jSND, mz) * ugeoDY(iSND, jSND, mz) &
                       + vgeoDY(iSND, jSND, mz) * vgeoDY(iSND, jSND, mz))
        if(ugeoDY(iSND, jSND, mz) /= zero) then
            ddtt(3) = atan(vgeoDY(iSND, jSND, mz) / ugeoDY(iSND, jSND, mz))
            if(ugeoDY(iSND, jSND, mz) < zero) &
                ddtt(3) = ddtt(nSND) + pi
        else
            if(vgeoDY(iSND, jSND, mz) > zero) then
                ddtt(3) = 0.5 * pi
            else
                ddtt(3) = -0.5 * pi
            endif
        endif
        ! +
        do nSND = 1, 3
            ddtt(nSND) = ddtt(nSND) * 180.0 / pi
            ddtt(nSND) = -ddtt(nSND) + 90.0
        enddo
        ! +
        write(4, 442)
442     format( &
            /, ' yyyy-MM-jj-UT-mm |  uL m/s |  vL m/s |  VL m/s |  dd deg |', &
            ' T(mx,mz) K |', &
            /, ' -----------------+---------+---------+---------+---------+', &
            '------------+')
        ! +
        i = iSND
        j = jSND
        ! +
        ! +  ***********
        call TIMcor(i, j)
        ! +  ***********
        ! +
        write(4, 443) iyrSND, mmaSND, jdaSND, jhuSND, izr, &
            ulij(mz, 1), vlij(mz, 1), &
            fftt(1), ddtt(1), ttij_1
        write(4, 443) iyrrGE, mmplus, jdplus, jhurGE, minuGE, &
            ugeoDY(iSND, jSND, mz), vgeoDY(iSND, jSND, mz), &
            fftt(3), ddtt(3), &
            vaxdLB(mx, 1, mz, 4) &
            * exp(cap * log(pstDY(mx, 1) * sigma(mz) + ptopDY))
        if(mmanew > 0) &
            write(4, 443) iyrnew, mmanew, jdanew, jhunew, izr, &
            ulij(mz, 2), vlij(mz, 2), &
            fftt(2), ddtt(2), ttij_2
443     format(i5, 4('-', i2), &
               ' |', f8.2, ' |', f8.2, ' |', f8.2, ' |', f8.1, ' |', &
               f10.3, '  |')
        write(4, 444)
444     format(/, 1x)
        ! +
        ! +
        ! +   ------
    endif
    ! +   ------
    ! +
    ! +
    ! +--Work Arrays Reset
    ! +  =================
    ! +
    ! +
    do j = 1, my
        do i = 1, mx
            WKxy1(i, j) = 0.0
        enddo
    enddo
    ! +
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                WKxyz1(i, j, k) = 0.0
                WKxyz2(i, j, k) = 0.0
            enddo
        enddo
    enddo
    return
endsubroutine inisnd
