#include "MAR_pp.def"
subroutine HYDmic(io1, io5, ioutIO, &
        ilmm, imm, jmm, ilmez, kk_pp, &
        ccni2D, ccnw2D, cfra2D, crys2D, &
        dqi2D, dqw2D, ect_2D, enr01D, &
        enr11D, enr21D, gplv2D, &
        gpmi2D, hlat2D, jhlr2D, mphy2D, &
        pk2D, pkta2D, prec2D, pst2D, &
        pst2Dn, qg2D, qi2D, qr2D, &
        qs2D, qv2D, qvsi2D, qvsw2D, &
        qw2D, rain2D, rolv2D, snoh2D, &
        snow2D, tair2D, tsrf2D, TUkv2D, &
        uair2D, vair2D, wair2D, wat01D, &
        wat11D, wat21D, watf1D, dtHyd2, &
        snf2D, sbl2D, dep2D, rnf2D, evp2D, &
        smt2D, qssbl2D, itPhys, ntHyd2)
    !------------------------------------------------------------------------+
    ! MAR HYDROLOGIC CYCLE                                  15-dec-2022  MAR |
    !   subroutine HYDmic computes Cloud Microphysical Processes             |
    !                                                                        |
    !------------------------------------------------------------------------+
    !                                                                        |
    !   INPUT / OUTPUT: qv2D(klon,klev): air   specific humidity     (kg/kg) |
    !   ^^^^^^^^^^^^^^  qw2D(klon,klev): cloud drops                 (kg/kg) |
    !                   qr2D(klon,klev): rain  drops                 (kg/kg) |
    !                   qi2D(klon,klev): ice   crystals concentration(kg/kg) |
    !                   qs2D(klon,klev): snow  flakes                (kg/kg) |
    !                   qssbl2D(klon,klev): sublimated snow  flakes  (kg/kg) |
    !   (to be added)   qg2D(klon,klev): graupels                    (kg/kg) |
    !                 ccnw2D(klon,klev): cloud droplets number       (Nb/m3) |
    !                 ccni2D(klon,klev): ice   crystals number       (Nb/m3) |
    !                                                                        |
    !                 cfra2D(klon,klev): cloud fraction                      |
    !                                                                        |
    !                 rain2D(klon)     : rain  Precipitation        (m w.e.) |
    !                 snow2D(klon)     : snow  Precipitation        (m w.e.) |
    !                 crys2D(klon)     : ice   Precipitation        (m w.e.) |
    !                                                                        |
    !                 Precipitation and sublimation in the atmosphere        |
    !                 snf2D(klon,klev): atm. snow precipitation     (m w.e.) |
    !                 sbl2D(klon,klev): atm. snow sublimation       (m w.e.) |
    !                 dep2D(klon,klev): atm. snow condensation      (m w.e.) |
    !                 rnf2D(klon,klev): atm. rain precipitation     (m w.e.) |
    !                 evp2D(klon,klev): atm. rain evaporation       (m w.e.) |
    !                 smt2D(klon,klev): atm. (integr) snow transport  (kg/m) |
    !                                                                        |
    !                 hlat2D(klon,klev): Latent Heat Release           (K/s) |
    !                  dqi2D(klon,klev): Ice    Water Formation      (kg/kg) |
    !                  dqw2D(klon,klev): Liquid Water Formation      (kg/kg) |
    !                 qvsi2D(klon,klev+1): Saturation Specific Humid.(kg/kg) |
    !                 qvsw2D(klon,klev+1): Saturation Specific Humid.(kg/kg) |
    !                                                                        |
    !   REFER. : 1) Ntezimana, unpubl.thes.LLN,          115 pp,     1993    |
    !   ^^^^^    2) Lin et al.       JCAM            22, 1065--1092, 1983    |
    !               (very similar, except that graupels are represented)     |
    !            3) Emde and Kahlig, Annal.Geophys.   7,  405-- 414, 1989    |
    !            4) Levkov et al.,   Contr.Atm.Phys. 65,   35--  57, 1992    |
    !            5) Meyers et al.,   JAM             31,  708-- 731, 1992    |
    !               (Primary Ice-Nucleation Parameterization)                |
    !            6) Delobbe and Gallee, BLM          89,   75-- 107  1998    |
    !               (Partial Condensation Scheme)                            |
    !                                                                        |
    !   CAUTION:     Partial Condensation Scheme NOT validated               |
    !   ^^^^^^^      for SCu -- Cu Transition                                |
    !                erf fonction is erroneous on HP                         |
    !                                                                        |
    !#  OPTIONS: #HM  Hallet-Mossop Theory (for Convective Updraft)          |
    !#  ^^^^^^^  #hm  idem                 (non vectorized code)             |
    !#          !#qf  Cloud Droplets Heterogeneous Freezing (not included)   |
    !#          !#qg  Graupel Conservation Equation         (to  include)    |
    !#           #hb  Snow particles distrib. parameter cnos set to BS value |
    !#           #hs  Emde & Kahlig Homogeneous Sublimation (not in Levkov)  |
    !#          !#pp  Emde & Kahlig Ice Crystal Deposition  (not included)   |
    !                                                                        |
    !#           #VW  Duynkerke et al. 1995, JAS 52, p.2763 Dropplets Fall   |
    !#           #LI  Lin  et  al (1983,JCAM 22, p.1076(50) Autoconv. Scheme |
    !#           #LO  Liou and Ou (1989, JGR 94, p.8599)    Autoconv. Scheme |
    !                                                                        |
    !#           #up  Snow Particles: Unrimed Side Planes                    |
    !#           #ur  Snow Particles: Aggregates of unrimed radiat. assembl. |
    !                                                                        |
    !#  DEBUG:   #WH  Additional Output (Each Process  is detailed)          |
    !#  ^^^^^    #WQ  FULL       Output (Each Process  is detailed)          |
    !#           #EW  Additional Output (Energy and Water Conservation)      |
    !                                                                        |
    !   REMARK : the sign '~' indicates that reference must be verified      |
    !   ^^^^^^^^                                                             |
    !------------------------------------------------------------------------+
    
    use mardim
    use marctr
    use marphy
    use margrd
    use mar_ge
    use mar_io
    use mar_dy
    use mar_hy
    use marmagic
#ifdef EW
    use mar_ew
#endif
    
    implicit none
    
    !  Input - Output
    !  ==============
    ! in
    ! --
    integer, intent(in) :: io1
    integer, intent(in) :: io5
    integer, intent(in) :: ioutIO(5)
    integer, intent(in) :: imm
    integer, intent(in) :: jmm
    integer, intent(in) :: ilmez
    integer, intent(in) :: kk_pp
    integer, intent(in) :: itPhys
    integer, intent(in) :: ntHyd2
    real, intent(in) :: dtHyd2
    ! inout
    ! ----
    integer, intent(inout) :: ilmm
    real, intent(inout) :: ccni2D(klon, klev)
    real, intent(inout) :: ccnw2D(klon, klev)
    real, intent(inout) :: cfra2D(klon, klev)
    real, intent(inout) :: crys2D(klon)
    real, intent(inout) :: dqi2D(klon, klev)
    real, intent(inout) :: dqw2D(klon, klev)
    real, intent(inout) :: ect_2D(klon, klev)
    real, intent(inout) :: enr01D(klon)
    real, intent(inout) :: enr11D(klon)
    real, intent(inout) :: enr21D(klon)
    real, intent(inout) :: gplv2D(klon, klev + 1)
    real, intent(inout) :: gpmi2D(klon, klev + 1)
    real, intent(inout) :: hlat2D(klon, klev)
    integer, intent(inout) :: jhlr2D(klon)
    character(len = 20), intent(inout) :: mphy2D(klon)
    real, intent(inout) :: pk2D(klon, klev)
    real, intent(inout) :: pkta2D(klon, klev)
    real, intent(inout) :: prec2D(klon)
    real, intent(inout) :: pst2D(klon)
    real, intent(inout) :: pst2Dn(klon)
    real, intent(inout) :: qg2D(klon, klev)
    real, intent(inout) :: qi2D(klon, klev)
    real, intent(inout) :: qr2D(klon, klev)
    real, intent(inout) :: qs2D(klon, klev)
    real, intent(inout) :: qv2D(klon, klev)
    real, intent(inout) :: qvsi2D(klon, klev + 1)
    real, intent(inout) :: qvsw2D(klon, klev + 1)
    real, intent(inout) :: qw2D(klon, klev)
    real, intent(inout) :: rain2D(klon)
    real, intent(inout) :: rolv2D(klon, klev)
    real, intent(inout) :: snoh2D(klon)
    real, intent(inout) :: snow2D(klon)
    real, intent(inout) :: tair2D(klon, klev)
    real, intent(inout) :: tsrf2D(klon)
    real, intent(inout) :: TUkv2D(klon, klev)
    real, intent(inout) :: uair2D(klon, klev)
    real, intent(inout) :: vair2D(klon, klev)
    real, intent(inout) :: wair2D(klon, klev)
    real, intent(inout) :: wat01D(klon)
    real, intent(inout) :: wat11D(klon)
    real, intent(inout) :: wat21D(klon)
    real, intent(inout) :: watf1D(klon)
    real, intent(inout) :: snf2D(klon, klev)
    real, intent(inout) :: sbl2D(klon, klev)
    real, intent(inout) :: dep2D(klon, klev)
    real, intent(inout) :: qssbl2D(klon, klev)
    real, intent(inout) :: rnf2D(klon, klev)
    real, intent(inout) :: evp2D(klon, klev)
    real, intent(inout) :: smt2D(klon, klev)

#ifdef wH
    !  Debug Variables
    !  ~~~~~~~~~~~~~~~
    integer i_fvv(klon), j_fvv(klon), klfvv, i0fvv, j0fvv, k0fvv
    common / DebuggHy / i_fvv, j_fvv, klfvv, i0fvv, j0fvv, k0fvv
    character * 70 debugH
    character * 10 proc_1, proc_2, proc_3, proc_4
    real procv1, procv2, procv3, procv4
    integer kv, nl
    real debugV(16, klev)
#endif
    
    !  Local  Variables
    !  ================
    
    integer k, m
    integer il, kl, itc, itmx, it, ii, io, ilmmi
    
    ! Work variables
    real :: W2xyz1(klon, klev)
    real :: W2xyz2(klon, klev)
    real :: W2xyz3(klon, klev)
    real :: W2xyz4(klon, klev)
    real :: W2xyz5(klon, klev + 1)
    real :: W2xyz6(klon, klev + 1)
    real :: W2xyz7(klon, klev + 1)
    real :: W2xyz8(klon, klev + 1)
    real :: W2xyz9(klon, klev)
    real :: W2xyz0(klon, klev)
    
    real :: qrevp2D(klon, klev)
    real :: qssub2D(klon, klev)
    real :: hsub2D(klon, klev)
    real :: vs2D(klon, klev)
    
    real argerf, erf, xt
    real signQw, signQr, signQi, signQs, signCi, signVR, &
            signVS, signHN, Qw0_OK, Qr0_OK, Qi0_OK, Qs0_OK, &
            Qi0qOK, Ci0cOK, Ci0_OK, vr__OK, vs__OK, qHoNuc, &
            qwOK, dpv, dqv, qHeNu1, qHeNu2, qHeNu3, qHeNuc, &
            qicnd1, qisign, qi1_OK, qicnd2, qicnd, qBerge, &
            a1, a2, am0, qidep, qvdfci, qSubl1, qSubl2, qSubli, &
            demde, sat, ab1, ab2, amf, pisub, qisub, qMelt1, &
            qMelt2, qMelt, qxmlt, qimlt, cimlt, qt, tl, pa_hPa, &
            es_hPa, qsl, dqt, wqt, ww, coefC2, sig2rh, sigqt, &
            err, alpha, t1, t2, signFR, cfraOK, SCuLim, &
            qw_new, dqw, signdq, fac_qv, updatw, dpw, signAU, &
            AutoOK, signFC, ClouOK, praut, qraut, signCC, qiOK, &
            qid, a1saut, c1saut, xtsaut, qsaut, cnsaut, ex1, psaut, &
            sign_W, WbyR_w, sign_R, WbyR_r, WbyROK, pracw, &
            qracw, WbyS_w, sign_S, WbyS_s, WbySOK, qsacw, &
            sign_T, Fact_R, SnoA, sign_C, CbyS_c, CbyS_T, &
            CbySOK, efc, psaci, qsaci, cnsaci, CbyR_c, CbyR_r, &
            CbyR_T, CbyROK, praci, qraci, CbyS_s, cnraci, &
            piacr, qiacr, qsacr, RbyS_r, RbyS_s, RbySOK, flR, &
            SbyR_r, SbyR_s, SbyROK, flS, pracs, qracs, qsacrS, &
            qracsS, Evap_r, EvapOK, sr, sign_Q, Evap_q, &
            qsacrR, almr, ab, prevp, qrevp, Evap_s, alms, si, &
            pssub, qssub, dqamx, Depo_s, SnoM_s, SnoM_T, &
            SnoMOK, qsmlt, xCoef, ACoef, BCoef, Tc, &
            Freezr, FreezT, FrerOK, psfr, akps, psmlt, &
            qsfr, Sedi_c, Sedicc, SediOK, vrmx, vsmx, vimx, &
            dzmn, xtmn, dwat, dsno, qcloud, pp, pkt0, vmmx, vmmi, &
            connw, qwclou, dmed0, dmedv, dmede, dmed5, waterb, &
            dmed, dmed2, dw0, dw4, rwbar, signHV, heavi, vwmx
    
    real dqi, dqi1, dqi2, dqi3
    real relhum, argexp, qvs_wi
    real ratio_rfsf, ratio_temp, ratio_prec
    
    real vi(klon, klev)
#ifdef VW
    real vw(klon, klev)
#endif
    real vr(klon, klev)
    real vs(klon, klev)
#ifdef qg
    real vh(klon, klev)
#endif
    real psacw(klon, klev), psacr(klon, klev)

#ifdef WH
    real wihm1(klev), wihm2(klev), wicnd(klev)
    real widep(klev), wisub(klev), wimlt(klev)
    real wwevp(klev)
    real wraut(klev), wsaut(klev)
    real wracw(klev), wsacw(klev)
    real wsaci(klev), wraci(klev), wiacr(klev)
    real wsacr(klev), wracs(klev), wrevp(klev)
    real wssub(klev), wsmlt(klev), wsfre(klev)
    real qiold(klev), qwold(klev)
#endif
    
    ! rad_ww: Droplet Radius, Meyers et al. (1992), JAM
    real rad_ww

#ifdef HM
    ! Levkov et al. (1992) CAM
    real SplinJ, SplinP
#endif
    
    !  DATA
    !  ====
    
    logical, parameter :: Meyers = .true.
    logical, parameter :: LevkovAUTO = .true.
    ! LevkovAUTX = .true. => Levkov parameterization Bergeron Processes
    ! LevkovAUTX = .false. => Emde&Kahlig parameterization Bergeron Processes
    logical, parameter :: LevkovAUTX = .true.
    
    logical, parameter :: EmdeKa = .false.
    ! fracSC: SCu Fractional Cloudiness Delobbe
    ! fracSC = .true. => Delobbe SCu Fractional Cloudiness Scheme
    ! fracSC = .true. => may be setup if fracld = .true.
    logical, parameter :: fracSC = .false.
    ! fraCEP: SCu Fractional Cloudiness ECMWF
    logical, parameter :: fraCEP = .false.
    
    real, parameter :: thir5 = 1.66e0 ! CAa 5. / 3.
    
    real, parameter :: eps1 = 1.e-01
    
    ! cnor: intercept parameter / rain distribution ! Tuning
    ! #ifdef LA
    !    real, parameter :: cnor = 3.0e06
    ! #else
    real, parameter :: cnor = 8.0e06
    ! #endif
    
    ! cnos: intercept parameter / snow distribution
    real :: cnos
    
    ! cnog: intercept parameter / graupel distribution
    ! cnog: Lin et al. 1983, JCAM 22, p.1068 (1,2 and 3)
    real, parameter :: cnog = 4.0e04
    
    ! cnos2: intercept parameter / snow distribution
    real, parameter :: cnos2 = 5.e06
    
    real, parameter :: ui50 = 0.1e0
    real, parameter :: ri50 = 5.e-5
    real, parameter :: beta = 0.5e0
    
    ! C1_EkM: Partial Condensation Scheme
    real, parameter :: C1_EkM = 0.14e-3
    ! C2_EkM: Ek and Mahrt 1991, An.Geoph. 9, 716--724
    real, parameter :: C2_EkM = 9.75e+0
    
    ! tsfo: minimum temperature (deg.C) before instant. cloud dropplets freezing
    ! tsfo: Levkov et al.1992, C.Atm.Ph.65, p.39
    real, parameter :: tsfo = -35.e0
    
    ! WatIce, ExpWat, ExpWa2: Saturation pressure over Water, Dudhia (1989) JAS
    real, parameter :: WatIce = 273.16e0
    real, parameter :: ExpWat = 5.138e0
    real, parameter :: ExpWa2 = 6827.e0
    
    ! aM_Nid, bM_Nid, TM_Nid: Deposition Condensation-Freezing Nucleation Param.
    ! aM_Nid, bM_Nid, TM_Nid: Meyers et al. (1992), p.713
    ! real, parameter :: aM_Nid = -0.639
    ! real, parameter :: bM_Nid =  0.1296
    !XF
    real, parameter :: aM_Nid = -1.488
    ! bM_Nid: Prenni et al. (2007), p.545, BAMS
    real, parameter :: bM_Nid = 0.0187
    real, parameter :: TM_Nid = -5.
    
    ! aM_Nic, bM_Nic, TM_Nic: Contact Freezing Nucleation Parameters
    ! aM_Nic, bM_Nic, TM_Nic: Meyers et al. (1992), p.713
    real, parameter :: aM_Nic = -2.80
    real, parameter :: bM_Nic = 0.262
    real, parameter :: TM_Nic = -2.

#ifdef HM
    ! TmnNhm, TmxNhm, w_svrl: Hallet-Mossop Theory
    ! see Levkov et al., (1992), Contr.Atm.Phy. 65, p.40
    real, parameter :: TmnNhm = -8.
    real, parameter :: TmxNhm = -3.
    real, parameter :: w_svrl = 1.
#endif
    
    ! qsd0: Smallest Diameter of Particles in the snow Class
    ! qsd0: Levkov et al. (1992), Contr. Atm. Phys. 65, p.41, para 1
    real, parameter :: qsd0 = 2.0e-4
    
    ! qi00: max. ice crystals concentr. before autoconv. of snow flakes occurs
    ! qi00: Lin et al. (1983), JCAM 22, p.1070 (21)
    real, parameter :: qi00 = 0.001e0
    ! qi00 = 0.0008: compromise when graupels are not included
    ! qi00 = 0.0008: Emde and Kahlig (1989), Ann.Geoph. 7, p.408  (18)
    ! _hl data qi00/0.0008/
    
    ! qg00: max. ice crystals concentr. before autoconversion of graupels occurs
    ! qg00: Lin et al. (1983), JCAM 22, p.1074 (37)
    real, parameter :: qg00 = 0.0006e0
    
    ! sigmaw = 1/3 ln(1/k), where k=0.8 (dispersion parameter)
    ! sigmaw: Martin et al. (1994), JAS 51, p.1823
    real, parameter :: sigmaw = 0.27e+0

#ifdef LO
    ! rcrilo: Autoconversion Critical Radius (Liou and Ou, 1989)
    real, parameter :: rcrilo = 10.0e-6
#endif

#ifdef LI
    ! qw00L: maximum cloud droplets concentration before autoconversion occurs
    ! qw00L: Lin et al. (1983), JCAM 22, p.1076 (50)
    real, parameter :: qw00L = 0.002e0
#endif
    
    ! qi0S: critical solid water mixing ratio (tuned Dome C) (FacFIk >   1)
    ! qi0S = 0.3e(-3) : critical solid water mixing ratio (standard)
    real, parameter :: qi0S = 0.10e-3
    
    ! qw00: critical liquid water mixing ratio
    ! qw00 = 0.30e-3 -> standard value
    ! qw00: Sundqvist (1988) : Physically-Based Modelling and
    ! qw00:                    Simulation of Climate and Climatic Change,
    ! qw00:                    M.E. Schlesinger, Ed., Reidel, 433-461.
    real, parameter :: qw00 = 0.10e-3

#ifdef SC
    real, parameter :: camart = 0.8e0
    ! connw : droplets number concentration (m-3)
    real, parameter :: connw = 1.2e8
#endif
    
    ! csud: charac. time scale for autoconversion (SUND), Sundqvist (1988)
    real, parameter :: csud = 1.0e-4
    
    ! typww: Typical Cloud Droplet Weight [Ton] (typ. diam.: 32.5 mim)
    ! typww: (used with air Density rolv2D [Ton/m3])
    real, parameter :: typww = 18.e-15
    
    ! cc1, cc2, dd0: cloud droplets autoconversion parameters
    real, parameter :: cc1 = 1.200e-04
    real, parameter :: cc2 = 1.569e-12
    real, parameter :: dd0 = 0.15e0
    
    ! ==========================================================================
    ! aa1, aa2: Bergeron Process Data (given by Koenig, 1971, J.A.S. 28,p235) ==
    
    real, parameter, dimension(31) :: aa1 = (/ 0.7939e-07, 0.7841e-06, 0.3369e-05, 0.4336e-05, &
            0.5285e-05, 0.3728e-05, 0.1852e-05, 0.2991e-06, &
            0.4248e-06, 0.7434e-06, 0.1812e-05, 0.4394e-05, &
            0.9145e-05, 0.1725e-06, 0.3348e-04, 0.1725e-04, &
            0.9175e-05, 0.4412e-05, 0.2252e-05, 0.9115e-06, &
            0.4876e-06, 0.3473e-06, 0.4758e-06, 0.6306e-06, &
            0.8573e-06, 0.7868e-06, 0.7192e-06, 0.6513e-06, &
            0.5956e-06, 0.5333e-06, 0.4834e-06/)
    
    real, parameter, dimension(31) :: aa2 = (/  0.4006e0, 0.4831e0, 0.5320e0, 0.5307e0, 0.5319e0, &
            0.5249e0, 0.4888e0, 0.3894e0, 0.4047e0, 0.4318e0, &
            0.4771e0, 0.5183e0, 0.5463e0, 0.5651e0, 0.5813e0, &
            0.5655e0, 0.5478e0, 0.5203e0, 0.4906e0, 0.4447e0, &
            0.4126e0, 0.3960e0, 0.4149e0, 0.4320e0, 0.4506e0, &
            0.4483e0, 0.4460e0, 0.4433e0, 0.4413e0, 0.4382e0, &
            0.4361e0 /)
    
    ! === Bergeron Process Data (given by Koenig, 1971, J.A.S. 28,p235) ========
    ! ==========================================================================
    
    !  Upper Limit for specific Humidity
    !  =================================
    
    ! SSImax: Maximum Sursaturation % ICE (900 ==> RH=1000%)
#ifdef kk
    real, parameter :: SSImax = 900.
#else
    real, parameter :: SSImax = 101.0
#endif
    
    real, parameter :: relCri = 1.0
    !#ifdef rc
    !    relCri = 0.9 + 0.08 * sqrt(max(0.,100. - dx*0.001) / 95.)
    !#endif
    
    ! ======== end declaration ============
    
    ! Define constants
    ! ================
    
    !  Cloud Droplets Autoconversion Threshold
    !  =======================================
    
    ! cminHY: Cloud Fraction under which no Autoconversion occurs
    cminHY = 1.0e-3
    
    ! cnos: intercept parameter / snow distribution
    cnos = 1.0e6 ! XF 28/08/2023
#ifdef AC
    cnos = 3.0e8
    ! old cnos over ANT
    ! cnos = 5.e7
    ! cnos = 3.e6
#endif
#ifdef GR
    cnos = 2.0e8
#endif
    ! #ifdef LA
    !     cnos = 4.0e6
    ! #endif

#ifdef LI
    qw00 = qw00L
#endif

#ifdef hb
    !  For Blown Snow Particles
    !  ========================
    !  do NOT USE unless for specific sensivity experiments
    cnos = 0.1e18
    if(itexpe == 0) write(6, 6000)
    6000 format(/, ' ****************************************************', &
            /, ' * cnos  = 0.1d18 for PURE BLOWING SNOW EXPERIMENTS *', &
            /, ' *             do not USE  OTHERWISE                *', &
            /, ' ****************************************************', &
            /)
#endif
    
    !  Update of Temperature
    !  =====================
    
    xt = min(dt, dtHyd)
    xt = dtHyd2
    
    qHeNu1 = 0
    qHeNu2 = 0
    qHeNu3 = 0
    
    psacw = 0
    psacr = 0
    
    do kl = mzhyd, klev
        do il = 1, klon
            tair2D(il, kl) = pkta2D(il, kl) * pk2D(il, kl)
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'HYDmic: Debugged Variables: Initial'
            debugH(36:70) = '                                   '
            proc_1 = 'R.Hum W[%]'
            procv1 = 0.1 * qv2D(il, kl) / (rhcrHY * qvsw2D(il, kl))
            proc_2 = 'R.Hum I[%]'
            procv2 = 0.1 * qv2D(il, kl) / (rhcrHY * qvsi2D(il, kl))
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            do kv = 1, 16
                debugV(kv, kl) = 0.
            enddo
#endif
        enddo
    enddo
    
    ! CAa   variables initialization
    do kl = 1, klev
        do il = 1, klon
            qrevp2D(il, kl) = 0.0
            qssub2D(il, kl) = 0.0
        enddo
    enddo

#ifdef EW
    !  Vertical Integrated Energy and Water Content
    !  ============================================
    do il = 1, klon
        enr01D(il) = 0.0
        wat01D(il) = 0.0
        do kl = 1, klev
            enr01D(il) = enr01D(il) &
                    + (tair2D(il, kl) &
                            - (qw2D(il, kl) + qr2D(il, kl)) * r_LvCp &
                            - (qi2D(il, kl) + qs2D(il, kl)) * r_LsCp) &
                            * dsigm1(kl)
            wat01D(il) = wat01D(il) &
                    + (qv2D(il, kl) &
                            + qw2D(il, kl) + qr2D(il, kl) &
                            + qi2D(il, kl) + qs2D(il, kl)) &
                            * dsigm1(kl)
        enddo
        ! mphy2D -->  '12345678901234567890'
        mphy2D(il) = '                    '
#ifdef ew
        enr01D(il) = enr01D(il) * pst2Dn(il) * grvinv
#endif
        !  wat01D [m] contains an implicit factor (10.**3) [kPa-->Pa] /ro_Wat
        wat01D(il) = wat01D(il) * pst2Dn(il) * grvinv
    enddo
#endif
#ifdef WH
    vmmx = 0.0
#endif
    
    !  Set lower limit on Hydrometeor Concentration
    !  ============================================
    
    if(itPhys == 1) then
        if(no_vec) then
            do kl = mzhyd, klev
                do il = 1, klon
                    
                    if(qw2D(il, kl) < eps9) then
                        qv2D(il, kl) = qv2D(il, kl) + qw2D(il, kl)
                        tair2D(il, kl) = tair2D(il, kl) - qw2D(il, kl) * r_LvCp
                        dqw2D(il, kl) = dqw2D(il, kl) - qw2D(il, kl)
                        qw2D(il, kl) = 0.0
                    endif
                    
                    if(qr2D(il, kl) < eps9) then
                        qv2D(il, kl) = qv2D(il, kl) + qr2D(il, kl)
                        tair2D(il, kl) = tair2D(il, kl) - qr2D(il, kl) * r_LvCp
                        dqw2D(il, kl) = dqw2D(il, kl) - qr2D(il, kl)
                        qr2D(il, kl) = 0.0
                    endif
                    
                    if(qi2D(il, kl) < eps9 .or. ccni2D(il, kl) < unun) then
                        qv2D(il, kl) = qv2D(il, kl) + qi2D(il, kl)
                        tair2D(il, kl) = tair2D(il, kl) - qi2D(il, kl) * r_LsCp
                        dqi2D(il, kl) = dqi2D(il, kl) - qi2D(il, kl)
                        qi2D(il, kl) = 0.0
                        ccni2D(il, kl) = 0.0
                    endif
                    
                    if(qs2D(il, kl) < eps9) then
                        qv2D(il, kl) = qv2D(il, kl) + qs2D(il, kl)
                        tair2D(il, kl) = tair2D(il, kl) - qs2D(il, kl) * r_LsCp
                        dqi2D(il, kl) = dqi2D(il, kl) - qs2D(il, kl)
                        qs2D(il, kl) = 0.0
                    endif
                enddo
            enddo
        else
            do kl = mzhyd, klev
                do il = 1, klon
                    signQw = sign(unun, eps9 - qw2D(il, kl))
                    Qw0_OK = max(zero, signQw) * qw2D(il, kl)
                    qw2D(il, kl) = qw2D(il, kl) - Qw0_OK
                    dqw2D(il, kl) = dqw2D(il, kl) - Qw0_OK
                    qv2D(il, kl) = qv2D(il, kl) + Qw0_OK
                    tair2D(il, kl) = tair2D(il, kl) - Qw0_OK * r_LvCp
                    
                    signQr = sign(unun, eps9 - qr2D(il, kl))
                    Qr0_OK = max(zero, signQr) * qr2D(il, kl)
                    qr2D(il, kl) = qr2D(il, kl) - Qr0_OK
                    dqw2D(il, kl) = dqw2D(il, kl) - Qr0_OK
                    qv2D(il, kl) = qv2D(il, kl) + Qr0_OK
                    tair2D(il, kl) = tair2D(il, kl) - Qr0_OK * r_LvCp
                    
                    signQi = sign(unun, eps9 - qi2D(il, kl))
                    Qi0qOK = max(zero, signQi)
                    signCi = sign(unun, unun - ccni2D(il, kl))
                    Ci0cOK = max(zero, signCi)
                    
                    Ci0_OK = max(Ci0cOK, Qi0qOK)
                    Qi0_OK = Ci0_OK * qi2D(il, kl)
                    
                    ccni2D(il, kl) = ccni2D(il, kl) * Ci0_OK
                    qi2D(il, kl) = qi2D(il, kl) - Qi0_OK
                    dqi2D(il, kl) = dqi2D(il, kl) - Qi0_OK
                    qv2D(il, kl) = qv2D(il, kl) + Qi0_OK
                    tair2D(il, kl) = tair2D(il, kl) - Qi0_OK * r_LsCp
                    
                    signQs = sign(unun, eps9 - qs2D(il, kl))
                    Qs0_OK = max(zero, signQs) * qs2D(il, kl)
                    qs2D(il, kl) = qs2D(il, kl) - Qs0_OK
                    dqi2D(il, kl) = dqi2D(il, kl) - Qs0_OK
                    qv2D(il, kl) = qv2D(il, kl) + Qs0_OK
                    tair2D(il, kl) = tair2D(il, kl) - Qs0_OK * r_LsCp
                enddo
            enddo
        endif
    endif
    
    !  Update of dummy Variables
    !  =========================
    
    do kl = mzhyd, klev
        do il = 1, klon
            W2xyz1(il, kl) = tair2D(il, kl) - TfSnow
            ! W2xyz2 : Ice Crystals Number (Fletcher, 1962)
            W2xyz2(il, kl) = 1.e-2 * exp(-0.6 * W2xyz1(il, kl))
            
            W2xyz3(il, kl) = qr2D(il, kl)
            W2xyz4(il, kl) = qs2D(il, kl)
#ifdef qg
            W2xyz0(il, kl) = qg2D(il, kl)
#endif
#ifdef WH
            ! old values
            if(il == ilmm) then
                qwold(kl) = qw2D(il, kl)
                qiold(kl) = qi2D(il, kl)
            endif
#endif
        enddo
    enddo
    
    !  Saturation Specific Humidity
    !  ============================
    
    !    ***********
    call qsat2D(tair2D, pst2D, tsrf2D, qvsi2D, qvsw2D)
    !    ***********
    
    do kl = mzhyd, klev
        do il = 1, klon
            ! W2xyz5:  Saturation Specific Humidity over Ice
            W2xyz5(il, kl) = rhcrHY * qvsi2D(il, kl)
            
            W2xyz6(il, kl) = sqrt((pst2Dn(il) + ptopDY) &
                    / (rolv2D(il, kl) * RDryAi * tair2D(il, klev)))
#ifdef VW
            !  Cloud Droplets Fall Velocity
            !  (Calcul de la Vitesse Terminale Moyenne)
            !  ----------------------------
            if(qw2D(il, kl) >= eps9) then
                !  ccnw2D: ASTEX case (Duynkerke et al. 1995, JAS 52, p.2763)
                ccnw2D(il, kl) = 1.2d8
                qwclou = qw2D(il, kl) / max(cminHY, cfra2D(il, kl))
                dmed0 = 4.5 * sigmaw * sigmaw
                dmedv = 12.5 * sigmaw * sigmaw
                dmede = qwclou * rolv2D(il, kl) &
                        * 6.d0 / (pi * ccnw2D(il, kl) * exp(dmed0))
                dmed5 = exp(thir5 * log(dmede))
                ! dmed        = exp(third*log(dmede))
                vw(il, kl) = 1.19d8 * pi * ccnw2D(il, kl) * dmed5 &
                        * exp(dmedv) / (24.0 * rolv2D(il, kl) * qwclou)
            else
                vw(il, kl) = 0.00
            endif
#endif
            
            !  Rain Fall Velocity
            !  ------------------
            
            ! W2xyz7(il,kl) : lambda_r : Marshall-Palmer Distribution Parameter
            !                            for Rain
            ! Note that a simplification occurs
            ! between the 1000. factor of rho, and rho_water=1000.
            ! Reference  : Emde and Kahlig (1989), Ann.Geoph. 7, p.407 (3)
            W2xyz7(il, kl) = exp(0.25 * log((pi * cnor) &
                    / (rolv2D(il, kl) * max(eps9, qr2D(il, kl)))))
            
            if(qr2D(il, kl) > eps9) then
                ! vr__OK = 1. if qr2D(il,kl)  > eps9
                ! vr__OK = 0. otherwise
                signVR = sign(unun, qr2D(il, kl) - eps9)
                vr__OK = max(zero, signVR)
                
                ! vr : Terminal Fall Velocity for Rain
                ! 392  = a Gamma[4+b] / 6 where  a = 842. and b = 0.8
                vr(il, kl) = vr__OK * 392. * W2xyz6(il, kl) &
                        / exp(0.8 * log(W2xyz7(il, kl)))
            else
                vr(il, kl) = 0.
            endif
            
            !  Snow Fall Velocity
            !  ------------------
            
            ! W2xyz8(il,kl) : lambda_s : Marshall-Palmer distribution parameter
            ! for Snow Flakes
            ! Note that a partial simplification occurs
            ! between the 1000. factor of rho, and rho_snow=500.
            ! Reference  : Emde and Kahlig 1989, Ann.Geoph.      7,  p.407 (3)
            ! (rho_snow)   Levkov et al.   1992, Cont.Atm.Phys. 65(1) p.37 (5)
#ifdef cn
            cnos = min(2.e8, cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
            W2xyz8(il, kl) = exp(0.25 * log((0.50 * pi * cnos) &
                    / (rolv2D(il, kl) * max(eps9, qs2D(il, kl)))))
            
            if(qs2D(il, kl) > eps9) then
                ! vs__OK = 1. if qs2D(il,kl)  > eps9
                ! vs__OK = 0. otherwise
                signVS = sign(unun, qs2D(il, kl) - eps9)
                vs__OK = max(zero, signVS)
                
                ! vs: Terminal Fall Velocity for Snow Flakes
                ! 2.19 = c Gamma[4+d] / 6
                ! where  c = 4.836 = 0.86 *1000.**0.25
                ! and d = 0.25
                ! (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                ! Graupellike Snow Flakes of Hexagonal Type)
                vs(il, kl) = vs__OK * 2.19 * W2xyz6(il, kl) &
                        / exp(0.25 * log(W2xyz8(il, kl)))
                ! old option #up
                ! OR 2976. = c Gamma[4+d] / 6
                ! where  c = 755.9 = 0.81 *1000.**0.99 and d = 0.99
                ! (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                ! Unrimed Side Planes)
                vs(il, kl) = vs__OK * 2976. * W2xyz6(il, kl) &
                        / exp(0.99 * log(W2xyz8(il, kl)))
#ifdef ur
                ! OR 2976. = c Gamma[4+d] / 6
                ! where c = 755.9 = 0.69 *1000.**0.41 and d = 0.41
                ! (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                ! Aggregates of unrimed radiating assemblages)
                vs(il, kl) = vs__OK * 20.06 * W2xyz6(il, kl) &
                        / exp(0.41 * log(W2xyz8(il, kl)))
#endif
#ifdef EU
                vs(il, kl) = vs__OK * 2.19 * W2xyz6(il, kl) &
                             / exp(0.25 * log(W2xyz8(il, kl)))
#endif
            else
                vs(il, kl) = 0.0
            endif

#ifdef qg
            !  Graupel Fall Velocity
            !  ---------------------
            if(qg2D(il, kl) >= eps9) then
                ! Don't forget "#hy" option !
                ! W2xyz9(il,kl) : lambda_g : Marshall-Palmer distrib. parameter
                ! for Graupel
                ! Note that a simplification occurs
                ! between the 1000. factor of rho, and rho_ice=1000.
                W2xyz9(il, kl) = exp(0.250 * log((pi * cnog) &
                        / (rolv2D(il, kl) * max(eps9, qg2D(il, kl)))))
                ! vh: Terminal Fall Velocity for Graupels
                ! 25.1 = c Gamma[4+d] / 6
                ! where  c = 4.836 = 1.10 *1000.**0.57 and d = 0.57
                ! (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                ! Hexagonal Graupel)
                vh(il, kl) = 25.1 * W2xyz6(il, kl) &
                        / exp(0.57 * log(W2xyz9(il, kl)))
            else
                vh(il, kl) = 0.0
                W2xyz9(il, kl) = 0.0
            endif
#endif
        enddo
    enddo
    
    !  ===================================================================
    !  Microphysical Processes affecting non Precipitating Cloud Particles
    !  ===================================================================
    !  Homogeneous Nucleation by Cloud Dropplets Solidification  ! BFREWI
    !  Ref: Emde and Kahlig 1989, Ann.Geoph. 7, p.407 (11) ! Levkov (24) p.40
    !  ---------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qHoNuc = 0.
            qHeNuc = 0.
            qwOK = 0.
#endif
            if(W2xyz1(il, kl) < tsfo) then
                ! qHoNuc = 1. if W2xyz1(il,kl) < tsfo
                ! qHoNuc = 0. otherwise
                signHN = -sign(unun, W2xyz1(il, kl) - tsfo)
                qHoNuc = max(zero, signHN)
#ifdef EW
                if(qHoNuc > epsi) then ! ctr
                    mauxEW = mphy2D(il)
                    mauxEW(01:01) = 'i'
                    mphy2D(il) = mauxEW
                ENDif            ! ctr
#endif
                qwOK = qw2D(il, kl) * qHoNuc
                qi2D(il, kl) = qi2D(il, kl) + qwOK
                ccni2D(il, kl) = ccni2D(il, kl) + rolv2D(il, kl) * qwOK / typww
                tair2D(il, kl) = tair2D(il, kl) + r_LcCp * qwOK
#ifdef WQ
                write(6, *) 'Qihm1', qw2D(il, kl), &
                        ' Qi', qi2D(il, kl), &
                        ' CcnI', ccni2D(il, kl), itexpe, il, kl
                if(il == ilmm) wihm1(kl) = qwOK
#endif
                qw2D(il, kl) = qw2D(il, kl) - qwOK
            
            endif
            
            !  Heterogeneous Freezing of Cloud Droplets ! BNUFWI
            !  Reference: Levkov et al., 1992 (21) p.40 ! Levkov (21) p.40
            !  ----------------------------------------
            
            !!#qf #hy   if(W2xyz1(il,kl) < 0.00)  then
            
            !!#qf           signHN = -sign(unun,W2xyz1(il,kl)  - 0.)
            !               qHeNuc =   1.0 if   W2xyz1(il,kl)  < 0.00dgC
            !                      =   0.0 otherwise
            !!#qf           qHeNuc =   max(zero,signHN)
            
            !!#qf           argexp = min(max(argmin,-W2xyz1(il,kl)),argmax)
            !!#qf           qHeNuc = qHeNuc*(exp(argexp)   -  1.  ) &
            !!#qf               *  qw2D(il,kl) * 100.0 *typww
            !!#qf           qHeNuc = min(qHeNuc,  qw2D(il,kl))
            
            !!#qf           qi2D(il,kl) = qi2D(il,kl) + qHeNuc
            !!#qf           ccni2D(il,kl) = ccni2D(il,kl) + rolv2D(il,kl)*qHeNuc/typww
            !!#qf           tair2D(il,kl) = tair2D(il,kl) + r_LcCp       *qHeNuc
            !!#qf           qw2D(il,kl) = qw2D(il,kl)   -               qHeNuc
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Homo+Hetero Nucleation by Droplets '
            debugH(36:70) = 'Solidification (BFREWI+BNUFWI)     '
            proc_1 = 'BFREWI    '
            procv1 = qHoNuc
            proc_2 = 'BNUFWI    '
            procv2 = qHeNuc
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(01, kl) = qwOK + qHeNuc
#endif
            !!#qf #hy   end if

#ifdef EW
            !===================================================================
            !  Homogeneous Sublimation                             ! XXXXXX
            !  Ref: Emde and Kahlig 1989, Ann.Geoph. 7, p.407 (12) ! Levkov
            !  ---------------------------------------------------------
            if(qHoNuc > epsi) then ! ctr
                mauxEW = mphy2D(il)
                mauxEW(02:02) = 'I'
                mphy2D(il) = mauxEW
            ENDif            ! ctr
#endif
#ifdef hs
            ! 1.733e7=Ls*Ls*0.622/Cpa/Ra with Ls = 2833600 J/kg
            dpv = (qv2D(il, kl) - W2xyz5(il, kl)) &
                    / (1.d0 + 1.733e7 * W2xyz5(il, kl) &
                            / (tair2D(il, kl) * tair2D(il, kl)))
            dpv = qHoNuc * max(zero, dpv)
            dqv = dpv
            qi2D(il, kl) = qi2D(il, kl) + dqv
            dqi2D(il, kl) = dqi2D(il, kl) + dqv
            ! ccni2D(il,kl) : NO VARIATION
            qv2D(il, kl) = qv2D(il, kl) - dqv
            tair2D(il, kl) = tair2D(il, kl) + r_LsCp * dqv
#endif
#ifdef WQ
            !  Full Debug
            !  ~~~~~~~~~~
            write(6, *) 'Qihm2', dqv, &
                    ' Qi', qi2D(il, kl), &
                    ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
            if(il == ilmm) wihm2(kl) = dqv
#endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Emde and Kahlig: Homogeneous Sublim'
            debugH(36:70) = 'ation                              '
            proc_1 = 'dQv   g/kg'
            procv1 = dqv
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = 'CCNI/1.e15'
            procv4 = ccni2D(il, kl) * 1.e-18
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(01, kl) = dqv + debugV(01, kl)
#endif
        enddo
    enddo
    
    !===========================================================================
    !  Nucleation  I: Deposition & Condensation-Freezing Nucleat.
    !  Source       : Water Vapor                           ! BNUCVI
    !  Reference: Meyers et al., 1992, JAM 31, (2.4) p.712  ! Levkov (20) p.40
    !  -----------------------------------------------------------
    if(Meyers) then
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qHeNuc = 0.
                qicnd1 = 0.
                dqi1 = 0.
                dqi2 = 0.
                dqi3 = 0.
#endif
                if(W2xyz1(il, kl) < TM_Nid) then
                    ! qHeNu1 = 1.0 if W2xyz1(il,kl) < TM_Nid
                    ! qHeNu1 = 0.0 otherwise
                    signHN = -sign(unun, W2xyz1(il, kl) - TM_Nid)
                    qHeNu1 = max(zero, signHN)
                    
                    ! Sursaturation
                    dqv = qv2D(il, kl) - W2xyz5(il, kl)
                    dqv = max(zero, dqv)
                    
                    if(dqv > 0.) then
                        ! qHeNu3 = 1.0 if qv2D(il,kl) > W2xyz5(il,kl)
                        ! qHeNu3 = 0.0 otherwise
                        signHN = sign(unun, dqv)
                        qHeNu3 = max(zero, signHN)
                        
                        qHeNuc = qHeNu1 * qHeNu3
                        
                        ! Sursaturation relative to ice
                        ! Meyers et al. (1992) JAM, 2.4
                        qicnd1 = 1.0e2 * dqv / W2xyz5(il, kl)
                        qicnd1 = min(qicnd1, SSImax)
                        qicnd1 = 1.0e3 * exp(aM_Nid + bM_Nid * qicnd1)
                        qicnd1 = max(qicnd1 - ccni2D(il, kl), zero) * qHeNuc
                        ccni2D(il, kl) = ccni2D(il, kl) + qicnd1
                        ! 1.e-15 = 0.001 * Initial Ice Crystal Mass
                        dqi = 1.0e-15 * qicnd1 / rolv2D(il, kl)
                        dqi = min(dqi, dqv)
                        qi2D(il, kl) = qi2D(il, kl) + dqi
                        dqi2D(il, kl) = dqi2D(il, kl) + dqi
                        qv2D(il, kl) = qv2D(il, kl) - dqi
                        tair2D(il, kl) = tair2D(il, kl) + dqi * r_LsCp
                        dqi1 = dqi
                    
                    endif
                endif
                
                !  Nucleation  I:              Contact     -Freezing Nucleat.
                !  Source       : Cloud Dropplets               ! BSPRWI
                !  Ref: Meyers et al. (1992), JAM 31, 2.6 p.713 ! Levkov (20) p.40
                !  -----------------------------------------------------------
#ifdef wH
                qicnd1 = 0.
                qicnd2 = 0.
                dqi = 0.
#endif
                if(qw2D(il, kl) > 0.) then
                    ! qHeNu3 = 1.0 if qw2D(il,kl) > 0.
                    ! qHeNu3 = 0.0 otherwise
                    signHN = sign(unun, qw2D(il, kl))
                    qHeNu3 = max(zero, signHN)
                    
                    if(W2xyz1(il, kl) < TM_Nic) then
                        ! qHeNu2 = 1.0 if W2xyz1(il,kl) < TM_Nic
                        ! qHeNu2 = 0.0 otherwise
                        signHN = -sign(unun, W2xyz1(il, kl) - TM_Nic)
                        qHeNu2 = max(zero, signHN)
                        
                        qHeNuc = qHeNu1 * qHeNu3
                        
                        ! Contact-Freez Potent.Nuclei
                        ! Meyers et al. (1992) JAM, 2.6
                        qicnd1 = 1.e3 * qHeNuc &
                                * exp(aM_Nic - bM_Nic * W2xyz1(il, kl))
                        rad_ww = (1.e3 * rolv2D(il, kl) &
                                * qw2D(il, kl) * .2e-11)**0.33
                        ! .2 e-11 = 1. / (1.2e+8       * 1.e3  * 4.19)
                        !                 ccnw2D (ASTEX) ro_w   4 pi /3
                        ! Levkov et al. 1992 CAM, (23)
                        qicnd2 = 603.2e+3 * qicnd1 * rad_ww &
                                * rolv2D(il, kl)
                        ! 603.2e3 =   4.0e-7  *  4 pi   * 1.2e+8  * 1.e3
                        !             DFar                ccnw2D    fact(rolv)
                        ccni2D(il, kl) = ccni2D(il, kl) + qicnd2
                        dqi = 1.e-15 * qicnd2 / rolv2D(il, kl)
                        !     1.e-15  =  1.0e-3 * Ice Crystal Mass
                        dqi = min(qw2D(il, kl), dqi)
                        ! XF 09/07/2019,too much qi vs qw
                        if(dqi > 0) dqi = dqi / 2.
                        qi2D(il, kl) = qi2D(il, kl) + dqi
                        qw2D(il, kl) = qw2D(il, kl) - dqi
                        tair2D(il, kl) = tair2D(il, kl) + dqi * r_LcCp
                        dqi2 = dqi
                    endif
                endif
#ifdef HM
                ! Nucleation II: Hallett-Mossop Ice-Multiplication Proc. ! BSPRWI
                ! Reference: Levkov et al., 1992, Contr.Atm.Ph.65,(25) p.40 ! Levkov (25) p.40
                ! -----------------------------------------------------------
                if(W2xyz1(il, kl) < TmxNhm .and. &
                        W2xyz1(il, kl) > TmnNhm .and. &
                        wair2D(il, kl) > w_svrl) then
                    ! qHeNu1 = 1.0 if W2xyz1(il,kl) < TmxNhm
                    ! qHeNu1 = 0.0 otherwise
                    signHN = -sign(unun, W2xyz1(il, kl) - TmxNhm)
                    qHeNu1 = max(zero, signHN)
                    ! qHeNu2 = 1.0 if W2xyz1(il,kl) > TmnNhm
                    ! qHeNu2 = 0.0 otherwise
                    signHN = sign(unun, W2xyz1(il, kl) - TmnNhm)
                    qHeNu2 = max(zero, signHN)
                    ! qHeNu3 = 1.0 if wair2D(il,kl) > w_svrl
                    ! qHeNu3 = 0.0 otherwise
                    signHN = sign(unun, wair2D(il, kl) - w_svrl)
                    qHeNu3 = max(zero, signHN)
#ifdef cn
                    cnos = min(2.e8, &
                            cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                    SplinJ = 1.358e12 * qw2D(il, kl) * cnos / &
                            (W2xyz8(il, kl)**.33)
                    ! 1.358e12=pi   *Gamma(3.5) *g   *ro_s /(3 *Cd  *4.19e-12)
                    !        [=3.14 *3.3233625  *9.81*0.1  /(3 *0.6 *4.19e-12)]
                    SplinP = 0.003 * (1. - 0.05 * SplinJ) * qHeNu1 * qHeNu2 &
                            * qHeNu3
                    SplinP = max(zero, SplinP)
                    dqi = 1.e-15 * SplinP / &
                            rolv2D(il, kl)
                    ! 1.e-15  =  1.0e-3 * Ice Crystal Mass
                    SplinP = (min(1.0, qs2D(il, kl) / max(dqi, eps9))) * SplinP
                    ccni2D(il, kl) = ccni2D(il, kl) + SplinP
                    dqi = min(qs2D(il, kl), dqi)
                    qi2D(il, kl) = qi2D(il, kl) + dqi
                    dqi2D(il, kl) = dqi2D(il, kl) + dqi
                    qs2D(il, kl) = qs2D(il, kl) - dqi
                    dqi3 = dqi
                endif
#endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Meyers: Nucl. I, Depot & Cond-Freez'
                debugH(36:70) = 'Nucl. / Freez / Nucl. II / Bergeron'
                proc_1 = 'dQi1 Meyer'
                procv1 = dqi1
                proc_2 = 'dQi2 Meyer'
                procv2 = dqi2
                proc_3 = 'dQi Ha-Mos'
                procv3 = dqi3
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(02, kl) = dqi1 + dqi2 + dqi3
#endif
            enddo
        enddo
        !=======================================================================
    else
        
        !=======================================================================
        !  Ice Crystals Nucleation Process between 0.C and -35.C
        !  (each crystal has a mass equal or less than 10d-12 kg)
        !  Reference: Emde and Kahlig 1989, Ann.Geoph. 7, p.408 (13)
        !  ---------------------------------------------------------
        
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qicnd1 = 0.
                qicnd2 = 0.
                qicnd = 0.
#endif
                if(W2xyz1(il, kl) > tsfo) then
                    
                    ! qHeNu1 = 1.0 if W2xyz1(il,kl)  > tsfo
                    ! qHeNu1 = 0.0 otherwise
                    signHN = sign(unun, W2xyz1(il, kl) - tsfo)
                    qHeNu1 = max(zero, signHN)
                    
                    if(W2xyz1(il, kl) < 0.) then
                        
                        ! qHeNu2 = 1.0 if W2xyz1(il,kl)  < 0.
                        ! qHeNu2 = 0.0 otherwise
                        signHN = -sign(unun, W2xyz1(il, kl))
                        qHeNu2 = max(zero, signHN)
                        
                        if(qv2D(il, kl) > W2xyz5(il, kl)) then
                            
                            ! qHeNu3 = 1.0 if qv2D(il,kl)  > W2xyz5(il,kl)
                            ! qHeNu3 = 0.0 otherwise
                            signHN = sign(unun, qv2D(il, kl) - W2xyz5(il, kl))
                            qHeNu3 = max(zero, signHN)
                            
                            qHeNuc = qHeNu1 * qHeNu2 * qHeNu3
#ifdef EW
                            if(qHeNuc > epsi) then ! ctr
                                mauxEW = mphy2D(il)
                                mauxEW(03:03) = 'I'
                                mphy2D(il) = mauxEW
                            ENDif             ! ctr
#endif
                            ! qicnd1 : amount of nucleated ice crystals
                            ! (first condition)
                            qicnd1 = qHeNuc * 1.e-15 * W2xyz2(il, kl) / &
                                    rolv2D(il, kl)
                            
                            qisign = sign(unun, qicnd1 - qi2D(il, kl))
                            qi1_OK = max(zero, qisign)
                            qicnd1 = qicnd1 * qi1_OK
                            
                            ! qicnd2 : amount of nucleated ice crystals
                            ! (second condition)
                            qicnd2 = (qv2D(il, kl) - W2xyz5(il, kl)) / &
                                    (1.d0 + 1.733d7 * W2xyz5(il, kl) / &
                                            (tair2D(il, kl) * tair2D(il, kl)))
                            qicnd2 = qHeNuc * max(zero, qicnd2)
                            
                            qicnd = min(qicnd1, qicnd2)
                            
                            qi2D(il, kl) = qi2D(il, kl) + qicnd
                            dqi2D(il, kl) = dqi2D(il, kl) + qicnd
                            ccni2D(il, kl) = ccni2D(il, kl) + rolv2D(il, kl) * &
                                    qicnd * 1.e15
                            qv2D(il, kl) = qv2D(il, kl) - qicnd
                            tair2D(il, kl) = tair2D(il, kl) + r_LsCp * qicnd
#ifdef WQ
                            !  Full Debug
                            !  ~~~~~~~~~~
                            write(6, *) 'QiCnd', qicnd, &
                                    ' Qi', qi2D(il, kl), &
                                    ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
                            if(il == ilmm) wicnd(kl) = qicnd
#endif
#ifdef wH
                            !  Debug
                            !  ~~~~~
                            debugH(1:35) = 'Emde and Kahlig: Ice Crystals Nucle'
                            debugH(36:70) = 'ation Process between 0.C and -35.C'
                            proc_1 = 'Qicnd1    '
                            procv1 = qicnd1
                            proc_2 = 'Qicnd2    '
                            procv2 = qicnd2
                            proc_3 = 'Qicnd g/kg'
                            procv3 = qicnd
                            proc_4 = '          '
                            procv4 = 0.
                            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                                    debugV(02, kl) = qicnd
#endif
                        endif
                    endif
                endif
            enddo
        enddo
    endif
    
    !===========================================================================
    !  Bergeron Process (water vapor diffusion-deposition on ice crystals)
    !  Reference: Koenig          1971, J.A.S.    28, p.235
    !             Emde and Kahlig 1989, Ann.Geoph. 7, p.408 (14)
    !  ---------------------------------------------------------
    
    if(.not. LevkovAUTX) then
        
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qBerge = 0.
                qidep = 0.
                qicnd = 0.
#endif
                if(qi2D(il, kl) > eps9 .and. W2xyz1(il, kl) < 0.) then
                    
                    ! qBerge = 1.0 if qi2D(il,kl)  > eps9
                    ! qBerge = 0.0 otherwise
                    signHN = sign(unun, qi2D(il, kl) - eps9)
                    qBerge = max(zero, signHN)
                    
                    ! qHeNuc = 1.0 if   W2xyz1(il,kl)  < 0.
                    ! qHeNuc = 0.0 otherwise
                    signHN = -sign(unun, W2xyz1(il, kl))
                    qHeNuc = max(zero, signHN)
                    
                    qBerge = qHeNuc * qBerge
#ifdef EW
                    if(qBerge > epsi) then ! ctr
                        mauxEW = mphy2D(il)
                        mauxEW(04:04) = 'i'
                        mphy2D(il) = mauxEW
                    ENDif            ! ctr
#endif
                    itc = abs(W2xyz1(il, kl) - unun)
                    itc = min(itc, 31)
                    itc = max(itc, 1)
                    a1 = aa1(itc)
                    a2 = aa2(itc)
                    
                    ! amf   :  analytical integration of
                    ! (14) p.408 Emde and Kahlig 1989, Ann.Geoph. 7
                    am0 = 1.d+3 * rolv2D(il, kl) * qi2D(il, kl) / W2xyz2(il, kl)
                    amf = (a1 * (1.0 - a2) * xt + am0**(1.0 - a2))**(1.0 / (1.0 - a2))
                    qidep = (1.d-3 * W2xyz2(il, kl) / rolv2D(il, kl)) * (amf - am0)
                    qidep = max(zero, qidep)
                    
                    ! qicnd :  to avoid the use of qw2D < 0.
                    qicnd = max(zero, qw2D(il, kl))
                    
                    qidep = qBerge * min(qicnd, qidep)
                    
                    ! XF 09/07/2019,too much qi vs qw
                    if(qidep > 0) qidep = qidep / 2.
                    
                    ! ccni2D(il,kl): NO VARIATION
                    qi2D(il, kl) = qi2D(il, kl) + qidep
                    
                    qw2D(il, kl) = qw2D(il, kl) - qidep
                    tair2D(il, kl) = tair2D(il, kl) + r_LcCp * qidep
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'QiDep', qidep, &
                            ' Qi', qi2D(il, kl), &
                            ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) widep(kl) = qidep
#endif
#ifdef wH
                    !  Debug
                    !  ~~~~~
                    debugH(1:35) = 'Bergeron Process (water vapor diffu'
                    debugH(36:70) = 'sion-deposition on ice crystals)   '
                    proc_1 = 'qBerge ICE'
                    procv1 = qBerge
                    proc_2 = 'Qicnd g/kg'
                    procv2 = qicnd
                    proc_3 = 'Qidep g/kg'
                    procv3 = qidep
                    proc_4 = '          '
                    procv4 = 0.
                    if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                            debugV(02, kl) = qidep + debugV(02, kl)
#endif
                endif
            enddo
        enddo
    endif
    
    !===========================================================================
    
    !  Ice Crystals Sublimation                     ! BDEPVI
    !  Reference: Emde and Kahlig, 1989 p.408 (15)  ! Levkov (27) p.40
    !  -------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qisub = 0.
#endif
            if(W2xyz5(il, kl) > qv2D(il, kl)) then
                qvdfci = W2xyz5(il, kl) - qv2D(il, kl)
                !     qSubl1 = 1.0 if   W2xyz5(il,kl) >  qv2D(il,kl)
                !            = 0.0 otherwise
                !!#pp           signHN = sign(unun,qvdfci)
                !!#pp           qSubl1 = max(zero,signHN)
                if(qi2D(il, kl) > eps9) then
                    ! qSubl2 = 1.0 if qi2D(il,kl)  > eps9
                    ! qSubl2 = 0.0 otherwise
                    signHN = sign(unun, qi2D(il, kl) - eps9)
                    qSubl2 = max(zero, signHN)
                    
                    qSubli = qSubl2
                    !!#pp               qSubli = qSubli * qSubl1
#ifdef EW
                    if(qSubli > epsi) then ! ctr
                        mauxEW = mphy2D(il)
                        mauxEW(05:05) = 'V'
                        mphy2D(il) = mauxEW
                    ENDif            ! ctr
#endif
                    demde = 1.1d+4
                    sat = qv2D(il, kl) / W2xyz5(il, kl)
                    ab1 = 6.959d+11 / (tair2D(il, kl) * tair2D(il, kl))
                    !     6.959e+11
                    != [Ls=2833600J/kg] * Ls / [kT=0.025W/m/K] / [Rv=461.J/kg/K]
                    ! kT: Air thermal Conductivity
                    ab2 = 1.d0 / (1.875d-2 * rolv2D(il, kl) * W2xyz5(il, kl))
                    !            1.875d-5: Water Vapor Diffusivity in Air !CAa WARNING
                    pisub = (1 - sat) * 4.d0 * demde * W2xyz2(il, kl) / (ab1 + ab2)
                    qisub = pisub * xt
                    ! H2O deposition limit = H2O content
                    qisub = max(qisub, -qv2D(il, kl))
                    ! qi sublimation limit = qi content
                    qisub = min(qisub, qi2D(il, kl))
                    ! qi sublimation limit = Saturation
                    qisub = min(qisub, qvdfci) * qSubli
                    
                    qi2D(il, kl) = qi2D(il, kl) - qisub
                    dqi2D(il, kl) = dqi2D(il, kl) - qisub
                    qv2D(il, kl) = qv2D(il, kl) + qisub
                    tair2D(il, kl) = tair2D(il, kl) - r_LsCp * qisub

#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'QiSub', qisub, &
                            ' Qi', qi2D(il, kl), &
                            ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wisub(kl) = qisub
#endif
                
                endif
            endif
#ifdef wH
            ! +--Debug
            ! +  ~~~~~
            debugH(1:35) = 'Emde and Kahlig: Ice Crystals Subli'
            debugH(36:70) = 'mation                             '
            proc_1 = 'Qisub g/kg'
            procv1 = qisub
            proc_2 = 'R.Hum I[%]'
            procv2 = 0.1 * sat
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(03, kl) = -qisub
#endif
        enddo
    enddo
    
    do kl = mzhyd, klev
        do il = 1, klon
            if(qi2D(il, kl) <= 0.) then
                qi2D(il, kl) = 0.
                ccni2D(il, kl) = 0.
            endif
        enddo
    enddo
    
    !===========================================================================
    
    !  Ice Crystals Instantaneous Melting
    !  ----------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qimlt = 0.
            cimlt = 0.
#endif
            if(W2xyz1(il, kl) > 0.) then
                
                ! qMelt1 = 1.0 if W2xyz1(il,kl) >  0.
                ! qMelt1 = 0.0 otherwise
                signHN = sign(unun, W2xyz1(il, kl))
                qMelt1 = max(zero, signHN)
                
                if(qi2D(il, kl) > eps9) then
                    
                    ! qMelt2 = 1.0 if     qi2D(il,kl) >  eps9
                    ! qMelt2 = 0.0 otherwise
                    signHN = sign(unun, qi2D(il, kl) - eps9)
                    qMelt2 = max(zero, signHN)
                    
                    qMelt = qMelt1 * qMelt2
#ifdef EW
                    ! ctr
                    if(qMelt > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(06:06) = 'w'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    qxmlt = W2xyz1(il, kl) / r_LcCp
                    qimlt = min(qi2D(il, kl), qxmlt) * qMelt
                    cimlt = ccni2D(il, kl) * qimlt / max(qi2D(il, kl), eps9)
                    qi2D(il, kl) = qi2D(il, kl) - qimlt
                    ccni2D(il, kl) = ccni2D(il, kl) - cimlt
                    qw2D(il, kl) = qw2D(il, kl) + qimlt
                    tair2D(il, kl) = tair2D(il, kl) - r_LcCp * qimlt
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'QiMlt', qimlt, &
                            ' Qi', qi2D(il, kl), &
                            ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wimlt(kl) = qimlt
#endif
                
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Emde and Kahlig: Ice Crystals Insta'
            debugH(36:70) = 'ntaneous Melting                   '
            proc_1 = 'Qimlt g/kg'
            procv1 = qimlt
            proc_2 = 'cimlt /e15'
            procv2 = cimlt * 1.e-18
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(04, kl) = -qimlt
#endif
        enddo
    enddo
    
    !================================================================
    !  Water Vapor Condensation / Evaporation (Fractional Cloudiness)
    !  Reference: Laurent Delobbe Thesis (Ek&Mahrt91)
    !  --------------------------------------------------------------
    
    ! Zeroing needed since cfra2D build from a maximization process
    do kl = mzhyd, klev
        do il = 1, klon
            ! cfra2D: Cloud Fraction
            cfra2D(il, kl) = 0.0
        enddo
    enddo
    
    if(fracld .and. fracSC) then
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                dqw = 0.
#endif
                if(W2xyz1(il, kl) >= tsfo) then
#ifdef EW
                    if(W2xyz1(il, kl) >= tsfo) then ! ctr
                        mauxEW = mphy2D(il)
                        mauxEW(07:07) = 'W'
                        mphy2D(il) = mauxEW
                    ENDif                    ! ctr
#endif
                    ! qHeNu1 = 1.0 if   W2xyz1(il,kl) > tsfo
                    ! qHeNu1 = 0.0 otherwise
                    signHN = sign(unun, W2xyz1(il, kl) - tsfo)
                    qHeNu1 = max(zero, signHN)
                    
                    ! qt : Total Water Mixing Ratio
                    qt = qv2D(il, kl) + qw2D(il, kl)
                    
                    ! tl : Liquid Temperature
                    tl = tair2D(il, kl) - r_LvCp * qw2D(il, kl)
                    
                    !  Saturation specific humidity over water,
                    !  corresponding to liquid temperature
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    ! Dudhia (1989) JAS, (B1) and (B2) p.3103
                    ! See also Pielke (1984), p.234 and Stull (1988), p.276
                    pa_hPa = (pst2Dn(il) * sigma(kl) + ptopDY) * 10.d0
                    es_hPa = 6.1078d0 * exp(ExpWat * log(WatIce / tl)) &
                            * exp(ExpWa2 * (unun / WatIce - unun / tl))
                    
                    ! Saturation Vapor Specific Concentration over Water
                    ! (even for temperatures less than freezing point)
                    qsl = .622d0 * es_hPa / (pa_hPa - .378d0 * es_hPa)
                    
                    !  Partial Condensation/Scheme
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    dqt = qv2D(il, MIN(kl + 1, klev)) - qv2D(il, kl) &
                            + qw2D(il, MIN(kl + 1, klev)) - qw2D(il, kl)
                    wqt = TUkv2D(il, kl) * dqt &
                            / (gplv2D(il, kl + 1) - gplv2D(il, kl)) * gravit
                    
                    ! ww : Vertical Velocity Variance
                    ww = 0.66d0 * ect_2D(il, kl)
                    
                    ! sig2rh : Relative Humidity Variance
                    ! (Ek and Mahrt, 1991, An. Geoph., 9, 716--724)
                    coefC2 = wqt / (sqrt(ww) * qsl)
                    sig2rh = C1_EkM + C2_EkM * coefC2 * coefC2
                    
                    ! sigqt : Total Water Variance
                    sigqt = sqrt(sig2rh) * qsl
                    
                    argerf = (qt - qsl) / (1.414d0 * sigqt)
                    err = erf(argerf)
                    
                    ! cfra2D: Cloud Fraction
                    cfra2D(il, kl) = 0.5d0 * (1.d0 + err)
                    
                    alpha = 1.d0 / (1.d0 + 1.349d7 * qsl / (tl * tl))
                    t1 = sigqt / sqrt(pi + pi) * exp(-min(argerf * argerf &
                            , argmax))
                    t2 = cfra2D(il, kl) * (qt - qsl)
                    
                    ! cfraOK = 1.0 if  cfra2D(il,kl) > cminHY
                    ! cfraOK = 0.0 otherwise
                    signFR = sign(unun, cfra2D(il, kl) - cminHY)
                    cfraOK = max(zero, signFR)
                    
                    ! qw_new : Mesh Averaged Liquid Water Mixing Ratio
                    cfra2D(il, kl) = cfra2D(il, kl) * cfraOK * qHeNu1
                    qw_new = alpha * (t1 + t2) * cfraOK
                    
                    dqw = qw_new - qw2D(il, kl)
                    
                    !  Vectorisation of the Atmospheric Water Update
                    !  ~~~+-------------------------------------------+
                    !     |       if (dqw > 0.) then      |
                    !     |        dqw = min(qv2D(il,kl), dqw)        |
                    !     |       else                                |
                    !     |        dqw =-min(qw2D(il,kl),-dqw)        |
                    !     |       end if                         |
                    !     +-------------------------------------------+
                    
                    signdq = sign(unun, dqw)
                    fac_qv = max(zero, signdq)
                    updatw = fac_qv * qv2D(il, kl) &
                            + (1.d0 - fac_qv) * qw2D(il, kl)
#ifdef kk
                    ! SCu Limitor
                    SCuLim = exp(min(0., 300. - tair2D(il, kl)))
#endif
                    dqw = signdq * min(updatw, signdq * dqw) &
                            * qHeNu1
#ifdef kk
                    ! SCu Limitor
                    dqw = dqw * SCuLim
                    cfra2D(il, kl) = cfra2D(il, kl) * SCuLim
#endif
                    !  Update of qv2D, qw2D and tair2D
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    qw2D(il, kl) = qw2D(il, kl) + dqw
                    dqw2D(il, kl) = dqw2D(il, kl) + dqw
                    qv2D(il, kl) = qv2D(il, kl) - dqw
                    tair2D(il, kl) = tair2D(il, kl) + r_LvCp * dqw
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'QwEvp', dqw, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wwevp(kl) = dqw
#endif
                endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Delobbe: Condensation              '
                debugH(36:70) = '                                   '
                proc_1 = 'dQw   g/kg'
                procv1 = dqw
                proc_2 = '          '
                procv2 = 0.
                proc_3 = '          '
                procv3 = 0.
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(05, kl) = dqw
#endif
            enddo
        enddo
    else
        !===========================================================
        !  Water Vapor Condensation / Evaporation
        !  Reference: Emde and Kahlig 1989, Ann.Geoph. 7, p.407 (7)
        !  --------------------------------------------------------
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                dqw = 0.
#endif
                if(W2xyz1(il, kl) >= tsfo) then
#ifdef EW
                    ! ctr
                    if(W2xyz1(il, kl) > tsfo) then
                        mauxEW = mphy2D(il)
                        mauxEW(07:07) = 'W'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    ! qHeNu1 = 1.0 if   W2xyz1(il,kl) > tsfo
                    ! qHeNu1 = 0.0 otherwise
                    signHN = sign(unun, W2xyz1(il, kl) - tsfo)
                    qHeNu1 = max(zero, signHN)
                    
                    dpw = (qv2D(il, kl) - qvsw2D(il, kl) * rhcrHY) / &
                            (1.d0 + 1.349d7 * qvsw2D(il, kl) / &
                                    (tair2D(il, kl) * tair2D(il, kl)))
                    !   1.349e7=Lv*Lv*0.622/Cpa/Ra with Lv = 2500000 J/kg
                    
                    dqw = dpw
                    
                    !  Vectorisation of the Atmospheric Water Update
                    !  ~~+-------------------------------------------+
                    !    |       if (dqw > 0.) then      |
                    !    |        dqw = min(qv2D(il,kl), dqw)        |
                    !    |       else                                |
                    !    |        dqw =-min(qw2D(il,kl),-dqw)        |
                    !    |       end if                         |
                    !    +-------------------------------------------+
                    
                    signdq = sign(unun, dqw)
                    fac_qv = max(zero, signdq)
                    updatw = fac_qv * qv2D(il, kl) &
                            + (1.d0 - fac_qv) * qw2D(il, kl)
                    dqw = signdq * min(updatw, signdq * dqw) &
                            * qHeNu1
                    
                    !  Update of qv2D, qw2D and tair2D
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    qw2D(il, kl) = qw2D(il, kl) + dqw
                    dqw2D(il, kl) = dqw2D(il, kl) + dqw
                    qv2D(il, kl) = qv2D(il, kl) - dqw
                    tair2D(il, kl) = tair2D(il, kl) + r_LvCp * dqw
                    ! [Ls=2500000J/kg]/[Cp=1004J/kg/K]=2490.04
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'QwEvp', dqw, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wwevp(kl) = dqw
#endif
                endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Emde and Kahlig: Water Vapor Conden'
                debugH(36:70) = 'sation / Evaporation               '
                proc_1 = 'dQw   g/kg'
                procv1 = dqw
                proc_2 = '          '
                procv2 = 0.
                proc_3 = '          '
                procv3 = 0.
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(05, kl) = dqw
#endif
            enddo
        enddo
    endif
    
    !===========================================================================
    
    !  Fractional  Cloudiness ! Guess may be computed (Ek&Mahrt91 fracSC=.T.)
    !  ====================== ! Final value  computed  below
    
    !!#sc if (fracld .and..not. fracSC) then
    if(fracld) then
        if(fraCEP) then
            ! ECMWF Large Scale Cloudiness
            ! ----------------------------
            do kl = mzhyd, klev
                do il = 1, klon
                    cfra2D(il, kl) = &
                            (qi2D(il, kl) + qw2D(il, kl) + qs2D(il, kl) * 0.33 &
                                    * (1. - min(1., exp((tair2D(il, kl) - 258.15) * 0.1)))) &
                                    / (0.02 * qvsw2D(il, kl))
                    cfra2D(il, kl) = min(1.000, cfra2D(il, kl))
                    cfra2D(il, kl) = max(0.001, cfra2D(il, kl)) &
                            * max(0., sign(1., &
                                    qi2D(il, kl) + qw2D(il, kl) + qs2D(il, kl) - 3.E-9))
                enddo
            enddo
        else
            ! XU and Randall  1996, JAS 21, p.3099 (4)
            ! ----------------------------
            do kl = mzhyd, klev
                do il = 1, klon
                    qvs_wi = qvsw2D(il, kl)
#ifdef wi
                    qvs_wi = max(eps9, ((qi2D(il, kl) + qs2D(il, kl)) * qvsi2D(il, kl) &
                            + qw2D(il, kl) * qvsw2D(il, kl)) / &
                            max(eps9, qi2D(il, kl) + qs2D(il, kl) + qw2D(il, kl)))
#endif
                    relhum = min(relCri, max(qv2D(il, kl), qv_MIN) / qvs_wi)
                    argexp = ((relCri - relhum) * qvs_wi)**0.49
                    argexp = min(100. * &
                            (qi2D(il, kl) + qw2D(il, kl) + qs2D(il, kl) * 0.33 &
                                    * (1. - min(1., exp((tair2D(il, kl) - 258.15) * 0.1)))) &
                            / max(eps9, argexp), argmax)
                    
                    cfra2D(il, kl) = (relhum**0.25) * (1. - exp(-argexp))
                enddo
            enddo
        endif
    
    else
        !!#sc   else if (.not.fracld) then
        !!#sc   if (fracSC) stop 'fracSC set up when fracld NOT'
        do kl = mzhyd, klev
            do il = 1, klon
                qcloud = qi2D(il, kl) + qw2D(il, kl)
                if(qcloud > eps9) then
                    
                    ! cfra2D(il,kl) = 1.0 if qcloud > eps9
                    ! cfra2D(il,kl) = 0.0 otherwise
                    signQW = sign(unun, qcloud - eps9)
                    cfra2D(il, kl) = max(zero, signQW)
                
                endif
            enddo
        enddo
    
    endif

#ifdef wH
    !  Debug
    !  ~~~~~
    do kl = mzhyd, klev
        do il = 1, klon
            debugH(1:35) = 'Fractional Cloudiness (XU .OR. CEP)'
            debugH(36:70) = '                                   '
            proc_1 = '          '
            procv1 = 0.
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
        enddo
    enddo
#endif
    
    !===========================================================================
    
    !  Autoconversion Processes (i.e., generation of precipitating particles)
    !  ======================================================================
    
    !  Cloud Droplets Autoconversion
    !  Reference: Lin et al. (1983), JCAM 22, p.1076 (50)
    !  --------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qraut = 0.0
#endif
            if(qw2D(il, kl) > eps9) then
                
                ! AutoOK = 1.0 if qw2D(il,kl)  > eps9
                ! AutoOK = 0.0 otherwise
                signAU = sign(unun, qw2D(il, kl) - eps9)
                AutoOK = max(zero, signAU)
                
                if(cfra2D(il, kl) > cminHY) then
                    
                    ! ClouOK = 1.0 if cfra2D(il,kl)  > cminHY
                    ! ClouOK = 0.0 otherwise
                    signFC = sign(unun, cfra2D(il, kl) - cminHY)
                    ClouOK = max(zero, signFC)
                    
                    AutoOK = AutoOK * ClouOK
#ifdef EW
                    ! ctr
                    if(AutoOK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(08:08) = 'r'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    ! Sundqvist (1988, Schlesinger, Reidel, p.433)
                    ! Autoconversion Scheme
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    dqw = AutoOK * qw2D(il, kl) / &
                            qw00 / max(cminHY, cfra2D(il, kl))
                    praut = AutoOK * qw2D(il, kl) * csud * &
                            (1. - exp(-min(dqw * dqw, argmax))) &
                            / max(cminHY, cfra2D(il, kl))
#ifdef LO
                    ! Liou and Ou (1989, JGR  94, p. 8599) Autoconversion Scheme
                    ! Boucher et al. (1995, JGR 100, p.16395)
                    ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    ! ASTEX (Duynkerke&al.1995, JAS 52, p.2763)
                    !       (polluted air, Rogers&Yau 89, p.90)
                    ccnw2D(il, kl) = 1.2e+8
                    ccnw2D(il, kl) = 1.e+11
                    qwclou = qw2D(il, kl) / cfra2D(il, kl)
                    dmed0 = 4.5d0 * sigmaw * sigmaw
                    dmede = qwclou * rolv2D(il, kl) &
                            * 6.d0 / pi / ccnw2D(il, kl) / exp(dmed0)
                    dmed = exp(third * log(dmede))
                    dmed2 = dmed * dmed
                    dw0 = 8.d0 * sigmaw * sigmaw
                    dw4 = exp(dw0) * dmed2 * dmed2
                    rwbar = 0.5d0 * sqrt(sqrt(dw4))
                    ! heavi : Heaviside Function
                    signHV = sign(unun, rwbar - rcrilo)
                    heavi = max(zero, signHV)
                    praut = AutoOK * cfra2D(il, kl) * heavi * 4.09d6 * pi &
                            * ccnw2D(il, kl) * dw4 * qwclou
#endif
#ifdef LI
                    !  Lin et al.(1983) Autoconversion Scheme
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    dqw = AutoOK * (qw2D(il, kl) - qw00)
                    praut = dqw * dqw * dqw / &
                            (cc1 * dqw + 1000.d0 * cc2 / dd0)
#endif
                    qraut = praut * xt
                    if(qraut > 0) qraut = qraut * min(0.9, (1. - cloud_magic))
                    qraut = min(qraut, qw2D(il, kl))
                    qw2D(il, kl) = qw2D(il, kl) - qraut
                    qr2D(il, kl) = qr2D(il, kl) + qraut

#ifdef WQ
                    write(6, *) 'QrAut', qraut, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wraut(kl) = qraut
#endif
                
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983) Autoconversion Sch'
            debugH(36:70) = 'eme                                '
            proc_1 = 'Qraut g/kg'
            procv1 = qraut
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(06, kl) = qraut
#endif
        enddo
    enddo
    
    !  Conversion from Cloud Ice Crystals to Snow Flakes
    !  Reference: Levkov et al.   1992, Contr.Atm.Phys. 65, p.41
    !  ---------------------------------------------------------
    
    if(LevkovAUTO) then
        
        !  Depositional Growth: Ice Crystals  => Snow Flakes     (BDEPIS)
        !  Reference: Levkov et al.   1992, Contr.Atm.Phys. 65, p.41 (28)
        !  --------------------------------------------------------------
        
        if(LevkovAUTX) then
            
            do kl = mzhyd, klev
                do il = 1, klon
#ifdef wH
                    qsaut = 0.0
#endif
                    if(qi2D(il, kl) > eps9) then
                        
                        ! AutoOK = 1.0 if qi2D(il,kl)  > eps9
                        ! AutoOK = 0.0 otherwise
                        signAU = sign(unun, qi2D(il, kl) - eps9)
                        AutoOK = max(zero, signAU)
                        
                        if(ccni2D(il, kl) > 1.) then
                            
                            ! ClouOK = 1.0 if   ccni2D(il,kl)  > 1.
                            ! ClouOK = 0.0 otherwise
                            signCC = sign(unun, ccni2D(il, kl) - 1.)
                            ClouOK = max(zero, signCC)
                            
                            AutoOK = AutoOK * ClouOK
                            qiOK = AutoOK * qi2D(il, kl)
                            
                            !  Pristine Ice Crystals Diameter
                            !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                            ! qid : Pristine Ice Crystals Diameter
                            !       Levkov et al. 1992,
                            !       Contr. Atm. Phys. 65, (5) p.37
                            !       where 6/(pi*ro_I)**1/3 ~ 0.156
                            qid = 0.156 * exp(third * log(thous * rolv2D(il, kl) &
                                    * max(eps9, qi2D(il, kl)) &
                                    / max(unun, ccni2D(il, kl))))
                            
                            !  Deposition Time Scale
                            !  ~~~~~~~~~~~~~~~~~~~~~
                            sat = max(epsq, qv2D(il, kl)) / W2xyz5(il, kl)
                            ! 0.702e12 ~ 0.702e12 = (2.8345e+6)**2/0.0248/461.5
                            !                        Ls_H2O    **2/Ka    /Rw
                            ! Dv = 2.36e-2 = 2.36e-5 * 10.**3
#ifdef a1
                            a1saut = max(eps9, sat - 1.) / &
                                    (0.702e12 / (tair2D(il, kl) * tair2D(il, kl)) &
                                            + 1. / (2.36e-2 * rolv2D(il, kl) * qv2D(il, kl) * sat))
#endif
                            ! Dv = 2.36e-2 = 2.36e-5 * 10.**3
                            xtsaut = 0.125 * (qsd0 * qsd0 - qid * qid) &
                                    * (0.702e12 / (tair2D(il, kl) * tair2D(il, kl)) &
                                            + 1.0 / (2.36e-2 * rolv2D(il, kl) &
                                                    * max(epsq, qv2D(il, kl)) * sat))
                            
                            !  Deposition
                            !  ~~~~~~~~~~
                            qsaut = xt * qiOK * (sat - 1.) / xtsaut
                            qsaut = (1. - cloud_magic) * qsaut
                            qsaut = min(qi2D(il, kl), qsaut)
                            qsaut = max(-qs2D(il, kl), qsaut)
                            
                            if(.not.isnan(qsaut)) then
                                qi2D(il, kl) = qi2D(il, kl) - qsaut
                                qs2D(il, kl) = qs2D(il, kl) + qsaut
                            endif
                        
                        endif
                    endif
#ifdef wH
                    !  Debug
                    !  ~~~~~
                    debugH(1:35) = 'Lin et al.(1983) Depositional Growt'
                    debugH(36:70) = 'h                                  '
                    proc_1 = 'Qsaut g/kg'
                    procv1 = qsaut
                    proc_2 = '          '
                    procv2 = 0.
                    proc_3 = '          '
                    procv3 = 0.
                    proc_4 = '          '
                    procv4 = 0.
                    if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                            debugV(07, kl) = qsaut
#endif
                enddo
            enddo
        endif
        
        !  Ice Crystals Aggregation           => Snow Flakes     (BAGRIS)
        !  Reference: Levkov et al.   1992, Contr.Atm.Phys. 65, p.41 (31)
        !  --------------------------------------------------------------
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qsaut = 0.0
                xtsaut = 0.0
#endif
                if(qi2D(il, kl) > eps9) then
                    ! AutoOK = 0.0 otherwise
                    ! AutoOK = 1.0 if qi2D(il,kl)  > eps9
                    signAU = sign(unun, qi2D(il, kl) - eps9)
                    AutoOK = max(zero, signAU)
                    
                    if(ccni2D(il, kl) > 1.) then
                        
                        ! ClouOK = 1.0 if ccni2D(il,kl)  > 1.
                        ! ClouOK = 0.0 otherwise
                        signCC = sign(unun, ccni2D(il, kl) - 1.)
                        ClouOK = max(zero, signCC)
                        
                        AutoOK = AutoOK * ClouOK
                        qiOK = AutoOK * qi2D(il, kl)
#ifdef EW
                        if(AutoOK > epsi) then
                            mauxEW = mphy2D(il)
                            mauxEW(09:09) = 's'
                            mphy2D(il) = mauxEW
                        endif
#endif
                        !  Pristine Ice Crystals Diameter
                        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        !qid : Pristine Ice Crystals Diameter
                        !      Levkov et al. 1992
                        !      Contr. Atm. Phys. 65, (5) p.37
                        !      where [6/(pi*ro_I)]**1/3 ~ 0.156
                        qid = 0.156 * exp(third * log(thous * rolv2D(il, kl) &
                                * max(eps9, qi2D(il, kl)) &
                                / max(unun, ccni2D(il, kl))))
                        
                        !  Time needed for Ice Crystals Diameter to reach Snow Diameter Threshold
                        !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        c1saut = max(eps9, qiOK) * rolv2D(il, kl) * 35. &
                                * exp(third * log(rolv2D(il, klev) / &
                                        rolv2D(il, kl)))
                        
                        ! qi fully used if xtsaut<xt
                        xtsaut = -6.d0 * log(qid / qsd0) / c1saut
                        xtsaut = max(xt, xtsaut)
#ifdef nt
                        ! Time needed for Ice Crystals Diameter
                        ! to reach Snow Diameter Threshold
                        !  ~(ALTERNATE PARAMETERIZATION)~
                        xtsaut = -2.0 * (3.0 * log(qid / qsd0) &
                                + log(max(qi2D(il, kl), eps9))) / c1saut
                        xtsaut = max(eps9, xtsaut)
#endif
                        !  Aggregation
                        !  ~~~~~~~~~~~
                        qsaut = xt * qiOK / xtsaut
                        if(qsaut > 0) qsaut = qsaut * (1. - cloud_magic)
                        qsaut = min(qi2D(il, kl), qsaut)
                        qsaut = max(-qs2D(il, kl), qsaut)
                        if(.not.isnan(qsaut)) then
                            qi2D(il, kl) = qi2D(il, kl) - qsaut
                            qs2D(il, kl) = qs2D(il, kl) + qsaut
                            
                            !  Decrease of Ice Crystals Number (BAGRII)
                            !  Reference: Levkov et al.   1992
                            !   Contr.Atm.Phys. 65, p.41 (34)
                            !  ----------------------------------------
                            ccni2D(il, kl) = ccni2D(il, kl) * exp(-0.5 * c1saut * xt)
#ifdef WQ
                            write(6, *) 'QsAut', qsaut, &
                                    ' Qi', qi2D(il, kl), &
                                    ' CcnI', ccni2D(il, kl), itexpe, il, kl
#endif
#ifdef WH
                            if(il == ilmm) wsaut(kl) = qsaut
#endif
                        endif
                    endif
                endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Lin et al.(1983) Ice Crystals Aggre'
                debugH(36:70) = 'gation                             '
                proc_1 = 'xtsaut sec'
                procv1 = xtsaut
                proc_2 = 'Qsaut g/kg'
                procv2 = qsaut
                proc_3 = '          '
                procv3 = 0.
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(07, kl) = qsaut + debugV(07, kl)
#endif
            enddo
        enddo
        !=======================================================================
    else if(EmdeKa) then
        !  Ice Crystals Autoconversion => Snow Flakes
        !  Reference: Lin et al.      1983, JCAM      22, p.1070 (21)
        !             Lin et al.      1983, JCAM      22, p.1074 (38)
        !             Emde and Kahlig 1989, Ann.Geoph. 7, p. 408 (18)
        !  ----------------------------------------------------------
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qsaut = 0.0
                cnsaut = 0.0
#endif
                if(qi2D(il, kl) >= qi00) then
#ifdef EW
                    if(qi2D(il, kl) >= qi00) then
                        mauxEW = mphy2D(il)
                        mauxEW(09:09) = 's'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    ex1 = 0.025d0 * W2xyz1(il, kl)      ! W2 = t?[K]
                    psaut = 0.001d0 * (qi2D(il, kl) - qi00) * exp(ex1)
                    qsaut = psaut * xt
                    qsaut = qsaut * (1. - cloud_magic)
                    qsaut = max(qsaut, zero)
                    qsaut = min(qsaut, qi2D(il, kl))
                    cnsaut = ccni2D(il, kl) * qsaut &
                            / max(qi00, qi2D(il, kl))
                    ccni2D(il, kl) = ccni2D(il, kl) - cnsaut
                    qi2D(il, kl) = qi2D(il, kl) - qsaut
                    qs2D(il, kl) = qs2D(il, kl) + qsaut
#ifdef WQ
                    write(6, *) 'QsAut', qsaut, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wsaut(kl) = qsaut
#endif
                endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Emde and Kahlig  Ice Crystals Autoc'
                debugH(36:70) = 'onversion                          '
                proc_1 = 'Qsaut g/kg'
                procv1 = qsaut
                proc_2 = 'cnsaut/e15'
                procv2 = cnsaut * 1.e-18
                proc_3 = '          '
                procv3 = 0.
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(07, kl) = qsaut
#endif
            enddo
        enddo
    else
        !  Sundqvist (1988, Schlesinger, Reidel, p.  433) Autoconversion Scheme
        !  --------------------------------------------------------------------
        do kl = mzhyd, klev
            do il = 1, klon
#ifdef wH
                qsaut = 0.0
                cnsaut = 0.0
#endif
                if(qi2D(il, kl) > eps9) then
                    
                    ! AutoOK = 1.0 if qi2D(il,kl)  > eps9
                    ! AutoOK = 0.0 otherwise
                    signAU = sign(unun, qi2D(il, kl) - eps9)
                    AutoOK = max(zero, signAU)
                    
                    dqi = AutoOK * qi2D(il, kl) / qi0S
#ifdef mf
                    dqi = dqi / max(cminHY, cfra2D(il, kl))
#endif
                    psaut = AutoOK * qi2D(il, kl) * csud &
                            * (1. - exp(-dqi * dqi))
#ifdef mf
                    psaut = psaut * max(cminHY, cfra2D(il, kl))
#endif
                    qsaut = psaut * xt
                    qsaut = (1. - cloud_magic) * qsaut
                    qsaut = min(qi2D(il, kl), qsaut)
                    qsaut = max(zero, qsaut)
                    cnsaut = ccni2D(il, kl) * qsaut &
                            / max(qi2D(il, kl), eps9)
                    ccni2D(il, kl) = ccni2D(il, kl) - cnsaut
                    qi2D(il, kl) = qi2D(il, kl) - qsaut
                    qs2D(il, kl) = qs2D(il, kl) + qsaut
                
                endif
#ifdef wH
                !  Debug
                !  ~~~~~
                debugH(1:35) = 'Sundqvist (1988) Ice Crystals Autoc'
                debugH(36:70) = 'onversion                          '
                proc_1 = 'Qsaut g/kg'
                procv1 = qsaut
                proc_2 = 'cnsaut/e15'
                procv2 = cnsaut * 1.e-18
                proc_3 = '          '
                procv3 = 0.
                proc_4 = '          '
                procv4 = 0.
                if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                        debugV(07, kl) = qsaut
#endif
            enddo
        enddo
    endif

#ifdef qg
    !  Ice Crystals Autoconversion => Graupels
    !  ---------------------------------------
    do kl = mzhyd, klev
        do il = 1, klon
            if(qi2D(il, kl) >= qg00) then
                ex1 = 0.090 * W2xyz1(il, kl)
                pgaut = 0.001 * (qi2D(il, kl) - qg00) * exp(ex1)
                qgaut = pgaut * xt
                qgaut = max(qgaut, zero)
                qgaut = min(qgaut, qi2D(il, kl))
                qi2D(il, kl) = qi2D(il, kl) - qgaut
                qg2D(il, kl) = qg2D(il, kl) + qgaut
            endif
        enddo
    enddo
#endif
    !===========================================================================
    
    !  Accretion Processes (i.e. increase in size of precipitating particles
    !  ====================      through a collision-coalescence process)===
    !                      ==============================================
    
    !  Accretion of Cloud Droplets by Rain
    !  Reference: Lin et al.      1983, JCAM      22, p.1076 (51)
    !             Emde and Kahlig 1989, Ann.Geoph. 7, p. 407 (10)
    !  ----------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qracw = 0.0
#endif
            if(qw2D(il, kl) > eps9) then
                
                ! WbyR_w = 1.0 if     qw2D(il,kl)  > eps9
                ! WbyR_w = 0.0 otherwise
                sign_W = sign(unun, qw2D(il, kl) - eps9)
                WbyR_w = max(zero, sign_W)
                
                if(W2xyz3(il, kl) > eps9) then
                    
                    ! WbyR_r = 1.0 if   W2xyz3(il,kl)  > eps9
                    ! WbyR_r = 0.0 otherwise
                    sign_R = sign(unun, W2xyz3(il, kl) - eps9)
                    WbyR_r = max(zero, sign_R)
                    
                    WbyROK = WbyR_w * WbyR_r
#ifdef EW
                    ! ctr
                    if(WbyROK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(10:10) = 'r'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    pracw = 3104.28d0 * cnor * W2xyz6(il, kl) &
                            * qw2D(il, kl) / exp(3.8d0 * log(W2xyz7(il, kl)))
                    ! 3104.28 = a pi Gamma[3+b] / 4
                    !    where  a = 842. and b = 0.8
                    qracw = pracw * xt * WbyROK
                    if(qracw > 0) qracw = qracw * min(0.9, (1. - cloud_magic))
                    qracw = min(qracw, qw2D(il, kl))
                    
                    qw2D(il, kl) = qw2D(il, kl) - qracw
                    qr2D(il, kl) = qr2D(il, kl) + qracw

#ifdef WQ
                    write(6, *) 'Qracw', qracw, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wracw(kl) = qracw
#endif
                
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Accretion of Clou'
            debugH(36:70) = 'd Droplets by Rain                 '
            proc_1 = 'Qracw g/kg'
            procv1 = qracw
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(08, kl) = qracw
#endif
        enddo
    enddo
    
    !  Accretion of Cloud Droplets by Snow Flakes
    !  Reference: Lin et al.      1983, JCAM      22, p.1070 (24)
    !  ----------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            snoA = 0.0
#endif
            if(qw2D(il, kl) > eps9) then


#ifdef cn
                ! psacw : taken into account in the snow melting process
                !         (if positive temperatures)
                !         9.682 = c pi  Gamma[3+d] / 4
                !         where   c = 4.836 and d = 0.25
                !      (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                !       Graupellike Snow Flakes of Hexagonal Type)
                cnos = min(2.e8, &
                        cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                psacw(il, kl) = 9.682d0 * cnos * W2xyz6(il, kl) &
                        * qw2D(il, kl) / exp(3.25d0 * log(W2xyz8(il, kl)))
#ifdef up
                ! psacw : taken into account in the snow melting process
                !         (if positive temperatures)
                !         3517. = c pi  Gamma[3+d] / 4
                !         where   c = 755.9 and d = 0.99
                !      (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                !       Unrimed Side Planes)
                psacw(il, kl) = 3517. * cnos * W2xyz6(il, kl) &
                        * qw2D(il, kl) / exp(3.99d0 * log(W2xyz8(il, kl)))
#endif
#ifdef ur
                ! psacw : taken into account in the snow melting process
                !         (if positive temperatures)
                !         27.73 = c pi  Gamma[3+d] / 4
                !         where   c = 11.718and d = 0.41
                !      (Locatelli and Hobbs, 1974, JGR: table 1 p.2188:
                !      Aggregates of unrimed radiating assemblages)
                psacw(il, kl) = 27.73 * cnos * W2xyz6(il, kl) &
                        * qw2D(il, kl) / exp(3.41d0 * log(W2xyz8(il, kl)))
#endif
                
                ! WbyS_w = 1.0 if     qw2D(il,kl)  > eps9
                ! WbyS_w = 0.0 otherwise
                sign_W = sign(unun, qw2D(il, kl) - eps9)
                WbyS_w = max(zero, sign_W)
                
                if(W2xyz4(il, kl) > eps9) then
                    
                    ! WbyS_s = 1.0 if   W2xyz4(il,kl)  > eps9
                    ! WbyS_s = 0.0 otherwise
                    sign_S = sign(unun, W2xyz4(il, kl) - eps9)
                    WbyS_s = max(zero, sign_S)
                    
                    WbySOK = WbyS_w * WbyS_s
#ifdef EW
                    ! ctr
                    if(WbySOK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(11:11) = 's'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    
                    qsacw = psacw(il, kl) * xt * WbySOK
                    if(qsacw > 0) qsacw = qsacw * min(0.9, (1. - cloud_magic))
                    qsacw = min(qsacw, qw2D(il, kl))
                    
                    ! Fact_R = 1.0 if   tair2D(il,kl) > TfSnow
                    ! Fact_R = 0.0 otherwise
                    sign_T = sign(unun, tair2D(il, kl) - TfSnow)
                    Fact_R = max(zero, sign_T)
                    if(.not.isnan(qsacw)) then
                        qw2D(il, kl) = qw2D(il, kl) - qsacw
                        qr2D(il, kl) = qr2D(il, kl) + Fact_R * qsacw
                        SnoA = (1.d0 - Fact_R) * qsacw
                        qs2D(il, kl) = qs2D(il, kl) + SnoA
                        ! Negative Temp. => Latent Heat is released by Freezing
                        tair2D(il, kl) = tair2D(il, kl) + r_LcCp * SnoA
                        
                        !  Full Debug
                        !  ~~~~~~~~~~
#ifdef WQ
                        write(6, *) 'Qsacw', qsacw, itexpe, il, kl
#endif
#ifdef WH
                        if(il == ilmm) wsacw(kl) = qsacw
#endif
                    endif
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Accretion of Cloud'
            debugH(36:70) = ' Droplets by Snow Particles         '
            proc_1 = 'Qsacw g/kg'
            procv1 = SnoA
            proc_3 = '          '
            procv2 = 0.
            proc_2 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(09, kl) = SnoA
#endif
        enddo
    enddo

#ifdef qg
    !  Accretion of Cloud Droplets by Graupels (Dry Growth Mode)
    !  Reference: Lin et al.      1983, JCAM      22, p.1075 (40)
    !             Emde and Kahlig 1989, Ann.Geoph. 7, p. 407 (~20)
    !  -----------------------------------------------------------
    do kl = mzhyd, klev
        do il = 1, klon
            if(qw2D(il, kl) > eps9) then
                ! WbyG_w = 1.0 if qw2D(il,kl)  > eps9
                ! WbyG_w = 0.0 otherwise
                sign_W = sign(unun, qw2D(il, kl) - eps9)
                WbyG_w = max(zero, sign_W)
                if(qg2D(il, kl) > eps9) then
                    ! WbyG_g = 1.0 if qg2D(il,kl)  > eps9
                    ! WbyG_g = 0.0 otherwise
                    sign_G = sign(unun, qg2D(il, kl) - eps9)
                    WbyG_g = max(zero, sign_G)
                    WbyGOK = WbyG_w * WbyG_g
                    if(tair2D(il, kl) < TfSnow) then
                        ! Fact_G = 1.0 if tair2D(il,kl)  > TfSnow
                        ! Fact_G = 0.0 otherwise
                        sign_T = -sign(unun, tair2D(il, kl) - TfSnow)
                        Fact_G = max(zero, sign_T)
                        !                        pgacw = ???
                        qgacw = pgacw * xt * WbyGOK
                        qgacw = min(qgacw, qw2D(il, kl))
                        qw2D(il, kl) = qw2D(il, kl) - qgacw
                        qg2D(il, kl) = qg2D(il, kl) + qgacw
                        tair2D(il, kl) = tair2D(il, kl) + r_LcCp * gacw
                    endif
                endif
            endif
        enddo
    enddo
#endif
    !  Accretion of Cloud Ice      by Snow Particles
    !  Reference: Lin et al.      1983, JCAM      22, p.1070 (22)
    !  ----------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qsaci = 0.0
            cnsaci = 0.0
#endif
            if(qi2D(il, kl) > eps9) then
                
                ! CbyS_c = 1.0 if qi2D(il,kl)  > eps9
                ! CbyS_c = 0.0 otherwise
                sign_C = sign(unun, qi2D(il, kl) - eps9)
                CbyS_c = max(zero, sign_C)
                
                if(W2xyz4(il, kl) > eps9) then
                    
                    ! CbyS_s = 1.0 if   W2xyz4(il,kl)  > eps9
                    ! CbyS_s = 0.0 otherwise
                    sign_S = sign(unun, W2xyz4(il, kl) - eps9)
                    CbyS_s = max(zero, sign_S)
                    
                    if(tair2D(il, kl) < TfSnow) then
                        
                        ! CbyS_T = 1.0 if   tair2D(il,kl)  < TfSnow
                        ! CbyS_T = 0.0 otherwise
                        sign_T = -sign(unun, tair2D(il, kl) - TfSnow)
                        CbyS_T = max(zero, sign_T)
                        
                        CbySOK = CbyS_c * CbyS_s * CbyS_T
#ifdef EW
                        ! ctr
                        if(CbySOK > epsi) then
                            mauxEW = mphy2D(il)
                            mauxEW(12:12) = 's'
                            mphy2D(il) = mauxEW
                        endif
#endif
                        ! efc   : Collection Efficiency
                        !         Lin et al. 1983 JCAM 22 p.1070 (23)
                        efc = exp(0.025d0 * W2xyz1(il, kl))
#ifdef cn
                        cnos = min(2.e8, &
                                cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                        psaci = efc * 9.682d0 * cnos * W2xyz6(il, kl) &
                                * qi2D(il, kl) / exp(3.25d0 * log(W2xyz8(il, kl)))
#ifdef up
                        psaci = efc * 3517.d0 * cnos * W2xyz6(il, kl) &
                                * qi2D(il, kl) / exp(3.99d0 * log(W2xyz8(il, kl)))
#endif
#ifdef ur
                        psaci = efc * 27.73d0 * cnos * W2xyz6(il, kl) &
                                * qi2D(il, kl) / exp(3.41d0 * log(W2xyz8(il, kl)))
#endif
                        qsaci = psaci * xt * CbySOK
                        if(qsaci > 0) qsaci = qsaci * (1. - cloud_magic)
                        qsaci = min(qsaci, qi2D(il, kl))
                        
                        cnsaci = ccni2D(il, kl) * qsaci / &
                                max(qi2D(il, kl), eps9)
                        ccni2D(il, kl) = ccni2D(il, kl) - cnsaci
                        qi2D(il, kl) = qi2D(il, kl) - qsaci
                        qs2D(il, kl) = qs2D(il, kl) + qsaci

#ifdef WQ
                        write(6, *) 'Qsaci', qsaci, itexpe, il, kl
#endif
#ifdef WH
                        if(il == ilmm) wsaci(kl) = qsaci
#endif
                    
                    endif
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Accretion of Clou'
            debugH(36:70) = 'd Ice by Snow Particles            '
            proc_1 = 'Qsaci g/kg'
            procv1 = qsaci
            proc_2 = 'CNsaci/e15'
            procv2 = cnsaci * 1.e-18
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(10, kl) = qsaci
#endif
        enddo
    enddo

#ifdef qg
    !  Accretion of Cloud Ice      by Graupel (Cloud Ice Sink)
    !  Reference: Lin et al.      1983, JCAM      22, p.1075 (41)
    !             Emde and Kahlig 1989, Ann.Geoph. 7, p. 407 (~19)
    !  -----------------------------------------------------------
    do kl = mzhyd, klev
        do il = 1, klon
            if(qw2D(il, kl) > eps9) then
                !           CbyG_c      =   1.0 if     qi2D(il,kl)  > eps9
                !                       =   0.0 otherwise
                sign_C = sign(unun, qi2D(il, kl) - eps9)
                CbyG_c = max(zero, sign_C)
                if(qg2D(il, kl) > eps9) then
                    ! CbyG_g = 1.0 if qg2D(il,kl)  > eps9
                    ! CbyG_g = 0.0 otherwise
                    sign_G = sign(unun, qg2D(il, kl) - eps9)
                    CbyG_g = max(zero, sign_G)
                    if(tair2D(il, kl) < TfSnow) then
                        ! Fact_G = 1.0 if tair2D(il,kl)  < TfSnow
                        ! Fact_G = 0.0 otherwise
                        sign_T = -sign(unun, tair2D(il, kl) - TfSnow)
                        Fact_G = max(zero, sign_T)
                        CbyGOK = CbyG_c * CbyG_g * Fact_G
                        !                        pgaci = ???
                        qgaci = pgaci * xt * CbyGOK
                        qgaci = min(qgaci, qi2D(il, kl))
                        qi2D(il, kl) = qi2D(il, kl) - qgaci
                        qg2D(il, kl) = qg2D(il, kl) + qgaci
                    endif
                endif
            endif
        enddo
    enddo
#endif
    
    !  Accretion of Cloud Ice by Rain (Cloud Ice Sink)
    !  Reference: Lin et al. 1983, JCAM 22, p.1071 (25)
    !  ------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qraci = 0.0
            qiacr = 0.0
#endif
            if(qi2D(il, kl) > eps9) then
                ! CbyR_c = 1.0 if qi2D(il,kl)  > eps9
                ! CbyR_c = 0.0 otherwise
                sign_C = sign(unun, qi2D(il, kl) - eps9)
                CbyR_c = max(zero, sign_C)
                if(W2xyz3(il, kl) > eps9) then
                    ! CbyR_r = 1.0 if W2xyz3(il,kl)  > eps9
                    ! CbyR_r = 0.0 otherwise
                    sign_R = sign(unun, W2xyz3(il, kl) - eps9)
                    CbyR_r = max(zero, sign_R)
                    if(tair2D(il, kl) < TfSnow) then
                        ! CbyR_T = 1.0 if tair2D(il,kl)  < TfSnow
                        ! CbyR_T = 0.0 otherwise
                        sign_T = -sign(unun, tair2D(il, kl) - TfSnow)
                        CbyR_T = max(zero, sign_T)
                        
                        CbyROK = CbyR_c * CbyR_r * CbyR_T
#ifdef EW
                        ! ctr
                        if(CbyROK > epsi) then
                            mauxEW = mphy2D(il)
                            if(mauxEW(13:13) == 's' .or. mauxEW(13:13) == 'A') then
                                mauxEW(13:13) = 'A'
                            else
                                mauxEW(13:13) = 'r'
                            endif
                            mphy2D(il) = mauxEW
                        endif
#endif
                        praci = 3104.28d0 * cnor * W2xyz6(il, kl) &
                                * qi2D(il, kl) / exp(3.8d0 * log(W2xyz7(il, kl)))
                        qraci = praci * xt * CbyROK
                        if(qraci > 0) qraci = qraci * (1. - cloud_magic)
                        qraci = min(qraci, qi2D(il, kl))
                        cnraci = ccni2D(il, kl) * qraci / &
                                max(qi2D(il, kl), eps9)
                        ccni2D(il, kl) = ccni2D(il, kl) - cnraci
                        qi2D(il, kl) = qi2D(il, kl) - qraci
#ifdef qg
                        ! CAUTION : Graupels Formation is not taken into account
                        ! This could be a reasonable assumption for Antarctica
                        ! ctr
                        if(qr2D(il, kl) > 1.e-4) then
                            qg2D(il, kl) = qg2D(il, kl) + qraci
                        else
#endif
                            qs2D(il, kl) = qs2D(il, kl) + qraci
#ifdef qg
                        endif
#endif

#ifdef WQ
                        write(6, *) 'Qraci', qraci, itexpe, il, kl
#endif
#ifdef WH
                        if(il == ilmm) wraci(kl) = qraci
#endif
                        
                        !  Accretion of Rain by Cloud Ice (Rain Sink)
                        !  Reference: Lin et al. 1983, JCAM 22, p.1071 (26)
                        !  ------------------------------------------------
#ifdef EW
                        ! ctr
                        if(CbyROK > epsi) then
                            mauxEW = mphy2D(il)
                            if(mauxEW(13:13) == 'r' .or. mauxEW(13:13) == 'A') then
                                mauxEW(13:13) = 'A'
                            else
                                mauxEW(13:13) = 's'
                            endif
                            mphy2D(il) = mauxEW
                        endif
#endif
                        ! Lin et al, 1983, JAM,p1071: mi:Ice Crystal Mass
                        piacr = 4.1e20 * cnor * W2xyz6(il, kl) &
                                * qi2D(il, kl) / exp(6.8d0 * log(W2xyz7(il, kl)))
                        ! 4.1e20 = a pi**2 rhow/mi Gamma[6+b] / 24
                        ! where    a=842., rhow=1000, mi=4.19e-13
                        !                             and b = 0.8
                        qiacr = piacr * xt * CbyROK
                        qiacr = min(qiacr, qr2D(il, kl))
                        qr2D(il, kl) = qr2D(il, kl) - qiacr
                        tair2D(il, kl) = tair2D(il, kl) + r_LcCp * qiacr
#ifdef qg
                        ! CAUTION : Graupels Formation is not taken into account
                        ! This could be a reasonable assumption for Antarctica
                        ! ctr
                        if(qr2D(il, kl) > 1.e-4) then
                            qg2D(il, kl) = qg2D(il, kl) + qiacr
                        else
#endif
                            qs2D(il, kl) = qs2D(il, kl) + qiacr
#ifdef qg
                        endif
#endif
#ifdef WQ
                        !  Full Debug
                        !  ~~~~~~~~~~
                        write(6, *) 'Qiacr', qiacr, itexpe, il, kl
#endif
#ifdef WH
                        if(il == ilmm) wiacr(kl) = qiacr
#endif
                    endif
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Accretion of Clou'
            debugH(36:70) = 'd Ice by Rain                      '
            proc_1 = 'Qraci g/kg'
            procv1 = qraci
            proc_2 = 'qiacr g/kg'
            procv2 = qiacr
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(11, kl) = qiacr
#endif
        enddo
    enddo
    
    !  Accretion of Rain           by Snow Flakes
    !  Accretion of Snow Flakes    by Rain
    !  Reference: Lin et al.      1983, JCAM      22, p.1071 (27)
    !             Lin et al.      1983, JCAM      22, p.1071 (28)
    !             Emde and Kahlig 1989, Ann.Geoph. 7, p. 408 (~21)
    !  -----------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
            psacr(il, kl) = 0.d0
            qsacr = 0.d0
#ifdef wH
            qraci = 0.0d0
            qracsS = 0.0d0
            qsacrR = 0.0d0
#endif
            if(W2xyz3(il, kl) > eps9) then
                ! RbyS_r = 1.0 if   W2xyz3(il,kl)  > eps9
                ! RbyS_r = 0.0 otherwise
                sign_R = sign(unun, W2xyz3(il, kl) - eps9) ! W2xyz3: Qr
                RbyS_r = max(zero, sign_R)
                if(W2xyz4(il, kl) > eps9) then
                    ! RbyS_s = 1.0 if   W2xyz4(il,kl)  > eps9
                    ! RbyS_s = 0.0 otherwise
                    sign_S = sign(unun, W2xyz4(il, kl) - eps9) ! W2xyz4: Qs
                    RbyS_s = max(zero, sign_S)
                    
                    RbySOK = RbyS_r * RbyS_s
#ifdef EW
                    if(CbyROK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(14:14) = 'A'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    !  Accretion of Rain by Snow --> Snow           | W2xyz7 : lambda_r
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~           | W2xyz8 : lambda_s
                    flS = (5.0d0 / (W2xyz8(il, kl) * W2xyz8(il, kl) * W2xyz7(il, kl)) &
                            + 2.0d0 / (W2xyz8(il, kl) * W2xyz7(il, kl) * W2xyz7(il, kl)) &
                            + 0.5d0 / (W2xyz7(il, kl) * W2xyz7(il, kl) * W2xyz7(il, kl))) &
                            / (W2xyz8(il, kl) * W2xyz8(il, kl) * W2xyz8(il, kl) * W2xyz8(il, kl))
#ifdef cn
                    cnos = min(2.e8, &
                            cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                    pracs = 986.96d-3 * (cnor * cnos / rolv2D(il, kl)) &
                            * abs(vr(il, kl) - vs(il, kl)) * flS
                    !                         986.96: pi**2 * rhos
                    !                        (snow    density assumed equal to  100 kg/m3)
                    qracs = pracs * xt * RbySOK
                    qracs = min(qracs, qr2D(il, kl))
#ifdef WQ
                    write(6, *) 'Qracs', qracs, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wracs(kl) = qracs
#endif
                    !  Accretion of Snow by Rain --> Rain
                    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    ! SbyR_r = 1.0 if W2xyz3(il,kl)  > 1.e-4
                    ! SbyR_r = 0.0 otherwise
                    sign_R = sign(unun, W2xyz3(il, kl) - 1.e-4)
                    SbyR_r = max(zero, sign_R)
                    
                    ! SbyR_s = 1.0 if W2xyz4(il,kl)  > 1.e-4
                    ! SbyR_s = 0.0 otherwise
                    sign_S = sign(unun, W2xyz4(il, kl) - 1.e-4)
                    SbyR_s = max(zero, sign_S)
                    
                    SbyROK = max(SbyR_r, SbyR_s)
                    
                    if(SbyROK > epsi) then
                        flR = (5.d0 / (W2xyz7(il, kl) * W2xyz7(il, kl) * W2xyz8(il, kl)) &
                                + 2.d0 / (W2xyz7(il, kl) * W2xyz8(il, kl) * W2xyz8(il, kl)) &
                                + 0.5d0 / (W2xyz8(il, kl) * W2xyz8(il, kl) * W2xyz8(il, kl))) &
                                / (W2xyz7(il, kl) * W2xyz7(il, kl) * W2xyz7(il, kl) * W2xyz7(il, kl))
                        
                        psacr(il, kl) = 9869.6d-3 * (cnor * cnos / rolv2D(il, kl)) &
                                * abs(vr(il, kl) - vs(il, kl)) * flR
                        !  9869.6: pi**2 * rhow
                        ! (water density assumed equal to 1000 kg/m3)
                        qsacr = psacr(il, kl) * xt * RbySOK * SbyROK
                        qsacr = min(qsacr, qs2D(il, kl))

#ifdef WQ
                        write(6, *) 'Qsacr', qsacr, itexpe, il, kl
#endif

#ifdef WH
                        if(il == ilmm) wsacr(kl) = qsacr
#endif
                    else
                        psacr(il, kl) = 0.d0
                        qsacr = 0.d0
                    endif
                    
                    !           CbyR_T      =   1.0 if   tair2D(il,kl)  < TfSnow
                    !                       =   0.0 otherwise
                    sign_T = -sign(unun, tair2D(il, kl) - TfSnow)
                    CbyR_T = max(zero, sign_T)
                    
                    qracsS = qracs * CbyR_T
                    qsacrR = qsacr * (1.d0 - CbyR_T)
                    
                    qr2D(il, kl) = qr2D(il, kl) - qracsS
#ifdef qg
                    ! CAUTION  : Graupel Formation is not taken into Account
                    if(W2xyz3(il, kl) < 1.e-4 .and. W2xyz4(il, kl) < 1.e-4) then
#endif
                        qs2D(il, kl) = qs2D(il, kl) + qracsS
#ifdef qg
                    else
                        qs2D(il, kl) = qs2D(il, kl) - qracsS
                        qg2D(il, kl) = qg2D(il, kl) + qsacrS + qracsS
                    endif
#endif
                    tair2D(il, kl) = tair2D(il, kl) + qracsS * r_LcCp
                    qr2D(il, kl) = qr2D(il, kl) + qsacrR
                    qs2D(il, kl) = qs2D(il, kl) - qsacrR
                    tair2D(il, kl) = tair2D(il, kl) - qsacrR * r_LcCp
                endif
            endif

#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Accretion of Snow'
            debugH(36:70) = '(Rain) by Rain(Snow)               '
            proc_1 = 'Qracs g/kg'
            procv1 = qracsS
            proc_2 = 'Qsacr g/kg'
            procv2 = qsacrR
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            include 'MAR_HY.Debug'
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(12, kl) = qracsS - qsacrR
#endif
        enddo
    enddo
    
    !  Accretion of Snow           by Graupels
    !  Reference: Lin et al.      1983, JCAM      22, p.1071 (29)
    !  ----------------------------------------------------------
    
    !    #ifdef qg
    !    do il=1,klon
    !        do kl=mzhyd,klev
    !
    !            if (W2xyz0(il,kl) > eps9) then
    !
    !                ! SbyG_g = 1.0 if W2xyz0(il,kl) > eps9
    !                ! SbyG_g = 0.0 otherwise
    !                sign_G      =  sign(unun,W2xyz0(il,kl)  - eps9)
    !                SbyG_g      =   max(zero,sign_G)
    !
    !                if (W2xyz4(il,kl) > eps9) then
    !
    !                    ! SbyG_s = 1.0 if W2xyz4(il,kl) > eps9
    !                    ! SbyG_s = 0.0 otherwise
    !                    sign_S      =  sign(unun,W2xyz4(il,kl)  - eps9)
    !                    SbyG_s      =   max(zero,sign_S)
    !
    !                    SbyGOK      =  SbyG_g *  SbyG_s
    !
    !                    ! efc : Collection Efficiency
    !                    ! Lin et al. 1983 JCAM 22 p.1072 (30)
    !                    efc   =  exp(0.090*W2xyz1(il,kl))
    !
    !                    flg=exp(-6.0d0*log(W2xyz8(il,kl)) &
    !                        *(5.0/W2xyz9(il,kl) &
    !                        +2.0*W2xyz8(il,kl)/(W2xyz9(il,kl)*W2xyz9(il,kl)) &
    !                        +0.5*W2xyz8(il,kl)* W2xyz8(il,kl) &
    !                        /exp(3.0d0*log(W2xyz9(il,kl))))
    !                #ifdef cn
    !                    cnos = min(2.e8 , &
    !                         cnos2*exp(-.12*min(0.,W2xyz1(il,kl))))
    !                #endif
    !                    pgacs      = 986.96d-3*(cnog*cnos/rolv2D(il,kl)) &
    !                                     * abs(vg(il,kl)-vs(il,kl))*flg*efc
    !                    ! 986.96: pi**2 * rhog
    !                    ! (graupel densitity assumed equal to snow density)
    !                    qgacs      =     pgacs * xt       * SbyGOK
    !                    qgacs      = min(qgacs,qs2D(il,kl))
    !                    qg2D(il,kl)  = qg2D(il,kl) +        qgacs
    !                    qs2D(il,kl)  = qs2D(il,kl) -        qgacs
    !
    !                end if
    !            end if
    !
    !        end do
    !    end do
    !    #endif
    
    !  Accretion of Rain by Graupels (Dry Growth Mode)
    !  Reference: Lin et al. 1983, JCAM 22, p.1075 (42)
    !  ------------------------------------------------
    !    #ifdef qg
    !    do kl=mzhyd,klev
    !        do il=1,klon
    !
    !           if (W2xyz0(il,kl) > eps9) then
    !
    !               ! RbyG_g = 0.0 otherwise
    !               ! RbyG_g = 1.0 if W2xyz0(il,kl) > eps9
    !               sign_G      =  sign(unun,W2xyz0(il,kl)  - eps9)
    !               RbyG_g      =   max(zero,sign_G)
    !
    !               if (W2xyz3(il,kl) > eps9) then
    !
    !                   ! RbyG_r = 0.0 otherwise
    !                   ! RbyG_r = 1.0 if W2xyz3(il,kl) > eps9
    !                   sign_R = sign(unun,W2xyz3(il,kl)  - eps9)
    !                   RbyG_r = max(zero,sign_R)
    !
    !                   if (tair2D(il,kl) < TfSnow) then
    !
    !                       ! Fact_G = 0.0 otherwise
    !                       ! Fact_G = 1.0 if tair2D(il,kl) < TfSnow
    !                       sign_T      = -sign(unun,tair2D(il,kl)  - TfSnow)
    !                       Fact_G      =   max(zero,sign_T)
    !
    !                       RbyGOK      =  RbyG_g *  RbyG_s         * Fact_G
    !
    !                       flg=exp(-6.0d0*log(W2xyz8(il,kl)) &
    !                          *(5.0/W2xyz9(il,kl) &
    !                           +2.0*W2xyz8(il,kl)/(W2xyz9(il,kl)*W2xyz9(il,kl)) &
    !                           +0.5*W2xyz8(il,kl)* W2xyz8(il,kl)) &
    !                                /exp(3.0d0*log(W2xyz9(il,kl))))
    !                       #ifdef cn
    !                       cnos = min(2.e8 , &
    !                           cnos2*exp(-.12*min(0.,W2xyz1(il,kl))))
    !                       #endif
    !                       pgacr = 986.96d-3*(cnog*cnos/rolv2D(il,kl)) &
    !                                               * abs(vg(i,kl) - vr(il,kl))*flg
    !                       qgacr = pgacr * xt  *        RbyGOK
    !                       qgacr = min(qgacr,qr2D(il,kl))
    !                       qg2D(il,kl) = qg2D(il,kl) + qgacr
    !                       qr2D(il,kl) = qr2D(il,kl) - qgacr
    !                       tair2D(il,kl) = tair2D(il,kl) + r_LcCp*qgacr
    !
    !                   end if
    !               end if
    !           end if
    !
    !        end do
    !    end do
    !    #endif

#ifdef qg
    !  Graupels Wet Growth Mode
    !  Reference: Lin et al.      1983, JCAM      22, p.1075 (43)
    !  ----------------------------------------------------------
    ! TO BE ADDED !
#endif
    
    !  Microphysical Processes affecting     Precipitating Cloud Particles
    !  ===================================================================
    
    !  Rain Drops Evaporation                                    ============
    !  Reference: Lin et al.      1983, JCAM      22, p.1077 (52)
    !  ----------------------------------------------------------
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qrevp = 0.0
#endif
            if(W2xyz3(il, kl) > eps9) then
                ! W2xyz3 : old Rain Concentration
                
                ! Evap_r = 1.0 if W2xyz3(il,kl) > eps9
                ! Evap_r = 0.0 otherwise
                sign_R = sign(unun, W2xyz3(il, kl) - eps9)
                Evap_r = max(zero, sign_R)
                
                EvapOK = Evap_r

#ifdef EW
                if(EvapOK > epsi) then
                    mauxEW = mphy2D(il)
                    mauxEW(15:15) = 'v'
                    mphy2D(il) = mauxEW
                endif
#endif
                ! sr : grid scale saturation humidity
                sr = qv2D(il, kl) / (rhcrHY * qvsw2D(il, kl))
                
                if(sr < unun) then
                    ! Evap_q = 1.0 if sr < unun
                    ! Evap_q = 0.0 otherwise
                    sign_Q = -sign(unun, sr - unun)
                    Evap_q = max(zero, sign_Q)
                    
                    EvapOK = EvapOK * Evap_q
                    
                    almr = 0.78d0 / (W2xyz7(il, kl) * W2xyz7(il, kl)) &
                            + 3940.d0 * sqrt(W2xyz6(il, kl)) &
                                    / exp(2.9d0 * log(W2xyz7(il, kl)))
                    ab = 5.423d11 / (tair2D(il, kl) * tair2D(il, kl)) &
                            + 1.d0 / (1.875d-2 * rolv2D(il, kl) * qvsw2D(il, kl))
                    
                    prevp = 2 * pi * (1.d0 - sr) * cnor * almr / ab
                    qrevp = prevp * xt
                    qrevp = min(qrevp, qr2D(il, kl))
                    
                    ! supersaturation is not allowed to occur
                    qrevp = min(qrevp, rhcrHY * qvsw2D(il, kl) - qv2D(il, kl))
                    
                    ! condensation is not allowed to occur
                    qrevp = max(qrevp, zero) * EvapOK
                    
                    qr2D(il, kl) = qr2D(il, kl) - qrevp
                    dqw2D(il, kl) = dqw2D(il, kl) - qrevp
                    qv2D(il, kl) = qv2D(il, kl) + qrevp
                    tair2D(il, kl) = tair2D(il, kl) - r_LvCp * qrevp
                    qrevp2D(il, kl) = qrevp
                    !  Full Debug
                    !  ~~~~~~~~~~
#ifdef WQ
                    write(6, *) 'Qrevp', qrevp, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wrevp(kl) = qrevp
#endif
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Rain Drops Evapor'
            debugH(36:70) = 'ation                              '
            proc_1 = 'Qrevp g/kg'
            procv1 = qrevp
            proc_2 = 'R.Hum  [%]'
            procv2 = sr * 0.1
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            include 'MAR_HY.Debug'
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(13, kl) = -qrevp
#endif
        enddo
    enddo
    
    !  (Deposition on) Snow Flakes (Sublimation)
    !   Reference: Lin et al.      1983, JCAM      22, p.1072 (31)
    !   ----------------------------------------------------------
    
    !    #ifdef BS
    !    do il=1,klon
    !        hlat2D(il,1)  = 0.d0
    !    end do
    !    #endif
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qssub = 0.0
#endif
            if(W2xyz4(il, kl) > eps9) then
                ! W2xyz4 : old Snow F. Concentration
                
                ! Evap_s = 1.0 if W2xyz4(il,kl) > eps9
                ! Evap_s = 0.0 otherwise
                sign_S = sign(unun, W2xyz4(il, kl) - eps9)
                Evap_s = max(zero, sign_S)

#ifdef EW
                if(Evap_s > epsi) then
                    mauxEW = mphy2D(il)
                    mauxEW(16:16) = 'V'
                    mphy2D(il) = mauxEW
                endif
#endif
                
                si = qv2D(il, kl) / W2xyz5(il, kl)
                
                alms = 0.78d0 / (W2xyz8(il, kl) * W2xyz8(il, kl)) &
                        + 238.d0 * sqrt(W2xyz6(il, kl)) &
                                / exp(2.625d0 * log(W2xyz8(il, kl)))
                ab = 6.959d11 / (tair2D(il, kl) * tair2D(il, kl)) &
                        + 1.d0 / (1.875d-2 * rolv2D(il, kl) * W2xyz5(il, kl))

#ifdef cn
                cnos = min(2.e8, &
                        cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                pssub = 2 * pi * (1.d0 - si) * cnos * alms / (1.d3 * rolv2D(il, kl) * ab)
                qssub = pssub * xt
                
                dqamx = W2xyz5(il, kl) - qv2D(il, kl)
                
                ! Depo_s = 1.0 if si > unun
                ! Depo_s = 0.0 otherwise
                sign_S = sign(unun, si - unun)
                Depo_s = max(zero, sign_S)
                
                ! qssub < 0 ... Deposition
                ! qssub > 0 ... Sublimation
                qssub = max(qssub, dqamx) * Depo_s &
                        + min(min(qssub, qs2D(il, kl)), dqamx) * (1.d0 - Depo_s)
                
                qssub = qssub * Evap_s
                
                qs2D(il, kl) = qs2D(il, kl) - qssub
                dqi2D(il, kl) = dqi2D(il, kl) - qssub
                qv2D(il, kl) = qv2D(il, kl) + qssub
                tair2D(il, kl) = tair2D(il, kl) - r_LsCp * qssub
                qssub2D(il, kl) = qssub
                ! #ifdef BS
                ! hlat2D(il ,1) = hlat2D(il ,1) + qssub * rolv2D(il,kl) &
                !     *(gpmi2D(il,kl)-gpmi2D(il,kl+1))*grvinv
                ! #endif
#ifdef WQ
                !  Full Debug
                !  ~~~~~~~~~~
                write(6, *) 'Qssub', qssub, itexpe, il, kl
#endif
#ifdef WH
                if(il == ilmm) wssub(kl) = -qssub
#endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): (Deposition on) S'
            debugH(36:70) = 'now Particles (Sublimation)        '
            proc_1 = 'Qssub g/kg'
            procv1 = qssub
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            include 'MAR_HY.Debug'
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(14, kl) = -qssub
#endif
        enddo
    enddo

#ifdef qg
    !  Graupels Sublimation
    !  Reference: Lin et al.      1983, JCAM      22, p.1076 (46)
    !  ----------------------------------------------------------
    ! TO BE ADDED !
#endif
    !  Snow Flakes Melting        PSMLT
    !  Reference: Lin et al.      1983, JCAM      22, p.1072 (32)
    !  ----------------------------------------------------------
    
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qsmlt = 0.0
#endif
            if(W2xyz4(il, kl) > eps9) then
                ! W2xyz4 : old Snow Fl.Concentration
                
                ! SnoM_s = 1.0 if W2xyz4(il,kl) > eps9
                ! SnoM_s = 0.0 otherwise
                sign_S = sign(unun, W2xyz4(il, kl) - eps9)
                SnoM_s = max(zero, sign_S)
                
                if(W2xyz1(il, kl) > 0.) then
                    ! W2xyz1 : old Celsius Temperature
                    
                    ! SnoM_T = 1.0 if W2xyz1(il,kl)  > 0.
                    ! SnoM_T = 0.0 otherwise
                    sign_T = sign(unun, W2xyz1(il, kl) - 0.)
                    SnoM_T = max(zero, sign_T)
                    
                    SnoMOK = SnoM_s * SnoM_T
#ifdef EW
                    if(SnoMOK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(17:17) = 'r'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    alms = 0.78 / (W2xyz8(il, kl) * W2xyz8(il, kl)) &
                            + 238. * sqrt(W2xyz6(il, kl)) &
                                    / exp(2.625d0 * log(W2xyz8(il, kl)))
#ifdef cn
                    cnos = min(2.e8, &
                            cnos2 * exp(-.12 * min(0., W2xyz1(il, kl))))
#endif
                    xCoef = 1.904d-8 * cnos * alms * r_LcCp / rolv2D(il, kl)
                    ! 1.904e-8: 2 pi / Lc /[10.**3 =rho Factor]
                    
                    ACoef = 0.0250d0 * xCoef &
                            + (psacw(il, kl) + psacr(il, kl)) * r_LcCp / 78.8d0
                    ! 78.8    : Lc /[Cpw=4.187e3 J/kg/K]
                    
                    Bcoef = 62.34d+3 * rolv2D(il, kl) * &
                            (qv2D(il, kl) - W2xyz5(il, kl)) &
                            * xCoef
                    Bcoef = min(-eps9, Bcoef)
                    
                    Tc = (tair2D(il, kl) - TfSnow - ACoef / Bcoef) * exp(-ACoef * xt)
                    qsmlt = (tair2D(il, kl) - TfSnow - Tc) / r_LcCp
                    qsmlt = max(qsmlt, 0.) * SnoMOK
                    qsmlt = min(qsmlt, qs2D(il, kl))
                    
                    if(tair2D(il, kl) - TfSnow>5) qsmlt = max(qsmlt, qs2D(il, kl) / 4.) ! no snow > 5°C
                    
                    ! #ifdef XF
                    ! ! this options increases the conversion of Snowfall to rainfall
                    ! alms  = 0.78d0  /(W2xyz8(il,kl)*W2xyz8(il,kl)) &
                    !          +  238.d0 *       sqrt(W2xyz6(il,kl)) &
                    !                /exp(2.625d0*log(W2xyz8(il,kl)))
                    ! akps  = 0.025d0  *W2xyz1(il,kl) &
                    !       + 46.875d3 *rolv2D(il,kl) *(qv2D(il,kl)-W2xyz5(il,kl))
                    ! ! 46.875  : Lv*[psiv=1.875e-5m2/s]
                    !
                    ! psmlt = 1.904d-8*cnos*akps*alms/rolv2D(il,kl) &
                    !       -(psacw(il,kl) + psacr(il,kl)) *W2xyz1(il,kl) / 78.8d0
                    ! ! 1.904e-8: 2 pi / Lc /[10.**3 =rho Factor]
                    ! ! Lc /[Cpw=4.187e3 J/kg/K]  = 78.8
                    ! qsmlt = psmlt * xt * SnoMOK
                    ! qsmlt = max(qsmlt,zero)
                    ! qsmlt = min(qsmlt,qs2D(il,kl))
                    ! #endif
                    qs2D(il, kl) = qs2D(il, kl) - qsmlt
                    qr2D(il, kl) = qr2D(il, kl) + qsmlt
                    tair2D(il, kl) = tair2D(il, kl) - r_LcCp * qsmlt
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'Qsmlt', qsmlt, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wsmlt(kl) = qsmlt
#endif
                
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Snow Particles Me'
            debugH(36:70) = 'lting                              '
            proc_1 = 'Qsmlt g/kg'
            procv1 = qsmlt
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            include 'MAR_HY.Debug'
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(15, kl) = -qsmlt
#endif
        enddo
    enddo
#ifdef qg
    !  Graupels    Melting
    !  Reference: Lin et al.      1983, JCAM      22, p.1076 (47)
    !  ----------------------------------------------------------
    ! TO BE ADDED !
#endif
    !  Rain Freezing
    !  Reference: Lin et al.      1983, JCAM      22, p.1075 (45)
    !  ----------------------------------------------------------
    !  **CAUTION**: Graupel Formation TO BE ADDED !
    do kl = mzhyd, klev
        do il = 1, klon
#ifdef wH
            qsfr = 0.0
#endif
            if(W2xyz3(il, kl) > eps9) then
                ! W2xyz3 : old Rain    Concentration
                ! Freezr = 1.0 if W2xyz3(il,kl) > eps9
                ! Freezr = 0.0 otherwise
                sign_R = sign(unun, W2xyz3(il, kl) - eps9)
                Freezr = max(zero, sign_R)
                if(W2xyz1(il, kl) < 0.) then
                    ! W2xyz1 : old Celsius Temperature
                    ! FreezT = 1.0 if W2xyz1(il,kl) < 0.
                    ! FreezT = 0.0 otherwise
                    sign_T = -sign(unun, W2xyz1(il, kl) - 0.)
                    FreezT = max(zero, sign_T)
                    FrerOK = Freezr * FreezT
#ifdef EW
                    if(FrerOK > epsi) then
                        mauxEW = mphy2D(il)
                        mauxEW(19:19) = 's'
                        mphy2D(il) = mauxEW
                    endif
#endif
                    psfr = 1.974d4 * cnor &
                            / (rolv2D(il, kl) * exp(7.d0 * log(W2xyz7(il, kl)))) &
                            * (exp(-0.66d0 * W2xyz1(il, kl)) - 1.d0)
                    qsfr = psfr * xt * FrerOK
                    qsfr = min(qsfr, qr2D(il, kl))
                    
                    if(tair2D(il, kl) - TfSnow<-5) qsfr = max(qsfr, qr2D(il, kl) / 4.) ! no rain < -5C
                    
                    qr2D(il, kl) = qr2D(il, kl) - qsfr
                    qs2D(il, kl) = qs2D(il, kl) + qsfr
                    ! CAUTION : graupel production is included into snow production
                    !           proposed modification in line below.
                    !                    #ifdef qg
                    !                    qg2D(il,kl) = qg2D(il,kl) + qsfr
                    !                    #endif
                    tair2D(il, kl) = tair2D(il, kl) + r_LcCp * qsfr
#ifdef WQ
                    !  Full Debug
                    !  ~~~~~~~~~~
                    write(6, *) 'Qsfre', qsfr, itexpe, il, kl
#endif
#ifdef WH
                    if(il == ilmm) wsfre(kl) = qsfr
#endif
                endif
            endif
#ifdef wH
            !  Debug
            !  ~~~~~
            debugH(1:35) = 'Lin et al.(1983): Rain Freezing    '
            debugH(36:70) = '                                   '
            proc_1 = 'Qsfr g/kg'
            procv1 = qsfr
            proc_2 = '          '
            procv2 = 0.
            proc_3 = '          '
            procv3 = 0.
            proc_4 = '          '
            procv4 = 0.
            include 'MAR_HY.Debug'
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) &
                    debugV(16, kl) = qsfr
#endif
        
        enddo
    enddo
    
    !  Debug (Summary)
    !  ~~~~~~~~~~~~~~~
#ifdef wH
    do kl = mzhyd, klev
        do il = 1, klon
            if(i_fvv(il) == i0fvv .and. j_fvv(il) == j0fvv) then
                if(kl == mzhyd) then
                    write(6, 6022)
                    6022                format(/, 'HYDmic STATISTICS=================')
                    write(6, 6026)
                    6026                format('    T_Air Qv   Qw g/kg  Qi g/kg  CLOUDS % '// &
                            ' Qs g/kg  Qr g/kg'// &
                            ' Qi+ E.K.'// &
                            ' Qi+ Mey.'// &
                            ' Qi- Sub.'// &
                            ' Qi- Mlt.'// &
                            ' Qw+ Cds.'// &
                            ' Qraut r+'// &
                            ' Qsaut s+'// &
                            ' Qracw r+')
                endif
                write(6, 6023) kl, &
                        tair2D(il, kl) - TfSnow, &
                        10.**3 * qv2D(il, kl), &
                        10.**3 * qw2D(il, kl), &
                        10.**3 * qi2D(il, kl), &
                        10.**2 * cfra2D(il, kl), &
                        10.**3 * qs2D(il, kl), &
                        10.**3 * qr2D(il, kl), &
                        (10.**3 * debugV(kv, kl), kv = 1, 8)
                6023            format(i3, f6.1, f5.2, 2f9.6, f9.1, 2f9.3, 8f9.6)
                if(kl == klev) then
                    write(6, 6026)
                    write(6, *) ' '
                    write(6, 6024)
                    6024                format(8x, 'Z [km]'// &
                            ' RH.w.[%]'// &
                            ' RH.i.[%]'//9x// &
                            ' Vss cm/s'// &
                            ' Vrr cm/s'// &
                            ' Qsacw s+'// &
                            ' Qsaci s+'// &
                            ' Qiacr r+'// &
                            ' Qracs ds'// &
                            ' Qrevp w-'// &
                            ' Qssub s-'// &
                            ' Qsmlt s-'// &
                            ' Qsfr  s+')
                    do nl = mzhyd, klev
                        write(6, 6025) nl, zsigma(nl) * 1.e-3, &
                                10.**2 * qv2D(il, nl) / qvsw2D(il, nl), &
                                10.**2 * qv2D(il, nl) / qvsi2D(il, nl), &
                                10.**2 * vs(il, nl), &
                                10.**2 * vr(il, nl), &
                                (10.**3 * debugV(kv, nl), kv = 9, 16)
                        6025                    format(i3, f11.3, 2f9.1, 9x, 2f9.1, 8f9.6)
                    enddo
                    write(6, 6024)
                    write(6, *) ' '
                endif
            
            endif
        
        enddo
    enddo
#endif

#ifdef EW
    !  Vertical Integrated Energy and Water Content
    !  ============================================
    do il = 1, klon
        enr11D(il) = 0.d0
        wat11D(il) = 0.d0
        
        do kl = 1, klev
            enr11D(il) = enr11D(il) &
                    + (tair2D(il, kl) &
                            - (qw2D(il, kl) + qr2D(il, kl)) * r_LvCp &
                            - (qi2D(il, kl) + qs2D(il, kl)) * r_LsCp) * dsigm1(kl)
            wat11D(il) = wat11D(il) &
                    + (qv2D(il, kl) &
                            + qw2D(il, kl) + qr2D(il, kl) &
                            + qi2D(il, kl) + qs2D(il, kl)) * dsigm1(kl)
        enddo
        enr11D(il) = enr11D(il) * pst2Dn(il) * grvinv
        ! wat11D [m] contains implicit factor 10.**3 [kPa-->Pa] /ro_Wat
        wat11D(il) = wat11D(il) * pst2Dn(il) * grvinv
    
    enddo
#endif
    
    !  Precipitation
    !  =============
    
    !  Hydrometeors Fall Velocity
    !  --------------------------
    
    !  Pristine Ice Crystals Diameter and Fall Velocity
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    do kl = mzhyd, klev
        do il = 1, klon
            
            if(qi2D(il, kl) > eps9) then
                
                ! Sedi_c = 1.0 if qi2D(il,kl)  > eps9
                ! Sedi_c = 0.0 otherwise
                sign_Q = sign(unun, qi2D(il, kl) - eps9)
                Sedi_c = max(zero, sign_Q)
                
                if(ccni2D(il, kl) > 1.) then
                    
                    ! Sedicc = 1.0 if ccni2D(il,kl)  > 1.
                    ! Sedi_c = 0.0 otherwise
                    signCC = sign(unun, ccni2D(il, kl) - 1.)
                    Sedicc = max(zero, signCC)
                    
                    SediOK = Sedi_c * Sedicc
                    
                    ! qid   : Pristine Ice Crystals Diameter,
                    !         Levkov et al. 1992, Contr. Atm. Phys. 65, (5) p.37
                    !         where 6/(pi*ro_I)**1/3 ~ 0.16
                    qid = 0.16d0 * exp(third * log(thous * rolv2D(il, kl) &
                            * max(eps9, qi2D(il, kl)) / max(ccni2D(il, kl), unun)))
                    ! vi : Terminal Fall Velocity for Pristine Ice Crystals
                    ! Levkov et al. 1992, Contr. Atm. Phys. 65, (4) p.37
                    vi(il, kl) = SediOK * 7.d2 * qid &
                            * exp(0.35d0 * log(rolv2D(il, klev) / rolv2D(il, kl)))
                else
                    vi(il, kl) = 0.d0
                endif
            endif
        
        enddo
    enddo
    
    !  Set Up of the Numerical Scheme
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifdef VW
    vwmx = 0.d0
#endif
    ! vrmx = 0.d0
    ! vsmx = 0.d0
    vimx = 0.d0
#ifdef EW
    do il = 1, klon
        ! watf1D : Water Flux (Atmosphere --> Surface)
        watf1D(il) = 0.d0
    enddo
#endif
    
    !  Snow and Rain Fall Velocity (Correction)
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    do kl = mzhyd, klev
        do il = 1, klon
            vi(il, kl) = vi(il, kl) * qi2D(il, kl) / max(qi2D(il, kl), eps9)
            vs(il, kl) = vs(il, kl) * qs2D(il, kl) / max(qs2D(il, kl), eps9)
#ifdef VW
            vw(il, kl) = vw(il, kl) * qw2D(il, kl) / max(qw2D(il, kl), eps9)
#endif
            vr(il, kl) = vr(il, kl) * qr2D(il, kl) / max(qr2D(il, kl), eps9)
            vimx = max(vi(il, kl), vimx)
            ! vsmx = max(vs(il,kl),vsmx)
            ! vrmx = max(vr(il,kl),vrmx)
#ifdef VW
            vwmx = max(vw(il, kl), vwmx)
#endif
#ifdef WH
            if(vsmx > vmmx) then
                vmmx = vsmx
                ilmmi = il
            endif
            if(vrmx > vmmi) then
                vmmi = vrmx
                ilmmi = il
            endif
#endif
        enddo
    enddo
    
    !    dzmn = 10000.
    !    do il=1,klon
    !        dzmn = min(dzmn,(gplv2D(il,mz1)-gplv2D(il,mz))*grvinv)
    !    end do
    
    !  Rain Drops  Precipitation (Implicit Scheme)
    !  -------------------------------------------
    
    do il = 1, klon
        W2xyz8(il, mzhyd - 1) = 0.
    enddo
    
    !  Precipitation Mass & Flux
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~
    do kl = mzhyd, klev
        do il = 1, klon
            ! Air  Mass  [mWE]
            W2xyz1(il, kl) = pst2Dn(il) * dsigm1(kl) * grvinv
            ! Flux Fact. [mWE]
            W2xyz6(il, kl) = xt * vr(il, kl) * rolv2D(il, kl)
        enddo
        
        do il = 1, klon
            ! Rain Mass From abov.
            W2xyz5(il, kl) = qr2D(il, kl) * W2xyz1(il, kl) &
                    + 0.5 * W2xyz8(il, kl - 1)
            if(kl < kk_pp) then
                ! Corr. Bug
                W2xyz7(il, kl) = W2xyz6(il, kl) / W2xyz1(il, kl)
            else
                ! Var. Fact. Flux Limi.
                W2xyz7(il, kl) = min(2., W2xyz6(il, kl) / W2xyz1(il, kl))
            endif
        enddo
        
        do il = 1, klon
            ! Mass Loss
            W2xyz8(il, kl) = W2xyz5(il, kl) * W2xyz7(il, kl) &
                    / (1. + W2xyz7(il, kl) * 0.5)
        enddo
        
        do il = 1, klon
            ! From abov.
            W2xyz5(il, kl) = W2xyz5(il, kl) - W2xyz8(il, kl) &
                    + 0.5 * W2xyz8(il, kl - 1)
            
            !  Cooling from above precipitating flux
            !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tair2D(il, kl) = (tair2D(il, kl) * W2xyz1(il, kl) &
                    + tair2D(il, kl - 1) * W2xyz8(il, kl - 1)) &
                    / (W2xyz1(il, kl) + W2xyz8(il, kl - 1))
        enddo
        
        do il = 1, klon
            qr2D(il, kl) = W2xyz5(il, kl) / W2xyz1(il, kl)
            rnf2D(il, kl) = rnf2D(il, kl) + W2xyz8(il, kl)
            ! evp2D : Net Evap. Mass [mWE]
            evp2D(il, kl) = evp2D(il, kl) + qrevp2D(il, kl) * W2xyz1(il, kl)
        enddo
    enddo
    
    !  Precipitation reaching the Surface
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    do il = 1, klon
        ! dwat contains an implicit factor 10**3 [kPa->Pa]/ro_Wat[kg/m2->m w.e.]
        dwat = W2xyz8(il, klev)
        ratio_temp = (tair2D(il, klev - 1) + tair2D(il, klev - 2) + &
                tair2D(il, klev - 3) + tair2D(il, klev - 4)) / 4.
        ratio_prec = dwat
        ! ratio_rfsf : -1C => snow ; 0C => rain
        ratio_rfsf = max(0., min(1., (ratio_temp - rain_snow_limit) / 2.))
        ! rain2D : rain precipitation height since start of run [m]
        rain2D(il) = rain2D(il) + ratio_prec * ratio_rfsf
        ! snow2D : snow precipitation height since start of run [m]
        snow2D(il) = snow2D(il) + ratio_prec * (1. - ratio_rfsf)
#ifdef EW
        watf1D(il) = watf1D(il) - dwat
#endif
        ! prec2D: rain precipitation height [m]
        !         is reset to zero after included in water reservoir
        prec2D(il) = prec2D(il) + dwat
    enddo
    dwat = 0.0

#ifdef VW
    !  Droplets Precipitation
    !  ----------------------
    ! normally, 0.5 is sufficient to take into account truncation effect
    itmx = int(1.d0 + xt * vwmx / dzmn)
    itmx = max(1, itmx)
    xtmn = xt / itmx
    
    !  Precipitation reaching the Surface
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    do it = 1, itmx
        do il = 1, klon
            ! dwat contains an implicit factor 10**3[kPa->Pa]/ro_Wat[kg/m2->m w.e.]
            dwat = vw(il, klev) * qw2D(il, klev) * rolv2D(il, klev) * xtmn
            
            ! rain2D : rain precipitation height since start of run [m]
            rain2D(il) = rain2D(il) + dwat
            
            watf1D(il) = watf1D(il) - dwat
            
            ! prec2D : rain precipitation height [m]
            !          is reset to zero after included in water reservoir
            !          (cfr. routine SRFfrm_XXX)
            prec2D(il) = prec2D(il) + dwat
        enddo
        
        !  Precipitation elsewhere
        !  ~~~~~~~~~~~~~~~~~~~~~~~
        do kl = klev, mzhyd + 1, -1
            do il = 1, klon
                W2xyz1(il, kl) = qw2D(il, kl) * pst2Dn(il) * dsigm1(kl) &
                        + gravit * xtmn * (qw2D(il, kl - 1) * vw(il, kl - 1) * rolv2D(il, kl - 1) &
                                - qw2D(il, kl) * vw(il, kl) * rolv2D(il, kl))
            enddo
        enddo
        
        do il = 1, klon
            W2xyz1(il, mzhyd) = qw2D(il, mzhyd) * pst2Dn(il) * dsigm1(mzhyd) &
                    - gravit * xtmn * qw2D(il, mzhyd) * vw(il, mzhyd) * rolv2D(il, mzhyd)
        enddo
        
        do kl = mzhyd, klev
            do il = 1, klon
                qw2D(il, kl) = W2xyz1(il, kl) / (pst2Dn(il) * dsigm1(kl))
            enddo
        enddo
    
    enddo
#endif
    
    !  Snow Flakes Precipitation (Implicit Scheme)
    !  -------------------------------------------
    do il = 1, klon
        W2xyz8(il, mzhyd - 1) = 0.
    enddo
    
    !  Precipitation Mass & Flux
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~
    do kl = mzhyd, klev
        do il = 1, klon
            ! Air  Mass  [mWE]
            W2xyz1(il, kl) = pst2Dn(il) * dsigm1(kl) * grvinv
            ! Flux Fact. [mWE]
            W2xyz6(il, kl) = xt * vs(il, kl) * rolv2D(il, kl)
        enddo
        
        do il = 1, klon
            ! Snow Mass From above
            W2xyz5(il, kl) = qs2D(il, kl) * W2xyz1(il, kl) &
                    + 0.5 * W2xyz8(il, kl - 1)
            if(kl < kk_pp) then
                ! Corr. Bug
                W2xyz7(il, kl) = W2xyz6(il, kl) / W2xyz1(il, kl)
            else ! for blowing snow
                ! Var. Fact. Flux Limit
                W2xyz7(il, kl) = &
                        min(2., W2xyz6(il, kl) / W2xyz1(il, kl))
                ! Var. Fact. Flux Limi.
#ifdef BS
                W2xyz7(il, kl) = &
                        min(min(2., 0.5 + (klev - kl) * 0.5), &
                                W2xyz6(il, kl) / W2xyz1(il, kl))
#endif
            endif
        enddo
        
        do il = 1, klon
            ! Mass Loss
            W2xyz8(il, kl) = W2xyz5(il, kl) * W2xyz7(il, kl) / &
                    (1. + W2xyz7(il, kl) * 0.5)
        enddo
        do il = 1, klon
            ! From abov.
            W2xyz5(il, kl) = W2xyz5(il, kl) - W2xyz8(il, kl) &
                    + 0.5 * W2xyz8(il, kl - 1)
            
            !  Cooling from above precipitating flux
            !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tair2D(il, kl) = &
                    (tair2D(il, kl) * W2xyz1(il, kl) &
                            + tair2D(il, kl - 1) * W2xyz8(il, kl - 1)) &
                            / (W2xyz1(il, kl) + W2xyz8(il, kl - 1))
        enddo
        
        do il = 1, klon
            qs2D(il, kl) = W2xyz5(il, kl) / W2xyz1(il, kl)
            snf2D(il, kl) = snf2D(il, kl) + W2xyz8(il, kl)
            ! sbl2D : Sublimation Mass [mWE]
            sbl2D(il, kl) = sbl2D(il, kl) + max(zero, qssub2D(il, kl)) * W2xyz1(il, kl)
            ! dep2D : Condensation Mass [mWE]
            dep2D(il, kl) = dep2D(il, kl) + max(zero, -qssub2D(il, kl)) * W2xyz1(il, kl)
            ! smt2D = UV*Qs*mass_air*dt
            ! qs2D * W2xyz1 : Integrated Mass Transp. [ton/m]
            smt2D(il, kl) = smt2D(il, kl) + &
                    qs2D(il, kl) * W2xyz1(il, kl) &
                            * sqrt((uair2D(il, kl)**2) + (vair2D(il, kl)**2)) &
                            * xt
            !Atm. Sublim.  ratio [kg/kg]
            qssbl2D(il, kl) = qssbl2D(il, kl) + qssub2D(il, kl)
        enddo
    enddo
    
    !  Precipitation reaching the Surface
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    do il = 1, klon
        ! dsno contains an implicit factor 10**3 [kPa->Pa]/ro_Wat[kg/m2->m w.e.]
        dsno = W2xyz8(il, klev)
        
        ! snow2D : snow precipitation height since start of run [m]
        snow2D(il) = snow2D(il) + dsno
#ifdef EW
        watf1D(il) = watf1D(il) - dsno
#endif
        ! snoh2D: snow precipitation height [m]
        !         is reset to zero after included in snow cover
        !        (cfr. routine SRFfrm_sno)
        snoh2D(il) = snoh2D(il) + dsno
    enddo
    dsno = 0.
    
    !  Pristine Ice Crystals Precipitation
    !  -----------------------------------
    
    ! itmx = int(1.d0 + xt * vimx / dzmn) cXF 04/2022
    ! normally, 0.5 is sufficient to take into account truncation effect
    !XF  itmx = max(1,itmx)
    itmx = 1
    ! XF, 20200309: could be very slow with some compilors
    xtmn = xt / itmx
    
    !  Precipitation reaching the Surface
    !  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    ! XF, 20200309: qi <0 if vimx>>
    
    if(vimx > eps9) then
        do it = 1, itmx
            do il = 1, klon
                ! dsno contains an implicit factor 10**3
                ! [kPa->Pa]/ro_Wat[kg/m2->m w.e.]
                dsno = vi(il, klev) * qi2D(il, klev) * rolv2D(il, klev) * xtmn
                
                ! snow2D : snow precipitation height since start of run [m]
                crys2D(il) = crys2D(il) + dsno
                snow2D(il) = snow2D(il) + dsno

#ifdef EW
                watf1D(il) = watf1D(il) - dsno
#endif
                
                ! snoh2D : snow precipitation height [m]
                !          is reset to zero after included in snow cover
                !          (cfr. routine SRFfrm_sno)
                snoh2D(il) = snoh2D(il) + dsno
            enddo
            
            !  Precipitation elsewhere
            !  ~~~~~~~~~~~~~~~~~~~~~~~
            do kl = klev, mzhyd + 1, -1
                do il = 1, klon
                    W2xyz1(il, kl) = qi2D(il, kl) * pst2Dn(il) * dsigm1(kl) &
                            + gravit * xtmn * (qi2D(il, kl - 1) * &
                                    vi(il, kl - 1) * rolv2D(il, kl - 1) &
                                    - qi2D(il, kl) * vi(il, kl) * rolv2D(il, kl))
                    W2xyz5(il, kl) = ccni2D(il, kl) * pst2Dn(il) * dsigm1(kl) &
                            + gravit * xtmn * (ccni2D(il, kl - 1) * &
                                    vi(il, kl - 1) * rolv2D(il, kl - 1) &
                                    - ccni2D(il, kl) * vi(il, kl) * rolv2D(il, kl))
                enddo
            enddo
            
            do il = 1, klon
                W2xyz1(il, mzhyd) = qi2D(il, mzhyd) * &
                        pst2Dn(il) * dsigm1(mzhyd) &
                        - gravit * xtmn * qi2D(il, mzhyd) * &
                                vi(il, mzhyd) * rolv2D(il, mzhyd)
                W2xyz5(il, mzhyd) = ccni2D(il, mzhyd) * &
                        pst2Dn(il) * dsigm1(mzhyd) &
                        - gravit * xtmn * ccni2D(il, mzhyd) * &
                                vi(il, mzhyd) * rolv2D(il, mzhyd)
            enddo
            
            do kl = mzhyd, klev
                do il = 1, klon
                    qi2D(il, kl) = max(0., W2xyz1(il, kl) / &
                            (pst2Dn(il) * dsigm1(kl)))
                    ccni2D(il, kl) = max(0., W2xyz5(il, kl) / &
                            (pst2Dn(il) * dsigm1(kl)))
                enddo
            enddo
        
        enddo
    endif
    
    !  Fractional  Cloudiness ! Guess may be computed (Ek&Mahrt91 fracSC=.T.)
    !  ====================== ! Final value  computed  below
    
    !    #ifdef sc
    !    if (fracld.and..not.fracSC) then
    !    #endif
    if(itPhys == ntHyd2) then
        if(fracld) then
            if(fraCEP) then
                ! ECMWF Large Scale Cloudiness
                ! ----------------------------
                do kl = mzhyd, klev
                    do il = 1, klon
                        cfra2D(il, kl) = (qi2D(il, kl) + qw2D(il, kl) &
                                + qs2D(il, kl) * 0.33 &
                                        * (1. - min(1., exp((tair2D(il, kl) - 258.15) * &
                                                0.1)))) / (0.02 * qvsw2D(il, kl))
                        cfra2D(il, kl) = min(1.000, cfra2D(il, kl))
                        cfra2D(il, kl) = max(0.001, cfra2D(il, kl)) &
                                * max(0., sign(1., qi2D(il, kl) + qw2D(il, kl) &
                                        + qs2D(il, kl) - 3.E-9))
                    enddo
                enddo
            else
                ! XU and Randall  1996, JAS 21, p.3099 (4)
                ! ----------------------------
                do kl = mzhyd, klev
                    do il = 1, klon
                        qvs_wi = qvsw2D(il, kl)
#ifdef wi
                        qvs_wi = max(eps9, ((qi2D(il, kl) + qs2D(il, kl)) * qvsi2D(il, kl) &
                                + qw2D(il, kl) * qvsw2D(il, kl)) / &
                                max(eps9, qi2D(il, kl) + qs2D(il, kl) + qw2D(il, kl)))
#endif
                        relhum = min(relCri, max(qv2D(il, kl), qv_MIN) &
                                / qvs_wi)
                        argexp = ((relCri - relhum) * qvs_wi)**0.49
                        argexp = min(100. * (qi2D(il, kl) + qw2D(il, kl) &
                                + qs2D(il, kl) * 0.33 &
                                        * (1. - min(1., exp((tair2D(il, kl) - 258.15) * &
                                                0.1)))) / &
                                max(eps9, argexp), argmax)
                        
                        cfra2D(il, kl) = (relhum**0.25) * (1. - exp(-argexp))
                    enddo
                enddo
            endif
        else
            !        #ifdef sc
            !        else if (.not.fracld) then
            !        #endif
            do kl = mzhyd, klev
                do il = 1, klon
                    qcloud = qi2D(il, kl) + qw2D(il, kl)
                    if(qcloud > eps9) then
                        
                        ! cfra2D(il,kl) = 1.0 if qcloud > eps9
                        ! cfra2D(il,kl) = 0.0 otherwise
                        signQW = sign(unun, qcloud - eps9)
                        cfra2D(il, kl) = max(zero, signQW)
                    
                    endif
                enddo
            enddo
        endif
    endif
#ifdef EW
    !  Vertically Integrated Energy and Water Content
    !  ==============================================
    do il = 1, klon
        ! Vertical Integrated Energy and Water Content
        enr21D(il) = 0.d0
        wat21D(il) = 0.d0
        
        do kl = 1, klev
            enr21D(il) = enr21D(il) &
                    + (tair2D(il, kl) &
                            - (qw2D(il, kl) + qr2D(il, kl)) * r_LvCp &
                            - (qi2D(il, kl) + qs2D(il, kl)) * r_LsCp) * dsigm1(kl)
            wat21D(il) = wat21D(il) &
                    + (qv2D(il, kl) &
                            + qw2D(il, kl) + qr2D(il, kl) &
                            + qi2D(il, kl) + qs2D(il, kl)) * dsigm1(kl)
        enddo
        
        ! wat21D [m] contains implicit factor 10**3 [kPa-->Pa] /ro_Wat
        enr21D(il) = enr21D(il) * pst2Dn(il) * grvinv
        wat21D(il) = wat21D(il) * pst2Dn(il) * grvinv
    
    enddo
#endif
#ifdef WH
    !  OUTPUT
    !  ======
    if(mod(minuGE, 6) == 0 .and. jsecGE == 0 .and. ilmm > 0) then
        write(6, 1030) jhlr2D(ilmm), minuGE, jsecGE, itexpe, imm, jmm
        1030    format(//, i4, 'LT', i2, 'm', i2, 's (iter.', i6, ')  /  Pt.(', 2i4, ')', &
                /, '  ==========================================')
        write(6, 1031)(kl, 0.1019d0 * gplv2D(ilmm, kl), qv2D(ilmm, kl), &
                (10.**3) * qiold(kl), (10.**3) * qi2D(ilmm, kl), &
                (10.**3) * wihm1(kl), (10.**3) * wihm2(kl), (10.**3) * wicnd(kl), &
                (10.**3) * widep(kl), (10.**3) * wisub(kl), (10.**3) * wimlt(kl), kl = mzhyd, klev)
        1031    format(/, &
                '            |  Water Vapor |  Cloud Ice, Time n & n+1', &
                '   Cloud Ice Nucleation Processes    |', &
                '   Bergeron   Sublimation   Melting  ', &
                /, '  k    z[m] |  qv   [g/kg] |  qi_n [g/kg] qi_n+[g/kg]', &
                ' QiHm1[g/kg] QiHm2[g/kg] QiCnd[g/kg] |', &
                '  QiDep[g/kg] QiSub[g/kg] QiMlt[q/kg]', &
                /, '------------+--------------+-------------------------', &
                '-------------------------------------+', &
                '-------------------------------------', &
                /, (i3, f8.1, ' | ', f12.6, ' | ', 2f12.6, 3d12.4, ' | ', 3d12.4))
        
        write(6, 1032)(kl, 0.1019d0 * gplv2D(ilmm, kl), &
                (10.**3) * W2xyz4(ilmm, kl), (10.**3) * qs2D(ilmm, kl), &
                (10.**3) * wsaut(kl), (10.**3) * wsaci(kl), (10.**3) * wsacw(kl), &
                (10.**3) * wiacr(kl), (10.**3) * wsacr(kl), (10.**3) * wssub(kl), vs(ilmm, kl), &
                kl = mzhyd, klev)
        1032    format(/, &
                '            |  Snow Flakes, Time n&n+1 Autoconver. |', &
                '  Accretion Processes ===> Snow Flakes            |', &
                '  Sublimation | Term.F.Vel', &
                /, '  k    z[m] |  qs_n [g/kg] qs_n+[g/kg] Qsaut[g/kg] |', &
                '  Qsaci[g/kg] Qsacw[g/kg] Qiacr[g/kg] Qsacr[g/kg] |', &
                '  QsSub[g/kg] | vs   [m/s]', &
                /, '------------+--------------------------------------+', &
                '--------------------------------------------------+', &
                '--------------+-----------', &
                /, (i3, f8.1, ' | ', 2f12.6, e12.4, ' | ', 4d12.4, ' | ', e12.4, &
                ' | ', f10.6))
        write(6, 1033)(kl, 0.1019d0 * gplv2D(ilmm, kl), tair2D(ilmm, kl), &
                (10.**3) * qwold(kl), (10.**3) * qw2D(ilmm, kl), &
                (10.**3) * wwevp(kl), 1.d2 * cfra2D(ilmm, kl), kl = mzhyd, klev)
        1033    format(/, &
                /, '            | Temperat.|  Cloud Water, Time n&n+1', &
                ' Condens/Evp | Cloud ', &
                /, '  k    z[m] | T    [K] |  qw_n [g/kg] qw_n+[g/kg]', &
                ' QwEvp[g/kg] | Fract.', &
                /, '------------+----------+-------------------------', &
                '-------------+-------', &
                /, (i3, f8.1, ' | ', f8.3, ' | ', 2f12.6, e12.4, ' | ', f5.1))
        
        write(6, 1034)(kl, 0.1019d0 * gplv2D(ilmm, kl), &
                (10.**3) * W2xyz3(ilmm, kl), (10.**3) * qr2D(ilmm, kl), &
                (10.**3) * wraut(kl), (10.**3) * wracw(kl), (10.**3) * wraci(kl), &
                (10.**3) * wracs(kl), (10.**3) * wrevp(kl), (10.**3) * wsfre(kl), vr(ilmm, kl), &
                kl = mzhyd, klev)
        1034    format(/, &
                /, '            | Rain Drops, Time n&n+1   Autoconver. |', &
                '  Accretion Processes ===> Rain Drops |', &
                '  Evaporation  Freezing   | Term.F.Vel', &
                /, '  k    z[m] |  qr_n [g/kg] qr_n+[g/kg] Qraut[g/kg] |', &
                '  Qracw[g/kg] Qraci[g/kg] Qracs[g/kg] |', &
                '  QrEvp[g/kg] QsFre[g/kg] | vr   [m/s]', &
                /, '------------+--------------------------------------+', &
                '--------------------------------------+', &
                '--------------------------+-----------', &
                /, (i3, f8.1, ' | ', 2f12.6, e12.4, ' | ', 3d12.4, ' | ', 2d12.4, &
                ' | ', f10.6))
        
        do kl = mzhyd, klev
            wihm1(kl) = 0.d0
            wihm2(kl) = 0.d0
            wicnd(kl) = 0.d0
            widep(kl) = 0.d0
            wisub(kl) = 0.d0
            wimlt(kl) = 0.d0
            wwevp(kl) = 0.d0
            wraut(kl) = 0.d0
            wsaut(kl) = 0.d0
            wracw(kl) = 0.d0
            wsacw(kl) = 0.d0
            wsaci(kl) = 0.d0
            wraci(kl) = 0.d0
            wiacr(kl) = 0.d0
            wsacr(kl) = 0.d0
            wracs(kl) = 0.d0
            wrevp(kl) = 0.d0
            wssub(kl) = 0.d0
            wsmlt(kl) = 0.d0
            wsfre(kl) = 0.d0
        enddo
    endif
#endif

#ifdef EW
    !  Vertical Integrated Energy and Water Content: OUTPUT
    !  ====================================================
    if(ilmez > 0) then
        waterb = wat21D(ilmez) - wat11D(ilmez) - watf1D(ilmez)
        write(6, 606) itexpe, &
                enr01D(ilmez), (10.**3) * wat01D(ilmez), &
                mphy2D(ilmez), &
                enr11D(ilmez), (10.**3) * wat11D(ilmez), &
                enr21D(ilmez), (10.**3) * wat21D(ilmez), &
                (10.**3) * watf1D(ilmez), &
                (10.**3) * waterb
        606     format(i9, '  Before mPhy:  E0 =', f12.6, '  W0 = ', &
                f9.6, 3x, a20, 3x, &
                9x, '  Before Prec:  E1 =', f12.6, '  W1 = ', f9.6, &
                9x, '  After  Prec:  E2 =', f12.6, '  W2 = ', f9.6, &
                '  W Flux =', f9.6, &
                '  Div(W) =', e9.3)
    endif
#endif
    if(jmmMAR == 0 .and. jssMAR == 0) then
        IO_loc = IO_gen + 2
        do io = io1, io5
            if(io > 0) then
                il = ioutIO(io)
                if((itexpe > 0 .and. jmmMAR == 0 .and. jssMAR == 0 .and. &
                        ((IO_loc >= 4 .and. jhurGE == 0) .or. &
                                (IO_loc >= 5 .and. mod(jhurGE, 3) == 0) .or. &
                                (IO_loc >= 6))) .or. IO_loc >= 7) then
                    !    ***********
                    !CAa call TIMcor(i, j)
                    !    ***********
                    !CAa write(4, 1037) jdplus, mmplus, jhlr2D(il), minuGE, &
                    !CAa        igrdIO(io), jgrdIO(io)
                    !CAa1037  format(' Ice-Crystal mPhy ', &
                    !CAa        i2, '/', i2, 1x, i2, 'h', i2, 'LT', &
                    !CAa        ' -- Grid Point (', i5, ',', i5, ')' // &
                    !CAa        ' =================================' // &
                    !CAa        '=========================' // &
                    !CAa        '     |  z  [m] | T  [K] | qi[g/kg] |' // &
                    !CAa        ' Ni [m-3] | Ni0[m-3] | vi [m/s] | qs[g/kg] |' // &
                    !CAa        '-----+---------+--------+----------+' // &
                    !CAa        '----------+----------+----------+----------+')
                    do kl = mzhyd, klev
                        write(4, 1038) kl, gplv2D(il, kl) * grvinv, tair2D(il, kl), &
                                qi2D(il, kl) * (10.**3), &
                                ccni2D(il, kl), W2xyz2(il, kl), vi(il, kl), &
                                qs2D(il, kl) * (10.**3)
                    enddo
                    1038                format((i4, ' |', f8.1, ' |', f7.2, ' |', f9.6, ' |', &
                            2(d9.3, ' |'), 2(f9.6, ' |')))
                endif
            endif
        enddo
        IO_loc = IO_gen
    endif

#ifdef WH
    ilmm = ilmmi
#endif
    !  Latent Heat Release
    !  ===================
    do kl = mzhyd, klev
        do il = 1, klon
            pkt0 = pkta2D(il, kl)
            pkta2D(il, kl) = tair2D(il, kl) / pk2D(il, kl)
            hlat2D(il, kl) = tair2D(il, kl) * (1.d0 - pkt0 / pkta2D(il, kl)) &
                    / xt
        enddo
    enddo
    
    !  Limits on Microphysical Variables
    !  =================================
    if(itPhys == ntHyd2) then
        do kl = 1, max(1, mzhyd - 1)
            do il = 1, klon
                qr2D(il, mzhyd) = qr2D(il, mzhyd) + qr2D(il, kl) + qw2D(il, kl)
#ifdef AC
                qs2D(il, mzhyd) = qs2D(il, mzhyd) + qs2D(il, kl) + qi2D(il, kl)
#else
                ! cXF BUGBUG 19/08/2022
                qs2D(il, mzhyd) = qs2D(il, mzhyd) + qs2D(il, kl) + qi2D(il, kl) + &
                        (qv2D(il, kl) - min(qv2D(il, kl), qvsi2D(il, kl)))
#endif
                ccni2D(il, mzhyd) = ccni2D(il, mzhyd) + ccni2D(il, kl)
            enddo
        enddo
        
        do kl = 1, max(1, mzhyd - 1)
            do il = 1, klon
                qv2D(il, kl) = max(qv2D(il, kl), qv_MIN)
                qv2D(il, kl) = min(qv2D(il, kl), qvsi2D(il, kl)) ! mass loss
                qw2D(il, kl) = zero
                qi2D(il, kl) = zero
                ccni2D(il, kl) = zero
                qr2D(il, kl) = zero
                qs2D(il, kl) = zero
                snf2D(il, kl) = zero
                sbl2D(il, kl) = zero
                dep2D(il, kl) = zero
                rnf2D(il, kl) = zero
                evp2D(il, kl) = zero
                smt2D(il, kl) = zero
                qssbl2D(il, kl) = zero
            enddo
        enddo
        
        do kl = mzhyd, klev
            do il = 1, klon
                qw2D(il, kl) = max(zero, qw2D(il, kl))
                qi2D(il, kl) = max(zero, qi2D(il, kl))
                ccni2D(il, kl) = max(zero, ccni2D(il, kl))
                qr2D(il, kl) = max(zero, qr2D(il, kl))
                qs2D(il, kl) = max(zero, qs2D(il, kl))
                snf2D(il, kl) = max(zero, snf2D(il, kl))
                sbl2D(il, kl) = max(zero, sbl2D(il, kl))
                dep2D(il, kl) = max(zero, dep2D(il, kl))
                rnf2D(il, kl) = max(zero, rnf2D(il, kl))
                evp2D(il, kl) = max(zero, evp2D(il, kl))
                smt2D(il, kl) = max(zero, smt2D(il, kl))
                qssbl2D(il, kl) = max(zero, qssbl2D(il, kl))
            enddo
        enddo
        
        do kl = 1, klev
            do il = 1, klon
                W2xyz1(il, kl) = 0.d0
                W2xyz2(il, kl) = 0.d0
                W2xyz3(il, kl) = 0.d0
                W2xyz4(il, kl) = 0.d0
                W2xyz5(il, kl) = 0.d0
                W2xyz6(il, kl) = 0.d0
                W2xyz7(il, kl) = 0.d0
                W2xyz8(il, kl) = 0.d0
            enddo
        enddo
    endif
    return
endsubroutine HYDmic
