#include "MAR_pp.def"
subroutine HYDadv_ver
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   SLOW                                    18-09-2001  MAR |
    ! |   subroutine HYDadv_ver  includes the Vertical  Advection Contribution |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT   (via common block)                                           |
    ! |   ^^^^^    iterun  : Run   Iteration Counter                           |
    ! |              qvDY, qwHY, qrHY, qiHY, ccniHY, qsHY / Time Step n        |
    ! |              qwHY  : Cloud Droplets Concentration             (kg/kg)  |
    ! |              qrHY  : Rain  Drops    Concentration             (kg/kg)  |
    ! |              qvDY  : Water Vapor    Concentration             (kg/kg)  |
    ! |              qiHY  : Ice   Crystals Concentration             (kg/kg)  |
    ! |            ccniHY  : Ice   Crystals Number                             |
    ! |              qsHY  : Snow  Flakes   Concentration             (kg/kg)  |
    ! |                                                                        |
    ! |   OUTPUT :   qvDY, qwHY, qrHY, qiHY, ccniHY, qsHY / Time Step n+1      |
    ! |   ^^^^^^^^                                                             |
    ! |                                                                        |
    ! |   METHOD:  Unstaggered Grid: 1st Accurate in Space Upstream Scheme     |
    ! |   ^^^^^^^^   Staggered Grid: 2nd Accurate in Space                     |
    ! |                                                                        |
    ! | # OPTIONS: #VA: Vertical Average preferred in Centered Conserv Scheme  |
    ! | # ^^^^^^^^ #NS: NO   Slip Surface BC used  in Centered Conserv Scheme  |
    ! | #          #WF: Water Conservation  along the Vertical                 |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_hy
    use mar_sl
    use mar_wk
#ifdef cA
    use mar_ca
#endif
    
    implicit none

#ifdef WA
    integer nadvrd
    common / DYNadv_ver_loc / nadvrd
#endif
    
    logical centrL
#ifdef ZU
    logical adv3rd
    real gat(mx, my, mz), ga0(mx, my)
    data adv3rd/.true./
#endif
    
    ! +--Local  Variables
    ! +  ================
    
    integer i, j, k, m
    integer ntimax, itimax
    
    real cflmax, cflsig, faccfl, dsgm
    real old__u, old__v, old__t
    real qw_0, qr_0, qv_0, qi_0, ccni_0, qs_0
#ifdef WF
    real qwVC, qrVC, qvVC, qiVC, ciVC, qsVC
#endif
#ifdef BS
    real dh_sno
#endif
    
    ! +--DATA
    ! +  ====
    data centrL/.true./
#ifdef UP
    centrL = .false.
#endif
    
    ! +--Slip condition for Mountain Wave Experiments
    ! +  ============================================

#ifdef OM
    do j = jp11, my1
        do i = ip11, mx1
            psigDY(i, j, mz) = 0.0
        enddo
    enddo
#endif
    
    ! +--First and Second Order Schemes
    ! +  ==============================

#ifdef ZU
    if(.not. adv3rd) then
#endif
        
        ! +--Courant Number
        ! +  --------------
        
        cflmax = 0.0
        
        ! +--Centered second Order Scheme on a       staggered Grid
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(centrL .and. staggr) then

#ifdef WA
            write(6, 6001) iterun
            6001        format(i6, ' 6001       centrL .and.       staggr /CFL Number')
#endif
            
            do k = 1, mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz7(i, j, k) = dt * psigDY(i, j, k) &
                                / (pstDYn(i, j) * dsigm1(k) * 2.0)
                        cflsig = abs(WKxyz7(i, j, k) + WKxyz7(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo
            enddo
            
            do j = jp11, my1
                do i = ip11, mx1
                    WKxyz8(i, j, 1) = 0.00
                enddo
            enddo
            
            do k = kp1(1), mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz8(i, j, k) = dt * psigDY(i, j, km1(k)) &
                                / (pstDYn(i, j) * dsigm1(k) * 2.0)
                        cflsig = abs(WKxyz8(i, j, k) + WKxyz8(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo
            enddo
        
        else
            
            ! +--Upstream first  Order Scheme on a       staggered Grid
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(staggr) then

#ifdef WA
                write(6, 6002) iterun
                6002            format(i6, ' 6002 .not. centrL .and.       staggr /Wind Speed')
#endif
                
                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz8(i, j, k) = (psigDY(i, j, k - 1) * dsig_1(k - 1) &
                                    + psigDY(i, j, k) * dsig_1(k)) &
                                    / (dsig_1(k - 1) + dsig_1(k))
                        enddo
                    enddo
                enddo
                
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz8(i, j, 1) = psigDY(i, j, 1) * dsig_1(1) &
                                / (dsig_1(0) + dsig_1(1))
                    enddo
                enddo
                
                ! +--Upstream first  Order Scheme on a  non  staggered Grid
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else

#ifdef WA
                write(6, 6003) iterun
                6003            format(i6, ' 6003 (.not.)centrL.and. .not. staggr /Wind Speed')
#endif
                
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz8(i, j, k) = psigDY(i, j, k)
                        enddo
                    enddo
                enddo
            
            endif
            
            ! +--Centered second Order Scheme on a  non  staggered Grid
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(centrL) then

#ifdef WA
                write(6, 6004) iterun
                6004            format(i6, ' 6004        centrL.and. .not. staggr /CFL Number')
#endif
                
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz7(i, j, k) = dt * WKxyz8(i, j, k) &
                                    / (pstDYn(i, j) * dsigm1(k) * 2.0)
                            cflsig = abs(WKxyz7(i, j, k))
                            cflmax = max(cflsig, cflmax)
                        enddo
                    enddo
                enddo
                
                ! +--Upstream first  Order Scheme on a (non) staggered Grid
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else

#ifdef WA
                write(6, 6005) iterun
                6005            format(i6, ' 6005  .not. centrL.and.(.not.)staggr /CFL Number')
#endif
                
                do k = 1, mmz1
                    do j = jp11, my1
                        do i = ip11, mx1
                            if(WKxyz8(i, j, k) > 0.0) then
                                WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k - 1))
                            else
                                WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k))
                            endif
                            cflsig = abs(WKxyz7(i, j, k))
                            cflmax = max(cflsig, cflmax)
                        enddo
                    enddo
                enddo
                
                k = mz
                do j = jp11, my1
                    do i = ip11, mx1
                        if(WKxyz8(i, j, k) > 0.0) then
                            WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k - 1))
                        else
                            WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k))
                        endif
                        cflsig = abs(WKxyz7(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo
                
                do j = 1, my
                    do i = 1, mx
                        WKxyz7(i, j, 1) = 0.0
                    enddo
                enddo
                
                ! +--Work Array Reset
                ! +  ~~~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz8(i, j, k) = 0.0
                        enddo
                    enddo
                enddo
            
            endif
        
        endif
        
        ! +--Set Up    of the Local Split Time Differencing
        ! +  ----------------------------------------------
        
        cflmax = 2.0 * cflmax
        ! +...      restricted CFL Criterion
        
        ntimax = cflmax
        if(centrL) then
            ntimax = max(2, ntimax)
#ifdef WA
            write(6, 6006) ntimax
            6006        format(i6, ' 6006        centrL.and.(.not.)staggr /Nb Iterat.')
#endif
        else
            ntimax = max(1, ntimax)
#ifdef WA
            write(6, 6007) ntimax
            6007        format(i6, ' 6007  .not. centrL.and.(.not.)staggr /Nb Iterat.')
#endif
        endif
        
        ! +--Update of CFL Number
        ! +  ~~~~~~~~~~~~~~~~~~~~
        if(ntimax > 1) then
            faccfl = 1.0d+0 / ntimax
            do k = 1, mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz7(i, j, k) = WKxyz7(i, j, k) * faccfl
                        WKxyz8(i, j, k) = WKxyz8(i, j, k) * faccfl
                    enddo
                enddo
            enddo
        endif
        
        ! +--OUTPUT for Verification
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~
#ifdef WA
        nadvrd = nadvrd + 1
        write(6, 6000) nadvrd, cflmax, ntimax
        6000    format(i6, ' CFLmax ', 3x, ' ', 3x, '  =', f7.4, &
                6x, ' ntimax ', 8x, '   =', i4)
#endif
        
        ! +--Warm Water Conservation
        ! +  -----------------------
#ifdef WF
        do j = 1, my
            do i = 1, mx
                WKxy4(i, j) = 0.0d+0
                WKxy5(i, j) = 0.0d+0
                WKxy6(i, j) = 0.0d+0
                WKxy7(i, j) = 0.0d+0
                WKxy8(i, j) = 0.0d+0
                WKxy9(i, j) = 0.0d+0
                do k = 1, mz
                    WKxy4(i, j) = WKxy4(i, j) + dsigm1(k) * qvDY(i, j, k)
                    WKxy5(i, j) = WKxy5(i, j) + dsigm1(k) * qwHY(i, j, k)
                    WKxy6(i, j) = WKxy6(i, j) + dsigm1(k) * qrHY(i, j, k)
                    WKxy7(i, j) = max(WKxy7(i, j), qvDY(i, j, k))
                    WKxy8(i, j) = max(WKxy8(i, j), qwHY(i, j, k))
                    WKxy9(i, j) = max(WKxy9(i, j), qrHY(i, j, k))
                enddo
            enddo
        enddo
#endif
        
        ! +--Warm Water
        ! +--Start Vertical Advection
        ! +  ------------------------
        
        if(centrL) then
            
            if(staggr) then

#ifdef WA
                write(6, 6008)
                6008            format(6x, ' 6008        centrL.and.       staggr /A Contrib.')
#endif
                
                ! +--2nd Order Centered Energy conserving:  Local Split Time Differencing
                ! +  ~~~~~~~~~ (Haltiner & Williams 1980 7.2.2, (7-47b) p.220) ~~~~~~~~~~
                ! +            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                do itimax = 1, ntimax
                    
                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then
                        
                        do j = jp11, my1
                            
                            ! +--Vertical Differences
                            
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = qwHY(i, j, k)
                                WKxzq(i, k) = qrHY(i, j, k)
                                WKxzx(i, k) = qvDY(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (qwHY(i, j, k) * dsigm1(k) * 2.0 &
                                        + qwHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (qrHY(i, j, k) * dsigm1(k) * 2.0 &
                                        + qrHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (qvDY(i, j, k) * dsigm1(k) * 2.0 &
                                        + qvDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                qw_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qw_0)
                                qr_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - qr_0)
                                qv_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qv_0)
                            enddo
                            
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = qwHY(i, j, k)
                                    WKxzq(i, k) = qrHY(i, j, k)
                                    WKxzx(i, k) = qvDY(i, j, k)

#ifdef VA
                                    WKxzp(i, k) = (qwHY(i, j, k - 1) * dsigm1(k - 1) &
                                            + qwHY(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + qwHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (qrHY(i, j, k - 1) * dsigm1(k - 1) &
                                            + qrHY(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + qrHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (qvDY(i, j, k - 1) * dsigm1(k - 1) &
                                            + qvDY(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + qvDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = qwHY(i, j, k)
                                WKxzq(i, k) = qrHY(i, j, k)
                                WKxzx(i, k) = qvDY(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (qwHY(i, j, k - 1) * dsigm1(k - 1) &
                                        + qwHY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (qrHY(i, j, k - 1) * dsigm1(k - 1) &
                                        + qrHY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (qvDY(i, j, k - 1) * dsigm1(k - 1) &
                                        + qvDY(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qvapSL(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            
                            ! +--Advection Contribution
                            
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxza(i, k)
                                    WKxyz1(i, j, k) = qwHY(i, j, k) - WKxzd(i, k)
                                    WKxyz4(i, j, k) = qwHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                    
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzb(i, k)
                                    WKxyz2(i, j, k) = qrHY(i, j, k) - WKxzd(i, k)
                                    WKxyz5(i, j, k) = qrHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                    
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzc(i, k)
                                    WKxyz3(i, j, k) = qvDY(i, j, k) - WKxzd(i, k)
                                    WKxyz6(i, j, k) = qvDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo
                            
                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                        + WKxyz8(i, j, k) * WKxza(i, k)
                                WKxyz1(i, j, k) = qwHY(i, j, k) - WKxzd(i, k)
                                WKxyz4(i, j, k) = qwHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                        + WKxyz8(i, j, k) * WKxzb(i, k)
                                WKxyz2(i, j, k) = qrHY(i, j, k) - WKxzd(i, k)
                                WKxyz5(i, j, k) = qrHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                        + WKxyz8(i, j, k) * WKxzc(i, k)
                                WKxyz3(i, j, k) = qvDY(i, j, k) - WKxzd(i, k)
                                WKxyz6(i, j, k) = qvDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                            enddo
                        
                        enddo
                        
                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then
                        
                        ! +--Vertical Differences
                        
                        do j = jp11, my1
                            
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                qw_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qw_0)
                                qr_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - qr_0)
                                qv_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qv_0)
                            enddo
                            
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz4(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz5(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz6(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qvapSL(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            
                            ! +--Advection Contribution
                            
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxza(i, k)
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))
                                    
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzb(i, k)
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))
                                    
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzc(i, k)
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo
                            
                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                        + WKxyz8(i, j, k) * WKxza(i, k)
                                old__u = WKxyz1(i, j, k)
                                WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))
                                
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                        + WKxyz8(i, j, k) * WKxzb(i, k)
                                old__v = WKxyz2(i, j, k)
                                WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))
                                
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                        + WKxyz8(i, j, k) * WKxzc(i, k)
                                old__t = WKxyz3(i, j, k)
                                WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                            enddo
                        
                        enddo
                        
                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else
                        
                        do j = jp11, my1
                            
                            ! +--Vertical Differences
                            
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                qw_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qw_0)
                                qr_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - qr_0)
                                qv_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qv_0)
                            enddo
                            
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz4(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz5(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz6(i, j, k) * dsigm1(k) * 2.0d+0 &
                                            + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)

#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz4(i, j, k) * dsigm1(k) * 2.0d+0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz5(i, j, k) * dsigm1(k) * 2.0d+0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz6(i, j, k) * dsigm1(k) * 2.0d+0) / dsgm
#endif
                            enddo
                            
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qvapSL(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            
                            ! +--Hydromet.Advection
                            
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    qwHY(i, j, k) = WKxyz1(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                                    + WKxyz8(i, j, k) * WKxza(i, k))
                                    qrHY(i, j, k) = WKxyz2(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                                    + WKxyz8(i, j, k) * WKxzb(i, k))
                                enddo
                            enddo
                            
                            k = mmz
                            do i = ip11, mx1
                                qwHY(i, j, k) = WKxyz1(i, j, k) &
                                        - (WKxyz7(i, j, k) * WKxy1(i, j) &
                                                + WKxyz8(i, j, k) * WKxza(i, k))
                                qrHY(i, j, k) = WKxyz2(i, j, k) &
                                        - (WKxyz7(i, j, k) * WKxy2(i, j) &
                                                + WKxyz8(i, j, k) * WKxzb(i, k))
                            enddo
                            
                            ! +--Wat.Vapr.Advect.avoids double Counting in case of convective Adjustment
                            
                            do k = 1, mmz1
                                do i = ip11, mx1
#ifdef cA
                                    if(adj_CA(i, j) == 0) then
#endif
                                        qvDY(i, j, k) = WKxyz3(i, j, k) &
                                                - (WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                                        + WKxyz8(i, j, k) * WKxzc(i, k))
#ifdef cA
                                    endif
#endif
                                enddo
                            enddo
                            
                            k = mmz
                            do i = ip11, mx1
#ifdef cA
                                if(adj_CA(i, j) == 0) then
#endif
                                    qvDY(i, j, k) = WKxyz3(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxy3(i, j) &
                                                    + WKxyz8(i, j, k) * WKxzc(i, k))
#ifdef cA
                                endif
#endif
                            enddo
                            ! +
                        enddo
                        ! +
                    endif
                    
                    ! +--Warm Water
                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------
                
                enddo
                
                ! +--Warm Water
                ! +--2nd Order Centered Leap-Frog Backward: Local Split Time Differencing
                ! +  --------------------------------------------------------------------
                ! +
            else
                ! +
#ifdef WA
                write(6, 6009)
                6009            format(6x, ' 6009        centrL.and. .not. staggr /A Contrib.')
#endif
                ! +
                do itimax = 1, ntimax
                    ! +
                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Advection Increment
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qw_0 = qwHY(i, j, k)
                                WKxza(i, k) = (qwHY(i, j, k + 1) - qw_0) &
                                        * WKxyz7(i, j, k)
                                qr_0 = qrHY(i, j, k)
                                WKxzb(i, k) = (qrHY(i, j, k + 1) - qrHY(i, j, k)) &
                                        * WKxyz7(i, j, k)
                                qv_0 = qvDY(i, j, k)
                                WKxzc(i, k) = (qvDY(i, j, k + 1) - qvDY(i, j, k)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (qwHY(i, j, k + 1) - qwHY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (qrHY(i, j, k + 1) - qrHY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (qvDY(i, j, k + 1) - qvDY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -qwHY(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -qrHY(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qvapSL(i, j) - qvDY(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    WKxyz1(i, j, k) = qwHY(i, j, k) - WKxza(i, k)
                                    WKxyz4(i, j, k) = qwHY(i, j, k) - (WKxza(i, k) + WKxza(i, k))
                                    WKxyz2(i, j, k) = qrHY(i, j, k) - WKxzb(i, k)
                                    WKxyz5(i, j, k) = qrHY(i, j, k) - (WKxzb(i, k) + WKxzb(i, k))
                                    WKxyz3(i, j, k) = qvDY(i, j, k) - WKxzc(i, k)
                                    WKxyz6(i, j, k) = qvDY(i, j, k) - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo
                        enddo
                        ! +
                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then
                        ! +
                        ! +--Advection Increment
                        ! +
                        do j = jp11, my1
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qw_0 = WKxyz4(i, j, k)
                                qr_0 = WKxyz5(i, j, k)
                                qv_0 = WKxyz6(i, j, k)
                                ! +
                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - qw_0) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - qr_0) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - qv_0) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qvapSL(i, j) - WKxyz6(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxza(i, k) + WKxza(i, k))
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzb(i, k) + WKxzb(i, k))
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo
                            ! +
                        enddo
                        ! +
                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Advection Increment
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qw_0 = WKxyz4(i, j, k)
                                qr_0 = WKxyz5(i, j, k)
                                qv_0 = WKxyz6(i, j, k)
                                ! +
                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - qw_0) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - qr_0) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - qv_0) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qvapSL(i, j) - WKxyz6(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Hydromet.Advection
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    qwHY(i, j, k) = WKxyz1(i, j, k) - WKxza(i, k)
                                    qrHY(i, j, k) = WKxyz2(i, j, k) - WKxzb(i, k)
                                enddo
                                ! +
                                ! +--Wat.Vapr.Advect.avoids double Counting in case of convective Adjustment
                                ! +
                                do i = ip11, mx1
#ifdef cA
                                    if(adj_CA(i, j) == 0) then
#endif
                                        qvDY(i, j, k) = WKxyz3(i, j, k) - WKxzc(i, k)
#ifdef cA
                                    endif
#endif
                                enddo
                            enddo
                            ! +
                        enddo
                        ! +
                    endif
                    ! +
                    ! +
                    ! +--Warm Water
                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------
                    ! +
                enddo
                ! +
            endif
            ! +
            ! +
            ! +--Warm Water
            ! +--First  Order Upstream Scheme:          Local Split Time Differencing
            ! +  --------------------------------------------------------------------
            ! +
        else
            ! +
#ifdef WA
            write(6, 6010)
            6010        format(6x, ' 6010  .not. centrL.and.(.not.)staggr /A Contrib.')
#endif
            ! +
            do itimax = 1, ntimax
                ! +
                ! +--Auxiliary Variables
                ! +  ~~~~~~~~~~~~~~~~~~~
#ifdef WA
                write(6, 6011) itimax, WKxyz1(imez, jmez, mz1), WKxyz1(imez, jmez, mz) &
                        , qwHY(imez, jmez, mz1), qwHY(imez, jmez, mz)
                6011            format(6x, ' 6011  .not. centrL.and.(.not.)staggr /A Contrib.', &
                        4f9.6)
#endif
                ! +
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = qwHY(i, j, k)
                            WKxyz2(i, j, k) = qrHY(i, j, k)
                            WKxyz3(i, j, k) = qvDY(i, j, k)
                        enddo
                    enddo
                enddo
                ! +
                ! +--Vertical Differences
                ! +  ~~~~~~~~~~~~~~~~~~~~
                k = 1
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz4(i, j, k) = 0.0d+0
                        WKxyz5(i, j, k) = 0.0d+0
                        WKxyz6(i, j, k) = 0.0d+0
                    enddo
                enddo
                ! +
                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = WKxyz1(i, j, k) - WKxyz1(i, j, k - 1)
                            WKxyz5(i, j, k) = WKxyz2(i, j, k) - WKxyz2(i, j, k - 1)
                            WKxyz6(i, j, k) = WKxyz3(i, j, k) - WKxyz3(i, j, k - 1)
                        enddo
                    enddo
                enddo
                ! +
                k = mzz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxy1(i, j) = -WKxyz1(i, j, k - 1)
                        WKxy2(i, j) = -WKxyz2(i, j, k - 1)
                        WKxy3(i, j) = qvapSL(i, j) - WKxyz3(i, j, k - 1)
                    enddo
                enddo
                ! +
                ! +--Advection Contribution
                ! +  ~~~~~~~~~~~~~~~~~~~~~~
                do k = 1, mmz1
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = qwHY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k + 1)
                            WKxyz2(i, j, k) = qrHY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k + 1)
                            WKxyz3(i, j, k) = qvDY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k + 1)
                        enddo
                    enddo
                enddo
                ! +
                k = mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz1(i, j, k) = qwHY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy1(i, j)
                        WKxyz2(i, j, k) = qrHY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy2(i, j)
                        WKxyz3(i, j, k) = qvDY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy3(i, j)
                    enddo
                enddo
                ! +
                ! +--Wind    Update
                ! +  ~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qwHY(i, j, k) = WKxyz1(i, j, k)
                            qrHY(i, j, k) = WKxyz2(i, j, k)
                        enddo
                        ! +
                        ! +--Pot.Temp.Update avoids double Counting in case of convective Adjustment
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        do i = ip11, mx1
#ifdef cA
                            if(adj_CA(i, j) == 0) then
#endif
                                qvDY(i, j, k) = WKxyz3(i, j, k)
#ifdef cA
                            endif
#endif
                        enddo
                    enddo
                enddo
                ! +
                ! +
                ! +--End of the                             Local Split Time Differencing
                ! +  --------------------------------------------------------------------
                ! +
#ifdef WA
                write(6, 6012) itimax, WKxyz1(imez, jmez, mz1), WKxyz1(imez, jmez, mz) &
                        , qwHY(imez, jmez, mz1), qwHY(imez, jmez, mz)
                6012            format(6x, ' 6012  .not. centrL.and.(.not.)staggr /A Contrib.', &
                        4f9.6)
#endif
            enddo
            ! +
        endif
#ifdef WF
        ! +
        ! +
        ! +--Warm Water Conservation
        ! +  -----------------------
        ! +
        do j = 1, my
            do i = 1, mx
                qvVC = 0.0d+0
                qwVC = 0.0d+0
                qrVC = 0.0d+0
                do k = 1, mz
                    ! +
                    ! +--Flux Limitor
                    ! +  ~~~~~~~~~~~~
                    qvDY(i, j, k) = max(eps9, qvDY(i, j, k))
                    qvDY(i, j, k) = min(WKxy7(i, j), qvDY(i, j, k))
                    qwHY(i, j, k) = max(zero, qwHY(i, j, k))
                    qwHY(i, j, k) = min(WKxy8(i, j), qwHY(i, j, k))
                    qrHY(i, j, k) = max(zero, qrHY(i, j, k))
                    qrHY(i, j, k) = min(WKxy9(i, j), qrHY(i, j, k))
                    ! +
                    ! +--Column Average
                    ! +  ~~~~~~~~~~~~~~
                    qvVC = qvVC + dsigm1(k) * qvDY(i, j, k)
                    qwVC = qwVC + dsigm1(k) * qwHY(i, j, k)
                    qrVC = qrVC + dsigm1(k) * qrHY(i, j, k)
                enddo
                ! +
                ! +--Surface Boundary Flux
                ! +  ~~~~~~~~~~~~~~~~~~~~~
                qvVC = qvVC - psigDY(i, j, mz) &
                        * (qvDY(i, j, mz) - qvapSL(i, j)) * dt / pstDYn(i, j)
                qwVC = qwVC - psigDY(i, j, mz) &
                        * qwHY(i, j, mz) * dt / pstDYn(i, j)
                qrVC = qrVC - psigDY(i, j, mz) &
                        * qrHY(i, j, mz) * dt / pstDYn(i, j)
                ! +
                ! +--Mass Restore
                ! +  ~~~~~~~~~~~~
                do k = 1, mz
                    qvDY(i, j, k) = qvDY(i, j, k) * WKxy4(i, j) / max(eps12, qvVC)
                    qwHY(i, j, k) = qwHY(i, j, k) * WKxy5(i, j) / max(eps12, qwVC)
                    qrHY(i, j, k) = qrHY(i, j, k) * WKxy6(i, j) / max(eps12, qrVC)
                enddo
            enddo
        enddo
#endif
        ! +
        ! +
        ! +--Warm Water
        ! +--Work Arrays Reset
        ! +  -----------------
        ! +
        do j = 1, my
            do i = 1, mx
                WKxy1(i, j) = 0.0
                WKxy2(i, j) = 0.0
                WKxy3(i, j) = 0.0
                WKxy4(i, j) = 0.0
                WKxy5(i, j) = 0.0
                WKxy6(i, j) = 0.0
            enddo
        enddo
        ! +
        do k = 1, mz
            do i = 1, mx
                WKxza(i, k) = 0.0
                WKxzb(i, k) = 0.0
                WKxzc(i, k) = 0.0
                WKxzd(i, k) = 0.0
            enddo
        enddo
        ! +
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    WKxyz1(i, j, k) = 0.0
                    WKxyz2(i, j, k) = 0.0
                    WKxyz3(i, j, k) = 0.0
                    WKxyz4(i, j, k) = 0.0
                    WKxyz5(i, j, k) = 0.0
                    WKxyz6(i, j, k) = 0.0
                enddo
            enddo
        enddo
        ! +
#ifdef WF
        ! +
        ! +--Ice  Water Conservation
        ! +  -----------------------
        ! +
        do j = 1, my
            do i = 1, mx
                WKxy4(i, j) = 0.0d+0
                WKxy5(i, j) = 0.0d+0
                WKxy6(i, j) = 0.0d+0
                WKxy7(i, j) = 0.0d+0
                WKxy8(i, j) = 0.0d+0
                WKxy9(i, j) = 0.0d+0
                do k = 1, mz
                    WKxy4(i, j) = WKxy4(i, j) + dsigm1(k) * ccniHY(i, j, k)
                    WKxy5(i, j) = WKxy5(i, j) + dsigm1(k) * qiHY(i, j, k)
                    WKxy6(i, j) = WKxy6(i, j) + dsigm1(k) * qsHY(i, j, k)
                    WKxy7(i, j) = max(WKxy7(i, j), ccniHY(i, j, k))
                    WKxy8(i, j) = max(WKxy8(i, j), qiHY(i, j, k))
                    WKxy9(i, j) = max(WKxy9(i, j), qsHY(i, j, k))
                enddo
            enddo
        enddo
        ! +
#endif
        ! +
        ! +--Ice  Water
        ! +--Start Vertical Advection
        ! +  ------------------------
        ! +
        if(centrL) then
            ! +
            if(staggr) then
                ! +
                ! +--2nd Order Centered Energy conserving:  Local Split Time Differencing
                ! +  ~~~~~~~~~ (Haltiner & Williams 1980 7.2.2, (7-47b) p.220) ~~~~~~~~~~
                ! +            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                do itimax = 1, ntimax
                    ! +
                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Vertical Differences
                            ! +
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = qiHY(i, j, k)
                                WKxzq(i, k) = ccniHY(i, j, k)
                                WKxzx(i, k) = qsHY(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (qiHY(i, j, k) * dsigm1(k) * 2.0 &
                                        + qiHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (ccniHY(i, j, k) * dsigm1(k) * 2.0 &
                                        + ccniHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (qsHY(i, j, k) * dsigm1(k) * 2.0 &
                                        + qsHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                qi_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qi_0)
                                ccni_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - ccni_0)
                                qs_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qs_0)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = qiHY(i, j, k)
                                    WKxzq(i, k) = ccniHY(i, j, k)
                                    WKxzx(i, k) = qsHY(i, j, k)
                                    ! +
#ifdef VA
                                    WKxzp(i, k) = (qiHY(i, j, k - 1) * dsigm1(k - 1) &
                                            + qiHY(i, j, k) * dsigm1(k) * 2.0 &
                                            + qiHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (ccniHY(i, j, k - 1) * dsigm1(k - 1) &
                                            + ccniHY(i, j, k) * dsigm1(k) * 2.0 &
                                            + ccniHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (qsHY(i, j, k - 1) * dsigm1(k - 1) &
                                            + qsHY(i, j, k) * dsigm1(k) * 2.0 &
                                            + qsHY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = qiHY(i, j, k)
                                WKxzq(i, k) = ccniHY(i, j, k)
                                WKxzx(i, k) = qsHY(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (qiHY(i, j, k - 1) * dsigm1(k - 1) &
                                        + qiHY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (ccniHY(i, j, k - 1) * dsigm1(k - 1) &
                                        + ccniHY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (qsHY(i, j, k - 1) * dsigm1(k - 1) &
                                        + qsHY(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            ! +
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qsrfHY(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxza(i, k)
                                    WKxyz1(i, j, k) = qiHY(i, j, k) - WKxzd(i, k)
                                    WKxyz4(i, j, k) = qiHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                    ! +
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzb(i, k)
                                    WKxyz2(i, j, k) = ccniHY(i, j, k) - WKxzd(i, k)
                                    WKxyz5(i, j, k) = ccniHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                    ! +
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzc(i, k)
                                    WKxyz3(i, j, k) = qsHY(i, j, k) - WKxzd(i, k)
                                    WKxyz6(i, j, k) = qsHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                        + WKxyz8(i, j, k) * WKxza(i, k)
                                WKxyz1(i, j, k) = qiHY(i, j, k) - WKxzd(i, k)
                                WKxyz4(i, j, k) = qiHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                ! +
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                        + WKxyz8(i, j, k) * WKxzb(i, k)
                                WKxyz2(i, j, k) = ccniHY(i, j, k) - WKxzd(i, k)
                                WKxyz5(i, j, k) = ccniHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                ! +
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                        + WKxyz8(i, j, k) * WKxzc(i, k)
                                WKxyz3(i, j, k) = qsHY(i, j, k) - WKxzd(i, k)
                                WKxyz6(i, j, k) = qsHY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                            enddo
                            ! +
                        enddo
                        ! +
                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then
                        ! +
                        ! +--Vertical Differences
                        ! +
                        do j = jp11, my1
                            ! +
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                qi_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qi_0)
                                ccni_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - ccni_0)
                                qs_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qs_0)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)
                                    ! +
#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz4(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz5(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz6(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            ! +
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qsrfHY(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxza(i, k)
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))
                                    ! +
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzb(i, k)
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))
                                    ! +
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                            + WKxyz8(i, j, k) * WKxzc(i, k)
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                        + WKxyz8(i, j, k) * WKxza(i, k)
                                old__u = WKxyz1(i, j, k)
                                WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))
                                ! +
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                        + WKxyz8(i, j, k) * WKxzb(i, k)
                                old__v = WKxyz2(i, j, k)
                                WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))
                                ! +
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                        + WKxyz8(i, j, k) * WKxzc(i, k)
                                old__t = WKxyz3(i, j, k)
                                WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                            enddo
                            ! +
                        enddo
                        ! +
                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Vertical Differences
                            ! +
                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                        + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                qi_0 = WKxzp(i, k)
                                WKxza(i, k) = (WKxzp(i, k) - qi_0)
                                ccni_0 = WKxzq(i, k)
                                WKxzb(i, k) = (WKxzq(i, k) - ccni_0)
                                qs_0 = WKxzx(i, k)
                                WKxzc(i, k) = (WKxzx(i, k) - qs_0)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)
                                    ! +
#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                            + WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                            + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
                                ! +
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz4(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz5(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                        + WKxyz6(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo
                            ! +
                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo
                            ! +
                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0
                                WKxy2(i, j) = 0.0
                                WKxy3(i, j) = 0.0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (qsrfHY(i, j) - WKxzx(i, k - 1))
#endif
                            enddo
                            ! +
                            ! +--Hydromet.Advection
                            ! +
                            do k = 1, mmz1
                                do i = ip11, mx1
                                    qiHY(i, j, k) = WKxyz1(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                                    + WKxyz8(i, j, k) * WKxza(i, k))
                                    ccniHY(i, j, k) = WKxyz2(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                                    + WKxyz8(i, j, k) * WKxzb(i, k))
                                    qsHY(i, j, k) = WKxyz3(i, j, k) &
                                            - (WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                                    + WKxyz8(i, j, k) * WKxzc(i, k))
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                qiHY(i, j, k) = WKxyz1(i, j, k) &
                                        - (WKxyz7(i, j, k) * WKxy1(i, j) &
                                                + WKxyz8(i, j, k) * WKxza(i, k))
                                ccniHY(i, j, k) = WKxyz2(i, j, k) &
                                        - (WKxyz7(i, j, k) * WKxy2(i, j) &
                                                + WKxyz8(i, j, k) * WKxzb(i, k))
                                qsHY(i, j, k) = WKxyz3(i, j, k) &
                                        - (WKxyz7(i, j, k) * WKxy3(i, j) &
                                                + WKxyz8(i, j, k) * WKxzc(i, k))
                            enddo
                            ! +
                        enddo
                        ! +
                    endif
                    ! +
                    ! +
                    ! +--Ice  Water
                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------
                    ! +
                enddo
                ! +
                ! +
                ! +--Ice  Water
                ! +--2nd Order Centered Leap-Frog Backward: Local Split Time Differencing
                ! +  --------------------------------------------------------------------
                ! +
            else
                ! +
                do itimax = 1, ntimax
                    ! +
                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Advection Increment
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qi_0 = qiHY(i, j, k)
                                WKxza(i, k) = (qiHY(i, j, k + 1) - qi_0) &
                                        * WKxyz7(i, j, k)
                                ccni_0 = ccniHY(i, j, k)
                                WKxzb(i, k) = (ccniHY(i, j, k + 1) - ccniHY(i, j, k)) &
                                        * WKxyz7(i, j, k)
                                qv_0 = qsHY(i, j, k)
                                WKxzc(i, k) = (qsHY(i, j, k + 1) - qsHY(i, j, k)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (qiHY(i, j, k + 1) - qiHY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (ccniHY(i, j, k + 1) - ccniHY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (qsHY(i, j, k + 1) - qsHY(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -qiHY(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -ccniHY(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qsrfHY(i, j) - qsHY(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    WKxyz1(i, j, k) = qiHY(i, j, k) - WKxza(i, k)
                                    WKxyz4(i, j, k) = qiHY(i, j, k) - (WKxza(i, k) + WKxza(i, k))
                                    WKxyz2(i, j, k) = ccniHY(i, j, k) - WKxzb(i, k)
                                    WKxyz5(i, j, k) = ccniHY(i, j, k) - (WKxzb(i, k) + WKxzb(i, k))
                                    WKxyz3(i, j, k) = qsHY(i, j, k) - WKxzc(i, k)
                                    WKxyz6(i, j, k) = qsHY(i, j, k) - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo
                        enddo
                        ! +
                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then
                        ! +
                        ! +--Advection Increment
                        ! +
                        do j = jp11, my1
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qi_0 = WKxyz4(i, j, k)
                                ccni_0 = WKxyz5(i, j, k)
                                qv_0 = WKxyz6(i, j, k)
                                ! +
                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - qi_0) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - ccni_0) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - qv_0) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qsrfHY(i, j) - WKxyz6(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Advection Contribution
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxza(i, k) + WKxza(i, k))
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzb(i, k) + WKxzb(i, k))
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo
                            ! +
                        enddo
                        ! +
                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else
                        ! +
                        do j = jp11, my1
                            ! +
                            ! +--Advection Increment
                            ! +
                            k = 1
                            do i = ip11, mx1
                                qi_0 = WKxyz4(i, j, k)
                                ccni_0 = WKxyz5(i, j, k)
                                qv_0 = WKxyz6(i, j, k)
                                ! +
                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - qi_0) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - ccni_0) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - qv_0) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                            * WKxyz7(i, j, k)
                                enddo
                            enddo
                            ! +
                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                        * WKxyz7(i, j, k)
                                WKxzc(i, k) = (qsrfHY(i, j) - WKxyz6(i, j, k - 1)) &
                                        * WKxyz7(i, j, k)
                            enddo
                            ! +
                            ! +--Hydromet.Advection
                            ! +
                            do k = 1, mmz
                                do i = ip11, mx1
                                    qiHY(i, j, k) = WKxyz1(i, j, k) - WKxza(i, k)
                                    ccniHY(i, j, k) = WKxyz2(i, j, k) - WKxzb(i, k)
                                    qsHY(i, j, k) = WKxyz3(i, j, k) - WKxzc(i, k)
                                enddo
                            enddo
                            ! +
                        enddo
                        ! +
                    endif
                    ! +
                    ! +
                    ! +--Ice  Water
                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------
                    ! +
                enddo
                ! +
            endif
            ! +
            ! +
            ! +--Ice  Water
            ! +--First  Order Upstream Scheme:          Local Split Time Differencing
            ! +  --------------------------------------------------------------------
            ! +
        else
            ! +
            do itimax = 1, ntimax
                ! +
                ! +--Auxiliary Variables
                ! +  ~~~~~~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = qiHY(i, j, k)
                            WKxyz2(i, j, k) = ccniHY(i, j, k)
                            WKxyz3(i, j, k) = qsHY(i, j, k)
                        enddo
                    enddo
                enddo
                ! +
                ! +--Vertical Differences
                ! +  ~~~~~~~~~~~~~~~~~~~~
                k = 1
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz4(i, j, k) = 0.0d+0
                        WKxyz5(i, j, k) = 0.0d+0
                        WKxyz6(i, j, k) = 0.0d+0
                    enddo
                enddo
                ! +
                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = WKxyz1(i, j, k) - WKxyz1(i, j, k - 1)
                            WKxyz5(i, j, k) = WKxyz2(i, j, k) - WKxyz2(i, j, k - 1)
                            WKxyz6(i, j, k) = WKxyz3(i, j, k) - WKxyz3(i, j, k - 1)
                        enddo
                    enddo
                enddo
                ! +
                k = mzz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxy1(i, j) = -WKxyz1(i, j, k - 1)
                        WKxy2(i, j) = -WKxyz2(i, j, k - 1)
                        WKxy3(i, j) = qsrfHY(i, j) - WKxyz3(i, j, k - 1)
                    enddo
                enddo
                ! +
                ! +--Advection Contribution
                ! +  ~~~~~~~~~~~~~~~~~~~~~~
                do k = 1, mmz1
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = qiHY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k + 1)
                            WKxyz2(i, j, k) = ccniHY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k + 1)
                            WKxyz3(i, j, k) = qsHY(i, j, k) &
                                    + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                    + max(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k + 1)
                        enddo
                    enddo
                enddo
                ! +
                k = mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz1(i, j, k) = qiHY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy1(i, j)
                        WKxyz2(i, j, k) = ccniHY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy2(i, j)
                        WKxyz3(i, j, k) = qsHY(i, j, k) &
                                + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                + max(zero, WKxyz7(i, j, k)) * WKxy3(i, j)
                    enddo
                enddo
                ! +
                ! +--Hydrom. Update
                ! +  ~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            qiHY(i, j, k) = WKxyz1(i, j, k)
                            ccniHY(i, j, k) = WKxyz2(i, j, k)
                            qsHY(i, j, k) = WKxyz3(i, j, k)
                        enddo
                    enddo
                enddo
                ! +
                ! +
                ! +--Ice  Water
                ! +--End of the                             Local Split Time Differencing
                ! +  --------------------------------------------------------------------
                ! +
            enddo
            ! +
        endif
        ! +
#ifdef WF
        ! +
        ! +--Ice  Water Conservation
        ! +  -----------------------
        ! +
        do j = 1, my
            do i = 1, mx
                ciVC = 0.0d+0
                qiVC = 0.0d+0
                qsVC = 0.0d+0
                do k = 1, mz
                    ! +
                    ! +--Flux Limitor
                    ! +  ~~~~~~~~~~~~
                    ccniHY(i, j, k) = max(zero, ccniHY(i, j, k))
                    ccniHY(i, j, k) = min(WKxy7(i, j), ccniHY(i, j, k))
                    qiHY(i, j, k) = max(zero, qiHY(i, j, k))
                    qiHY(i, j, k) = min(WKxy8(i, j), qiHY(i, j, k))
                    qsHY(i, j, k) = max(zero, qsHY(i, j, k))
                    qsHY(i, j, k) = min(WKxy9(i, j), qsHY(i, j, k))
                    ! +
                    ! +--Column Average
                    ! +  ~~~~~~~~~~~~~~
                    ciVC = ciVC + dsigm1(k) * ccniHY(i, j, k)
                    qiVC = qiVC + dsigm1(k) * qiHY(i, j, k)
                    qsVC = qsVC + dsigm1(k) * qsHY(i, j, k)
                enddo
                ! +
                ! +--Surface Boundary Flux
                ! +  ~~~~~~~~~~~~~~~~~~~~~
                ciVC = ciVC - psigDY(i, j, mz) &
                        * ccniHY(i, j, mz) * dt / pstDYn(i, j)
                qiVC = qiVC - psigDY(i, j, mz) &
                        * qiHY(i, j, mz) * dt / pstDYn(i, j)
                qsVC = qsVC - psigDY(i, j, mz) &
                        * (qsHY(i, j, mz) - qsrfHY(i, j)) * dt / pstDYn(i, j)
                ! +
                ! +--Mass Restore
                ! +  ~~~~~~~~~~~~
                do k = 1, mz
                    ccniHY(i, j, k) = ccniHY(i, j, k) * (WKxy4(i, j) / max(eps12, ciVC))
                    qiHY(i, j, k) = qiHY(i, j, k) * WKxy5(i, j) / max(eps12, qiVC)
                    qsHY(i, j, k) = qsHY(i, j, k) * WKxy6(i, j) / max(eps12, qsVC)
                enddo
            enddo
        enddo
#endif
        ! +
        ! +
        ! +--Work Arrays Reset
        ! +  -----------------
        ! +
        do j = 1, my
            do i = 1, mx
                WKxy1(i, j) = 0.0
                WKxy2(i, j) = 0.0
                WKxy3(i, j) = 0.0
                WKxy4(i, j) = 0.0
                WKxy5(i, j) = 0.0
                WKxy6(i, j) = 0.0
            enddo
        enddo
        ! +
        do k = 1, mz
            do i = 1, mx
                WKxza(i, k) = 0.0
                WKxzb(i, k) = 0.0
                WKxzc(i, k) = 0.0
                WKxzd(i, k) = 0.0
            enddo
        enddo
        ! +
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    WKxyz1(i, j, k) = 0.0
                    WKxyz2(i, j, k) = 0.0
                    WKxyz3(i, j, k) = 0.0
                    WKxyz4(i, j, k) = 0.0
                    WKxyz5(i, j, k) = 0.0
                    WKxyz6(i, j, k) = 0.0
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Courant Number
        ! +--Work Arrays Reset
        ! +  -----------------
        ! +
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    WKxyz7(i, j, k) = 0.0
                    WKxyz8(i, j, k) = 0.0
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Third Order Vertical Scheme
        ! +  ===========================
#ifdef ZU
    else
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = 0.
#endif
#ifdef ZO
                ga0(i, j) = qwHY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = qwHY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    qwHY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
#endif
#ifdef ZO
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = qrHY(i, j, mz)
            enddo
        enddo
#endif
#ifdef ZU
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = qrHY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    qrHY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = qvapSL(i, j)
#endif
#ifdef ZO
                ga0(i, j) = qvDY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = qvDY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    qvDY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = 0.0
#endif
#ifdef ZO
                ga0(i, j) = qiHY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = qiHY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    qiHY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
#endif
#ifdef ZO
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = ccniHY(i, j, mz)
            enddo
        enddo
#endif
#ifdef ZU
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = ccniHY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    ccniHY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = qsrfHY(i, j)
#endif
#ifdef ZO
                ga0(i, j) = qsHY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = qsHY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    qsHY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
    endif
#endif
    ! +
#ifdef BS
    ! +
    ! +--Impact on Snow Erosion/Deposition
    ! +  =================================
    ! +
    do j = jp11, my1
        do i = ip11, mx1
            ! dh_sno contains an implicit factor 1.d3[kPa-->Pa]/ro_Wat[kg/m2-->mWE]
            !        > 0 ==> Atmospheric Loss ==> Surface Gain
            dh_sno = psigDY(i, j, mz) * (qsHY(i, j, mz) - qsrfHY(i, j)) * dt * grvinv
            snohSL(i, j) = snohSL(i, j) + max(zero, dh_sno)
            snobSL(i, j) = snobSL(i, j) + min(zero, dh_sno)
            snowHY(i, j) = snowHY(i, j) + dh_sno
        enddo
    enddo
#endif
    return
endsubroutine HYDadv_ver
