#include "MAR_pp.def"
subroutine grdstr(xxmar, yymar, GE0lon, GE0lat, GElonM, GElatM, GEtruL)
    ! +------------------------------------------------------------------------+
    ! | MAR GRID                                               19-11-2004  MAR |
    ! |   subroutine grdstr computes the Latitudes, Longitudes                 |
    ! |                              of a MAR Domain Grid Point                |
    ! |                     assuming Inverse Stereographic Oblique Projection  |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT:  xxmar ,yymar : MAR        Coordinates                        |
    ! |   ^^^^^^  GE0lon,GE0lat: Geographic Coordinates of MAR Domain Center   |
    ! |                          (3-D): South-North Direction along            |
    ! |                                 90E, 180E, 270E or 360E Meridians      |
    ! |                                                                        |
    ! |   OUTPUT: GElatM       : Latitude  of the MAR grid point      (radian) |
    ! |   ^^^^^^^ GElonM       : Longitude of the MAR grid point        (hour) |
    ! |                                                                        |
    ! |   REFERENCE: F. Pearson, Map projection methods, CRC Press, 1990.      |
    ! |   ^^^^^^^^^^                                                           |
    ! +------------------------------------------------------------------------+
    use marphy
    use mardim
    use margrd

    implicit none

    ! +--local  Parameters
    ! +  =================
    integer i, j, k, m
    real pidemi, CphiP, SphiP, Sphi
    real dGElon, GElonM, GElatM
    real GEtruL, GE0lon, GE0lat
    real denomi, OBLlon, OBLlat
    real ddista, xxmar, yymar
    real costru

    pidemi = pi / 2.0

    CphiP = cos(degrad * GE0lat)
    SphiP = sin(degrad * GE0lat)

    costru = cos(degrad * GEtruL)

    ! +--Coordinates relative to a Pole set to the Domain Center
    ! +  =======================================================

    ! +--Relative Longitude -OBLlon (0 <= OBLlon < 2pi)
    ! +  ----------------------------------------------
    if(xxmar > 0.) then
        OBLlon = pidemi - atan(yymar / xxmar)
    else if(xxmar == 0. .and. yymar < 0.) then
        OBLlon = pi
    else if(xxmar < 0.) then
        OBLlon = 3.00 * pidemi - atan(yymar / xxmar)
    else if(xxmar == 0. .and. yymar >= 0.) then
        OBLlon = 0.0
    endif

    ! +--Relative  Latitude  OBLlat
    ! +  --------------------------
    ddista = sqrt(xxmar * xxmar + yymar * yymar)
    OBLlat = 0.50 * pi - 2.0 * atan(ddista / (earthr * (1.+costru)))

    ! +--Coordinates Change (OBLlon,OBLlat) -> (GElonM,GElatM)
    ! +                   / (rotation, Pearson p.57)
    ! +  =====================================================

    ! +--Latitude (radians)
    ! +  ------------------
    Sphi = SphiP * sin(OBLlat) + CphiP * cos(OBLlat) * cos(OBLlon)
    GElatM = asin(Sphi)

    ! +--Longitude  (hours)
    ! +  ------------------
    ! +--dGElon = GElonM - GE0lon  (-pi < dGElon <= pi)
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    denomi = CphiP * tan(OBLlat) - SphiP * cos(OBLlon)

    if(OBLlon > epsi .and. OBLlon < (pi - epsi)) then
        ! +--1) OBLlon in trigonometric quadrant 1 or 4 ("right"):
        ! +     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        dGElon = atan(sin(OBLlon) / denomi)
        if(dGElon < 0.0) then
            ! Go to Quadrant 1 by adding180 degrees
            dGElon = dGElon + pi
        endif
        ! +--2) OBLlon is in trigonometric quadrant 2or3 ("left "):
        ! +     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    else if(OBLlon > (pi + epsi) .and. OBLlon < (2.0 * pi - epsi)) then
        dGElon = atan(sin(OBLlon) / denomi)
        if(dGElon > 0.0) then
            ! Go to Quadrant 2 by substracting 180 degrees
            dGElon = dGElon - pi
        endif
    else if(OBLlon <= epsi .or. OBLlon >= (2.0 * pi - epsi)) then
        ! +--3) OBLlon = 0 -> dGElon = 0 or pi :
        ! +     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if((pidemi - OBLlat) > (pidemi - degrad * GE0lat)) then
            !  North pole crossed ==> add 180 degrees to Longitude
            dGElon = pi
        else
            dGElon = 0.0
        endif
    else if(OBLlon >= (pi - epsi) .and. OBLlon <= (pi + epsi)) then
        ! +--4) OBLlon = pi -> dGElon = 0 or pi :
        ! +     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if((pidemi - OBLlat) > (pidemi + degrad * GE0lat)) then
            !  South pole crossed ==> add 180 degrees to Longitude
            dGElon = pi
        else
            dGElon = 0.0
        endif
    endif
    ! +--Longitude (hours)
    ! +  ~~~~~~~~~
    GElonM = (dGElon + GE0lon * degrad) / hourad
    return
endsubroutine grdstr
