#include "MAR_pp.def"
subroutine grdsig(zmin, aavu, bbvu, ccvu, vertic)
    ! +------------------------------------------------------------------------+
    ! | MAR GRID                                               15-02-2008  MAR |
    ! |   subroutine grdsig is used to initialize the vertical grid            |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT:   zmin           : Height above Surface / 1st Sigma Level (m) |
    ! |   ^^^^^    aavu,bbvu,ccvu : Vertical Discretization Parameters         |
    ! |            vertic         : Logical Variable caracteris.vertic.discris.|
    ! |                                                                        |
    ! |   DATA:    sigpar(10)     : Parish Model Vertical Discretisation       |
    ! |   ^^^^^                                                                |
    ! |                                                                        |
    ! |   OUTPUT (via common block)                                            |
    ! |   ^^^^^^   Variable  which is  initialized is:                         |
    ! |               sigma(mz): Independant Variable (Normalized Pressure)    |
    ! |                                                                        |
    ! |   ASSUMPTION: sigma is calculated from initial level height amsl       |
    ! |   ^^^^^^^^^^^                     assumig that T(msl) = SST            |
    ! |                                                dT/dz  = -0.0065 K/m    |
    ! |                                                p_s    = 100     hPa    |
    ! |                                                                        |
    ! | # OPTIONS: #SA  Regular      Vertical Discretisation                   |
    ! | # ^^^^^^^^ #PA  Parish Model Vertical Discretisation                   |
    ! | #          #ll  LMDZ   Model Vertical Discretisation (L. Li)           |
    ! | #          #HE  NORLAM       Vertical Discretisation (G. Heineman)     |
    ! | #          #L1  Alternate    Vertical Discretisation (when very fine)  |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marphy
    use mardim
    use margrd
    use marsnd
    use mar_dy
    use mar_tu
    use mar_sl
    use mar_io
    use mar_wk

    implicit none

    logical, intent(in) :: vertic

    ! +--Local  Variables
    ! +  ================
    integer i, j, k, m
    integer lsf, lvg, kk, km, kn, k1
    real ga0, gaz, zmin, dzz, rz, rzb, zzo, sh_min
    real ps_sig, vu, aavu, bbvu, ccvu, ps_max, pstar0
#ifdef HE
    real sighei(29)
#endif
#ifdef lm
    real siglmd(11)
#endif
#ifdef PA
    real sigpar(10), sigp11, sigp12(11:12), sigp13(10:13)
#endif

    ! +--DATA
    ! +  ====
    ! data ps_sig/101.3e0/
    data ps_sig/100.0e0/
#ifdef HE
    ! sighei: DNMI model Vertical Discretisat.  (Heinemann 1996)
    data sighei/0.10015, 0.19077, 0.27276, 0.34695, 0.41409, &
        0.47483, 0.52979, 0.57952, 0.62452, 0.66524, 0.70208, &
        0.73542, 0.76558, 0.79288, 0.81757, 0.83992, 0.86014, &
        0.87844, 0.89499, 0.90997, 0.92352, 0.93579, 0.94688, &
        0.95692, 0.96601, 0.97423, 0.98167, 0.98840, 0.99111/
#endif
#ifdef lm
    ! siglmd: Vertical Discretisation of LMDZ   Model
    !         (Laurent LI, personal communication,  5 dec. 2000)
    data siglmd/0.014767, 0.071835, 0.150135, 0.270661, 0.410669, &
        0.565832, 0.708390, 0.829996, 0.913837, 0.966484, &
        0.990723/
#endif
#ifdef PA
    ! sigpar: Vertical Discretisation of Parish Model
    !         (Bromwich, Du and Parish 1994 MWR 122 No 7 p.1418)
    data sigpar/0.100, 0.350, 0.600, 0.800, 0.900, &
        0.930, 0.950, 0.970, 0.985, 0.996/
    ! sigp1x: Vertical Discretisation of Parish Model (modified)
    data sigp11/0.998/
    data(sigp12(k), k=11, 12) / 0.998, 0.999 /
    data(sigp13(k), k=10, 13) / 0.990, 0.996, 0.998, 0.999 /
#endif
    data lsf/1/
    ! ga0 : Standard Atmospheric Lapse Rate
    data ga0/0.0065e0/
    ! lvg : set to 1 if |Vg(sounding)| .ne. 0 anywhere
    lvg = 0

    ! +--Entry Checking Point
    ! +  ====================
    if(IO_loc >= 2) write(21, 999)
999 format(//, '   --- Initialisation / grdsig ---')

    ! +--Temperature Vertical Profile
    ! +  ============================
    gaz = ga0

    if(IO_loc >= 2) write(21, 1) gaz, sst_SL, ps_sig, gravit, RDryAi
1   format(/, '  dT/dz  =', f8.5, ' K/m', &
            /, '  SST    =', f8.2, ' K', &
            /, '  ps_sig =', f8.2, ' kPa', &
            /, '  gravit =', f8.2, ' m/s2', &
            /, '  RDryAi =', f8.2, ' J/kg/K')

    ! +--Sigma Levels
    ! +  ============

    ! +- 1) Coarse Resolution of the Surface Layer
    ! +  -----------------------------------------
    if(.not. vertic) then
        ! Reference : E. Richard, these, 1991, p.29
        ! aa = 0.5
        ! bb = 1.5
        ! cc =-1.0
        vu = 0.0
        do k = 1, mz
            vu = vu + 1.0 / dble(mzz)
            sigma(k) = aavu * vu + bbvu * vu * vu + ccvu * vu * vu * vu
#ifdef HE
            ! +- Vertical Discretisation of NORLAM
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            sigma(k) = sighei(k)
#endif
        enddo
#ifdef lm
        do k = 1, 11
            sigma(k) = siglmd(k)
        enddo
#endif
#ifdef PA
        ! +- Vertical Discretisation of Parish Model
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        do k = 1, 10
            sigma(k) = sigpar(k)
        enddo
        mmz = mz
        if(mmz > 10) then
            if(mmz == 11) sigma(11) = sigp11
            if(mmz == 12) then
                do k = 11, 12
                    sigma(k) = sigp12(k)
                enddo
            endif
            if(mmz == 13) then
                do k = 10, 13
                    sigma(k) = sigp13(k)
                enddo
            endif
        endif
#endif
        do k = 1, mz
            if(abs(gaz) > 1.d-5) then
                zsigma(k) = -(sst_SL / gaz) * ((1.e0 + (sigma(k) - 1.e0) &
                                                * (1.e2 / ps_sig))**(RDryAi * gaz / gravit) - 1.e0)
            else
                if(IO_loc >= 2 .and. k == 1) write(21, 116)
116             format(/, '  t(z)   = CONSTANT')
                zsigma(k) = -(RDryAi * sst_SL / gravit) * log((unun + (sigma(k) - unun) &
                                                               * (1.d2 / ps_sig)))
            endif
        enddo
    else
        ! +- 2) Fine   Resolution of the Surface Layer
        ! +  -----------------------------------------
        gaz = max(gaz, epsi)
        km = 2
        km = min(km, mz)
        kn = 1
#ifdef L1
        kn = 2
#endif
        zsigma(1) = zmin
        zsigma(km) = 2.0 * zmin
        do k = min(3, mz), mz
            rz = zmin * aavu**(k - 1)
            rzb = ccvu * bbvu**(k - 1)
            if(TUkhmx > 0.0) then
                zsigma(k) = rzb * rz / (rz + rzb)
            else
                zsigma(k) = rz
            endif

            zsigma(k) = max(zsigma(k), zsigma(k - 1) + zsigma(kn))
        enddo
        ! sh_min_0 : Everest
        sh_min = 8807.0
        do j = 1, my
            do i = 1, mx
                sh_min = min(sh_min, sh(i, j))
            enddo
        enddo
        ps_max = ps_sig * (1.0 - gaz * sh_min / sst_SL) &
                 **(gravit / (gaz * RDryAi))
        pstar0 = ps_max - ptopDY
        do k = 1, mz
            kk = mz + 1 - k
            ! sigma(kk): the fine resolution  of the surface layer
            !            is computed using a geometric progression
            sigma(kk) = (ps_sig / pstar0) &
                        * ((1.0 - gaz * (sh_min + zsigma(k)) / sst_SL) &
                           **(gravit / (gaz * RDryAi)) &
                           - 1.0) &
                        + 1.0 + (ps_sig - ps_max) / pstar0
        enddo
    endif

    ! +--Output
    ! +  ======
    do k = 1, mz
        kk = mzz - k
        WKxza(1, k) = zsigma(kk)
    enddo

    do k = 1, mz
        zsigma(k) = WKxza(1, k)
        WKxza(1, k) = 0.0
    enddo

    if(IO_loc >= 2) then
        write(21, 130)(sigma(k), k=1, mz)
130     format(/, '  Sigma    Levels :', /,(1x, 15f8.4))
        write(21, 131)(zsigma(k), k=1, mz)
131     format(/, '  Altitude Levels :', /,(1x, 15f8.1))
    endif
    return
endsubroutine grdsig
