#include "MAR_pp.def"
subroutine grdmar
    ! +------------------------------------------------------------------------+
    ! | MAR GRID                                               20-02-2021  MAR |
    ! |   subroutine grdmar is used to initialize the grid parameters          |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |    INPUT (via common block)                                            |
    ! |    ^^^^^    sigma(mz) : Independant Variable on      Levels            |
    ! |             FIslot    : Implicit Filter Parameter                      |
    ! |                                 (Slow Dynamics / Temperature)          |
    ! |             FIslou    : ...     (Slow Dynamics / Wind Speed)           |
    ! |             FIslop    : ...     (Slow Dynamics / Pressure)             |
    ! |                                                                        |
    ! |   OUTPUT (via common block)                                            |
    ! |   ^^^^^^     xxkm(mx) : Distance along the x-axis                 (km) |
    ! |              yykm(my) : Distance along the y-axis                 (km) |
    ! |                                                                        |
    ! |            sigmid(mzz): Independant Variable between Levels (i.e.k-1/2)|
    ! |            dsigm1(mz ):      Difference  d(sigma)|k                    |
    ! |            qsigm1(mz ): 1 / [Difference  d(sigma)|k    ]               |
    ! |            dsigm2(mz ):      Difference 2d(sigma)|k                    |
    ! |            qsigm2(mz ): 1 / [Difference 2d(sigma)|k    ]               |
    ! |            dsig_1(mzz):      Difference  d(sigma)|k+1/2                |
    ! |            qsig_1(mzz): 1 / [Difference  d(sigma)|k+1/2]               |
    ! |            dsig_2(mzz):      Difference 2d(sigma)|k+1/2                |
    ! |                                                                        |
    ! |            Ray_UB(mzabso) : Top Absorbing Layer  Contribution to       |
    ! |                             Rayleigh             Friction        (-/s) |
    ! |                                                                        |
    ! |            TUspon(mzabso) : Top Absorbing Layer  Contribution to       |
    ! |                             Horizontal Diffusion Coefficient    (m2/s) |
    ! |                                                                        |
    ! |            FIspon(mzabso) : Top Absorbing Layer  Contribution          |
    ! |                             to Implicit Filter Parameter               |
    ! |                    FIk_st(mz): Implicit Filter Parameter               |
    ! |                                   (Slow Dynamics / Temperature)        |
    ! |                    FIk_su(mz): ...(Slow Dynamics / Wind Speed)         |
    ! |            FIfstu, FIk_fu(mz): ...(Fast Dynamics / Wind Speed)         |
    ! |            FIfstu, FIk_fp(mz): ...(Fast Dynamics / Pressure,Velocity)  |
    ! |                                                                        |
    ! |            n6mxLB, n7mxLB : Effective Length of Lateral Sponge (x-Axe) |
    ! |            n6myLB, n7myLB : Effective Length of Lateral Sponge (y-Axe) |
    ! |                                                                        |
    ! |            im1(mx),2,..: max(i-1, 1), max(i-2, 1), etc...              |
    ! |            ip1(mx),2,..: min(i+1,mx), min(i+2,mx), etc...              |
    ! |                                                                        |
    ! |            jm1(my),2,..: max(j-1, 1), max(j-2, 1), etc...              |
    ! |            jp1(my),2,..: min(j+1,my), min(j+2,my), etc...              |
    ! |                                                                        |
    ! |            km1(mz),2,..: max(k-1, 1), max(k-2, 1), etc...              |
    ! |            kp1(mz),2,..: min(k+1,mz), min(k+2,mz), etc...              |
    ! |                                                                        |
    ! |            CUspxh(mx)  : Cubic Spline Auxiliary Variable (x Direction) |
    ! |            CUspxb(mx)  : idem                                          |
    ! |            CUspyh(mx)  : Cubic Spline Auxiliary Variable (y Direction) |
    ! |            CUspyb(mx)  : idem                                          |
    ! |            CUspzh(mx)  : Cubic Spline Auxiliary Variable (z Direction) |
    ! |            CUspzb(mx)  : idem                                          |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marphy
    use mardim
    use margrd
    use mar_cu
    use mar_lb
    use mar_ub
    use mar_tu
    use mar_fi
    use mar_io
#ifdef NH
    use mar_nh
#endif

    implicit none

    ! +--Local  Variables
    ! +  ================
    ! +
    integer i, j, k, m
    integer im10, ip10, im20, ip20
    integer jm10, jp10, jm20, jp20
    integer km10, kp10, km20, mzabs
    real FIabs
    ! +
    ! +
    ! +--DATA
    ! +  ====
    ! +
    logical DFspon
    data DFspon/.true./
#ifdef KS
    DFspon = .false.
#endif
    ! +
    ! +--Entry Checking Point
    ! +  ====================
    ! +
    if(IO_loc >= 2) write(21, 999)
999 format(//, '   --- Initialisation / grdmar ---')
    ! +
    ! +
    ! +--Auxiliary Horizontal Independant Variables
    ! +  ==========================================
    ! +
    mmx = mx
    mmx1 = mx1
    mmx1 = max(1, mmx1)
    mmx2 = mx - 2
    mmx2 = max(1, mmx2)
    mmx3 = mx - 3
    mmx3 = max(1, mmx3)
    mmx4 = mx - 4
    mmx4 = max(1, mmx4)
    mmx5 = mx - 5
    mmx5 = max(1, mmx5)
    mmx6 = mx - 6
    mmx6 = max(1, mmx6)
    m0x2 = 2
    m0x2 = min(mx, m0x2)
    m0x3 = 3
    m0x3 = min(mx, m0x3)
    m0x4 = 4
    m0x4 = min(mx, m0x4)
    m0x5 = 5
    m0x5 = min(mx, m0x5)
    m0x6 = 6
    m0x6 = min(mx, m0x6)
    ! +
    mmy = my
    mmy1 = my1
    mmy1 = max(1, mmy1)
    mmy2 = my - 2
    mmy2 = max(1, mmy2)
    mmy3 = my - 3
    mmy3 = max(1, mmy3)
    mmy4 = my - 4
    mmy4 = max(1, mmy4)
    mmy5 = my - 5
    mmy5 = max(1, mmy5)
    mmy6 = my - 6
    mmy6 = max(1, mmy6)
    m0y2 = 2
    m0y2 = min(my, m0y2)
    m0y3 = 3
    m0y3 = min(my, m0y3)
    m0y4 = 4
    m0y4 = min(my, m0y4)
    m0y5 = 5
    m0y5 = min(my, m0y5)
    m0y6 = 6
    m0y6 = min(my, m0y6)
    ! +
    mmz = mz
    mmz1 = mz1
    mmz1 = max(1, mmz1)
    mmz2 = mz - 2
    mmz2 = max(1, mmz2)
    ! +
    dx2 = dx * 2.0
    dy2 = dy * 2.0
    ! +
    if(mmx > 1) then
        ! +
        dtx = dt / dx
        dty = dt / dy
        ! +
        dxinv = 1.0 / dx
        dyinv = 1.0 / dy
        dxinv2 = 1.0 / dx2
        dyinv2 = 1.0 / dy2
        ! +
        do i = 1, mx
            ! xxkm : in km
            xxkm(i) = (i - imez) * dx / 1000.
        enddo
        ! +
        do j = 1, my
            ! yykm : in kms
            yykm(j) = (j - jmez) * dy / 1000.
        enddo
        ! +
    endif
    ! +
    ! +
    ! +--4th Order Centered Difference Parameter
    ! +  ---------------------------------------
    ! +
    fac43 = 4.0 / 3.0
    ! +
    ! +
    ! +--Effective Length of the Lateral Sponge
    ! +  --------------------------------------
    ! +
    if(mmx == 1) then
        n40xLB = 1
        n50xLB = 1
        n5mxLB = 1
        n6mxLB = 0
        n7mxLB = 1
        n40yLB = 1
        n50yLB = 1
        n5myLB = 1
        n6myLB = 0
        n7myLB = 1
        ! +
    else
        n40xLB = mx - n6 + 2
        n50xLB = mx - n6 + 1
        n5mxLB = n6 - 1
        n6mxLB = n6
        n7mxLB = n7
        ! +
        if(mmy == 1) then
            n40yLB = 1
            n50yLB = 1
            n5myLB = 1
            n6myLB = 0
            n7myLB = 1
        else
            n40yLB = my - n6 + 2
            n50yLB = my - n6 + 1
            n5myLB = n6 - 1
            n6myLB = n6
            n7myLB = n7
        endif
        ! +
    endif
    ! +
    ! +
    ! +--Boundaries Masks
    ! +  ----------------
    ! +
    do i = 1, mx
        im10 = i - 1
        ip10 = i + 1
        im1(i) = max(im10, 1)
        ip1(i) = min(ip10, mx)
        im20 = i - 2
        ip20 = i + 2
        im2(i) = max(im20, 1)
        ip2(i) = min(ip20, mx)
    enddo
    ! +
    do j = 1, my
        jm10 = j - 1
        jp10 = j + 1
        jm1(j) = max(jm10, 1)
        jp1(j) = min(jp10, my)
        jm20 = j - 2
        jp20 = j + 2
        jm2(j) = max(jm20, 1)
        jp2(j) = min(jp20, my)
    enddo
    ! +
    ! +
    ! +--Auxiliary Vertical   Independant Variables
    ! +  ==========================================
    ! +
    ! +
    ! +--Boundaries Masks
    ! +  ----------------
    ! +
    do k = 1, mz
        km10 = k - 1
        kp10 = k + 1
        km1(k) = max(km10, 1)
        kp1(k) = min(kp10, mz)
        km20 = k - 2
        km2(k) = max(km20, 1)
    enddo
    ! +
    ! +
    ! +--Discretisation
    ! +  --------------
    ! +
    dsig_1(0) = sigma(1)
    dsig_1(1) = sigma(kp1(1)) - sigma(1)
    dsig_2(1) = sigma(kp1(1))
    sigmid(1) = 0.0
    sigmid(mzz) = 1.0
    ! +
    do k = kp1(1), mmz1
        dsig_1(k) = sigma(kp1(k)) - sigma(k)
        dsig_2(k) = sigma(kp1(k)) - sigma(km1(k))
        sigmid(k) = (sigma(k) + sigma(km1(k))) / 2.0
        dsigm1(km1(k)) = sigmid(k) - sigmid(km1(k))
        dsigm2(km1(k)) = sigmid(k) - sigmid(km2(k))
    enddo
    ! +
    ! +--The lowest layer of the model is assumed to be a constant flux layer
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    dsig_1(mz) = 1.0 - sigma(mz)
    dsig_2(mz) = sigma(mz) - sigma(km1(mz))
    sigmid(mz) = 0.50 * (sigma(mz) + sigma(km1(mz)))
    dsigm1(km1(mz)) = sigmid(mz) - sigmid(km1(mz))
    dsigm2(km1(mz)) = sigmid(mz) - sigmid(km2(mz))
    ! +
    dsig_1(mzz) = dsig_1(mz)
    dsig_2(mzz) = 2.00 * dsig_1(mzz)
    dsigm1(mz) = 1.00 - sigmid(mz)
    dsigm2(mz) = 1.00 - sigmid(km1(mz))
    ! +
    do k = 0, mzz
        qsig_1(k) = 1.00 / dsig_1(k)
    enddo
    ! +
    do k = 1, mz
        qsigm1(k) = 1.00 / dsigm1(k)
        qsigm2(k) = 1.00 / dsigm2(k)
    enddo
    ! +
    ! +
    ! +--Filter Parameter    Initialisation (rapidly propagating Waves Dynamics)
    ! +  =======================================================================
    ! +
    !XF
    !           FIslou=max(FIslou,0.008) ! higher is, smoother the wind is
    !           FIslop=max(FIslop,0.008)
    !           FIslot=max(FIslot,0.008) ! higher is, colder MAR is
    FIslot = 0.007
    FIfstu = FIslou / (ntFast + 1)
    FIfstp = FIslop / (ntFast + 1)
    ! +
    do k = 1, mz
        FIk_st(k) = FIslot / max(0.1, sigma(k))
        FIk_su(k) = FIslou / max(0.1, sigma(k))
        FIk_fu(k) = FIfstu / max(0.1, sigma(k))
        FIk_fp(k) = FIfstp / max(0.1, sigma(k))
    enddo
    ! +
    ! +
    ! +--Top Absorbing Layer Initialisation
    ! +  ==================================
    ! +
    FIabs = TUkhmx * 4.0 * dtfast / (dx * dx)
    ! +
    if(mz > 1) then
        mzabs = mzabso + 1
        mzabs = min(mz, mzabs)
        if(DFspon) then
            do k = 1, mzabso
                FIspon(k) = FIabs * (sigma(mzabs) - sigma(k)) &
                            / (sigma(mzabs) - sigma(1))
                FIk_st(k) = FIk_st(k) + FIspon(k) * dt / dtfast
                FIk_su(k) = FIk_su(k) + FIspon(k) * dt / dtfast
                FIk_fu(k) = FIk_fu(k) + FIspon(k)
                FIk_fp(k) = FIk_fp(k) + FIspon(k)
                TUspon(k) = zero
            enddo
        else
            do k = 1, mzabso
                FIspon(k) = zero
                TUspon(k) = TUkhmx * (sigma(mzabs) - sigma(k)) &
                            / (sigma(mzabs) - sigma(1))
            enddo
        endif
    endif

    ! +--Rayleigh Friction (Ref. ARPS 4.0 User's Guide, para 6.4.3 p.152)
    ! +  =================
    do k = 1, mzabso
        Ray_UB(k) = 0.5 * (1.-cos(pi * (sigma(mzabso) - sigma(k)) &
                                  / (sigma(mzabso) - sigma(1)))) / (1.5 * dt)
#ifdef rf
        Ray_UB(k) = (sigma(mzabso) - sigma(k)) / &
                    (sigma(mzabso) - sigma(1)) / (10.0 * dt)
#endif
    enddo

    ! +--Cubic Spline Initialisation
    ! +  ===========================
    ! +
    ! +  1) x - Direction
    ! +  ----------------
    CUspxh(1) = 0.0
    CUspxh(mx) = 0.0
    CUspxb(1) = 0.0
    CUspxb(mx) = 0.0
    do i = ip11, mx1
        CUspxh(i) = CUspxb(im1(i)) + 4.0
        CUspxb(i) = -1.0 / CUspxh(i)
    enddo
    ! +
    ! +  2) y - Direction
    ! +  ----------------
    CUspyh(1) = 0.0
    CUspyh(my) = 0.0
    CUspyb(1) = 0.0
    CUspyb(my) = 0.0
    if(mmy > 1) then
        do j = jp11, my1
            CUspyh(j) = CUspyb(jm1(j)) + 4.0
            CUspyb(j) = -1.0 / CUspyh(j)
        enddo
    endif
    ! +
    ! +  3) Sigma - Direction (to be used in routine DYNadv_cubv)
    ! +  --------------------------------------------------------
#ifdef ZU
    CUspzh(1) = dsig_1(1) / (dsig_1(1) + sigma(1))
    CUspzh(mz) = dsig_1(mz) / (dsig_1(mz) + dsig_1(mmz1))
    CUspzb(1) = sigma(1) / (dsig_1(1) + sigma(1))
    CUspzb(mz) = dsig_1(mmz1) / (dsig_1(mz) + dsig_1(mmz1))
    do k = kp1(1), mmz1
        CUspzh(k) = dsig_1(k) / (dsig_1(k) + dsig_1(k - 1))
        CUspzb(k) = dsig_1(k - 1) / (dsig_1(k) + dsig_1(k - 1))
    enddo
#endif
    ! +
    return
endsubroutine grdmar
