#include "MAR_pp.def"
subroutine grdgeo
    ! +------------------------------------------------------------------------+
    ! | MAR GRID                                           Tue 14-04-2021  MAR |
    ! |   subroutine grdgeo computes the Latitudes, Longitudes and             |
    ! |                              the Time Zone of each Grid Point          |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |    INPUT: abs(maptyp)=0: Polar   Stereogr. Project. (ALL        LATITUDES) |
    ! |    ^^^^^              1: Oblique Stereogr. Project. (ALL    LATITUDES  |
    ! |                                                                        |
    ! |    INPUT (via common block)                                            |
    ! |    ^^^^^  imez,jmez    : Indices of the MAR Domain Center              |
    ! |           GEddxx       : (2-D): x-Axis      Direction                  |
    ! |                          (3-D): South-North Direction along            |
    ! |                                 90E, 180E, 270E or 360E Meridians      |
    ! |           GElat0       : Latitude  of (0,0) in  MAR              (deg) |
    ! |           GElon0       : Longitude of (0,0) in  MAR              (deg) |
    ! |                                                                        |
    ! |   OUTPUT (via common block)                                            |
    ! |   ^^^^^^  GElatr(mx,my): Latitude  of the (x,y) MAR coordinate   (rad) |
    ! |           GElonh(mx,my): Longitude of the (x,y) MAR coordinate     (h) |
    ! |           itizGE(mx,my): Time Zone                                     |
    ! |           fcorDY(mx,my): Coriolis Parameter (may be variable)          |
    ! |                                                                        |
    ! |   MODIF.  3 Nov 2009   : Map Scaling Factor SFm_DY computed only       |
    ! |   ^^^^^                  for a domain which is North/South Pole        |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_ge
    use mar_dy

    implicit none

    ! +--Local  Variables
    ! +  ================
    integer i, j, k, m
    integer i1_gg, i2_gg, id10, jd10
    real GElon, GElat, RadLat, clat_s
    real argrot, cosrot, sinrot
    real xxmar0, yymar0, x0, y0
    real ddista, xdista, ydista
    real lon2, lat2, lon1, lat1, distance, d1, d2

    ! +--GEOGRAPHIC Coordinates
    ! +  ======================
    if(.not. geoNST) then
        ! +--1-D and 2-D Cases
        ! +  -----------------
        if(mmy == 1) then
            argrot = (GEddxx - 90.0) * degrad
            cosrot = cos(argrot)
            sinrot = sin(argrot)
            do j = 1, my
                do i = 1, mx
                    xxmar0 = cosrot * (i - imez) * dx + sinrot * (j - jmez) * dx
                    yymar0 = cosrot * (j - jmez) * dx - sinrot * (i - imez) * dx
                    ! +  ******
                    call grdstr(xxmar0, yymar0, GElon0, GElat0, GElon, GElat, GEtrue)
                    ! +  ******
                    GElatr(i, j) = GElat
                    GElonh(i, j) = GElon
                enddo
            enddo
        else
            ! +--3-D Cases
            ! +  ---------

            ! +- ANTARCTICA (Polar   Stereographic Projection is assumed)
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(maptyp == 0) then
                call stereosouth_inverse(GElon0, GElat0, GEddxx, x0, y0)
                xxkm = xxkm + x0
                yykm = yykm + y0
                do j = 1, my
                    do i = 1, mx
                        xxmar0 = (i - imez) * dx / 1000.+x0
                        yymar0 = (j - jmez) * dy / 1000.+y0
                        ! +  ***********
                        call StereoSouth(xxmar0, yymar0, GEddxx, GElon, GElat, GElat0)
                        ! +  ***********
                        ! Conversion: degrees->hour
                        GElonh(i, j) = GElon / 15.
                        ! Conversion: rad ->degrees
                        GElatr(i, j) = GElat * degrad
                    enddo
                enddo
            endif

#ifdef PP
            if(maptyp == 0) then
                ! transformation stereographic coordinates (center = South Pole)
                !             -> spherical     coordinates
                ddista = earthr * 2.0 * tan((45.0 + GElat0 * 0.50) * degrad)
                xdista = ddista * cos((90.0 - GElon0) * degrad)
                ydista = ddista * sin((90.0 - GElon0) * degrad)
                do j = 1, my
                    do i = 1, mx
                        if(abs(GEddxx - 90.0) < epsi) then
                            xxmar0 = (i - imez) * dx
                            yymar0 = (j - jmez) * dy
                        endif
                        if(abs(GEddxx) < epsi) then
                            xxmar0 = (j - jmez) * dy
                            yymar0 = -(i - imez) * dx
                        endif
                        if(abs(GEddxx - 270.0) < epsi) then
                            xxmar0 = -(i - imez) * dx
                            yymar0 = -(j - jmez) * dy
                        endif
                        if(abs(GEddxx - 180.0) < epsi) then
                            xxmar0 = -(j - jmez) * dy
                            yymar0 = (i - imez) * dx
                        endif

                        xxmar0 = xxmar0 + xdista
                        yymar0 = yymar0 + ydista

                        ddista = sqrt(xxmar0 * xxmar0 + yymar0 * yymar0)
                        GElatr(i, j) = -0.5 * pi + 2.*atan(ddista * 0.5 / earthr)
                        if(abs(xxmar0) > zero) then
                            GElonh(i, j) = atan(yymar0 / xxmar0)
                            if(xxmar0 < zero) then
                                GElonh(i, j) = GElonh(i, j) + pi
                            endif
                            GElonh(i, j) = 0.50 * pi - GElonh(i, j)
                            if(GElonh(i, j) > pi) then
                                GElonh(i, j) = -2.00 * pi + GElonh(i, j)
                            endif
                            if(GElonh(i, j) < -pi) then
                                GElonh(i, j) = 2.00 * pi + GElonh(i, j)
                            endif
                        else
                            if(yymar0 > zero) then
                                GElonh(i, j) = 0.00
                            else
                                GElonh(i, j) = pi
                            endif
                        endif
                        ! Conversion : radian -> Hour
                        GElonh(i, j) = GElonh(i, j) / hourad
                    enddo
                enddo
            endif
#endif
            ! +- OTHERS (Oblique Stereographic Projection is assumed)
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(abs(maptyp) == 1) then
                do j = 1, my
                    do i = 1, mx
                        argrot = (GEddxx - 90.0) * degrad
                        cosrot = cos(argrot)
                        sinrot = sin(argrot)
                        xxmar0 = cosrot * (i - imez) * dx + sinrot * (j - jmez) * dy
                        yymar0 = cosrot * (j - jmez) * dy - sinrot * (i - imez) * dx
                        ! +  ***********
                        call grdstr(xxmar0, yymar0, GElon0, GElat0, GElon, GElat, GEtrue)
                        ! +  ***********
                        GElatr(i, j) = GElat
                        GElonh(i, j) = GElon
                    enddo
                enddo
            endif
        endif
    endif

    do i = 1, mx
        lon2 = GElonh(i, jmez) * 15.
        lat2 = GElatr(i, jmez) / degrad
        lon1 = GElonh(imez, jmez) * 15.
        lat1 = GElatr(imez, jmez) / degrad
        xxkm2(i) = distance(lon2, lat2, lon1, lat1)
        if(i <= imez) xxkm2(i) = -1.*xxkm2(i)
    enddo

    do j = 1, my
        lon2 = GElonh(imez, j) * 15.
        lat2 = GElatr(imez, j) / degrad
        lon1 = GElonh(imez, jmez) * 15.
        lat1 = GElatr(imez, jmez) / degrad
        yykm2(j) = distance(lon2, lat2, lon1, lat1)
        if(j <= jmez) yykm2(j) = -1.*yykm2(j)
    enddo

    do i = 2, mx - 1
        do j = 2, my - 1
            d1 = 0; d2 = 0
            do k = -1, 1, 2
                lon1 = GElonh(i, j) * 15.
                lon2 = GElonh(i + k, j) * 15.
                lat1 = GElatr(i, j) / degrad
                lat2 = GElatr(i + k, j) / degrad
                d1 = d1 + distance(lon2, lat2, lon1, lat1) / 2.
            enddo
            do k = -1, 1, 2
                lon1 = GElonh(i, j) * 15.
                lon2 = GElonh(i, j + k) * 15.
                lat1 = GElatr(i, j) / degrad
                lat2 = GElatr(i, j + k) / degrad
                d2 = d2 + distance(lon2, lat2, lon1, lat1) / 2.
            enddo
            area(i, j) = d1 * d2
            d1 = 0; d2 = 0
            do k = -1, 1, 2
                lon1 = GElonh(i, j) * 15.
                lon2 = GElonh(i + k, j) * 15.
                lat1 = GElatr(i, j) / degrad
                lat2 = GElatr(i + k, j) / degrad
                d1 = d1 + distance(lon2, lat2, lon1, lat1) / 2.
            enddo
            do k = -1, 1, 2
                lon1 = GElonh(i, j) * 15.
                lon2 = GElonh(i, j + k) * 15.
                lat1 = GElatr(i, j) / degrad
                lat2 = GElatr(i, j + k) / degrad
                d2 = d2 + distance(lon2, lat2, lon1, lat1) / 2.
            enddo
            dx3(i, j) = d1 * 1000.
            dy3(i, j) = d2 * 1000.
            dxy3(i, j) = (dx3(i, j) + dy3(i, j)) / 2.
            dxinv3(i, j) = 1./(dx3(i, j) * 2.)
            dyinv3(i, j) = 1./(dy3(i, j) * 2.)
        enddo
    enddo

    do i = 1, mx
        area(i, 1) = area(i, 2)
        area(i, my) = area(i, my - 1)
        dx3(i, 1) = dx3(i, 2)
        dx3(i, my) = dx3(i, my - 1)
        dy3(i, 1) = dy3(i, 2)
        dy3(i, my) = dy3(i, my - 1)
    enddo

    do j = 1, my
        area(1, j) = area(2, j)
        area(mx, j) = area(mx - 1, j)
        dx3(1, j) = dx3(2, j)
        dx3(mx, j) = dx3(mx - 1, j)
        dy3(1, j) = dy3(2, j)
        dy3(mx, j) = dy3(mx - 1, j)
    enddo

    ! +--Sine, Cosine of Latitude
    ! +  ========================
    do j = 1, my
        do i = 1, mx
            clatGE(i, j) = cos(GElatr(i, j))
            slatGE(i, j) = sin(GElatr(i, j))
        enddo
    enddo

    ! +--Scaling Map Factor
    ! +  ==================
    if(abs(GElat0) >= 90.-epsi) then
        clat_s = 1.+sin((90.-GEtrue) * degrad)
        do j = 1, my
            do i = 1, mx
                SFm_DY(i, j) = clat_s / (1.+abs(slatGE(i, j)))
            enddo
        enddo
    else
        do j = 1, my
            do i = 1, mx
                SFm_DY(i, j) = 1.0
            enddo
        enddo
    endif

    ! +--Numerical Equator
    ! +  ~~~~~~~~~~~~~~~~~
    do j = 1, my
        do i = 1, mx
            if(abs(GElatr(i, j)) < epsi) then
                RadLat = epsi
                slatGE(i, j) = sin(RadLat)
                clatGE(i, j) = cos(RadLat)
            endif

            ! +--Numerical North Pole
            ! +  ~~~~~~~~~~~~~~~~~~~~
            if(GElatr(i, j) > demi * pi - epsi) then
                RadLat = demi * pi - epsi
                !XF slatGE(i,j) = sin(RadLat )
                !XF clatGE(i,j) = cos(RadLat )
                slatGE(i, j) = 1.
                clatGE(i, j) = 0.
            endif

            ! +--Numerical South Pole
            ! +  ~~~~~~~~~~~~~~~~~~~~
            if(GElatr(i, j) < epsi - demi * pi) then
                RadLat = epsi - demi * pi
                !XF slatGE(i,j) = sin(RadLat )
                !XF clatGE(i,j) = cos(RadLat )
                slatGE(i, j) = -1.
                clatGE(i, j) = 0.
            endif
        enddo
    enddo

    ! +--Coriolis Parameter
    ! +  ==================
    do j = 1, my
        do i = 1, mx
            fcorDY(i, j) = 2.0 * earthv * sin(GElatr(i, j))
#ifdef CC
            ! fcorDY: Coriolis Parameter
            fcorDY(i, j) = 2.0 * earthv * sin(GElatr(imez, jmez))
#endif
        enddo
    enddo

    ! +--Time Zone
    ! +  =========
    do j = 1, my
        do i = 1, mx
            itizGE(i, j) = GElonh(i, j)
            if(itizGE(i, j) > 12) itizGE(i, j) = itizGE(i, j) - 24
            if(itizGE(i, j) < -12) itizGE(i, j) = itizGE(i, j) + 24
        enddo
    enddo

    ! +--OUTPUT
    ! +  ======
    i1_gg = imez - 50
    i2_gg = imez + 50
    i1_gg = max(i1_gg, 1)
    i2_gg = min(i2_gg, mx)
    id10 = 1 + min(mx - 1, 10)
    jd10 = 1 + min(my - 1, 10)

    write(4, 990)(i, i=i1_gg, i2_gg, id10)
990 format(/, ' LATITUDES / LONGITUDES / TOPOGRAPHY:  x ->  y ^ ', &
            /, ' ===================================', /, 9x, 13i9)
    do j = my, 1, -jd10
        do i = i1_gg, i2_gg, id10
            write(4, 991) j,(GElatr(i, j) / degrad)
991         format(i9, 11f9.3)
            write(4, 992)(GElonh(i, j) * 1.5d+1)
992         format(9x, 11f9.3)
            write(4, 993)(sh(i, j) * 1.0d-3)
993         format(9x, 11f9.3)
            write(4, 994)(itizGE(i, j))
994         format(9x, 11i9)
            write(4, 995)(fcorDY(i, j))
995         format(9x, 11f9.6)
        enddo
    enddo

    return
endsubroutine grdgeo

function distance(lon2, lat2, lon1, lat1)
    use marphy
    implicit none

    real, parameter :: R = 6378.1370
    real :: lon1, lat1
    real :: lon2, lat2, distance
    real :: dlat, dlon, a, c

    lat2 = lat2 * degrad
    lon2 = lon2 * degrad
    lat1 = lat1 * degrad
    lon1 = lon1 * degrad

    dlat = (lat2 - lat1)
    dlon = (lon2 - lon1)
    a = sin(dLat / 2.) * sin(dLat / 2.) &
        + cos(lat1) * cos(lat2) * sin(dLon / 2.) * sin(dLon / 2.)
    c = 2.*atan2(sqrt(a), sqrt(1.-a))
    distance = max(0.0, R * c)
endfunction distance
