! greenhouse_climatology.F90
! 
!   Created on: 29/09/2022
!       Author: J.-F. Grailet (jefgrailet)
! 
! Module created for the sole purpose of managing a few NetCDF files provided by the ECMWF to 
! prepare greenhouse gas data for the ecRad radiative scheme. The class greenhouse_type it 
! provides is responsible for scaling monthly mean volume mixing ratios for a given period to a 
! given year in a given CMIP scenario and for interpolating the resulting climatologies on a 
! regional grid.
! 
! Some remarks:
! 1) The class is not responsible for managing the unit used in the NetCDF files (typically volume 
!    mixing ratio). The radiation_gas module of ecRad already gives the option to input gas 
!    concentrations as mass mixing ratio or volume mixing ratio.
! 2) While the class was designed to manage the data from a specific format of NetCDF file, it may 
!    be edited later to accept other formats. It would typically require making the VarNames* and 
!    Mappings* arrays non static.

module greenhouse_climatology

  use parkind1, only : jprb, jprd
  use radiation_io, only : nulout, nulerr, my_abort => radiation_abort
  use easy_netcdf
  
  implicit none
  public
  
  ! Various constants tied to the format of the NetCDF files used by this class
  integer, parameter :: NClimatologies = 8 ! Greenhouse gases in the main NetCDF file
  integer, parameter :: NGasesCMIP = 7 ! Number of gases stored in time series
  integer, parameter :: NYearsCMIP = 2501 ! Covers 0-2500 time period (years)

  ! Name of the variables to read in the NetCDF climatology files
  character*10, dimension(NClimatologies), parameter :: VarNamesVMR &
  &  = (/'co2_vmr   ', 'o3_vmr    ', 'n2o_vmr   ', 'ch4_vmr   ', &
  &      'cfc11_vmr ', 'cfc12_vmr ', 'hcfc22_vmr', 'ccl4_vmr  '/)
  
  ! Name of the variables to read in the NetCDF CMIP scenarios files
  character*10, dimension(NGasesCMIP), parameter :: VarNamesCMIP &
  &  = (/'co2_vmr   ', 'ch4_vmr   ', 'n2o_vmr   ', 'cfc11_vmr ', & 
  &      'cfc12_vmr ', 'hcfc22_vmr', 'ccl4_vmr  '/)
  
  ! Mappings between indices from CMIP scenario and greenhouse gases in climatologies
  integer, dimension(NGasesCMIP), parameter :: MappingsCMIP & 
  &  = (/ 1, 4, 3, 5, 6, 7, 8 /)
  
  ! Mappings between the ecRad constants (in radiation_gas_constants) and the indices used here
  integer, dimension(12), parameter :: MappingsEcRad & 
  & = (/ -1, 1, 2, 3, -1, 4, -1, 5, 6, 7, 8, -1 /)
  
  ! Single longitudinal slice (used to declare fixed-size array of allocatable arrays)
  type climatology_type
    real(kind=jprb), allocatable :: slice(:,:)
  end type
  
  ! Single time series (same as above)
  type time_series_type
    real(kind=jprb), allocatable :: series(:)
  end type
  
  ! Collection of 2D slices
  type greenhouse_type

    ! Axes details for interpolation (must be allocatable for easy interfacing with easy_netcdf)
    real(kind=jprb), allocatable :: latitudes(:)
    real(kind=jprb), allocatable :: pressure_levels(:)
    
    ! Monthly values for a selected number of greenhouse gases and a given month (assumes a MAR 
    ! simulation never overlaps two distinct months, given the block decomposition).
    type(climatology_type), dimension(NClimatologies) :: monthly_values

  contains
  
    procedure :: allocate => allocate_greenhouse
    procedure :: deallocate => deallocate_greenhouse
    procedure :: interpolate
    procedure :: output_monthly_values
  
  end type greenhouse_type

contains

  ! Creates an instance of the greenhouse_type derived type. It requires two NetCDF files: a first 
  ! one provides monthly mean volume mixing ratios for greenhouse gases, while the second file 
  ! should provide the annual mean volume mixing ratios of each greenhouse gas in a given CMIP 
  ! scenario. Four additional parameters are required: the first year of the period covered by the 
  ! monthly means NetCDF file, the last year for the same period, a month whose means should be 
  ! extracted, and the year to which the mixing ratios should be scaled via the CMIP scenario. 
  ! By the end of the subroutine, the instance encapsulates monthly means for each greenhouse gas 
  ! for a given month and scaled to a given year according to a given scenario.
  
  subroutine allocate_greenhouse(this, means_path, CMIP_path, first_year, last_year, month, year)
  
    class(greenhouse_type), intent(inout) :: this
    character(len=*), intent(in) :: means_path, CMIP_path
    integer, intent(in) :: first_year, last_year, month, year
    
    type(netcdf_file) :: means_file, cmip_file
    
    ! Local variables
    integer :: i, j, k, l, max_l
    real(kind=jprb) :: scaling_factor
    real(kind=jprb) :: sum_period, mean_period
    type(time_series_type), dimension(NGasesCMIP) :: CMIP_time_series ! Array of time series
    
    if (first_year > last_year) then
      call my_abort('Greenhouse: the provided period is invalid (first year > last year).')
    end if
    
    ! Loads axes data and monthly values from the monthly means NetCDF file
    call means_file%open(trim(means_path))
    
    call means_file%get_real_vector('latitude', this%latitudes)
    call means_file%get_real_vector('pressure', this%pressure_levels)
    
    do i=1, NClimatologies
      call means_file%get_real_matrix_indexed(trim(VarNamesVMR(i)), & 
       &                                      this%monthly_values(i)%slice, month)
    end do
    
    call means_file%close()
    
    ! Loads the time series of the selected CMIP scenario
    call cmip_file%open(trim(CMIP_path))
    
    do i=1, NGasesCMIP
      call cmip_file%get_real_vector(trim(VarNamesCMIP(i)), CMIP_time_series(i)%series)
    end do
    
    call cmip_file%close()
    
    ! Scales the mixing ratios to the selected year for each gas
    do i=1, NGasesCMIP
      
      ! Uses the CMIP time series to compute the scaling factor. Scaling is based on the mean 
      ! value for the period covered by the monthly means NetCDF file in the selected scenario.
      sum_period = 0.0_jprb
      max_l = last_year - first_year + 1
      do l=1, max_l
        sum_period = sum_period + CMIP_time_series(i)%series(first_year + l)
      end do
      mean_period = sum_period / real(max_l)
      scaling_factor = CMIP_time_series(i)%series(year) / mean_period
      
      ! Scales the climatology accordingly
      do j=1, size(this%latitudes); do k=1, size(this%pressure_levels)

        this%monthly_values(MappingsCMIP(i))%slice(j, k) = & 
         & this%monthly_values(MappingsCMIP(i))%slice(j, k) * scaling_factor

      end do; end do
      
    end do
    
    ! Frees the memory allocated by easy_netcdf while reading the CMIP scenario
    do i=1, NGasesCMIP
      deallocate(CMIP_time_series(i)%series)
    end do
  
  end subroutine allocate_greenhouse
  
  ! Frees the memory allocated to store greenhouse gas data (previously allocated by easy_netcdf).
  
  subroutine deallocate_greenhouse(this)
  
    class(greenhouse_type), intent(inout) :: this
    integer :: i
    
    deallocate(this%latitudes)
    deallocate(this%pressure_levels)
    
    do i=1, NClimatologies
      deallocate(this%monthly_values(i)%slice)
    end do
    
  end subroutine deallocate_greenhouse
  
  ! Interpolates the monthly values for a selected gas. Given that the NetCDF file that provides 
  ! the volume mixing ratios typically has a lower resolution than MAR on the latitude axis but a 
  ! higher resolution on the levels, the interpolation process only selects the closest value 
  ! w.r.t. latitude and pressure level. Starting from December 2022, also receives a "smoothing" 
  ! parameter telling how the interpolated values should be smoothed (0 corresponding to no 
  ! smoothing at all). WARNING: the sel_lev input parameter MUST provide pressure in Pascal for 
  ! each level; do not mix up with the sigma levels used in the MAR !
  
  subroutine interpolate(this, sel_gas, sel_lat, sel_lev, smoothing, output)
  
    class(greenhouse_type), intent(inout) :: this
    integer, intent(in) :: sel_gas
    real(kind=jprb), dimension(:), intent(in) :: sel_lat
    real(kind=jprb), dimension(:), intent(in) :: sel_lev
    integer, intent(in) :: smoothing
    real(kind=jprb), dimension(:,:), intent(inout), allocatable :: output
    
    integer :: dimLat, nbLvls ! Based on the provided arrays
    integer :: gas ! Gas index in the VarNamesVMR/monthly_values arrays
    integer :: j, k
    integer :: il, ip ! il = index latitude, ip = index pressure
    real(kind=jprb) :: diff1, diff2
    real(kind=jprd) :: sum_level, avg_level ! For smoothing with "Horizontal average" (1) method
    
    ! !!! J.-F. Grailet remark (02/02/2023): jprd precision (see aux/parkind1.F90) is used here on 
    ! purpose to avoid bad roundings while computing horizontal averages, as volume or mass mixing 
    ! ratios typically have many decimals that go beyond the significant digits of simple 
    ! precision (default precision used in ecRad).
    
    dimLat = size(sel_lat)
    nbLvls = size(sel_lev)
    gas = MappingsEcRad(sel_gas)
    allocate(output(dimLat, nbLvls))
    
    if (gas <= 0 .or. gas > NClimatologies) then
      call my_abort('Greenhouse: cannot interpolate because the selected gas does not exist')
    end if
    
    il=1
    do j=1, dimLat
      
      ! Finds next column whose latitude is greater than sel_lat(j)
      do while(il <= size(this%latitudes) .and. sel_lat(j) > this%latitudes(il))
        il = il+1
      end do
      
      ! North pole
      if(il > size(this%latitudes)) then
        il = il-1
      end if
      
      ! Adjusts if latitudes(il-1) is closer
      if (il > 1) then
        diff1 = this%latitudes(il) - sel_lat(j)
        diff2 = sel_lat(j) - this%latitudes(il-1)
        if (diff1 > diff2) then
          il = il-1
        end if
      end if
      
      ! N.B.: pressure levels are in increasing order (i.e., greatest ip -> closest to surface)
      ip = 1
      do k=1, nbLvls
      
        ! Finds next level whose pressure is greater than sel_lev(k)
        do while(ip <= size(this%pressure_levels) .and. sel_lev(k) > this%pressure_levels(ip))
          ip = ip+1
        end do
        
        ! Level closest to surface
        if(ip > size(this%pressure_levels)) then
          ip = ip-1
        end if
        
        ! Adjusts if pressure_levels(ip-1) is closer
        if (ip > 1) then
          diff1 = this%pressure_levels(ip) - sel_lev(k)
          diff2 = sel_lev(k) - this%pressure_levels(ip-1)
          if (diff1 > diff2) then
            ip = ip-1
          end if
        end if
        
        ! Saves (il, ip) value
        output(j, k) = this%monthly_values(gas)%slice(il,ip)
        
      end do
    end do
    
    ! Smoothes the interpolated values if requested
    if (smoothing == 1) then ! Horizontal average

      do k=1, nbLvls

        ! Computes the average for this level
        sum_level = 0.
        do j=1, dimLat
          sum_level = sum_level + output(j, k)
        end do
        avg_level = sum_level / dimLat

        ! Replaces the values for this level by the average
        do j=1, dimLat
          output(j, k) = real(avg_level, kind=jprb)
        end do

      end do

    end if
  
  end subroutine interpolate
  
  ! Writes monthly values for a selected month to an output file. This is only for test purpose 
  ! (i.e., check the monthly values after scaling via ferret).
  
  subroutine output_monthly_values(this, filename)
  
    class(greenhouse_type), intent(inout) :: this
    character(len=*), intent(in) :: filename
    
    type(netcdf_file) :: new_nc_file
    integer :: i
    
    ! Long names for metadata (copy/paste from drivers/radiation_climatologies.F90)
    character*55, dimension(NClimatologies), parameter :: VarLongNames = (/ &
    & 'CO2 (carbon dioxide) volume mixing ratio               ', &
    & 'O3 (ozone) volume mixing ratio                         ', &
    & 'N2O (nitrous oxide) volume mixing ratio                ', &
    & 'CH4 (methane) volume mixing ratio                      ', &
    & 'CFC-11 (trichlorofluoromethane) volume mixing ratio    ', &
    & 'CFC-12 (dichlorodifluoromethane) volume mixing ratio   ', &
    & 'HCFC-22 (chlorodifluoromethane) volume mixing ratio    ', &
    & 'CCl4 (carbon tetrachloride) volume mixing ratio        ' /)
    
    call new_nc_file%create(filename)
    call new_nc_file%define_dimension('level', size(this%pressure_levels))
    call new_nc_file%define_dimension('latitude', size(this%latitudes))
    
    ! Defines the variables
    call new_nc_file%define_variable('latitude', dim1_name='latitude')
    call new_nc_file%define_variable('pressure', dim1_name='level')
    do i=1, NClimatologies
      call new_nc_file%define_variable(trim(VarNamesVMR(i)), dim1_name='latitude', & 
       &                               dim2_name='level')
    end do
    
    ! Writes some metadata
    call new_nc_file%put_attribute('latitude', 'long_name', 'Latitude')
    call new_nc_file%put_attribute('latitude', 'units', 'degrees')
    call new_nc_file%put_attribute('pressure', 'long_name', 'Pressure levels')
    call new_nc_file%put_attribute('pressure', 'units', 'Pascal (Pa)')
    
    do i=1, NClimatologies
      call new_nc_file%put_attribute(trim(VarNamesVMR(i)), 'long_name', trim(VarLongNames(i)))
      call new_nc_file%put_attribute(trim(VarNamesVMR(i)), 'units', '1')
    end do
    
    ! Stores the data
    call new_nc_file%put_real_vector('latitude', this%latitudes)
    call new_nc_file%put_real_vector('pressure', this%pressure_levels)
    do i=1, NClimatologies
      call new_nc_file%put_real_matrix(trim(VarNamesVMR(i)), this%monthly_values(i)%slice)
    end do
    
    call new_nc_file%close()
    
  end subroutine output_monthly_values

end module greenhouse_climatology
