! aerosols_climatology.F90
! 
!   Created on: 25/10/2022
!       Author: J.-F. Grailet (jefgrailet)
! 
! Module created for the sole purpose of managing one NetCDF file provided by the ECMWF to 
! prepare aerosols data for the ecRad radiative scheme. The class aerosols_type it provides is 
! meant to interpolate the mass mixing ratios 3D grids on a regional grid.
! 
! Some remarks:
! 1) The class is not responsible for managing the unit used in the NetCDF file (typically mass 
!    mixing ratio). The radiation_gas module of ecRad already gives the option to input gas 
!    concentrations as mass mixing ratio or volume mixing ratio.
! 2) While the class was designed to manage the data from a specific (format of) NetCDF file, it 
!    may be edited later to accept other files. It may require making VarNamesMMR non static.

module aerosols_climatology

  use parkind1, only : jprb, jprd
  use radiation_io, only : nulout, nulerr, my_abort => radiation_abort
  use easy_netcdf
  
  implicit none
  public

  ! Aerosols expected in the provided NetCDF file
  integer, parameter :: NClimatologies = 11
  
  ! Name of the variables to read in the NetCDF file for aerosols
  character*30, dimension(NClimatologies), parameter :: AerosolsVarNames &
  &  = (/'Sea_Salt_bin1                 ', 'Sea_Salt_bin2                 ', & 
  &      'Sea_Salt_bin3                 ', 'Mineral_Dust_bin1             ', & 
  &      'Mineral_Dust_bin2             ', 'Mineral_Dust_bin3             ', &
  &      'Organic_Matter_hydrophilic    ', 'Organic_Matter_hydrophobic    ', &
  &      'Black_Carbon_hydrophilic      ', 'Black_Carbon_hydrophobic      ', &
  &      'Sulfates                      ' /)
  
  ! J.-F. Grailet note: the next derived type solely exist to be able to declare a fixed-size 
  ! array of allocatable's for convenient handling of climatologies via the easy_netcdf module.
  
  ! Single 3D grid for aerosol mass mixing ratios
  type climatology_3D_type
    real(kind=jprb), allocatable :: array3D(:,:,:)
  end type
  
  ! Collection of 3D arrays (aerosolS because several climatologies are stored in a same object)
  type aerosols_type
    
    ! Axes details for interpolation (must be allocatable for easy interfacing with easy_netcdf)
    real(kind=jprb), allocatable :: longitudes(:)
    real(kind=jprb), allocatable :: latitudes(:)
    real(kind=jprb), allocatable :: pressure_levels(:,:,:)
    
    ! Monthly values for a selected number of aerosols and a given month (assumes a MAR simulation 
    ! never overlaps two distinct months, given the block decomposition).
    type(climatology_3D_type), dimension(NClimatologies) :: monthly_values
  
  contains
  
    procedure :: allocate => allocate_aerosols
    procedure :: deallocate => deallocate_aerosols
    procedure :: interpolate
  
  end type aerosols_type

contains

  ! Creates a new instance of the aerosols_type derived type by loading aerosol mass mixing ratios 
  ! for a given month in a given NetCDF file.
  
  subroutine allocate_aerosols(this, netcdf_path, month)
  
    class(aerosols_type), intent(inout) :: this
    
    character(len=*), intent(in) :: netcdf_path
    integer, intent(in) :: month
    
    type(netcdf_file) :: nc_file
    integer :: i
    
    call nc_file%open(trim(netcdf_path))
    
    ! Loads axes data
    call nc_file%get_real_vector('lon', this%longitudes)
    call nc_file%get_real_vector('lat', this%latitudes)
    call nc_file%get_real_array3_indexed('pressure', this%pressure_levels, month)
    
    ! Loads monthly values
    do i=1, NClimatologies
      call nc_file%get_real_array3_indexed(trim(AerosolsVarNames(i)), & 
                                        &  this%monthly_values(i)%array3D, month)
    end do
    
    call nc_file%close()
  
  end subroutine allocate_aerosols
  
  ! Frees the memory allocated to store aerosols data (previously allocated by easy_netcdf).
  
  subroutine deallocate_aerosols(this)
  
    class(aerosols_type), intent(inout) :: this
    
    integer :: i
    
    deallocate(this%longitudes)
    deallocate(this%latitudes)
    deallocate(this%pressure_levels)
    
    do i=1, NClimatologies
      deallocate(this%monthly_values(i)%array3D)
    end do
  
  end subroutine deallocate_aerosols
  
  ! Interpolates the monthly values for a selected aerosol. Given that the NetCDF file that 
  ! provides the mass mixing ratios typically has a lower resolution than MAR on the longitude and 
  ! latitude axes but a higher resolution on the levels, the interpolation process only selects 
  ! the closest value w.r.t. latitude and pressure level after picking a longitudinal slice based 
  ! on the average longitude of the regional grid.
  ! 
  ! Important remarks:
  ! 1) The sel_lev input parameter MUST provide pressure in Pascal for each level; do not mix up 
  !    with the sigma levels used in the MAR !
  ! 2) Contrary to interpolate in greenhouse_climatology, the 'output' array is not allocated 
  !    here. This is due to the fact that ecRad expects aerosols mass mixing ratios in a 3D format 
  !    (dimLat x nbLvls x nbAerosols). Such an array is therefore allocated outside, and it is 
  !    assumed sel_aerosol is the (aerosol) index where interpolated values must be written.
  ! 3) Starting from December 2022, also receives a "smoothing" parameter telling how the 
  !    interpolated values should be smoothed (0 corresponding to no smoothing at all).
  
  subroutine interpolate(this, sel_aerosol, sel_lon, sel_lat, sel_lev, smoothing, output)
  
    class(aerosols_type), intent(inout) :: this
    integer, intent(in) :: sel_aerosol
    real(kind=jprb), intent(in) :: sel_lon ! Average
    real(kind=jprb), dimension(:), intent(in) :: sel_lat
    real(kind=jprb), dimension(:), intent(in) :: sel_lev
    integer, intent(in) :: smoothing
    real(kind=jprb), dimension(:,:,:), intent(inout) :: output
    real(kind=jprd) :: sum_level, avg_level ! For smoothing with "Horizontal average" (1) method
    
    ! !!! J.-F. Grailet remark (02/02/2023): jprd precision (see aux/parkind1.F90) is used here on 
    ! purpose to avoid bad roundings while computing horizontal averages, as volume or mass mixing 
    ! ratios typically have many decimals that go beyond the significant digits of simple 
    ! precision (default precision used in ecRad).
    
    integer :: dimLat, nbLvls ! Based on the provided arrays
    integer :: myLat ! Number of latitude points in the NetCDF file
    integer :: j, k
    integer :: ilon, ilat, ip ! ilon = index longitude, ilat = index latitude, ip = index pressure
    real(kind=jprb) :: adj_lon
    real(kind=jprb) :: diff1, diff2
    
    myLat = size(this%latitudes)
    dimLat = size(sel_lat)
    nbLvls = size(sel_lev)
    
    if (sel_aerosol <= 0 .or. sel_aerosol > NClimatologies) then
      call my_abort('Aerosols: cannot interpolate because the selected aerosol does not exist')
    end if
    
    ! Adjusts average longitude if negative (file from ECMWF uses 0E -> 360E)
    adj_lon = sel_lon
    if (adj_lon < 0) then
      adj_lon = adj_lon + 360.
    end if
    
    ! Picks ilon the same way as the latitude is chosen in interpolate from greenhouse_climatology
    ilon = 1
    do while(ilon <= size(this%longitudes) .and. adj_lon > this%longitudes(ilon))
      ilon = ilon+1
    end do
    
    ! Greenwhich
    if(ilon > size(this%longitudes)) then
      ilon = ilon-1
    end if
    
    ! Adjusts ilon if longitudes(ilon-1) is closer
    if (ilon > 1) then
      diff1 = this%longitudes(ilon) - adj_lon
      diff2 = adj_lon - this%longitudes(ilon-1)
      if (diff1 > diff2) then
        ilon = ilon-1
      end if
    end if
    
    ! Selecting latitude and pressure level is performed just like in greenhouse_climatology
    ilat=1
    do j=1, dimLat
      
      ! WARNING !!! Latitudes are decreasing in the aerosols CAMS NetCDF file, contrary to 
      ! conventions. The this%latitudes array is therefore read backwards (having myLat 
      ! cells). Same goes for the latitude axis of the this%pressure_levels array and the 
      ! selected value in this%monthly_values(sel_aerosol)%array3D.
      
      ! Finds next column whose latitude is greater than sel_lat(j)
      do while(ilat <= myLat .and. sel_lat(j) > this%latitudes(myLat-ilat+1))
        ilat = ilat+1
      end do
      
      ! North pole
      if(ilat > myLat) then
        ilat = ilat-1
      end if
      
      ! Adjusts if latitudes(myLat - ilat) is closer
      if (ilat > 1) then
        diff1 = this%latitudes(myLat-ilat+1) - sel_lat(j)
        diff2 = sel_lat(j) - this%latitudes(myLat - ilat)
        if (diff1 > diff2) then
          ilat = ilat-1
        end if
      end if
      
      ! N.B.: pressure levels are in increasing order (i.e., greatest ip -> closest to surface)
      ip = 1
      do k=1, nbLvls
      
        ! Finds next level whose pressure is greater than sel_lev(k)
        do while(ip <= size(this%pressure_levels, 3) .and. &
         &       sel_lev(k) > this%pressure_levels(ilon, myLat-ilat+1, ip))
          ip = ip+1
        end do
        
        ! Level closest to surface
        if(ip > size(this%pressure_levels, 3)) then
          ip = ip-1
        end if
        
        ! Adjusts if pressure_levels(ip-1) is closer
        if (ip > 1) then
          diff1 = this%pressure_levels(ilon, myLat-ilat+1, ip) - sel_lev(k)
          diff2 = sel_lev(k) - this%pressure_levels(ilon, myLat-ilat+1, ip-1)
          if (diff1 > diff2) then
            ip = ip-1
          end if
        end if
        
        ! Saves (ilon, ilat, ip) value
        output(j, k, sel_aerosol) = this%monthly_values(sel_aerosol)%array3D(ilon,myLat-ilat+1,ip)
        
      end do
    end do
    
    ! Smoothes the interpolated values if requested
    if (smoothing == 1) then ! Horizontal average

      do k=1, nbLvls

        ! Computes the average for this level
        sum_level = 0.
        do j=1, dimLat
          sum_level = sum_level + output(j, k, sel_aerosol)
        end do
        avg_level = sum_level / dimLat

        ! Replaces the values for this level by the average
        do j=1, dimLat
          output(j, k, sel_aerosol) = avg_level
        end do

      end do

    end if
    
  end subroutine interpolate

end module aerosols_climatology
