MODULE RADIATION_SETUP

! RADIATION_SETUP - Setting up modular radiation scheme
!
! (C) Copyright 2015 - ECMWF.
!
! This software is licensed under the terms of the Apache Licence Version 2.0
! which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
!
! In applying this licence, ECMWF does not waive the privileges and immunities
! granted to it by virtue of its status as an intergovernmental organisation
! nor does it submit to any jurisdiction.
!
! PURPOSE
! -------
!   The modular radiation scheme is contained in a separate
!   library. SETUP_RADIATION_SCHEME in this module sets up a small
!   number of global variables needed to store the information for it.
!
!   Lower case is used for variables and types taken from the
!   radiation library
!
! INTERFACE
! ---------
!   SETUP_RADIATION_SCHEME is called from SUECRAD.  The radiation
!   scheme is actually run using the RADIATION_SCHEME routine (not in
!   this module).
!
! AUTHOR
! ------
!   Robin Hogan, ECMWF
!   Original: 2015-09-16
!
! MODIFICATIONS
! -------------
! 
! Modifications brought to the routine to prepare the inclusion in the MAR 
! model, by J.-F. Grailet (September, October and November 2022):
! -removed all instructions related to the Dr. Hook library.
! -added a local directory to access radiation data.
! -included NICEOPT and NLIQOPT via the YOERAD module (declared outside of any 
!  derived type) due to the lack of a declaration for a YRERAD derived type in 
!  the ecRad 1.5.0 archive.
! -included NAERMACC via the additional module INCMAR (not declared in the IFS 
!  files provided in the archive; normally part of the YRERAD derived type).
! -likewise, again because YRERAD derived type was not provided, added 
!  NLWSOLVER, NSWSOLVER and NLWSCATTERING to INCMAR. Their typical values 
!  could be inferred from the code. A fourth option was added for both 
!  NLWSOLVER and NSWSOLVER: Tripleclouds (=3 for both variables). For this 
!  very reason, ISolverTripleclouds was added in the list of variables used 
!  from the radiation_config module.
! -removal of the "POSNAME(...)" sequence of instructions due to the absence 
!  of POSNAME itself in the ecRad 1.5.0 archive. Another variable only used 
!  in this sequence (ISTAT) has been removed as well. The "USE YOMLUN" 
!  instruction also no longer requires NULNAM (tied to the aforementioned 
!  sequence), which also does not appear in the provided yomlun.F90.
! -added a global logical variable to signal whether the setup routine has 
!  been already called once (matters in the context of the MAR model - for the 
!  time being, at least).
! -added a mandatory parameter (thus, before LOUTPUT) to provide the path to 
!  the directory containing the NetCDF files typically used by ecRad.
! -renamed LOUTPUT as IOUTPUT and modified the associated code so that a MAR 
!  user can fully control the verbosity of ecRad via a namelist file.
! -modified the code regarding aerosols to reduce the number of aerosol 
!  classes to 11 (see code below "IF (NAERMACC > 0) THEN").
! -commented all instructions related to background aerosols after discussing 
!  the matter with Robin Hogan on November 8, 2022 (see also comment above the 
!  ITYPE_* variables).
! -refreshed the code regarding longwave scattering due to the NLWSCATTERING 
!  variable having been split into LCLD_SCATTERING and LAER_SCATTERING (use 
!  longwave scattering for clouds and for aerosols, respectively).
! -added code at the end of the subroutine to prepare the matrix used to 
!  compute a broader spectral decomposition of the surface solar downward 
!  fluxes, using an updated ecRad published by R. Hogan in late November 2022.
! -added some additional flag setting instructions to prevent a few error 
!  messages that happened with the new version of ecRad (see above).
! 
!-----------------------------------------------------------------------

  USE PARKIND1,         ONLY : JPRB
  USE radiation_config, ONLY : config_type, &
       &                       ISolverMcICA, ISolverSpartacus, &
       &                       ISolverTripleclouds, ILiquidModelSlingo, &
       &                       ILiquidModelSOCRATES, IIceModelFu, &
       &                       IIceModelBaran, IOverlapExponential, &
       &                       IOverlapExponentialRandom

  IMPLICIT NONE

  ! Store configuration information for the radiation scheme in a
  ! global variable
  type(config_type) :: rad_config
  
  ! Ultraviolet weightings
  INTEGER         :: NWEIGHT_UV
  INTEGER         :: IBAND_UV(100)
  REAL(KIND=JPRB) :: WEIGHT_UV(100)
  ! Photosynthetically active radiation weightings
  INTEGER         :: NWEIGHT_PAR
  INTEGER         :: IBAND_PAR(100)
  REAL(KIND=JPRB) :: WEIGHT_PAR(100)

  ! J.-F. Grailet (09/11/2022): all instructions regarding tropospheric and 
  ! stratospheric background aerosols have been commented (effectively 
  ! removing their effect) for three reasons. First, they concerned only 
  ! two types of aerosols (hydrophobic organic matter and stratospheric 
  ! sulphate). Second, one of both types, i.e. stratospheric sulphate, is not 
  ! taken into account at all in the MAR model (the aerosols forcings file 
  ! being used does not provide data for it). Third and last, background 
  ! aerosols have only a minor effect on ecRad in the IFS (after discussing 
  ! the matter with Robin Hogan), so the most sensible choice regarding the 
  ! inclusion of ecRad in the MAR was to comment said instructions.

!  ! Background aerosol is specified in an ugly way: using the old
!  ! Tegen fields that are in terms of optical depth, and converted to
!  ! mass mixing ratio via the relevant mass-extinction coefficient
!  INTEGER, PARAMETER :: ITYPE_TROP_BG_AER = 8 ! hydrophobic organic
!  INTEGER, PARAMETER :: ITYPE_STRAT_BG_AER=12 ! non-absorbing sulphate
!  REAL(KIND=JPRB)    :: TROP_BG_AER_MASS_EXT
!  REAL(KIND=JPRB)    :: STRAT_BG_AER_MASS_EXT

CONTAINS

  ! (description slightly modified by J.-F. Grailet)
  ! This routine copies information between MAR/ecRad radiation configuration 
  ! (stored in global variables, with the exception of DATA_PATH) and the 
  ! radiation configuration of the modular radiation scheme (stored in 
  ! rad_config). The optional input integer IOUTPUT controls the verbosity of 
  ! ecRad while setting up (verbosity of ecRad itself is hard-coded to only 
  ! print errors/warnings).

  SUBROUTINE SETUP_RADIATION_SCHEME(DATA_PATH, IOUTPUT)

    USE YOMLUN,   ONLY : NULOUT, NULERR
    USE YOESRTWN, ONLY : NMPSRTM
    USE YOERAD,   ONLY : NICEOPT, NLIQOPT
    USE INCMAR,   ONLY : LCLOUD_BETA_OVERLAP, & 
     &                   NAERMACC, NLWSOLVER, NSWSOLVER, &
     &                   LCLD_SCATTERING, LAER_SCATTERING, &
     &                   NGASMODEL, ECCKD_LW_FILE, ECCKD_SW_FILE

    USE radiation_interface,      ONLY : setup_radiation
    USE radiation_aerosol_optics, ONLY : dry_aerosol_mass_extinction
    USE radiation_config,         ONLY : IGasModelECCKD
    USE extra_ecRad_outputs,      ONLY : N_SW_SPECTRAL_BANDS, &
     &                                   sw_spectral_bounds, &
     &                                   sw_spectral_mapping
    
    IMPLICIT NONE
    
    ! Path to directory with NetCDF files used by ecRad
    CHARACTER(LEN=*), INTENT(IN) :: DATA_PATH
    
    ! Whether or not to provide information on the radiation scheme
    ! configuration (adjusted into integer by J.-F. Grailet)
    INTEGER, INTENT(IN), OPTIONAL :: IOUTPUT

    ! Verbosity of configuration information 0=silent, 1=print main 
    ! configuration only, 2=detail everything related to setup
    INTEGER :: IVERBOSESETUP

    ! -------------------------------------------------------------------

#include "abor1.intfb.h"

    ! -------------------------------------------------------------------

    ! *** GENERAL SETUP ***

    ! Configure verbosity of setup of radiation scheme
    IVERBOSESETUP = 2 ! Provide all details by default
    IF (PRESENT(IOUTPUT)) THEN
      IF (IOUTPUT >= 0 .and. IOUTPUT <= 2) THEN
        IVERBOSESETUP = IOUTPUT ! Simply copies the provided IOUTPUT (JFG)
      ENDIF
    ENDIF
    rad_config%iverbosesetup = IVERBOSESETUP

    ! Tweak w.r.t. initial code: > 0 dumps at least the final configuration
    IF (IVERBOSESETUP > 0) THEN
      WRITE(NULOUT,'(a)') '-------------------------------------------------------------------------------'
      WRITE(NULOUT,'(a)') 'RADIATION_SETUP'
    ENDIF

    ! Normal operation of the radiation scheme displays only errors
    ! and warnings (hard-coded while ecRad is included in MAR)
    rad_config%iverbose = 1

    ! Directory with data files used by ecRad is provided by calling code
    rad_config%directory_name = DATA_PATH

    ! Do we do Hogan and Bozzo (2014) approximate longwave updates?
    ! J.-F. Grailet remark (09/11/2022): always compute it as additional and 
    ! optional output (see rFLopt variable in PHYrad_ecRad.F90).
    rad_config%do_lw_derivatives = .TRUE.

    ! Surface spectral fluxes are needed for spectral shortwave albedo
    ! calculation and spectral decomposition
    rad_config%do_surface_sw_spectral_flux = .TRUE.
    
    ! J.-F. Grailet remark (22/11/2022): do NOT change the next params ! They 
    ! are necessary for the computation of PSWDIFFUSEBAND and PSWDIRECTBAND 
    ! (surface shortwave fluxes according to albedo bands).
    rad_config%use_canopy_full_spectrum_sw = .FALSE.
    rad_config%do_nearest_spectral_sw_albedo = .TRUE.
    
    ! J.-F. Grailet remark (01/12/2022): do NOT change the next params ! They 
    ! are required for when ecRad is run with the ecCKD gas model. Flipping 
    ! them will cause an abort by ecRad in the radiation_single_level module 
    ! (just like above).
    rad_config%use_canopy_full_spectrum_lw = .FALSE.
    rad_config%do_nearest_spectral_lw_emiss = .TRUE.

    ! *** SETUP GAS OPTICS ***

    ! WARNING !!! For the MAR model, this must be set to true, 
    ! otherwise ecRad crashes (almost) immediately. Within the IFS, it 
    ! is normally set to .FALSE. because the IFS already did the 
    ! set-up of RRTM prior to calling ecRad.
    rad_config%do_setup_ifsrrtm = .TRUE.

    ! *** SETUP CLOUD OPTICS ***

    ! Setup liquid optics
    IF (NLIQOPT == 2) THEN
      rad_config%i_liq_model = ILiquidModelSlingo
    ELSEIF (NLIQOPT == 3) THEN
      rad_config%i_liq_model = ILiquidModelSOCRATES
    ELSE
      WRITE(NULERR,'(a,i0)') 'Unavailable liquid optics model in modular radiation scheme: NLIQOPT=', &
           &  NLIQOPT
      CALL ABOR1('RADIATION_SETUP: error interpreting NLIQOPT')   
    ENDIF

    ! Setup ice optics
    IF (NICEOPT == 3) THEN
      rad_config%i_ice_model = IIceModelFu
    ELSEIF (NICEOPT == 4) THEN
      rad_config%i_ice_model = IIceModelBaran
    ELSE
      WRITE(NULERR,'(a,i0)') 'Unavailable ice optics model in modular radiation scheme: NICEOPT=', &
           &  NICEOPT
      CALL ABOR1('RADIATION_SETUP: error interpreting NICEOPT')   
    ENDIF

    ! For consistency with earlier versions of the IFS radiation
    ! scheme, we perform shortwave delta-Eddington scaling *after* the
    ! merge of the cloud, aerosol and gas optical properties.  Set
    ! this to "false" to do the scaling on the cloud and aerosol
    ! properties separately before merging with gases. Note that this
    ! is not compatible with the SPARTACUS solver.
    rad_config%do_sw_delta_scaling_with_gases = .TRUE.

    ! Use Exponential-Exponential cloud overlap to match original IFS
    ! implementation of Raisanen cloud generator
    if (NLWSOLVER > 0 .or. NSWSOLVER > 0) then
      rad_config%i_overlap_scheme = IOverlapExponentialRandom ! JFG edit
    else
      rad_config%i_overlap_scheme = IOverlapExponential
    endif
    
    ! J.-F. Grailet addition on 20/04/2023: do we use beta overlap (by 
    ! (Shonk et al., 2010) instead of alpha overlap (by Hogan and 
    ! Illingworth, 2000) ?
    rad_config%use_beta_overlap = LCLOUD_BETA_OVERLAP

    ! *** SETUP AEROSOLS ***

    rad_config%use_aerosols = .TRUE.

    IF (NAERMACC > 0) THEN
      ! Using MACC climatology - in this case the aerosol optics file
      ! will be chosen automatically

      ! 11 aerosol classes: 1-3 Sea salt, 4-6 Boucher desert dust,
      ! 7 hydrophilic organics, 8 hydrophobic organics, 9&10
      ! hydrophobic black carbon, 11 ammonium sulphate
      rad_config%n_aerosol_types = 11
      ! J.-F. Grailet remark (09/11/2022): IFS normally also includes 
      ! stratopheric sulphate (hydrophilic at 20~30% RH) as a 12th class 
      ! (the associated index being 14), but the aerosols forcings file used 
      ! with the MAR does not provide data for such a class. The aerosols map 
      ! is therefore restricted to the 11 types given by said file.

      ! Indices to the aerosol optical properties in
      ! aerosol_ifs_rrtm_*.nc, for each class, where negative numbers
      ! index hydrophilic aerosol types and positive numbers index
      ! hydrophobic aerosol types
      rad_config%i_aerosol_type_map = 0 ! There can be up to 256 types
      rad_config%i_aerosol_type_map(1:11) = (/ &
           &  -1, &  ! Sea salt, size bin 1 (OPAC)
           &  -2, &  ! Sea salt, size bin 2 (OPAC)
           &  -3, &  ! Sea salt, size bin 3 (OPAC)
           &   7, &  ! Desert dust, size bin 1 (Woodward 2001)
           &   8, &  ! Desert dust, size bin 2 (Woodward 2001)
           &   9, &  ! Desert dust, size bin 3 (Woodward 2001)
           &  -4, &  ! Hydrophilic organic matter (OPAC)
           &  10, &  ! Hydrophobic organic matter (OPAC)
           &  11, &  ! Black carbon (Boucher)
           &  11, &  ! Black carbon (Boucher)
           &  -5 /)  ! Hydrophilic ammonium sulphate (OPAC)

      ! Background aerosol mass-extinction coefficients are obtained
      ! after the configuration files have been read - see later in
      ! this routine.

    ELSE
      ! Using Tegen climatology
      rad_config%n_aerosol_types = 6
      rad_config%i_aerosol_type_map = 0 ! There can be up to 256 types
      rad_config%i_aerosol_type_map(1:6) = (/ &
           &  1, &  ! Continental background
           &  2, &  ! Maritime
           &  3, &  ! Desert
           &  4, &  ! Urban
           &  5, &  ! Volcanic active
           &  6 /)  ! Stratospheric background

      ! Manually set the aerosol optics file name (the directory will
      ! be added automatically)
      rad_config%aerosol_optics_override_file_name = 'aerosol_ifs_rrtm_tegen.nc'
    ENDIF

    ! *** SETUP GAS MODEL ***
    
    ! Addition brought on 17/11/2022 to allow MAR users to use the ecCKD gas 
    ! model instead of the default RRTM-G.
    
    IF (NGASMODEL > 0) THEN
      rad_config%i_gas_model = IGasModelECCKD ! Default is IGasModelIFSRRTMG
      rad_config%gas_optics_lw_override_file_name = ECCKD_LW_FILE
      rad_config%gas_optics_sw_override_file_name = ECCKD_SW_FILE
    ENDIF
    
    ! *** SETUP SOLVER ***

    ! 3D effects are off by default
    rad_config%do_3d_effects = .FALSE.

    ! Select longwave solver
    SELECT CASE (NLWSOLVER)
    CASE(0)
      rad_config%i_solver_lw = ISolverMcICA
    CASE(1)
      rad_config%i_solver_lw = ISolverSpartacus
    CASE(2)
      rad_config%i_solver_lw = ISolverSpartacus
      rad_config%do_3d_effects = .TRUE.
    CASE(3) ! JFG addition
      rad_config%i_solver_lw = ISolverTripleclouds
    CASE DEFAULT
      WRITE(NULERR,'(a,i0)') 'Unknown value for NLWSOLVER: ', NLWSOLVER
      CALL ABOR1('RADIATION_SETUP: error interpreting NLWSOLVER')
    END SELECT

    ! Select shortwave solver
    SELECT CASE (NSWSOLVER)
    CASE(0)
      rad_config%i_solver_sw = ISolverMcICA
    CASE(1)
      rad_config%i_solver_sw = ISolverSpartacus
      rad_config%do_3d_effects = .FALSE.
      IF (NLWSOLVER /= 1) THEN ! JFG edit: NLWSOLVER must be 1 for consistency
        CALL ABOR1('RADIATION_SETUP: cannot represent 3D effects in LW but not SW')
      ENDIF
    CASE(2)
      rad_config%i_solver_sw = ISolverSpartacus
      rad_config%do_3d_effects = .TRUE.
      IF (NLWSOLVER /= 2) THEN ! JFG edit: NLWSOLVER must be 2 for consistency
        CALL ABOR1('RADIATION_SETUP: cannot represent 3D effects in SW but not LW')
      ENDIF
    CASE(3) ! JFG addition
      rad_config%i_solver_sw = ISolverTripleclouds
    CASE DEFAULT
      WRITE(NULERR,'(a,i0)') 'Unknown value for NSWSOLVER: ', NSWSOLVER
      CALL ABOR1('RADIATION_SETUP: error interpreting NSWSOLVER')
    END SELECT

    ! SPARTACUS solver requires delta scaling to be done separately
    ! for clouds & aerosols
    !IF (rad_config%i_solver_sw == ISolverSpartacus) THEN
    !  rad_config%do_sw_delta_scaling_with_gases = .FALSE.
    !ENDIF
    
    ! Suggestion of R. Hogan (08/12/2022)
    rad_config%do_sw_delta_scaling_with_gases = .FALSE.

    ! Do we represent longwave scattering?
    rad_config%do_lw_cloud_scattering = .FALSE.
    rad_config%do_lw_aerosol_scattering = .FALSE.
    IF (LCLD_SCATTERING) THEN
      rad_config%do_lw_cloud_scattering = .TRUE.
    ENDIF
    IF (NAERMACC > 0 .and. LAER_SCATTERING) THEN
      ! Tegen climatology omits data required to do longwave
      ! scattering by aerosols, so only turn this on with a more
      ! recent scattering database
      rad_config%do_lw_aerosol_scattering = .TRUE.
    ENDIF

    ! JFG 08/02/2023: commented this to minimize redundancy
    ! Print configuration before consolidating it
    ! IF (IVERBOSESETUP > 1) THEN
    !  WRITE(NULOUT,'(a)') 'Radiation scheme settings:'
    !  CALL rad_config%print(IVERBOSE=IVERBOSESETUP)
    ! ENDIF

    ! Use configuration data to set-up radiation scheme, including
    ! reading scattering datafiles
    CALL setup_radiation(rad_config)
    
    ! JFG 04/05/2023, tweak w.r.t. initial code: 1 now means the final 
    ! consolidated ecRad configuration is dumped in the console so MAR users 
    ! can check it at the start of a run. 2 will print additional details, 
    ! e.g., about the ecCKD model.
    if (IVERBOSESETUP > 0) then
      CALL rad_config%print(IVERBOSE=2)
    end if

    ! If using RRTM-G gas model (default), maps the shortwave bands with the 
    ! albedo bands to later compute PSWDIRECTBAND and PSWDIFFUSEBAND.
    
    IF (NGASMODEL == 0) THEN
      ! Populate the mapping between the 14 RRTM shortwave bands and the
      ! 6 albedo inputs. The mapping according to the stated wavelength
      ! ranges of the 6-band model does not match the hard-wired mapping
      ! in NMPSRTM, but only the hard-wired values produce sensible
      ! results...
      ! Note that NMPSRTM(:)=(/  6, 6, 5, 5, 5, 5, 5, 4, 4, 3, 2, 2, 1, 6 /)
      rad_config%i_albedo_from_band_sw = NMPSRTM
      !    call rad_config%define_sw_albedo_intervals(6, &
      !         &  (/ 0.25e-6_jprb, 0.44e-6_jprb, 1.19e-6_jprb, &
      !         &     2.38e-6_jprb, 4.00e-6_jprb /),  (/ 1,2,3,4,5,6 /))
      
      ! Likewise between the 16 RRTM longwave bands and the 2 emissivity
      ! inputs (info taken from rrtm_ecrt_140gp_mcica.F90) representing
      ! outside and inside the window region of the spectrum
      ! rad_config%i_emiss_from_band_lw = (/ 1,1,1,1,1,2,2,2,1,1,1,1,1,1,1,1 /)
      call rad_config%define_lw_emiss_intervals(3, &
           &  (/ 8.0e-6_jprb,13.0e-6_jprb /),  (/ 1,2,1 /))
      ! J.-F. Grailet remark (21/09/2022): rrtm_ecrt_140gp_mcica.F90 has been 
      ! removed from ecRad code used in MAR, and will likely be removed from 
      ! the ecRad repository at some point due to being not used in practice.
    ENDIF

    ! Get spectral weightings for UV and PAR
    call rad_config%get_sw_weights(0.2e-6_jprb, 0.4415e-6_jprb, &
         &  NWEIGHT_UV, IBAND_UV, WEIGHT_UV, 'ultraviolet')
    call rad_config%get_sw_weights(0.4e-6_jprb, 0.7e-6_jprb, &
         &  NWEIGHT_PAR, IBAND_PAR, WEIGHT_PAR, &
         &  'photosynthetically active radiation, PAR')

    ! J.-F. Grailet remark (09/11/2022): the following instructions have been 
    ! commented not only because they concerned background aerosols (see above 
    ! comment in the constants of this module), but also because they require 
    ! a specific setting to work, which would be done through a call to the 
    ! set_aerosol_wavelength_mono subroutine from the radiation_config 
    ! module. Without this setting, a seg fault occurs. After discussing the 
    ! matter with Robin Hogan, the best was to comment these instructions.

!    IF (NAERMACC > 0) THEN
!      ! With the MACC aerosol climatology we need to add in the
!      ! background aerosol afterwards using the Tegen arrays.  In this
!      ! case we first configure the background aerosol mass-extinction
!      ! coefficient at 550 nm, which corresponds to the 10th RRTMG
!      ! shortwave band.
!      TROP_BG_AER_MASS_EXT  = dry_aerosol_mass_extinction(rad_config, &
!           &                                   ITYPE_TROP_BG_AER, 550.0e-9_jprb)
!      STRAT_BG_AER_MASS_EXT = dry_aerosol_mass_extinction(rad_config, &
!           &                                   ITYPE_STRAT_BG_AER, 550.0e-9_jprb)
!      ! J.-F. Grailet remark: the third arg equals 550nm, which amounts to 
!      ! shortwave band 10 with RRTMG band structure. The function 
!      ! dry_aerosol_mass_extinction no longer receives a band number (previous 
!      ! version of IFS) for a wavelength but said wavelength in meters.
!      
!      WRITE(NULOUT,'(a,i0)') 'Tropospheric background uses aerosol type ', &
!           &                 ITYPE_TROP_BG_AER
!      WRITE(NULOUT,'(a,i0)') 'Stratospheric background uses aerosol type ', &
!           &                 ITYPE_STRAT_BG_AER
!    ENDIF

    ! Compute matrix to get (custom) spectral decomposition of surface solar 
    ! downward fluxes
    if (N_SW_SPECTRAL_BANDS > 0) then
      call rad_config%get_sw_mapping(sw_spectral_bounds, sw_spectral_mapping)
    end if

    ! Tweak w.r.t. initial code: > 0 dumps at least the final configuration
    IF (IVERBOSESETUP > 0) THEN
      WRITE(NULOUT,'(a)') '-------------------------------------------------------------------------------'
    ENDIF

  END SUBROUTINE SETUP_RADIATION_SCHEME

END MODULE RADIATION_SETUP
