! radiation_climatologies.F90
! 
!   Created on: 30/09/2022
!       Author: J.-F. Grailet (jefgrailet)
! 
! This is a simple module to gather global variables storing climatological data for ecRad, namely 
! greenhouse gases and aerosols climatologies. Such data is generated only once in practice, as a 
! single instance of the MAR typically runs over a period for which said climatologies can be 
! assumed to be constant.

module radiation_climatologies

use parkind1, only : jprb

implicit none

! (26/10/2022) Constants regarding aerosols: in the MAR, old aerosol format (Tegen climatology) 
! uses 6 types of aerosols while the new format (MACC climatology) uses 11.
integer, parameter :: N_AEROSOLS_TEGEN = 6
integer, parameter :: N_AEROSOLS_MACC = 11

! Longitude (average) used to pick a longitudinal slice in aerosols 3D grids (saved in metadata)
real(kind=jprb) :: CLIM_LONGITUDE

! Latitudes and pressure levels for which mixing ratios were interpolated
real(kind=jprb), dimension(:), allocatable :: CLIM_LATITUDES
real(kind=jprb), dimension(:), allocatable :: CLIM_LEVELS

! Greenhouse gases volume mixing ratios
real(kind=jprb), dimension(:,:), allocatable :: PCO2
real(kind=jprb), dimension(:,:), allocatable :: PCH4
real(kind=jprb), dimension(:,:), allocatable :: PN2O
real(kind=jprb), dimension(:,:), allocatable :: PCFC11
real(kind=jprb), dimension(:,:), allocatable :: PCFC12
real(kind=jprb), dimension(:,:), allocatable :: PHCFC22
real(kind=jprb), dimension(:,:), allocatable :: PCCL4
real(kind=jprb), dimension(:,:), allocatable :: PO3

! Old aerosols format (Tegen)
real(kind=jprb), dimension(:,:,:), allocatable :: PAEROSOL_OLD

! Aerosols mass mixing ratios (MACC)
real(kind=jprb), dimension(:,:,:), allocatable :: PAEROSOL

! Note how greenhouse gases are stored in separate arrays while aerosols are in a single 3D array. 
! This is the format ecRad expects: indeed, ecRad allows a lot of flexibility regarding aerosols 
! (up to 256 types, in theory), so storing everything in a 3D array is more convenient.

contains
  
  ! A subroutine to write the interpolated data to some output NetCDF file. The purpose of this 
  ! subroutine is to allow the MAR user verify the results of the interpolations of greenhouse 
  ! gases/aerosols mixing ratios on the MAR grid in case of problem. Its use is purely optional.
  
  subroutine output_interpolated(filename)

    use radiation_io, only : nulout, nulerr, my_abort => radiation_abort
    use easy_netcdf
    
    implicit none
    
    ! Variables to write in the NetCDF file
    integer, parameter :: NGreenhouseGases = 8
    character*10, dimension(NGreenhouseGases), parameter :: VarNames1 &
    &  = (/'co2_vmr   ', 'o3_vmr    ', 'n2o_vmr   ', 'ch4_vmr   ', &
    &      'cfc11_vmr ', 'cfc12_vmr ', 'hcfc22_vmr', 'ccl4_vmr  '/)
    
    character*55, dimension(NGreenhouseGases), parameter :: VarLongNames1 = (/ &
    & 'CO2 (carbon dioxide) volume mixing ratio               ', &
    & 'O3 (ozone) volume mixing ratio                         ', &
    & 'N2O (nitrous oxide) volume mixing ratio                ', &
    & 'CH4 (methane) volume mixing ratio                      ', &
    & 'CFC-11 (trichlorofluoromethane) volume mixing ratio    ', &
    & 'CFC-12 (dichlorodifluoromethane) volume mixing ratio   ', &
    & 'HCFC-22 (chlorodifluoromethane) volume mixing ratio    ', &
    & 'CCl4 (carbon tetrachloride) volume mixing ratio        ' /)
    
    integer, parameter :: NAerosols = 11 ! Fixed for now (given the data used by MAR)
    character*30, dimension(NAerosols), parameter :: VarNames2 &
    &  = (/'Sea_Salt_bin1                 ', 'Sea_Salt_bin2                 ', & 
    &      'Sea_Salt_bin3                 ', 'Mineral_Dust_bin1             ', & 
    &      'Mineral_Dust_bin2             ', 'Mineral_Dust_bin3             ', &
    &      'Organic_Matter_hydrophilic    ', 'Organic_Matter_hydrophobic    ', &
    &      'Black_Carbon_hydrophilic      ', 'Black_Carbon_hydrophobic      ', &
    &      'Sulfates                      ' /)
    
    character*44, dimension(NAerosols), parameter :: VarLongNames2 = (/ &
    & 'Sea salt mass mixing ratio (bin 1)          ', &
    & 'Sea salt mass mixing ratio (bin 2)          ', &
    & 'Sea salt mass mixing ratio (bin 3)          ', &
    & 'Mineral dust mass mixing ratio (bin 1)      ', &
    & 'Mineral dust mass mixing ratio (bin 2)      ', &
    & 'Mineral dust mass mixing ratio (bin 3)      ', &
    & 'Hydrophilic organic matter mass mixing ratio', &
    & 'Hydrophobic organic matter mass mixing ratio', &
    & 'Hydrophilic black carbon mass mixing ratio  ', &
    & 'Hydrophobic black carbon mass mixing ratio  ', &
    & 'Sulfates mass mixing ratio                  ' /)
    
    character(len=*), intent(in) :: filename ! Plus path (and assumed to have the .nc extension)
    type(netcdf_file) :: nc_file
    
    integer :: i
    
    if(.not.allocated(PCO2)) then
      call my_abort('Climatologies: no interpolated value to output yet.')
    end if
    
    ! Creates the new NetCDF file
    call nc_file%create(filename)
    call nc_file%define_dimension('level', size(CLIM_LEVELS))
    call nc_file%define_dimension('latitude', size(CLIM_LATITUDES))
    
    call nc_file%define_variable('latitude', dim1_name='latitude')
    call nc_file%define_variable('pressure', dim1_name='level')
    do i=1, NGreenhouseGases
      call nc_file%define_variable(trim(VarNames1(i)), dim1_name='latitude', dim2_name='level')
    end do
    
    ! Aerosols are added only if PAEROSOL is allocated (only when MACC climatology is used)
    if (allocated(PAEROSOL)) then
      do i=1, NAerosols
        call nc_file%define_variable(trim(VarNames2(i)), dim1_name='latitude', dim2_name='level')
      end do
      ! Adds an attribute to signal the average longitude used
      call nc_file%put_global_attribute_real('average_longitude', CLIM_LONGITUDE)
    end if
    
    ! Metadata
    call nc_file%put_attribute('latitude', 'long_name', 'Latitude')
    call nc_file%put_attribute('latitude', 'units', 'degrees')
    call nc_file%put_attribute('pressure', 'long_name', 'Pressure levels')
    call nc_file%put_attribute('pressure', 'units', 'Pascal (Pa)')
    
    do i=1, NGreenhouseGases
      call nc_file%put_attribute(trim(VarNames1(i)), 'long_name', trim(VarLongNames1(i)))
      call nc_file%put_attribute(trim(VarNames1(i)), 'units', '1')
    end do
    
    if (allocated(PAEROSOL)) then
      do i=1, NAerosols
        call nc_file%put_attribute(trim(VarNames2(i)), 'long_name', trim(VarLongNames2(i)))
        call nc_file%put_attribute(trim(VarNames2(i)), 'units', 'kg kg-1')
      end do
    end if
    
    ! Stores dimension data
    call nc_file%put_real_vector('latitude', CLIM_LATITUDES)
    call nc_file%put_real_vector('pressure', CLIM_LEVELS)
    
    ! Greenhouse gas data
    call nc_file%put_real_matrix(trim(VarNames1(1)), PCO2)
    call nc_file%put_real_matrix(trim(VarNames1(2)), PO3)
    call nc_file%put_real_matrix(trim(VarNames1(3)), PN2O)
    call nc_file%put_real_matrix(trim(VarNames1(4)), PCH4)
    call nc_file%put_real_matrix(trim(VarNames1(5)), PCFC11)
    call nc_file%put_real_matrix(trim(VarNames1(6)), PCFC12)
    call nc_file%put_real_matrix(trim(VarNames1(7)), PHCFC22)
    call nc_file%put_real_matrix(trim(VarNames1(8)), PCCL4)
    
    ! Aerosol data
    if (allocated(PAEROSOL)) then
      do i=1, NAerosols
        call nc_file%put_real_matrix(trim(VarNames2(i)), PAEROSOL(:,:,i))
      end do
    end if
    
    call nc_file%close()

  end subroutine output_interpolated

end module radiation_climatologies
