! extra_ecRad_outputs.F90
! 
!   Created on: 27/10/2022
!       Author: J.-F. Grailet (jefgrailet)
! 
! Module created to gather constants, variables and subroutines to handle the output variables of 
! ecRad that are not practically used by the MAR model. This includes the spectral decomposition 
! of the surface solar downward fluxes based on the spectral bands picked by the user or as used 
! internally by ecRad and a dozen of "extra" fluxes (surface/TOA fluxes complementing the net and 
! clear-sky fluxes) which this module is named after. Managing and outputting these variables is 
! entirely optional in the MAR.
! 
! The spectral decomposition of the surface solar downward fluxes is a 3D variable, the two first 
! dimensions corresponding to the surface grid and the third dimension being the custom spectral 
! bands. Such a decomposition already exists within ecRad and can be outputted via this module, 
! though the used spectral bands may not always be suitable to the purpose of the simulation. 
! Therefore, this module allows users to define custom spectral bands that are later used to build 
! a matrix in RADIATION_SETUP for turning outputs of ecRad into the user-picked spectral bands in 
! the RADIATION_SCHEME subroutine. These bands can be built either from a range (described with 
! three values) or from a file listing the bounds. Note that verifying the format and soundness of 
! either input is done via the MAR_ecRad_config module.
! 
! The "extra" fluxes are in 2D format (i.e., longitude by latitude map providing surface or TOA 
! fluxes) and, for convenience, are returned by PHYrad_ecRad as a single 3D array where each level 
! corresponds to a distinct variable. Several constants are therefore defined to distinguish these 
! levels by variable name, should these outputs be used by the MAR model in some way. The same 
! constants are also used to write these fluxes to an additional and optional output file as 
! separate variables.
! 
! Both types of optional outputs have an associated NetCDF file with an unlimited time dimension 
! that is appended with the extra data produced at each call of PHYrad_ecRad. Initializing these 
! files and/or initializing the custom spectral bands must be done at the initialization of ecRad.

module extra_ecRad_outputs

  use parkind1, only : jprb

  ! Constants for the extra ecRad output variables
  integer, parameter :: N_EXTRA_OUTPUTS = 12

  character*20, dimension(N_EXTRA_OUTPUTS), parameter :: EXTRA_VAR_NAMES = (/ &
   & 'PFLUX_SW_DN         ', &
   & 'PFLUX_LW_DN         ', &
   & 'PFLUX_SW_DN_CLEAR   ', &
   & 'PFLUX_LW_DN_CLEAR   ', &
   & 'PFLUX_DIR           ', &
   & 'PFLUX_DIR_CLEAR     ', &
   & 'PFLUX_DIR_INTO_SUN  ', &
   & 'PFLUX_UV            ', &
   & 'PFLUX_PAR           ', &
   & 'PFLUX_PAR_CLEAR     ', &
   & 'PFLUX_SW_DN_TOA     ', &
   & 'PEMIS_OUT           ' /)

  character*88, dimension(N_EXTRA_OUTPUTS), parameter :: EXTRA_VAR_LONG_NAMES = (/ &
   & 'Surface downward shortwave fluxes                                                       ', &
   & 'Surface downward longwave fluxes                                                        ', &
   & 'Clear-sky surface downward shortwave fluxes                                             ', & 
   & 'Clear-sky surface downward longwave fluxes                                              ', &
   & 'Direct component of surface downward shortwave fluxes                                   ', & 
   & 'Direct component of clear-sky surface downward shortwave fluxes                         ', &
   & 'Direct component of surface downward shortwave fluxes, perpendicular to the Sun         ', &
   & 'Surface downward UV fluxes                                                              ', &
   & 'Surface downward shortwave fluxes in the Photosynthetically Active Range (PAR)          ', &
   & 'Clear-sky surface downward shortwave fluxes in the Photosynthetically Active Range (PAR)', &
   & 'Top-Of-Atmosphere (TOA) downward shortwave fluxes                                       ', &
   & 'Surface emissivity in the longwave                                                      ' /)

  integer, parameter :: ID_PFLUX_SW_DN = 1 ! Formerly "rFSOst" or "FSOs_t" with radCEP
  integer, parameter :: ID_PFLUX_LW_DN = 2
  integer, parameter :: ID_PFLUX_SW_DN_CLEAR = 3
  integer, parameter :: ID_PFLUX_LW_DN_CLEAR = 4
  integer, parameter :: ID_PFLUX_DIR = 5
  integer, parameter :: ID_PFLUX_DIR_CLEAR = 6
  integer, parameter :: ID_PFLUX_DIR_INTO_SUN = 7
  integer, parameter :: ID_PFLUX_UV = 8
  integer, parameter :: ID_PFLUX_PAR = 9
  integer, parameter :: ID_PFLUX_PAR_CLEAR = 10
  integer, parameter :: ID_PFLUX_SW_DN_TOA = 11
  integer, parameter :: ID_PEMIS_OUT = 12

  ! Constant and arrays for common variables (used in all extra output files) and their metadata
  integer, parameter :: N_COMMON_VAR = 9

  character*4, dimension(N_COMMON_VAR), parameter :: COMMON_VAR_NAMES = (/ &
   & 'X   ', &
   & 'Y   ', &
   & 'LON ', &
   & 'LAT ', &
   & 'YYYY', & 
   & 'MM  ', &
   & 'DD  ', &
   & 'HH  ', &
   & 'MIN ' /)

  ! 0 = x and y, 1 = x, 2 = y, 3 = time
  integer, dimension(N_COMMON_VAR), parameter :: COMMON_VAR_DIMS = (/ 1, 2, 0, 0, 3, 3, 3, 3, 3 /)

  character*42, dimension(N_COMMON_VAR), parameter :: COMMON_VAR_LONG_NAMES = (/ &
   & 'MAR longitude indices                     ', &
   & 'MAR latitude indices                      ', &
   & 'Longitude                                 ', &
   & 'Latitude                                  ', &
   & 'YYYY in HH:MIN DD/MM/YYYY (MAR time step) ', &
   & 'MM in HH:MIN DD/MM/YYYY (MAR time step)   ', &
   & 'DD in HH:MIN DD/MM/YYYY (MAR time step)   ', &
   & 'HH in HH:MIN DD/MM/YYYY (MAR time step)   ', &
   & 'MIN in HH:MIN DD/MM/YYYY (MAR time step)  ' /)
   
  ! J.-F. Grailet remark: ideally, there should also be arrays to provide units for each variable. 
  ! However, for each group, most variables have the same units except for one or two exceptions.

  ! Constants for managing a spectrum (i.e., describing a sequence of spectral bands)
  integer, parameter :: MAX_FILE_PATH_LENGTH = 500
  integer, parameter :: MAX_LINES_PER_FILE = 2000

  ! Non-constant variables to manage the (user-selected) SW spectral bands
  integer :: N_SW_SPECTRAL_BANDS = 0 ! 0 means no spectral decomposition; modified if requested
  real(kind=jprb), dimension(:), allocatable :: sw_spectral_bounds
  
  ! Logicals to remember whether or not SW spectral data for nocturnal time steps is saved too
  logical :: L_SW_SPECTRAL_PADDING = .true.
  logical :: L_SW_SPECTRAL_RAW_PADDING = .true.
  
  ! Matrix used to compute the custom spectral decomposition of surface solar downward fluxes 
  ! (created in the SETUP_RADIATION_SCHEME subroutine of the RADIATION_SETUP module)
  real(kind=jprb), allocatable :: sw_spectral_mapping(:,:)
  
  ! Logical to remember whether or not we output extra fluxes from ecRad
  logical :: L_EXTRA = .false.
  
  ! Global variable to remember the number of SW spectral bands used internally by ecRad
  integer :: N_ECRAD_SW_BANDS = 0 ! 0 means this type of output has not been requested
  
  ! Global logical telling whether or not only the hourly time steps (min = 0) are saved (any file)
  logical :: TSTEPS_HOURLY_ONLY = .true.
  
  ! Coding style remark (J.-F. Grailet): N_SW_SPECTRAL_BANDS, L_SW_SPECTRAL_PADDING, 
  ! L_SW_SPECTRAL_RAW_PADDING, L_EXTRA and N_ECRAD_SW_BANDS, TSTEPS_HOURLY_ONLY are in uppercase 
  ! to match with the naming conventions of the subroutines that use them (namely, RADIATION_SETUP 
  ! and PHYrad_ecRad).
  
  ! Non-constant variables to manage the additional output files
  character(len=MAX_FILE_PATH_LENGTH) :: sw_spectral_out_file
  character(len=MAX_FILE_PATH_LENGTH) :: sw_spectral_raw_out_file
  character(len=MAX_FILE_PATH_LENGTH) :: extra_out_file
  
  ! Next time step to store in the additional output files; note there are three such variables 
  ! and not just one in case this module would have to record any extra data at a different pace.
  integer :: extra_time_step
  integer :: sw_spectral_time_step
  integer :: sw_spectral_raw_time_step

contains

  ! Subroutine building the spectral bands given a range. The range is assumed to be described as 
  ! an array of 3 real values: lower bound, upper bound and band width (all in nm), in this order. 
  ! It is also assumed to have been previously verified by the MAR/ecRad configuration module.

  subroutine sw_spectral_bands_from_range(spectral_range)

    implicit none

    real(kind=jprb), intent(in), dimension(3) :: spectral_range
    integer :: i

    ! Initializes the variables regarding spectral bands
    N_SW_SPECTRAL_BANDS = int(spectral_range(2) - spectral_range(1)) / int(spectral_range(3))
    allocate(sw_spectral_bounds(N_SW_SPECTRAL_BANDS+1))

    sw_spectral_bounds(1) = spectral_range(1)
    do i=2, N_SW_SPECTRAL_BANDS+1
      sw_spectral_bounds(i) = sw_spectral_bounds(i-1) + spectral_range(3)
    end do

    ! Converts to meters
    do i=1, N_SW_SPECTRAL_BANDS+1
      sw_spectral_bounds(i) = sw_spectral_bounds(i) * 1e-9
    end do
  
  end subroutine sw_spectral_bands_from_range
  
  ! Subroutine building the spectral bands given a file providing the bounds (one per line). The 
  ! file is assumed to be properly formatted, as it has normally been previously verified by the 
  ! MAR/ecRad configuration module.
  ! 
  ! 08/02/2023: this subroutine will likely no longer be used in order to restrict the 
  ! configuration of the MAR-embedded ecRad to a single file. The subroutine is kept just in case 
  ! (also, is the basis for the subsequent sw_spectral_bands_from_array subroutine).
  
  subroutine sw_spectral_bands_from_file(spectrum_file)

    use radiation_io, only : nulerr, my_abort => radiation_abort

    implicit none

    character(len=MAX_FILE_PATH_LENGTH), intent(in) :: spectrum_file

    integer :: iosopen
    real(kind=jprb), dimension(MAX_LINES_PER_FILE+1) :: bounds
    integer :: nb_bounds
    integer :: i

    ! Opens the spectrum file and reads it line by line
    open(unit=10, iostat=iosopen, file=trim(spectrum_file))
    if (iosopen /= 0) then
      write(nulerr, '(a)') 'ecRad error: spectrum file "'//trim(spectrum_file)//'" is missing.'
      call my_abort('ecRad extra outputs error')
    end if

    nb_bounds = 0
    do
      read(10, '(f10.7)', iostat=iosopen) bounds(nb_bounds+1)
      if (iosopen /= 0) then
        exit
      end if
      nb_bounds = nb_bounds + 1
    end do

    close(unit=10)

    ! Initializes the variables regarding spectral bands
    N_SW_SPECTRAL_BANDS = nb_bounds - 1
    allocate(sw_spectral_bounds(N_SW_SPECTRAL_BANDS + 1))

    ! Also converts nm to m
    do i=1, N_SW_SPECTRAL_BANDS + 1
      sw_spectral_bounds(i) = bounds(i) * 1e-9
    end do
  
  end subroutine sw_spectral_bands_from_file
  
  ! Subroutine building the spectral bands from an array providing the complete list of spectral 
  ! bounds in ascending order. The array is assumed to be properly formatted, as it has normally 
  ! been previously verified by the MAR/ecRad configuration module.
  
  subroutine sw_spectral_bands_from_array(bounds)
  
    implicit none
    
    real(kind=jprb), intent(in), dimension(:) :: bounds
    integer :: nb_bounds, i
    
    nb_bounds = size(bounds, 1)
    N_SW_SPECTRAL_BANDS = nb_bounds - 1
    allocate(sw_spectral_bounds(N_SW_SPECTRAL_BANDS + 1))
    
    do i=1, N_SW_SPECTRAL_BANDS + 1
      sw_spectral_bounds(i) = bounds(i) * 1e-9 ! nm to m
    end do
  
  end subroutine sw_spectral_bands_from_array
  
  ! Subroutine meant to initialize the NetCDF file that will provide the spectral decomposition of 
  ! the solar surface downward fluxes at each time step ecRad was called. Appending the file 
  ! (i.e., writing spectral data for one time step) is done in a subsequent subroutine, which 
  ! relies on the global variable spectral_out_file to re-locate the file initialized here. 
  ! Because the dimensions of the MAR grid are required, and because users of such file needs 
  ! additional geographical data, the subroutine receives a few more parameters than just the path 
  ! to the new output file.
  
  subroutine create_sw_spectral_file(output_file_path, night_padding, dimLon, dimLat, & 
   &                                 longitudes, latitudes)
  
    use radiation_io, only : nulerr, my_abort => radiation_abort
    use easy_netcdf

    implicit none

    character(len=MAX_FILE_PATH_LENGTH), intent(in) :: output_file_path
    logical, intent(in) :: night_padding
    integer, intent(in) :: dimLon
    integer, intent(in) :: dimLat
    real(kind=jprb), intent(in) :: longitudes(:,:)
    real(kind=jprb), intent(in) :: latitudes(:,:)

    type(netcdf_file) :: out_file

    integer :: x_axis(dimLon)
    integer :: y_axis(dimLat)
    real(kind=jprb), allocatable :: low_bounds(:), up_bounds(:)

    integer :: i
    character(len=80) :: global_attr ! For a comment regarding night time

    if (.not. allocated(sw_spectral_bounds)) then
      write(nulerr, '(a)') 'ecRad error: attempt to write spectral bands without a definition.'
      call my_abort('ecRad extra outputs error')
    end if

    sw_spectral_out_file = output_file_path

    ! Dimensions
    call out_file%create(trim(output_file_path))
    call out_file%define_dimension('time', 0) ! Unlimited dimension
    call out_file%define_dimension('x', dimLon)
    call out_file%define_dimension('y', dimLat)
    call out_file%define_dimension('sw_band', N_SW_SPECTRAL_BANDS)
    
    ! Global attribute to remind the user night time values (null) are not stored to save space
    if (.not. night_padding) then
      write(global_attr, '(a,a)') 'Time steps without shortwave radiation (= night) were not ', &
       &                          'stored to save space.'
      call out_file%put_global_attribute('about_night_time', global_attr)
      L_SW_SPECTRAL_PADDING = .false. ! Prevents calling append in PHYrad_ecRad at night
    end if

    ! Defines the variables
    do i=1, N_COMMON_VAR
      if (COMMON_VAR_DIMS(i) == 0) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x', dim2_name='y')
      else if (COMMON_VAR_DIMS(i) == 1) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x')
      else if (COMMON_VAR_DIMS(i) == 2) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='y')
      else
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='time')
      end if
    end do
    
    call out_file%define_variable('LOW_BOUNDS', dim1_name='sw_band')
    call out_file%define_variable('UP_BOUNDS', dim1_name='sw_band')
    
    ! Spectral fluxes are provided in three forms: total, direct, diffuse (total - direct)
    call out_file%define_variable('PSW_SPECTRAL', dim1_name='x', dim2_name='y', & 
     &                                            dim3_name='sw_band', dim4_name='time')
    call out_file%define_variable('PSW_SPECTRAL_DIR', dim1_name='x', dim2_name='y', & 
     &                                                dim3_name='sw_band', dim4_name='time')

    ! Attributes to describe the variables
    do i = 1, N_COMMON_VAR
      call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'long_name', & 
       &                          trim(COMMON_VAR_LONG_NAMES(i)))
      if (i >= 3 .and. i < 5) then
        call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'units', 'degrees') ! LON/LAT units
      end if
    end do
    
    call out_file%put_attribute('LOW_BOUNDS', 'long_name', 'Lower bounds of the spectral bands')
    call out_file%put_attribute('LOW_BOUNDS', 'unit', 'm')
    call out_file%put_attribute('UP_BOUNDS', 'long_name', 'Upper bounds of the spectral bands')
    call out_file%put_attribute('UP_BOUNDS', 'unit', 'm')
    
    call out_file%put_attribute('PSW_SPECTRAL', 'long_name', & 
     &                          'Spectral bands for surface downward shortwave fluxes')
    call out_file%put_attribute('PSW_SPECTRAL', 'units', 'W/m2')
    call out_file%put_attribute('PSW_SPECTRAL_DIR', 'long_name', & 
     &                          'Spectral bands for direct surface downward shortwave fluxes')
    call out_file%put_attribute('PSW_SPECTRAL_DIR', 'units', 'W/m2')
    
    ! Prepares and stores dimension stuff
    do i=1, dimLon
      x_axis(i) = i
    end do

    do i=1, dimLat
      y_axis(i) = i
    end do

    allocate(low_bounds(N_SW_SPECTRAL_BANDS))
    allocate(up_bounds(N_SW_SPECTRAL_BANDS))

    do i=1, N_SW_SPECTRAL_BANDS
      low_bounds(i) = sw_spectral_bounds(i)
      up_bounds(i) = sw_spectral_bounds(i+1)
    end do

    call out_file%put_int_vector('X', x_axis)
    call out_file%put_int_vector('Y', y_axis)
    call out_file%put_real_matrix('LON', longitudes)
    call out_file%put_real_matrix('LAT', latitudes)
    call out_file%put_real_vector('LOW_BOUNDS', low_bounds)
    call out_file%put_real_vector('UP_BOUNDS', up_bounds)

    deallocate(low_bounds)
    deallocate(up_bounds)

    call out_file%close()

    sw_spectral_time_step = 1 ! Next time step to store is 1
  
  end subroutine create_sw_spectral_file
  
  ! Subroutine designed to append the NetCDF file that will provide the spectral decomposition of 
  ! the solar surface downward fluxes, previously created by create_sw_spectral_file. This 
  ! subroutine requires both the spectral fluxes computed at a given time step and the date/time 
  ! information of that time step, so users of the output file have everything they need w.r.t. 
  ! time and geography (previous subroutine also writing the longitude/latitude coordinates of 
  ! each grid point). The file to append is re-located thanks to the spectral_out_file variable 
  ! while sw_spectral_time_step provides how many time steps have been stored already. Note that 
  ! no diffuses fluxes are received nor stored, because they can be simply deduced by substracting 
  ! the direct fluxes from the total fluxes (spectral_fluxes - spectral_fluxes_direct).
  
  subroutine append_sw_spectral_file(spectral_fluxes, spectral_fluxes_direct, & 
   &                                 yyyy, mm, dd, seconds_cur_day)

    use radiation_io, only : nulerr, my_abort => radiation_abort
    use easy_netcdf

    implicit none

    real(kind=jprb), intent(in) :: spectral_fluxes(:,:,:)
    real(kind=jprb), intent(in) :: spectral_fluxes_direct(:,:,:)
    integer, intent(in) :: yyyy
    integer, intent(in) :: mm
    integer, intent(in) :: dd
    integer, intent(in) :: seconds_cur_day ! irhhss in PHYrad_ecRad

    type(netcdf_file) :: out_file
    integer :: hours
    integer :: minutes

    if (.not. allocated(sw_spectral_bounds)) then
      write(nulerr, '(a)') 'ecRad error: attempt to write spectral bands without a definition.'
      call my_abort('ecRad extra outputs error')
    end if

    hours = (seconds_cur_day - modulo(seconds_cur_day, 3600)) / 3600
    minutes = seconds_cur_day - hours * 3600
    minutes = (minutes - modulo(minutes, 60)) / 60
    
    if (TSTEPS_HOURLY_ONLY .and. minutes > 0) then
      return
    end if
    
    call out_file%open_for_append(sw_spectral_out_file, iverbose=1)

    call out_file%put_int_scalar_indexed('YYYY', sw_spectral_time_step, yyyy)
    call out_file%put_int_scalar_indexed('MM', sw_spectral_time_step, mm)
    call out_file%put_int_scalar_indexed('DD', sw_spectral_time_step, dd)
    call out_file%put_int_scalar_indexed('HH', sw_spectral_time_step, hours)
    call out_file%put_int_scalar_indexed('MIN', sw_spectral_time_step, minutes)
    call out_file%put_real_array3_indexed('PSW_SPECTRAL', spectral_fluxes, sw_spectral_time_step)
    call out_file%put_real_array3_indexed('PSW_SPECTRAL_DIR', spectral_fluxes_direct, & 
     &                                    sw_spectral_time_step)

    call out_file%close()

    sw_spectral_time_step = sw_spectral_time_step+1

  end subroutine append_sw_spectral_file
  
  ! Subroutine that creates the NetCDF file with the extra surface/TOA fluxes produced by ecRad. 
  ! This subroutine is very similar to create_sw_spectral_file and differs mostly by the fact that 
  ! it does not require additional configuration by the user (hence the slightly simpler code). 
  ! Just like create_sw_spectral_file, it must be called at the time ecRad is being configured in 
  ! the MAR, and an additional subroutine is responsible for appending the additional output file.
  
  subroutine create_extra_fluxes_file(output_file_path, dimLon, dimLat, longitudes, latitudes)
  
    use easy_netcdf

    implicit none

    character(len=MAX_FILE_PATH_LENGTH), intent(in) :: output_file_path
    integer, intent(in) :: dimLon
    integer, intent(in) :: dimLat
    real(kind=jprb), intent(in) :: longitudes(:,:)
    real(kind=jprb), intent(in) :: latitudes(:,:)

    type(netcdf_file) :: out_file

    integer :: x_axis(dimLon)
    integer :: y_axis(dimLat)

    integer :: i

    extra_out_file = output_file_path

    ! Dimensions
    call out_file%create(trim(output_file_path))
    call out_file%define_dimension('time', 0) ! Unlimited dimension
    call out_file%define_dimension('x', dimLon)
    call out_file%define_dimension('y', dimLat)

    ! Defines the variables
    do i=1, N_COMMON_VAR
      if (COMMON_VAR_DIMS(i) == 0) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x', dim2_name='y')
      else if (COMMON_VAR_DIMS(i) == 1) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x')
      else if (COMMON_VAR_DIMS(i) == 2) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='y')
      else
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='time')
      end if
    end do

    do i = 1, N_EXTRA_OUTPUTS
      call out_file%define_variable(trim(EXTRA_VAR_NAMES(i)), dim1_name='x', dim2_name='y', & 
       &                                                      dim3_name='time')
    end do
    
    ! Attributes to describe the variables
    do i = 1, N_COMMON_VAR
      call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'long_name', & 
       &                          trim(COMMON_VAR_LONG_NAMES(i)))
      if (i >= 3 .and. i < 5) then
        call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'units', 'degrees') ! LON/LAT units
      end if
    end do
    
    do i = 1, N_EXTRA_OUTPUTS
      call out_file%put_attribute(trim(EXTRA_VAR_NAMES(i)), 'long_name', & 
       &                          trim(EXTRA_VAR_LONG_NAMES(i)))
      if (i < N_EXTRA_OUTPUTS) then
        call out_file%put_attribute(trim(EXTRA_VAR_NAMES(i)), 'units', 'W/m2')
      else
        call out_file%put_attribute(trim(EXTRA_VAR_NAMES(i)), 'units', '-') ! Emissivity (no dim)
      end if
    end do

    ! Prepares and stores dimension stuff
    do i=1, dimLon
      x_axis(i) = i
    end do

    do i=1, dimLat
      y_axis(i) = i
    end do

    call out_file%put_int_vector('X', x_axis)
    call out_file%put_int_vector('Y', y_axis)
    call out_file%put_real_matrix('LON', longitudes)
    call out_file%put_real_matrix('LAT', latitudes)

    call out_file%close()

    L_EXTRA = .true.
    extra_time_step = 1 ! Next time step to store is 1
  
  end subroutine create_extra_fluxes_file
  
  ! Subroutine for appending the output file providing the extra surface/TOA fluxes produced by 
  ! ecRad. It works in the same way as append_sw_spectral_file, although simpler.
  
  subroutine append_extra_fluxes_file(extra_fluxes, yyyy, mm, dd, seconds_cur_day)

    use easy_netcdf

    implicit none

    real(kind=jprb), intent(in) :: extra_fluxes(:,:,:)
    integer, intent(in) :: yyyy
    integer, intent(in) :: mm
    integer, intent(in) :: dd
    integer, intent(in) :: seconds_cur_day ! irhhss in PHYrad_ecRad

    type(netcdf_file) :: out_file
    integer :: hours
    integer :: minutes
    integer :: i

    hours = (seconds_cur_day - modulo(seconds_cur_day, 3600)) / 3600
    minutes = seconds_cur_day - hours * 3600
    minutes = (minutes - modulo(minutes, 60)) / 60
    
    if (TSTEPS_HOURLY_ONLY .and. minutes > 0) then
      return
    end if

    call out_file%open_for_append(extra_out_file, iverbose=1)

    call out_file%put_int_scalar_indexed('YYYY', extra_time_step, yyyy)
    call out_file%put_int_scalar_indexed('MM', extra_time_step, mm)
    call out_file%put_int_scalar_indexed('DD', extra_time_step, dd)
    call out_file%put_int_scalar_indexed('HH', extra_time_step, hours)
    call out_file%put_int_scalar_indexed('MIN', extra_time_step, minutes)
    do i=1, N_EXTRA_OUTPUTS
      call out_file%put_real_matrix_indexed(trim(EXTRA_VAR_NAMES(i)), extra_fluxes(:,:,i), & 
       &                                    extra_time_step)
    end do

    call out_file%close()

    extra_time_step = extra_time_step+1

  end subroutine append_extra_fluxes_file
  
  ! Additional subroutine allowing a user to output the spectral shortwave data used internally by 
  ! ecRad, i.e., without the additional operations that make the fluxes match the spectral bands 
  ! defined by the user. This subroutine was added mainly for debugging after discussing problems 
  ! with spectral data produced by the MAR with R. Hogan in December 2022. In addition to the same 
  ! parameters as create_sw_spectral_file, this subroutine requires the spectral definition for 
  ! shortwave used internally by ecRad and needs to know whether the fluxes are re-arranged in 
  ! bands if using g-points (with ecCKD). As such, it SHOULD ONLY be called after the ecRad setup 
  ! to ensure the spectral definition object is available. Note also that the output NetCDF file 
  ! does not store the bounds of the spectral shortwave bands but provides instead a global 
  ! attribute reminding what gas model was used during the simulation, for simplicity's sake.
  
  subroutine create_sw_spectral_raw_file(output_file_path, night_padding, dimLon, dimLat, & 
   &                                     longitudes, latitudes, spectral_def, use_bands)
  
    use radiation_spectral_definition
    use easy_netcdf

    implicit none

    ! Main parameters for creating the NetCDF file
    character(len=MAX_FILE_PATH_LENGTH), intent(in) :: output_file_path
    logical, intent(in) :: night_padding
    integer, intent(in) :: dimLon
    integer, intent(in) :: dimLat
    real(kind=jprb), intent(in) :: longitudes(:,:)
    real(kind=jprb), intent(in) :: latitudes(:,:)

    ! Additional info regarding the spectral definition
    type(spectral_definition_type), intent(in) :: spectral_def
    logical, intent(in) :: use_bands ! Should be the negation of do_cloud_aerosol_per_sw_g_point

    type(netcdf_file) :: out_file

    integer :: x_axis(dimLon)
    integer :: y_axis(dimLat)

    integer :: i
    character(len=80) :: global_attr ! For comments regarding ecCKD and night time

    sw_spectral_raw_out_file = output_file_path

    ! Dimensions
    call out_file%create(trim(output_file_path))
    call out_file%define_dimension('time', 0) ! Unlimited dimension
    call out_file%define_dimension('x', dimLon)
    call out_file%define_dimension('y', dimLat)
    
    ! Number of bands stored in raw fluxes depend on the spectral definition. For now, it is 
    ! assumed there are only RRTM-G and ecCKD as gas models, the latter allowing to output fluxes 
    ! per g-point or per band (consisting in rearranged g-points). WARNING ! This MUST be edited 
    ! if more gas models were added in the future !

    if (spectral_def%ng > 0) then
      if (use_bands) then
        write(global_attr, '(a,i2,a,i2,a)') 'ecCKD with ', spectral_def%ng, &
         &                                  ' g-points (shortwave) rearranged as ', &
         &                                  spectral_def%nband, ' bands'
        call out_file%define_dimension('sw_band', spectral_def%nband)
        call out_file%put_global_attribute('gas_model', trim(global_attr))
        N_ECRAD_SW_BANDS = spectral_def%nband
      else
        write(global_attr, '(a,i2,a)') 'ecCKD with ', spectral_def%ng, ' g-points (shortwave)'
        call out_file%define_dimension('sw_band', spectral_def%ng)
        call out_file%put_global_attribute('gas_model', trim(global_attr))
        N_ECRAD_SW_BANDS = spectral_def%ng
      end if
    else
      call out_file%define_dimension('sw_band', spectral_def%nband)
      call out_file%put_global_attribute('gas_model', 'RRTM-G (14 bands for shortwave)')
      N_ECRAD_SW_BANDS = spectral_def%nband
    end if
    
    ! Global attribute to remind the user night time values (null) are not stored to save space
    if (.not. night_padding) then
      write(global_attr, '(a,a)') 'Time steps without shortwave radiation (= night) were not ', &
       &                          'stored to save space.'
      call out_file%put_global_attribute('about_night_time', global_attr)
      L_SW_SPECTRAL_RAW_PADDING = .false. ! Prevents appending in PHYrad_ecRad at night time
    end if

    ! Defines the variables
    do i=1, N_COMMON_VAR
      if (COMMON_VAR_DIMS(i) == 0) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x', dim2_name='y')
      else if (COMMON_VAR_DIMS(i) == 1) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='x')
      else if (COMMON_VAR_DIMS(i) == 2) then
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='y')
      else
        call out_file%define_variable(trim(COMMON_VAR_NAMES(i)), dim1_name='time')
      end if
    end do
    
    ! Spectral fluxes are provided in three forms: total, direct, diffuse (total - direct)
    call out_file%define_variable('PSW_SPECTRAL_RAW', dim1_name='x', dim2_name='y', & 
     &                                                dim3_name='sw_band', dim4_name='time')
    call out_file%define_variable('PSW_SPECTRAL_RAW_DIR', dim1_name='x', dim2_name='y', & 
     &                                                    dim3_name='sw_band', dim4_name='time')

    ! Attributes to describe the variables
    do i = 1, N_COMMON_VAR
      call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'long_name', & 
       &                          trim(COMMON_VAR_LONG_NAMES(i)))
      if (i >= 3 .and. i < 5) then
        call out_file%put_attribute(trim(COMMON_VAR_NAMES(i)), 'units', 'degrees') ! LON/LAT units
      end if
    end do
    
    call out_file%put_attribute('PSW_SPECTRAL_RAW', 'long_name', & 
     &                          'ecRad spectral data for surface downward shortwave fluxes')
    call out_file%put_attribute('PSW_SPECTRAL_RAW', 'units', 'W/m2')
    call out_file%put_attribute('PSW_SPECTRAL_RAW_DIR', 'long_name', & 
     &                          'ecRad spectral data for direct surface downward shortwave fluxes')
    call out_file%put_attribute('PSW_SPECTRAL_RAW_DIR', 'units', 'W/m2')
    
    ! Prepares and stores dimension stuff
    do i=1, dimLon
      x_axis(i) = i
    end do

    do i=1, dimLat
      y_axis(i) = i
    end do

    call out_file%put_int_vector('X', x_axis)
    call out_file%put_int_vector('Y', y_axis)
    call out_file%put_real_matrix('LON', longitudes)
    call out_file%put_real_matrix('LAT', latitudes)

    call out_file%close()

    sw_spectral_raw_time_step = 1 ! Next time step to store is 1
  
  end subroutine create_sw_spectral_raw_file
  
  ! Subroutine for appending the NetCDF file that provides the spectral shortwave data used 
  ! internally by ecRad. It is almost identical to append_sw_spectral_file, excepts that it does 
  ! not check whether custom spectral bands exist since it does not need them.
  
  subroutine append_sw_spectral_raw_file(spectral_fluxes, spectral_fluxes_direct, & 
   &                                     yyyy, mm, dd, seconds_cur_day)

    use easy_netcdf

    implicit none

    real(kind=jprb), intent(in) :: spectral_fluxes(:,:,:)
    real(kind=jprb), intent(in) :: spectral_fluxes_direct(:,:,:)
    integer, intent(in) :: yyyy
    integer, intent(in) :: mm
    integer, intent(in) :: dd
    integer, intent(in) :: seconds_cur_day ! irhhss in PHYrad_ecRad

    type(netcdf_file) :: out_file
    integer :: hours
    integer :: minutes

    hours = (seconds_cur_day - modulo(seconds_cur_day, 3600)) / 3600
    minutes = seconds_cur_day - hours * 3600
    minutes = (minutes - modulo(minutes, 60)) / 60
    
    if (TSTEPS_HOURLY_ONLY .and. minutes > 0) then
      return
    end if

    call out_file%open_for_append(sw_spectral_raw_out_file, iverbose=1)

    call out_file%put_int_scalar_indexed('YYYY', sw_spectral_raw_time_step, yyyy)
    call out_file%put_int_scalar_indexed('MM', sw_spectral_raw_time_step, mm)
    call out_file%put_int_scalar_indexed('DD', sw_spectral_raw_time_step, dd)
    call out_file%put_int_scalar_indexed('HH', sw_spectral_raw_time_step, hours)
    call out_file%put_int_scalar_indexed('MIN', sw_spectral_raw_time_step, minutes)
    call out_file%put_real_array3_indexed('PSW_SPECTRAL_RAW', spectral_fluxes, & 
     &                                    sw_spectral_raw_time_step)
    call out_file%put_real_array3_indexed('PSW_SPECTRAL_RAW_DIR', spectral_fluxes_direct, & 
     &                                    sw_spectral_raw_time_step)
    call out_file%close()

    sw_spectral_raw_time_step = sw_spectral_raw_time_step+1

  end subroutine append_sw_spectral_raw_file

end module extra_ecRad_outputs
