SUBROUTINE CLOUD_OVERLAP_DECORR_LEN &
     & (KIDIA, KFDIA, KLON, PGEMU, NDECOLAT, &
     &  PDECORR_LEN_EDGES_KM, PDECORR_LEN_WATER_KM, PDECORR_LEN_RATIO)

! CLOUD_OVERLAP_DECORR_LEN
!
! (C) Copyright 2016- ECMWF.
!
! This software is licensed under the terms of the Apache Licence Version 2.0
! which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
!
! In applying this licence, ECMWF does not waive the privileges and immunities
! granted to it by virtue of its status as an intergovernmental organisation
! nor does it submit to any jurisdiction.
!
! PURPOSE
! -------
!   Calculate the cloud overlap decorrelation length as a function of
!   latitude for use in the radiation scheme
!
! INTERFACE
! ---------
!   CLOUD_OVERLAP_DECORR_LEN is called from RADIATION_SCHEME
!
! AUTHOR
! ------
!   Robin Hogan, ECMWF (using code extracted from radlswr.F90)
!   Original: 2016-02-16
!
! MODIFICATIONS
! -------------
! 
! Modifications for the inclusion in the MAR model by J.-F. Grailet (09/2022):
! -removed all instructions related to the Dr. Hook library.
! -included RDECORR_CF and RDECORR_CW via the additional module INCMAR, as the 
!  ecRad 1.5.0 archive did not provide the YOECLD module and its YRECLD 
!  derived type.
! 
! Additional modifications on 20/04/2023:
! -renamed RDECORR_CF and RDECORR_CW as RCLOUD_DECORR_CF and RCLOUD_DECORR_CW.
! -added RCLOUD_DECORR_LEN_RATIO (new INCMAR variable) to control the ratio of 
!  decorrelation length for cloud water content inhomogeneities and cloud 
!  boundaries (initially hard-coded as 0.5, a default value).
! 
! -------------------------------------------------------------------

USE PARKIND1 , ONLY : JPIM, JPRB
USE YOMCST   , ONLY : RPI
USE INCMAR   , ONLY : RCLOUD_DECORR_CF, RCLOUD_DECORR_CW, & 
 &                    RCLOUD_DECORR_LEN_RATIO

! -------------------------------------------------------------------

IMPLICIT NONE

! INPUT ARGUMENTS

! *** Array dimensions and ranges
INTEGER(KIND=JPIM),INTENT(IN) :: KIDIA    ! Start column to process
INTEGER(KIND=JPIM),INTENT(IN) :: KFDIA    ! End column to process
INTEGER(KIND=JPIM),INTENT(IN) :: KLON     ! Number of columns

! *** Configuration variable controlling the overlap scheme
INTEGER(KIND=JPIM),INTENT(IN) :: NDECOLAT

! *** Single-level variables 
REAL(KIND=JPRB),   INTENT(IN) :: PGEMU(KLON) ! Sine of latitude

! OUTPUT ARGUMENTS

! *** Decorrelation lengths for cloud edges and cloud water content,
! *** in km
REAL(KIND=JPRB), INTENT(OUT)           :: PDECORR_LEN_EDGES_KM(KLON)
REAL(KIND=JPRB), INTENT(OUT), OPTIONAL :: PDECORR_LEN_WATER_KM(KLON)
  
! Ratio of water-content to cloud-edge decorrelation lengths
REAL(KIND=JPRB), INTENT(OUT), OPTIONAL :: PDECORR_LEN_RATIO

! LOCAL VARIABLES

REAL(KIND=JPRB) :: ZRADIANS_TO_DEGREES, ZABS_LAT_DEG, ZCOS_LAT

INTEGER(KIND=JPIM) :: JL

! -------------------------------------------------------------------

IF (NDECOLAT == 0) THEN

  ! Decorrelation lengths are constant values
  PDECORR_LEN_EDGES_KM(KIDIA:KFDIA) = RCLOUD_DECORR_CF
  IF (PRESENT(PDECORR_LEN_WATER_KM)) THEN
    PDECORR_LEN_WATER_KM(KIDIA:KFDIA) = RCLOUD_DECORR_CW
  ENDIF
  IF (PRESENT(PDECORR_LEN_RATIO)) THEN
    PDECORR_LEN_RATIO = RCLOUD_DECORR_CW / RCLOUD_DECORR_CF
  ENDIF

ELSE

  ZRADIANS_TO_DEGREES = 180.0_JPRB / RPI

  IF (NDECOLAT == 1) THEN
    ! Shonk et al. (2010) Eq. 13 formula
    DO JL = KIDIA,KFDIA
      ZABS_LAT_DEG = ABS(ASIN(PGEMU(JL)) * ZRADIANS_TO_DEGREES)
      PDECORR_LEN_EDGES_KM(JL) = 2.899_JPRB - 0.02759_JPRB * ZABS_LAT_DEG
    ENDDO
  ELSE ! NDECOLAT == 2
    DO JL = KIDIA,KFDIA
      ! Shonk et al. (2010) but smoothed over the equator
      ZCOS_LAT = COS(ASIN(PGEMU(JL)))
      PDECORR_LEN_EDGES_KM(JL) = 0.75_JPRB + 2.149_JPRB * ZCOS_LAT*ZCOS_LAT
    ENDDO
  ENDIF

  ! Both NDECOLAT = 1 and 2 assume that the decorrelation length for
  ! cloud water content is half that for cloud edges
  IF (PRESENT(PDECORR_LEN_WATER_KM)) THEN
    PDECORR_LEN_WATER_KM(KIDIA:KFDIA) = PDECORR_LEN_EDGES_KM(KIDIA:KFDIA) &
     &                                  * RCLOUD_DECORR_LEN_RATIO
  ENDIF

  IF (PRESENT(PDECORR_LEN_RATIO)) THEN
    PDECORR_LEN_RATIO = RCLOUD_DECORR_LEN_RATIO
  ENDIF

ENDIF

! -------------------------------------------------------------------

END SUBROUTINE CLOUD_OVERLAP_DECORR_LEN
