! MAR_ecRad_config.F90
! 
!   Created on: 20/10/2022
!       Author: J.-F. Grailet (jefgrailet)
! 
! This module allows one to read a configuration file (extension .nam) dedicated to the parameters 
! used by ecRad embedded in the MAR model. Its goal is to let MAR users adjust the configuration 
! of ecRad to their needs, e.g., which equation solver is used, or how the effective radius of ice 
! or water particles is handled. It is, for the most part, derived from the ecrad_driver_config 
! module found in the offline driver for ecRad as provided in the ecRad 1.5.0 archive (written by 
! Robin Hogan).
! 
! It is worth noting the radiation_config module (in radiation/ sub-folder) already provides a 
! subroutine for reading a configuration from a (namelist) file. However, on the one hand, it is 
! far more detailed and complex, and on the other hand, this specific module adds options that 
! control parameters and features that are not from ecRad itself.
! 
! For instance, it adds parameters for the management of the greenhouse gases/aerosols forcings 
! that do not exist in the radiation_config module, and starting from Spring 2023, it also allows 
! MAR users to choose and tune the parametrization used to compute cloud fraction values sent to 
! ecRad. Moreover, it also provides options to write additional output files that provide 
! completementary or control data. There are nevertheless similarities between both, and 
! additional parameters for ecRad itself may be added in the future (see ecRad documentation).
! 
! Starting from Spring 2023, this module also provides a subroutine to enforce a default 
! configuration for MAR users to be able to run a simulation without having to review or fine-tune 
! the configuration of ecRad or the code interfacing it with MAR.

module MAR_ecRad_config

  use parkind1, only : jprb

  implicit none

  public

  ! Max lengths of: a file path, a string value, a range, a namelist array, etc.
  integer, parameter :: FilePathMaxLength = 500
  integer, parameter :: FileNameMaxLength = 100 ! For the NetCDF files used by ecCKD
  integer, parameter :: StringVarNameLength = 40
  integer, parameter :: StringValMaxLength = 20 ! Largest expected string is 19 characters
  integer, parameter :: NbValuesInRange = 3
  integer, parameter :: MaxNamelistArraySize = 501 ! +1 for 500 spectral bands (arbitrary)
  integer, parameter :: MaxLinesPerFile = 2000 ! Arbitrary constant for reading files
  
  ! Potential string values expected for various parameters
  character(len=StringValMaxLength), dimension(2), parameter :: smoothingMethods &
   & = (/ 'None                ', 'Horizontal average  ' /)
  character(len=StringValMaxLength), dimension(2), parameter :: aerosolClimatologies & 
   & = (/ 'Tegen               ', 'MACC                ' /)
  character(len=StringValMaxLength), dimension(5), parameter :: cloudParametrizations &
   & = (/ 'MAR                 ', 'ECMWF crude         ', 'ECMWF               ',  & 
   &      'Sundqvist           ', 'Xu and Randall      ' /)
  character(len=StringValMaxLength), dimension(2), parameter :: cloudOpticsIceSchemes & 
   & = (/ 'Fu                  ', 'Baran               ' /)
  character(len=StringValMaxLength), dimension(2), parameter :: cloudOpticsWatSchemes & 
   & = (/ 'Slingo              ', 'SOCRATES            ' /)
  character(len=StringValMaxLength), dimension(3), parameter :: decorrelationSchemes &
   & = (/ 'Independent         ', 'Shonk-Hogan         ', 'Shonk-Hogan-I       ' /)
  character(len=StringValMaxLength), dimension(2), parameter :: gasModels &
   & = (/ 'RRTM-G              ', 'ecCKD               ' /)
  character(len=StringValMaxLength), dimension(4), parameter :: particlesIceRadiusSchemes &
   & = (/ 'Fixed               ', 'Liou-Ou             ', 'Liou-Ou Jacob-Klein ', &
   &      'Sun-Rikus           ' /)
  character(len=StringValMaxLength), dimension(3), parameter :: particlesLiqRadiusSchemes &
   & = (/ 'Pressure            ', 'Fixed               ', 'Martin              ' /)
  ! N.B.: available solvers are the same for both shortwave and longwave
  character(len=StringValMaxLength), dimension(4), parameter :: availableSolvers & 
   & = (/ 'McICA               ', 'SPARTACUS           ', 'SPARTACUS-3D        ', & 
   &      'Tripleclouds        ' /)
  
  ! Additionnal array listing all string parameters
  character(len=StringVarNameLength), dimension(11), parameter :: varNames &
   & = (/ 'interpolation_smoothing_method          ', &
   &      'aerosol_climatology                     ', & 
   &      'cloud_fraction_parametrization_name     ', & 
   &      'cloud_optics_ice_scheme                 ', &
   &      'cloud_optics_water_scheme               ', & 
   &      'decorrelation_scheme                    ', & 
   &      'gas_model_name                          ', &
   &      'particles_ice_radius_scheme             ', & 
   &      'particles_liquid_radius_scheme          ', &
   &      'solvers_longwave_name                   ', &
   &      'solvers_shortwave_name                  ' /)

  ! Absolute values for the upper limits of blanket layers (stratosphere) in hPa
  real(kind=jprb), parameter :: LOWER_BLANKET_LIMIT = 50.0
  real(kind=jprb), parameter :: UPPER_BLANKET_LIMIT = 1.0 ! Stratopause

  ! Absolute bounds for solar spectral bands (nm)
  real(kind=jprb), parameter :: LOWER_SW_BOUND = 115.5 ! Based on Coddington et al. in BAMS 2016
  real(kind=jprb), parameter :: UPPER_SW_BOUND = 5000.0 ! Arbitrary (but read comment below)
  
  ! Important remark by J.-F. Grailet after some exchanges with R. Hogan (03/02/2022): spectral 
  ! data above 3333.33 nm (i.e., wavenumber lower than 3000 cm^-1) will be incomplete as thermal 
  ! radiation from the atmosphere becomes non-negligible starting from that wavelength. Any use of 
  ! spectral bands beyond 3333.33nm and the absolute upper bound should be used with caution.
  
  ! Minimum and maximum value for any scaling factor (dimensionless)
  real(kind=jprb), parameter :: MIN_FACTOR = 0.8
  real(kind=jprb), parameter :: MAX_FACTOR = 1.2

  type ecRad_config_type
  
    ! A few remarks:
    ! 1) The parameters are listed with the same ordering conventions as the sample .nam file, for 
    !    readability's sake. This also applies to the rest of the module.
    ! 2) All integer parameters in the "radiation" section are selected with the help of strings 
    !    upon reading the configuration file. They are translated by this module into integer 
    !    values for simplicity's sake (and also because these parameters are defined as such in 
    !    the sample IFS code).
    
    ! General parameters
    integer :: columns_per_cpu
    integer :: greenhouse_period_start
    integer :: greenhouse_period_end
    integer :: interpolation_smoothing_method
    character(len=FilePathMaxLength) :: path_radiation_data
    character(len=FilePathMaxLength) :: path_greenhouse_data
    character(len=FilePathMaxLength) :: path_aerosols_data
    character(len=FilePathMaxLength) :: path_cmip_scenarios
    integer :: verbosity
    
    ! Radiation parameters (in six sub-groups; see text file detailing them)
    integer :: aerosol_climatology
    logical :: aerosol_longwave_scattering
    
    real(kind=jprb) :: cloud_fraction_alpha
    real(kind=jprb) :: cloud_fraction_gamma
    real(kind=jprb) :: cloud_fraction_p
    integer :: cloud_fraction_parametrization
    logical :: cloud_longwave_scattering
    integer :: cloud_optics_ice_scheme
    integer :: cloud_optics_water_scheme
    real(kind=jprb) :: cloud_water_fractional_std
    
    real(kind=jprb) :: decorrelation_depth_cloud
    real(kind=jprb) :: decorrelation_depth_condensate
    integer :: decorrelation_depth_scheme
    real(kind=jprb) :: decorrelation_length_ratio
    logical :: decorrelation_use_beta_overlap
    
    integer :: gas_model ! _name in namelist file
    character(len=FileNameMaxLength) :: gas_optics_lw_file
    character(len=FileNameMaxLength) :: gas_optics_sw_file
    
    real(kind=jprb) :: particles_ccn_concentration_ocean
    real(kind=jprb) :: particles_ccn_concentration_land
    real(kind=jprb) :: particles_conversion_radius_size
    logical :: particles_ice_latitude_dependence
    real(kind=jprb) :: particles_ice_minimum_diameter
    integer :: particles_ice_radius_scheme
    integer :: particles_liquid_radius_scheme
    
    integer :: solvers_longwave ! _name in namelist file
    integer :: solvers_shortwave ! _name in namelist file
    
    logical :: upper_blanket_layers ! Tells if blanket layers are used
    real(kind=jprb), dimension(:), allocatable :: upper_blanket_limits
    real(kind=jprb) :: upper_solar_scaling
    
    real(kind=jprb) :: water_species_scaling_vapor
    real(kind=jprb) :: water_species_scaling_ice
    real(kind=jprb) :: water_species_scaling_liquid
    real(kind=jprb) :: water_species_scaling_rain
    real(kind=jprb) :: water_species_scaling_snow
    
    ! Additional parameters
    logical :: additional_fluxes_output
    character(len=FilePathMaxLength) :: additional_fluxes_filename ! Filename, but can be a path
    logical :: climatologies_interpolated_output
    character(len=FilePathMaxLength) :: climatologies_interpolated_filename ! Ditto
    logical :: climatologies_scaled_output
    character(len=FilePathMaxLength) :: climatologies_scaled_filename ! Ditto
    logical :: solar_spectral_output
    character(len=FilePathMaxLength) :: solar_spectral_filename
    logical :: solar_spectral_bands_regular
    real(kind=jprb), dimension(NbValuesInRange) :: solar_spectral_bands_range ! Range description
    real(kind=jprb), dimension(:), allocatable :: solar_spectral_bands_full
    
    logical :: solar_spectral_padding
    logical :: solar_spectral_raw_output
    character(len=FilePathMaxLength) :: solar_spectral_raw_filename
    logical :: solar_spectral_raw_padding
    logical :: time_steps_hourly_only
    
    ! Remark: spectral bands are built in the extra_ecRad_outputs module.
  
  contains
  
   procedure :: read => read_config_from_namelist
   procedure :: default_config
  
  end type ecRad_config_type

contains
  
  ! Simple subroutine to check that a string appears in a short list of possible strings. It 
  ! prevents redundant code within the read_config_from_namelist subroutine.
  
  subroutine check_string(param_name, param_val, possible_vals, int_result)
    
    use radiation_io, only : nulerr, my_abort => radiation_abort
  
    implicit none

    character(len=StringVarNameLength), intent(in) :: param_name
    character(len=StringValMaxLength), intent(in) :: param_val
    character(len=StringValMaxLength), intent(in) :: possible_vals(:)
    integer, intent(out) :: int_result
    
    integer :: i
    logical :: stringOK
    
    stringOK = .false.
    
    ! Checks the value
    do i=1, size(possible_vals)
      if (.not.stringOK .and. (trim(param_val) == trim(possible_vals(i)))) then
        stringOK = .true.
        int_result = i
      end if
    end do
    
    ! Prints an error message and aborts if the value does not match known ones
    if (.not.stringOK) then
      write(nulerr, '(a,a)') 'ecRad config error: unknown value "'//trim(param_val)//'" ', &
       &                     'for '//trim(param_name)//'. Possible values are:'
      do i=1, size(possible_vals)
        write(nulerr, '(a,a,a)') '-"', trim(possible_vals(i)), '"'
      end do
      call my_abort('ecRad configuration error')
    end if
  
  end subroutine check_string
  
  ! Simple subroutine to check a real value is within the [MIN_COEF, MAX_COEF] interval. It 
  ! prevents redundant code within the read_config_from_namelist subroutine.
  
  subroutine check_scaling_factor(variable_name, scaling_factor)
  
    use radiation_io, only : nulerr, my_abort => radiation_abort
  
    implicit none

    character(len=*), intent(in) :: variable_name
    real(kind=jprb), intent(in) :: scaling_factor
    
    character(len=3) :: lower_str, upper_str
    
    if (scaling_factor < MIN_FACTOR .or. scaling_factor > MAX_FACTOR) then
      write(lower_str, '(f3.1)') MIN_FACTOR
      write(upper_str, '(f3.1)') MAX_FACTOR
      write(nulerr, '(a,a,a)') 'ecRad config error: scaling factor for '//trim(variable_name), &
       &                       ' mixing ratio must be included in ', &
       &                       '['//trim(lower_str)//', '//trim(upper_str)//'].'
      call my_abort('ecRad configuration error')
    end if
  
  end subroutine check_scaling_factor
  
  ! Simple subroutine to copy a fixed-size array from MAR (coming from the mar_ecrad_mod module) 
  ! into an allocatable array that is part of the ecRad_config_type derived type.
  
  subroutine copy_MAR_param_array(param_array, config_array)
  
    implicit none
    
    real(kind=jprb), intent(in) :: param_array(:)
    real(kind=jprb), intent(inout), allocatable :: config_array(:)
    
    integer :: size_param_array, i
    
    ! Arrays from ecrad_config_type may be already allocated with default values
    if (allocated(config_array)) then
      deallocate(config_array)
    end if
    
    size_param_array = size(param_array, 1)
    allocate(config_array(size_param_array))
    do i=1, size_param_array
      config_array(i) = param_array(i)
    end do
  
  end subroutine copy_MAR_param_array
  
  ! Simple subroutine to copy an array read from a namelist, which has an upper limit in size, 
  ! into an allocatable array that is part of the ecRad_config_type derived type. Before copying 
  ! the values, the subroutine first determines how many useful values there are, assuming all 
  ! useful values are non-zero.
  
  subroutine copy_namelist_array(namelist_array, config_array)
  
    implicit none
    
    real(kind=jprb), intent(in), dimension(MaxNamelistArraySize) :: namelist_array
    real(kind=jprb), intent(inout), dimension(:), allocatable :: config_array
    
    integer :: size_namelist_array, i
    
    size_namelist_array = 0
    do i=1, MaxNamelistArraySize
      if (namelist_array(i) == 0.) then
        exit
      end if
      size_namelist_array = size_namelist_array + 1
    end do
    
    allocate(config_array(size_namelist_array))
    do i=1, size_namelist_array
      config_array(i) = namelist_array(i)
    end do
  
  end subroutine copy_namelist_array
  
  ! Simple function checking that a description of stratospheric "blanket" levels is valid, this 
  ! description consisting in the upper limits of said levels, provided in ascending order. The 
  ! top level (first in the description) must have its upper limit at the stratopause (1 hPa).
  
  function is_valid_blanket(upper_limits)

    implicit none

    real(kind=jprb), dimension(:), intent(in) :: upper_limits
    logical :: is_valid_blanket
    integer :: nb_layers, i
    
    is_valid_blanket = .true.

    if (upper_limits(1) /= UPPER_BLANKET_LIMIT) then
      is_valid_blanket = .false.
    else
      ! Loops over upper limits as long as ascending
      nb_layers = size(upper_limits, 1)
      do i=1, nb_layers - 1
        if (upper_limits(i) >= upper_limits(i + 1)) then
          exit
        end if
      end do
      
      if (i /= nb_layers) then
        is_valid_blanket = .false.
      else if (upper_limits(nb_layers) > LOWER_BLANKET_LIMIT) then
        is_valid_blanket = .false.
      end if
    end if

  end function is_valid_blanket
  
  ! Simple function checking that a triplet of real values describes a valid range that can be 
  ! used to build spectral bands.
  
  function is_valid_range(input_range)

    implicit none
    
    real(kind=jprb), intent(in) :: input_range(:)
    logical :: is_valid_range
    
    is_valid_range = .true.
    
    ! Range must be included in [LOWER_SW_BOUND, UPPER_SW_BOUND]
    if (input_range(1) < LOWER_SW_BOUND .or. input_range(2) > UPPER_SW_BOUND) then
      is_valid_range = .false.
    ! Lower bound must be lower than upper bound
    else if (input_range(1) >= input_range(2)) then
      is_valid_range = .false.
    ! Upper - bound % step == 0
    else if (modulo(input_range(2) - input_range(1), input_range(3)) /= 0.0) then
      is_valid_range = .false.
    end if
  
  end function is_valid_range

  ! Simple function checking that a given text file is providing a valid description of a spectral 
  ! decomposition of solar radiation. This means: one real value per line, in nanometers, in 
  ! ascending order with all values included in [LOWER_SW_BOUND, UPPER_SW_BOUND].
  ! 
  ! 08/02/2023: this function will likely no longer be used in order to restrict the configuration 
  ! of the MAR-embedded ecRad to a single file. The function is kept just in case (also, is the 
  ! basis for the subsequent is_valid_spectrum function).
  
  function is_valid_spectrum_file(input_file)

    use radiation_io, only : nulerr, my_abort => radiation_abort
    
    implicit none
    
    character(len=*), intent(in) :: input_file
    logical :: is_valid_spectrum_file
    
    integer :: iosopen
    real(kind=jprb), dimension(MaxLinesPerFile+1) :: bounds
    integer :: nb_bounds, i
    
    ! Opens the spectrum file and reads it line by line
    open(unit=10, iostat=iosopen, file=trim(input_file))
    if (iosopen /= 0) then
      write(nulerr, '(a)') 'ecRad config error: spectrum file "'//trim(input_file)//'" is missing.'
      call my_abort('ecRad configuration error')
    end if
    
    nb_bounds = 0
    do
      read(10, '(f12.7)', iostat=iosopen) bounds(nb_bounds+1)
      if (iosopen /= 0) then
        exit
      end if
      nb_bounds = nb_bounds + 1
    end do
    
    close(unit=10)
    
    ! Loops over bounds as long as ascending
    do i=1, nb_bounds-1
      if (bounds(i) >= bounds(i+1)) then
        exit
      end if
    end do
    
    is_valid_spectrum_file = .true.
    ! Returns false if not exclusively ascending or if bounds are problematic
    if (i /= nb_bounds) then
      is_valid_spectrum_file = .false.
    else if (bounds(1) < LOWER_SW_BOUND .or. bounds(nb_bounds) > UPPER_SW_BOUND) then
      is_valid_spectrum_file = .false.
    end if
  
  end function is_valid_spectrum_file
  
  ! Simple function that processes a one-dimensional array to ensure it describes a valid 
  ! sequence of spectral bounds that can be used to build custom spectral bands and output a 
  ! spectral decomposition of the surface shortwave fluxes. It is derived from the function 
  ! is_valid_spectrum_file (see above).
  
  function is_valid_spectrum(spectral_bounds)
  
    implicit none
    
    real(kind=jprb), dimension(:), intent(in) :: spectral_bounds
    integer :: nb_bounds, i
    logical :: is_valid_spectrum
    
    ! Loops over bounds as long as ascending
    nb_bounds = size(spectral_bounds, 1)
    do i=1, nb_bounds - 1
      if (spectral_bounds(i) >= spectral_bounds(i + 1)) then
        exit
      end if
    end do
    
    ! Returns false if: too few bounds, not exclusively ascending or problematic bounds
    is_valid_spectrum = .true.
    if (nb_bounds < 3 .or. i /= nb_bounds) then
      is_valid_spectrum = .false.
    else if (spectral_bounds(1) < LOWER_SW_BOUND .or. & 
     &       spectral_bounds(nb_bounds) > UPPER_SW_BOUND) then
      is_valid_spectrum = .false.
    end if
  
  end function is_valid_spectrum

  ! Reads ecRad parameters from a namelist file. Unknown string values for string-based parameters 
  ! lead to aborting the program. Aborting also happens if the namelist file cannot be opened. It 
  ! is up to the calling code to make sure this file exists in the first place and to use the 
  ! default configuration (see next subroutine) if no such file can be found.
  
  subroutine read_config_from_namelist(this, file_name)
  
    use radiation_io, only : nulerr, my_abort => radiation_abort
    
    class(ecRad_config_type), intent(inout) :: this
    character(len=*), intent(in) :: file_name
    
    ! Parameters as read from the namelist file
    integer :: columns_per_cpu
    integer :: greenhouse_period_start
    integer :: greenhouse_period_end
    character(len=StringValMaxLength) :: interpolation_smoothing_method
    character(len=FilePathMaxLength) :: path_radiation_data
    character(len=FilePathMaxLength) :: path_greenhouse_data
    character(len=FilePathMaxLength) :: path_aerosols_data
    character(len=FilePathMaxLength) :: path_cmip_scenarios
    integer :: verbosity
    
    ! Note how integers from above are now character(len=StringValMaxLength)
    character(len=StringValMaxLength) :: aerosol_climatology
    logical :: aerosol_longwave_scattering
    
    real(kind=jprb) :: cloud_fraction_alpha
    real(kind=jprb) :: cloud_fraction_gamma
    real(kind=jprb) :: cloud_fraction_p
    character(len=StringValMaxLength) :: cloud_fraction_parametrization_name
    logical :: cloud_longwave_scattering
    character(len=StringValMaxLength) :: cloud_optics_ice_scheme
    character(len=StringValMaxLength) :: cloud_optics_water_scheme
    real(kind=jprb) :: cloud_water_fractional_std

    real(kind=jprb) :: decorrelation_depth_cloud
    real(kind=jprb) :: decorrelation_depth_condensate
    character(len=StringValMaxLength) :: decorrelation_depth_scheme
    real(kind=jprb) :: decorrelation_length_ratio
    logical :: decorrelation_use_beta_overlap
    
    character(len=StringValMaxLength) :: gas_model_name
    character(len=FileNameMaxLength) :: gas_optics_lw_file
    character(len=FileNameMaxLength) :: gas_optics_sw_file
    
    real(kind=jprb) :: particles_ccn_concentration_ocean
    real(kind=jprb) :: particles_ccn_concentration_land
    real(kind=jprb) :: particles_conversion_radius_size
    logical :: particles_ice_latitude_dependence
    real(kind=jprb) :: particles_ice_minimum_diameter
    character(len=StringValMaxLength) :: particles_ice_radius_scheme
    character(len=StringValMaxLength) :: particles_liquid_radius_scheme

    character(len=StringValMaxLength) :: solvers_longwave_name
    character(len=StringValMaxLength) :: solvers_shortwave_name

    logical :: upper_blanket_layers
    real(kind=jprb), dimension(MaxNamelistArraySize) :: upper_blanket_limits ! Namelist -> max size
    real(kind=jprb) :: upper_solar_scaling
    
    real(kind=jprb) :: water_species_scaling_vapor
    real(kind=jprb) :: water_species_scaling_ice
    real(kind=jprb) :: water_species_scaling_liquid
    real(kind=jprb) :: water_species_scaling_rain
    real(kind=jprb) :: water_species_scaling_snow

    logical :: additional_fluxes_output
    character(len=FilePathMaxLength) :: additional_fluxes_filename
    logical :: climatologies_interpolated_output
    character(len=FilePathMaxLength) :: climatologies_interpolated_filename
    logical :: climatologies_scaled_output
    character(len=FilePathMaxLength) :: climatologies_scaled_filename
    logical :: solar_spectral_output
    character(len=FilePathMaxLength) :: solar_spectral_filename
    logical :: solar_spectral_bands_regular
    real(kind=jprb), dimension(NbValuesInRange) :: solar_spectral_bands_range ! In nanometers
    real(kind=jprb), dimension(MaxNamelistArraySize) :: solar_spectral_bands_full ! Ditto
    logical :: solar_spectral_padding
    logical :: solar_spectral_raw_output
    character(len=FilePathMaxLength) :: solar_spectral_raw_filename
    logical :: solar_spectral_raw_padding
    logical :: time_steps_hourly_only
    
    namelist /MAR_ecRad_parameters/ columns_per_cpu, greenhouse_period_start, & 
     & greenhouse_period_end, interpolation_smoothing_method, path_radiation_data, & 
     & path_greenhouse_data, path_aerosols_data, path_cmip_scenarios, verbosity, & 
     & aerosol_climatology, aerosol_longwave_scattering, &
     & cloud_fraction_alpha, cloud_fraction_gamma, cloud_fraction_p, &
     & cloud_fraction_parametrization_name, cloud_longwave_scattering, cloud_optics_ice_scheme, &
     & cloud_optics_water_scheme, cloud_water_fractional_std, &
     & decorrelation_depth_cloud, decorrelation_depth_condensate, decorrelation_depth_scheme, &
     & decorrelation_length_ratio, decorrelation_use_beta_overlap, &
     & gas_model_name, gas_optics_lw_file, gas_optics_sw_file, &
     & particles_ccn_concentration_ocean, particles_ccn_concentration_land, & 
     & particles_conversion_radius_size, particles_ice_latitude_dependence, &
     & particles_ice_minimum_diameter, particles_ice_radius_scheme, & 
     & particles_liquid_radius_scheme, &
     & solvers_longwave_name, solvers_shortwave_name, &
     & upper_blanket_layers, upper_blanket_limits, upper_solar_scaling, &
     & water_species_scaling_vapor, water_species_scaling_ice, water_species_scaling_liquid, &
     & water_species_scaling_rain, water_species_scaling_snow, &
     & additional_fluxes_output, additional_fluxes_filename, &
     & climatologies_interpolated_output, climatologies_interpolated_filename, &
     & climatologies_scaled_output, climatologies_scaled_filename, & 
     & solar_spectral_output, solar_spectral_filename, solar_spectral_bands_regular, & 
     & solar_spectral_bands_range, solar_spectral_bands_full, solar_spectral_padding, & 
     & solar_spectral_raw_output, solar_spectral_raw_filename, solar_spectral_raw_padding, &
     & time_steps_hourly_only
    
    ! Status after calling open
    integer :: iosopen
    
    ! Return value of check_string subroutine
    integer :: ret
    
    ! For error messages providing limits of an expected range
    character(len=6) :: lower_str, upper_str
    
    ! Default values (a.k.a. default configuration); starting from 13/03/2023, there are default 
    ! values for all variables to ensure MAR can rely on a default ecRad configuration if a .nam 
    ! configuration file is missing. This also means that MAR scripts have to enforce some paths.
    
    ! Default paths to radiation/greenhouse/aerosol/CMIP data
    columns_per_cpu = 4
    greenhouse_period_start = 2003
    greenhouse_period_end = 2010
    interpolation_smoothing_method = 'Horizontal average'
    path_radiation_data = './radiation_data'
    path_greenhouse_data = './climatologies/greenhouse_gas_climatology_46r1.nc'
    path_aerosols_data = './climatologies/aerosol_cams_climatology_43r3_v2_3D.nc'
    path_cmip_scenarios = './climatologies/scenarios/greenhouse_gas_timeseries_'
    verbosity = 1 ! Dumps at least the consolidated ecRad configuration in the console
    
    ! Default values are partly derived from the default configuration of ecRad
    aerosol_climatology = 'MACC'
    aerosol_longwave_scattering = .true.
    
    this%cloud_fraction_alpha = 100. ! Default for Xu & Randall
    this%cloud_fraction_gamma = 0.49 ! Default for Xu & Randall (0.02 for old ECMWF in EU dom)
    this%cloud_fraction_p = 0.25 ! Default for Xu & Randall
    this%cloud_fraction_parametrization = 3 ! Sundqvist (1~2 = old ECMWF, 4 = Xu & Randall)
    ! N.B.: the difference between 1 and 2 (old ECMWF) is 2 takes account of ice crystals.
    this%cloud_longwave_scattering = .true.
    this%cloud_optics_ice_scheme = 3 ! Fu
    this%cloud_optics_water_scheme = 3 ! SOCRATES
    this%cloud_water_fractional_std = 0.75 ! 0.75±0.18 (Shonk et al., 2010), but 0.5 OK for MAR
    
    decorrelation_depth_cloud = 2.0
    decorrelation_depth_condensate = 1.0
    decorrelation_depth_scheme = 'Shonk-Hogan-I'
    decorrelation_length_ratio = 0.5
    decorrelation_use_beta_overlap = .false. ! Alpha overlap (Hogan and Illingworth, 2000)
    
    ! Note how default file names exist, because they normally exist at path_radiation_data
    gas_model_name = 'ecCKD'
    gas_optics_lw_file = 'ecckd-1.2_lw_climate_narrow-64b_ckd-definition.nc'
    gas_optics_sw_file = 'ecckd-1.5_sw_climate_vfine-96b_ckd-definition.nc'
    
    particles_ccn_concentration_ocean = 50.0
    particles_ccn_concentration_land = 900.0
    particles_conversion_radius_size = 0.64952
    particles_ice_latitude_dependence = .true.
    particles_ice_minimum_diameter = 60.0
    particles_ice_radius_scheme = 'Sun-Rikus'
    particles_liquid_radius_scheme = 'Martin'

    ! McICA (default for ecRad) is tailored for operational forecast (with IFS)
    solvers_longwave_name = 'Tripleclouds'
    solvers_shortwave_name = 'Tripleclouds'

    upper_blanket_layers = .true.
    upper_blanket_limits = 0. ! Sets all cells to zero
    upper_blanket_limits(1) = 1. ! Expressed in hPa (1 hPa = stratopause)
    upper_blanket_limits(2) = 10.
    upper_blanket_limits(3) = 30.
    upper_solar_scaling = 0.9962 ! Adjusts 1366 to 1360.8 (old MAR config)
    
    water_species_scaling_vapor = 1. ! No scaling by default (ditto below)
    water_species_scaling_ice = 1.
    water_species_scaling_liquid = 1.
    water_species_scaling_rain = 1.
    water_species_scaling_snow = 1.

    additional_fluxes_output = .false.
    additional_fluxes_filename = 'extra_fluxes.nc'
    climatologies_interpolated_output = .false.
    climatologies_interpolated_filename = 'clim_interpolated.nc'
    climatologies_scaled_output = .false.
    climatologies_scaled_filename = 'clim_scaled.nc'
    solar_spectral_output = .false.
    solar_spectral_filename = 'SW_spectral.nc'
    solar_spectral_bands_regular = .false.
    solar_spectral_bands_range = (/ 200., 700., 25. /) ! Example range
    solar_spectral_bands_full = 0. ! Zero everywhere -> no description yet
    solar_spectral_padding = .true.
    solar_spectral_raw_output = .false.
    solar_spectral_raw_filename = 'SW_spectral_raw.nc'
    solar_spectral_raw_padding = .true.
    time_steps_hourly_only = .true.
    
    ! Opens the provided file and reads the MAR_ecRad_parameters namelist
    open(unit=10, iostat=iosopen, file=trim(file_name))
    if (iosopen /= 0) then
      write(nulerr, '(a)') 'ecRad config error: namelist file "'//trim(file_name)//'" is missing.'
      call my_abort('ecRad configuration error')
    end if
    read(unit=10, nml=MAR_ecRad_parameters)
    close(unit=10)
    
    ! Verifies each string parameter and adjusts the return value
    call check_string(varNames(1), interpolation_smoothing_method, smoothingMethods, ret)
    this%interpolation_smoothing_method = ret-1 ! 0, 1
    call check_string(varNames(2), aerosol_climatology, aerosolClimatologies, ret)
    this%aerosol_climatology = ret-1 ! 0, 1
    call check_string(varNames(3), cloud_fraction_parametrization_name, cloudParametrizations, ret)
    this%cloud_fraction_parametrization = ret-1 ! 0 to 4
    call check_string(varNames(4), cloud_optics_ice_scheme, cloudOpticsIceSchemes, ret)
    this%cloud_optics_ice_scheme = ret+2 ! 3, 4
    call check_string(varNames(5), cloud_optics_water_scheme, cloudOpticsWatSchemes, ret)
    this%cloud_optics_water_scheme = ret+1 ! 2, 3
    call check_string(varNames(6), decorrelation_depth_scheme, decorrelationSchemes, ret)
    this%decorrelation_depth_scheme = ret-1 ! 0 to 2
    call check_string(varNames(7), gas_model_name, gasModels, ret)
    this%gas_model = ret-1 ! 0, 1
    call check_string(varNames(8), particles_ice_radius_scheme, particlesIceRadiusSchemes, ret)
    this%particles_ice_radius_scheme = ret-1 ! 0 to 3
    call check_string(varNames(9), particles_liquid_radius_scheme, particlesLiqRadiusSchemes, ret)
    this%particles_liquid_radius_scheme = ret-1 ! 0 to 2
    call check_string(varNames(10), solvers_longwave_name, availableSolvers, ret)
    this%solvers_longwave = ret-1 ! 0 to 3
    call check_string(varNames(11), solvers_shortwave_name, availableSolvers, ret)
    this%solvers_shortwave = ret-1 ! 0 to 3
    
    ! Additional check: file paths cannot be empty
    if (len(trim(path_radiation_data)) == 0 .or. len(trim(path_greenhouse_data)) == 0 .or. & 
     &  len(trim(path_aerosols_data)) == 0 .or. len(trim(path_cmip_scenarios)) == 0) then
      write(nulerr, '(a,a,a)') 'ecRad config error: the paths to the files required by ecRad ', &
       &                       '(ecRad radiation data, greenhouse data, aerosols data, ', &
       &                       'CMIP scenarios) cannot be empty.'
      call my_abort('ecRad configuration error')
    end if
    
    ! Additional check: periods covered by mean mixing ratios NetCDF files MUST be provided
    if (greenhouse_period_start <= 0 .or. greenhouse_period_end <= 0) then
      write(nulerr, '(a,a)') 'ecRad config error: period covered by the greenhouse gas data ', &
       &                     'NetCDF file cannot be described with null or negative bounds.'
      call my_abort('ecRad configuration error')
    end if
    
    ! Additional check: aforementioned periods MUST make sense
    if (greenhouse_period_start > greenhouse_period_end .or. greenhouse_period_end > 2500) then
      write(nulerr, '(a,a)') 'ecRad config error: period covered by the greenhouse gas data ', &
       &                     'NetCDF file must be such that 1 <= start <= end <= 2500.'
      call my_abort('ecRad configuration error')
    end if
    
    ! Additional check: if SPARTACUS-3D, MUST be used for both shortwave AND longwave
    if (this%solvers_longwave == 2 .or. this%solvers_shortwave == 2) then
      if (this%solvers_longwave /= this%solvers_shortwave) then
        write(nulerr, '(a,a)') 'ecRad config error: if using SPARTACUS with 3D effects as ', & 
         &                     'solver, it MUST be used for both shortwave and longwave.'
        call my_abort('ecRad configuration error')
      end if
    end if
    
    ! Additional check: aerosol longwave scattering only makes sense with MACC
    if (aerosol_longwave_scattering) then
      if (this%aerosol_climatology /= 1) then
        write(nulerr, '(a,a)') 'ecRad config error: aerosol longwave scattering can be ', & 
         &                     'enabled if and only if using MACC aerosol climatology.'
        call my_abort('ecRad configuration error')
      end if
    end if
    
    ! Additional check: when an additional output file is requested, the filename (with or without 
    ! a path) MUST be given. This check is performed for all four types of additional output files.
    
    if (additional_fluxes_output .and. len(trim(additional_fluxes_filename)) == 0) then
      write(nulerr, '(a,a)') 'ecRad config error: to output additional fluxes computed by ', &
       &                     'ecRad into a NetCDF file, a filename MUST be given.'
      call my_abort('ecRad configuration error')
    end if
    
    if (climatologies_interpolated_output &
     &  .and. len(trim(climatologies_interpolated_filename)) == 0) then
      write(nulerr, '(a,a)') 'ecRad config error: to output climatologies interpolated on the ', &
       &                     'MAR grid (and scaled) into a NetCDF file, a filename MUST be given.'
      call my_abort('ecRad configuration error')
    end if
    
    if (climatologies_scaled_output .and. len(trim(climatologies_scaled_filename)) == 0) then
      write(nulerr, '(a,a,a)') 'ecRad config error: to output climatologies scaled to a CMIP ', &
       &                       'scenario (whole longitudinal slice) into a NetCDF file, a ', &
       &                       'filename MUST be given.'
      call my_abort('ecRad configuration error')
    end if
    
    if (solar_spectral_output .and. len(trim(solar_spectral_filename)) == 0) then
      write(nulerr, '(a,a,a)') 'ecRad config error: to output shortwave spectral data ', &
       &                       'following user-defined bands into a NetCDF file, a ', &
       &                       'filename MUST be given.'
      call my_abort('ecRad configuration error')
    end if
    
    if (solar_spectral_raw_output .and. len(trim(solar_spectral_raw_filename)) == 0) then
      write(nulerr, '(a,a,a)') 'ecRad config error: to output shortwave spectral data ', &
       &                       'as internally defined in ecRad into a NetCDF file, a ', &
       &                       'filename MUST be given.'
      call my_abort('ecRad configuration error')
    end if
    
    ! Additional check: if blanket levels are used, checks their description is valid
    if (upper_blanket_layers) then
      call copy_namelist_array(upper_blanket_limits, this%upper_blanket_limits)
      if (.not. is_valid_blanket(this%upper_blanket_limits)) then
        write(lower_str, '(f5.1)') UPPER_BLANKET_LIMIT
        write(upper_str, '(f6.1)') LOWER_BLANKET_LIMIT
        write(nulerr, '(a,a,a,a,a,a)') 'ecRad config error: the pressure levels provided to ', &
         &                             'create blanket levels (stratosphere) is invalid. It ', &
         &                             'must consist in a list of ascending pressure levels ', &
         &                             'expressed in hPa starting at 1 hPa (stratopause) ', &
         &                             'with the subsequent values being included in ', &
         &                             ']'//trim(lower_str)//', '//trim(upper_str)//'].'
        call my_abort('ecRad configuration error')
      end if
    end if
    
    ! Additional checks: for any water species scaling factor, must be in a hard-coded interval
    call check_scaling_factor('vapor', water_species_scaling_vapor)
    call check_scaling_factor('crystals', water_species_scaling_ice)
    call check_scaling_factor('droplets', water_species_scaling_liquid)
    call check_scaling_factor('raindrops', water_species_scaling_rain)
    call check_scaling_factor('snowflakes', water_species_scaling_snow)
    
    ! Additional check: verifies the provided spectral bands description is valid
    if (solar_spectral_output) then
      if (solar_spectral_bands_regular) then
        if (.not. is_valid_range(solar_spectral_bands_range)) then
          write(lower_str, '(f5.1)') LOWER_SW_BOUND
          write(upper_str, '(f6.1)') UPPER_SW_BOUND
          write(nulerr, '(a,a,a,a,a,a)') 'ecRad config error: the range provided to build ', &
           &                             'spectral bands is invalid. It must consist in three ', &
           &                             'values: lower bound, upper bound and step. The ', &
           &                             'bounds must fulfill '//trim(lower_str)//' <= lower < ', &
           &                             'upper <= '//trim(upper_str)//' and the step must be ', &
           &                             'a divisor of (upper - lower).'
          call my_abort('ecRad configuration error')
        end if
      else
        call copy_namelist_array(solar_spectral_bands_full, this%solar_spectral_bands_full)
        if (.not. is_valid_spectrum(this%solar_spectral_bands_full)) then
          write(lower_str, '(f5.1)') LOWER_SW_BOUND
          write(upper_str, '(f6.1)') UPPER_SW_BOUND
          write(nulerr, '(a,a,a,a,a,a)') 'ecRad config error: the described sequence of ', &
           &                             'spectral bands is invalid. It must consist in a ', &
           &                             'sequence of strictly ascending spectral bounds, with ', &
           &                             'a minimum number of 3 bounds. The bounds must also ', &
           &                             'fulfill '//trim(lower_str)//' <= lower < ', &
           &                             'upper <= '//trim(upper_str)//'.'
          call my_abort('ecRad configuration error')
        end if
      end if
    end if
    
    ! Copies values in the config object for remaining parameters
    this%columns_per_cpu = columns_per_cpu
    this%greenhouse_period_start = greenhouse_period_start
    this%greenhouse_period_end = greenhouse_period_end
    this%path_aerosols_data = path_aerosols_data
    this%path_cmip_scenarios = path_cmip_scenarios
    this%path_greenhouse_data = path_greenhouse_data
    this%path_radiation_data = path_radiation_data 
    this%verbosity = verbosity
    
    this%aerosol_longwave_scattering = aerosol_longwave_scattering
    
    this%cloud_fraction_alpha = cloud_fraction_alpha
    this%cloud_fraction_gamma = cloud_fraction_gamma
    this%cloud_fraction_p = cloud_fraction_p
    this%cloud_longwave_scattering = cloud_longwave_scattering
    this%cloud_water_fractional_std = cloud_water_fractional_std
    
    this%decorrelation_depth_cloud = decorrelation_depth_cloud
    this%decorrelation_depth_condensate = decorrelation_depth_condensate
    this%decorrelation_length_ratio = decorrelation_length_ratio
    this%decorrelation_use_beta_overlap = decorrelation_use_beta_overlap
    
    this%gas_optics_lw_file = gas_optics_lw_file
    this%gas_optics_sw_file = gas_optics_sw_file
    
    this%particles_ccn_concentration_ocean = particles_ccn_concentration_ocean
    this%particles_ccn_concentration_land = particles_ccn_concentration_land
    this%particles_conversion_radius_size = particles_conversion_radius_size
    this%particles_ice_latitude_dependence = particles_ice_latitude_dependence
    this%particles_ice_minimum_diameter = particles_ice_minimum_diameter

    this%upper_blanket_layers = upper_blanket_layers
    ! this%upper_blanket_limits -> Already set via copy_namelist_array
    this%upper_solar_scaling = upper_solar_scaling
    
    this%water_species_scaling_vapor = water_species_scaling_vapor
    this%water_species_scaling_ice = water_species_scaling_ice
    this%water_species_scaling_liquid = water_species_scaling_liquid
    this%water_species_scaling_rain = water_species_scaling_rain
    this%water_species_scaling_snow = water_species_scaling_snow

    this%additional_fluxes_output = additional_fluxes_output
    this%additional_fluxes_filename = additional_fluxes_filename
    this%climatologies_interpolated_output = climatologies_interpolated_output
    this%climatologies_interpolated_filename = climatologies_interpolated_filename
    this%climatologies_scaled_output = climatologies_scaled_output
    this%climatologies_scaled_filename = climatologies_scaled_filename
    this%solar_spectral_output = solar_spectral_output
    this%solar_spectral_filename = solar_spectral_filename
    this%solar_spectral_bands_regular = solar_spectral_bands_regular
    this%solar_spectral_bands_range = solar_spectral_bands_range
    ! this%solar_spectral_bands_full -> Already set via copy_namelist_array
    this%solar_spectral_padding = solar_spectral_padding
    this%solar_spectral_raw_output = solar_spectral_raw_output
    this%solar_spectral_raw_filename = solar_spectral_raw_filename
    this%solar_spectral_raw_padding = solar_spectral_raw_padding
    this%time_steps_hourly_only = time_steps_hourly_only
    
  end subroutine read_config_from_namelist
  
  ! Sets all variables of the configuration object with default values. These values are the same 
  ! as those in read_config_from_namelist, except this subroutine does not perform any kind of 
  ! verification or processing and sets some variables a bit differently because of that (e.g., 
  ! parameters received as strings are directly set with their associated integer value).
  
  subroutine default_config(this)
  
    class(ecRad_config_type), intent(inout) :: this
  
    this%columns_per_cpu = 4
    this%greenhouse_period_start = 2003
    this%greenhouse_period_end = 2010
    this%interpolation_smoothing_method = 1 ! Horizontal average
    this%path_radiation_data = './radiation_data'
    this%path_greenhouse_data = './climatologies/greenhouse_gas_climatology_46r1.nc'
    this%path_aerosols_data = './climatologies/aerosol_cams_climatology_43r3_v2_3D.nc'
    this%path_cmip_scenarios = './climatologies/scenarios/greenhouse_gas_timeseries_'
    this%verbosity = 1 ! Dumps at least the consolidated ecRad configuration in the console
    
    this%aerosol_climatology = 1 ! MACC
    this%aerosol_longwave_scattering = .true.
    
    this%cloud_fraction_alpha = 100. ! Default for Xu & Randall
    this%cloud_fraction_gamma = 0.49 ! Default for Xu & Randall (0.02 for old ECMWF in EU dom)
    this%cloud_fraction_p = 0.25 ! Default for Xu & Randall
    this%cloud_fraction_parametrization = 3 ! Sundqvist (1~2 = old ECMWF, 4 = Xu & Randall)
    ! N.B.: the difference between 1 and 2 (old ECMWF) is 2 takes account of ice crystals.
    this%cloud_longwave_scattering = .true.
    this%cloud_optics_ice_scheme = 3 ! Fu
    this%cloud_optics_water_scheme = 3 ! SOCRATES
    this%cloud_water_fractional_std = 0.75 ! 0.75±0.18 (Shonk et al., 2010), but 0.5 OK for MAR
    
    this%decorrelation_depth_cloud = 2.0
    this%decorrelation_depth_condensate = 1.0
    this%decorrelation_depth_scheme = 2 ! Shonk-Hogan-I
    this%decorrelation_length_ratio = 0.5
    this%decorrelation_use_beta_overlap = .false. ! Alpha overlap (Hogan and Illingworth, 2000)
    
    this%gas_model = 1 ! ecCKD
    this%gas_optics_lw_file = 'ecckd-1.2_lw_climate_narrow-64b_ckd-definition.nc'
    this%gas_optics_sw_file = 'ecckd-1.5_sw_climate_vfine-96b_ckd-definition.nc'
    
    this%particles_ccn_concentration_ocean = 50.0
    this%particles_ccn_concentration_land = 900.0
    this%particles_conversion_radius_size = 0.64952
    this%particles_ice_latitude_dependence = .true.
    this%particles_ice_minimum_diameter = 60.0
    this%particles_ice_radius_scheme = 3 ! Sun-Rikus
    this%particles_liquid_radius_scheme = 2 ! Martin

    this%solvers_longwave = 3 ! Tripleclouds
    this%solvers_shortwave = 3 ! Tripleclouds

    this%upper_blanket_layers = .true.
    allocate(this%upper_blanket_limits(3))
    this%upper_blanket_limits(1) = 1.
    this%upper_blanket_limits(2) = 10.
    this%upper_blanket_limits(3) = 30.
    this%upper_solar_scaling = 0.9962
    
    this%water_species_scaling_vapor = 1.
    this%water_species_scaling_ice = 1.
    this%water_species_scaling_liquid = 1.
    this%water_species_scaling_rain = 1.
    this%water_species_scaling_snow = 1.

    this%additional_fluxes_output = .false.
    this%additional_fluxes_filename = 'extra_fluxes.nc'
    this%climatologies_interpolated_output = .false.
    this%climatologies_interpolated_filename = 'clim_interpolated.nc'
    this%climatologies_scaled_output = .false.
    this%climatologies_scaled_filename = 'clim_scaled.nc'
    this%solar_spectral_output = .false.
    this%solar_spectral_filename = 'SW_spectral.nc'
    this%solar_spectral_bands_regular = .false.
    this%solar_spectral_bands_range = (/ 200., 700., 25. /) ! Example range
    ! this%solar_spectral_bands_full remains unallocated by default
    this%solar_spectral_padding = .true.
    this%solar_spectral_raw_output = .false.
    this%solar_spectral_raw_filename = 'SW_spectral_raw.nc'
    this%solar_spectral_raw_padding = .true.
    this%time_steps_hourly_only = .true.
  
  end subroutine default_config
  
end module MAR_ecRad_config
