subroutine PHYrad_ecRad_init(dimLon,dimLat,nbLvls,dx,yyyy,mm, &
      &                      config_path,CMIP_scenario,       &
      &                      rp_RAD,rLaRAD,rLoRAD,            &
      &                      MAR_override_logical,            &
      &                      MAR_override_int,                &
      &                      MAR_override_real,               &
      &                      MAR_override_strato,             &
      &                      MAR_override_spectral)

! +------------------------------------------------------------------------+
! | MAR PHYSICS                                         September 2023 MAR |
! |                                                                        |
! | PHYrad_ecRad_init initializes the ECMWF solar/infrared radiative       |
! | transfer scheme ecRad 1.5.0 for the needs of MAR, taking account of    |
! | various parameters (hard-coded or tunable by MAR users). It MUST be    |
! | called prior to calling PHYrad_ecRad, which runs ecRad for MAR.        |
! |                                                                        |
! | Code by J.-F. Grailet, based on previous subroutine PHYrad2CEP (main   |
! | structure) and on the ecRad offline driver.                            |
! |                                                                        |
! | Main body has been written in September and October 2022 and the first |
! | successful MAR runs with ecRad have been conducted in November 2022,   |
! | but numerous additions and adjustements were brought since then to     |
! | tune the new radiation scheme and make the most of its features, such  |
! | as spectral decomposition of shortwave radiation.                      |
! |                                                                        |
! | The subroutine has been split in two (PHYrad_ecRad_init and            |
! | PHYrad_ecRad) in September 2023 to clearly separate the operations of  |
! | initializing and running the radiation scheme while improving code     |
! | readability.                                                           |
! |                                                                        |
! | ECMWF code by Hogan et al., cf. https://github.com/ecmwf-ifs/ecrad     |
! |                                                                        |
! | General remarks:                                                       |
! | -IFS = Integrated Forecast System, a weather prediction model used     |
! |  by the ECMWF.                                                         |
! |                                                                        |
! +------------------------------------------------------------------------+

  use parkind1, only : jpim, jprb

! +--Global variables via modules
! +  ============================

  use YOERAD, only : NSW, NDECOLAT, NRADIP, RMINICE, NMINICE, &
   &                 NRADLP, NICEOPT, NLIQOPT, LCCNO, LCCNL, &
   &                 RCCNSEA, RCCNLND, RRE2DE
  use YOERDU, only : REPLOG, REPSCW
  use INCMAR, only : RI0_SCALING, NAERMACC, NLWSOLVER, NSWSOLVER, & 
   &                 LCLD_SCATTERING, LAER_SCATTERING, & 
   &                 RCLOUD_FRAC_STD, RCLOUD_DECORR_CF, &
   &                 RCLOUD_DECORR_CW, RCLOUD_DECORR_LEN_RATIO, &
   &                 LCLOUD_BETA_OVERLAP, & 
   &                 NGASMODEL, ECCKD_LW_FILE, ECCKD_SW_FILE, & 
   &                 NCLOUD_PARAM, RCLOUD_PARAM_GAMMA, &
   &                 RCLOUD_PARAM_ALPHA, RCLOUD_PARAM_P, &
   &                 RCLOUD_RH_THRESHOLD_LAND, & 
   &                 RCLOUD_RH_THRESHOLD_SEA, &
   &                 BLANKET_LAYERS, &
   &                 SCALING_PQ, SCALING_PQIWP, SCALING_PQLWP, &
   &                 SCALING_PQRAIN, SCALING_PQSNOW, &
   &                 COLUMNS_PER_THREAD
  
! +--Additional modules
! +  ==================
  
  ! Module for configuring ecRad for MAR via a namelist file
  use MAR_ecRad_config, only : ecRad_config_type, copy_MAR_param_array, & 
   &                           is_valid_blanket, is_valid_spectrum
  
  ! ecRad setup (based on IFS code) (rad_config to get spectral definition)
  use radiation_setup, only : SETUP_RADIATION_SCHEME, rad_config
  
  ! Variables and class to prepare and store greenhouse gases mixing ratios
  use radiation_gas_constants, only : ICO2, IO3, IN2O, ICH4, & 
   &                                  ICFC11, ICFC12, IHCFC22, ICCl4
  use radiation_climatologies, only : N_AEROSOLS_TEGEN, & 
   &                                  N_AEROSOLS_MACC, & 
   &                                  CLIM_LONGITUDE, &
   &                                  CLIM_LATITUDES, CLIM_LEVELS, &
   &                                  PCO2, PCH4, PN2O, PCFC11, & 
   &                                  PCFC12, PHCFC22, PCCL4, PO3, &
   &                                  PAEROSOL_OLD, PAEROSOL, & 
   &                                  output_interpolated
  use greenhouse_climatology, only : greenhouse_type
  use aerosols_climatology, only : aerosols_type
  
  ! Module to handle extra ecRad output variables not used by MAR (no "only": everything is used)
  use extra_ecRad_outputs

  implicit none
  
! +--Input variables
! +  ===============

  integer, intent(in) :: dimLon    ! Number of longitude points (per line)
  integer, intent(in) :: dimLat    ! Number of latitude points (per column)
  integer, intent(in) :: nbLvls    ! Number of vertical levels
  real, intent(in) :: dx           ! Horizontal grid size (m) (from margrd_mod)
  
  integer, intent(in) :: yyyy   ! Year
  integer, intent(in) :: mm     ! Month
  
  character(len=*), intent(in) :: config_path   ! Path to .nam file to configure ecRad for MAR
  character(len=*), intent(in) :: CMIP_scenario ! Scenario for gas mixing ratios evolution
  
  real, intent(in) :: rp_RAD(dimLon, dimLat, nbLvls)     ! Pressure (layer)            [Pa]
  real, intent(in) :: rLaRAD(dimLon, dimLat)             ! Latitude                    [degree]
  real, intent(in) :: rLoRAD(dimLon, dimLat)             ! Longitude                   [degree]

  ! 27/04/2023: MAR "override", i.e., configuration of ecRad in MAR with the help of hard-coded 
  ! parameters to mimic the old way of tuning MAR parameters (e.g. via marmagic_mod.f90).
  
  logical, intent(in), optional :: MAR_override_logical(6)
  integer, intent(in), optional :: MAR_override_int(2)
  real, intent(in), optional :: MAR_override_real(8)
  real, intent(in), optional :: MAR_override_strato(:)
  real, intent(in), optional :: MAR_override_spectral(:)
  
! +--Local variables
! +  ===============

  ! Iterators (i = longitude, j = latitude, k = level, l = aerosol)
  integer :: i, j, k, l
  
  logical :: config_file_exists
  type(ecRad_config_type) :: ecRad_config
  
  ! Climatology objects (paths to NetCDF files via ecRad_config)
  type(greenhouse_type) :: greenhouse_clim
  type(aerosols_type) :: aerosols_clim
  
  ! Local variables used to pick sample latitudes and pressures (+ longitude for aerosols) to 
  ! interpolate greenhouse gases/aerosols climatologies on the regional grid, and smoothing method
  real(kind=jprb) :: max_pressure
  integer :: max_pressure_i, max_pressure_j
  real(kind=jprb) :: sum_longitudes
  integer :: smoothing ! 0 = none, 1 = horizontal average
  
  ! Number of blanket layers
  integer :: nbStrato
  
  ! J.-F. Grailet (02/02/2023): additional arrays used to write the additional NetCDF output files 
  ! of ecRad when requested. These arrays solely exist to solve a real precision issue: sometimes, 
  ! ecRad needs to be compiled in double precision for some of its options to work properly (e.g.: 
  ! SPARTACUS radiation solver with 3-D cloud effects enabled). The MAR can still use simple 
  ! precision simultaneously thanks to the type casting of ecRad outputs (see the code that 
  ! follows the call of RADIATION_SCHEME) to the default real precision used while compiling MAR. 
  ! Still, to avoid some pesky compilation errors with subroutines of the extra_ecRad_outputs 
  ! module (which uses the precision of ecRad), the following arrays are used to cast MAR inputs 
  ! to ecRad precision or to keep ecRad outputs to their initial precision.
  
  real(kind=jprb), dimension(:,:), allocatable :: MAR_LON
  real(kind=jprb), dimension(:,:), allocatable :: MAR_LAT
  
  ! Extra variable used for a short console message about climatologies
  real(kind=jprb) :: mean_CO2
  
! +--Beginning of the initialization of ecRad for MAR
! +  ================================================
  
  ! Reads configuration file for ecRad, if it exists, otherwise uses default a configuration
  inquire(file=trim(config_path), exist=config_file_exists)
  
  if (config_file_exists) then
    call ecRad_config%read(config_path)
  else
    call ecRad_config%default_config()
    
    ! 27/04/2023: MAR_override_ parameters are optional parameters that allow MAR users to tune 
    ! the most important parameters of the ecRad configuration in the same way they tune MAR with 
    ! the marmagic module. Only the parameters that have a significant impact on the radiative 
    ! balance or the output files are tunable in this way, the other parameters being set to 
    ! default values. More elaborate sets of parameters (e.g., using other ecCKD gas models) may 
    ! be implemented with the help of a proper namelist configuration file.
    
    if (present(MAR_override_logical)) then
      if (.not. MAR_override_logical(1)) then
        ecRad_config%gas_model = 0
      end if        
      
      if (MAR_override_logical(2)) then
        ecRad_config%aerosol_longwave_scattering = .true.
        ecRad_config%cloud_longwave_scattering = .true.
      end if

      ! File names for additional output files will be those of the default configuration
      if (MAR_override_logical(3)) ecRad_config%additional_fluxes_output = .true.
      if (MAR_override_logical(4)) ecRad_config%climatologies_interpolated_output = .true.
      if (MAR_override_logical(5)) ecRad_config%climatologies_scaled_output = .true.
      if (MAR_override_logical(6)) ecRad_config%solar_spectral_raw_output = .true.
    end if
    
    if (present(MAR_override_int)) then
      ! Cloud parametrization (enforces parameters for Xu & Randall)
      if (MAR_override_int(1) >= 0 .and. MAR_override_int(1) <= 4) then
        ecRad_config%cloud_fraction_parametrization = MAR_override_int(1)
        
        if (MAR_override_int(1) == 4) then ! Xu & Randall
          ecRad_config%cloud_fraction_alpha = 100.
          ecRad_config%cloud_fraction_gamma = 0.49
          ecRad_config%cloud_fraction_p = 0.25
        else if (MAR_override_int(1) == 1 .or. MAR_override_int(1) == 2) then ! ECMWF
          if (present(MAR_override_real)) then
            ecRad_config%cloud_fraction_gamma = MAR_override_real(3)
          else
            ecRad_config%cloud_fraction_gamma = 0.005
            ! J.-F. Grailet: 0.005 is for Greenland, the use case where ECMWF cloudiness is still 
            ! being used (value tuned by X. Fettweis). For European grids, it should be 0.02.
          end if
        end if
      end if
      
      ! Cloud decorrelation length formula
      if (MAR_override_int(2) >= 0 .and. MAR_override_int(2) <= 2) then
        ecRad_config%decorrelation_depth_scheme = MAR_override_int(2)
      end if
    end if
    
    if (present(MAR_override_real)) then
      ! Next two parameters (names self-explaining) cannot be negative
      ecRad_config%upper_solar_scaling = max(0.01, MAR_override_real(1))
      ecRad_config%cloud_water_fractional_std = max(0.01, MAR_override_real(2))
      
      ! Water species scaling ratios are restricted in the [0.8, 1.2] range
      ecRad_config%water_species_scaling_vapor = min(1.2, max(0.8, MAR_override_real(4)))
      ecRad_config%water_species_scaling_ice = min(1.2, max(0.8, MAR_override_real(5)))
      ecRad_config%water_species_scaling_liquid = min(1.2, max(0.8, MAR_override_real(6)))
      ecRad_config%water_species_scaling_rain = min(1.2, max(0.8, MAR_override_real(7)))
      ecRad_config%water_species_scaling_snow = min(1.2, max(0.8, MAR_override_real(8)))
    end if
    
    ! User-defined stratospheric "blanket" layers
    if (present(MAR_override_strato)) then
      if (size(MAR_override_strato, 1) > 0 .and. MAR_override_strato(1) /= 0.) then
        call copy_MAR_param_array(MAR_override_strato, ecRad_config%upper_blanket_limits)
        if (is_valid_blanket(ecRad_config%upper_blanket_limits)) then
          ecRad_config%upper_blanket_layers = .true.
        else
          write(6, *) 'ERROR: blanket layers in mar_ecrad_mod are wrongly formatted. Fix them !'
          stop
        end if
      else
        ecRad_config%upper_blanket_layers = .false.
      end if
    end if
    
    ! User-defined spectral decomposition (range is still checked just in case)
    if (present(MAR_override_spectral)) then
      if (size(MAR_override_spectral, 1) > 1 .and. MAR_override_spectral(1) /= 0.) then
        call copy_MAR_param_array(MAR_override_spectral, ecRad_config%solar_spectral_bands_full)
        if (is_valid_spectrum(ecRad_config%solar_spectral_bands_full)) then
          ecRad_config%solar_spectral_output = .true.
        else
          write(6, *) 'ERROR: spectral bands in mar_ecrad_mod are wrongly formatted. Fix them !'
          stop
        end if
      end if
    end if

  end if
  
! +--Control options for radiation configuration (YOERAD)
! +  ====================================================

  ! Decorrelation length for cloud fraction/condensate
  !  0   Independant of latitude
  !  1   Shonk/Hogan
  !  2   Improved Shonk/Hogan
  NDECOLAT = ecRad_config%decorrelation_depth_scheme
  
  ! Ice effective radius scheme
  !  0   Fixed (40 microns)
  !  1   Ice effective radius from Liou and Ou (1994), 40-130 microns
  !  2   Same as above with Jacob/Klein modifications
  !  3   f(T,IWC) from Sun and Rikus (1999), revised by Sun (2001)
  NRADIP = ecRad_config%particles_ice_radius_scheme
  
  ! Minimum diameter for ice particles
  ! E.g., 60.0 micrometers is the value used by the ECMWF
  RMINICE = ecRad_config%particles_ice_minimum_diameter

  ! Dependence of ice particle size on latitude
  !  0   Independant of latitude
  !  1   Varies with latitude
  if (ecRad_config%particles_ice_latitude_dependence) then ! Is a logical in ecRad_config_type
    NMINICE = 1
  else
    NMINICE = 0
  end if

  ! Liquid effective radius scheme
  !  0   Effective radius as f(Pressure)
  !  1   Fixed 10 microns over land, 13 over ocean
  !  2   Computed from LWC; Martin et al. (1994)
  NRADLP = ecRad_config%particles_liquid_radius_scheme
  
  ! Cloud optical properties w.r.t. ice
  !  3   LW: Fu et al., 1998; SW: Fu, 1996
  !  4   Baran et al. (see ecRad code, IIceModelBaran)
  NICEOPT = ecRad_config%cloud_optics_ice_scheme

  ! Cloud optical properties w.r.t. water
  !  2   LW: Lindner et Li, 2000; SW: Slingo, 1989
  !  3   SOCRATES (see ecRad code, ILiquidModelSOCRATES)
  ! XF WARNING: NLIQOPT=2 increases SWD and LWD but MAR is more unstable
  NLIQOPT = ecRad_config%cloud_optics_water_scheme

  ! Cloud Condensation Nuclei (CCN) concentration over ocean
  !  T   If diagnosed (to provide with PCCN_LAND, TODO)
  !  F   Use constant value RCCNSEA
  LCCNO = .false.

  ! Cloud Condensation Nuclei (CCN) concentration over land
  !  T   If diagnosed (to provide with PCCN_SEA, TODO)
  !  F   Use constant value RCCNLND
  LCCNL = .false.

  ! Constant CCN concentration for ocean
  RCCNSEA = ecRad_config%particles_ccn_concentration_ocean
  
  ! Constant CCN concentration for land
  RCCNLND = ecRad_config%particles_ccn_concentration_land
  
  ! Conversion factor between radius and particle size
  RRE2DE = ecRad_config%particles_conversion_radius_size
  
! +--Mathematical Constants used by IFS routines (YOERDU)
! +  ----------------------------------------------------

  REPLOG = 1.e-12 ! Minimum Logarithm Argument
  REPSCW = 1.e-12

! +--Additional ecRad parameters (INCMAR)
! +  ====================================
  
  ! Adjustement of the solar constant (RI0) to account for loss of energy in upper atmosphere
  RI0_SCALING = ecRad_config%upper_solar_scaling
  
  ! Type of aerosol climatology (variable named "NAERMACC" because MACC is more recent)
  !  0   Tegen aerosol climatology
  !  1   MACC aerosol climatology
  NAERMACC = ecRad_config%aerosol_climatology

  ! Longwave solver
  !  0   McICA (N.B.: McICA is for operational forecasting)
  !  1   SPARTACUS
  !  2   SPARTACUS with 3D effects
  !  3   Tripleclouds
  NLWSOLVER = ecRad_config%solvers_longwave

  ! Shortwave solver
  !  0   McICA (N.B.: McICA is for operational forecasting)
  !  1   SPARTACUS
  !  2   SPARTACUS with 3D effects
  !  3   Tripleclouds
  NSWSOLVER = ecRad_config%solvers_shortwave

  ! Representation of longwave scattering, respectively for clouds and aerosols
  !  T   If longwave scattering is simulated (for aerosols, works only with MACC climatology)
  !  F   Otherwise
  LCLD_SCATTERING = ecRad_config%cloud_longwave_scattering
  LAER_SCATTERING = ecRad_config%aerosol_longwave_scattering

  ! Fractional std dev. of in-cloud water content; cf. Table 1 in 
  ! http://www.met.rdg.ac.uk/~swrhgnrj/publications/shonk_part1.pdf
  RCLOUD_FRAC_STD = ecRad_config%cloud_water_fractional_std

  ! Constant decorrelation depth scale (if NDECOLAT=0), cloud fraction
  RCLOUD_DECORR_CF = ecRad_config%decorrelation_depth_cloud

  ! Constant decorrelation depth scale (if NDECOLAT=0), condensate
  RCLOUD_DECORR_CW = ecRad_config%decorrelation_depth_condensate
  
  ! Ratio of decorrelation length for cloud water content inhomogeneities and cloud boundaries
  RCLOUD_DECORR_LEN_RATIO = ecRad_config%decorrelation_length_ratio
  
  ! Do we use beta overlap (Shonk et al., 2010) rather than alpha (Hogan & Illingworth, 2000) ?
  LCLOUD_BETA_OVERLAP = ecRad_config%decorrelation_use_beta_overlap

  ! 17/11/2022: additional variables to use ecCKD instead of RRTM-G as gas model
  NGASMODEL = ecRad_config%gas_model
  if (NGASMODEL > 0) then
    ECCKD_LW_FILE = ecRad_config%gas_optics_lw_file
    ECCKD_SW_FILE = ecRad_config%gas_optics_sw_file
  end if

  ! Cloud fraction parametrization
  NCLOUD_PARAM = ecRad_config%cloud_fraction_parametrization
  RCLOUD_PARAM_ALPHA = ecRad_config%cloud_fraction_alpha
  RCLOUD_PARAM_GAMMA = ecRad_config%cloud_fraction_gamma
  RCLOUD_PARAM_P = ecRad_config%cloud_fraction_p
  
  ! 14/04/2023: relative humidity thresholds for Sundqvist's parametrization (1989). The 
  ! thresholds are adjusted with formulas provided in Wang. et al. (2023), "Evaluating two 
  ! diagnostic schemes of cloud-fraction parameterization using the CloudSat data". These 
  ! formulas require the horizontal resolution of MAR (dx in the margrd module, in meters) in km.
  
  RCLOUD_RH_THRESHOLD_LAND = 0.7 + sqrt(1. / (25. + (dx / 1000.)**3))
  RCLOUD_RH_THRESHOLD_SEA = 0.81 + sqrt(1. / (50. + (dx / 1000.)**3))

  ! Number of spectral intervals for the albedo (fixed; DON'T CHANGE)
  NSW = 6
  
  ! Important note on NSW (22/11/2022): currently, ecRad seems hard-wired to use 6 spectral bands 
  ! for albedo, as adjusting NSW to higher values just add slices of zeroes to PSWDIFFUSEBAND and 
  ! PSWDIRECTBAND. If this was fixed by a future version of ecRad, NSW may be set by the user. In 
  ! the meantime, variables dimensioned with NSW are still left allocatable to ease future changes.
  
  ! Blanket layers (modeling the ozone peak found in the stratosphere)
  if (ecRad_config%upper_blanket_layers) then
    allocate(BLANKET_LAYERS(size(ecRad_config%upper_blanket_limits, 1)))
    do i=1, size(BLANKET_LAYERS, 1)
      BLANKET_LAYERS(i) = ecRad_config%upper_blanket_limits(i) * 100 ! hPa -> Pa
    end do
  end if
  
  ! Scaling factors for water species mixing ratios
  SCALING_PQ = ecRad_config%water_species_scaling_vapor
  SCALING_PQIWP = ecRad_config%water_species_scaling_ice
  SCALING_PQLWP = ecRad_config%water_species_scaling_liquid
  SCALING_PQRAIN = ecRad_config%water_species_scaling_rain
  SCALING_PQSNOW = ecRad_config%water_species_scaling_snow

  ! Number of air columns each thread should process (with OpenMP)
  COLUMNS_PER_THREAD = ecRad_config%columns_per_cpu

! +--Extra ecRad outputs (extra_ecRad_outputs)
! +  =========================================

  ! Saves the configuration regarding whether or not time steps with minutes != 0 are saved
  TSTEPS_HOURLY_ONLY = ecRad_config%time_steps_hourly_only

  ! MAR longitudes/latitudes in ecRad precision (see comment in local variables)
  allocate(MAR_LON(dimLon, dimLat))
  allocate(MAR_LAT(dimLon, dimLat))
  
  do i=1, dimLon; do j=1, dimLat
    MAR_LON(i, j) = rLoRAD(i, j) ! Implicit conversion
    MAR_LAT(i, j) = rLaRAD(i, j)
  end do; end do

  ! Spectral decomposition of surface solar downward fluxes (requires additional configuration)
  if (ecRad_config%solar_spectral_output) then
    
    ! Deals with user-selected spectral bands
    if (ecRad_config%solar_spectral_bands_regular) then
      call sw_spectral_bands_from_range(ecRad_config%solar_spectral_bands_range)
    else
      call sw_spectral_bands_from_array(ecRad_config%solar_spectral_bands_full)
    end if
    
    ! Prepares the output file for spectral decomposition
    call create_sw_spectral_file(ecRad_config%solar_spectral_filename, & 
     &                           ecRad_config%solar_spectral_padding, &
     &                           dimLon, dimLat, MAR_LON, MAR_LAT)
    
  end if
  
  ! Additional surface/TOA fluxes from ecRad
  if (ecRad_config%additional_fluxes_output) then
    call create_extra_fluxes_file(ecRad_config%additional_fluxes_filename, & 
     &                            dimLon, dimLat, MAR_LON, MAR_LAT)
  end if

! +--Call to the ecRad setup subroutine (plus console output about configuration)
! +  ============================================================================

  call SETUP_RADIATION_SCHEME(ecRad_config%path_radiation_data, IOUTPUT=ecRad_config%verbosity)
  
  ! If verbosity > 0, more details on the configuration (MAR side) are appended to ecRad dump
  if (ecRad_config%verbosity > 0) then
    write(6, '(a)') 'ADDITIONAL RADIATIVE TRANSFER SCHEME PARAMETERS (MAR SIDE)'
    
    write(6, '(a,f6.4)') 'Scaling of the solar constant is ', RI0_SCALING
    
    if (allocated(BLANKET_LAYERS)) then
      write(6, '(a,i2,a)') 'Stratospheric layers (blanket) activated with', & 
       &                   size(BLANKET_LAYERS, 1), ' layers'
      if (ecRad_config%verbosity > 1) then
        write(6, '(a)', advance='no') '|--> Upper limits are: '
        do i=1, size(BLANKET_LAYERS, 1)
          if (i > 1) write(6, '(a)', advance='no') ', '
          write(6, '(f7.2,a)', advance='no') BLANKET_LAYERS(i), ' Pa'
        end do
        write(6, *) ! New line
      end if
    end if
    
    ! For each cloud fraction parametrization, parameters are detailed if verbosity is > 1
    if (NCLOUD_PARAM == 1) then
      write(6, '(a)') 'Cloud fraction computed with ECMWF Large Scale Cloudiness (no snow)'
      if (ecRad_config%verbosity > 1) then
        write(6, '(a,f4.2)') '|--> Gamma parameter is ', RCLOUD_PARAM_GAMMA
      end if
    else if(NCLOUD_PARAM == 2) then
      write(6, '(a)') 'Cloud fraction computed with ECMWF Large Scale Cloudiness (with snow)'
      if (ecRad_config%verbosity > 1) then
        write(6, '(a,f4.2)') '|--> Gamma parameter is ', RCLOUD_PARAM_GAMMA
      end if
    else if(NCLOUD_PARAM == 3) then
      write(6, '(a)') 'Cloud fraction computed with Sundqvist humidity-based formula (1989)'
      if (ecRad_config%verbosity > 1) then
        write(6, '(a)', advance='no') '|--> Humidity thresholds are: '
        write(6, '(f6.4,a)', advance='no') RCLOUD_RH_THRESHOLD_LAND, ' (land), '
        write(6, '(f6.4,a)') RCLOUD_RH_THRESHOLD_SEA, ' (sea)'
      end if
    else if(NCLOUD_PARAM == 4) then
      write(6, '(a)') 'Cloud fraction computed with Xu & Randall semiempirical formula (1996)'
      if (ecRad_config%verbosity > 1) then
        write(6, '(a)', advance='no') '|--> Parameters are '
        write(6, '(a,f6.2)', advance='no') 'alpha=', RCLOUD_PARAM_ALPHA
        write(6, '(a,f4.2)', advance='no') ', gamma=', RCLOUD_PARAM_GAMMA
        write(6, '(a,f4.2)') ', p=', RCLOUD_PARAM_P
      end if
    else
      write(6, '(a)') 'Cloud fraction values directly received from MAR'
    end if
    
    write(6, '(a,a,f6.4)') 'Fractional standard deviation of in-cloud water content fed to ', &
     &                     'ecRad is ', RCLOUD_FRAC_STD
    
    ! Water species scaling ratios are printed only when != 1.0
    if(SCALING_PQ /= 1.) write(6, '(a,f5.3)') 'Vapor mixing ratio scaling is ', SCALING_PQ
    if(SCALING_PQIWP /= 1.) write(6, '(a,f5.3)') 'Ice mixing ratio scaling is ', SCALING_PQIWP
    if(SCALING_PQLWP /= 1.) write(6, '(a,f5.3)') 'Liquid mixing ratio scaling is ', SCALING_PQLWP
    if(SCALING_PQRAIN /= 1.) write(6, '(a,f5.3)') 'Rain mixing ratio scaling is ', SCALING_PQRAIN
    if(SCALING_PQSNOW /= 1.) write(6, '(a,f5.3)') 'Snow mixing ratio scaling is ', SCALING_PQSNOW
    
    ! Additional output files (+ spectral bands if verbosity > 1 and spectral bands used)
    if (ecRad_config%additional_fluxes_output .or. &
     &  ecRad_config%climatologies_scaled_output .or. & 
     &  ecRad_config%climatologies_interpolated_output .or. &
     &  ecRad_config%solar_spectral_raw_output .or. ecRad_config%solar_spectral_output) then
      write(6, '(a)') '-------------------------------------------------------------------------------'
      write(6, '(a)') 'ADDITIONAL OUTPUT FILES'
    end if
    
    if (ecRad_config%additional_fluxes_output) then
      write(6, '(a,a)') 'Extra surface fluxes: ', &
       &                trim(ecRad_config%additional_fluxes_filename)
    end if
    
    if (ecRad_config%climatologies_scaled_output) then
      write(6, '(a,a)') 'Climatologies scaled to CMIP scenario: ', &
       &                trim(ecRad_config%climatologies_scaled_filename)
    end if
    
    if (ecRad_config%climatologies_interpolated_output) then
      write(6, '(a,a)') 'Climatologies scaled to CMIP scenario and interpolated on MAR grid: ', &
       &                trim(ecRad_config%climatologies_interpolated_filename)
    end if
    
    if (ecRad_config%solar_spectral_raw_output) then
      write(6, '(a,a)') 'Raw spectral surface fluxes: ', &
       &                trim(ecRad_config%solar_spectral_raw_filename)
    end if
    
    if (ecRad_config%solar_spectral_output) then
      write(6, '(a,a)') 'Spectral surface fluxes: ', &
       &                trim(ecRad_config%solar_spectral_filename)
      if (ecRad_config%verbosity > 1) then
        write(6, '(a)') '|--> Boundaries of spectral bands (nm):'
        do i=1, N_SW_SPECTRAL_BANDS+1 ! From the extra_ecRad_outputs module
          write(6, '(f8.2)', advance='no') (sw_spectral_bounds(i) * 1e9) ! m -> nm
          if(i <= N_SW_SPECTRAL_BANDS) write(6, '(a)', advance='no') ', '
          if(modulo(i, 10) == 0 .and. i <= N_SW_SPECTRAL_BANDS) write(6, *) ! New line
        end do
        write(6, *) ! New line
      end if
    end if
    
    write(6, '(a)') '-------------------------------------------------------------------------------'
  end if
  
  if (ecRad_config%solar_spectral_raw_output) then
    call create_sw_spectral_raw_file(ecRad_config%solar_spectral_raw_filename, &
     &                               ecRad_config%solar_spectral_raw_padding, &
     &                               dimLon, dimLat, MAR_LON, MAR_LAT, & 
     &                               rad_config%gas_optics_sw%spectral_def, & 
     &                               .not. rad_config%do_cloud_aerosol_per_sw_g_point)
  end if

! +--Preparation of climatologies
! +  ============================
! +  J.-F. Grailet remarks:
! +  -Greenhouse gases/aerosols concentrations are assumed to be constant throughout a single MAR 
! +   run, since a MAR run usually covers up to 2 weeks within the same month. The concentrations 
! +   are therefore interpolated only once.
! +  -To get a representative interpolation, the code finds the air column within the MAR grid 
! +   whose surface pressure is the highest. It then uses the pressures from this air column and 
! +   the MAR latitudes from the surface (2D grid) column containing said air column.
! +  -To interpolate aerosols, given that aerosols mass mixing ratios are provided in 3D grids 
! +   covering the whole Earth surface but with 3° of longitude of difference per surface cell 
! +   (which is much, much larger than the typical resolution of MAR), the average longitude of 
! +   the regional grid is computed to select a longitudinal slice in each aerosol 3D grid.

  nbStrato = 0
  if (allocated(BLANKET_LAYERS)) then
    nbStrato = size(BLANKET_LAYERS)
  end if

  allocate(CLIM_LATITUDES(dimLat))
  allocate(CLIM_LEVELS(nbStrato + nbLvls))

  max_pressure = 0.0
  max_pressure_i = -1
  max_pressure_j = -1
  sum_longitudes = 0.0
  do i=1, dimLon; do j=1, dimLat
    if (rp_RAD(i, j, nbLvls) > max_pressure) then
      max_pressure = rp_RAD(i, j, nbLvls)
      max_pressure_i = i
      max_pressure_j = j
    end if
    sum_longitudes = sum_longitudes + MAR_LON(i, j)
  end do; end do
  
  ! Average longitude used to pick aerosols mass mixing ratios (longitudinal slice)
  CLIM_LONGITUDE = sum_longitudes / (dimLon * dimLat)
  
  do j=1, dimLat
    CLIM_LATITUDES(j) = rLaRAD(max_pressure_i, j)
  end do

  do k=1, nbLvls
    CLIM_LEVELS(nbStrato + k) = rp_RAD(max_pressure_i, max_pressure_j, k)
  end do
  
  if (nbStrato > 0) then
    do k=1, nbStrato
      CLIM_LEVELS(k) = BLANKET_LAYERS(k)
    end do
  end if
  
  smoothing = ecRad_config%interpolation_smoothing_method
  
  ! MAR longitude/latitude values are no longer needed from here
  deallocate(MAR_LON)
  deallocate(MAR_LAT)
  
  ! Reads and interpolate the greenhouse gas climatologies
  call greenhouse_clim%allocate(trim(ecRad_config%path_greenhouse_data), &
   &                            trim(ecRad_config%path_cmip_scenarios) &
   &                            //trim(CMIP_scenario)//'.nc', &
   &                            ecRad_config%greenhouse_period_start, & 
   &                            ecRad_config%greenhouse_period_end, mm, yyyy)

  call greenhouse_clim%interpolate(ICO2, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PCO2)
  call greenhouse_clim%interpolate(ICH4, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PCH4)
  call greenhouse_clim%interpolate(IN2O, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PN2O)
  call greenhouse_clim%interpolate(ICFC11, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PCFC11)
  call greenhouse_clim%interpolate(ICFC12, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PCFC12)
  call greenhouse_clim%interpolate(IHCFC22, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PHCFC22)
  call greenhouse_clim%interpolate(ICCl4, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PCCL4)
  call greenhouse_clim%interpolate(IO3, CLIM_LATITUDES, CLIM_LEVELS, smoothing, PO3)
  
  ! Option to output the (scaled) monthly data so MAR users can compare with interpolations
  if (ecRad_config%climatologies_scaled_output) then
    call greenhouse_clim%output_monthly_values(trim(ecRad_config%climatologies_scaled_filename))
  end if
  
  call greenhouse_clim%deallocate()
  
  ! Aerosols mass mixing ratios for MACC aerosol climatology
  if (NAERMACC > 0) then
    call aerosols_clim%allocate(trim(ecRad_config%path_aerosols_data), mm)
    
    allocate(PAEROSOL(dimLat, nbStrato + nbLvls, N_AEROSOLS_MACC))
    
    do i=1, N_AEROSOLS_MACC
      call aerosols_clim%interpolate(i, CLIM_LONGITUDE, CLIM_LATITUDES, & 
       &                             CLIM_LEVELS, smoothing, PAEROSOL)
    end do
    
    call aerosols_clim%deallocate()
  end if
  
  ! Old aerosol format, used for Tegen climatology but also for some adjustments of the MACC 
  ! aerosols climatology (see code of drivers/radiation_scheme.F90).
  allocate(PAEROSOL_OLD(dimLat, N_AEROSOLS_TEGEN, nbStrato + nbLvls))
  
  do j=1, dimLat; do l=1, N_AEROSOLS_TEGEN; do k=1, nbStrato + nbLvls
    PAEROSOL_OLD(j, l, k) = 1.e-12
  end do; end do; end do
  
  ! Option to output the interpolated data, so MAR users can check it in case of problem
  if (ecRad_config%climatologies_interpolated_output) then
    call output_interpolated(trim(ecRad_config%climatologies_interpolated_filename))
  end if
  
  ! Advertises mean CO2 in ppm (X. Fettweis).
  mean_CO2 = 0.
  do i=1, dimLat; do j=1, nbStrato + nbLvls
    mean_CO2 = mean_CO2 + PCO2(i, j)
  end do; end do
  write(6, '(a,f6.2,a)') 'Mean CO2 concentration: ', & 
   &                     1000000. * (mean_CO2 / real(dimLat * (nbStrato + nbLvls))), ' ppm.'
  
  write(6, '(a)') '-------------------------------------------------------------------------------'
  
  return
end subroutine PHYrad_ecRad_init
